/* pseudo.c */
/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */

/*
 *	Gould, Inc. Base Register Assembler
 */

#ifndef lint
/*NOBASE*/
static char *rcsid = "@(#) (Gould) $Header: pseudo.c,v 5.5 89/05/12 12:37:27 pcc Rel-3_0 $";
#endif

#include "parse.h"
#include "struct.h"
#include "operand.h"
#ifdef JCB
#include <a.out.h>
#else
#include "a.out.h"
#endif
#include <stdio.h>

extern OPND oper[];

Tuchar safe_place[MAX_STR];	/* temporary token storage */

bvalT bval[8] =
{
    last_SEGS,0,last_SEGS,0,last_SEGS,0,last_SEGS,0,
    last_SEGS,0,last_SEGS,0,last_SEGS,0,last_SEGS,0
};
int bvflags;			/* bits in these are set if .using in effect */






/*
NAME: alloc_lcomm
PURPOSE: allocate bss space for a local common
PRECONDITIONS:
called only from op_comm, this routine exists to simplify op_comm.
symp is a pointer to the symbol table entry for the lcomm label.
flags is the value assigned to the sflags symbol table field by op_comm.
POSTCONDITIONS:
the value of dot for the appropriate segment is set with appropriate
rounding. the value of the lcomm label (which is the offset into the bss
or farbss segment at which the common BEGINS) is set after rounding but
before the space is allocated.
ALGORITHM:
the rounding algorithm was taken directly from the loader (recall that
common (as opposed to current case of local common) must be allocated by
the loader).
HISTORY:
split out of op_comm RJM
Tue Nov  1 10:07:09 CST 1983
*/






alloc_lcomm (symp, flags)
register SYM *symp;
register int flags;
{
    enum SEGS oldseg;
    unsigned long rnd;

    oldseg = dot -> sspace;	/* remember where we are */
    if (flags & S_FAR)		/* which segment? */
    {
	setseg (S_farbss);
    }
    else
    {
	setseg (S_bss);
    }
    /* get the alignment right first */
    /* rounding algorithm taken from the loader */
    if (result.constval >= sizeof (double))
    {
	rnd = sizeof (double) - 1;
    }
    else if (result.constval >= sizeof (long))
    {
	rnd = sizeof (long) - 1;
    }
    else
    {
	rnd = sizeof (short) - 1;
    }

    dot -> svalue += rnd;
    dot -> svalue &= ~(long) rnd;

    symp -> svalue = dot -> svalue;

    dot -> svalue += result.constval;	/* increase the space */
    setseg (oldseg);	/* return to the regularly scheduled ... */
}






/*
NAME: do_align
PURPOSE: fix alignment within current segment.
PRECONDITIONS:
parameter align must be a power of 2, otherwise the masking will not
work correctly.
this routine can be called from a variety of places, but especially from
opcode handling routines that require specific alignment and from
op_align, which forces a specified alignment.
POSTCONDITIONS:
if called in pass 1 (where only size is needed), the current segment
size is set to the appropriate boundary.
if called in pass 2, data bytes are emitted as required to get the
object file aligned (except for bss and farbss space, which is never
emitted to). note that the nop instruction is emitted to the text and
fartext space and the value 0 (zero) is emitted otherwise. the text
segment should never be sitting on an odd byte (not a halfword)
boundary, but if so emit a 0 byte first to align properly.
ALGORITHM:
the standard add and truncate rounding algorithm is used. as long as
align is a power of 2, the assignment of align - 1 to mask will set all
the appropriate low order bits to 1 in mask and the emission of padding
will be done correctly (pad emission will not work if not a power of two
although dot will be set correctly as per the value of align).
HISTORY:
*/






do_align (align)
register int align;
{				/* routine to do alignment */
    register int mask;
    extern SYM *newest_label;

    mask = align - 1;

    if (pass == 1)
    {
	/* check if at label which needs to be padded past NOP */
	if (mask == 3
		&& (dot -> sspace == S_fartext || dot -> sspace == S_text)
		&& (dot -> svalue & mask) == 2
		&& newest_label && newest_label -> sspace == dot -> sspace
		&& newest_label -> svalue == dot -> svalue)
	{
	    newest_label -> svalue += 2;
	    newest_label -> sflags |= S_PAD;
	}

				/* in pass1, just round up */
	dot -> svalue += mask;
	dot -> svalue &= ~(mask);
#if defined(COFF)
	/* set segment alignment */
	if (align > seginf[(int) dot -> sspace].si_align)
	    seginf[(int) dot -> sspace].si_align = align;
#endif /* COFF */
    }
    else
    {				/* fill in vacated space */
	switch (dot -> sspace)
	{
	    case S_bss:
	    case S_farbss:	/* for bss, just round */
		dot -> svalue += mask;
		dot -> svalue &= ~(mask);
		break;

	    case S_text:
	    case S_fartext:
		if (dot -> svalue & mask & 1)
		{
		    emit_dbyte (0);
		}

		while (dot -> svalue & mask)
		{
		    emit_half (0x0002);		/* put out a NOP */
		}
		break;

	    case S_data:
	    case S_fardata:
#ifdef POOL
	    case S_Lconstant:
	    case S_Dconstant:
	    case S_Wconstant:
	    case S_Xconstant:
	    case S_Hconstant:
#else /* not POOL */
	    case S_constant:
#endif /* POOL */
	    case S_dat2:
	    case S_fdat2:
	    case S_tex2:
	    case S_ftex2:
		while (dot -> svalue & mask)
		{
		    emit_dbyte (0);
		}
		break;
	}
    }
}






/*
NAME: scan_ascii
PURPOSE: scan an ascii string and put it into a buffer.
PRECONDITIONS:
buf is a pointer to a Tuchar, the address of a buffer to put the scanned
string into. string can be no longer than MAX_STR chars due to the
declaration in op_ascii.
POSTCONDITIONS:
the length of the string in characters is returned on success.
if an error is found, the string is still scanned, but the result is
unpredictable.
the string is copied into the space pointed to by buf.
ALGORITHM:
the first non-space character of the operand is assumed to be the
delimiter and must appear at the end of the string. the usual C backslash
escape sequence is used.
we force the delimitor to be a double quote (") to prevent
unpleasantries with the routine scan_cmt in pass.c, which is too dumb
(because it must be fast) to handle an arbitrary string delimitor.
HISTORY:
20 Jun 83	D A Willcox of Compion	Initial coding
*/






scan_ascii (buf)
Tuchar *buf;			/* buffer to put converted string into */
{
    register Tuchar *yyl;		/* temp copy of yylinept */
    register Tuchar delim;
    register Tuchar *bp;

    bp = buf;
    yyl = yylinept;

    delim = *yyl++;		/* remember delimiter char */

    if (delim != '"')
    {
	error ("scan_ascii: illegal string delimitor %c", delim);
    }

    while (*yyl != delim)
    {
	switch (*yyl)
	{
	    default:
		*bp++ = *yyl;
		break;

	    case '\n':		/* whoops! */
		error ("scan_ascii: unterminated string");
		--yyl;
		goto out;

	    case '\\':		/* check next character */
		++yyl;
		switch (*yyl)
		{
		    default:
			*bp++ = *yyl;
			break;

		    case 'b':
			*bp++ = '\b';
			break;
		    
		    case 'f':
			*bp++ = '\f';
			break;
		    
		    case 'n':
			*bp++ = '\n';
			break;
		    
		    case 't':
			*bp++ = '\t';
			break;

		    case 'r':
			*bp++ = '\r';
			break;

		    case '0': case '1': case '2': case '3':
		    case '4': case '5': case '6': case '7':
			{		/* convert string like \123 */
			    int i;
			    int ch;

			    ch = 0;
			    for (i=0; i<3; ++i)
			    {
				ch = ch << 3;
				ch += (*yyl++ - '0') & 0377;
				if (!(isdigit (*yyl)))
				    break;
			    }

			    --yyl;
			    *bp++ = ch;
			}
			break;
		}
		break;
	}

	++yyl;
    }

  out:
    yylinept = yyl + 1;

    return (bp - buf);
}






/*
NAME: setseg
PURPOSE: change the current segment.
PRECONDITIONS:
parameter seg is the SEGS value of the segment to change to.
POSTCONDITIONS:
this routine WILL change the segment, it does not check whether the
current segment is the same as seg or that the far_flag should prevent
a change. these checks must be done by the caller (typically op_space).
ALGORITHM:
the size of the current segment is taken to be the value of dot and is
stored in the seginf structure. the size of the new segment is then taken
from the seginf entry for the new space to set up the new location.
HISTORY:
part of initial code RJM
*/






setseg (seg)
enum SEGS seg;			/* change to end of given segment */
{
    cur_seg -> si_dot = dot -> svalue;
    cur_seg = &seginf[(int) seg];
    dot -> svalue = cur_seg -> si_dot;
    dot -> sspace = (enum SEGS) seg;
}






#ifndef GOULD_NP1
/*
NAME: op_addr
PURPOSE: similar to .word, but allow high byte of non-relocatable data
PRECONDITIONS:
the intent of this pseudo op is to allow the use of the high byte of a
relocatable word for constant data. the primary use is in building
things like a processor status doubleword, where there are state bits in
the high byte and an address in the lower 3 bytes.
parameter ptr is the pointer to the opcode table entry for .addr
the first operand should be a byte of (constant) data, the second
operand should be up to 24 bits of (possibly relocatable) data.
POSTCONDITIONS:
alignment to a word boundary is guaranteed. the result is emitted to the
"current" segment. if the second operand is relocatable, emit_rel is
called to handle the requisite relocation information.
HISTORY:
Fri Aug 19 15:05:30 EDT 1983 RJM
*/






op_addr (ptr)
register struct opcode *ptr;
{
    int val;

    do_align (4);		/* word align whatever this thing is */

    SKIPSPACE (yylinept);
    validate_op (ptr, 0);	/* get the first operand */
    SKIPSPACE (yylinept);
    if (*yylinept != ',')	/* assure there is a comma */
    {
	error ("op_addr: comma expected");
	SKIPEOS (yylinept);
	return;
    }
    ++yylinept;			/* skip over the comma */
    SKIPSPACE (yylinept);
    validate_op (ptr, 1);	/* get the second operand */

    if (pass == 1)
    {
	dot -> svalue += 4;	/* just set segment size in this pass */
    }
    else
    {
	val = oper[1].constval;
	if (oper[1].haveseen & NAME)	/* relocatable? */
	{
	    val = emit_rel (R_ADDRLITERAL, R_WORD, 
		oper[1].symname, oper[1].constval);
	}
	if (((unsigned) oper[0].constval > 0xFF) ||
	    ((unsigned) oper[1].constval > 0xFFFFFF))
	{
	    warning ("op_addr: operand precision lost");
	}
	emit_dword ((oper[0].constval << 24) | val);
    }
}






#endif /* NOT GOULD_NP1 */
/*
NAME: op_align
PURPOSE: handle the .align directive
PRECONDITIONS:
ptr is a pointer to the .align opcode table entry.
expects 1 operand, must have a constant value, used as a power of 2.
ALGORITHM:
actual work is done by routine do_align. this routine just handles the
syntax checking.
HISTORY:
part of initial code DAW
explicit cast of NULL to Tuchar * in DEBUG section - fix from rmiller
*/






op_align(ptr)
struct opcode *ptr;
{

    if (yyparse())		/* returns 1 on failure, 0 on success */
    {
	error ("op_align: yyparse syntax error");
	SKIPEOS (yylinept);
	return;
    }
    else
    {
#ifdef DEBUG
	if(dbflag)
		printf ( "op_align: dot %d result %d - %d - %d - %d - %d\n%s\n",
		  dot -> svalue, result.type, result.haveseen, result.regval,
		  result.baseval, result.constval,
		  result.symname ? result.symname -> sname : (Tuchar *) "NULL");
#endif

	if (result.haveseen != CONST)
	{
	    error ("op_align: non-constant expression in operand");
	}

	do_align ((unsigned) 1 << result.constval);

    }
    SKIPSPACE (yylinept);
}






/*
NAME: op_ascii
PURPOSE: handle the .ascii pseudo op.
PRECONDITIONS:
op is the usual pointer to the optable.
one operand, a single string delimited by the first nonspace character
in the operand field, is expected.
the declaration of local variable tbuf determines the maximum length
that the string can be.
POSTCONDITIONS:
this routine is essentially a special case of op_setmem.
HISTORY:
20 Jun 83	D A Willcox of Compion	Initial coding
*/






op_ascii (op)
register struct opcode *op;
{
    Tuchar tbuf[MAX_STR];			/* buffer to build data */
    register Tuchar *bp;
    register int len;
    register int val;

    SKIPSPACE (yylinept);		/* skip white space */

    len = scan_ascii (tbuf);		/* process the string */

    if (pass == 1)
    {					/* in pass 1, just bump size */
	dot -> svalue += len;
    }
    else
    {					/* in pass 2, put it out */
	bp = tbuf;

	while (--len >= 0)
	    emit_dbyte (*bp++);
    }
}






/*
NAME: op_base
PURPOSE: handle the .using pseudo op
PRECONDITIONS:
ptr is the usual optable pointer.
the first operand must be a base register, the second must be a constant
or the name of an internal (to this file) symbol.
POSTCONDITIONS:
if the second operand is the constant value 0 (zero), then the effect of
a previous .using directive (if any), is turned off. otherwise, the
specified base register is set to the address of the internal symbol and
later memory references within the current segment are made relative to
this base.
HISTORY:
part of initial code DAW
*/






extern SYM *curr_func_symp;
extern int dom_max, dom_begin;
extern int curr_domain;
extern int dom_txtadd, dom_tbladd;

op_base(ptr)
register struct opcode *ptr;
{
register bvalT *bp;			/* pointer into bval table */
register SYM *symp = 0;

    if (validate_op (ptr, 0) == 0)
    {
	SKIPEOS (yylinept);
	return;				/* parse error */
    }

    SKIPSPACE (yylinept);

    if (*yylinept != ',')
    {
	error ("op_base: comma expected");
	SKIPEOS (yylinept);
	return;
    }
    ++yylinept;

    if (validate_op (ptr, 1) == 0)
    {
	SKIPEOS (yylinept);
	return;				/* parse error */
    }
    
    bp = &bval[oper[0].baseval];	/* point at bval table entry */

    switch (oper[1].type)
    {					/* look at 2nd arg */
	case CONSTANT:
	case IMMEDIATE:
	case MEMORY:
	    if ((oper[1].haveseen & NAME) == 0)
	    {				/* this is just a constant */
		if (pass == 2) {
		    if (oper[1].constval == 0)
		    {			/* we are dropping reg value */
			bp -> bv_seg = last_SEGS;
			bvflags &= ~((unsigned) 1 << oper[0].baseval);
		    	bp -> bv_offset = 0;
		    }
		    else
		    {
		    	error("op_base: illegal base value");
		    }
		}
	    }
	    else
	    {
		symp = oper[1].symname;
#ifndef OLDCALL
		if (oper[0].baseval == 1) {
		    curr_func_symp = symp;
		}
#endif /* NOT OLDCALL */
		if (pass == 1) {
		    if (curr_func_symp && curr_func_symp->sspace == S_fartext)
			dom_begin = curr_func_symp->svalue;
		}
		if (pass == 2) {
		    if (symp -> sflags & S_DEF)
		    {			/* symbol is internal */
			bvflags |= ((unsigned) 1 << oper[0].baseval);
			bp -> bv_seg = symp -> sspace;
			bp -> bv_offset = symp -> svalue + oper[1].constval;
		    }
		    else
		    {
			error("op_base: .using must be w.r.t. internal symbol");
		    }
		    if (curr_func_symp && curr_func_symp->sspace == S_fartext)
			dom_max = curr_func_symp->sextra.domain;
		}
	    }
	    break;

	default:
	    if (pass == 2)
		error ("op_base: Illegal .using type %d", oper[1].type);
	    SKIPEOS (yylinept);
	    return;
    }
}






/*
NAME: op_be
PURPOSE: handle the .bf and .ef pseudo ops
POSTCONDITIONS:
these ops may be used to put out debug information and to flag when a
local constant segment may be written in a later release.
for now, they do nothing other than accept the directives and skip past
the statement.
HISTORY:
part of initial code DAW
*/






extern int procno;
#ifndef OLDCALL
extern int func_lab_flg;
extern SYM *func_lab;
#endif /* NOT OLDCALL */

typedef struct e_stk *E_STKP;
typedef struct e_lchn *E_LCHNP;

struct e_stk {
	E_STKP e_next;
	enum SEGS e_space;
	SYM *e_sym0;
	int e_off0;
	SYM *e_sym1;
	int e_off1;
	E_LCHNP e_lfirst, e_llast;
	short int e_retns, e_entrys;
#ifdef R_SV
	short int e_funcs;
#endif /* R_SV */
	unsigned char e_in, e_var;
#ifdef GOULD_PN
	int e_domain;
#endif /* GOULD_PN */
};

struct e_lchn {
	E_LCHNP e_lnext;
	SYM *e_lab;
	short int e_lretns, e_lentrys;
#ifdef R_SV
	short int e_lfuncs;
#endif /* R_SV */
	unsigned char e_lin;
};

extern E_STKP e_stkp;
extern HIST *h_start;

op_be(ptr)
struct opcode *ptr;
{
    E_STKP ep;

    if (!strcmp(ptr->oname, ".bf")) {
	++procno;
	dom_init();
#ifndef OLDCALL
	if (pass == 1)
	    e_begin();
	else if (pass == 2) {
	    ++func_lab_flg;
	    func_lab = 0;
	}
#endif /* NOT OLDCALL */
    }
    else if (!strcmp(ptr->oname, ".ef")) {
	if (pass == 1) {

	   {
	   register HIST *hp, *tp;
	   for (hp = h_start; hp;)
	   {
		switch(hp->kind) {
		    case H_UNBR:
		    case H_COBR:
			tp = hp->fwrd;
			if (!(hp->refp->sflags & S_DEF)) {
	    		    hp->refp->sextra.domain = 0;
			    dom_histdel(hp);
			}
			hp = tp;
			free(tp);
			break;
	
		    default:
			hp = hp->fwrd;
			break;
		    }
	   }
	   }
	    dom_fixup();
	    if (curr_func_symp && curr_func_symp->sextra.domain > 0
		&& dom_txtadd+dom_tbladd != 0) {
		    if (ep = e_stkp) {
#ifdef E_SV
			ep->e_retns = 0;
			ep->e_entrys = 0;
#else
#ifdef R_SV
			ep->e_funcs = 0;
#endif /* R_SV */
#endif /* E_SV */
		    }
	        dot->svalue = dom_begin + 0x10000 +
#ifdef GOULD_NP1
		8;	/* Rev E call/return */
#else /* GOULD_PN */
		4;
#endif /* GOULD_NP1 */
		}
  
#ifndef OLDCALL
	    e_end();
#endif /* NOT OLDCALL */
	}
	else if (pass == 2) {
	    put_dom_tbl();
#ifndef OLDCALL
	    func_lab = 0;
#endif /* NOT OLDCALL */
	}
	curr_func_symp = 0;
    }
    SKIPEOS (yylinept);
}






/*
NAME: op_comm
PURPOSE: handle the .comm and .lcomm pseudo ops
PRECONDITIONS:
ptr is the usual optable pointer.
the first operand is a symbol that (for .comm) will be put in bss or
farbss spaces IF IT IS NOT LATER DEFINED IN THE SOURCE or (for .lcomm)
will be put in bss or farbss spaces.
the second operand must be a constant value specifying the number of
bytes to allocate.
POSTCONDITIONS:
common symbols are defined in the a.out symbol table as UNDEFINED and
EXTERNAL with a nonzero value (actually the size of the common) if they
are not defined in this file after the .comm statement.
local common symbols are treated as labels in bss or farbss space (as
per the far_flag) with some amount of space allocated after the label.
caution is required with local common alignment (the loader handles
alignment for common).
HISTORY:
part of initial code DAW
local common fixed RJM
Fri Aug 19 11:04:21 EDT 1983
code restructured as part of label handling clean up RJM
Tue Nov  1 20:01:03 CST 1983
*/






op_comm(ptr)
register struct opcode *ptr;
{
    register SYM *symp;
    extern Tuchar token[];		/* output of get_token */
    register int flags;

    SKIPSPACE (yylinept);

    get_token ();			/* scan a symbol */
    if (ptr -> ocode == 5 || ptr -> ocode == 6)
    {
	Tuchar tmptok[80];
	int len, n;

	if (*yylinept != ':')
	    error ("op_comm: .(near)dpool expects ':' in name");
	strcpy(tmptok, token);
	strcat(tmptok, ":");
	len = strlen(tmptok);
	++yylinept;
	get_token();
	for (n =strlen(token); n >= 0; --n)
	    token[n + len] = token[n];
	strncpy(token, tmptok, len);
    }

    if (pass == 1)
    {
	symp = defsym (token);
    }
    else	/* better be pass == 2 */
    {
	symp = lookup (token);
    }
    /* the following return should NEVER happen */
    if (symp == 0)
	return;

    flags = S_COMM;

    /* .comm == 0 and .farcomm == 2, must be external */
    if (ptr -> ocode == 0 || ptr -> ocode == 2)
    {
	flags |= S_EXT;
    }

    /* .farcomm == 2 and .farlcomm == 3, must be far */
    if (far_flag || ptr -> ocode == 2 || ptr -> ocode == 3)
    {
	flags |= S_FAR;
    }

    /* .gcomm == 4 is eligible for sharing */
    if (ptr -> ocode == 4)
	flags |= (S_SHARE | S_EXT);

    /* .dpool == 5 is eligible for (unbounded) datapool resolution */
    if (ptr -> ocode == 5)
	flags |= (S_UBDP | S_EXT);

    /* .bpool == 6 is eligible for (bounded) datapool resolution */
    if (ptr -> ocode == 6)
	flags |= (S_BDDP | S_EXT);

    /* .lcomm == 1 and .farlcomm == 3, must be defined */
    /* local common ONLY for relocation (see emit_rel in aout.c) */
    if (ptr -> ocode == 1 || ptr -> ocode == 3)
    {
	flags |= S_DEF;
    }

    if (pass == 1)
    {
	if (symp -> sflags & S_DEF)
	{
	    if (symp -> sflags != flags)
	    {
		error ("op_comm: %s redefined", symp -> sname);
		SKIPEOS (yylinept);
		return;
	    }
	}
	else
	{
	    symp -> sflags = flags;		/* record type of symbol */

	    if (flags & S_FAR)
		symp -> sspace = S_farbss;
	    else
		symp -> sspace = S_bss;
	}

    }

    SKIPSPACE (yylinept);
    if (*yylinept == ',')
    {
	yylinept++;
    }
    else
    {
	error ("op_comm: comma expected");
	SKIPEOS (yylinept);
	return;
    }

    SKIPSPACE (yylinept);
    if (yyparse ())
    {
	error ("op_comm: yyparse syntax error");
	SKIPEOS (yylinept);
	return;
    }
    else
    {
	if (pass == 1)
	{
	    if (result.haveseen == CONST)
	    {
		if (symp -> svalue < result.constval)
		    symp -> svalue = result.constval;
	    }
	    else
	    {
		error ("op_comm: illegal operand");
		SKIPEOS (yylinept);
		return;
	    }
	}
    }

/* local common symbols are treated as labels in bss space */
/* now, allocate the space in the appropriate bss segment */
    if (ptr -> ocode == 1 || ptr -> ocode == 3)
    {
	alloc_lcomm (symp, flags);
    }
}






/*
NAME: op_float
PURPOSE: a (temporary) routine to help parse floating point constants.
PRECONDITIONS:
op is the usual optable pointer.
this routine is called from op_setmem when a floating point constant is
found (by detecting prefix 0d or 0f). yylinept should already be
pointing at the beginning of the operand and only one operand is parsed.
POSTCONDITIONS:
in pass 1, yylinept is moved past the constant while it is copied into a
temporary location.
in pass 2, the copy is done followed by a call to atof to convert the
string into a floating point value which is then emitted with the
appropriate length (either float (32 bits) or double (64 bits)).
HISTORY:
6 Jul 83	D A Willcox of Compion	Initial coding
*/






op_float (op)
register struct opcode *op;
{
    Tuchar tbuf[MAX_STR];			/* buffer to accumulate */
    register Tuchar *bp;
    register int c;
    double atof ();
    union
    {
	int dwords[2];
#ifdef LDPOOL
	float fval;
#endif /* LDPOOL */
	double dval;
    } d_un;
#ifdef REP_COUNT
    int repct;
#endif /* REP_COUNT */

    yylinept += 2;			/* first two are "0f" or "0d" */

    bp = tbuf;

    while ((c = *yylinept)
	&& !isspace (c)
	&& c != ';'
#ifdef REP_COUNT
	&& c != ','
	&& c != '@')
#else /* not REP_COUNT */
	&& c != ',')
#endif /* REP_COUNT */
    {					/* copy the fp constant */
	*bp++ = c;
	++yylinept;
    }

    *bp = 0;

    if (pass == 1)
	return;				/* only emit in pass 2 */

#ifdef REP_COUNT
    if (op -> ocode != 4 && op -> ocode != 8) {
	error ("op_float: invalid type");
	return;
    }

#endif /* REP_COUNT */
    d_un.dval = atof (tbuf);		/* convert the constant */
#ifdef LDPOOL
    if (op -> ocode == 4)
	d_un.fval = d_un.dval;		/* for rounding */
#endif /* LDPOOL */

#ifdef REP_COUNT
    repct = 1;

    if (*yylinept == '@') {
	++yylinept;
	if (!yyparse()) {
	    if (result.type != MEMORY || result.haveseen != CONST)
		error("op_float: bad repeat count");
	    else
		repct = result.constval;
	}
    }

    while (repct-- > 0) {
	if (op -> ocode == 8) {
#else /* not REP_COUNT */
    switch (op -> ocode)
    {
	case 8:
#endif /* REP_COUNT */
	    emit_dword (d_un.dwords[0]);
	    emit_dword (d_un.dwords[1]);
#ifdef REP_COUNT
	}
	else
#else /* not REP_COUNT */
	    break;

	case 4:
#endif /* REP_COUNT */
	    emit_dword (d_un.dwords[0]);
#ifndef REP_COUNT
	    break;

	default:
	    error ("op_float: invalid type");
	    break;
#endif /* NOT REP_COUNT */
    }
}






/*
NAME: op_globl
PURPOSE: handle the .globl pseudo op
PRECONDITIONS:
ptr is the usual optable pointer.
op_globl expects a list of names separated by commas as operands.
POSTCONDITIONS:
if a symbol does not exist in the symbol table (pass 1 only), it will
be created by the defsym call in yyparse.
the external and declared flags are set for the symbol in the symbol
table.
ALGORITHM:
yyparse does the work, each operand must be a name.
if a comma is not found after a name, assume that the list of names has
ended and return to op1.
HISTORY:
part of initial code DAW
*/






op_globl(ptr)
struct opcode *ptr;
{
    for (;;)
    {
	if (yyparse())
	{
	    error ("op_globl: yyparse syntax error");
	    break;
	}

	if (pass == 1)
	{
	    if (result.haveseen == NAME)
	    {
		result.symname -> sflags |= S_EXT | S_DECL;
	    }
	    else
	    {
		error ("op_globl: illegal operand");
	    }
	}

	SKIPSPACE(yylinept);
	if (*yylinept != ',')
	    break;

	++yylinept;			/* get another parameter */
    }
}






/*
NAME: op_nearfar
PURPOSE: handle the .near and .far pseudo ops
PRECONDITIONS:
ptr is the usual optable pointer.
no operands are expected.
the variable far_flag is a global flag that indicates when symbol
declarations should be defaulted into a far space.
POSTCONDITIONS:
far_flag is set appropriately, then the space is changed if the current
space is not appropriate to the pseudo op.
HISTORY:
part of initial code DAW
*/






op_nearfar(ptr)
struct opcode *ptr;
{
    enum SEGS seg = last_SEGS;

    far_flag = ptr -> ocode;

    switch (dot -> sspace)
    {
    /* if we have .far while in a near space */
    case S_text:
	if (far_flag)
	{
	    seg = S_fartext;
	}
	break;
    case S_tex2:
	if (far_flag)
	{
	    seg = S_ftex2;
	}
	break;
    case S_data:
	if (far_flag)
	{
	    seg = S_fardata;
	}
	break;
    case S_dat2:
	if (far_flag)
	{
	    seg = S_fdat2;
	}
	break;
    case S_bss:
	if (far_flag)
	{
	    seg = S_farbss;
	}
	break;
    /* .near while in a far space, just turn off far_flag */
    case S_fartext:
    case S_ftex2:
    case S_fardata:
    case S_fdat2:
    case S_farbss:
	break;
    /* just set the far_flag if in constant space */
#ifdef POOL
    case S_Lconstant:
    case S_Dconstant:
    case S_Wconstant:
    case S_Xconstant:
    case S_Hconstant:
#else /* not POOL */
    case S_constant:
#endif /* POOL */
	break;
    default:
	error ("op_nearfar: current space not recognized %d",
	    dot -> sspace);
	break;
    }

    /* now, if we changed anything, make sure it gets changed */
    if (seg != last_SEGS)
    {
	setseg (seg);
    }
}






/*
NAME: op_res
PURPOSE: handle the .space pseudo op.
PRECONDITIONS:
op is the usual optable pointer.
a constant valued operand is expected.
POSTCONDITIONS:
dot is fixed appropriately in pass 1. in pass 2, bytes with the value
zero are emitted if the current space is not a bss space.
a warning is printed if the operand is <= zero.
HISTORY:
24 Jun 83	D A Willcox of Compion	Initial coding
*/






op_res (op)
register struct opcode *op;
{
    register int i;

    if (yyparse())		/* returns 1 on failure, 0 on success */
    {
	SKIPEOS (yylinept);
	return;
    }

    if (result.haveseen != CONST)
    {
	error ("op_res: non-constant for %s", op -> oname);
	SKIPEOS (yylinept);
	return;
    }

    if (pass == 1)
    {
	if (result.constval <= 0)
	{
	    warning ("op_res: strange value for .space");
	}
	else
	{
	    dot -> svalue += result.constval;
	}
    }
    else
    {
	switch (dot -> sspace)
	{
#ifdef POOL
	    case S_Lconstant:
	    case S_Dconstant:
	    case S_Wconstant:
	    case S_Xconstant:
	    case S_Hconstant:
#else /* not POOL */
	    case S_constant:
#endif /* POOL */
	    case S_text:
	    case S_tex2:
	    case S_fartext:
	    case S_ftex2:
	    case S_fdat2:
	    case S_dat2:
	    case S_fardata:
	    case S_data:
		for (i = result.constval; --i >= 0;)
		{			/* fill in empty space */
		    emit_dbyte (0);
		}
		break;

	    default:
		dot -> svalue += result.constval;
		break;
	}
    }
}






/*
NAME: op_set
PURPOSE: handle the .set pseudo op.
PRECONDITIONS:
ptr is the usual optable pointer.
two operands are expected, although only the second is put into an OPND
structure. the first must be a symbol name. the second may be an address
or an expression yielding a constant value.
POSTCONDITIONS:
the symbol is defined by defsym if necessary and then its svalue field is
set to the value of the second operand.
the symbolname "." (dot) is allowed, but this is not always the best
usage.
HISTORY:
part of initial code DAW
*/






op_set(ptr)
struct opcode *ptr;
{
    register SYM *symp;
    extern Tuchar token[];		/* output from get_token */

    SKIPSPACE (yylinept);

    get_token ();			/* scan a symbol */

    if ((symp = defsym (token)) == 0)
    {
	SKIPEOS (yylinept);
	return;
    }

    SKIPSPACE (yylinept);

    if (*yylinept != ',')
    {
	error ("op_set: comma expected");
	SKIPEOS (yylinept);
	return;
    }
    ++yylinept;

    if (validate_op (ptr, 0) == 0)
    {
	SKIPEOS (yylinept);
	return;
    }

    switch (oper[0].type)
    {
	case CONSTANT:
	case IMMEDIATE:
	case MEMORY:
	    symp -> svalue = oper[0].constval;
	    if ((oper[0].haveseen & NAME) == 0)
	    {
		symp -> sflags |= S_CONST | S_DEF;
	    }
	    else if (oper[0].symname && (oper[0].symname -> sflags & S_DEF)) {
		symp -> svalue += oper[0].symname -> svalue;
		symp -> sflags |= oper[0].symname -> sflags;
		symp -> sspace = oper[0].symname -> sspace;
	    }
	    break;

	default:
	    if (pass == 2)
		error ("op_set: illegal operand");
	    break;
    }

    if (pass == 2 && (symp -> sflags & S_DEF) == 0)
    {
	error ("op_set: couldn't resolve %s", symp -> sname);
    }
}






/*
NAME: op_setmem
PURPOSE: handle the .byte, .half, .word, and .long pseudo ops.
PRECONDITIONS:
ptr is the usual optable pointer.
the ocode field of the optable defines which pseudo op we are handling.
one or more operands are expected, the last being determined by the lack
of a trailing comma.
POSTCONDITIONS:
space in the appropriate size is allocated for each operand in the
current space.
ALGORITHM:
floating point constants are handled by op_float, otherwise validate_op
is used. if relocation is necessary, emit_rel is called.
pass 1 actions differ significantly from those of pass 2 due to the code
emission that is done only in pass 2.
HISTORY:
part of initial code DAW
revision for negative offset handling RJM
*/






op_setmem(ptr)
register struct opcode *ptr;
{
    register int count = 0;
    register int r_len;			/* length for relocation */
    register int val;
#ifdef REP_COUNT
    int relflg;
    int (*emit_data)();
    extern int emit_dbyte(), emit_dhalf(), emit_dword(), emit_dlong();
#endif /* REP_COUNT */
#if defined(NEW_LIL)
    int reltype;
#endif /* NEW_LIL */

#ifdef DEBUG
    if(dbflag)
	printf ( "set_mem: length %d, line is %s\n", ptr -> ocode, yylinept);
#endif

    do_align (ptr -> ocode);		/* align the space */

    if (pass == 1)
    {
	for (;;)
	{				/* parse all parameters */
	    ++count;

	    SKIPSPACE (yylinept);

	    /*
	     *  temporary kludge to handle floating point constants
	     */
	    if (*yylinept == '0'
		&& (*(yylinept+1) == 'd'
		    || *(yylinept+1) == 'D'
		    || *(yylinept+1) == 'f'
		    || *(yylinept+1) == 'F'))
	    {
		op_float(ptr);		/* go process a floating point const */
	    }
	    else
	    {
		validate_op (ptr, 0);	/* need to define symbols */
	    }

#ifdef REP_COUNT
	    /*
	     *	kludge over kludge for repeat count - syntax: e1 @ e2
	    */

	    if (*yylinept == '@') {
		++yylinept;
		if (!yyparse()) {
		    if (result.type == MEMORY && result.haveseen == CONST
				&& result.constval >= 0)
			count += result.constval - 1;
		}
	    }

#endif /* REP_COUNT */
	    SKIPSPACE (yylinept);
	    if (*yylinept != ',')
		break;
	    ++yylinept;
	}

#ifdef DEBUG
	if(dbflag)
		printf ( "set_mem: pass1 %d items reserved\n", count);
#endif

	dot -> svalue += count * ptr -> ocode;
    }
    else
    {
	switch (ptr -> ocode)
	{				/* figure type for relocation */
	    case 1:
		r_len = R_BYTE;
#ifdef REP_COUNT
		emit_data = emit_dbyte;
#endif /* REP_COUNT */
		break;

	    case 2:
		r_len = R_HALFWORD;
#ifdef REP_COUNT
		emit_data = emit_dhalf;
#endif /* REP_COUNT */
		break;

	    case 4:
		r_len = R_WORD;
#ifdef REP_COUNT
		emit_data = emit_dword;
#endif /* REP_COUNT */
		break;

	    case 8:
		r_len = R_DOUBLEWORD;
#ifdef REP_COUNT
		emit_data = emit_dlong;
#endif /* REP_COUNT */
		break;

	    default:
		error ("op_setmem: bad length for %s", ptr -> oname);
		SKIPEOS (yylinept);
		return;
	}

	for (;;)
	{				/* do for each word on line */
	    SKIPSPACE (yylinept);

	    /*
	     * temporary kludge for floating point constants
	     */

	    if (*yylinept == '0'
		&& (*(yylinept+1) == 'd'
		    || *(yylinept+1) == 'D'
		    || *(yylinept+1) == 'f'
		    || *(yylinept+1) == 'F'))
	    {
		op_float(ptr);		/* go process a floating point const */
		SKIPSPACE (yylinept);
		if (*yylinept != ',')
		    break;
		++yylinept;
		continue;
	    }

#ifdef REP_COUNT
	    if (!validate_op (ptr, 0))
	    {				/* zero indicates error */
		SKIPEOS (yylinept);
		return;
	    }
	    val = oper[0].constval;
	    if (relflg = oper[0].haveseen & NAME) {
		/* we must relocate this one! */
#if defined(NEW_LIL)
		if (r_len == R_BYTE
			|| (r_len == R_HALFWORD & ! oper[0].ishalf)) {
#else /* NOT NEW_LIL */
		if (r_len == R_BYTE || r_len == R_HALFWORD) {
#endif /* NEW_LIL */
		 error ("op_setmem: relocatable symbol used in .byte or .half");
		 relflg = 0;
		}
	    }

	    count = 1;

	    if (*yylinept == '@') {
		++yylinept;
		if (!yyparse()) {
		    if (result.type != MEMORY || result.haveseen != CONST
				|| result.constval < 0)
			error("op_setmem: bad repeat count");
		    else
			count = result.constval;
		}
	    }

	    while (count-- > 0) {
#if defined(NEW_LIL)
		if (oper[0].ishalf == 1)
		    reltype = R_LO_ADDR;
		else if (oper[0].ishalf == 2) {
		    reltype = R_HI_ADDR;
		    if (relflg) {
			if (val > 0x7fff || val < ~0x7fff) {
			    error("op_setmem: R_HI_ADDR offset overflow");
			    val = (val & 0x7fff) | ((val >> 16) & 0x8000);
			}
		    }
		    else
			val = ((unsigned) val >> 16);
		}
		if (relflg) {
		    if (oper[0].ishalf) {
			dot -> svalue += ptr -> ocode - 2;
			val &= 0xffff;
			val = emit_rel (reltype, R_HALFWORD, oper[0].symname,
					val);
			dot -> svalue -= ptr -> ocode - 2;
		    }
		    else
			val = emit_rel (R_ADDRLITERAL, r_len, oper[0].symname,
				val);
		}
		else {
			if (curr_domain != 0
			    && curr_func_symp != 0
			    && oper[0].symname != 0
			    && dot->sspace == curr_func_symp->sspace
			    && r_len == R_WORD
			    && !strcmp(oper[0].symname->sname,
					curr_func_symp->sname)
			) {
			    val += (curr_domain << 16);
			}
#else /* NOT NEW_LIL */
		if (relflg)
		    val = emit_rel (R_ADDRLITERAL, r_len, oper[0].symname,
				val);
		else {
			if (curr_domain != 0
			    && curr_func_symp != 0
			    && oper[0].symname != 0
			    && dot->sspace == curr_func_symp->sspace
			    && r_len == R_WORD
			    && !strcmp(oper[0].symname->sname,
					curr_func_symp->sname)
			) {
			    val += (curr_domain << 16);
			}
		}
#endif /* NEW_LIL */
		emit_data(val);	/* put out the data */
	    }
#else /* not REP_COUNT */
	    if (validate_op (ptr, 0))
	    {				/* zero indicates error */
		val = oper[0].constval;
		if (oper[0].haveseen & NAME)
		{			/* we must relocate this one! */
#if defined(NEW_LIL)
		    if (r_len == R_BYTE
			|| (r_len == R_HALFWORD & ! oper[0].ishalf))
#else /* NOT NEW_LIL */
		    if (r_len == R_BYTE || r_len == R_HALFWORD)
#endif /* NEW_LIL */
		    {
	    error ("op_setmem: relocatable symbol used in .byte or .half");
		    }
#if defined(NEW_LIL)
		    else if (oper[0].ishalf) {
			if (oper[0].ishalf == 1)
			    reltype = R_LO_ADDR;
			else if (oper[0].ishalf == 2) {
			    reltype = R_HI_ADDR;
			    if (val > 0x7fff || val < ~0x7fff) {
				error("op_setmem: R_HI_ADDR offset overflow");
				val = (val & 0x7fff) | ((val >> 16) & 0x8000);
			    }
			}
			dot -> svalue += ptr -> ocode - 2;
			val &= 0xffff;
			val = emit_rel(reltype, R_HALFWORD, oper[0].symname,
					val);
			dot -> svalue -= ptr -> ocode - 2;
		    }
#endif /* NEW_LIL */
		    else
		    {
			if (curr_domain != 0
			    && curr_func_symp != 0
			    && oper[0].symname != 0
			    && dot->sspace == curr_func_symp->sspace
			    && r_len == R_WORD
			    && !strcmp(oper[0].symname->sname,
					curr_func_symp->sname)
			) {
			    val += (curr_domain << 16);
			}
			val = emit_rel (R_ADDRLITERAL, r_len,
			    oper[0].symname, val);
		    }
		}
#if defined(NEW_LIL)
		else if (oper[0].ishalf == 1)
		    val &= 0xffff;
		else if (oper[0].ishalf == 2)
		    val = ((unsigned) val >> 16);
#endif /* NEW_LIL */
	    }
	    else
	    {
		SKIPEOS (yylinept);
		return;
	    }

	    switch (r_len)
	    {				/* put out the data */
		case R_BYTE:
		    emit_dbyte (val);
		    break;

		case R_HALFWORD:
		    emit_dhalf (val);
		    break;

		case R_WORD:
		    emit_dword (val);
		    break;

		case R_DOUBLEWORD:
		    emit_dlong (val);
		    break;
	    }
#endif /* REP_COUNT */

	    SKIPSPACE (yylinept);
	    if (*yylinept != ',')
		break;			/* got to end of line */
	    ++yylinept;
	}
    }
}






/*
NAME: op_space
PURPOSE: handle the "change to a new space" pseudo ops.
PRECONDITIONS:
ptr is the usual optable pointer.
the ocode field of the optable entry contains the SEGS value of the
segment to change to.
only multi subsegment segments (eg data with data and data 2) should
have an operand. it should identify the subsegment (not necessary if
subsegment is the first one).
POSTCONDITIONS:
based on the value of the far_flag and the current space, a new current
space will be set using setseg.
HISTORY:
part of initial code DAW
*/






op_space(ptr)
struct opcode *ptr;
{
    int subseg = 1;		/* assume data/fardata 1 */
    int param = 0;		/* assume no subsegment parameter */
    enum SEGS seg;

    SKIPSPACE (yylinept);

    if (*yylinept
	&& *yylinept != ';'
	&& yyparse () == 0)
    {					/* there was some param */
	if (result.haveseen != CONST
		|| result.constval < 1
		|| result.constval > 2)
	{
	    error ("op_space: illegal operand");
	}
	else
	{
	    param = 1;
	    subseg = result.constval;
	}
    }

    seg = ptr -> ocode;

/*
*  the following should be fixed if many partitions of a
*  space are implemented (such as s_dat9 or s_bss423).
*  this would require some major structural changes however.
*/
    switch (seg)
    {					/* we might have to check param */
    case S_data:
	if (subseg == 2)
	    seg = S_dat2;
	break;
    case S_text:
	if (subseg == 2)
	    seg = S_tex2;
	break;
    case S_fardata:
	if (subseg == 2)
	    seg = S_fdat2;
	break;
    case S_fartext:
	if (subseg == 2)
	    seg = S_ftex2;
	break;
    default:
	if (param)
	{
	    error ("op_space: illegal operand");
	}
	break;
    }

/* note that far_flag can prevent a change from far to near space */
    if (far_flag)
    {
	switch (seg)
	{
	case S_text:
	    seg = S_fartext;
	    break;
	case S_tex2:
	    seg = S_ftex2;
	    break;
	case S_data:
	    seg = S_fardata;
	    break;
	case S_dat2:
	    seg = S_fdat2;
	    break;
	case S_bss:
	    seg = S_farbss;
	    break;
	}
    }

    /* only do it if we are actually changing to a new space */
    if (seg != dot -> sspace)
    {
	setseg (seg);
    }
}






/*
NAME: op_stamp
PURPOSE: handle the .stamp pseudo op
POSTCONDITIONS:
for now, this does nothing other than accept the directive and skip past
the statement.
the header stamps are currently set in post_pass2 of aout.c
HISTORY:
dummy function created RJM
Thu Sep 22 15:00:42 EDT 1983
*/






op_stamp(ptr)
struct opcode *ptr;
{
    SKIPEOS (yylinept);
}






/*
NAME: op_using
PURPOSE: handle the .drop and .using pseudo ops
POSTCONDITIONS:
.using currently uses op_base instead of this routine.
these ops may be used to put out optimization information in a later release.
for now, they do nothing other than accept the directives and move past
the statement.
HISTORY:
dummy function created RJM
Tue Sep  6 15:06:20 EDT 1983
*/






op_using(ptr)
struct opcode *ptr;
{
    SKIPEOS (yylinept);
}

/*
 * 	(c) Copyright 1987 Gould Inc.
 * 	    All Rights Reserved.
 */
