/*
 * xdm - display manager daemon
 *
 * $XConsortium: session.c,v 1.29 90/02/07 18:47:19 keith Exp $
 *
 * Copyright 1988 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of M.I.T. not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  M.I.T. makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * Author:  Keith Packard, MIT X Consortium
 */

/*
 * session.c
 */

# include "dm.h"
# include <X11/Xlib.h>
# include <signal.h>
# include <X11/Xatom.h>
# include <setjmp.h>
# include <sys/errno.h>
# include <stdio.h>

extern int  errno;
static jmp_buf abortSession;

static SIGVAL
catchTerm ()
{
	longjmp (abortSession, 1);
}


extern void	exit ();

/*
 * We need our own error handlers because we can't be sure what exit code Xlib
 * will use, and our Xlib does exit(1) which matches REMANAGE_DISPLAY, which
 * can cause a race condition leaving the display wedged.  We need to use
 * RESERVER_DISPLAY for IO errors, to ensure that the manager waits for the
 * server to terminate.  For other X errors, we should give up.
 */

static
IOErrorHandler (dpy)
    Display *dpy;
{
    extern char *sys_errlist[];
    extern int sys_nerr;
    char *s = ((errno >= 0 && errno < sys_nerr) ? sys_errlist[errno]
						: "unknown error");

    LogError("fatal IO error %d (%s)\n", errno, s);
    exit(RESERVER_DISPLAY);
}

static int
ErrorHandler(dpy, event)
    Display *dpy;
    XErrorEvent *event;
{
    LogError("X error\n");
    if (XmuPrintDefaultErrorMessage (dpy, event, stderr) == 0) return 0;
    exit(UNMANAGE_DISPLAY);
}

ManageSession (d)
     struct display	*d;
{
  Display		*dpy, *Init ();
  int			pid;

  Debug ("ManageSession %s\n", d->name);
  (void)XSetIOErrorHandler(IOErrorHandler);
  (void)XSetErrorHandler(ErrorHandler);

  SetTitle(d->name, (char *) 0);
  /*
   * Step 5: Load system default Resources
   */
  LoadXloginResources (d);
  Debug ("name now %s\n", d->name);
  ReadSessionHosts (d->sessionHostFile);
  dpy = Init (d);
  InitGreet (d);
  Greet (d);
  DeleteXloginResources (d, dpy);
  CloseGreet (d);
  Debug ("Greet loop finished\n");

  Debug ("Will try host \"%s\".\n", GetSessionHost ());
  if (!setjmp (abortSession)) {
    signal (SIGTERM, catchTerm);
    if (StartRemote (d)) {
      Debug ("Remote session started\n");
      AckRemote (d);
      while (!WaitForLogout (d))
	Debug ("restoring logout window on %s\n", d->name);
    } else {
      LogError ("session start failed\n");
    }
  }
  SessionExit (d, OBEYSESS_DISPLAY);
}

LoadXloginResources (d)
struct display	*d;
{
    char	cmd[1024];

    if (d->resources[0] && access (d->resources, 4) == 0) {
	if (d->authorization && d->authFile && d->authFile[0]) {
	    sprintf (cmd, "XAUTHORITY=%s %s -display %s -load %s",
			    d->authFile,
			    d->xrdb, d->name, d->resources);
	} else {
	    sprintf (cmd, "%s -display %s -load %s",
			    d->xrdb, d->name, d->resources);
	}
	Debug ("Loading resource file: %s\n", cmd);
	system (cmd);
    }
}

/*ARGSUSED*/
DeleteXloginResources (d, dpy)
struct display	*d;
Display		*dpy;
{
    XDeleteProperty(dpy, RootWindow (dpy, 0), XA_RESOURCE_MANAGER);
}

static jmp_buf syncJump;

static SIGVAL
syncTimeout ()
{
    longjmp (syncJump, 1);
}

SecureDisplay (d, dpy)
struct display	*d;
Display		*dpy;
{
    Debug ("SecureDisplay %s\n", d->name);
    signal (SIGALRM, syncTimeout);
    if (setjmp (syncJump)) {
	LogError ("WARNING: display %s could not be secured\n",
		   d->name);
	SessionExit (d, RESERVER_DISPLAY);
    }
    alarm ((unsigned) d->grabTimeout);
    Debug ("Before XGrabServer %s\n", d->name);
    XGrabServer (dpy);
    /*if (XGrabKeyboard (dpy, DefaultRootWindow (dpy), True, GrabModeAsync,
		       GrabModeAsync, CurrentTime) != GrabSuccess)
    {
	alarm (0);
	signal (SIGALRM, SIG_DFL);
	LogError ("WARNING: keyboard on display %s could not be secured\n",
		  d->name);
	SessionExit (d, RESERVER_DISPLAY);
    }
    Debug ("XGrabKeyboard succeeded %s\n", d->name);*/
    alarm (0);
    signal (SIGALRM, SIG_DFL);
    pseudoReset (dpy);
    if (!d->grabServer)
    {
	XUngrabServer (dpy);
	XSync (dpy, 0);
    }
    Debug ("done secure %s\n", d->name);
}

UnsecureDisplay (d, dpy)
struct display	*d;
Display		*dpy;
{
    Debug ("Unsecure display %s\n", d->name);
    if (d->grabServer)
	XUngrabServer (dpy);
    /*XUngrabKeyboard (dpy, CurrentTime);*/
    XSync (dpy, 0);
}

SessionExit (d, status)
    struct display  *d;
{
    /* make sure the server gets reset after the session is over */
    if (d->serverPid >= 2)
	kill (d->serverPid, SIGHUP);
    else
	ResetServer (d);
    exit (status);
}

static int
StartRemote (d)
     struct display *d;
{
/*	Invoke the session server on the requested host.
	Wait for the session server to report success in bringing up a
	session.
 */
  int t0;

  if (RequestRemoteSession (d)) {
    Debug ("Request successfully transmitted.\n");
    /* Request has been transmitted.  Wait for status return
       transmission. */
    t0 = time (0);
    /* t1 = t0; */
    do {
      int i;
      i = ReadRemoteAck (d);
      if (i < 0) return d->sessionStatus;
      if (i == 0) continue;
      if (i > 0) {
	do {
	  if (ReadRemoteStatus (d)) return d->sessionStatus;
	  /* if ((t1 - time (0)) > d->pingInterval) {
	     if (!PingServer (d, (Display *) NULL))
	     SessionExit (d, RESERVER_DISPLAY);
	     t1 = time (0);
	     }
	     */
	} while (time (0) < (t0 + 6000));
	break;
      }
      /* Keep trying for 10 seconds. */
    } while (time (0) < (t0 + 10));
    LogError ("Giving up on host \"%s\".\n", GetSessionHost ());
    FailRemote (d);
    /* Host apparently isn't up to finishing its message. */
  }
  LogError ("Host \"%s\" failed to respond.\n", GetSessionHost ());
  return 0;			/* Host would not talk. */
  /* Should put an error message window on screen, or set insensitive. */
}
