/* Copyright 1990 Network Computing Devices, Inc.  All rights reserved. */

#ident "@(#)pcftosnf.c	13.2	90/08/16"

/*
 * pcftosnf
 *
 * converter from DEC pcf format to NCD snf format
 *
 * Dave Lemke
 *
 */

#ifndef lint
#include "version.h"
#endif /* lint */

#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<errno.h>

#include	<X11/X.h>
#include	<X11/Xmd.h>
#include	<X11/Xproto.h>

#ifndef CONVERTER
#include	"dixfontstr.h"
#else
#include	"fontstruct.h"
#include	"snfstruct.h"
#endif

#include	"bdftosnf.h"

#include	"pcftosnf.h"

#include	"fontlib.h"
#include	"fontos.h"
#include	"fosfile.h"

#include	"pcf.h"
#include	"encode.h"

#ifdef CONVERTER
typedef struct _TempFont	*TempFontPtr;
#define	FONTPTR	TempFontPtr
#define	FONTREC	TempFont
#else
#define	FONTPTR	FontPtr
#define	FONTREC	FontRec
#endif

static int msb[8] = {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
static int lsb[8] = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80};

#ifdef DEBUG
#define assert(ex)	{if (!(ex)){(void)fprintf(stderr,"Assertion \"ex\" failed: file \"%s\", line %d\n", __FILE__, __LINE__);abort();}}
#else
#define assert(ex)
#endif

static void	do_file(), write_snf_font(), free_font();
static FONTPTR	convert_font();
static int	*read_pcf_indices();

Bool		force_flag = FALSE;
Bool		dump_names = FALSE;

extern char	*NameForAtom();

Encoding	encoding = (Encoding) 0;

#ifdef CONVERTER
#define	Xalloc(n)	calloc(n, 1)
#define	Xfree(n)	free(n)
#define	Xrealloc(p, n)	realloc((p), (n))

char	*program;

static union BitOrderCheck {
    long l;
    struct {
#ifndef STRUCT_BITS_FILL_FROM_LOW
        int b:1;                /* sign bit */
        int pad:31;
#else STRUCT_BITS_FILL_FROM_LOW
        int pad:31;
        int b:1;                /* sign bit */
#endif STRUCT_BITS_FILL_FROM_LOW
    } bitfields;
} BitOrderChecker;

usage()
{
	fprintf(stderr, "%s:  [-m] [-l] [-f] [-e encoding_file] [pcf files]\n", program);
	exit(0);
}

main(argc, argv)
int	argc;
char	**argv;
{
fosFilePtr	pFile;
BuildParamsPtr	params;
int	i;
int	fcnt = 0;

	InitAtoms();

	program = argv[0];
	params = fosNaturalParams;

	BitOrderChecker.l = 0;
	BitOrderChecker.bitfields.b = 1;
	if (BitOrderChecker.l >= 0) {
		fprintf(stderr, "Bit order within structures is incorrect.\n");
		fprintf(stderr, "Change definition of STRUCT_BITS_FILL_FROM_LOW,\n");
		fprintf(stderr, "and recompile %s\n", argv[0]);
		exit(1);
	}

	for (i = 1; i < argc; i++)	{
		if (argv[i][0] == '-')	{
			switch (argv[i][1])	{
			case 'd':
				dump_names = TRUE;
				break;
			case 'f':
				force_flag = TRUE;
				break;
			case 'e':
				if (!argv[i+1])
					usage();
				/* try for the internal first, then the
				 * file name */
				encoding = find_encoding(argv[i+1]);
				if (!encoding)	{
					encoding = read_enc_file(argv[i + 1]);
					if (!encoding)	{
						printf("unknown encoding %s\n", 
								argv[i + 1]);
						usage();
					}
				}
				i++;
				break;
			case 'L':
				if (argv[i][2] != '\0')
					usage();
				params->byteOrder = LSBFirst;
				break;
			case 'M':
				if (argv[i][2] != '\0')
					usage();
				params->byteOrder = MSBFirst;
				break;
			case 'l':
				if (argv[i][2] != '\0')
					usage();
				params->bitOrder = LSBFirst;
				break;
			case 'm':
				if (argv[i][2] != '\0')
					usage();
				params->bitOrder = MSBFirst;
				break;
			default:
				usage();
			}
		} else	{
			fcnt++;
			pFile = fosOpenFile(argv[i], "r");
			if (pFile == NULL)	{
				fprintf(stderr, "%s: \n", program);
				perror(argv[i]);
			} else	{
				do_file(pFile, params, argv[i]);
				fosCloseFile(pFile, TRUE);
			}
		}
	}
	if (fcnt == 0)
		fprintf(stderr, "No fonts specfied\n");
	exit (0);
}
#endif CONVERTER

static void
do_file(pFile, params, name)
fosFilePtr	pFile;
BuildParamsPtr	params;
char		*name;
{
EncodedFontPtr	font = NullEncodedFont;
FONTPTR		newfont;

	pcfReadFont(pFile, &font, NullCharSet, FONT_EVERYTHING, params);
	if (font == NullEncodedFont)	{
		fprintf(stderr, "%s:  format failure for '%s'\n", 
			program, name);
		return;
	}
	if (dump_names)	{
		dump_pcf_names(font->pCS);
		goto dropout;
	}
	newfont = convert_font(font, params, name);
	if (!newfont)	{
		fprintf(stderr, "%s:  conversion failure for '%s'\n", 
			program, name);
		goto dropout;
	}
#ifdef	CONVERTER
	write_snf_font(newfont, name);
#endif	CONVERTER
	free_font(newfont);

dropout:
	fontFree(font);
}

static void
free_font(pfont)
FONTPTR	pfont;
{
	(void)Xfree(pfont->pFP);
	(void)Xfree(pfont->pGlyphs);
	if (pfont->pFI->inkMetrics)
		(void)Xfree(pfont->pInkCI);
	(void)Xfree(pfont);
}

static FONTPTR
convert_font(pcf_font, params, fname)
EncodedFontPtr	pcf_font;
BuildParamsPtr	params;
char	*fname;
{
FONTPTR	pfont;
char	*fontspace, *propspace;
FontInfoRec	fi;
CharSetPtr	pCS = pcf_font->pCS;
int	bytestoalloc, bytestoink;
pcfFontPropPtr	props = pCS->props;
int		i;
#ifdef	CONVERTER
FontPropPtr	pffp;
#else
DIXFontProp	*pdfp;
#endif	CONVERTER

	
	if (!encoding)	{
		encoding = find_encoding("UNKNOWN");
		if (!encoding)	{
			printf("can't find encoding UNKNOWN\n");
			exit(0);
		}
	}

	fill_font_info(&fi, pcf_font);

	bytestoalloc = sizeof(FONTREC);
	bytestoalloc += BYTESOFFONTINFO(&fi);
	bytestoalloc += BYTESOFCHARINFO(&fi);
	bytestoink = bytestoalloc;
	if (fi.inkMetrics)
		bytestoalloc += BYTESOFINKINFO(&fi);

	fontspace = (char *) Xalloc(bytestoalloc);

	if (!fontspace)
		return (FONTPTR) 0;

	pfont = (FONTPTR) fontspace;

	pfont->pFI = (FontInfoPtr) (fontspace + sizeof(FONTREC));

	*pfont->pFI = fi;	/* copy font info */

	pfont->pCI = ADDRCharInfoRec(pfont->pFI);
#ifdef CONVERTER
	pfont->pFP = (FontPropPtr) Xalloc(fi.nProps * sizeof (FontPropRec));
#else CONVERTER
	pfont->pFP = (DIXFontPropPtr) Xalloc(fi.nProps * sizeof (DIXFontProp));
#endif CONVERTER
	if (pfont->pFI->inkMetrics)	{
		pfont->pInkMin = (CharInfoPtr) (fontspace + bytestoink);
		pfont->pInkMax = pfont->pInkMin + 1;
		pfont->pInkCI = pfont->pInkMax + 1;
	} else	{
		pfont->pInkMin = &pfont->pFI->minbounds;
		pfont->pInkMax = &pfont->pFI->maxbounds;
		pfont->pInkCI = pfont->pCI;
	}

	/* convert the char info */
	if (copy_bitmaps(pfont, pcf_font, params, fname) < 0)	{
		(void) Xfree(pfont->pFP);
		(void) Xfree(pfont);
		return (FONTPTR) 0;
	}

#ifndef CONVERTER
	/* read and convert the props */
/* XXX -- this is done by WriteNFont() as well */
	fi.lenStrings = 0;
	for (i = 0; i < pCS->nProps; i++)	{
		fi.lenStrings += strlen(NameForAtom(props[i].name)) + 1;
#ifdef DEBUG
printf("name: %s (%d)\n", NameForAtom(props[i].name), props[i].name);
#endif
		if (pCS->isStringProp[i])	{
			fi.lenStrings += 
				strlen(NameForAtom(props[i].value)) + 1;
#ifdef DEBUG
printf("value: %s (%d)\n", NameForAtom(props[i].value), props[i].value);
		} else	{
printf("value: %d\n", props[i].value);
#endif
		}
	}
#endif !CONVERTER

#ifdef	CONVERTER
	pffp = pfont->pFP;
#else	CONVERTER
	pdfp = pfont->pFP;
#endif	CONVERTER

	for (i = 0; i < pCS->nProps; i++)	{
#ifdef	CONVERTER
		pffp->indirect = pCS->isStringProp[i];
		pffp->name = props[i].name;
		pffp->value = props[i].value;

		pffp++;
#else	CONVERTER
		pdfp->name = props[i].name;
		pdfp->value = props[i].value;
		pdfp++;
#endif	CONVERTER
	}

	/* now reset the accelerators to what they sould be */
	computeNaccelerators(pfont, 0, 0, params->glyphPad);

	return pfont;
}

dump_pcf_names(pCS)
CharSetPtr	pCS;
{
Atom	*name = pCS->glyphNames;

	while (*name != None)	{
		printf("Name: %s\n", NameForAtom(*name));
		name++;
	}
}

find_char(atom, pCS)
Atom	atom;
CharSetPtr	pCS;
{
Atom	*name = pCS->glyphNames;
int	i = 0;
	
	while (*name != None)	{
		if (*name == atom)
			return i;
		i++, name++;
	}
	return -1;
}

copy_bitmaps(pfont, pcf_font, params, fname)
FONTPTR		pfont;
EncodedFontPtr	pcf_font;
BuildParamsPtr	params;
char	*fname;
{
int	ch;
int	ndx;
int	r;
int	b;
CharInfoPtr	pCI = pfont->pCI;
CharSetPtr	pCS = pcf_font->pCS;
CharInfoPtr	pCI_src = pCS->ci.pCI;
int	*masks;
unsigned int	bytesGlUsed = 0;
Atom	atom;
int	bytesGlAlloced = 1024;
unsigned char	*pGI = (unsigned char *)Xalloc((unsigned)bytesGlAlloced);
int	*indices = read_pcf_indices(fname);
int	*cur_index;


	cur_index = indices;
	pfont->pFI->minbounds.byteOffset = bytesGlUsed;
	masks = (params->bitOrder == MSBFirst ? msb : lsb);
/* XXXX -- this is extremely inefficient */
	for (ch = pfont->pFI->firstCol; 
		ch < n2dChars(pfont->pFI) + pfont->pFI->firstCol;
		ch++, pCI++) {

		int bpr;

		/* ignore encoding translations if 'unknown' encoding */
		if (encoding->num_encs != 0)	{
			atom = index_to_atom(encoding, ch);
			if (atom == 0)		/* skip non-existent chars */
				continue;

			/* find the character for this slot in our encoding */
			ndx = find_char(atom, pCS);
			if (ndx < 0)	{
				if (!force_flag)
					printf("Fatal ");
				printf("error -- encoding mismatch for %s\n",
					NameForAtom(atom));
				if (!force_flag)	{
					return -1;
				}
				ndx = ch;
			}
		} else	{
			if (indices)	{
				if ((ndx = *cur_index++) == -1)	{
					continue;
				}
			} else	{
				ndx = ch - pfont->pFI->firstCol;
				/* don't run off the end if it lies about
				 * the columns (decw$session) */
				if (ndx >= pcf_font->pCS->nChars)
					goto all_done;
			}
		}

		pCI->byteOffset = bytesGlUsed;
		pCI_src = &pCS->ci.pCI[ndx];

		/* copy charinfo */
		bcopy(&pCI_src->metrics, &pCI->metrics, sizeof(xCharInfo));

		pCI->exists = TRUE;

		bpr = BYTES_PER_ROW(GLYPHWIDTHPIXELS(pCI_src), 
							params->glyphPad);
		for (r = 0;
			r < pCI_src->metrics.descent + pCI_src->metrics.ascent;
			r++)	{
			unsigned char	*row, *row2;

			/* make sure there's enough room */
			if ((bytesGlUsed + bpr) >= bytesGlAlloced)	{
				bytesGlAlloced = (bytesGlUsed + bpr) * 2;
				pGI = (unsigned char *)Xrealloc((char *)pGI,
						(unsigned) bytesGlAlloced);
				/* clear them out */
				bzero(pGI+bytesGlUsed, 
					bytesGlAlloced - bytesGlUsed);
			}
			row = ((unsigned char*)
				pcf_font->pCS->pBitOffsets[ndx]) + (r*bpr);
			row2 = pGI + bytesGlUsed;

			for (b = 0;
				b < pCI_src->metrics.rightSideBearing - 
				pCI_src->metrics.leftSideBearing; 
				b++)	{
					row2[b >> 3]  |= row[b >> 3] & 
						masks[b&7];
			}
			bytesGlUsed += bpr;
		}
	}
all_done:
	pfont->pFI->maxbounds.byteOffset = bytesGlUsed;
	pfont->pGlyphs = pGI;
	(void) Xfree(indices);

	return 1;
}

fill_font_info(pfi, pcf_font)
FontInfoPtr	pfi;
EncodedFontPtr	pcf_font;
{
CharSetPtr	pCS = pcf_font->pCS;

	pfi->version1 = FONT_FILE_VERSION;
	pfi->version2 = FONT_FILE_VERSION;

	pfi->noOverlap = pCS->noOverlap;
	pfi->constantMetrics = pCS->constantMetrics;
	pfi->terminalFont = pCS->terminalFont;
	pfi->linear = pcf_font->linear;
	pfi->constantWidth = pCS->constantWidth;
	pfi->inkInside = pCS->inkInside;
	pfi->inkMetrics = pCS->inkMetrics;
	pfi->drawDirection = pCS->drawDirection;

	pfi->firstCol = pcf_font->firstCol;
	pfi->lastCol = pcf_font->lastCol;
	pfi->firstRow = pcf_font->firstRow;
	pfi->lastRow = pcf_font->lastRow;

	pfi->nProps = pCS->nProps;

	pfi->chDefault = pcf_font->defaultCh;
	pfi->fontDescent = pCS->fontDescent;
	pfi->fontAscent = pCS->fontAscent;

	pfi->glyphSets = 1;
}

static void
write_snf_font(pfont, name)
FONTPTR	pfont;
char	*name;
{
FILE	*fp;
int	len = strlen(name);

	/* XXX -- check file name */
	assert (name[len-3] == 'p' && name[len-2] == 'c' && name[len-1] == 'f');
	name[len-3] = 's';
	name[len-2] = 'n';
	if ((fp = fopen(name, "w")) == NULL)	{
		perror(name);
		return;
	}
	WriteNFont(fp, pfont, NameForAtom);
	(void) fclose(fp);
}

#include	"fosfilestr.h"
#include	"pcfint.h"

static int	*
read_pcf_indices(name)
char	*name;
{
fosFilePtr	fp;
int	nt, i;
TableDesc	toc[NUM_FONT_TABLES];
short	fc, lc, fr, lr;
int	nchars;
Mask	format;
int	*indices;
int	position;
int	skip = 0;

	if ((fp = fosOpenFile(name, "r")) == NULL)	{
		perror(name);
		return (int *) 0;
	}

	/* read version */
	fosReadLSB32(fp);

	/* read tables */
	nt = fosReadLSB32(fp);
	for (i = 0; i < nt; i++)	{
		toc[i].label = fosReadLSB32(fp);
		toc[i].format = fosReadLSB32(fp);
		toc[i].size = fosReadLSB32(fp);
		toc[i].start = fosReadLSB32(fp);
	}

	/* skip to the proper place */
	for (i = 0; i < nt; i++)	{
		assert(fosFilePosition(fp) == toc[i].start);
		if (toc[i].label == FONT_BDF_ENCODINGS)	{
			break;
		} else	{
			fosSkip(fp, toc[i].size);
		}
	}

	if (toc[i].label != FONT_BDF_ENCODINGS)	{
		printf("no encoding for %s\n", name);
		fosCloseFile(fp, TRUE);
		exit(0);
	}

	format = fosReadLSB32(fp);
	pcfSetByteOrder(fp, FMT_BYTE_ORDER(format));
	fc = pcfReadInt16(fp);
	lc = pcfReadInt16(fp);
	fr = pcfReadInt16(fp);
	lr = pcfReadInt16(fp);
	pcfReadInt16(fp);	/* skip default char */

	nchars = (lc - fc + 1) * (lr - fr + 1);

	indices = (int *) calloc(nchars + 1, (sizeof (int)));


	for (i = 0; i < nchars; i++)	{
		indices[i] = pcfReadInt16(fp);
#ifdef undef
#ifdef DEBUG
		printf("index @ %d is %d\n", i, indices[i]);
#endif
#endif
	}

	fosCloseFile(fp, TRUE);

	return indices;
}
