comment
***************************************************************************
**        Sketch -- a drawing program -- Written for Microsphere Inc.    **
**                                                                       **
***       By DmC                                                        ***
****               To Compile Enter SBASIC Sketch                      ****
*****                                                                 *****
***************************************************************************
end of comment

$lines

$constant	true = 0ffffh
$constant	false = 0000h

$constant	KNULL = 0Fh
$constant 	KUP = 38h
$constant	KDOWN = 32h
$constant	KLEFT = 34h
$constant	KRIGHT = 36h
$constant	RIGHTUP = 39h
$constant	LEFTUP = 37h
$constant	LEFTDOWN = 31h
$constant	RIGHTDOWN = 33h

var forgrnd, row_mode, direction, joy_stick_on, xpos, ypos, single_pass, \
	altrestart, loop, xtemp, ytemp, speed, plotting, keyhit = integer


$constant VDP  = 0FFFFh
$constant PIO = 0

$constant CLEAR = 0
$constant BLACK = 1
$constant GREEN = 2
$constant LTGREEN = 3
$constant BLUE = 4
$constant LTBLUE = 5
$constant DKRED = 6
$constant CYAN = 7
$constant RED = 8
$constant LTRED = 9
$constant YELLOW = 10
$constant LTYELLOW = 11
$constant DKGREEN = 12
$constant MAGENTA = 13
$constant GRAY = 14
$constant WHITE = 15

$constant READ_ADDR = 1
$constant WRITE_ADDR = 0
$constant READRAM = 1
$constant WRITERAM = 0

$constant JOY_PORT = 30
$constant JOY_CONTROL = 31

$constant VDPDATA = 30		: REM  VDPDATA = 10 for BigBoard
$constant VDPCNTL = 31		: REM  VDPCNTL = 11 for BigBoard	
$constant SYSPORT = 1ch
$constant KEYPORT = 1eh
$constant BITPORT = 0Ah		: REM same as sysport for BigBoard
$constant BITCNTL = 0Bh		: REM bitport + 1

$constant PGTBASE = 0
$constant SGTBASE = 1800h
$constant PNTBASE = 1c00h
$constant SATBASE = 1f00h
$constant PCTBASE = 2000h

$constant PGTLENGTH = 1800h
$constant SGTLENGTH = 0400h
$constant PNTLENGTH = 0300h
$constant SATLENGTH = 0100h
$constant PCTLENGTH = 1800h

$constant RAMSTART  = 0
$constant RAMLENGTH = 3800h

$constant COLORPORT = 87h

comment
***************************************************************************
***  VDPRAM ........... This is the standard fare for the Color Board   ***
***          accepts an address in video RAM and a mode (read or write) ***
***          from the caller and sets up the VDP chip to perform the    ***
***          requested operation to the requested address               ***
***************************************************************************
end of comment


procedure vdpram ( address, mode = integer )
     var read_mask, write_mask = integer
     read_mask = 03fh
     write_mask = 040h
     out VDPCNTL, address
     if ( mode ) then
	  out VDPCNTL, (address / 100h) and read_mask
     else
	  out VDPCNTL, (address / 100h) or write_mask
end of vdpram

comment
***************************************************************************
*** This procedure moves a sprite ( 0-31 ) to the position requested by ***
*** caller............................................................. *** 
*** call by move_sprite sprite#, x-position, y-position                 ***
***                  sprite number should be  0-31                      ***
***                  xpos and ypos should be valid moves                ***
***************************************************************************
end of comment

procedure move_sprite ( sprite, posx, posy = integer )
	vdpram SATBASE + ( sprite * 4 ), WRITERAM
	out VDPDATA, posy
	out VDPDATA, posx
end of move_sprite

comment
***************************************************************************
*** Sleep is used to insert a delay into a loop or program ............ ***
***       The larger the TIME number the longer the delay               ***
*************************************************************************** 
end of comment

procedure sleep ( time = integer )
     var i, j = integer
     for i = 0 to time
	  for j = 0 to 50
	  next j
     next i
end of sleep

comment
***************************************************************************
*** This routine places the cursor to the x,y position requested by     ***
*** caller...... 0,0 is upper left corner of the Kaypro screen and      ***
*** 79,23 is lower right hand corner                                    ***
***************************************************************************
end of comment 

procedure cursor ( x, y = integer )
     print chr(27);
     print "=";
     print chr(y + 32);
     print chr(x + 32);
end

REM ***** The next procedure clears the Kaypro screen *********************

procedure page
     print chr(26);
end

REM ***** This procedure clears the requested line on the Kaypro screen ***

procedure clearline ( line = integer )
	cursor 0, line
	print chr(18h);
end


comment
***************************************************************************
*** This procedure sets up a menu of options for the Sketch main loop   ***
***************************************************************************
end of comment

procedure set_up_menu

page
cursor 0,0
print "   <<<< SKETCH >>>>   Pro Color Graphics Drawing Program    DmC ";
cursor 0,2
print "              Commands                                       COLORS"
print "d -- pen down            u -- pen up                       0    clear"
print "j -- joy stick mode      k -- key board mode               1    black"
print "a -- set box mode        z -- set row mode                 2    green"
print "c -- color               b -- plot background pixels       3 lt green"
print "s -- set repeat speed    f -- plot foreground pixels       4     blue"
print "m -- alternate menu      r -- refresh this screen          5  lt blue"
print "          q -- quit sketch and return to CP/M              6   dk red"
print "                                                           7     cyan"
print "                                                           8      red"
print "           8                                               9   lt red"
print "      7    ^    9             -- STATUS --------------    10   yellow"
print "        \  |  /               Color Mode >                11 ltyellow"
print "   4<---- MOVE --->6          Joy Stick  >                12 dk green"
print "       /   |   \              Pixels     >                13  magenta"
print "      1    v     3            Pen        >                14     gray"
print "           2                  Speed      >                15    white"

	cursor 10, 23
	print "Pen position x, y ===============================>";
end of set_up_menu

comment
***************************************************************************
*** This procedure selects between the VDP chip and the PIO chip which  ***
*** reside at the same port address ..... The PIO 'b' side was unused   ***
*** on the Kaypro but we use it for the joystick                        ***
***************************************************************************
end of comment

procedure port_select ( flag = integer )
     var on_mask, off_mask, status = integer
     on_mask = 010h
     off_mask = 0efh
	
     status = inp( BITPORT )
     if ( flag ) then
	  out BITPORT, status or on_mask
     else
	  out BITPORT, status and off_mask
end of port_select


comment
***************************************************************************
*** This program gets a character from the keyboard by calling BDOS     ***
***************************************************************************
end of comment

function keyboard = integer
     var hl, de, bc, af = integer
     hl = 0
     de = 0ffh
     bc = 6
     af = 0
     call ( 5, hl, de, bc, af )
end = af / 100h


comment
***************************************************************************
*** This function does a MOD on an integer                              ***
***************************************************************************
end of comment  

function mod ( x, y = integer ) = integer
end = x - (x / y) * y


REM *** This function returns lower case equivalents for upper case *******
REM ***      letters                                                *******

function tolower ( cc = char ) = char
var d = char
	if ( cc >= 'A' and cc <= 'Z' ) then
		d = cc - ( 'A' - 'a' )
		else d = cc
end = d


comment
***************************************************************************
***  This procedure plots individual pixels. Or it can unplot (which is ***
***  the same as plotting background pixels) depending on the state of  ***
***  the global variable 'forgrnd'                                      ***
***************************************************************************
end of comment  

procedure plot ( x, y = integer ) : REM  bullet-proof version
     var bit, pixaddr = integer
	  bit = 2 ^ (7 - mod(x, 8))
	  pixaddr = PGTBASE + 8 * (x / 8 + 32 * (y / 8)) + mod(y, 8)
	  vdpram pixaddr, READ_ADDR
	  x = inp( VDPDATA )
	  if forgrnd THEN x = x or bit
			ELSE x = x AND ( NOT BIT )
	  vdpram pixaddr, WRITE_ADDR
	  out VDPDATA, x
end of plot

comment
***************************************************************************
*** This procedure updates the status portion of the menu to reflect    ***
*** the present values of the tested global variables                   ***
***************************************************************************
end of comment

procedure update_status

	cursor 42, 14
	if row_mode THEN print " row      ";
		    ELSE print " box      ";
	cursor 42, 15
	if joy_stick_on THEN print " on       ";
		      ELSE print " off      ";
	cursor 42, 16
	if forgrnd THEN print " foreground";
		   ELSE print " background";
	cursor 42, 17
	if plotting THEN print " down      ";
		    ELSE print " up        ";

	cursor 42, 18
	print " "; speed;"   ";

end of update_status


comment
***************************************************************************
***  This procedure reads the joy stick and returns a direction to      ***
***  Sketch........................................................     ***
***************************************************************************
end of comment

procedure read_stick
   var  temp, plotstrip, plotmask, direct, plot_flag = integer

	plotstrip = 00Fh
	plotmask = 010h
	port_select PIO
	direct = inp( JOY_PORT )

	REM ******* first find out if fire button is depressed ************
	REM *******       and set plotting true if it is       ************

			temp = direct
			temp = temp AND plotmask
			if temp = 0 then plot_flag = true
				ELSE plot_flag = false
			if plot_flag <> plotting THEN
				begin
					plotting = plot_flag
					update_status
				end

	REM ********      now remove fire button bit     ******************

	direct = direct AND plotstrip

	REM ***************** and get direction ***************************

	case direct of

		0eh	:	direction = KUP
		0dh	:	direction = KDOWN
		0bh	:	direction = KLEFT
		07h	:	direction = KRIGHT
		06h	:	direction = RIGHTUP
		0ah	:	direction = LEFTUP
		05h	:	direction = RIGHTDOWN
		09h	:	direction = LEFTDOWN		
		0fh	:	direction = KNULL
	end of case
	port_select VDP

end of read_stick


comment
***************************************************************************
***  This procedure initializes the PIO to be used for the Joy Stick    ***
***     It need be only called once                                     ***
***************************************************************************
end of comment

procedure set_stick_port

	port_select PIO
	out JOY_CONTROL, 0cfh
	out JOY_CONTROL, 01fh
	out JOY_PORT, 0E0h
	port_select VDP

end of set_stick_port


comment
***************************************************************************
***   This procedure updates the x,y position portion of the menu       ***
***************************************************************************
end of comment

procedure display_position
	IF ( xtemp <> xpos ) or ( ytemp <> ypos ) THEN
	begin
		cursor 60, 23
		print "             ";
		cursor 60, 23
		print xpos;"  ";ypos;"  ";
		xtemp = xpos
		ytemp = ypos
	end
end of display_position

comment
***************************************************************************
*** These five procedures move the pen in the direction set by the      ***
***  User and kept in the global variable 'direction'                   ***
***************************************************************************
end of comment

procedure go_up
	IF ypos > 0 THEN ypos = ypos - 1
	direction = KUP
end of go_up

procedure go_down
	IF ypos < 191 THEN ypos = ypos + 1
	direction = KDOWN
end of go_down

procedure go_left
	IF xpos > 0 THEN xpos = xpos - 1
	direction = KLEFT
end of go_left

procedure go_right
	IF xpos < 255 THEN xpos = xpos + 1
	direction = KRIGHT
end of go_right


procedure move_pen

	case direction of
		KUP:	go_up
		KDOWN:	go_down
		KRIGHT:	go_right
		KLEFT:	go_left
		LEFTUP:	begin
				go_up
				go_left
			end
		RIGHTUP:	begin
				go_up
				go_right
				end
		LEFTDOWN:	begin
				go_down
				go_left
				end
		RIGHTDOWN:	begin
				go_down
				go_right
				end
	end of case

end of move_pen

REM ****** end of pen moving procedures ***********************************



comment
***************************************************************************
*** This procedure allows coloring of pixels at the area selected by    ***
***  the pen .... Two modes are supported ... In the box mode the       ***
***   entire screen box (8x8 pixels) is colored ... In the row mode     ***
***    any number of rows (8 pixels horizontally) from one to the       ***
***     entire screen may be colored                                    ***
***************************************************************************
end of comment

procedure colors

	var selecting, fore_color, back_color, scrn_box, box_color, \
		last_box, box_row, first_row, last_row = integer
 

REM **** First do ROW mode code ******************************************* 

	IF row_mode THEN
		begin
		 cursor 8,20
		 print "Select by moving pen to row and putting pen down";
		 cursor 8,21
		 print "Select first row ..";
		 plotting = false
		 update_status 
		 selecting = true
		 While selecting do begin

		     IF ( joy_stick_on ) then
				read_stick
					else
					  direction = keyboard()
			if ( direction =  'd' or direction = 'D' ) then
					plotting = true
			move_pen
			move_sprite 0, xpos, ypos
			display_position
			IF plotting THEN
				begin
				selecting = false
				scrn_box =  xpos/8 + 32 * ( ypos/8 )
				first_row = (scrn_box * 8) + ( MOD(ypos, 8)) 
				end
		 end of while
	
		 cursor 30, 21
		 print "       Select last row ..";
		 sleep 30
		 plotting = false
		 selecting = true
		 While selecting do begin
			if ( joy_stick_on ) then
				 	read_stick
				else
					direction = keyboard()
			if ( direction = 'd' or direction = 'D' ) then
					plotting = true 
			move_pen
			move_sprite 0, xpos, ypos 
			display_position
			IF plotting THEN
				begin
				selecting = false
				scrn_box = xpos/8 + 32 * ( ypos/8 )
				last_row = (scrn_box * 8) + ( MOD(ypos, 8))
				end
			 end of while

		 clearline 20
		 clearline 21

		 plotting = false
		 update_status
		 cursor 0, 21
		 input3 "Foreground color please ? "; fore_color;
		 input3 "       Background color please ? "; back_color;
		 box_color = (fore_color*16) + back_color
 
		vdpram PCTBASE + first_row, WRITERAM
			 for box_row = first_row to last_row
				 out VDPDATA, box_color
			 next box_row
		 clearline 21

	end of if

REM ***** Now do BOX mode *************************************************

IF ( NOT row_mode ) then
	       begin
		 first_row = 1
		 last_row = 8
		 cursor 0, 21
		 input3 "Foreground color please ? "; fore_color
		 input3 "       Background color please ? "; back_color; 

		 box_color = ( fore_color*16 ) + back_color
 	 	 scrn_box = xpos/8 + 32 * (ypos/8)
		 vdpram PCTBASE + (8 * scrn_box), WRITERAM

		 FOR box_row = first_row to last_row
			out vdpdata, box_color
		 NEXT box_row
		 clearline 21
	      end

end of colors


REM **** QUIT checks for exit confirmation then clears screen and boots ***  

procedure quit
	var answer = char
	clearline 20
	clearline 21
	cursor 10, 20
	input3 "Quit (y or n)? "; answer;
	IF answer = 'y' then
		begin
			page
			stop
		end
end of quit

comment
***************************************************************************
*** Set Speed sets the length of delay in the joystick reading loop and ***
*** thereby sets joystick speed ....................................    ***
***************************************************************************
end of comment

procedure set_speed
		clearline 20
		clearline 21
	cursor 8, 21
	input3 "New repeat speed ?"; speed;
	clearline 21
	update_status
end of set_speed


REM *** This procedure sets up the joy stick port as a bit port ***********
REM ******   Needs to run only once  **************************************

procedure set_bit_port
	out SYSPORT + 1, 0cfh
	out SYSPORT + 1, 008h
end of set_bit_port


comment
***************************************************************************
***  MAIN_LOOP used to be the main loop of this program. Well, the menu ***
***   kept getting more and more crowded. I needed another menu and I   ***
***    I wanted to enter it first. But I also wanted to do some other   ***
***     things that were defined in this program. The only solution     ***
***     (since sbasic does not support forward declarations was to put  ***
***     the old main loop in a procedure that would be called by a new  ***
***    main loop and then this procedure could be entered at any time   ***
***************************************************************************
end of comment

procedure main_loop

While ( loop ) do begin

IF ( NOT joy_stick_on ) OR ( direction = KNULL ) THEN

	begin

	keyhit = keyboard ()
	case keyhit of
		'j'	:	begin
					joy_stick_on = true
					update_status
				end
		'k'	:	begin
					joy_stick_on = false
					update_status
				end
		'u'	:	begin
					plotting = false
					update_status
				end
		'd'	:	begin
					plotting = true
					update_status
				end
		'c'	:	colors
		's'	:	set_speed
		'a'	:	begin
					row_mode = false
					update_status
				end
		'z'	:	begin
					row_mode = true 
					update_status
				end
		'f'	:	begin
					forgrnd = true
					update_status
				end
		'b'	:	begin
					forgrnd = false
					update_status
				end
		'r'	:	begin
					set_up_menu
					update_status
					xtemp = 0
					display_position
				end
		'm'	:	begin
					loop = false
					altrestart = true
				end
		KUP	:	begin
					go_up
					display_position
				end
		KDOWN	:	begin
					go_down
					display_position
				end 
		KLEFT	:	begin
					go_left
					display_position
				end
		KRIGHT	:	begin
					go_right
					display_position
				end
		RIGHTDOWN:	BEGIN
					GO_RIGHT
					GO_DOWN
					display_position
				END
		RIGHTUP:	BEGIN
					GO_UP
					GO_RIGHT
					display_position
				END
		LEFTUP:		BEGIN
					GO_UP
					GO_LEFT
					display_position
				END
		LEFTDOWN:	BEGIN
					GO_left
					GO_DOWN
					display_position
				END 
		'Q'	:	quit
		'q'	:	quit
	end of case

	move_sprite 0,xpos,ypos
	if plotting then plot xpos, ypos
end

	if ( joy_stick_on ) then
		begin
			read_stick
			move_pen
			if plotting then plot xpos, ypos
			sleep speed
			IF direction = KNULL then
			display_position
		end 


   if single_pass then loop = false
 end of while
end of main_loop



comment
***************************************************************************
*** This is the other menu. It is entered first when running sketch and ***
*** provides initialization choices, total screen coloring, fetching of ***
*** a blank pad (SKETCH.SCR) etc....................................... ***
*** There is still plenty of room on the menu for ??................... ***
***************************************************************************
end of comment

procedure alternate_menu
	var key = integer


comment
***************************************************************************
*** The first sub procedure prints the menu on the screen.............. ***
***************************************************************************
end of comment

procedure set_up_alt

page
cursor 0,0
print "   <<<< SKETCH >>>>   Pro Color Graphics Drawing Program    DmC ";
cursor 0,2
print "---------------------  OPTIONS  ----------------------       COLORS"
print "e -- enter SKETCH             r -- return to SKETCH        0    clear"
print "b -- set backplane color      p -- change pen color        1    black"
print "g -- get blank sketch pad     c -- color entire pad        2    green"
print "n -- refresh this menu                                     3 lt green"
print "                                                           4     blue"
print "                                                           5  lt blue"
print "                                                           6   dk red"
print "                                                           7     cyan"
print "                                                           8      red"
print "                                                           9   lt red"
print "                                                          10   yellow"
print "                                                          11 ltyellow"
print "                                                          12 dk green"
print "                                                          13  magenta"
print "                                                          14     gray"
print "                                                          15    white"
cursor 5, 23
print "Menu Selection ====================================";
cursor 70, 23

end of set_up_alt


REM *** This procedure is the portal for reentering the other menu ********

procedure reenter
	set_up_menu
	update_status
	xtemp = 0
	display_position
	single_pass = false
	loop = true
	main_loop
end of reenter


REM *** This procedure sets up the globals for the first entry to the *****
REM *** other menu                                         ****************

procedure start_sketch
	xpos = 121
	ypos = 94
	plotting = false
	joy_stick_on = false
	forgrnd = true
	row_mode = false
	speed = 6
	reenter
end of start_sketch

REM *** Color the backplane ***********************************************

procedure backplane
  var bcolor = integer
	cursor 10, 20
	input3 " Backplane Color ? "; bcolor;
	out VDPCNTL, bcolor
	out VDPCNTL, COLORPORT
	clearline 20
	cursor 70, 23
end of backplane

REM *** Set foreground and background pixel colors for entire screen ******

procedure color_pad
var forcolor, baccolor, color_byte, k = integer

cursor  5, 20
input3 "Foreground color ?"; forcolor;
input3 "               Background color ?"; baccolor;
color_byte = (forcolor * 16) + baccolor
vdpram PCTBASE, WRITERAM
for k = 1 to PCTLENGTH
	out VDPDATA, color_byte
next k
clearline 20
cursor 70, 23

end of color_pad

REM *** Change color of pen (in case it is same as graphics image) ******** 

procedure color_pen
var pen_color = integer

cursor 5, 20
input3 " What color pen would you like ? "; pen_color;
vdpram SATBASE + 3, WRITERAM
out VDPDATA, pen_color
clearline 20
cursor 70, 23

end of color_pen

REM *** Initialize VDP and load SKETCH.SCR into the VDP RAM ***************

procedure get_blank_pad
	var j = integer
	var c, reply = char

	procedure init

		dim char a(10)
		var i = integer

		port_select VDP

		a[0] = 002h
		a[1] = 0c2h
		a[2] = 007h
		a[3] = 0ffh
		a[4] = 003h
		a[5] = 03eh
		a[6] = 003h
		a[7] = 000h
		for i = 0 to 7
			out VDPCNTL, a[i]
			out VDPCNTL, 80h + i
		next i
	end of init


cursor 8, 20
input3 "Confirm..... Get blank pad (y or n) ?"; reply;
clearline 20
IF ( reply = 'y' or reply = 'Y' ) THEN
	begin
		clearline 23
		cursor 5, 23
		print "Loading SKETCH.SCR    Please wait ..................";
		init
		out VDPCNTL, 082h
		out VDPCNTL, 081h
		files S(1)
		open #0, "SKETCH.SCR"
		vdpram RAMSTART, WRITERAM
		for j = 1 to RAMLENGTH
		read #0; c
		out VDPDATA, c
		next j
		out VDPCNTL, 0C2h
		out VDPCNTL, 081h
		clearline 23
		cursor 5, 23
		print "Menu Selection =====================================";
		cursor 70, 23
	end

end of get_blank_pad


REM *** This is the main loop of the alternate menu procedure *************  

  set_bit_port
  set_stick_port
  altrestart = true
  WHILE true DO begin

	IF altrestart THEN
		begin
			set_up_alt
			altrestart = false
		end
	key = keyboard ()
	key = tolower ( key )
	case key of
		'r'	:	reenter
		'e'	:	start_sketch
		'b'	:	backplane
		'c'	:	color_pad
		'p'	:	color_pen
		'g'	:	get_blank_pad
		'n'	:	set_up_alt
	end of case
  end of while
end of alternate_menu

comment
***************************************************************************
***  The main loop of the program is an endless loop that allows the    ***
***   User to switch between the two menus at any time since both       ***
***    menus are in procedures                                          ***
***************************************************************************
end of comment

While ( true ) do begin
	
	alternate_menu

end of while
