/* if_bp.c - UniWorks backplane driver */

static char *copyright = "Copyright 1987,1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
...
*/

#include "UniWorks.h"
#include "param.h"
#include "mbuf.h"
#include "protosw.h"
#include "ioctl.h"
#include "socket.h"
#include "errno.h"
#include "if.h"
#include "route.h"
#include "in.h"
#include "in_systm.h"
#include "in_var.h"
#include "ip.h"
#include "ip_var.h"
#include "if_ether.h"
#include "if_ring.h"
#include "if_bp.h"
#include "iv68k.h"
#include "taskLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "etherLib.h"
#include "systm.h"
#include "ioLib.h"

IMPORT int ringput ();
IMPORT int ipintr ();

#define	MAXCPU		32		/* don't change! */
#define	NCPU		32		/* number of input rings to init */
#define NBP		2		/* max bp interfaces */
#define	MINPKTSIZE	60		/* minimum packet size to send */
#define	BP_BUFSIZ	((2 * 1024) + 128)	/* buffer size */

typedef struct ring RING;

typedef struct bufNode		/* buffer control block */
    {
    char	*b_addr;
    short	b_len;
    } BUF_NODE;

typedef struct			/* cpu descriptor in bp header */
    {
    BOOL	cd_active;
    int		cd_cpuNum;
    int		cd_unit;	/* unit on own cpu */
    RING	*cd_pInputRing;
    int		cd_intType;
    int		cd_intArg1;
    int		cd_intArg2;
    int		cd_intArg3;
    } CPU_DESC;

typedef struct			/* bp header */
    {
    int		hdr_ready;
    short	hdr_ncpu;
    u_char	hdr_enetAdrs[6];
    RING	*hdr_pFreeRing;
    CPU_DESC	hdr_cpuDesc[MAXCPU];
    } BP_HEADER;

typedef struct			/* bp anchor */
    {
    int		an_ready;
    UINT	an_heartbeat;
    BP_HEADER	*an_pHdr;
    WDOG_ID	an_wdog;
    } BP_ANCHOR;

#define BP_READY	0x12345678	/* distinctive ready value */

#define	ItoKval(p,type,base)	((type)((int)(p) + (int)(base)))
#define	KtoIval(p,type,base)	((type)((int)(p) - (int)(base)))
#define	ItoK(p,type,base)	((p) = ItoKval(p,type,base))
#define	KtoI(p,type,base)	((p) = KtoIval(p,type,base))

/*
 * Net software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * bs_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */

typedef struct
    {
    struct	arpcom bs_ac;	/* common ethernet structures */
    int		bs_procNum;	/* processor number on this net */
    BP_ANCHOR	*bs_pAnchor;	/* pointer to bp anchor */
    BP_HEADER	*bs_pHdr;	/* pointer to bp header */
    RING	*bs_pFreeRing;	/* pointer to free ring */
    RING	*bs_pInputRing;	/* pointer to our input ring */
    BOOL	bs_readPending;	/* read interrupt is outstanding */
    int		bs_intType;
    int		bs_intArg1;
    int		bs_intArg2;
    int		bs_intArg3;
    } BP_SOFTC;

#define bs_if           bs_ac.ac_if     /* network-visible interface */
#define bs_enaddr       bs_ac.ac_enaddr /* hardware ethernet address */


/* bpPollTask parameters */

int bpPollTaskId;
int bpPollTaskOptions	= VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int bpPollTaskStackSize	= 2000;
int bpPollTaskPriority	= 250;

u_char bpEnetAdrs [6]	= { 60, 123, 0, 0, 0, 0 };

/* LOCAL variables */

LOCAL int bpHeartbeatRate;
LOCAL BP_SOFTC *bp_softc[NBP];


/* forward declarations */

LOCAL VOID bpInitIf ();
LOCAL int bpIoctl ();
LOCAL VOID bpOutput ();
LOCAL VOID bpReset ();
VOID bpPollTask ();
LOCAL int bpReadAll ();
LOCAL struct mbuf *bpGet ();
VOID bpIntr ();
LOCAL VOID bpHeartbeat ();

/********************************************************************
*
* bpattach - make interface available
*
* make interface available by filling in network interface record.
* System will initialize the interface when it is ready to accept packets. 
*/

STATUS bpattach (unit, pAnchor, procNum, intType, intArg1, intArg2, intArg3)
    int unit;
    char *pAnchor;	/* bus pointer to bp anchor */
    int procNum;
    int intType;
    int intArg1;
    int intArg2;
    int intArg3;

    {
    FAST BP_SOFTC *bs;
    FAST struct ifnet *ifp;
    FAST BP_HEADER *pHdr;


    /* wait for shared memory to be initialized */

    if (bpProbe ((BP_ANCHOR *) pAnchor) == ERROR)
	return (ERROR);

    pHdr = ItoKval (((BP_ANCHOR *)pAnchor)->an_pHdr, BP_HEADER *, pAnchor);


    /* allocate bp_softc structure */

    if ((bs = (BP_SOFTC *) malloc (sizeof (BP_SOFTC))) == NULL)
	return (ERROR);

    bp_softc [unit] = bs;
    ifp = &bs->bs_if;


    /* initialize the interface structure */

    bzero ((char *) bs, sizeof (BP_SOFTC));

    bs->bs_pAnchor = (BP_ANCHOR *) pAnchor;
    bs->bs_pHdr    = pHdr;
    bs->bs_procNum = procNum;
    bs->bs_intType = intType;
    bs->bs_intArg1 = intArg1;
    bs->bs_intArg2 = intArg2;
    bs->bs_intArg3 = intArg3;

    bs->bs_pFreeRing = ItoKval (pHdr->hdr_pFreeRing, RING *, pAnchor);

    bs->bs_pInputRing = ItoKval(pHdr->hdr_cpuDesc[procNum].cd_pInputRing,
				RING *, pAnchor);

    /* fill in ethernet address, which is the backplane ethernet address with
     * our processor number as the last byte. */

    bcopy ((char *) pHdr->hdr_enetAdrs, (char *) bs->bs_enaddr, 6);
    bs->bs_enaddr [5] = procNum;

    bzero ((char *) ifp, sizeof (*ifp));

    ifp->if_unit   = unit;
    ifp->if_name   = "bp";
    ifp->if_mtu    = BP_BUFSIZ;		/* was ETHERMTU */
    ifp->if_init   = bpInitIf;
    ifp->if_ioctl  = bpIoctl;
    ifp->if_output = bpOutput;
    ifp->if_reset  = bpReset;
    ifp->if_flags  = IFF_BROADCAST;


    if_attach (ifp);

    return (OK);
    }

/*********************************************************************
*
* bpReboot - mark a bp inactive
*
* called from reboot and panic
*/

LOCAL VOID bpReset ()

    {
    FAST BP_SOFTC *bs;
    FAST BUF_NODE *pBufNode;
    FAST int unit;
	
    for (unit = 0;  unit < NBP;  ++unit)
	{
	bs = bp_softc [unit];
	if ((bs != NULL) && (bs->bs_pHdr != NULL))
	    {
	    /* turn off active flag in our cpu descriptor in the
	     * shared header, then throw away anything on our input ring */

	    bs->bs_pHdr->hdr_cpuDesc[bs->bs_procNum].cd_active = FALSE;

	    while (pBufNode = (BUF_NODE *) ringget (bs->bs_pInputRing))
		ringput (bs->bs_pFreeRing, (int) pBufNode);
	    }
	}
    }

/*********************************************************************
*
* bpInitIf - initialize interface
*/

LOCAL STATUS bpInitIf (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = bp_softc[unit];
    FAST CPU_DESC *pCpu = &bs->bs_pHdr->hdr_cpuDesc [bs->bs_procNum];

    /* check that our input buffer was established
     * (in case NCPU < our cpu < MAXCPU) */

    if (pCpu->cd_pInputRing == NULL)
	return (ERROR);


    /* fill in our info in the cpu descriptor in shared memory */

    pCpu->cd_unit = unit;	/* set our internal unit num */

    pCpu->cd_intType = bs->bs_intType;	/* tell others how to interrupt us */
    pCpu->cd_intArg1 = bs->bs_intArg1;
    pCpu->cd_intArg2 = bs->bs_intArg2;
    pCpu->cd_intArg3 = bs->bs_intArg3;


    /* connect and enable our interprocessor interrupt */

    if (bpConnect (bs, unit) != OK)
	return (ERROR);

    if (bpIntEnable (bs) != OK)
	return (ERROR);


    /* tell others we're here now */

    pCpu->cd_active = TRUE;

    if ((bs->bs_if.if_flags & IFF_RUNNING) == 0)
	bs->bs_if.if_flags |= IFF_UP | IFF_RUNNING;

    return (OK);
    }

/**********************************************************************
*
* bpIntr - interrupt handler
*/

VOID bpIntr (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = bp_softc [unit];

    if ((bs != NULL) && (bs->bs_pInputRing != NULL))	/* input ring exists? */
	{
	bpIntAck (bs);				/* ack interrupt */

	if (!bs->bs_readPending)
	    {
	    bs->bs_readPending = TRUE;
	    netToDoAdd (bpReadAll, unit);	/* read from input ring */
	    }
	}
    }

/*********************************************************************
*
* bpReadAll -
*/

LOCAL VOID bpReadAll (unit)
    int unit;

    {
    FAST BP_SOFTC *bs = bp_softc [unit];
    FAST BUF_NODE *pBufNode;

    bs->bs_readPending = FALSE;

    while (pBufNode = (BUF_NODE *) ringget (bs->bs_pInputRing))
	{
	bpRead (bs, ItoKval (pBufNode, BUF_NODE *, bs->bs_pAnchor));
	ringput (bs->bs_pFreeRing, (int) pBufNode);
	}
    }
/*********************************************************************
*
* bpRead -
*/

LOCAL VOID bpRead (bs, pBufNode)
    BP_SOFTC *bs;
    BUF_NODE *pBufNode;

    {
    FAST struct ether_header *eh;
    FAST struct mbuf *m;
    FAST int len;
    int off;
    int resid;
    FAST struct ifqueue *inq;
    FAST unsigned char *pData;

    bs->bs_if.if_ipackets++;	/* count input packets */

    /* get length and pointer to packet */

    len = pBufNode->b_len;
    eh  = ItoKval (pBufNode->b_addr, struct ether_header *, bs->bs_pAnchor);


    /* call input hook if any */

    if ((etherInputHookRtn != NULL) &&
	(* etherInputHookRtn) (&bs->bs_if, (char *) eh, pBufNode->b_len))
	return;		/* input hook has already processed packet */


    /*
     * Deal with trailer protocol: if type is PUP trailer
     * get true type from first 16-bit word past data.
     * Remember that type was trailer by setting off.
     */

    len -= sizeof (struct ether_header);
    pData = ((unsigned char *) eh) + sizeof (struct ether_header);

    if ((eh->ether_type >= ETHERTYPE_TRAIL)  &&
	(eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER))
	{
	off = (eh->ether_type - ETHERTYPE_TRAIL) * 512;

	if (off >= BP_BUFSIZ)	/* was ETHERMTU */
	    return;		/* sanity */

	eh->ether_type = *(u_short *) (pData + off);
	resid          = *(u_short *) (pData + off + 2);

	if ((off + resid) > len)
	    return;		/* sanity */

	len = off + resid;
	}
    else
	off = 0;

    if (len == 0)
	return;

    /*
     * Pull packet off interface.  Off is nonzero if packet
     * has trailing header; bpget will then force this header
     * information to be at the front, but we still have to drop
     * the type and length which are at the front of any trailer data.
     */

    m = bpGet (pData, len, off, &bs->bs_if);
    if (m == 0)
	return;

    if (off)
        {
	struct ifnet *ifp;

	ifp = *(mtod (m, struct ifnet **));
	m->m_off += 2 * sizeof (u_short);
	m->m_len -= 2 * sizeof (u_short);
	*(mtod (m, struct ifnet **)) = ifp;
	}

    switch (eh->ether_type)
	{
#ifdef INET
	case ETHERTYPE_IP:
	    netToDoAdd (ipintr);
	    inq = &ipintrq;
	    break;

	case ETHERTYPE_ARP:
	    arpinput (&bs->bs_ac, m);
	    return;
#endif
	default:
	    m_freem (m);
	    return;
	}

    {
    int s = splnet ();

    if (IF_QFULL (inq))
	{
	IF_DROP (inq);
	m_freem (m);
	splx (s);
	return;
	}

    IF_ENQUEUE (inq, m);
    splx (s);
    }
    }

/*************************************************************************
*
* bpOutput - net output routine
*
* Encapsulate a packet of type family for the local net.
* Use trailer local net encapsulation if enough data in first
* packet leaves a multiple of 512 bytes of data in remainder.
*
* RETURNS: 0 if successful, errno otherwise (as per network requirements).
*/

LOCAL int bpOutput (ifp, m0, dst)
    struct ifnet *ifp;
    struct mbuf *m0;
    struct sockaddr *dst;

    {
    FAST BP_SOFTC *bs = bp_softc[ifp->if_unit];
    FAST struct mbuf *m = m0;
    FAST struct ether_header *bp;
    FAST int off;
    int type, error;
    u_char edst[6];
    int usetrailers;
    struct in_addr idst;
    int spl;

    if ((bs->bs_if.if_flags & IFF_UP) == 0)	/* not ready yet */
	{
	m_freem (m0);
	return (0);
	}

    switch (dst->sa_family)
	{
#ifdef INET
	case AF_INET:
	    idst = ((struct sockaddr_in *) dst)->sin_addr;
	    if (!arpresolve (&bs->bs_ac, m, &idst, edst, &usetrailers))
		return (0);     /* if not yet resolved */

	    off = ((u_short) mtod (m, struct ip *)->ip_len) - m->m_len;
	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
		type = ETHERTYPE_TRAIL + (off>>9);
		m->m_off -= 2 * sizeof (u_short);
		m->m_len += 2 * sizeof (u_short);
		*mtod (m, u_short *) = ETHERTYPE_IP;
		*(mtod (m, u_short *) + 1) = m->m_len;
		goto gottrailertype;
		}
	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif
	case AF_IMPLINK:
	    /* should do some ARP here? */
	    bp = mtod (m, struct ether_header *);
	    goto gotheader;

	case AF_UNSPEC:
	    bp = (struct ether_header *) dst->sa_data;
	    bcopy ((char *) bp->ether_dhost, (char *) edst, 6);
	    type = bp->ether_type;
	    goto gottype;

	default:
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /* Add local net header.  If no space in first mbuf, allocate another.  */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
	m = m_get (M_DONTWAIT, MT_DATA);
	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
	m->m_next = m0;
	m->m_off = MMINOFF;
	m->m_len = sizeof (struct ether_header);
	}
    else
	{
	m->m_off -= sizeof (struct ether_header);
	m->m_len += sizeof (struct ether_header);
	}

    bp = mtod (m, struct ether_header *);
    bcopy ((char *) edst, (char *) bp->ether_dhost, 6);
    bp->ether_type = type;

gotheader:
    bcopy ((char *) bs->bs_enaddr, (char *) bp->ether_shost, 6);

    /* Queue message on interface if possible */

    spl = splnet ();

    if (bpPut (bs, m) != OK)
	{
	error = ENOBUFS;
	m0 = m;
	splx (spl);
	goto bad;
	}

    bs->bs_if.if_opackets++;

    splx (spl);
    return (0);

bad:
    m_freem (m0);
    return (error);
    }

/**********************************************************************
*
* bpPut - copy from mbuf chain to shared memory buffer.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS bpPut (bs, mp)
    FAST BP_SOFTC *bs;
    FAST struct mbuf *mp;

    {
    struct mbuf *m;
    FAST BUF_NODE *pBufNode;
    FAST u_char *buf;
    u_char *bufStart;

    if ((pBufNode = (BUF_NODE *) ringget (bs->bs_pFreeRing)) == NULL)
	return (ERROR);

    ItoK (pBufNode, BUF_NODE *, bs->bs_pAnchor);
    bufStart = ItoKval (pBufNode->b_addr, u_char *, bs->bs_pAnchor);

    for (buf = bufStart, m = mp; m != NULL; buf += m->m_len, m = m->m_next)
	bcopy ((caddr_t) mtod (m, u_char *), (caddr_t) buf, (unsigned)m->m_len);

    pBufNode->b_len = max (MINPKTSIZE, buf - bufStart);


    /* call output hook if any */

    if ((etherOutputHookRtn != NULL) &&
	(* etherOutputHookRtn) (&bs->bs_if, bufStart, pBufNode->b_len))
	return (OK);	/* output hook has already processed packet */


    /* send to specified processor:
     * proc number is in last byte of ethernet address */

    bpXmit (bs, pBufNode,
	    (int) ((struct ether_header *)(bufStart))->ether_dhost[5],
	    bs->bs_procNum);

    m_freem (mp);

    return (OK);
    }

/************************************************************************
*
* bpXmit -
*/

LOCAL VOID bpXmit (bs, pBufNode, destProcNum, srcProcNum)
    FAST BP_SOFTC *bs;
    FAST BUF_NODE *pBufNode;
    FAST int destProcNum;
    FAST int srcProcNum;

    {
    FAST BP_ANCHOR *pAnchor = bs->bs_pAnchor;
    FAST CPU_DESC *pCpu;

    if ((destProcNum < 0) || (destProcNum >= MAXCPU))
	{
	/* broadcast to all cpus on network */

	FAST u_char *buf;

	buf = ItoKval (pBufNode->b_addr, u_char *, pAnchor);

	for (destProcNum = MAXCPU - 1;  destProcNum >= 0;  --destProcNum)
	    {
	    FAST BUF_NODE *pBufNode0;

	    pCpu = &bs->bs_pHdr->hdr_cpuDesc[destProcNum];

	    if (!pCpu->cd_active)
		continue;

	    if ((pBufNode0 = (BUF_NODE *) ringget (bs->bs_pFreeRing)) == NULL)
		break;

	    ItoK (pBufNode0, BUF_NODE *, pAnchor);

	    pBufNode0->b_len = pBufNode->b_len;
	    bcopy ((caddr_t) buf,
		   (caddr_t) ItoKval (pBufNode0->b_addr, u_char *, pAnchor),
		   (unsigned) pBufNode->b_len);

	    switch (ringput (ItoKval (pCpu->cd_pInputRing, RING *, pAnchor),
			     KtoIval (pBufNode0, int, pAnchor)))
		{
		/* if input ring was full, then put bufNode back on free ring */
		case 0:
		    ringput (bs->bs_pFreeRing,
			     KtoIval (pBufNode0, int, pAnchor));
		    break;

		/* if input ring was empty, then interrupt destination cpu */
		case 1:
		    bpIntGen (pCpu, srcProcNum);
		    break;
		}
	    }

	ringput (bs->bs_pFreeRing, KtoIval (pBufNode, int, pAnchor));
	}
    else
	{
	pCpu = &bs->bs_pHdr->hdr_cpuDesc [destProcNum];

	if (!pCpu->cd_active)
	    ringput (bs->bs_pFreeRing, KtoIval (pBufNode, int, pAnchor));
	else
	    {
	    switch (ringput (ItoKval (pCpu->cd_pInputRing, RING *, pAnchor),
	    		     KtoIval (pBufNode,int, pAnchor)))
		{
		/* if input ring was full, then put bufNode back on free ring */
		case 0:
		    ringput (bs->bs_pFreeRing,
			     KtoIval (pBufNode, int, pAnchor));
		    break;

		/* if input ring was empty, then interrupt destination cpu */
		case 1:
		    bpIntGen (pCpu, srcProcNum);
		    break;
		}
	    }
	}
    }

/************************************************************************
*
* bpGet - copy from shared memory into mbufs.
*/

LOCAL struct mbuf *bpGet (buf0, totlen, off0, ifp)
    FAST unsigned char *buf0;
    int totlen, off0;
    struct ifnet *ifp;

    {
    FAST struct mbuf *m;
    FAST int off = off0;
    FAST unsigned char *cp = buf0;
    struct mbuf *top = 0;
    struct mbuf **mp = &top;
    int len;

    while (totlen > 0)
	{
	MGET (m, M_DONTWAIT, MT_DATA);

	if (m == 0)
	    goto bad;

	if (off)
	    {
	    len = totlen - off;
	    cp = buf0 + off;
	    }
	else
	    len = totlen;

	m->m_off = MMINOFF;

	if (ifp)
	    {
	    /* Leave room for ifp. */
	    m->m_len = MIN (MLEN - sizeof (ifp), len);
	    m->m_off += sizeof (ifp);
	    }
	else 
	    m->m_len = MIN (MLEN, len);

	bcopy ((caddr_t) cp, (caddr_t) mtod (m, u_char *), (unsigned) m->m_len);
	cp += m->m_len;
	*mp = m;
	mp = &m->m_next;

	if (off)
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen)
		{
		cp = buf0;
		off = 0;
		totlen = off0;
		}
	    }
	else
	    totlen -= m->m_len;

	if (ifp)
	    {
	    /* Prepend interface pointer to first mbuf. */
	    m->m_len += sizeof (ifp);
	    m->m_off -= sizeof (ifp);
	    *(mtod (m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}

    return (top);

bad:
    m_freem (top);
    return ((struct mbuf *)0);
    }

/*************************************************************************
*
* bpIoctl - process an ioctl request
*
*    this can be called via the "socket" route for SIOCSIFADDR or
*	by the cdev/inode route for SIOCSIFCCFWR/RD
*
*/

LOCAL int bpIoctl (ifp, cmd, data)
    FAST struct ifnet *ifp;
    int cmd;
    caddr_t data;

    {
    FAST struct ifaddr *ifa = (struct ifaddr *) data;
    int error = 0;
    int s = splimp ();

    switch (cmd)
	{
        case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;

	    error = bpInitIf (ifp->if_unit);
	    if (error != OK)
		break;

	    switch (ifa->ifa_addr.sa_family)
		{
#ifdef INET
		case AF_INET:
		    ((struct arpcom *) ifp)->ac_ipaddr = IA_SIN (ifa)->sin_addr;
		    arpwhohas ((struct arpcom *) ifp, &IA_SIN (ifa)->sin_addr);
		    break;
#endif
#ifdef NS
		case AF_NS:
		    {
		    FAST struct ns_addr *ina = &(IA_SNS (ifa)->sns_addr);
			
		    if (ns_nullhost (*ina))
			ina->x_host = *(union ns_host *)(xs->xs_addr);
		    else
			ex_setaddr (ina->x_host.c_host,ifp->if_unit);
		    break;
		    }
#endif
		}

	    break;

	default:
	    error = EINVAL;
	}

    splx (s);
    return (error);
    }

/********************************************************************
*
* bpPollTask - backplane poll task
*
* This routine is spawned as task to poll the backplane input rings for
* this cpu.  It is only used if no backplane interrupt mechanism is
* used.
*/

VOID bpPollTask ()

    {
    FAST int unit;

    FOREVER
	{
	for (unit = 0;  unit < NBP;  ++unit)
	    bpIntr (unit);
	}
    }

/************************************************************************
*
* bpIntAck - acknowledge our backplane interrupt
*/

LOCAL VOID bpIntAck (bs)
    FAST BP_SOFTC *bs;

    {
    switch (bs->bs_intType)
	{
	case BP_INT_NONE:
	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	    break;

	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

	case BP_INT_BUS:
	    sysIntAck (bs->bs_intArg1);
	    break;
	}
    }

/************************************************************************
*
* bpConnect - connect to our backplane using the requested intType
*
*/

LOCAL STATUS bpConnect (bs, unit)
    FAST BP_SOFTC *bs;
    FAST int unit;

    {
    STATUS status;

    switch (bs->bs_intType)
	{
	case BP_INT_NONE:
	    /* no interprocessor interrupt - spawn background polling task */
	    bpPollTaskId = taskSpawn ("bpPollTask", bpPollTaskPriority,
				      bpPollTaskOptions, bpPollTaskStackSize,
				      bpPollTask, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	    status = OK;
	    break;

	/* interrupt by mailbox (write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	    status = sysMailboxConnect (bpIntr, unit);
	    break;

	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

	case BP_INT_BUS:
	    status = intConnect (INUM_TO_IVEC (bs->bs_intArg2), bpIntr, unit);
	    break;

	default:
	    status = ERROR;
	    printErr ("if_bp: bad int type: type = %d\n", bs->bs_intType);
	}

    return (status);
    }

/************************************************************************
*
* bpIntEnable - enable our backplane interrupt
*
*/

LOCAL STATUS bpIntEnable (bs)
    FAST BP_SOFTC *bs;

    {
    STATUS status;

    switch (bs->bs_intType)
	{
	case BP_INT_NONE:
	    /* no interrupt (we will poll) */
	    status = OK;
	    break;

	/* interrupt by mailbox (write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

	case BP_INT_MAILBOX_1:
	case BP_INT_MAILBOX_2:
	case BP_INT_MAILBOX_4:
	    status = sysMailboxEnable ((char *) bs->bs_intArg2);
	    if (status == ERROR)
		printErr ("if_bp: error enabling mailbox interrupt.\n");
	    break;

	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

	case BP_INT_BUS:
	    status = sysIntEnable (bs->bs_intArg1);
	    if (status == ERROR)
		printErr ("if_bp: error enabling bus interrupt level %d.\n",
			  bs->bs_intArg1);
	    break;

	default:
	    status = ERROR;
	    printErr ("if_bp: bad int type: type = %d\n", bs->bs_intType);
	}

    return (status);
    }

/************************************************************************
*
* bpIntGen - interrupt another cpu
*
*/

LOCAL VOID bpIntGen (pCpu, srcProcNum)
    FAST CPU_DESC *pCpu;	/* ptr to descriptor of cpu to interrupt */
    int srcProcNum;		/* our procNum on this network */

    {
    char *pMailbox;

    /* if we're trying to interrupt ourself (a bad idea on some cpus (isi))
     * simply call the isr directly */

    if (pCpu->cd_cpuNum == srcProcNum)
	{
	netToDoAdd (bpReadAll, pCpu->cd_unit);	/* read input ring */
	return;
	}

    switch (pCpu->cd_intType)
	{
	case BP_INT_NONE:
	    /* no interrupt (cpu is polling) */
	    break;

	/* interrupt by mailbox (write to bus address):
	 *   arg1 = bus address space
	 *   arg2 = address
	 *   arg3 = value
	 */

	case BP_INT_MAILBOX_1:
	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		*pMailbox = pCpu->cd_intArg3;
	    break;

	case BP_INT_MAILBOX_2:
	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		* (short *) pMailbox = pCpu->cd_intArg3;
	    break;

	case BP_INT_MAILBOX_4:
	    if (sysBusToLocalAdrs (pCpu->cd_intArg1,
				   (char *) pCpu->cd_intArg2,
				   &pMailbox) == OK)
		* (long *) pMailbox = pCpu->cd_intArg3;
	    break;


	/* bus interrupt:
	 *   arg1 = level
	 *   arg2 = vector
	 */

	case BP_INT_BUS:
	    sysIntGen (pCpu->cd_intArg1, pCpu->cd_intArg2);
	    break;


	default:
	    logMsg ("bpIntGen: bad int type: proc num = %d, type = %d\n",
		     pCpu->cd_cpuNum, pCpu->cd_intType);
	}
    }

/*********************************************************************
*
* bpInit -
*
* NOTE: addresses are in local space
*/

STATUS bpInit (pAnchor, pMem, memSize)
    FAST char *pAnchor;		/* pointer to bp anchor */
    FAST char *pMem;		/* pointer to start of bp shared memory */
    int memSize;		/* num bytes in bp shared memory */

    {
    FAST BP_ANCHOR *pBpAnchor = (BP_ANCHOR *) pAnchor;
    FAST RING *pFreeRing;
    FAST int i;
    FAST CPU_DESC *pCpu;
    FAST BP_HEADER *pHdr;
    WDOG_ID wdog;

    /* create watchdog timer for heartbeat */

    wdog = wdCreate ();
    if (wdog == NULL)
	return (ERROR);

    /* if bp header specified, probe it, otherwise malloc it */

    if (pMem != (char *) NONE)
	{
	if ((memSize = bpSizeMem (pMem, memSize)) == 0)
	    {
	    printErr ("bp: no shared memory at address 0x%x\n", pMem);
	    return (ERROR);
	    }

	/* if anchor is at start of shared region, skip over it */

	if (pMem == pAnchor)
	    {
	    pMem += sizeof (BP_ANCHOR);
	    memSize -= sizeof (BP_ANCHOR);
	    }
	}
    else
	{
	if ((pMem = malloc (memSize)) == NULL)
	    {
	    printErr ("bp: can't allocate shared memory\n", pMem);
	    return (ERROR);
	    }
	}

    pHdr = (BP_HEADER *) pMem;

    /* initialize bp anchor region */

    pBpAnchor->an_ready     = 0;		/* set bp not ready */
    pBpAnchor->an_heartbeat = 0;		/* clear heartbeat */
    pBpAnchor->an_wdog      = wdog;		/* set watchdog id */
    pBpAnchor->an_pHdr      = KtoIval (pHdr, BP_HEADER *, pAnchor);

    /* initialize bp header region */

    pHdr->hdr_ready = 0;		/* set icpu region not ready */
    pHdr->hdr_ncpu = NCPU;		/* set number of cpus */
    bcopy ((char *) bpEnetAdrs, (char *) pHdr->hdr_enetAdrs, 6);

    /* initialize each cpu descriptor */

    for (i = 0;  i < MAXCPU;  ++i)
	{
	pCpu = &pHdr->hdr_cpuDesc[i];

	pCpu->cd_active      = FALSE;
	pCpu->cd_cpuNum      = i;
	pCpu->cd_pInputRing  = NULL;
	}

    /* initialize free ring */

    pMem += sizeof (BP_HEADER);

    ringinit ((pFreeRing = (RING *) pMem), 256);
    pHdr->hdr_pFreeRing = KtoIval (pFreeRing, RING *, pAnchor);
    pMem += sizeof (struct ring256);

    /* initialize each input ring */

    for (i = 0;  i < NCPU;  ++i)
	{
	ringinit ((RING *) pMem, 32);
	pHdr->hdr_cpuDesc[i].cd_pInputRing = KtoIval (pMem, RING *, pAnchor);
	pMem += sizeof (struct ring32);
	}

    /* initialize as many buffer blocks as will fit in remaining shared memory*/

    while ((pMem + sizeof (BUF_NODE) + BP_BUFSIZ) <= ((caddr_t) pHdr + memSize))
	{
	((BUF_NODE *)pMem)->b_addr = KtoIval (pMem + sizeof (BUF_NODE),
					      char*, pAnchor);
	((BUF_NODE *)pMem)->b_len  = BP_BUFSIZ;

	ringput (pFreeRing, KtoIval (pMem, int, pAnchor));
	pMem += sizeof (BUF_NODE) + BP_BUFSIZ;
	}

    /* mark bp as ready */

    pBpAnchor->an_ready = BP_READY;
    pHdr->hdr_ready = BP_READY;

    /* start heartbeat */

    bpHeartbeatRate = sysClkGetRate ();
    wdStart (wdog, bpHeartbeatRate, bpHeartbeat, (int) pAnchor);

    return (OK);
    }

/**********************************************************************
*
* bpSizeMem - size shared memory pool
*/

LOCAL int bpSizeMem (memAdrs, memSize)
    caddr_t memAdrs;	/* address of start of pool to start probing */
    int memSize;	/* max size of memory pool (0 == 0x100000) */

    {
    FAST int i;
    char c;

    if (memSize == 0)
	memSize = 0x100000;

    for (i = 0;  i < memSize;  i += 4096, memAdrs += 4096)
	{
	if (vxMemProbe (memAdrs, READ, 1, &c) != OK)
	    break;
	}

    /* Return the amount of memory we found.  */

    return (i);
    }

/*************************************************************************
*
* bpProbe - probe for bp established
*/

LOCAL STATUS bpProbe (pAnchor)
    BP_ANCHOR *pAnchor;

    {
    FAST int cnt = 10 * 60 * 60 * sysClkGetRate ();	/* wait 10 minutes! */
    int test;
    unsigned int oldHeartbeat = 0;

    while (--cnt > 0)
	{
	/* probe test:
	 *  1) anchor must be readable (memory must be visable to us),
	 *  2) anchor.ready must have BP_READY,
	 *  3) anchor.heartbeat must be increasing
	 */

	if ((vxMemProbe ((char *) &pAnchor->an_ready,
			 READ, 4, (char *) &test) == OK) &&
	    (test == BP_READY) &&
	    (vxMemProbe ((char *) &pAnchor->an_heartbeat,
			 READ, 4, (char *) &test) == OK))
	    {
	    if (oldHeartbeat == 0)
		oldHeartbeat = test;
	    else
		{
		if (test > oldHeartbeat)
		    return (OK);	/* heartbeat increased; passed test */
		else
		    oldHeartbeat = test;
		}
	    }

	taskDelay (1);
	}

    return (ERROR);	/* give up - didn't get initialized */
    }
/*************************************************************************
*
* bpHeartbeat -
*/

LOCAL VOID bpHeartbeat (pAnchor)
    FAST BP_ANCHOR *pAnchor;

    {
    ++pAnchor->an_heartbeat;
    wdStart (pAnchor->an_wdog, bpHeartbeatRate, bpHeartbeat, (int) pAnchor);
    }
