/* sysLib.c - Mizar mz8115 system dependent library */

static char *copyright = "Copyright 1988, Mizar Digital Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts

AUTHOR
Mizar Digital Sytems, Carrollton, Texas
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "iv68k.h"

typedef unsigned char UCHAR;	/* mcl */

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

#define	HZ	3686400			/* clock rate */

/* locals */

LOCAL FUNCPTR duartTxInt       = NULL;
LOCAL FUNCPTR duartRxInt       = NULL;
LOCAL BOOL duartIsConnected    = FALSE;

LOCAL FUNCPTR sysClkRoutine    = NULL;
LOCAL int sysClkArg	       = NULL;
LOCAL int clkTicksPerSecond    = 60;
LOCAL BOOL sysClkIsRunning     = FALSE;

/* external declarations */

VOID sysBclrIntA ();		/* Handle /BCLR interrupt in assembly */
VOID sysAbortIntA ();		/* Handle abort interrupt in assembly */

/* forward declarations */

LOCAL VOID sysDuartInt ();		/* The handler of autovector 3 ints. */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Mizar mz8115"
*/

char *sysModel ()
    {
    return ("Mizar mz8115");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
*
* The timers are initialized and turned off.
* The DUART is initialized for use by the ty driver.
*/

VOID sysHwInit ()

    {
    /* disable bus clear interrupts by setting "Release On Request" */

    sysEnableROR ();

    *DUART_IMR  = 0;			/* mask serial interrupts */
    *DUART_IVR  = (UCHAR)INT_VEC_SERIAL;	/* set interrupt autovector */

    *DUART_ACR  = BRG_SELECT | TMR_EXT_CLK;	/* brg #2, xtal clock */

    /* handle bus clear and abort switch interrupts */
    intSetVec (IV_AUTOVEC_7, sysAbortIntA);
    intSetVec (IV_AUTOVEC_5, sysBclrIntA);
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* INTERNAL
* We don't bother sizing memory, just return a constant.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return (char *) (LOCAL_MEM_LOCAL_ADRS + LOCAL_MEM_SIZE);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot.
			   The possible types are defined in h/sysLib.h */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

VOID sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    if (!duartIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_SERIAL), sysDuartInt, NULL) == ERROR)
	{
	return (ERROR);
	}

    duartIsConnected = TRUE;

    sysClkRoutine    = routine;
    sysClkArg        = arg;

    sysImrSet (CTR_RDY_INT, 0);

    return (OK);
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()
    {
    if (sysClkIsRunning)
	{
	/* stop timer */

	if (*DUART_CTROFF);	/* EOI on the timer */

	sysClkIsRunning = FALSE;
	}
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int tc;

    /* calculate the divide ratio, and write it to the timer chip */

    tc = HZ / (2 * clkTicksPerSecond);

    *DUART_CTUR = MSB(tc);
    *DUART_CTLR = LSB(tc);

    sysClkIsRunning = TRUE;

    if (*DUART_CTRON);	/* EOI on the timer */
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (clkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	clkTicksPerSecond = ticksPerSecond;

    if (sysClkIsRunning)                    
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*
* RETURNS: ERROR to indicate that there is no auxiliary clock
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*
* ARGSUSED
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*/

VOID sysAuxClkDisable ()
    {
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*/

VOID sysAuxClkEnable ()
    {
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*
* RETURNS: ERROR as there is no auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
*
* NOTE:
* There is no auxiliary clock.
* This routine has no effect.
*
* SEE ALSO: sysAuxClkGetRate (2)
*
* ARGSUSED
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VMEbus address
* that would have to be accessed to get to that byte.
*
* NOTE MZ8115:
* In A24 space, only the lower 24 address bits are examined.  In A16 space,
* only the lower 16 address bits are examined.  Unused bits are set to zeros
* on exit.
*
* RETURNS:  OK, or ERROR if unable to get to that local address from the VMEbus.
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    UINT temp;

    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pBusAdrs = (char *) ((UINT) localAdrs & 0x0000ffff);
	    return (OK);	/* no on-board-only A16 space */

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    temp = (UINT) localAdrs & 0x00ffffff;
	    if (temp > 0x00efffff)
		return (ERROR);
	    else
		if ((temp >= LOCAL_MEM_BUS_ADRS) &&
		    (temp < (LOCAL_MEM_BUS_ADRS + LOCAL_MEM_SIZE)))
		    {
		    return (ERROR); /* illegal address in the first place! */
		    }
		else
		    {
		    if ((temp >= LOCAL_MEM_LOCAL_ADRS) ||
			(temp < (LOCAL_MEM_LOCAL_ADRS + LOCAL_MEM_SIZE)))
			    temp += LOCAL_MEM_BUS_ADRS;
		    *pBusAdrs = (char *) temp;
		    return (OK);
		    }

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    return (ERROR);	/* sorry, only an A24 board */

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VMEbus memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* NOTE MZ8115:
* In A24 space, only the lower 24 address bits are examined.  In A16 space,
* only the lower 16 address bits are examined.  Unused bits are set to zeros
* on exit.
*
* RETURNS: OK, or ERROR if unknown address space.
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in
			 * vme.h, such as VME_AM_STD_SUP_DATA */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    UINT temp;

    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0x00ff0000 | ((UINT) busAdrs & 0x0000ffff));
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    temp = (UINT) busAdrs & 0x00ffffff;
	    if ((temp < 0x00080000) || (temp > 0x00efffff))
		return (ERROR);
	    else
		{
		if ((temp >= LOCAL_MEM_BUS_ADRS) &&
		    (temp < (LOCAL_MEM_BUS_ADRS + LOCAL_MEM_SIZE)))
			temp -= LOCAL_MEM_BUS_ADRS;	/* remap local RAM */
		*pLocalAdrs = (char *) temp;
		return (OK);
		}

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    return (ERROR);	/* sorry, only an A24 board */

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable VMEbus interrupt level
*
* This routine disables the specified VMEbus interrupt level.
*
* NOTE:
* VMEbus interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable VMEbus interrupt level
*
* This routine enables the specified VMEbus interrupt level.
*
* NOTE:
* VMEbus interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge VMEbus interrupt
*
* This routine acknowledges VMEbus interrupts.
*
* NOTE:
* Hardware acknowledges VMEbus interrupts automatically.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysIntGen - generate VMEbus interrupt
*
* This routine generates a VMEbus interrupt.
*
* NOTE:
* Not a VMEbus interrupter.
*
* RETURNS: ERROR
*
* ARGSUSED
*/

STATUS sysIntGen (intLevel,vector)
    int intLevel;	/* interrupt level to generate */
    int vector;		/* interrupt vector to generate */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* NOTE:
* Does not support mailbox interrupts.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt*/
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* NOTE:
* Does not support mailbox interrupts.
*
* RETURNS: ERROR, since there is no mailbox facility
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    return (ERROR);
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VMEbus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* NOTE:
* This is equivalent to the vxTas (2) routine.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysEnableROR - disable bus clear interrupts by setting "Release On Request"
*
* This routine disables bus clear interrupts by setting "Release On Request"
* to the VMEbus, and clearing "Programmed Release On Request."
* "Release On Request" is the default condition on the mz8115.
*/

VOID sysEnableROR ()

    {
    *DUART_ROPBC = ENABLE_VME_ROR;
    }
/*******************************************************************************
*
* sysDisableROR - enable bus clear interrupts by clearing "Release On Request"
*
* This routine enables bus clear interrupts by clearing "Release On Request"
* to the VMEbus, and restoring "Programmed Release On Request."
* "Release On Request" is the default condidtion on the mz8115.
*/

VOID sysDisableROR ()

    {
    *DUART_SOPBC = DISABLE_VME_ROR;
    }
/*******************************************************************************
*
* sysImrSet - set and clear bits in the m68681 DUART interrupt mask register
*
* This routine sets and clears bits in the DUART's IMR.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the DUART.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*/

VOID sysImrSet (setBits, clearBits)
    char setBits;	/* which bits to set in the IMR */
    char clearBits;	/* which bits to clear in the IMR */

    {
    static UCHAR imr;		/* current value of DUART imr register */

    *DUART_IMR = imr = (imr | setBits) & (~clearBits);
    }
/*******************************************************************************
*
* sysDuartInt - interrupt level processing for mz8115 DUART.
*
* This routine handles an interrupt from the on-board DUART.  The interrupt
* is decoded by checking the DUART's interrupt status register, and the
* appropriate routine(s) invoked.  
*
* This routine is located here because we use the extra timer on the DUART
* chip as the system clock, which is handled here.
*/

LOCAL VOID sysDuartInt ()

    {
    if (*DUART_ISR & TX_RDY_A)
	{
	/* transmitter channel A interrupt.  Call DUART routine, if
	   there is one, else issue an EOI */

	if (duartTxInt == NULL)
	    sysImrSet (0, TX_RDY_A_INT);	/* EOI */
	else
	    (*duartTxInt) (0);
	}

    if (*DUART_ISR & TX_RDY_B)
	{
	/* transmitter channel B interrupt.  Call DUART routine, if
	   there is one, else issue an EOI */

	if (duartTxInt == NULL)
	    sysImrSet (0, TX_RDY_B_INT);	/* EOI */
	else
	    (*duartTxInt) (1);
	}

    if ((*DUART_ISR & RX_RDY_A) && (duartRxInt != NULL))
	(*duartRxInt) (0);			/* receiver channel A */

    if ((*DUART_ISR & RX_RDY_B) && (duartRxInt != NULL))
	(*duartRxInt) (1);			/* receiver channel B */

    if (*DUART_ISR & CTR_RDY)			/* on-chip timer */
	{
	if (sysClkRoutine != NULL)
	    (*sysClkRoutine) (sysClkArg);

	if (*DUART_CTROFF);		/* EOI on the timer */
	}
    }
/*******************************************************************************
*
* sysDuartConnect - connect interrupt routines for the mz8115 DUART
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*/

STATUS sysDuartConnect (recvRoutine, xmitRoutine)
    FUNCPTR recvRoutine;	/* receive routine */
    FUNCPTR xmitRoutine;	/* transmit routine */

    {
    if (!duartIsConnected &&
	intConnect (INUM_TO_IVEC (INT_VEC_SERIAL), sysDuartInt, NULL) == ERROR)
	{
	return (ERROR);
	}

    duartIsConnected = TRUE;

    duartTxInt       = xmitRoutine;
    duartRxInt       = recvRoutine;

    return (OK);
    }
