/* sysLib.c - Mizar mz7120 system dependent library */

static char *copyright = "Copyright 1988, Mizar Digital Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains system-dependent routines to set and get the
system clock rate, initialize and handle system hardware, find
the top of memory, deal with interrupts, etc.
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "mz7120.h"
#include "mz7120sp4.h"
#include "config.h"
#include "m68681.h"
#include "m68230.h"
#include "iv68k.h"

IMPORT char end;		/* end of system, created automatically by ld */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL int clkTicksPerSecond;
LOCAL int auxClkTicksPerSecond;
LOCAL FUNCPTR duartTxInt = NULL;
LOCAL FUNCPTR duartRxInt = NULL;

LOCAL FUNCPTR sysClkRoutine = NULL;
LOCAL int sysClkRoutineParm;
LOCAL int sysClkFreq;

LOCAL FUNCPTR auxClkRoutine = NULL;
LOCAL int auxClkRoutineParm;
LOCAL int auxClkFreq;

LOCAL ULONG bcl;		/* current value of board control latch */
LOCAL ULONG imr;		/* current value of duart imr register */

/* forward declarations */

VOID sysBclrIntA ();		/* Handle /BCLR interrupt in assembly */
VOID sysClkIntA ();		/* Handle heartbeat interrupt in assembly */
VOID sysAbortIntA ();		/* Handle abort interrupt in assembly */
VOID sysMemParityError ();	/* memory parity error interrupt rtn */
VOID sysDuartInt ();		/* The handler of autovector 4 ints. */

/* XXX dummy routines not implemeted! */
VOID sysClkEnable () {}
VOID sysAuxClkEnable () {}

/************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Mizar mz7120".
*/

char *sysModel ()
    {
    return ("Mizar mz7120");
    }
/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board including the
* control register, the DUART, and the PI/T.
* It is normally called from usrInit (2) in usrConfig (1).
*
*/

VOID sysHwInit ()

    {
    *DUART_IMR  = 0;		/* mask serial interrupts */

    *DUART_ACR  = BRG_SELECT | TMR_EXT_CLK;	/* brg #2, xtal clock */

    /* handle bus clear interrupts */
    intSetVec (INUM_TO_IVEC (MZ_AVEC_7), sysAbortIntA);

    intSetVec (INUM_TO_IVEC (MZ_AVEC_6), sysClkIntA);

    /* enable heart beat interrupts */
    sysSetBCL (0xffffffff, MZ_BD_TIMER_ENABLE);

    intSetVec (INUM_TO_IVEC (MZ_AVEC_5), sysBclrIntA);
    }
/**********************************************************************
*
* sysMemTop - get top of memory address
*
* This routine finds the size of system RAM.
*
* RETURNS: The address of the first missing byte of memory.
*/

char *sysMemTop ()

    {
    return (MZ_RAM_END_ADRS + 1);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot.
			   The possible type are defined in h/sysLib.h */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }
/**********************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

VOID sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each system clock
			 * interrupt */
    int arg;		/* argument with which to call routine */

    {
    char setButNotUsed;		/* used to generate EOI on the timer */

    intConnect (INUM_TO_IVEC (MZ_AVEC_4), sysDuartInt, NULL);
    sysClkRoutine     = routine;
    sysClkRoutineParm = arg;
    sysImrSet (CTR_RDY_INT, 0);
    setButNotUsed = *DUART_CTROFF;	/* EOI on the timer */
    }
/**********************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2)
*/
VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    int ctr;
    char setButNotUsed;		/* used to generate EOI on the timer */

    /* calculate the divide ratio, and write it to the timer chip.  If
       ticksPerSecond == 0, write a value of 0 to stop the clock. */

    ctr = (ticksPerSecond == 0) ? 0 : (3686400 / (2 * ticksPerSecond));

    *DUART_CTUR = (ctr & 0xff00) >> 8;
    *DUART_CTLR = (ctr & 0x00ff);
    setButNotUsed = *DUART_CTRON;	/* EOI on the timer */

    sysClkFreq = ticksPerSecond;
    }
/**********************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: The number of ticks of the system clock per second.
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (sysClkFreq);
    }
/**************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt,
* and enables the auxiliary clock interrupt.
*
* RETURNS: OK, to indicate that there is, indeed,
* an auxiliary clock on this CPU board.
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, parm)
    FUNCPTR routine;	/* routine to be called at each auxiliary clock
			 * interrupt */
    int parm;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/**************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    }
/**********************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
*
* SEE ALSO: sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    }
/**************************************************************************
*
* sysAuxClkGetRate - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: The number of ticks of the auxiliary clock per second.
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()

    {
    return (ERROR);
    }
/************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS:  OK, or ERROR if unable to get to that local address from the VME.
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

/* ARGSUSED */
STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    if (localAdrs >= sysMemTop ())
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	return (OK);
	}
    else
	{
	return (ERROR);
	}

    }
/*************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space.
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

/* ARGSUSED */
STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in
			 * vme.h, such as VME_AM_STD_SUP_DATA */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (0x00800000 | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if (busAdrs < (char *) 0x01000000)
		return (ERROR);

	    *pLocalAdrs = busAdrs;
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*************************************************************************
*
* sysIntEnable - enable VME interrupt level
*
* This routine enables the specified VME interrupt level.
* Interrupts 1 - 3 are jumper enabled.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-3.
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    if ((intLevel < 1) || (intLevel > 3))
	return (ERROR);
    else
	return (OK);
    }
/*************************************************************************
*
* sysIntDisable - disable VME interrupt level
*
* This routine alway returns error because VME interrupts are not under
* software control.
*
* RETURNS: ERROR.
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (ERROR);
    }
/*************************************************************************
*
* sysIntAck - acknowledge VME interrupt
*
* This is a null routine because hardware acknowledges VME interrupts.
*
* RETURNS: OK.
*/

STATUS sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (OK);
    }
/*************************************************************************
*
* sysIntGen - generate VME interrupt
*
* The mz7120 is not a VMEbus interrupter.
*
* RETURNS: ERROR.
*/

STATUS sysIntGen (intLevel)
    int intLevel;	/* interrupt level to generate */

    {
    return (ERROR);
    }
/**********************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* The mz7120 does not support mailbox interrupts.
*
* RETURNS: ERROR.
*
* SEE ALSO: intConnect (2)
*/

/* ARGSUSED */

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each mailbox interrupt*/
    int arg;		/* argument with which to call routine */

    {
    return (ERROR);
    }
/************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* The mz7120 does not support mailbox interrupts.
*
* RETURNS: ERROR.
*/

/* ARGSUSED */
STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    return (ERROR);
    }
/***************************************************************************
*
* sysSetBCL - set bits in the board control latch
*
* This routine sets bits in the board control latch on the mz7120 board.
* The parameter "mask" determines which bits will be set, and "value"
* determines the value to which those bits will be set.
* In  other words, newBCL = (oldBCL & ~mask) | (value & mask).
*/

VOID sysSetBCL (mask, value)
    ULONG mask;		/* which bits to change */
    ULONG value;	/* what to change bits to */

    {
    bcl = (bcl & (~mask)) | (value & mask);
    *MZ_BD_CONTROL_LATCH = bcl;
    }
/***********************************************************************
*
* sysGetBCL - return the current value of the board control latch
*
* This routine returns the last value to which the board control latch
* was set.  It is only effective if all changes to the board control
* latch are done through sysSetBCL.
*/

ULONG sysGetBCL ()

    {
    return (bcl);
    }
/**********************************************************************
*
* sysImrSet - set and clear bits in the m68681's interrupt mask register
*
* This routine sets and clears bits in the duart's IMR.  It may be called
* either in user or supervisor state.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the duart.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* NOTE: This routine must be called in supervisor mode.
*/

sysImrSet (setBits, clearBits)
    char setBits;	/* which bits to set in the IMR */
    char clearBits;	/* which bits to clear in the IMR */

    {
    imr = (imr | setBits) & (~clearBits);
    *DUART_IMR = imr;
    }
/**********************************************************************
*
* sysDuartInt - interrupt level processing for mz7120 sp4 daughter board
*
* This routine handles an interrupt from the on-board duart.  The interrupt
* is decoded by checking the duart's interrupt status reg, and the appropriate 
* routine invoked.  
*
* This routine is located here because we use the extra timer on the duart
* chip as the system clock, which is handled here.  Second, the 68230 is
* multiplexed on this interrupt level.
*/

LOCAL VOID sysDuartInt ()

    {
    char setButNotUsed;		/* used to generate EOI on the timer */

    if (*DUART_ISR & TX_RDY_A)
	{
	/* transmitter channel A interrupt.  call duart routine, if
	   there is one, else issue an EOI */

	if (duartTxInt == NULL)
	    sysImrSet (0, TX_RDY_A); 	/* EOI */
	else
	    (*duartTxInt) (0);
	}

    if (*DUART_ISR & TX_RDY_B)
	{
	/* transmitter channel B interrupt.  call duart routine, if
	   there is one, else issue an EOI */

	if (duartTxInt == NULL)
	    sysImrSet (0, TX_RDY_B); 	/* EOI */
	else
	    (*duartTxInt) (1);
	}

    if ((*DUART_ISR & RX_RDY_A) && (duartRxInt != NULL))
	(*duartRxInt) (0);			/* receiver channel A */

    if ((*DUART_ISR & RX_RDY_B) && (duartRxInt != NULL))
	(*duartRxInt) (1);			/* receiver channel B */

    if (*DUART_ISR & CTR_RDY)			/* on-chip timer */
	{
	if (sysClkRoutine != NULL)
	    (*sysClkRoutine) (sysClkRoutineParm);

	setButNotUsed = *DUART_CTROFF;		/* EOI on the timer */
	}
    }
/************************************************************************
*
* sysDuartConnect - set interrupt routines for the mz7120 sp4
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*/

VOID sysDuartConnect (recvRoutine, xmitRoutine)
    FUNCPTR recvRoutine;	/* receive routine */
    FUNCPTR xmitRoutine;	/* transmit routine */

    {
    intConnect (INUM_TO_IVEC (MZ_AVEC_4), sysDuartInt, NULL);
    duartTxInt = xmitRoutine;
    duartRxInt = recvRoutine;
    }
/**********************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for the mz7120.  Processor numbers should be
* unique on a single backplane.
*
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/**********************************************************************
*
* sysGetProcNum - get processor number
*
* Get the processor number for the mz7120.  Processor numbers should be
* unique on a single backplane.
*
*/

VOID sysGetProcNum (procNum)
    int procNum;		/* processor number */

    {
    return (sysProcNum);
    }
/************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* On the mz7120 this is equivalent to the vxTas (2) routine.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }
