/* tyCoDrv.c - MV133, MV133A, MV133XT & MV134 tty handler */

static char *copyright = "Copyright 1986-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
01h,30may88,dnw  changed to v4 names.
01g,29apr88,gae  fixed MFP usage.
01f,11mar88,gae  merged with MV133A & MV134 ports.
		   corrected baud rate table.
01e,09dec87,jcf  does not include mv133.h; comes via config.h
01d,04dec87,dnw  removed erroneous set of {} in initialization of tyCoDv.
01c,13nov87,ecs  documentation.
01b,23oct87,ecs  delinted
01a,06aug87,jcf  written by modifying 01a of dy4 version.
*/

/*
DESCRIPTION
This is the driver for the Multi-Function Peripheral Chip used
on the MV133, MV133A, MV133XT & MV134 boards.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 110, 300, 600, 1200, 2400, 4800, 9600.

MV133 VARIATIONS
This driver supports several variations of the MV133 board.
When hardware is initialized in sysHwInit (2), the appropriate
memory map addresses are determined.
*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "iv68k.h"
#include "mv133.h"


IMPORT char *MFP_BASE_ADR;	/* Multi-Function Peripheral address */

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *ucr;		/* USART control register */
    char *rsr;		/* receiver status register */
    char *tsr;		/* transmit status register */
    char *udr;		/* USART data register */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv;

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char timeConstant;	/* time constant to write to the timers C and D data
			   registers (TCDR & TDDR) to get that baud rate */
    } BAUD;

LOCAL BAUD baudTable [] =
    {
    {110,	0x57}, {300,	0x20}, {600,	0x10}, {1200,	0x08},
    {2400,	0x04}, {4800,	0x02}, {9600,	0x01}
    };

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoTxInt ();
LOCAL VOID tyCoRxInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */

    if (tyCoDrvNum > 0)
	return (OK);

    /* The MFP chip interrupts on lvl. 5 and supplies the vector to jump thru */

    intConnect (INUM_TO_IVEC(MFP_INT_VECT_NUM + MFP_INT_TRANS),tyCoTxInt, NULL);
    intConnect (INUM_TO_IVEC(MFP_INT_VECT_NUM + MFP_INT_RECV), tyCoRxInt, NULL);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*
* NOTE:
* The channel parameter is ignored since the MV133 has only one
* serial port on this USART.  The parameter is included only to
* make the calling sequence compatible with other versions.
*
* ARGSUSED
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 only!) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    int temp;

    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if this device already exists, don't create it */

    if (tyCoDv.created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    tyCoDv.ucr = MFP_UCR(MFP_BASE_ADR);
    tyCoDv.rsr = MFP_RSR(MFP_BASE_ADR);
    tyCoDv.tsr = MFP_TSR(MFP_BASE_ADR);
    tyCoDv.udr = MFP_UDR(MFP_BASE_ADR);

    if (tyDevInit ((TY_DEV_ID) &tyCoDv, 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)
	{
	return (ERROR);
	}

    /* enable the receiver */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_RECV_BUFFER_FULL;	/* unmask recv interrupts */
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* mark the device as created, and add the device to the I/O system */

    tyCoDv.created = TRUE;
    return (iosDevAdd ((DEV_HDR *) &tyCoDv, name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the USART
*
* This routine initializes the MV133 USART for the UniWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    int temp;

    oldlevel = intLock ();

    *MFP_UCR(MFP_BASE_ADR) = 0x88;	/* 8 data bits, 1 stop bit, no parity */
  
    temp = *MFP_TCDCR(MFP_BASE_ADR);
    temp = 0x10 | (temp & 0x0f);	/* set timer c for /4 prescale */
    *MFP_TCDCR(MFP_BASE_ADR) = temp;
    *MFP_TCDR(MFP_BASE_ADR) = 0x01;	/* set for 9600 baud */

    /* enable the receivers and transmitters on both channels*/

    *MFP_RSR(MFP_BASE_ADR) = 1;		/* enable receiver status */
    *MFP_TSR(MFP_BASE_ADR) = 1;		/* enable transmitter status */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_RECV_BUFFER_FULL;	/* enable recv interrupts */
    *MFP_IERA(MFP_BASE_ADR) = temp;

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TRANS_BUFFER_EMPTY;	/* enable transmit interrupts */
    *MFP_IERA(MFP_BASE_ADR) = temp;

    /* all interrupts are still masked out via the interrupt mask registers: 
       turn the receiver on and off by twiddling the interrupt mask reg (IMRA),
       the receiver interrupt will be enabled in tyCoDevCreate,
       the transmitter interrupt will be enabled in tyCoStartup */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to MV133 USART
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for MV133 USART
*
* This routine fields all read calls to the MV133 USART.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - Task level write routine for MV133 USART
*
* This routine fields all write calls to the MV133 USART.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles FIOBAUDRATE requests and passes all others to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    int temp;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:

	    temp = *MFP_TCDCR(MFP_BASE_ADR);
	    temp = 0x10 | (temp & 0x0f);
            *MFP_TCDCR(MFP_BASE_ADR) = temp;	/* delay mode, prescale = 4 */
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)
		    {
		    *MFP_TCDR(MFP_BASE_ADR) = baudTable [i].timeConstant;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}

    return (status);
    }
/*******************************************************************************
*
* tyCoTxInt - handle a transmitter interrupt
*
* This routine gets called to handle transmitter interrupts.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyCoTxInt ()

    {
    char outChar;
    int temp;

    if ((tyCoDv.created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv, &outChar) == OK))

	*(tyCoDv.udr) = outChar;
    else

	/* turn off the transmitter */
	{
	temp = *MFP_IMRA(MFP_BASE_ADR);
	temp &= ~MFP_TRANS_BUFFER_EMPTY;
	*MFP_IMRA(MFP_BASE_ADR) = temp;
	}
    }
/*******************************************************************************
*
* tyCoRxInt - handle a receiver interrupt
*
* This routine gets called to handle receiver interrupts.
*/

LOCAL VOID tyCoRxInt ()

    {
    tyIRd ((TY_DEV_ID) &tyCoDv, *tyCoDv.udr);
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for MV133 USART.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outchar;
    int temp;

    /* all we gotta do is enable the transmitter and it should interrupt
     * to write the next char out. We don't use IMR_ON, because this routine
     * is called from task level, so sysIoWB must be used. */

    if (tyITx ((TY_DEV_ID) pTyCoDv, &outchar) == OK)
	*pTyCoDv->udr = outchar;

    /* turn the transmitter on */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TRANS_BUFFER_EMPTY;
    *MFP_IMRA(MFP_BASE_ADR) = temp;
    }
