/* tyCoDrv.c - The Motorola MVME-131 USART tty handler */

static char *copyright = "Copyright 1987, Wind River Systems, Inc.";

/*
modification history
--------------------
01c,13nov87,ecs  documentation.
01b,23oct87,ecs  delinted
*/

/*
DESCRIPTION
This is the driver for on-board serial ports of the Motorola MVME-131.
The serial ports are provided by a MK68564 serial I/O chip.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.

*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "mv131.h"
#include "mk68564.h"

IMPORT tyRead ();
IMPORT tyWrite ();

#define INPUT_FREQUENCY		2457600		/* 2.4576 Mhz input clock */

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;		/* TRUE = this device has really been created */
    UTINY *base;		/* port registers base address */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv[] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, (UTINY *) 0xfffb0040 },
    {{{{NULL}}}, FALSE, (UTINY *) 0xfffb0050 }
    };

LOCAL int tyCoNum;		/* driver number assigned to this driver */


/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();

/***********************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    FUNCPTR *intVec = (FUNCPTR *) 0x110;

    intConnect (intVec, tyCoInt, 0);

    tyCoHrdInit (intVec);

    tyCoNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen, (FUNCPTR) NULL,
			     tyRead, tyWrite, tyCoIoctl);

    return (OK);
    }
/******************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 only!) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    FAST TY_CO_DEV *pTyCoDv = &tyCoDv [channel];
    FAST UTINY *base = pTyCoDv->base;

    /* if this device already exists, don't create it */

    if (pTyCoDv->created)
	return (ERROR);

    /* initialize the ty descriptor */

    if (tyDevInit (&pTyCoDv->tyDev, rdBufSize, wrtBufSize, tyCoStartup) != OK)
	return (ERROR);


    /* initialize channel parameters and initialize interrupts */

    *MK564_BAUD_REG (base) = MK564_BAUD_RCV_CLK_INTERNAL |
			     MK564_BAUD_XMIT_CLK_INTERNAL;

    tyCoIoctl (pTyCoDv, FIOBAUDRATE, 9600);	/* set baud rate = 9600 */

    *MK564_MODE_CTRL_REG (base) = MK564_MODE_CLK_X16 | MK564_MODE_STOP_1;
    *MK564_INT_CTRL_REG (base)  = MK564_INT_ON_EACH_RCV_CHAR |
				  MK564_INT_XMIT_INT_ENABLE  |
				  /* XXX MK564_INT_EXT_INT_ENABLE   | */
				  MK564_INT_STATUS_AFFECTS_VECTOR;
    *MK564_RCV_CTRL_REG (base)  = MK564_RCV_8_BITS | MK564_RCV_ENABLE;
    *MK564_XMIT_CTRL_REG (base) = MK564_XMIT_8_BITS | MK564_XMIT_ENABLE |
				  MK564_XMIT_DTR | MK564_XMIT_RTS;


    /* Mark the device as having been created, and add device to the io sys */

    pTyCoDv->created = TRUE;
    return (iosDevAdd (&pTyCoDv->tyDev.devHdr, name, tyCoNum));
    }
/******************************************************************
*
* tyCoHrdInit - initialize the usart.
*
* This routine initializes the MK68564 usart for the UniWorks environment. 
*
*/

LOCAL VOID tyCoHrdInit (intVec)
    FUNCPTR *intVec;

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* reset both channels and program interrupt vector */

    /* *MK564_COMMAND_REG (tyCoDv[0].base)    = MK564_CMD_CHANNEL_RESET; */
    *MK564_COMMAND_REG (tyCoDv[1].base)    = MK564_CMD_CHANNEL_RESET;
    *MK564_INT_VECTOR_REG (tyCoDv[0].base) = (int) intVec >> 2;

    intUnlock (oldlevel);
    } 

/**********************************************************************
*
* tyCoOpen - open file to usart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/***********************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    STATUS status;
    FAST int timeConst;
    BOOL divideBy64;

    switch (request)
	{
	case FIOBAUDRATE:

	    /* compute time constant with divide by 4 value to start with,
	     * (assumes x16 clock) */

	    timeConst = (INPUT_FREQUENCY / (4 * 16)) / arg;


	    /* make it divide by 64 if necessary */

	    divideBy64 = (timeConst > 256);
	    if (divideBy64)
		{
		timeConst /= 16;

		if (timeConst > 256)		/* error if still too big */
		    return (ERROR);
		}

	    if (timeConst == 256)		/* count 256 is set as 0 */
		timeConst = 0;


	    /* disable baud gen, set divider and time constant, re-enable gen */

	    *MK564_BAUD_REG (pTyCoDv->base) &= ~MK564_BAUD_GEN_ENABLE;

	    if (divideBy64)
		*MK564_BAUD_REG (pTyCoDv->base) |= MK564_BAUD_DIVIDE_BY_64;
	    else
		*MK564_BAUD_REG (pTyCoDv->base) &= ~MK564_BAUD_DIVIDE_BY_64;

	    *MK564_TIME_CONST_REG (pTyCoDv->base) = timeConst;	/* set count */

	    *MK564_BAUD_REG (pTyCoDv->base) |= MK564_BAUD_GEN_ENABLE;

	    status = OK;
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/**********************************************************************
*
* tyCoRxInt - handle receive interrupt
*
* This routine gets called to handle interrupts.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyCoInt ()

    {
    FAST TY_CO_DEV *pTyCoDv;
    int status;
    char outChar;

    status = *MK564_INT_VECTOR_REG (tyCoDv[0].base);
    pTyCoDv = ((status & 4) == 0) ? &tyCoDv[1] : &tyCoDv[0];

    switch (status & 3)
	{
	case 0:			/* XMIT BUFFER EMPTY */
	    if (pTyCoDv->created && (tyITx (&pTyCoDv->tyDev, &outChar) == OK))
		*MK564_DATA_REG (pTyCoDv->base) = outChar;
	    else
		*MK564_COMMAND_REG (pTyCoDv->base) = MK564_CMD_RESET_XMIT_INT;
	    break;

	case 1:			/* EXTERNAL/STATUS CHANGE */
	    logMsg ("tyCoDrv: external/status interrupt: chan=%d st0=0x%x\n",
		    (pTyCoDv == &tyCoDv[0]) ? 0 : 1,
		    *MK564_STATUS_0_REG (pTyCoDv->base));

	    *MK564_COMMAND_REG (pTyCoDv->base) = MK564_CMD_RESET_EXT_INT;
	    break;

	case 2:			/* RECEIVE CHAR AVAILABLE */
	    if (pTyCoDv->created)
		tyIRd (&pTyCoDv->tyDev, (char) *MK564_DATA_REG (pTyCoDv->base));
	    break;

	case 3:			/* SPECIAL RECEIVE CONDITION */
	    logMsg ("tyCoDrv: special receive interrupt: chan=%d st1=0x%x\n",
		    (pTyCoDv == &tyCoDv[0]) ? 0 : 1,
		    *MK564_STATUS_1_REG (pTyCoDv->base));

	    *MK564_COMMAND_REG (pTyCoDv->base) = MK564_CMD_ERROR_RESET;
	    break;
	}
    }
/**********************************************************************
*
* tyCoStartup - Transmitter startup routine
*
* Call interrupt level character output routine for heurikon usart.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if (tyITx (&pTyCoDv->tyDev, &outChar) == OK)
	*MK564_DATA_REG (pTyCoDv->base) = outChar;
    }
