/* romInit.s - init module for code in rom for DY4_134 */

/*
modification history
--------------------
01d,19nov87,dnw changed to use ROM_BASE_ADRS & LOCAL_MEM_LOCAL_ADRS in config.h.
01c,14nov87,jcf	romInit2 now takes an argument from sysToMonitor
01b,16oct87,jcf	cleaned up, clearing boot state is now before clearing ram
		so that all memory is dual ported.
01a,03oct87,jcf	written by modifying v01a of hkv2f/romInit.s
*/

#define ASMLANGUAGE
#include "vxWorks.h"
#include "sysLib.h"
#include "config.h"

	/* internals */

	.globl	_romInit	/* start of system code */

	/* externals */

	.globl	_usrInit	/* system initialization routine */


#define COPY_LOC	0xc0000		/* rom will be copied to 0x90000 as per
					 * Makefile.
					 */

	.text
	.even

_romInit:
	movew	#0x3700,sr	/* disable interrupts, turn on M bit */
	bra	romInit1	/* initial start in rom */

	movew	#0x3700,sr	/* disable interrupts, turn on M bit */
	bra	romInit2	/* subsequent starts in rom */


/***********************************************************************
*
* romInit1 -
*
* enable cache, clear all memory, and copy romCopy to ram
* This routine is position independent.

* romInit1 ()

*/

romInit1:
	/* Enable the cache by writing to the cache control register.
	 * Since the assemblers differ on the syntax for this one
	 * (if they even have it), it's done with two .word's. */

	movel	#0x09,d0	/* clear and enable cache */
	.word	0x4e7b		/* movec ... 			*/
	.word	0x0002		/* .....	d0,cacr 	*/

	movew	#0xff27,0xfffd4000	/* Turn off boot state sending eprom */
					/* to high mem, and turn on top led */

	/* clear all of memory;
	 * outer loop counter in d2 = 0x3 =
	 *
	 *   1M bytes to clear
	 *   ------------------  - 1
	 *   64K * 4               ^
	 *    ^    ^               |
	 *    |    |               for dbra at end of outer loop
	 *    |    bytes per inner loop
	 *    inner loops per outer loop
	 */

	movel	#0x3,d2			/* outer loop counter */
	movel	#LOCAL_MEM_LOCAL_ADRS,a0 /* get start of local memory */
	clrl	d0			/* value to fill */


fb1:	movel	#0xffff,d1		/* set to fill 64K */
fb2:	movel	d0,a0@+			/* move 4 bytes */
	dbra	d1,fb2			/* inner loop test */
	dbra	d2,fb1			/* outer loop test */

	/* set stack to grow down from where code will eventually be */

	movel	#_romInit,a7		/* set stack to start of code */
	movel	#BOOT_COLD,d7		/* arg = start type */
	bra	_romCopyCopy		/* goto romCopyCopy routine */


/***********************************************************************
*
* romInit2 -
*
* same as romInit but doesn't clear memory.
* This routine is position independent.

* romInit2 (startType)
*    int startType;

*/

romInit2:
	link	a6,#0		/* no unlk necessary; we change stacks! */

	/* Enable the cache by writing to the cache control register.
	 * Since the assemblers differ on the syntax for this one
	 * (if they even have it), it's done with two .word's. */

	movel	#0x09,d0	/* clear and enable cache */
	.word	0x4e7b		/* movec ... 			*/
	.word	0x0002		/* .....	d0,cacr 	*/

	/* set stack to grow down from start of code */

	movel	a6@(0x08),d7		/* put startType in d7 */
	movel	#_romInit,a7		/* set stack to start of code */
	bra	_romCopyCopy		/* goto romCopyCopy */

/***********************************************************************
*
* romCopyCopy - copy the rom copying routine to ram
*
* this routine copies the rom copy routine to ram so that the seperate pages may
* loaded.
*/
_romCopyCopy:

	movel	#COPY_LOC,a0		/* destination address of _romCopy */
	movel	#romCopyEnd,d0		/* end of _romCopy */
	subl	#_romCopy,d0		/* length of routine in bytes */
	movel	#_romCopy,a1		/* source address of _romCopy */
	subl	#_romInit,a1		/* find offset into code to _romCopy */
	addl	#ROM_BASE_ADRS + 8,a1	/* + vectors length to get start adrs */
loop1:
	moveb	a1@+,a0@+		/* copy _romCopy to ram */
	dbf	d0,loop1		/* until we are finished */
	jmp	COPY_LOC		/* get out of this crazy rom */
    					/* this base must be higher than 128k */
					/* because in boot state low address */
					/* space is occupied by the eprom */


/***********************************************************************
*
* romCopy - copies the eprom into on board ram
*
* it copies all the pages of the eprom to wherever
* romInit has been linked to run.  Care should be taken in choosing this
* address.  Don't copy the eprom onto the copy routine and leave enough
* space below for vxWorks to be booted over the network.
* The safest order from lowest to highest address is init. stack, RAM_TEXT_ADRS,
* ROM_TEXT_ADRS, COPY_LOC.  The sizes of these sections should
* be, respectively, 1k, 256k, 128k, 100 bytes.
*
* The contents of d7 at entry contains the argument to determine whether this
* was a warm or cold boot.  This argument gets pushed onto the stack.
*
* THIS IS NOT A CALLABLE ROUTINE.
*/
_romCopy:

	/* copy system from rom to ram */
	movel	#_romInit,a0		/* where system should be in ram */
	subl	#8,a0			/* start 2 longs early for vectors */
	movel	#ROM_BASE_ADRS,a1	/* where system is in rom */
	movel	#0x7,d0			/* page counter */
	movel	#0x0,d2			/* page selector */
loop2a:
	movel	#0xfff,d1		/* page length in long words */
loop2b:
	movel	a1@+,a0@+		/* copy 4 bytes of code */
	dbf	d1,loop2b		/* do all of this page */
	addl	#1,d2			/* select the next page */
	moveb	d2,ROM_BASE_ADRS	/* write to rom for new page */
	movel	#ROM_BASE_ADRS,a1	/* the adrs. of the start of each page*/
	dbf	d0,loop2a		/* do every page */

	/* 
	* On the early intel 27011 chips the reset line does not work.
	* We reset the rom to page zero so that a sysReset will find the
	* initial sp, pc vectors.
	* NOTE: This is not a problem at power up.  The chip resets correctly 
	* by latching on the rising edge of the Vcc level, but if one were to
	* reset the board while in this routine, the board would crash because
	* the initial pc and sp will not be present during 7 of the 8 pages.
	* The only two solutions are to power down the board or use the 
	* newer 27011's.
	*/

	moveb	#0,ROM_BASE_ADRS	/* reset rom to page zero */
	movel	d7,a7@-			/* push arg = start type */
	jsr	_usrInit		/* never returns - starts up kernel */
romCopyEnd:				/* end address of _romCopy */
