/* ibDrv.c -- The vxWorks driver for the National 
 *	Instruments GPIB 1014 gpib controller board.
 *  
 *	Written by Kevin Meier from a 4.2bsd driver
 *  supplied by National Instruments.
 *	November 7, 1987.
 */


 /*
 * The vxWorks handler implemented here consists of a
 * straightforward set of independent functions used to
 * manipulate an interface board.  It is at the level of
 * example code provided in the hardware reference manual but
 * structured to be a subset of the complete integrated driver
 * package.  Sufficient functionality is implemented so that
 * an application program can control and coordinate data
 * transfers among a suite of instruments on the GPIB.
 *
 * The function calls support synchronous, DMA I/O
 * for a single interface board which is always the
 * system controller and controller in charge.
 * Prior to calling ibrd or ibwrt the appropriate devices,
 * including the interface board, must be addressed by calling
 * ibcmd with the proper addressing commands.
 */

#include "vxWorks.h"
#include "iosLib.h"
#include "semLib.h"
#include "wdLib.h"
#include "ibDrv.h"

/* Define ASTRMD only if the driver will be used with
 * the Astromed MT9500 chart recorder. Be aware that 
 * if defined, this may cause the driver to not work
 * with other devices, e.g. HP6032 power supplies.
 *
 * This is a bug that will be fixed later.
#define ASTRMD
#undef ASTRMD
 */

/* Status bits */

static struct ibregs *ib=(struct ibregs *)BASEOF1014;   /* board address */

static int ibcmd_next = 0;          /* next write call will be a command */
int no_timeout = 0;	/* next read call will not start watchdog timer */

char r_isr1, r_isr2;     /* software copies of read-once registers */
char r_imr1, r_imr2;     /* software copies of write side of above registers */

unsigned long vme_start;	/* start of vme address space */

/* carry cycle array */
char cc_byte;
struct cc_ary {
	char *cc_ccb;
	short   cc_ONE;
	char *cc_n_1addr;
	short   cc_TWO;
} cc_array = {0, 1, 0, 2};

LOCAL IB_DEV ibDv;		 /* device type */
LOCAL int IbNum;	 	 /* driver number for this device */

LOCAL SEMAPHORE IbSem;		/* semaphore for driver lockout */
LOCAL SEMAPHORE srq_sem;	/* semaphore to allow only one task*/
							/* to wait for srq's */
long ib_mail;				/* interrupt sync mailbox */

LOCAL WDOG_ID ib_wd;		/* watchdog timer id */

first_read=1;				/* indicated first time reading from bus */

/* forward declarations */
int ibOpen();
int ibRead();
int ibWrite();
int ibIoctl();
int ib_dma();
LOCAL ib_watch ();
VOID ibIntr(); 


/*
 * IBDRV
 * Install gpib driver in vxWorks operating
 * system, initialize any data structures
 *
 */
STATUS ibDrv ()

{ 
    short id_code;

    intConnect (IBVEC << 2,ibIntr);

	/* install driver functions in vxWorks */
    IbNum = iosDrvInstall (ibOpen,(FUNCPTR) NULL,ibOpen, (FUNCPTR) NULL,
				ibRead, ibWrite, ibIoctl);
    
    /* initialize semaphore */
    semInit(&IbSem);
    semGive(&IbSem);
    semInit(&srq_sem);
    semGive(&srq_sem);

    /* initialize mailbox */
    ib_mail = 0;

	/* initialize watchdog timer */
	ib_wd = wdCreate ();

	/* WARNING: this vme offset code ports this driver
	 * specifically to the DY4 DVME134 CPU board
	 */
    /* find the location of the 134 memory in vme memory space */
	/* find the code for the starting */
	/* address of vme memory space.   */
    id_code = *((short *)(0xFFFD0000)); 	

	/* clear unwanted bits */
    id_code &= 0x001F;				  	

	/* compute the starting address   */
    vme_start = ((id_code / 2) * 0x100000);		

    printf("VME_START = 0x%x\r\n",vme_start) ;
}

/*
 * IBDEVCREATE
 *      Perform initialization on first open.
 *	    Create device for operating system.
 *      Allow only one open at a time.
 */

ibDevCreate(name) 
char *name;
{
	static first_open=1;

	/* set TLC chip online and initialize board */
	if (first_open) {
		first_open = 0;
		ibonl(1);
	}
	if (ibDv.created)
		return(ERROR);
	ibDv.created = 1;
	/* add device to operation system */
	return(iosDevAdd((DEV_HDR *)&ibDv, name, IbNum));
}

/* IBOPEN
 * 	Open a filename to the device 
 */

int ibOpen (pIbDv,name,mode)
IB_DEV *pIbDv;
char *name;
int mode;

{
    return((int) pIbDv);
}



/*
 * IBREAD
 * Read up to cnt bytes of data from the GPIB into buf. In
 * addition to I/O complete, the read operation terminates
 * on detection of EOI.  Read operation always holds off the
 * handshake.  Prior to beginning the read the
 * interface is placed in the controller standby state.
 * Prior to calling ibRead, the intended devices as well as
 * the interface board itself must be addressed by calling
 * ibcmd.
 *
 */
ibRead(pIbDv, buffer, maxbytes)
IB_DEV *pIbDv;
char *buffer;
int maxbytes;
{
	int status, count;

	semTake (&IbSem);
	if (! maxbytes) {
		semGive (&IbSem);
		return(ERROR);
	}
	ib->auxmr = AUX_LTN; 
	/* read one byte from gpib */
	if (maxbytes == 1) {
		status = ibrdone(buffer);
		semGive (&IbSem);
		return(status);
	}
	/* if more than one byte, use dma transfer */
	ib_dma(pIbDv, buffer,maxbytes, READ);
	count = dmaresid();
	semGive (&IbSem);
	/* XXX this return value is a kludge which must be */
	/* fixed later */
	return(maxbytes - count);
}

/*
 * IBWRITE
 * Write cnt bytes of data from buf to the GPIB.  The write
 * operation terminates only on I/O complete.  By default,
 * EOI is always sent along with the last byte.  Prior to
 * beginning the write the interface is placed in the
 * controller standby state.  Prior to calling ibwrt, the
 * intended devices as well as the interface board itself
 * must be addressed by calling ibcmd.
 *
 * Write entry point is also used as a branch to ibcmd() if
 * the appropriate ioctl() call was previously made.
 */
ibWrite(pIbDv, buffer, nbytes)
IB_DEV *pIbDv;
char *buffer;
int nbytes;
{
	STATUS status;

	semTake (&IbSem);
	if (ibcmd_next)  {
		/* send a command */
		status =  ibcmd(pIbDv,buffer,nbytes);
	}
	else if (nbytes == 1) {
		/* write only one character to bus */
		status = ibwrtone(buffer);
	}
	else {
		/* use dma to transfer more than 1 character */
		status = ib_dma(pIbDv, buffer,nbytes, WRITE);
	}
	semGive (&IbSem);
	return(status);
}

/*
 * GPIB-1014 dma routine:
 *      Always does DMA, always does carry cycle operation.
 * 		If we get here, we are guaranteed that more than 1
 *		byte will be transfered.
 */
ib_dma(pIbDv, buffer, nbytes, operation)
IB_DEV *pIbDv;
char *buffer;
int nbytes;
int operation;
{

	int status = 0;
	register struct  ibregs *b;
	register char *dmaddr;
	char  w_imr2;
	short   cnt;
	long msg;

	b = ib;
	dmaddr = buffer + vme_start; 
	cnt = nbytes;

	b->auxmr = AUX_GTS;     /* goto standby */

	if (operation == READ) 
		{
		if (!first_read) b->auxmr = AUX_FH;
		else first_read = 0;
		b->auxmr= AUXRA|HR_HLDE;      /* holdoff on end */
		cc_byte = AUXRA|HR_HLDA;      /* (carry cycle) holdoff on all */

		b->ch0.ocr = D_DTM|D_XRQ;
		b->cfg1 = D_ECC|D_IN|(IBIRQ << 5)|D_BRG3|D_DBM;

		b->ch1.ocr = D_DTM|D_ACH|D_XRQ;

		/* enable interrupts and dma */
		b->imr1 = HR_ENDIE;
		w_imr2 = HR_DMAI;
	} else { 			/* do a write operation */ 
		cc_byte = AUX_SEOI;        /* send EOI with last byte */

		b->ch0.ocr = D_MTD|D_XRQ;
		b->cfg1 = D_ECC|D_OUT|(IBIRQ << 5)|D_BRG3|D_DBM;
		b->ch1.ocr = D_MTD|D_ACH|D_XRQ;

		/* enable interrupts and dma */
		b->imr1 = 0; 
		w_imr2 = HR_DMAO;
	}
	/* setup channel 1 (carry cycle) */

	cc_array.cc_ccb = (char *)(&cc_byte) + vme_start;
	cc_array.cc_n_1addr = dmaddr + cnt - 1;
	cnt--;

	b->ch1.csr = D_CLEAR;
	b->ch1.bar = (unsigned long)(&cc_array) + vme_start; 
	b->ch1.btc = 2;

	/* setup channel 0 (main transfer) */
	b->ch0.csr = D_CLEAR;
	b->ch0.mtc = cnt;
	b->ch0.mar = (unsigned long)dmaddr;
#ifdef IBDEBUG
	printf("ib_dma, transfer count channel 0 = %d\r\n",cnt);
	printf("ib_dma, transfer address channel 0 = 0x%x\r\n",(long)dmaddr);
#endif IBDEBUG

	/* start dma (ch1 first) and enable interrupts */
	b->ch1.ccr = D_SAB;
	b->ch1.ccr = D_EINT|D_SRT;
	b->ch0.ccr = D_SRT;

	/* start watchdog timer for read, write will return with an error if */
	/* no device on the bus will answer */
	/*
	if (operation == READ) 
	*/
	if (!no_timeout) {
		no_timeout = 0;
		wdStart(ib_wd,
		    (operation == READ) ? IB_RD_WD_WAIT : IB_WRT_WD_WAIT,
		    ib_watch,dmaddr);  
	}

	b->auxmr = AUX_FH;     /* finish handshake */
	b->imr2 = w_imr2;       /* this must be done last */

	/* wait for interrupt */
	vxPend(&ib_mail, 0, &msg);

	/* disable interrupts */
	b->imr2 = 0; 
	b->imr1 = 0; 

	ib->cfg1 = (IBIRQ << 5)|D_BRG3|D_DBM; 

	/* check for error in DMAC initialization */
	if((b->ch0.csr & D_ERR) || (b->ch1.csr & D_ERR)) {
		printf("Error in DMAC initialization\r\n");
		status = ERROR;
	} else 
		if (msg == NO_ERR) status = OK;
			else status = ERROR;

	/* abort any ongoing dma and clear status registers 
	 */
	b->ch0.ccr = D_SAB;
	b->ch0.csr = D_CLEAR;
	b->ch1.ccr = D_SAB;
	b->ch1.csr = D_CLEAR;
	b->auxmr = AUX_TCA; 			/* take control back */ 

	return(status);
}

/* get number of bytes not transferred */
dmaresid() 
{
	register int cnt;

	cnt = ib->ch0.mtc;
	if (ib->ch1.mtc == 2 || cnt)    /* add one to count if carry-cycle */
		cnt++;                  /* never started */
	return cnt;
}


/* Watchdog timer routine 
 *
 * Posts to the interrupt mailbox if
 * no device answers on the bus
 */

LOCAL ib_watch(dmaddr)
unsigned long dmaddr;
{
#ifdef IBDEBUG 
	logMsg("Gpib Timed Out\n");
#endif IBDEBUG 
	vxPost(&ib_mail,IB_TMOUT);
	/* abort idc hangup */
}



/*
 * IBIOCTL
 *      The ioctl routine is used to implement the remaining functions.
 *      Since the functions require at most one argument the third
 *      address is taken to be an argument passed by value.
 *
 */

ibIoctl(dev,code,v) 
{
	switch(code) {
		case IBONL:
			/* send TLC chip on or offline */
			return ibonl(v);
		case IBCAC:
			/* go to controller active state */
			return ibcac();
		case IBGTS:
			/* go to standby */
			return ibgts();
		case IBRPP:
			/* conduct a parallel poll */
			/* WARNING: this has never been used or tested */
			return ibrpp(v);
		case IBSIC:
			/* generate an interface clear (IFC) message */
			return ibsic();
		case IBSRE:
			/* assert or unassert the REN line */
			return ibsre(v);
		case IBWAIT:
			/* wait indefinitely for an srq */
			return ibwaitsrq(v);
		case IBCMD:
			/* the next write call will be a command
			 * and not data.
			 */
			semTake(&IbSem);
			ibcmd_next++;
			semGive(&IbSem);
			return(OK);
		case IBNTM:
			/* the next read call will be not start
			 * the watchdog timer.
			 */
			semTake(&IbSem);
			no_timeout++;
			semGive(&IbSem);
			return(OK);
	}
}

/*
 * IBINTR
 */
VOID ibIntr() 
{
	int ilevel;
	int status;

	/* lock out any other interrupts */
	/*
	ilevel = intLock(IBVEC);
	*/
	/* cancel watchdog timer */
	wdCancel(ib_wd);

#ifdef IBDEBUG 
	logMsg("Interrupt received from gpib board\r\n"); 
#endif IBDEBUG 

	/*
	 * determine if board caused interrupt
	 */
	if ((ib->ch1.csr & (D_PCLT|D_ERR)) == 0) {
#ifdef IBDEBUG 
		logMsg("Fell out at first error check\r\n");
#endif IBDEBUG 
		/*
		intUnlock(ilevel);
		*/
		return;
	}

	/* read status and config1 registers to clear interrupt */
	r_isr2 |= ib->isr2; 
	r_isr1 |= ib->isr1;
	ib->cfg1 = (IBIRQ << 5)|D_BRG3|D_DBM;

	ib->ch1.csr = D_CLEAR;   /* reset channel 1*/

	if (r_isr1 & HR_ERR) {
#ifdef IBDEBUG 
		logMsg("GPIB Error:  Device not answering.\r\n"); 
#endif IBDEBUG 
		r_isr1 &= ~HR_ERR;
		status = ERROR;
	} else status = NO_ERR;

#ifdef IBDEBUG 
	logMsg("Status register 1:%x\r\n",r_isr1);
	logMsg("Status register 2:%x\r\n",r_isr2); 
#endif IBDEBUG 

	if ((r_isr1 & HR_DI) || (ib->ch0.csr & D_COC) || (r_isr1 & HR_END) || 
		(r_isr2 & HR_CO) || (r_isr1 & HR_DO)) {  
		vxPost(&ib_mail, status);
		/*
		intUnlock(ilevel);
		*/
	} else {
		vxPost(&ib_mail, 1);
		logMsg("Error in gpib interrupt handler\r\n");
		logMsg("isr1 = %x\r\n    isr2 = %x\r\n",r_isr1, r_isr2);
		/*
		intUnlock(ilevel);
		*/
	}
	ib->ch0.csr = D_CLEAR;  /* reset channel 0 */

}

/*
 * IBCMD
 * Write cnt command bytes from buf to the GPIB.  The
 * command operation terminates only on I/O complete.  Prior
 * to beginning the command the interface is placed in the
 * controller active state.  Before calling ibcmd for the
 * first time, ibsic must be called to initialize the GPIB
 * and enable the interface to leave the controller idle
 * state.
 *
 * ibcmd() should be called by:
 *      ioctl(f,IBCMD,0);
 *      write(f,buf,cnt);
 *
 */
ibcmd(pIbDv, buffer, nbytes)
IB_DEV *pIbDv;
char *buffer;
int nbytes;
{
	int count;
	char ch;
	long err_msg;

	ibcmd_next = 0;
	ib->auxmr = AUX_GTS;
	ib->auxmr = AUX_TCA;    /* if standby, go to CAC */
	ib->ch1.ccr = D_EINT; 

 	for (count = 0; count < nbytes; count++) {
		ch = *((char *)(buffer+count));
#ifdef IBDEBUG
		printf("character to send is 0x%x, or %c\r\n",ch,ch);
		printf("Sending character number %d\r\n",count);
#endif IBDEBUG
		/*output command */
		ib->cdor =  ch;
#ifdef IBDEBUG
		logMsg("ibcmd: waiting for interrupt\r\n");
		logMsg("before enable: \r\nADSR = 0x%x\r\n",ib->adsr);
		logMsg("ADR0 = 0x%x\r\n",ib->adr0);
		logMsg("ADR1 = 0x%x\r\n",ib->adr1);
#endif IBDEBUG
		/* enable interrupts */
		ib->imr2 = HR_COIE; 
        ib->ch1.ccr = D_EINT; 
		/*
		 * wait for interrupt
		 */
		vxPend(&ib_mail, 0, &err_msg);    
		r_isr2 &= ~HR_COIE;
	}
	/* disable interrupts */
	ib->imr2 = 0; 
	ib->ch1.ccr = 0; 
	return((err_msg == ERROR) ? ERROR : OK);
}

/*
 * IBWAITSRQ
 * Check or wait for a GPIB srq to occur.  
 *
 * XXX This is a kludge now. Hopefully we 
 * will get this driven by interrupts later.
 */
ibwaitsrq(mask)
{
	long msg;

	semTake(&srq_sem);
	while ( ib->ch0.csr & D_NSRQ) {
		vxTdelay(50);
	}
	semGive(&srq_sem);
	return(OK);
}

/*
 * IBRPP
 * Conduct a parallel poll and return the byte in buf.  Prior
 * to conducting the poll the interface is placed in the
 * controller active state.
 * WARNING: This has not been tested.
 */
static ibrpp(buf) 
char *buf;
{
	semTake(&IbSem);
	ib->auxmr = AUX_TCA;           /* if standby, go to CAC */
	ib->auxmr = AUX_EPP;           /* enable parallel poll */
	*buf = ib->cptr;   	       	   /* store the response byte */
	semGive(&IbSem);
	return (*buf);    
}

/*
 * IBONL
 * Initialize the interface hardware.  If v is non-zero then
 * the GPIB chip is enabled online.  If v is zero then it is
 * left disabled and offline.  Ibonl is called on the first open.
 */
static ibonl(v)
{
	char s;

	semTake(&IbSem);
	ib->cfg2 = D_LMR|D_SFL; /* local master reset */
	ib->cfg2 = D_SFL;       /* clear LMR */

	/* 7210 setup */

	s= ib->cptr;            /* clear registers by reading */
	s= ib->isr1;
	s= ib->isr2;
	ib->imr1 = 0;           /* disable all interrupts */
	ib->imr2 = 0;
	ib->spmr = 0;
	ib->adr = 0;            /* set GPIB address; MTA= 100, MLA= 040 */
	ib->adr = HR_ARS | HR_DT | HR_DL;     /* disable secondary addressing */
	ib->admr = HR_TRM1 | HR_TRM0 | HR_ADM0;
	ib->eosr = 0;
	ib->auxmr = ICR | 8;            /* set internal counter register N= 8 */
	ib->auxmr = PPR | HR_PPU;       /* parallel poll unconfigure */
	ib->auxmr = AUXRA | 0;
	ib->auxmr = AUXRB | 0;
	ib->auxmr = AUXRE | 0;

	/* DMAC setup */

	ib->cfg2 = D_SPAC|D_SFL|D_SC;
	ib->cfg1 = (IBIRQ << 5)|D_BRG3|D_DBM;
	ib->ch1.niv = IBVEC ;       /* 50 == interrupt vector */
	ib->ch1.eiv = IBVEC ;
	ib->ch1.ccr = D_EINT;
	ib->ch0.cpr = 3;                /* highest priority */
	ib->ch1.cpr = 3;                /* highest priority */
	ib->ch1.dcr = D_CS|D_IACK|D_IPCL;
	ib->ch0.dcr = D_CS|D_IACK;
	ib->ch1.scr = 0;
	ib->ch1.mfc = D_SUP|D_S24;
	ib->ch1.bfc = D_SUP|D_S24;
	ib->ch0.scr = D_MCU;
	ib->ch0.mfc = D_SUP|D_S24;

	if(v) ib->auxmr = AUX_PON;      /* release pon state to bring online */
	semGive(&IbSem);
	return(OK);
}


/*
 * IBSIC
 * Send IFC for at least 100 microseconds.  Ibsic must be
 * called prior to the first call to ibcmd in order to
 * initialize the bus and enable the interface to leave
 * the controller idle state.
 */
static ibsic()
{
	int i;

	semTake(&IbSem);
	ib->auxmr = AUX_SIFC;   /* assert IFC */
	for(i= 0; i<100; i++) ; /* busy wait >=100us */
	ib->auxmr = AUX_CIFC;   /* clear IFC */
	semGive(&IbSem);
	return(OK);
}

/*
 * IBSRE
 * Send REN true if v is non-zero or false if v is zero.
 */
static ibsre(v)
{
	semTake(&IbSem);
	ib->auxmr = (v? AUX_SREN : AUX_CREN);   /* set or clear REN */
	semGive(&IbSem);
	return(OK);
}

/*
 * IBGTS
 * Go to the controller standby state from the controller
 * active state, i.e., turn ATN off.
 */
static ibgts()
{
	semTake(&IbSem);
	ib->auxmr = AUX_GTS;    /* go to standby */
	semGive(&IbSem);
	return(OK);
}

/*
 * IBCAC
 * Return to the controller active state from the
 * controller standby state, i.e., turn ATN on.  Note
 * that in order to enter the controller active state
 * from the controller idle state, ibsic must be called.
 */
static ibcac()
{
	semTake(&IbSem);
	ib->auxmr = AUX_TCA;    /* assert ATN (asynchronously) */
	semGive(&IbSem);
	return(OK);
}


/*  ibrdone  -- read one byte from the gpib
 */
ibrdone(buffer)
char *buffer;
{
	int count;
	char ch;
	long err_msg;

	/* go to standby to read byte */
	ib->auxmr = AUX_GTS; 
	/* finish any handshake holdoff */
	if (!first_read) ib->auxmr = AUX_FH;
	else first_read = 0;    

	/* don't holdoff */
	ib->auxmr = AUXRA|HR_HLDA;          

	/* check to see if byte is already in DIR */
	r_isr2 |= ib->isr2;
	r_isr1 |= ib->isr1;
	if (r_isr1 & HR_DI) {
		/* read byte from DIR register */
		*buffer = ib->dir;
		r_isr1 &= ~HR_DI;
		/* take control back */
		ib->auxmr = AUX_TCS;   
		return(OK);
	}
	
	/* byte has not been read yet, enable interrupts
	 * and wait for interrupt mail.
	 */
	ib->ch1.ccr = D_EINT; 
	r_imr1 = HR_DIIE;
	ib->imr1 = HR_DIIE; 
	
	/* start watchdog timer */
	if (!no_timeout) {
		wdStart(ib_wd,IB_WD_WAIT,ib_watch,0);  
	} else no_timeout = 0;

	vxPend(&ib_mail,0,&err_msg);

	/* disable interrupts and clear int bits */
	ib->ch1.ccr = 0; 
	ib->imr1 = 0;
	r_imr1 &= ~HR_DI;  

	/* read DIR */
	if (err_msg == NO_ERR)
		*buffer = ib->dir;
	else {
		*buffer = ib->dir;
		*buffer = 0;
	}

	/* clear DI bit and take control */
	r_isr1 &= ~HR_DI;
	ib->auxmr = AUX_TCS; 

	return((err_msg == NO_ERR) ? OK : ERROR);
}



/* ibwrtone -- write one byte to the gpib
 */
ibwrtone(buffer)
char *buffer;
{
	long err_msg;

	/* go to standby, send EOI with byte */
	ib->auxmr = AUX_GTS;
	ib->auxmr = AUX_SEOI;

	/* enable interrupts */
	ib->ch1.ccr = D_EINT; 
	ib->imr1 = HR_DOIE; 

	/*output command */
	ib->cdor =  *buffer;
		/*
		 * wait for interrupt
		 */
	if (!no_timeout) {
		wdStart(ib_wd,IB_WD_WAIT,ib_watch,0);  
	} else no_timeout = 0;

	vxPend(&ib_mail, 0, &err_msg);    

	/*disable interrupts*/
	ib->imr1 = 0;
	ib->ch1.ccr = 0; 
	r_isr1 &= ~HR_DOIE;

	/* resume control */
	ib->auxmr = AUX_TCS;

	return((err_msg == NO_ERR) ? OK : ERROR);
}




/* dump readable registers */
ibregdump ()
{
		register struct  ibregs *b;
		
		b = ib;
		printf("DMAC register dump:\r\n");
		printf("CER 0 : 0x%x\r\n",b->ch0.cer);
		printf("MAR 0 : 0x%x\r\n",b->ch0.mar);
		printf("DAR 0 : 0x%x\r\n",b->ch0.dar);
		printf("BAR 0 : 0x%x\r\n",b->ch0.bar);
		printf("MTC 0 : 0x%x\r\n",b->ch0.mtc);
		printf("BTC 0 : 0x%x\r\n",b->ch0.btc);
		printf("CSR 0 : 0x%x\r\n",b->ch0.csr);
		printf("CCR 0 : 0x%x\r\n",b->ch0.ccr);
		printf("CER 1 : 0x%x\r\n",b->ch1.cer);
		printf("MAR 1 : 0x%x\r\n",b->ch0.mar);
		printf("DAR 1 : 0x%x\r\n",b->ch0.dar);
		printf("BAR 1 : 0x%x\r\n",b->ch0.bar);
		printf("MTC 1 : 0x%x\r\n",b->ch0.mtc);
		printf("BTC 1 : 0x%x\r\n",b->ch0.btc);
		printf("CSR 1 : 0x%x\r\n",b->ch0.csr);
		printf("CCR 1 : 0x%x\r\n",b->ch0.ccr);
		printf("isr1 = 0x%x\r\nisr2 = 0x%x\r\n", r_isr1, r_isr2);
}
