/*
 * Revision 1.1  87/05/07  17:36:49  sdc
 * 
 */
#include "dr.h"
#if NDR > 0 || Ndr > 0

/*
 * This driver is for the Uniworks OS and was ported from the ISI unix driver.
 * ------------------------------------------------------------------
 * This driver is derived from Copyrighted work by, and was developed
 * under the auspices of, Chromatics and Integrated Solutions.
 *
 * 	(C) Copyright 1987 Chromatics and Integrated Solutions
 *
 * This software may not be reproduced, copied, used or disclosed in
 * whole or in part, by any means except as provided in your contract
 * with Chromatics/ISI, or to the limits of your TALENT, which ever
 * is more restrictive.
 * 
 * NOTE BENE:
 * This driver supports two logical CX devices per physical DR11-W
 * interface.  The CX may have two display heads attached.  A properly
 * installed DR driver will have /dev/dr0 and /dev/dr1 as two logical
 * devices with the same controller.  
 *
 *
 * A few words about a DR11-W interface.
 *
 * The DR11-W has cabling (signals) that appear to support a full-duplex
 * connection but in fact is a half-duplex device since there is only
 * one set of DMA registers (address/xfer-count, etc).  In order to use
 * a DR11-W in something other than a strict master/slave relationship
 * a mechanism must be used for one peer to inform the other of its
 * intentions, in other words, strict unsolicited i/o is not available.
 * The function/status lines on the DR11-W are used for this purpose.
 * Their meaning, with respect to the host, is as follows:
 *
 *	ICR_FCN3 (to CX's ISR_ST_A): 0 => CX head 0, 1 => CX head 1
 *	ICR_FCN2:  used to generate attention interrupt at CX
 *	ICR_FCN1 (to CX's ISR_ST_C): 0 => CX to host, 1 => host to CX
 *
 *	ISR_ST_A (from CX's ICR_FCN3): 0 => CX head 0, 1 => CX head 1
 *	ISR_ST_B: not used, the CX's ICR_FCN2 connects to our attention intr.
 *	ISR_ST_C (from CX's ICR_FCN1): 0 => CX to host, 1 => host to CX
 *
 * I/o involves several machinations by each peer in order to negotiate
 * the transfer size, direction, and logical device (head 0 vs. head 1).
 *
 * A write to the CX works something like this:
 *	(1) load work count in DR11-W data reg (for the CX to see)
 *	(2) send attention interrupt to CX
 *	(3) wait for attention interrupt from CX indicating he has read
 *		the word count and will start a DMA read cycle
 *	(4) do DMA write cycle.
 *
 * A read to the CX works like:
 *	(1) wait for attention interrupt from the CX
 *	(2) get word count from DR11-W data reg (put there by the CX)
 *	(3) generate attention to CX indicating we have accepted the
 *		word count.
 *	(4) do DMA read cycle.
 *
 * There are interrupt complications because attention interrupts and
 * DMA end-of-range interrupts may occur simultaneously.  These are
 * appropriately handled (it says here in the fine print).
 *
 *
 * A few words about Ikon 10084 and CX1500 cabling are in order.
 * 
 * Normally, there are six (6) cables involved.  The Diagram on page 3-6 of
 * the "CX Installation and Configuration Manual" (Chromatics p/n 070272)
 * shows that each of the two DR11-W connections has three cables;
 * host DR11-W board to host bulkhead, host bulkhead to CX bulkhead, and
 * CX bulkhead to CX IOP board.  Unfortunately, there are several potential
 * pitfalls.
 * 
 * Often, bulkhead connectors engage a odd/even pin swap.  If two such swaps
 * occur, no problem.  For testing purposes, lets ignore the bulkhead
 * connections and discuss board-to-board cabling.
 * 
 * The Ikon 10084 DR11-W board's connectors correspond exactly to the
 * DEC DR11-W connectors.  The pin labelled on the connector as pin 1
 * (denoted by the etched arrow on the board connectors) corresponds to the
 * DEC pin designator VV.  The fact that DEC letters their pins in reverse
 * order when compared to the physical connectors used is CONFUSING.
 * 
 * A further confusing factor is that the cables are assembled with a
 * stripe on the pin 40 end---not the typical pin 1 end!!!!!!!!!!!
 * This is necessary since the cables's internal groud plane drain
 * wire is at the striped end of the cable and must be terminated to pin 40.
 *
 * DO NOT go by the etched triangle on the cable connectors.  It is
 * best to use a continuity checker between the cable ground plane (it
 * is usually accessible near the ends of the cable) and locate the
 * proper ground pin on the connector next to the striped end.  USE THIS
 * AS A GUIDE, since it is usually the case that one end of the cable will
 * have the connector installed in a reverse fashion.
 * 
 * Now, on each of the two Ikon connectors lets consider pins 1, 2, 39, and 40.
 * Of these four pins, only pin 40 is connected to GROUND.  By using this
 * fact, we can (via visual and ohm-meter check) derive the proper
 * cabling topology:
 * 
 * The CX (of course) does things differently.  PIN 1 on the CX IOP CONNECTOR
 * IS THE GROUND PIN!!!!!!
 * 
 * Thus the cable is installed to that the Ikon's pin 1 goes to the CX IOP's
 * pin 40, and the Ikon's pin 40 goes to the CX IOP's pin 1.  Furthermore
 * these must be connected to the cable's ground plane!!!!!!!!!!!!!
 * 
 * 
 *      IKON 10084				   CX IOP BOARD
 * 	#					#
 * 	#					#		^  Front
 * 	~					~		|   of
 * 	~ +-- Pin 40 is ground			~		|  CX1500
 * 	# |					#		|
 * 	# V   J1				#
 * 	#-----------+				#
 * 	# *40   39* | 				#
 * 	# *38   37* |				#
 * 	~           ~				#
 * 	~           ~				#
 * 	# *4     3* |				#
 * 	# *2     1* |				#
 * 	#-----------+				#
 * 	#					#     P1 on top
 * 	#					#     P2 on bottom (unseen)
 * 	~					#-----------+
 * 	~					# *40   39* |
 * 	# +-- Pin 40 is ground			# *38   37* |
 * 	# |					~           ~
 * 	# V   J2				~           ~
 * 	#-----------+				# *4     3* |
 * 	# *40   39* |				# *2     1* | <- Pin 1 is
 * 	# *38   37* |				#-----------+    GROUND
 * 	~           ~				#
 * 	~           ~				#
 * 	# *4     3* |				#
 * 	# *2     1* |				#
 * 	#-----------+				#
 * 	#					#
 * 	#					#
 * 	~					#
 * 	~     |					#
 * 	#     | Front				#
 * 	#     |  of				#
 * 	#     V ISI				#
 * 
 * There is more,  the J1 connector on the Ikon board must be connected to
 * J2 on the IOP board, thus:
 * 
 * 
 *        Ikon			     CX IOP
 *   -----------+		+-------------
 * 		|		|
 * 	     J1 :-----   -------: P1
 * 		|     \	/	|
 * 		|      X	|
 * 		|     / \	|
 * 	     J2 :-----   -------: P2
 *		|		|
 *   -----------+		+-------------
 * 
 */

#include "/usr/sys/h/types.h"
#include "/usr/sys/machine/param.h"
#include "drreg.h"
#include "/usr/uw/h/vxWorks.h"
#include "/usr/uw/h/ioLib.h"
#include "/usr/uw/h/iosLib.h"
#include "/usr/uw/h/memLib.h"
#include "/usr/uw/h/wdLib.h"
#include "/usr/uw/h/vme68k20.h"


/*
 * macros used in the driver
 */
#define	reg		register
#define DRUNIT(dev)	(minor(dev))	/* minor unit no. from device no. */
#define USERID(a)	(((a)&ISR_ST_A)>>11)

#define TIMEOUTINT	(60 * CLOCK_RATE)/* initial DMA timeout value */

#define MAXUSERS	2		/* number of CX heads */
#define	TOTALUNITS	(MAXUSERS * NDR)	/* total # of logical units */
#define	DRADMD		0x3a		/* dma cycle: std non-priv program */
#define BLOCKSIZ	131068

LOCAL	int	drOpen(), drRead(), drInterrupt(), drWrite(), drClose(), 
		drIoctl(), drTimeout(), settimer(), unsettimer();

LOCAL	int	dr_num;	/* driver number assigned by UniWorks */

#define DMAFREE		0x01
typedef struct {
	BOOL	alive;
	struct drdevice	*draddr;
	ushort	dmastate;
	int	dmauser;
	int	users;		/* number of devices open on controller */
} DR_CONT;
LOCAL DR_CONT drcont[NDR];


/*
 * Software control structure
 */
typedef struct {
	DEV_HDR		dev_header;
	BOOL		created;	
	int		unit;
	int		timeoutint;
	int		timercnt;
	int 		sleepreason;
#		define READREASON		0
#		define WRITEREASON		1
#		define READFINISHREASON		2
#		define RESETREASON		3
#		define WAITREASON 		4
#		define WRITEFINISHREASON	5
#		define IOCTLREASON		6
	int     	state;
#		define DR_CLOSE		0x00
#		define DR_OPEN		0x01
#		define DR_REQUESTED	0x04
#		define DR_DMADONE	0x08
#		define HOST2CX		0x10
#		define DR_TIMEOUT	0x20
#		define DR_GRANTED 	0x40
#		define ATTNDCX		0x80
	int     	read_count;	/* in words */
	WDOG_ID		wd_id;		/* watchdog timer id */
	int		semaphore;		/* task doing I/O on unit */
	DR_CONT		*cont;		/* ptr to ctlr */
}DR_DEV;

DR_DEV	dr_dev[ TOTALUNITS ];

typedef struct {
	u_short	*csr;
	char	vec;
}DEVADDS;

DEVADDS	draddrs[] = {		/* Bus and vector address of DR11-W */
	(u_short *)0xfffe00, (char)0x50, (u_short *)0xffff00, (char)0x51,
	(u_short *)0, (char)0
};



drDrv()

{
register struct drdevice *draddr;
register BOOL someone_alive;
register int	i, j;

/*
 * check for presence of hardware at boot time
 */
	for (i = 0, someone_alive = FALSE; i < NDR; i++)
	{
	/* connect controller to interrupt routine */
		intConnect(draddrs[i].vec << 2, drInterrupt, i);
		draddr = (struct drdevice*)draddrs[i].csr;    
		draddr->dr_intvect = draddrs[i].vec; 	
		draddr->dr_addmod = DRADMD;
		draddr->dr_walo = 0x8000;
		draddr->dr_wahi = 0x00;
		draddr->dr_data = 0x55;		/* garbage pad char */
		if( draddr->dr_cstat & ISR_ATTN ) {
			draddr->dr_range = 1;
			draddr->dr_pulse = IPR_SMSK | IPR_GO;
       		 	draddr->dr_pulse = IPR_CYCL;
		} else {
			draddr->dr_cstat = ICR_MCLR;
			draddr->dr_range = 0x00;
			draddr->dr_cstat = ICR_IENB | ICR_RATN;
			draddr->dr_pulse = IPR_CYCL | IPR_GO;
		}
		vxTdelay(60);

#ifdef DEBUG
	printf( "drDrv: dr_cstat = 0x%x\n", draddr->dr_cstat );
#endif

	/* setup data structure for controller */
		if ((draddr->dr_cstat & ISR_REDY)!=0)
		{
			drcont[i].alive = TRUE;
			drcont[i].draddr = draddr;
			drcont[i].dmastate = DMAFREE;
			drcont[i].dmauser = 0;
			drcont[i].users = 0;
			someone_alive = TRUE;
		}
		else
			drcont[i].alive = FALSE;
	/* initialize structures for each device on controller */
		for (j = 0; j < MAXUSERS; j++)
		{
			dr_dev[i*MAXUSERS+j].created = FALSE;
			dr_dev[i*MAXUSERS+j].timeoutint = TIMEOUTINT;
			dr_dev[i*MAXUSERS+j].timercnt = TIMEOUTINT;
			dr_dev[i*MAXUSERS+j].state = DR_CLOSE;
			dr_dev[i*MAXUSERS+j].cont = &drcont[i];
			dr_dev[i*MAXUSERS+j].unit = i*MAXUSERS+j;
			dr_dev[i*MAXUSERS+j].semaphore = 0;
			if ((dr_dev[i*MAXUSERS+j].wd_id = wdCreate()) == NULL)
				return(ERROR);
		}
	}
	if (someone_alive)
	{
		dr_num = iosDrvInstall(drOpen, (FUNCPTR)NULL, drOpen, 
					drClose, drRead, drWrite, drIoctl);
		return(OK);
	}
	else
		return(ERROR);
}


STATUS drDevCreate(name, unit, rbs, wbs)

register char	*name;		/* name of this device */
register int	unit;		/* unit number for this device */
register int	rbs, wbs;  /* read and write buffer sizes; not used for DR11W */

{

#ifdef DEBUG
	printf( "drDevCreate: name = %s unit = %d\n", name, unit );
#endif

/* if a device exits already don't create it over again */
	if (dr_dev[unit].created)
		return(ERROR);

/* check to see if the controller for the device is present */
	if (!dr_dev[unit].cont->alive)
		return(ERROR);

/* mark the device created and add to I/O system */
	dr_dev[unit].created = TRUE;

/* initialize the semaphore for this logical unit */
	semInit(&dr_dev[unit].semaphore);

	return (iosDevAdd((DR_DEV *)&dr_dev[unit], name, dr_num));
}



LOCAL int drOpen(pdr_dev, name, mode)

register DR_DEV	*pdr_dev;
register char	*name;
register int	mode;

{
register int	ipl;

#ifdef DEBUG
	printf( "drOpen: unit = %d \n", pdr_dev->unit);
#endif

/* should not have a stream identifier */
	if (name[0] != 0)
		return(ERROR);

/* device should not be open */
	if (pdr_dev->state != DR_CLOSE)
		return(ERROR);

/* lock out interrupts while changing software state */
	ipl = intLock();
	pdr_dev->state = DR_OPEN;
	pdr_dev->read_count = 0;
	pdr_dev->timercnt = 0;
	pdr_dev->cont->users++;

/* initialize semaphore for exclusion of device use */
	semInit(&pdr_dev->semaphore);
	intUnlock(ipl);
	
	return((int)pdr_dev);
}



LOCAL drClose(pdr_dev)

register DR_DEV	*pdr_dev;

{
register int	ipl;

#ifdef DEBUG
	printf( "unit %d, drclose called\n", pdr_dev->unit ); 
#endif
	if (pdr_dev->state == DR_CLOSE)
		return(ERROR);
	ipl = intLock();
	pdr_dev->state = DR_CLOSE;
	pdr_dev->cont->users--;
	intUnlock(ipl);
	return(OK);
}


LOCAL drRead(pdr_dev, buffer, nbytes) 

register DR_DEV	*pdr_dev;
register char	*buffer;
register int	nbytes;

{
register int	ipl;

#ifdef DEBUG
	printf( "unit %d, drread called\n", pdr_dev->unit );
#endif

	/*
	 * We must block drintr from diddling with the state vector
	 */

	ipl = intLock();


/* init semaphore to a known state */
	semInit(&pdr_dev->semaphore);
/* 
 * Set up to receive.
 */
#ifdef DEBUG
	printf( "unit %d, attempting dma read\n", pdr_dev->unit );
#endif
	pdr_dev->state &= ~DR_TIMEOUT;

	if (!(pdr_dev->state & DR_GRANTED) ) 
	{
		/*
		 * Wait for int. routine to set readcnt and go.
		 * N.B. - For reads the interrupt service
		 * procedure (drintr) marks "dmasta" as
		 * DMA in use when the attention interrupt is
		 * received.
		 */
#ifdef DEBUG
		printf("sem take wait for readcnt\n");
#endif

		settimer(pdr_dev, READREASON, "readcnt");
		intUnlock(ipl);
		semTake(&pdr_dev->semaphore);
	}
	else
		intUnlock(ipl);

		pdr_dev->state &= ~DR_GRANTED;
		pdr_dev->state &= ~DR_REQUESTED;

		if( !(pdr_dev->state & DR_TIMEOUT) ) {
			unsettimer(pdr_dev);
			return(drgo(pdr_dev, buffer, nbytes, READ));
		} else {
			pdr_dev->state &= ~DR_TIMEOUT;
			return(ERROR);
		}
/*
	intUnlock(ipl);
*/
}

LOCAL drWrite(pdr_dev, buffer, nbytes)

register DR_DEV	*pdr_dev;
register char	*buffer;
register int	nbytes;

{
register	ipl;
register struct	drdevice *draddr;

#ifdef DEBUG
	printf( "unit %d, drwrite called\n", pdr_dev->unit );
#endif

	draddr = pdr_dev->cont->draddr;

	/*
	 * We must block drintr from diddling with the state vector
	 */

	ipl = intLock();


/* init semaphore to a known state */
	semInit(&pdr_dev->semaphore);

	/* 
	 * Set up to transmit.
	 */
#ifdef DEBUG
	printf( "unit %d, attempting dma write\n", pdr_dev->unit );
#endif
	pdr_dev->state &= ~(DR_GRANTED | DR_TIMEOUT);
	pdr_dev->state |= DR_REQUESTED;


	if(!(pdr_dev->cont->dmastate & DMAFREE) && 
	      !(pdr_dev->state & DR_GRANTED)) 
	{
	/*
	 * Wait for the DMA hardware to become available.
	 */
#ifdef DEBUG2
		printf("sem take waiting for dma device DMAFREE = %x, DR_GRANTED = %x\n",
			pdr_dev->cont->dmastate & DMAFREE, 
			pdr_dev->state & DR_GRANTED);
#endif
		settimer(pdr_dev, WAITREASON, "write wait dma");
		intUnlock(ipl);
		semTake(&pdr_dev->semaphore);

	}
	else
		intUnlock(ipl);

	pdr_dev->cont->dmastate &= ~DMAFREE;
	pdr_dev->state &= ~(DR_REQUESTED | DR_GRANTED);

	if (pdr_dev->state & DR_TIMEOUT) 
	{
		pdr_dev->state &= ~DR_TIMEOUT;
		return(ERROR);
	} 
	else 
	{
	/*
	 * Cancel the timeout timer and mark the
	 * DMA hardware (the controller) as in use.
	 */
		unsettimer(pdr_dev);
	/*
	 * Put the desired xfer word count out
	 * for the CX to see.
	 */
		draddr->dr_data = nbytes >> 1;
	/*
	 * Indicate head odd (1,3,5..) vs. head even (2,4,6..)
	 */
		draddr->dr_cstat = (pdr_dev->unit & 1) ? (ICR_IENB | ICR_FCN3) :
			ICR_IENB ;
	/*
	 * Generate an attention interrupt to the CX
	 */
#ifdef DEBUG
		printf("sem take attn interrupt for device %d\n", pdr_dev->unit);
#endif

		draddr->dr_pulse = ICR_FCN2;
		pdr_dev->state |= ATTNDCX;

		/*
		 * Wait for the CX to respond that
		 * he strobed the xfer count and wants
		 * do start the DMA transfer.
		 */

		if( !(pdr_dev->state & HOST2CX))
		{
			settimer(pdr_dev, WRITEREASON, "write strobe xfer");
			semTake(&pdr_dev->semaphore);
		}

#ifdef DEBUG
			printf("wakeup from xfer count wait\n");
#endif

		if ( !(pdr_dev->state & HOST2CX) ) 
		{
		/*
		 * Got a timeout, deallocate
		 * the DMA hardware and declare
		 * the i/o operation complete
		 * with error.
		 */
			pdr_dev->state &= ~DR_TIMEOUT;
			pdr_dev->cont->dmastate |= DMAFREE;
			return(ERROR);
		} 
		else 
		{
			unsettimer(pdr_dev);
			pdr_dev->state &= ~HOST2CX;
			pdr_dev->state &= ~ATTNDCX;
			return(drgo(pdr_dev, buffer, nbytes, WRITE));
		}
	}		
}



drgo(pdr_dev, buffer, nbytes, dir)

register DR_DEV	*pdr_dev;
register char	*buffer;
register int	nbytes;
register int	dir;

{
register struct drdevice   	*draddr;
register unsigned int	bufadr;
register unsigned short		dmareg;
register int	i;


#ifdef DEBUG
	printf( "drgo called for unit %d\n", pdr_dev->unit);
#endif
	/*
	 * after all conditions met, initiate the transfer
	 */
	draddr = pdr_dev->cont->draddr;		/* get controller csr address */
	pdr_dev->cont->dmauser = pdr_dev->unit;

	draddr->dr_cstat = 0;		/* reset the control reg */
	/*
	 * Set DMA address register.
	 */
	draddr->dr_walo = (u_short)((unsigned int)buffer >> 1);
	draddr->dr_wahi = (char)((unsigned int)buffer >> 17);
	/*
	 * Set DMA count register.
	 */
	draddr->dr_range = (u_short)((nbytes >> 1) - 1);
	draddr->dr_addmod = (char)DRADMD;

#ifdef DEBUG
	printf( "transfer count        = 0x%x\n", nbytes );
	printf( "low address read back = 0x%x\n", draddr->dr_ralo );
	printf( "hi address read back  = 0x%x\n", (u_short)draddr->dr_rahi );
	printf( "range read back       = 0x%x\n", draddr->dr_range );
	printf( "address modifier read = 0x%x\n", (u_short)draddr->dr_addmod );
	printf( "interrupt vector read = 0x%x\n", (u_short)draddr->dr_intvect );
#endif
	pdr_dev->state &= ~DR_DMADONE;

	dmareg = (pdr_dev->unit & 1) ? ICR_FCN3 : 0;

	if (dir == READ) {		/* We are reading */
		if( (draddr->dr_range + 1) != pdr_dev->read_count )
			printf( "dr%d word counts do not agree!\n", 
				pdr_dev->unit );

		draddr->dr_cstat = (ICR_FCN1 | ICR_GO | ICR_IENB | dmareg );
		draddr->dr_pulse = ICR_FCN2;
	} else {				/* We are writing */
		draddr->dr_cstat = (ICR_IENB | ICR_GO | ICR_CYCL | dmareg );
	}

	/*
	 * wait for dma finish interrupt
	 */
#ifdef DEBUG
	printf( "unit %d sem take dma done\n", pdr_dev->unit );
#endif

	if (!(pdr_dev->state & DR_DMADONE))
	{
		settimer(pdr_dev, (dir == READ) ?
	  	      READFINISHREASON : WRITEFINISHREASON, "wait dma finish" );
		semTake(&pdr_dev->semaphore);
	}

#ifdef DEBUG2
	printf( "unit %d wake\n", pdr_dev->unit );
	if (dir == READ)
		for (i = 0; i < pdr_dev->read_count*2; i++)
			printf("data word %d = %d\n", i, buffer[i]);
#endif

	if ( pdr_dev->state & DR_TIMEOUT ) {
		pdr_dev->state &= ~DR_TIMEOUT;
		pdr_dev->read_count = 0;
		return(ERROR);
	}
	else
	{
		unsettimer(pdr_dev);
		return(nbytes);
	}
}




LOCAL drInterrupt(ctlr)

reg int			ctlr;

{
	reg struct drdevice	*draddr;

	DR_DEV	*pdr_dev;
	DR_DEV	*otheruser;
	int     		i, status, granted, user, unit;
	unsigned int 		userstates;

	/*
	 * field interrupt:
	 */
	draddr = drcont[ctlr].draddr;
	status = draddr->dr_cstat;

#ifdef DEBUG
	printf( "ctlr %d, drintr....stat = %x\n",
		ctlr, status );
#endif

	granted = 0;
	userstates = 0;

	if(status & ISR_ATTF) 
	{
		/*
		 * Got attention interrupt from CX.  Clear the
		 * interrupt.  If the local DR device is not open,
		 * ignore the interrupt from the CX.
		 */
		draddr->dr_pulse = ICR_RATN;
		if(drcont[ctlr].users == 0)
		{
#ifdef DEBUG
			printf("drInterrupt no devices open\n");
#endif
			return;
		}
	}
	if(status & ISR_DMAF)
	{
		/*
		 * Got DMA end-of-range interrupt.  Clear the
		 * interrupt and cancel the timeout timer.
		 * Indicate the controller is available and wake
		 * up any top level guy that was waiting for
		 * i/o completion.
		 */
		draddr->dr_cstat = ICR_RDMA | ICR_IENB;
		pdr_dev = &dr_dev[drcont[ctlr].dmauser];
		pdr_dev->state |= DR_DMADONE;
		drcont[ctlr].dmastate |= DMAFREE;
/*
#ifdef DEBUG
		printf("sem give dma done\n");
#endif
*/
		semGive(&pdr_dev->semaphore);
	}
	/*
	 * Get the status bit (defined by the USERID macro) what indicates
	 * which CX head is involved.
	 */
	pdr_dev = &dr_dev[USERID(status)];
	otheruser = &dr_dev[!USERID(status)];

	if((status & ISR_ATTF) && !(status & ISR_ST_C)) 
	{
	/*
	 * CX to host (determined by the DR11 status C signal).
	 *
	 * The CX placed the desired word count in the data
	 * register.  This will tell the (currently blocked)
	 * read request topside for how many words to initiate
	 * the DMA transfer.
	 */
		pdr_dev->read_count = draddr->dr_data;
		if( (status & ISR_DMAF) ||
		    (drcont[ctlr].dmastate & DMAFREE) || 
		    (otheruser->state & DR_GRANTED) ) {
		/*
		 * The other head finished his i/o
		 * and we want to use the controller now.
		 */
			otheruser->state &= ~DR_GRANTED;
			drcont[ctlr].dmastate &= ~DMAFREE;
			pdr_dev->state |= DR_GRANTED;
			granted = 1;
/*
#ifdef DEBUG
		printf("sem give readcnt latched\n");
#endif
*/
			semGive(&pdr_dev->semaphore);
		} 
		else 
		{
			if( otheruser->state & ATTNDCX ) 
			{
				draddr->dr_cstat = USERID(status) ?
				    (ICR_FCN2 | ICR_IENB) :
				    (ICR_FCN2 | ICR_IENB | ICR_FCN3 );
			}
			pdr_dev->state |= DR_REQUESTED;
		}
	} 
	else if ((status & ISR_ATTF) && (status & ISR_ST_C)) 
	{
	/*
	 * Host to CX
	 */
		drcont[ctlr].dmastate &= ~DMAFREE;
		pdr_dev->state |= HOST2CX;
/*
#ifdef DEBUG
		printf("sem give host to CX attn\n");
#endif
*/
		semGive(&pdr_dev->semaphore);
	}

	if( (status & ISR_DMAF) && !granted ) {
	/*
	 * The next line says:
	 * Get the adr of the local state entry for other
	 * head reletive to the head currently using the
	 * controller (DR11-W), NOT the other head releative
	 * to who generated the interrupt that got us here.
	 */
		pdr_dev = &dr_dev[drcont[ctlr].dmauser  ^ 1 ];
		if (pdr_dev->state & DR_REQUESTED) 
		{
			drcont[ctlr].dmastate &= ~DMAFREE;
			pdr_dev->state |= DR_GRANTED;
/*
#ifdef DEBUG2
		printf("sem give DMAF and !granted\n");
#endif
*/
			semGive(&pdr_dev->semaphore);
		} 
		else 
		{
		/*
		 * A DMA end-of-range interrupt occured and
		 * the other device (CX head) is not requesting
		 * use of the DMA, so mark it free.
		 */
			drcont[ctlr].dmastate |= DMAFREE;
		}
	}
}


LOCAL drIoctl(pdr_dev, cmd, data )

register DR_DEV	*pdr_dev;
register int   	cmd;
register int	*data;

{
register struct drdevice	*draddr;
register int 			iodata, ipl,  user;

/*
 * provide misc. services
 * 1) user control over timeout interval
 * 2) warmstart the CX
 * 3) coldstart the CX
 * 4) reset the dma hardware - will zap ANY operations in progress
 */

#ifdef DEBUG
	printf("drIoctl - command = %d\n", cmd);
#endif
	iodata = 0;
	draddr = pdr_dev->cont->draddr;

	switch(cmd & 0xff) 
	{

		case DRSETTIMER:
			pdr_dev->timeoutint = *data;
			break;

		case DRGETTIMER:
			*data = pdr_dev->timeoutint;
			break;

		case DRLOCK:
			break;

		case DRUNLOCK:
			break;

		case DRWARMSTART:
		iodata = 0xffff;
		case DRCOLDSTART:
			ipl = intLock();
			pdr_dev->state &= ~DR_GRANTED;
			pdr_dev->state |= DR_REQUESTED;

			if(!(pdr_dev->cont->dmastate & DMAFREE) &&
			      !(pdr_dev->state & DR_GRANTED)) 
			{
				settimer(pdr_dev, IOCTLREASON);
				intUnlock(ipl);

#ifdef DEBUG2
				printf("sem take ioctl start\n");
#endif
				semTake(&pdr_dev->semaphore);
			}

			pdr_dev->state &= ~(DR_GRANTED | DR_REQUESTED);
		/*
		 * Interrupts are disabled and we are not going
		 * block so cheat and mark the DMA free even
		 * though we are about to diddle with it.
		 */
			pdr_dev->cont->dmastate |= DMAFREE;

			if(!(pdr_dev->state & DR_TIMEOUT)) 
			{
				unsettimer(pdr_dev);
				/*
				 * The CX interprets 0 as coldstart and -1
				 * as warmstart request.
				 */
				draddr->dr_data = iodata;
			/*
			 * Odd or even head (unit) determines ICR_FCN3 state.
			 */
				draddr->dr_cstat = (pdr_dev->unit & 1) ?
					(ICR_FCN3 | ICR_IENB) :
					(ICR_IENB);
			/*
			 * Generate attention interrupt.
			 */
				draddr->dr_pulse = ICR_FCN2;
			/*
			 * Wake up anybody who just might now be waiting
			 * on the the "other" logical device
			 * to get use of the DMA hardware.
			 */
#ifdef DEBUG
				printf("sem give ioctl start\n");
#endif
/*
				semGive(&dr_dev[pdr_dev->unit ^ 1].semaphore);
				vxTresume(dr_dev[pdr_dev->unit ^ 1].task);
*/
			} 
			else 
			{
				pdr_dev->state &= ~DR_TIMEOUT;
			}
			intUnlock(ipl);
			break;
	
		case DRRESET:
			pdr_dev->read_count = 0;
			pdr_dev->timeoutint = TIMEOUTINT ;
			pdr_dev->timercnt = 0;
			pdr_dev->sleepreason = RESETREASON;
			drTimeout(pdr_dev);	/* NB - will mark DMA free */
			pdr_dev->state = DR_CLOSE;
			break;
	
	}
	return( 0 );
}


drclr(draddr)

register struct drdevice	*draddr;

{
	draddr->dr_cstat = ICR_MCLR;
	draddr->dr_data = 0;	/* Data register */
}


LOCAL drTimeout(pdr_dev)

register DR_DEV	*pdr_dev;

{
register struct drdevice *draddr;
register int cnt;
static char *reason[] = 
		{
		"read interrupt",
		"write acknowledge",
		"read dma finish",
		"dma manual reset",
		"acquiring dma to write",
		"write dma finish",
		"ioctl to acquire dma",
	};

	draddr = pdr_dev->cont->draddr;

	printf("DMA timeout: dr%d waiting on %s, %d timers remaining\n",
		pdr_dev->unit, reason[ pdr_dev->sleepreason ], 
		pdr_dev->timercnt );

	unsettimer(pdr_dev);

	pdr_dev->state |= DR_TIMEOUT;
	pdr_dev->read_count = 0;

#ifdef DEBUG
	printf("sem give timeout\n");
#endif
	semGive(&pdr_dev->semaphore);
/*
	vxTresume(pdr_dev->task);
*/

	switch(pdr_dev->sleepreason) 
	{

		case READFINISHREASON:
		case WRITEFINISHREASON:
		case RESETREASON:
			drclr( draddr );
			draddr->dr_cstat = ICR_IENB;
			pdr_dev->cont->dmastate = DMAFREE;
			break;
	}
}


LOCAL settimer(pdr_dev, reason, caller)

register DR_DEV	*pdr_dev;	
register int	reason;
char	*caller;

{


	if ( pdr_dev->timercnt > 0 )
	{
		printf("timer already set!\n");
		printf("caller was %s\n", caller);
		return(0);
	}

	pdr_dev->timercnt++;
	wdStart(pdr_dev->wd_id, pdr_dev->timeoutint, drTimeout, pdr_dev);
	pdr_dev->sleepreason = reason;
}


LOCAL unsettimer(pdr_dev)

register DR_DEV	*pdr_dev;	

{

	if ( pdr_dev->timercnt > 0 )
	{
		pdr_dev->timercnt--;
		wdCancel(pdr_dev->wd_id);
	}
}

#endif
