/* kernelLib.c - WRS kernel library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library provides a generic interface to the WRS kernel
management routines.  Kernel management includes:

    - initialize the kernel,
    - start the kernel,
    - get the version number of the kernel,
    - turn on/off round-robin (time slice) scheduling.

SEE ALSO: "Architecture", taskLib (1), tickLib (1)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "wind.h"

IMPORT ULONG vxTicks;		/* global concept of time */
IMPORT BOOL  contextSwitching;	/* context switching state */
IMPORT int   intLockLevel;	/* interrupt lock out level */

#define MAX_WIND_JOBS	50	/* the number of jobs that can be q'ed*/
#define MEM_TAG		32	/* this is the size of the memory tag */
				/* placed at the end of memory by memInit */


/* global variables */

char *vxIsp;				/* interrupt stack pointer */
char *ispBot;				/* bottom of interrupt stack pointer */
JOB jobPool [MAX_WIND_JOBS];		/* memory for job list */
 
LIST activeList;        		/* active list of TCBs*/
LIST readyList;         		/* ready list of TCBs */
LIST delayList;         		/* delayed list of TCBs */
LIST workList;          		/* list of jobs */
LIST freeJobList;       		/* free list of jobs */

/* local variables */

LOCAL int usrTaskId;

/* forward declarations */

LOCAL VOID kernelRoot ();			/* the kernel root routine */


/*******************************************************************************
*
* kernelInit - initialize kernel
*
* This routine initializes and starts the kernel.  It should only be called
* once.  The routine `rootRtn' is the user's startup code that subsequently
* installs system facilities (eg. initialize the I/O system),
* spawns tasks, etc..
*
* INTERNAL
* The routine kernelRoot is called before the user's root routine so that
* memory management can be initialized.
*
* ARGSUSED
*/

VOID kernelInit (trapNum, rootRtn, rootStackSize, memPoolStart,
		 memPoolEnd, intStackSize, lockOutLevel)
    int trapNum;		/* kernel trap number             */
    FUNCPTR rootRtn;		/* user startup routine           */
    int rootStackSize;		/* stack size for startup routine */
    char *memPoolStart;		/* beginning of memory            */
    char *memPoolEnd;		/* end of memory                  */
    int intStackSize;		/* interrupt stack size           */
    int lockOutLevel;		/* interrupt lock out level (1-7) */

    {
    char *bottomOfStack;
    int job;

    /* initialize WRS kernel interrupt lock out level */
    intLockLevel = lockOutLevel;

    /* initialize the kernel's lists */
    lstInit (&activeList);
    lstInit (&readyList);
    lstInit (&delayList);
    lstInit (&workList);
    lstInit (&freeJobList);

    /* initialize the the dead list of jobs */

    bzero ((char *) jobPool, sizeof (JOB) * MAX_WIND_JOBS);	/* zero jobs */

    for (job = 0; job < MAX_WIND_JOBS; job++)
	lstAdd (&freeJobList, &jobPool [job].node);

    /* create a dummy task for calls made from usrInit */
    usrTaskId = windDummyCreate ();

    /* initialize the time to zero */
    vxTicks = 0;				/* good morning */

    /* initialize context switching to on */
    contextSwitching = TRUE;			/* allow context switching */

    /* memory setup is as follows:
     *
     * ------------------------ <--- memPoolEnd		HIGH MEMORY
     * |                      |
     * |     memPoolTag       |   We have to leave room for this tag so we
     * |     < 32 bytes       |   can add these areas to the free mem pool
     * ------------------------   without clobbering the TCB, which would
     * |                      |   make it impossible to delete the root task.
     * |     TCB & TCBX       |
     * |                      |
     * ------------------------ <--- bottomOfStack;
     * |                      |
     * |      ROOT STACK      |   rootStackSize includes the memPoolTag and TCBX
     * |                      |
     * ------------------------
     * ------------------------ <--- tempMemTop;
     * |                      |
     * -   FREE MEMORY POOL   -
     * |                      |	
     * ------------------------ <--- memPoolStart + intStackSize;
     * |                      |	  
     * |   INTERRUPT STACK    |	  
     * |                      |	 
     * ------------------------ <--- memPoolStart      LOW  MEMORY
     */ 
    
    vxIsp  = memPoolStart;
    ispBot = (char *) ((int) memPoolStart + intStackSize);
    bfill (vxIsp, (int) intStackSize, 0xee);	/* to determine stack usage */
    windIntStackSet ((int) ispBot);

    bottomOfStack = (char *) (memPoolEnd - (ULONG) sizeof (WIND_TCB) - MEM_TAG);

    /* spawn the root task to do the rest of the work */

    taskActivate (taskCreate ("rootTask", 0, VX_SUPERVISOR_MODE, bottomOfStack,
			      rootStackSize - sizeof (WIND_TCB) - MEM_TAG,
			      (WIND_TCB *) bottomOfStack, kernelRoot,
			      (int) memPoolStart + intStackSize,
			      (int) memPoolEnd, rootRtn,
			      rootStackSize));

    taskSuspend (0);	/* just in case we ever run */
    }
/*******************************************************************************
*
* kernelVersion - get kernel version
*
* RETURNS: pointer to string of format `kernel version xxx.xxx'
*/

char *kernelVersion ()

    {
    return ("WIND version 1.0");
    }
/*******************************************************************************
*
* kernelTimeSlice - enable round-robin select
*
* This routine enables round-robin selection among tasks of same priority.
*
* NOTE:
* Not implemented in the WRS kernel, yet.
*
* RETURNS: ERROR (not implemented)
*
* ARGSUSED
*/

STATUS kernelTimeSlice (ticks)
    int ticks;		/* number of ticks for each task */

    {
    /* NOT IMPLEMENTED YET */
    return (ERROR);
    }
/******************************************************************************
*
* kernelRoot - temp root
*
* This routine is the first task to run under UniWorks.  In initializes the
* memory management, calls a user specified root routine (usually usrRoot(1)),
* and then cleans up after itself.
*/  

LOCAL VOID kernelRoot (memPoolStart, memPoolEnd, rootRtn, rootStackSize)
    char *memPoolStart;
    ULONG memPoolEnd;
    FUNCPTR rootRtn;
    ULONG rootStackSize;

    {
    ULONG tempMemTop = memPoolEnd - rootStackSize;

    /* initialize the memory pool.
     * rootStackSize bytes at the top of memory are NOT added
     * to the pool now, because kernelRoot's stack is at the top of memory.
     * Our last act, below, will be to add that missing top of memory to
     * the pool.
     */

    memInit (memPoolStart, (unsigned) (tempMemTop - (int)memPoolStart));

    (* rootRtn) ();				/* call rootRtn */

    taskDelete (usrTaskId);	/* kill usrInit pseudo task */

    /* add to memory pool our stack, which was excluded from initial
     * pool in memInit above.  We assume (hope!) that this memory will not
     * be allocated until after we have exited and gone away.
     */

    memAddToPool ((char *) tempMemTop, (unsigned) rootStackSize);

    taskDelete (0);		/* commit suicide */
    }
