/* ioLib.c - I/O interface library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library contains the interface to the I/O system.  Calls to the
routines in this module cause calls to the corresponding routines in the
driver for a particular device.  Those driver routines do all the actual
work, so much of what actually transpires during a read, write, etc.  is
device dependent.  If the drivers adhere to convention, however, the
results should be consistent from device to device.

INCLUDE FILE: ioLib.h

SEE ALSO: "I/O System", iosLib (1)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "memLib.h"
#include "ioLib.h"
#include "iosLib.h"
#include "strLib.h"
#include "taskLib.h"

IMPORT char ioDefPath [MAX_FILENAME_LENGTH];	/* current default i/o prefix */

#define STD_VALID(fd)   (((fd) >= 0) && ((fd) < 3))

LOCAL int ioStdFd [3];		/* global standard input/output/error */


/*******************************************************************************
*
* creat - create a file
*
* Create a new file of the specified name, and open it with the specified
* mode.  This routine figures out which device the file is to be
* created on, then calls that device driver's create routine to do
* most of the actual work.  Therefore, much of what really transpires
* here is device/driver dependent.
*
* The `mode' parameter is used for the file descriptor, and is set as either
* READ, WRITE, or UPDATE for the duration that the file is opened.
* To create NFS files with a particular file mode of the UNIX chmod (2) type,
* call open() with the file mode specified in the third argument.
*
* A driver's create routine will return FOLLOW_LINK if any part of the file name
* contains a link (in the directory path).  In this case, it will also
* have changed the name of the file being opened to incorporate the name of
* the link.  The new file name is then repeatedly resubmitted to the driver's
* open routine until all links are resolved.
*
* RETURNS:
*    File descriptor number, or
*    ERROR if no filename specified,
*	      no such device,
*             no more fd's available (see iosInit),
*             or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*/

int creat (name, mode)
    char *name;		/* name of the file to create */
    int mode;		/* file creation mode */

    {
    DEV_HDR *pDevHdr1;
    DEV_HDR *pDevHdr2;
    int value;
    int fd;
    char fullFileName [MAX_FILENAME_LENGTH];
    char *pPartFileName;
    char *pName;

    /* don't allow null filename (for user protection) but change
     * "." into what null filename would be if we allowed it */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (strcmp (name, ".") == 0)
	++name;		/* point to EOS (turn it into null filename) */

    if (ioFullFileNameGet (name, &pDevHdr1, fullFileName) == ERROR)
	return (ERROR);

#ifdef	is68k
    fd = iosFdGetFree();
    iosFdSetName (fd, name);
    if (fd == ERROR)
	return(ERROR);
    iosFdSet (fd, pDevHdr1, NULL, 0);
#else	is68k
    if ((fd = iosFdNew (pDevHdr1, (char *) NULL, 0)) == ERROR)
	return (ERROR);
#endif	is68k

    value = iosCreate (pDevHdr1, fullFileName, mode);

    if (value == ERROR)
	{
	iosFdFree (fd);
	return (ERROR);
	}

    while (value == FOLLOW_LINK)
	{
	/* if a file name contains a link, the driver's create routine changed
	 * fullFileName to incorporate the link's name. Try to create the file
	 * that the driver's create routine handed back.
	 */

	if ((pDevHdr2 = iosDevFind (fullFileName, &pPartFileName)) == NULL)
	    return (ERROR);

	if (fullFileName != pPartFileName)
	    {
	    /* link file name starts with a UniWorks device name,
	     * possibly a different device from the current one.
	     */

	    strcpy (fullFileName, pPartFileName);
	    }
	else
	    {
	    /* link file name does not start with a UniWorks device name.
	     * create the file on the current device.
	     */

	    pDevHdr2 = pDevHdr1;
	    }

	if ((value = iosCreate (pDevHdr2, fullFileName, mode)) == ERROR)
	    {
	    iosFdFree (fd);
	    return (ERROR);
	    }
	} /* while */

    pName = malloc ((unsigned) (strlen (name) + 1));
    strcpy (pName, name);

    iosFdSet (fd, pDevHdr1, pName, value);

    return (fd);
    }
/*******************************************************************************
*
* delete - delete a file
* 
* Delete the specified file.  Delete calls the device driver's delete
* routine for the particular device on which the file is located to do
* all the actual work.
*
* RETURNS:
*    OK if there is no delete routine for the device, or driver return OK,
*    ERROR if there is no such device, or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*/

STATUS delete (name)
    char *name;		/* name of the file to delete */

    {
    DEV_HDR *pDevHdr;
    char fullFileName [MAX_FILENAME_LENGTH];

    /* don't allow null filename (for user protection) */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (ioFullFileNameGet (name, &pDevHdr, fullFileName) == ERROR)
	return (ERROR);

    return (iosDelete (pDevHdr, fullFileName));
    }
/*******************************************************************************
*
* open - open a file
*
* Open the file for reading, writing, or updating (reading and writing)
* as specified by the flags argument, and returns a descriptor for that file.
* The flags argument may indicate the file is to be created if it does not 
* already exist (by specifying the O_CREAT flag), in which case the file is
* created with mode 'mode' as described in the UNIX definition of "chmod".
* "flags" values are constructed by ORing flags form the following list
* (only one of the first three flags below may be used):
*
*	READ    (or O_RDONLY)	Open for reading only.
*	WRITE   (or O_WRONLY)	Open for writing only.
*	UPDATE  (or O_RDWR)	Open for reading and writing.
*	O_CREAT
*
* A driver's open routine will return FOLLOW_LINK if any part of the file name
* contains a link (directory path or file name).  In this case, it will also
* have changed the name of the file being opened to incorporate the name of
* the link.  The new file name is then repeatedly resubmitted to the driver's
* open routine until all links are resolved.
*
* RETURNS:
*    File descriptor number, or
*    ERROR if no filename specified,
*	      no such device,
*	      no more fd's (see iosInit (2)),
*             or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*
* VARARGS2
*/

int open (name, flags, mode)
    char *name;		/* name of the file to open */
    int flags;		/* READ, WRITE, UPDATE, or O_CREAT */
    int mode;		/* mode of file or directory to be created
			   (UNIX chmod style).  Only useful for file systems.
			   Use 0 to get default values. */

    {
    DEV_HDR *pDevHdr1;
    DEV_HDR *pDevHdr2;
    int value;
    int fd;
    char fullFileName [MAX_FILENAME_LENGTH];
    char *pPartFileName;
    char *pName;

    /* don't allow null filename (for user protection) but change
     * "." into what null filename would be if we allowed it */

    if (name[0] == EOS)
	{
	errnoSet (S_ioLib_NO_FILENAME);
	return (ERROR);
	}

    if (strcmp (".", name) == 0)
	++name;		/* point to EOS (turn it into null filename) */

    if (ioFullFileNameGet (name, &pDevHdr1, fullFileName) == ERROR)
	return (ERROR);

#ifdef	is68k
    fd = iosFdGetFree();
    iosFdSetName (fd, name);
    if (fd == ERROR)
	return(ERROR);
    iosFdSet (fd, pDevHdr1, (char *) NULL, 0);
#else	is68k
    if ((fd = iosFdNew (pDevHdr1, (char *) NULL, 0)) == ERROR)
	return (ERROR);
#endif	is68k

    value = iosOpen (pDevHdr1, fullFileName, flags, mode);

    if (value == ERROR)
	{
	iosFdFree (fd);
	return (ERROR);
	}

    while (value == FOLLOW_LINK)
	{
	/* if a file name contains a link, the driver's open routine changed
	 * fullFileName to incorporate the link's name. Try to open the file
	 * that the driver's open routine handed back.
	 */

	if ((pDevHdr2 = iosDevFind (fullFileName, &pPartFileName)) == NULL)
	    return (ERROR);

	if (fullFileName != pPartFileName)
	    {
	    /* link file name starts with a UniWorks device name,
	     * possibly a different device from the current one.
	     */

	    strcpy (fullFileName, pPartFileName);
	    }
	else
	    {
	    /* link file name does not start with a UniWorks device name.
	     * open the file on the current device.
	     */

	    value = iosOpen (pDevHdr1, fullFileName, flags, mode);
	    pDevHdr2 = pDevHdr1;
	    }

	if ((value = iosOpen (pDevHdr2, fullFileName, flags, mode)) == ERROR)
	    {
	    iosFdFree (fd);
	    return (ERROR);
	    }
	} /* while */

    pName = malloc ((unsigned) (strlen (name) + 1));
    strcpy (pName, name);

    iosFdSet (fd, pDevHdr1, pName, value);

    return (fd);
    }
/*******************************************************************************
*
* close - close a file
*
* Close the specified file.
* Calls the device driver's close routine to do the actual work,
* therefore, what really transpires here is device dependent.
* The file descriptor is free'd.
*
* RETURNS:
*    status of driver's close routine, or
*    ERROR if invalid fd
*
* SEE ALSO: "I/O System"
*/

STATUS close (fd)
    int fd;		/* fd of file to close */

    {
    return (iosClose (fd));
    }
/*******************************************************************************
*
* read - read bytes from a file or device
*
* Reads some number of bytes (less than or equal to maxbytes) from the
* specified stream.  Calls the device driver to do all the actual work.
* 
* RETURNS:
*    Number of bytes actually read;
*        between 1 and maxbytes, 0 if end of file.
*    ERROR if no such open fd, or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*/

int read (fd, buffer, maxbytes)
    int fd;		/* fd of file from which to read */
    char *buffer;	/* pointer to buffer to receive bytes */
    int maxbytes;	/* max number of bytes to read into buffer */

    {
    return (iosRead (fd, buffer, maxbytes));
    }
/*******************************************************************************
*
* write - write bytes to a file
*
* Writes nbytes from buffer to fd.
* Calls the driver write routine to do all the actual work.
* Actual operation is driver dependent.
*
* RETURNS:
*    Number of bytes actually written;
*        if not equal to nbytes, some error has occurred.
*    ERROR if no such open fd, or driver returns ERROR.
*
* SEE ALSO: "I/O System"
*/

int write (fd, buffer, nbytes)
    int fd;		/* fd of file to which to write */
    char *buffer;	/* pointer to buffer from which to
			   write bytes */
    int nbytes;		/* number of bytes to write */

    {
    return (iosWrite (fd, buffer, nbytes));
    }
/*******************************************************************************
*
* ioctl - do file specific control function
*
* Performs a device-specific special function on a device.  Most
* requests are passed on to the driver, which actually handles them
* as it wishes.  One request is handle directly at the I/O interface level:
*
*   ioctl (fd, FIOGETNAME, &nameBuf)
*	return in buffer `nameBuf' the file
*	name of the file descriptor
*
* RETURNS:
*    ERROR if no such open fd, or
*    whatever the driver returns.
*
* SEE ALSO: "I/O System"
*
* VARARGS2
*/

int ioctl (fd, function, arg)
    int fd;		/* fd of file to control	*/
    int function;	/* function code		*/
    int	arg;		/* arbitrary argument		*/

    {
    return (iosIoctl (fd, function, arg));
    }
/*******************************************************************************
*
* lseek - set file read/write pointer
*
* Sets the file read/write pointer of file 'fd' to 'offset'.
* The `whence' has three values which affect the file position pointer:
*
*      L_SET  - set to `offset',
*      L_INCR - set to current position plus `offset',
*      L_XTND - set to the size of the file plus `offset'.
*
* NOTE: L_XTND is not implemented.
*
* This routine uses ioctl (2) with functions FIOWHERE and FIOSEEK.
*
* RETURNS:
*    the new offset, or ERROR
*
* ARGSUSED
*/

int lseek (fd, offset, whence)
    int fd;		/* fd of file to seek */
    long offset;	/* new byte offset to seek to */
    int whence;		/* relative file position */
    
    {
    int where, howmany;

    switch (whence)
	{
	case L_SET:
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	case L_INCR:
	    if (ioctl (fd, FIOWHERE, &where) == ERROR)
		return (ERROR);

	    offset += where;
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	case L_XTND:
	    if (ioctl (fd, FIOSEEK, 0L) == ERROR)
		return (ERROR);
	    if (ioctl (fd, FIONREAD, &howmany) == ERROR)
		return (ERROR);
	    offset += howmany;
	    return ((ioctl (fd, FIOSEEK, offset) == OK) ? offset : ERROR);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* ioFullFileNameGet - get the full file name using the current directory
*
* The device header gets returned, along with the complete name of the
* file relative to that device (i.e. the device name is not in the returned
* file name).
*/

STATUS ioFullFileNameGet (partName, ppDevHdr, fullFileName)
    char *partName;	/* original name */
    DEV_HDR **ppDevHdr;	/* pointer to device header that gets returned */
    char *fullFileName;	/* resulting complete file name */

    {
    char *pPathName;
    char defDir [MAX_FILENAME_LENGTH];

    if ((*ppDevHdr = iosDevFind (partName, &pPathName)) == NULL)
	return (ERROR);

    if (partName != pPathName)
	{
	/* partName starts with a device name, pass on the directory */

	strcpy (fullFileName, pPathName);
	}
    else
	{
	/* use default path */

	ioDefDirGet (defDir);
	pathCat (defDir, partName, fullFileName);
	}
    return (OK);
    }

/*******************************************************************************
*
* ioDefPathSet - set current default path
*
* This routine sets up the default I/O path.
* Path name must start with an existing device's name.
*
* SEE ALSO: ioDefPathGet (2)
*/

STATUS ioDefPathSet (name)
    char *name;		/* Name of the new default device */

    {
    char *pTail = name;

    if (iosDevFind (name, &pTail) == NULL)
	return (ERROR);

    if (pTail == name)
	{
	/* name does not begin with an existing device's name */

	errnoSet (S_ioLib_NO_DEVICE_NAME_IN_PATH);
	return (ERROR);
	}

    strncpy (ioDefPath, name, sizeof (ioDefPath));
    return (OK);
    }

/*******************************************************************************
*
* ioDefPathGet - get current default directory
*
* Copies the name of the current default path to the character array
* passed as a parameter.
*
* SEE ALSO: ioDefPathSet (2)
*/

VOID ioDefPathGet (name)
    char *name;		/* where to return the name */

    {
    strncpy (name, ioDefPath, sizeof(ioDefPath));
    }
/*******************************************************************************
*
* ioDefDevGet - get current default device
*
* Copies the name of the current default device to the character array
* passed as a parameter.
*/

VOID ioDefDevGet (devName)
    char *devName;		/* where to return the device name */

    {
    char *pTail = devName;
    FAST int devNameLen;			/* device name length */

    /* find the device name in the default path name */

    if (iosDevFind (ioDefPath, &pTail) == NULL)
	{
	/* no default device, return null device name */
	*devName = EOS;
	}
    else
	{
	/* extract device name from path name */

	devNameLen = (int) (pTail - ioDefPath);
	bcopy (ioDefPath, devName, devNameLen);
	devName [devNameLen] = EOS;
	}
    }
/*******************************************************************************
*
* ioDefDirGet - get current default directory
*
*/

VOID ioDefDirGet (dirName)
    char *dirName;		/* where to return the directory name */

    {
    char *pTail = dirName;

    /* find the directory name in the default path name */

    if (iosDevFind (ioDefPath, &pTail) == NULL)
	{
	/* no default device, return null directory name */

	*dirName = EOS;
	}
    else
	strcpy (dirName, pTail);
    }

/*******************************************************************************
*
* ioGlobalStdSet - set underlying global fd for standard input/output/error
*
* If `stdFd' is not 0, 1, or 2 then this routine has no effect.
*/

VOID ioGlobalStdSet (stdFd, newFd)
    int stdFd;	/* either 0, 1, or 2 */
    int newFd;	/* new underlying fd */

    {
    if (STD_VALID (stdFd))
	ioStdFd [stdFd] = newFd;
    }
/*******************************************************************************
*
* ioGlobalStdGet - get underlying global fd for standard input/output/error
*
* RETURNS:
*    underlying fd for task, or
*    ERROR if `stdFd' is not 0, 1, or 2
*/

int ioGlobalStdGet (stdFd)
    int stdFd;	/* either 0, 1, or 2 */

    {
    return (STD_VALID (stdFd) ? ioStdFd [stdFd] : ERROR);
    }
/*******************************************************************************
*
* ioTaskStdSet - set underlying task fd for standard input/output/error
*
*
* The console file descriptor as set by ioGlobalStdSet, is ordinarily
* the input, output and error fd used by a task.
* If a particular task should use different input, output, and error fd's
* use this routine to change them.
* If `stdFd' is not 0, 1, or 2 then this routine has no effect.
*
* This routine has no effect if called at interrupt level.
*
* SEE ALSO: ioGlobalStdGet (2), ioTaskStdGet (2)
*
*/

VOID ioTaskStdSet (taskId, stdFd, newFd)
    int taskId; /* id of task whose std fd is to be set (0 = self) */
    int stdFd;	/* either 0, 1, or 2 */
    int newFd;	/* new underlying fd */

    {
    TCBX *pTcbX;

    if (STD_VALID (stdFd) && (pTcbX = taskTcbX (taskId)) != NULL)
	pTcbX->taskStd [stdFd] = newFd;
    }
/*******************************************************************************
*
* ioTaskStdGet - get underlying task fd for standard input/output/error
*
* RETURNS:
*    underlying fd, or
*    ERROR if `stdFd' is not 0, 1, or 2, or
*    if called at interrupt level
*/

int ioTaskStdGet (taskId, stdFd)
    int taskId; /* id of desired task (0 = self) */
    int stdFd;	/* either 0, 1, or 2 */

    {
    TCBX *pTcbX;
    int taskFd;

    if (STD_VALID (stdFd) && (pTcbX = taskTcbX (taskId)) != NULL)
	{
	taskFd = pTcbX->taskStd [stdFd];
	if (STD_VALID (taskFd))
	    return (ioStdFd [taskFd]);
	else
	    return (taskFd);
	}
 
    return (ERROR);
    }
