/* unixLib.c - unix kernal compatability library */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library provides routines that simulate or replace Unix kernel functions
that are used in the network code.  This includes the spl...() processor
level locking routines, sleep() and wakeup(), perror() and panic(),
and a checksum routine.
*/

#include "semLib.h"
#include "UniWorks.h"
#include "types.h"
#include "mbuf.h"
#include "in.h"
#include "in_systm.h"

IMPORT ULONG taskIdCurrent;	/* id of cur task, faster than taskIdSelf() */

/* GLOBAL VARIABLES */

SEM_ID splSemId;	/* the semaphore used by splnet, splimp, etc */

int splTid;		/* task id of the process who has the 
			 * network processor level semaphore */

/******************************************************************************
*
* splnet - set network processor level
*
* splnet may be used to insure mutual exclusion among processes which
* share common network data structures.  In unixland, this is done by
* setting the processor interrupt level.  Here in UniWorksland, however,
* we use a simple semaphore mechanism.  splx should be called after
* completion of the critical section.  We must check first to see if we're 
* already at network level.
*
* RETURNS: previous "processor level"
*/

int splnet ()

    {
    if (taskIdCurrent == splTid)
	return (1);

    semTake (splSemId);
    splTid = taskIdCurrent;
    return (0);
    }
/******************************************************************************
*
* splimp - set imp processor level
*
* This routine is intentical to splnet().
* Currently, all network interlocking uses the same network semaphore.
*
* RETURNS: previous "processor level"
*/

int splimp ()

    {
    if (taskIdCurrent == splTid)
	return (1);

    semTake (splSemId);
    splTid = taskIdCurrent;
    return (0);
    }
/******************************************************************************
*
* splx - set processor level
*
* splx is used in unix to restore the processor level.
* It is used in UniWorks in conjunction with splnet() and splimp() to provide
* mutual exclusion of network data structures.  splx marks the end of the
* critical section.
*/

VOID splx (x)
    int x;	/* processor level to restore */
    {
    if (x == 0)
	{
	splTid = 0;
	semGive (splSemId);
	}
    }

/***********************************************************************
*
* sleep -  put a process to sleep
*
* In Unix, processes can sleep on any value (typically an address).
* In UniWorks, we substitute a semaphore mechanism so sleep()
* calls have to be changed to provide the address of a semaphore.
*/

VOID sleep (semId)
    SEM_ID semId;

    {
    BOOL hadSplSem;

    /* first see if we've got the network semaphore (splSemId);
     * if so, give it, and then take it back when we are awakened. */

    hadSplSem = (splTid == taskIdCurrent);
    if (hadSplSem)
	{
	splTid = 0;
	semGive (splSemId);
	}

    semTake (semId);

    if (hadSplSem)
	{
	semTake (splSemId);
	splTid = taskIdCurrent;
	}

    }

/******************************************************************************
*
* wakeup - wakeup a sleeping process
*/

VOID wakeup (semId)
    SEM_ID semId;
    {
    semGive (semId);
    }

/******************************************************************************
*
* in_cksum - Checksum routine for Internet Protocol family headers 
*/

in_cksum(m, len)
    FAST struct mbuf *m;
    FAST int len;
    {
    FAST u_short sum = 0;
    FAST int mlen = 0;
    FAST int slen = 0;

    while ((m != NULL) && (len != 0))
	{
	if ((mlen = m->m_len) > 0)
	    {
	    if (len < mlen)
		mlen = len;
	    sum = cksum (sum, mtod(m, u_short *), mlen, slen);
	    slen += mlen;
	    len -= mlen;
	    }
	m = m->m_next;
	}

    if (len != 0)
	logMsg ("cksum: out of data\n");

        sum = (~sum) & 0xFFFF;
        return (sum);
    }

/******************************************************************************
*
* perror - print error
*
* perror prints the given string plus the calling task's taskid and current
* status value on stderr.
*
*/

VOID perror (s)
    char *s;
    {
    printErr ("\nt%d: %s; status=0x%x\n", taskIdSelf (), s, errnoGet ());
    }

/******************************************************************************
*
* panic - something is very wrong
*
* panic simply logs a message to the console prepended with "panic".
* If the panic occurs at task level, the current task is suspended.
*/

VOID panic (msg)
    char *msg;
    {
    logMsg ("panic: %s\n", msg);

    if (!intContext ())
	taskSuspend (0);
    }
