/* uipc_mbuf.c - mbuf routines */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
 * Copyright (c) 1987 Wind River Systems, Inc.
 * Copyright (c) 1982, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * and the VxWorks Software License Agreement specify the terms and
 * conditions for redistribution.
 *
 *	@(#)uipc_mbuf.c	7.1 (Berkeley) 6/5/86
 */

/*
modification history
--------------------
*/

#include "UniWorks.h"
#include "param.h"
#include "memLib.h"
#include "mbuf.h"
#include "systm.h"

struct mbstat mbstat;
struct mbuf *mfree;

/*************************************************************************
* 
* mbinit - initialize mbuf package
*
* mbufs must reside on a 128 byte boundry, so we overallocate, and align.
*
*/

mbinit()
    {
    int s;
    int i;
    register struct mbuf *m;
    char *pool;

    s = splimp();

    bzero ((char *) &mbstat, sizeof (mbstat));

    pool = malloc ((INIT_FREE_MBUFS + 1) * MSIZE);
    if (pool == NULL)
	panic ("mbinit");
    pool  = (char *) ((int)pool + MSIZE - ((int) pool & (MSIZE - 1)));

    for (i = 0; i < INIT_FREE_MBUFS; i++)
	{
	m = (struct mbuf *) pool;
	pool += MSIZE;
	m->m_off = 0;
	m->m_type = MT_DATA;
	mbstat.m_mtypes[MT_DATA]++;
	mbstat.m_mbufs++;
	(void) m_free(m);
	}

    splx(s);
    }

/*********************************************************************
*
* mbufShow - report mbuf statistics
*
* mbufShow prints on standard out the distribution of mbufs in 
* the network.  Mbufs are a construct used internally in the network.
*
*/

VOID mbufStat ()

    {
#define	NUM_MT_TYPES	14
    static int mt_types [NUM_MT_TYPES] =
	{ MT_FREE, MT_DATA, MT_HEADER, MT_SOCKET, MT_PCB,
	  MT_RTABLE, MT_HTABLE, MT_ATABLE, MT_SONAME, MT_ZOMBIE,
	  MT_SOOPTS, MT_FTABLE, MT_RIGHTS, MT_IFADDR };
    static char mt_names [NUM_MT_TYPES][10] =
	{ "FREE", "DATA", "HEADER", "SOCKET", "PCB",
	  "RTABLE", "HTABLE", "ATABLE", "SONAME", "ZOMBIE",
	  "SOOPTS", "FTABLE", "RIGHTS", "IFADDR" };
    int totalMbufs = 0;
    FAST int i;

    printf ("type        number\n");
    printf ("---------   ------\n");

    for (i = 0; i < NUM_MT_TYPES; i++)
	{
	printf ("%-8s:    %3d\n", mt_names [i],
		mbstat.m_mtypes [mt_types [i]]);
	totalMbufs += mbstat.m_mtypes[mt_types [i]];
	}

    printf ("%-8s:    %3d\n", "TOTAL", totalMbufs);
    }

/*
 * Must be called at splimp.
 *
 * ARGSUSED
 */
m_expand(canwait)
	int canwait;
{
	register struct mbuf *m;
	int i;
	char *pool;

       pool = malloc ((NUM_MBUFS_TO_EXPAND + 1) * MSIZE);
       if (pool == NULL)
	    panic ("m_expand");

       pool  = (char *) ((int)pool + MSIZE - ((int) pool & (MSIZE - 1)));

       for (i = 0; i < NUM_MBUFS_TO_EXPAND; i++)
	   {
	   m = (struct mbuf *) pool;
	   pool += MSIZE;
	   m->m_off = 0;
	   m->m_type = MT_DATA;
	   mbstat.m_mtypes[MT_DATA]++;
	   mbstat.m_mbufs++;
	   (void) m_free(m);
	   }

	return (1);

}

/* NEED SOME WAY TO RELEASE SPACE */

/*
 * Space allocation routines.
 * These are also available as macros
 * for critical paths.
 */
struct mbuf *
m_get(canwait, type)
	int canwait, type;
{
	register struct mbuf *m;

	MGET(m, canwait, type);
	return (m);
}

struct mbuf *
m_getclr(canwait, type)
	int canwait, type;
{
	register struct mbuf *m;

	MGET(m, canwait, type);
	if (m == 0)
		return (0);
	bzero(mtod(m, caddr_t), MLEN);
	return (m);
}

struct mbuf *
m_free(m)
	struct mbuf *m;
{
	register struct mbuf *n;

	MFREE(m, n);
	return (n);
}

/*
 * Get more mbufs; called from MGET macro if mfree list is empty.
 * Must be called at splimp.
 */
/*ARGSUSED*/
struct mbuf *
m_more(canwait, type)
	int canwait, type;
{
	register struct mbuf *m;

        m_expand (canwait);
#define m_more(x,y) (panic("m_more"), (struct mbuf *)0)
	MGET(m, canwait, type);
#undef m_more
	return (m);
}

m_freem(m)
	register struct mbuf *m;
{
	register struct mbuf *n;
	register int s;

	if (m == NULL)
		return;
	s = splimp();
	do {
		MFREE(m, n);
	} while (m = n);
	splx(s);
}

/*
 * Mbuffer utility routines.
 */

/*
 * Make a copy of an mbuf chain starting "off" bytes from the beginning,
 * continuing for "len" bytes.  If len is M_COPYALL, copy to end of mbuf.
 * Should get M_WAIT/M_DONTWAIT from caller.
 */
struct mbuf *
m_copy(m, off, len)
	register struct mbuf *m;
	int off;
	register int len;
{
	register struct mbuf *n, **np;
	struct mbuf *top, *p;

	if (len == 0)
		return (0);
	if (off < 0 || len < 0)
		panic("m_copy");
	while (off > 0) {
		if (m == 0)
			panic("m_copy");
		if (off < m->m_len)
			break;
		off -= m->m_len;
		m = m->m_next;
	}
	np = &top;
	top = 0;
	while (len > 0) {
		if (m == 0) {
			if (len != M_COPYALL)
				panic("m_copy");
			break;
		}
		MGET(n, M_DONTWAIT, m->m_type);
		*np = n;
		if (n == 0)
			goto nospace;
		n->m_len = MIN(len, m->m_len - off);
		if (m->m_off > MMAXOFF) {
			p = mtod(m, struct mbuf *);
			n->m_off = ((int)p - (int)n) + off;
		} else
			bcopy(mtod(m, caddr_t)+off, mtod(n, caddr_t),
			    (unsigned)n->m_len);
		if (len != M_COPYALL)
			len -= n->m_len;
		off = 0;
		m = m->m_next;
		np = &n->m_next;
	}
	return (top);
nospace:
	m_freem(top);
	return (0);
}

m_cat(m, n)
	register struct mbuf *m, *n;
{
	while (m->m_next)
		m = m->m_next;
	while (n) {
		if (m->m_off >= MMAXOFF ||
		    m->m_off + m->m_len + n->m_len > MMAXOFF) {
			/* just join the two chains */
			m->m_next = n;
			return;
		}
		/* splat the data from one into the other */
		bcopy(mtod(n, caddr_t), mtod(m, caddr_t) + m->m_len,
		    (u_int)n->m_len);
		m->m_len += n->m_len;
		n = m_free(n);
	}
}

m_adj(mp, len)
	struct mbuf *mp;
	register int len;
{
	register struct mbuf *m;
	register count;

	if ((m = mp) == NULL)
		return;
	if (len >= 0) {
		while (m != NULL && len > 0) {
			if (m->m_len <= len) {
				len -= m->m_len;
				m->m_len = 0;
				m = m->m_next;
			} else {
				m->m_len -= len;
				m->m_off += len;
				break;
			}
		}
	} else {
		/*
		 * Trim from tail.  Scan the mbuf chain,
		 * calculating its length and finding the last mbuf.
		 * If the adjustment only affects this mbuf, then just
		 * adjust and return.  Otherwise, rescan and truncate
		 * after the remaining size.
		 */
		len = -len;
		count = 0;
		for (;;) {
			count += m->m_len;
			if (m->m_next == (struct mbuf *)0)
				break;
			m = m->m_next;
		}
		if (m->m_len >= len) {
			m->m_len -= len;
			return;
		}
		count -= len;
		/*
		 * Correct length for chain is "count".
		 * Find the mbuf with last data, adjust its length,
		 * and toss data from remaining mbufs on chain.
		 */
		for (m = mp; m; m = m->m_next) {
			if (m->m_len >= count) {
				m->m_len = count;
				break;
			}
			count -= m->m_len;
		}
		while (m = m->m_next)
			m->m_len = 0;
	}
}

/*
 * Rearange an mbuf chain so that len bytes are contiguous
 * and in the data area of an mbuf (so that mtod and dtom
 * will work for a structure of size len).  Returns the resulting
 * mbuf chain on success, frees it and returns null on failure.
 * If there is room, it will add up to MPULL_EXTRA bytes to the
 * contiguous region in an attempt to avoid being called next time.
 */
struct mbuf *
m_pullup(n, len)
	register struct mbuf *n;
	int len;
{
	register struct mbuf *m;
	register int count;
	int space;

	if (n->m_off + len <= MMAXOFF && n->m_next) {
		m = n;
		n = n->m_next;
		len -= m->m_len;
	} else {
		if (len > MLEN)
			goto bad;
		MGET(m, M_DONTWAIT, n->m_type);
		if (m == 0)
			goto bad;
		m->m_len = 0;
	}
	space = MMAXOFF - m->m_off;
	do {
		count = MIN(MIN(space - m->m_len, len + MPULL_EXTRA), n->m_len);
		bcopy(mtod(n, caddr_t), mtod(m, caddr_t)+m->m_len,
		  (unsigned)count);
		len -= count;
		m->m_len += count;
		n->m_len -= count;
		if (n->m_len)
			n->m_off += count;
		else
			n = m_free(n);
	} while (len > 0 && n);
	if (len > 0) {
		(void) m_free(m);
		goto bad;
	}
	m->m_next = n;
	return (m);
bad:
	m_freem(n);
	return (0);
}
