/* if_ex.c - excelan network interface */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
 * Copyright (c) 1987 Wind River Systems, Inc.
 * Copyright (c) 1980, 1986 Regents of the University of California.
 * All rights reserved.  The Berkeley software License Agreement
 * and the UniWorks Software License Agreement specify the terms and
 * conditions for redistribution.
 *
 *	@(#)if_ex.c	7.1 (Berkeley) 6/5/86
 */

/*
modification history
--------------------
00a,24oct88,krm  added vdma
... adapted from original Excelan interface by George Powers
*/

#include "UniWorks.h"
#include "vme.h"
#include "param.h"
#include "mbuf.h"
#include "protosw.h"
#include "socket.h"
#include "ioctl.h"
#include "errno.h"
#include "iv68k.h"
#include "memLib.h"

#include "if.h"
#include "route.h"

#ifdef	INET
#include "in.h"
#include "in_systm.h"
#include "in_var.h"
#include "ip.h"
#include "if_ether.h"
#endif

#ifdef NS
#include "../netns/ns.h"
#include "../netns/ns_if.h"
#endif


#include "if_exreg.h"

#include "sysLib.h"
#include "wdLib.h"
#include "iosLib.h"
#include "etherLib.h"
#include "systm.h"

IMPORT VOID ipintr ();

#define DEBUG			/* check for "impossible" events */

#define MAX_EX	4		/* max number of excelan controllers online */

#define	NH2X	4		/* a sufficient number is critical */
#define	NX2H	1		/* this is pretty arbitrary */
#define	EXWATCHINTVL 10		/* call exwatch() every 10 seconds */

#define	INCORE_BASE(p)	((caddr_t)((u_long)(&(p)->xs_h2xhdr) & 0xFFFFFFF0))
#define	RVAL_OFF(unit, n) \
    ((caddr_t)(&(ex_softc[unit]->n)) - INCORE_BASE(ex_softc[unit]))
#define	LVAL_OFF(unit, n) \
    ((caddr_t)(ex_softc[unit]->n) - INCORE_BASE(ex_softc[unit]))
#define	H2XHDR_OFFSET(unit)	RVAL_OFF(unit, xs_h2xhdr)
#define	X2HHDR_OFFSET(unit)	RVAL_OFF(unit, xs_x2hhdr)
#define	H2XENT_OFFSET(unit)	LVAL_OFF(unit, xs_h2xent)
#define	X2HENT_OFFSET(unit)	LVAL_OFF(unit, xs_x2hent)
#define CM_OFFSET(x) ((u_long)(&((x)->xs_cm)) - (u_long)(x)) 


extern char vme_std[];

u_short *exstd[] = {
	(u_short *) &vme_std[0xFF0000],
	(u_short *) &vme_std[0xFE0000],
	(u_short *) &vme_std[0xFF7FE0],
	(u_short *) &vme_std[0xFE7FE0],
	0};

/*
 * Ethernet software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * xs_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface,
 * its address, etc.
 */

struct	ex_softc
    {
    struct	arpcom xs_ac;		/* Ethernet common part */
#define	xs_if	xs_ac.ac_if		    /* network-visible interface */
#define	xs_enaddr xs_ac.ac_enaddr	    /* hardware Ethernet address */
#ifdef DEBUG
    int		xs_wait;
#endif
    caddr_t	readBuf;		/* pointer to read buffer */
    caddr_t	writeBuf;		/* pointer to write buffer */
    int		xs_flags;		/* private flags */
#define	EX_XPENDING	1		    /* xmit rqst pending on EXOS */
#define	EX_STATPENDING	(1<<1)		    /* stats rqst pending on EXOS */
#define	EX_RUNNING	(1<<2)		    /* board is running */
#define EX_SETADDR	(1<<3)		    /* physaddr has been changed */
    struct	ex_msg *xs_h2xnext;	/* host pointer to request queue */
    struct	ex_msg *xs_x2hnext;	/* host pointer to reply queue */
    u_short	xs_h2xhdr;		/* EXOS's request queue header */
    u_short	xs_x2hhdr;		/* EXOS's reply queue header */
    struct	ex_msg xs_h2xent[NH2X];	/* request msg buffers */
    struct	ex_msg xs_x2hent[NX2H];	/* reply msg buffers */
    struct	confmsg xs_cm;		/* configuration message */
    struct	stat_array xs_xsa;	/* EXOS writes stats here */
    BOOL	promiscuous;		/* TRUE = cntrlr receiving all packets,
					 * FALSE = cntrlr address filter on */
    int		ivec;			/* interrupt vector */
    int		ilevel;			/* interrupt level */
    struct	exdevice *devAdrs;	/* device address */
    caddr_t	readWind;		/* vme_std window for read xfers */
    caddr_t	writeWind;		/* vme_std window for write xfers */
    struct 	ex_softc *xs_vdma;	/* vme_std window for xs struct */
    };

LOCAL struct ex_softc *ex_softc [MAX_EX];

#undef EX_RESET
LOCAL int EX_RESET = 0;	/* because using constant causes "clr" inst. */

/* global variables */

int exBusAdrsSpace = VME_AM_STD_SUP_DATA;	/* VME address space that
						 * excelan should use to
						 * access host buffers, etc.
						 * This is normally std space
						 * but we make it global so
						 * it can be changed if
						 * necessary */

/* forward declarations */

int exInt();
LOCAL VOID exinit();
LOCAL int exoutput();
LOCAL int exioctl();
LOCAL VOID exreset();
LOCAL VOID exwatch();
LOCAL VOID exstart ();
LOCAL VOID exRecvAndHang ();
LOCAL struct ex_msg *exgetcbuf();
LOCAL char *exHostAdrs();
LOCAL struct mbuf *exGet ();

/***********************************************************************
*
* exattach -
*
* Interface exists: make available by filling in network interface
* record.  System will initialize the interface when it is ready
* to accept packets.  Board is temporarily configured and issues
* a NET_ADDRS command, only to get the Ethernet address.
*/

STATUS exattach (unit)
    int unit; 

    {
    FAST struct ex_softc *xs;
    FAST struct ifnet *ifp;
    FAST struct ex_msg *bp;

    /* allocate device structure and put pointer in static array */

    xs = (struct ex_softc *) malloc (sizeof (struct ex_softc));
    if (xs == NULL)
	return (ERROR);

    bzero ((char *) xs, sizeof (*xs));		/* zero out xs */

    xs->readBuf = malloc (EXMAXRBUF);
    if (xs->readBuf == NULL)
	{
	free ((char *) xs);
	return (ERROR);
	}
	    
    xs->writeBuf = malloc (EXMAXRBUF);
    if (xs->writeBuf == NULL)
	{
	free (xs->readBuf);
	free ((char *) xs);
	return (ERROR);
	}

    /* allocate vdma registers 
     */
    if ((xs->readWind = (caddr_t)sysSetVdma(xs->readBuf, EXMAXRBUF)) < 0) {
	printf ("ex: could not set vdma registers for read\n");
	free (xs->readBuf);
	free (xs->writeBuf);
	free ((char *) xs);
	return (ERROR);
    }
    if ((xs->writeWind = (caddr_t)sysSetVdma(xs->writeBuf, EXMAXRBUF)) < 0) {
	printf ("ex: could not set vdma registers for write\n");
	sysGiveVdma (xs->readWind);
	free (xs->readBuf);
	free (xs->writeBuf);
	free ((char *) xs);
	return (ERROR);
    }
    if ((xs->xs_vdma = (caddr_t)sysSetVdma(xs, sizeof (struct ex_softc))) < 0) {
	printf ("ex: could not set vdma registers for xs\n");
	sysGiveVdma (xs->readWind);
	sysGiveVdma (xs->writeWind);
	free (xs->readBuf);
	free (xs->writeBuf);
	free ((char *) xs);
	return (ERROR);
    }

    ex_softc [unit] = xs;

    /* initialize device structure */

    xs->ivec    = freevec();
    xs->ilevel  = 4;
    xs->devAdrs = (struct exdevice *) exstd[unit];

    ifp = &xs->xs_if;

    ifp->if_unit = unit;
    ifp->if_name = "ex";
    ifp->if_mtu  = ETHERMTU;

    /* configure controller and get ethernet address */

    if (exconfig (unit, 0) != OK)		/* without interrupts */
	return (ERROR);

    bp = exgetcbuf(xs);
    bp->mb_rqst       = LLNET_ADDRS;
    bp->mb_na.na_mask = READ_OBJ;
    bp->mb_na.na_slot = PHYSSLOT;

    exGiveRequest (bp, xs->devAdrs);

    bp = xs->xs_x2hnext;
    while ((bp->mb_status & MH_OWNER) == MH_EXOS)	/* poll for reply */
    	;

    bcopy((caddr_t)bp->mb_na.na_addrs, (caddr_t)xs->xs_enaddr,
          sizeof (xs->xs_enaddr));

    ifp->if_init   = exinit;
    ifp->if_output = exoutput;
    ifp->if_ioctl  = exioctl;
    ifp->if_reset  = exreset;
    ifp->if_flags  = IFF_BROADCAST;

    if_attach(ifp);

    return (OK);
    }
/********************************************************************
*
* exreset -
*/

LOCAL VOID exreset (unit)
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    int dummy;

    xs->xs_if.if_flags &= ~IFF_RUNNING;
    xs->xs_flags &= ~EX_RUNNING;

    if (sysBus == VME_BUS)
	xs->devAdrs->bus.vme.xd_porta = EX_RESET;
    else
	dummy = xs->devAdrs->bus.multi.xd_porta;
    }
/**************************************************************************
*
* exinit -
*
* Initialization of interface; clear recorded pending operations.
* Called at boot time (with interrupts disabled?),
* and at ifAddrSet time via exioctl, with interrupts disabled.
*/

LOCAL VOID exinit (unit)
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ifnet *ifp = &xs->xs_if;
    FAST struct ex_msg *bp;

    /* not yet, if address still unknown */

    if (xs->xs_flags & EX_RUNNING)
    	return;


    /* attach and enable the excelan interrupt service routine to vector */

    intConnect (INUM_TO_IVEC (xs->ivec), exInt, unit);
    sysIntEnable (xs->ilevel);

    if (sysBus == VME_BUS)
        exconfig (unit, 4);		/* with vectored interrupts*/
    else
        exconfig (unit, 3);		/* with level interrupts*/

    /* Put EXOS on the Ethernet, using NET_MODE command */

    bp = exgetcbuf(xs);
    bp->mb_rqst       = LLNET_MODE;
    bp->mb_nm.nm_mask = WRITE_OBJ;
    bp->mb_nm.nm_optn = 0;
    bp->mb_nm.nm_mode = MODE_PERF;

    exGiveRequest (bp, xs->devAdrs);

    ifp->if_watchdog = exwatch;
    ifp->if_timer    = EXWATCHINTVL;

    exhangrcv(unit);			/* hang receive request */

    xs->xs_if.if_flags |= IFF_RUNNING;
    xs->xs_flags       |= EX_RUNNING;

    if (xs->xs_flags & EX_SETADDR)
    	exSetaddr((u_char *)0, unit);

    exstart (unit);			/* start transmits */
    }
/***********************************************************************
*
* exconfig -
*
* Reset, test, and configure EXOS.  This routine is called by exinit,
* and should also be callable by exattach.
*/

LOCAL STATUS exconfig (unit, itype)
    int unit;
    int itype;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct exdevice *devAdrs = xs->devAdrs;
    FAST struct confmsg *cm = &xs->xs_cm;
    FAST struct ex_msg *bp;
    char i;
    int ix;
    u_long shiftreg;

    xs->xs_flags = 0;

    /* Reset EXOS via probe (in case board doesn't exist) */

    if (sysBus == VME_BUS)
	{
	i = EX_RESET;
	if (vxMemProbe (&devAdrs->bus.vme.xd_porta, WRITE,
		        sizeof (devAdrs->bus.vme.xd_porta), &i) == ERROR)
	    {
	    errnoSet (S_iosLib_CONTROLLER_NOT_PRESENT);
	    return (ERROR);
	    }
	}
    else
	{
	if (vxMemProbe (&devAdrs->bus.multi.xd_porta, READ,
		        sizeof (devAdrs->bus.multi.xd_porta), &i) == ERROR)
	    {
	    errnoSet (S_iosLib_CONTROLLER_NOT_PRESENT);
	    return (ERROR);
	    }
	}

    /* wait for test to complete (5 secs max) */

    for (ix = 5 * sysClkGetRate ();
	 (ix > 0) &&
	 ((((sysBus == VME_BUS) ?
	    devAdrs->bus.vme.xd_portb : devAdrs->bus.multi.xd_portb) &
							      EX_TESTOK) == 0);
	 --ix)
	 {
	 taskDelay (1);
	 }

    if (ix == 0)
	{
	printf("ex%d: self-test failed.\n", unit);
	errnoSet (S_ioLib_DEVICE_ERROR);
	return (ERROR);
	}

    /* Set up configuration message. */

    cm->cm_1rsrv       = 1;
    cm->cm_cc          = 0xFF;
    cm->cm_opmode      = 0;		/* link-level controller mode */
    cm->cm_dfo         = 0x0101;	/* enable host data order conversion */
    cm->cm_dcn1        = 1;
    cm->cm_2rsrv[0]    = 0;
    cm->cm_2rsrv[1]    = 0;
    cm->cm_ham         = 3;		/* absolute address mode */
    cm->cm_3rsrv       = 0;
    cm->cm_mapsiz      = 0;
    cm->cm_byteptrn[0] = 0x01;		/* EXOS deduces data order of host */
    cm->cm_byteptrn[1] = 0x03;		/*  by looking at this pattern */
    cm->cm_byteptrn[2] = 0x07;
    cm->cm_byteptrn[3] = 0x0F;
    cm->cm_wordptrn[0] = 0x0103;
    cm->cm_wordptrn[1] = 0x070F;
    cm->cm_lwordptrn   = 0x0103070F;

    for (ix = 0; ix < 20; ix++)
	cm->cm_rsrvd[ix] = 0;

    cm->cm_mba         = 0xFFFFFFFF;
    cm->cm_nproc       = 0xFF;
    cm->cm_nmbox       = 0xFF;
    cm->cm_nmcast      = 0xFF;
    cm->cm_nhost       = 1;

    cm->cm_h2xba       = (int) exHostAdrs (INCORE_BASE(xs->xs_vdma));
    cm->cm_h2xhdr      = H2XHDR_OFFSET(unit);
    cm->cm_h2xtyp      = 0;		/* should never wait for rqst buffer */
    cm->cm_x2hba       = cm->cm_h2xba;
    cm->cm_x2hhdr      = X2HHDR_OFFSET(unit);
    cm->cm_x2htyp      = itype;	/* 0 for none, 4 for vectored, 3 for level  */

    if (sysBus == VME_BUS)
        cm->cm_x2haddr = xs->ivec;
    
    /*
     * Set up message queues and headers.
     * First the request queue.
     */
    for (bp = &xs->xs_h2xent[0]; bp < &xs->xs_h2xent[NH2X]; bp++)
	{
    	bp->mb_link   = (u_short)((char *)(bp+1)-INCORE_BASE(xs));
    	bp->mb_rsrv   = 0;
    	bp->mb_length = MBDATALEN;
    	bp->mb_status = MH_HOST;
    	bp->mb_next   = bp+1;
	}

    xs->xs_h2xhdr =
    	xs->xs_h2xent[NH2X-1].mb_link = (u_short)H2XENT_OFFSET(unit);

    xs->xs_h2xnext =
    	xs->xs_h2xent[NH2X-1].mb_next = xs->xs_h2xent;

    /* Now the reply queue. */

    for (bp = &xs->xs_x2hent[0]; bp < &xs->xs_x2hent[NX2H]; bp++)
	{
    	bp->mb_link   = (u_short)((char *)(bp+1)-INCORE_BASE(xs));
    	bp->mb_rsrv   = 0;
    	bp->mb_length = MBDATALEN;
    	bp->mb_status = MH_EXOS;
    	bp->mb_next   = bp+1;
	}

    xs->xs_x2hhdr =
    	xs->xs_x2hent[NX2H-1].mb_link = (u_short)X2HENT_OFFSET(unit);

    xs->xs_x2hnext =
    	xs->xs_x2hent[NX2H-1].mb_next = xs->xs_x2hent;

    /*
     * Write config msg address to EXOS and wait for
     * configuration to complete (guaranteed response within 2 seconds).
     */


    shiftreg = (u_long)0x0000FFFF;

    for (ix = 0; ix < 8; ix++)
	{
    	if (ix == 4)
	shiftreg = 
	    (u_long) exHostAdrs ((char *)((int)xs->xs_vdma + CM_OFFSET(xs)));

    	if (sysBus == VME_BUS)
    	    {
    	    while (devAdrs->bus.vme.xd_portb & EX_UNREADY)
    		;
    	    devAdrs->bus.vme.xd_portb = (u_char)(shiftreg & 0xFF);
    	    }
    	else
    	    {
    	    while (devAdrs->bus.multi.xd_portb & EX_UNREADY)
    		;
    	    devAdrs->bus.multi.xd_portb = (u_char)(shiftreg & 0xFF);
    	    }

    	shiftreg >>= 8;
	}

    for (ix = 3 * sysClkGetRate (); (cm->cm_cc == 0xFF) && ix; --ix)
	taskDelay (1);


    if (cm->cm_cc)
	{
    	printf("ex%d: configuration failed; cc=0x%x\n", unit, cm->cm_cc);
	errnoSet (S_ioLib_DEVICE_ERROR);
	return (ERROR);
	}

    return (OK);
    }
/**************************************************************************
*
* exstart -
*
* Start or re-start output on interface.
* Get another datagram to send off of the interface queue,
* and map it to the interface before starting the output.
* This routine is called by exinit(), exoutput(), and exInt().
* In all cases, interrupts by EXOS are disabled.
*/

LOCAL VOID exstart(unit)
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ex_msg *bp;
    FAST struct mbuf *m;
    struct mbuf *mp;
    FAST caddr_t buf;
    FAST int len;
    int s;

    /* get first packet to send */

again:
    s = splnet ();
    xs->xs_flags &= ~EX_XPENDING;
    IF_DEQUEUE(&xs->xs_if.if_snd, m);
    splx (s);

    if (m == 0)
    	return;

    mp = m;


    /* copy packet to write buffer;
     * note that freeing mbuf is delayed until after we give EXOS
     * the request so we can overlap operation */

    buf = xs->writeBuf;

    for (; m != NULL; m = m->m_next)
	{
	bcopy (mtod (m, caddr_t), buf, (unsigned)m->m_len);
	buf += m->m_len;
	}

    len = max (ETHERMIN + sizeof(struct ether_header), buf - xs->writeBuf);


    /* call output hook if any */

    if ((etherOutputHookRtn != NULL) &&
	(* etherOutputHookRtn) (&xs->xs_if, xs->writeBuf, len))
	goto again;	/* output hook has already processed this packet */


    /* Place a transmit request. */

    bp = exgetcbuf(xs);
    bp->mb_rqst                 = LLRTRANSMIT;
    bp->mb_et.et_nblock         = 1;
    bp->mb_et.et_blks[0].bb_len = (u_short)len;
    *(char **)bp->mb_et.et_blks[0].bb_addr = exHostAdrs ((char *) xs->writeWind);

    xs->xs_flags |= EX_XPENDING;	/* indicate that xmit is in progress */

    exGiveRequest (bp, xs->devAdrs);

    m_freem (mp);
    }
/*************************************************************************
*
* exInt - handle controller interrupt
*
* This routine is called at interrupt level in response to an interrupt from
* the controller.
*/

VOID exInt (unit)
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ex_msg *bp = xs->xs_x2hnext;

    sysIntAck (xs->ilevel);	/* acknowledge interrupt */

    while ((bp->mb_status & MH_OWNER) == MH_HOST)
	{
    	switch (bp->mb_rqst)
	    {
	    case LLRECEIVE:
    		netToDoAdd (exRecvAndHang, unit, bp->mb_er.er_blks[0].bb_len,
    			    bp->mb_rply);
    		break;

	    case LLRTRANSMIT:
#ifdef DEBUG
    		if ((xs->xs_flags & EX_XPENDING) == 0)
		    panic("exxmit: no xmit pending");
#endif
    		xs->xs_if.if_opackets++;

    		if (bp->mb_rply == LL_OK)
		    ;
		else if (bp->mb_rply & LLXM_1RTRY)
		    {
		    xs->xs_if.if_collisions++;
		    }
		else if (bp->mb_rply & LLXM_RTRYS)
		    {
		    xs->xs_if.if_collisions += 2;	/* guess */
		    }
		else if (bp->mb_rply & LLXM_ERROR)
		    {
		    xs->xs_if.if_oerrors++;
		    logMsg ("ex%d: transmit error=%#x\n", unit, bp->mb_rply);
		    }

    		netToDoAdd (exstart, unit);
    		break;

	    case LLNET_STSTCS:
    		xs->xs_if.if_ierrors = xs->xs_xsa.sa_crc;
    		xs->xs_flags &= ~EX_STATPENDING;
    		break;

	    case LLNET_ADDRS:
	    case LLNET_RECV:
	    case LLNET_MODE:
    		break;
	    } /* end of switch */

    	if (sysBus == MULTI_BUS)
    	    xs->devAdrs->bus.multi.xd_porta = 7;	/* ack the interrupt */

    	bp->mb_length = MBDATALEN;

	exGiveRequest (bp, xs->devAdrs);

    	bp = xs->xs_x2hnext = xs->xs_x2hnext->mb_next;
	}
    }
/***********************************************************************
*
* exgetcbuf -
*
* Get a request buffer, fill in standard values, advance pointer.
*/

LOCAL struct ex_msg *exgetcbuf (xs)
    struct ex_softc *xs;

    {
    FAST struct ex_msg *bp = xs->xs_h2xnext;

#ifdef DEBUG
    if ((bp->mb_status & MH_OWNER) == MH_EXOS)
    	panic("exgetcbuf(): EXOS owns message buffer");
#endif

    bp->mb_1rsrv  = 0;
    bp->mb_length = MBDATALEN;

    xs->xs_h2xnext = xs->xs_h2xnext->mb_next;

    return (bp);
    }
/*************************************************************************
*
* exRecvAndHang - receive packet and hang another receive request
*/

LOCAL VOID exRecvAndHang (unit, len, status)
    int unit;
    int len;
    int status;

    {
    exrecv (unit, len, status);
    exhangrcv (unit);
    }
/*************************************************************************
*
* exrecv -
*
* Process Ethernet receive completion:
*	If input error just drop packet.
*	Otherwise purge input buffered data path and examine 
*	packet to determine type.  If can't determine length
*	from type, then have to drop packet.  Otherwise decapsulate
*	packet based on type and pass to type-specific higher-level
*	input routine.
*/

LOCAL VOID exrecv (unit, len, status)
    int unit;
    FAST int len;
    int status;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ether_header *eh;
    FAST struct mbuf *m;
    FAST struct ifqueue *inq;
    FAST int off, resid;
    char *pData;

    xs->xs_if.if_ipackets++;

    if (status != LL_OK)
	{
    	xs->xs_if.if_ierrors++;
    	printErr ("ex%d: receive error=%b\n", unit, status, RECV_BITS);
    	return;
	}

    /* get input packet length and ptr to packet */

    len -= 4;
    eh = (struct ether_header *)(xs->readBuf);


    /* call input hook if any */

    if ((etherInputHookRtn != NULL) &&
	(* etherInputHookRtn) (&xs->xs_if, (char *) eh, len))
	return;	/* input hook has already processed this packet */


    /* do software filter if controller is in promiscuous mode */

    if (xs->promiscuous)
	{
	if ((bcmp ((char *)eh->ether_dhost, (char *)xs->xs_enaddr,
		     sizeof (eh->ether_dhost)) != 0) &&	    /* not our adrs? */
	    (bcmp ((char *)eh->ether_dhost, (char *)etherbroadcastaddr,
		     sizeof (eh->ether_dhost)) != 0))	    /* not broadcast? */

	    return;					    /* not for us */
	}


    /*
     * Deal with trailer protocol: if type is trailer
     * get true type from first 16-bit word past data.
     * Remember that type was trailer by setting off.
     */

    len  -= sizeof (struct ether_header);
    pData = ((char *) eh) + (sizeof (struct ether_header));

    eh->ether_type = ntohs((u_short)eh->ether_type);
    if (eh->ether_type >= ETHERTYPE_TRAIL &&
        eh->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER)
	{
    	off = (eh->ether_type - ETHERTYPE_TRAIL) * 512;

    	if (off >= ETHERMTU)
	    return;		/* sanity */

	eh->ether_type = ntohs (*(u_short *) (pData + off));
	resid          = ntohs (*(u_short *) (pData + off + 2));

    	if (off + resid > len)
	    return;		/* sanity */

    	len = off + resid;
	}
    else
    	off = 0;

    if (len == 0)
        return;		/* sanity */


    /*
     * Pull packet off interface.  Off is nonzero if packet
     * has trailing header; exGet will then force this header
     * information to be at the front, but we still have to drop
     * the type and length which are at the front of any trailer data.
     */

    m = exGet (pData, len, off, (struct ifnet *) &xs->xs_if);
    if (m == 0)
    	return;

    if (off)
	{
    	struct ifnet *ifp;

    	ifp = *(mtod(m, struct ifnet **));
    	m->m_off += 2 * sizeof (u_short);
    	m->m_len -= 2 * sizeof (u_short);
    	*(mtod(m, struct ifnet **)) = ifp;
	}

    switch (eh->ether_type)
	{
#ifdef INET
	case ETHERTYPE_IP:
	    netToDoAdd (ipintr, 0);
	    inq = &ipintrq;
	    break;

	case ETHERTYPE_ARP:
	    arpinput(&xs->xs_ac, m);
	    return;

#endif
#ifdef NS
	case ETHERTYPE_NS:
	    schednetisr(NETISR_NS);
	    inq = &nsintrq;
	    break;

#endif
	default:
	    m_freem(m);
	    return;
	}

    {
    int s = splnet ();

    if (IF_QFULL(inq))
	{
    	IF_DROP(inq);
    	m_freem(m);
    	splx (s);
    	return;
	}

    IF_ENQUEUE(inq, m);

    splx (s);
    }
    }
/**********************************************************************
*
* exhangrcv -
*
* Send receive request to EXOS.
* This routine is called by exinit and exInt,
* with interrupts disabled in both cases.
*/

LOCAL VOID exhangrcv (unit)
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ex_msg *bp = exgetcbuf(xs);
    
    bp->mb_rqst                            = LLRECEIVE;
    bp->mb_er.er_nblock                    = 1;
    bp->mb_er.er_blks[0].bb_len            = EXMAXRBUF;
    *(char **)bp->mb_er.er_blks[0].bb_addr = exHostAdrs ((char *) xs->readWind);

    exGiveRequest (bp, xs->devAdrs);
    }
/**************************************************************************
* 
* exGet - Pull read data off an interface.
*
* Len is length of data, with local net header stripped.
* Off is non-zero if a trailer protocol was used, and
* gives the offset of the trailer information.
* We copy the trailer information and then all the normal
* data into mbufs.  
* Prepend a pointer to the interface structure,
* so that protocols can determine where incoming packets arrived.
*/

LOCAL struct mbuf *exGet (buf0, totlen, off0, ifp)
    caddr_t buf0;
    int totlen;
    int off0;
    struct ifnet *ifp;

    {
    struct mbuf *top;
    struct mbuf **mp;
    FAST struct mbuf *m;
    FAST int len;
    FAST caddr_t buf;
    FAST int off;

    buf = buf0;
    off = off0;

    top = 0;
    mp = &top;

    while (totlen > 0)
	{
	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0)
	    {
	    m_freem(top);
	    return (0);
	    }

	if (off)
	    {
	    len = totlen - off;
	    buf += off;
	    }
	else
	    len = totlen;

	m->m_off = MMINOFF;
	if (ifp)
	    {
	    /* Leave room for ifp. */
	    m->m_len = MIN(MLEN - sizeof(ifp), len);
	    m->m_off += sizeof(ifp);
	    }
	else 
	    m->m_len = MIN(MLEN, len);

	bcopy (buf, mtod(m, caddr_t), (unsigned)m->m_len);
	buf += m->m_len;

	*mp = m;
	mp = &m->m_next;

	if (off)
	    {
	    /* sort of an ALGOL-W style for statement... */
	    off += m->m_len;
	    if (off == totlen)
		{
		/* reached end of data; setup to move trailer front */

		buf = buf0;
		off = 0;
		totlen = off0;
		}
	    }
	else
	    totlen -= m->m_len;

	if (ifp)
	    {
	    /* Prepend interface pointer to first mbuf. */
	    m->m_len += sizeof(ifp);
	    m->m_off -= sizeof(ifp);
	    *(mtod(m, struct ifnet **)) = ifp;
	    ifp = (struct ifnet *)0;
	    }
	}

    return (top);
    }
/***********************************************************************
*
* exoutput -
*
* Ethernet output routine.
* Encapsulate a packet of type family for the local net.
* Use trailer local net encapsulation if enough data in first
* packet leaves a multiple of 512 bytes of data in remainder.
*/

LOCAL int exoutput(ifp, m0, dst)
    FAST struct ifnet *ifp;
    FAST struct mbuf *m0;
    struct sockaddr *dst;

    {
    int type, s, error;
    u_char edst[6];
    struct in_addr idst;
    FAST struct ex_softc *xs = ex_softc[ifp->if_unit];
    FAST struct mbuf *m = m0;
    FAST struct ether_header *eh;
    FAST int off;
    int usetrailers;

    if ((ifp->if_flags & (IFF_UP|IFF_RUNNING)) != (IFF_UP|IFF_RUNNING))
	{
    	error = ENETDOWN;
    	goto bad;
	}

    switch (dst->sa_family)
	{

#ifdef INET
	case AF_INET:
	    idst = ((struct sockaddr_in *)dst)->sin_addr;

	    if (!arpresolve(&xs->xs_ac, m, &idst, edst, &usetrailers))
    		return (0);	/* if not yet resolved */

	    off = ntohs((u_short)mtod(m, struct ip *)->ip_len) - m->m_len;

	    if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		m->m_off >= MMINOFF + 2 * sizeof (u_short))
		{
    		type = ETHERTYPE_TRAIL + (off>>9);
    		m->m_off -= 2 * sizeof (u_short);
    		m->m_len += 2 * sizeof (u_short);
    		*mtod(m, u_short *) = htons(ETHERTYPE_IP);
    		*(mtod(m, u_short *) + 1) = htons((u_short)m->m_len);
    		goto gottrailertype;
		}

	    type = ETHERTYPE_IP;
	    off = 0;
	    goto gottype;
#endif
#ifdef NS
	case AF_NS:
	    type = ETHERTYPE_NS;
	    bcopy((caddr_t)&(((struct sockaddr_ns *)dst)->sns_addr.x_host),
    		  (caddr_t)edst, sizeof (edst));
	    off = 0;
	    goto gottype;
#endif

	case AF_UNSPEC:
	    eh = (struct ether_header *)dst->sa_data;
	    bcopy((caddr_t)eh->ether_dhost, (caddr_t)edst, sizeof (edst));
	    type = eh->ether_type;
	    goto gottype;

	default:
	    printf("ex%d: can't handle af%d\n", ifp->if_unit, dst->sa_family);
	    error = EAFNOSUPPORT;
	    goto bad;
	}

gottrailertype:
    /*
     * Packet to be sent as trailer: move first packet
     * (control information) to end of chain.
     */
    while (m->m_next)
    	m = m->m_next;
    m->m_next = m0;
    m = m0->m_next;
    m0->m_next = 0;
    m0 = m;

gottype:
    /*
     * Add local net header.  If no space in first mbuf,
     * allocate another.
     */
    if (m->m_off > MMAXOFF || MMINOFF + sizeof (struct ether_header) > m->m_off)
	{
    	m = m_get(M_DONTWAIT, MT_HEADER);
    	if (m == 0)
	    {
	    error = ENOBUFS;
	    goto bad;
	    }
    	m->m_next = m0;
    	m->m_off = MMINOFF;
    	m->m_len = sizeof (struct ether_header);
	}
    else
	{
    	m->m_off -= sizeof (struct ether_header);
    	m->m_len += sizeof (struct ether_header);
	}
    eh = mtod(m, struct ether_header *);
    eh->ether_type = htons((u_short)type);
    bcopy((caddr_t)edst, (caddr_t)eh->ether_dhost, sizeof (edst));
    bcopy((caddr_t)xs->xs_enaddr, (caddr_t)eh->ether_shost, 6);

    /*
     * Queue message on interface, and start output if interface
     * not yet active.
     */

    s = splimp();
    if (IF_QFULL(&ifp->if_snd))
	{
    	IF_DROP(&ifp->if_snd);
    	splx(s);
    	m_freem(m);
    	return (ENOBUFS);
	}

    IF_ENQUEUE(&ifp->if_snd, m);

    /*
     * If transmit request not already pending, then
     * kick the back end.
     */

    if ((xs->xs_flags & EX_XPENDING) == 0)
    	exstart(ifp->if_unit);

#ifdef DEBUG
    else
    	xs->xs_wait++;
#endif

    splx(s);
    return (0);

bad:
    m_freem(m0);
    return (error);
    }
/*******************************************************************
*
* exwatch -
*
* Watchdog routine - place stats request to EXOS
* (This could be dispensed with, if you don't care
*  about the if_ierrors count, or are willing to receive
*  bad packets in order to derive it.)
*
* ARGSUSED
*/

LOCAL VOID exwatch(unit)
    int unit;
    {
    }
/********************************************************************
*
* exioctl -
*
* Process an ioctl request.
*/

LOCAL int exioctl(ifp, cmd, data)
    FAST struct ifnet *ifp;
    int cmd;
    caddr_t data;

    {
    FAST struct ifaddr *ifa = (struct ifaddr *)data;
    FAST struct ex_softc *xs = ex_softc[ifp->if_unit];
    int s = splimp(), error = 0;

    switch (cmd)
	{
	case SIOCSIFADDR:
	    ifp->if_flags |= IFF_UP;
	    exinit(ifp->if_unit);

	    switch (ifa->ifa_addr.sa_family)
		{
#ifdef INET
		case AF_INET:
		    ((struct arpcom *)ifp)->ac_ipaddr = IA_SIN(ifa)->sin_addr;
		    arpwhohas((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
		    break;
#endif
#ifdef NS
		case AF_NS:
		    {
		    FAST struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);
    		
		    if (ns_nullhost(*ina))
    			ina->x_host = *(union ns_host *)(xs->xs_enaddr);
		    else
    			exSetaddr(ina->x_host.c_host,ifp->if_unit);
		    break;
		    }
#endif
		}
	    break;

	case SIOCSIFFLAGS:
	    if ((ifp->if_flags & IFF_UP) == 0 && xs->xs_flags & EX_RUNNING)
		{
		int dummy;

    		if (sysBus == VME_BUS)
		    xs->devAdrs->bus.vme.xd_porta = EX_RESET;
    		else
    		    dummy = xs->devAdrs->bus.multi.xd_porta;

    		xs->xs_flags &= ~EX_RUNNING;
		}

	    else if (ifp->if_flags & IFF_UP && (xs->xs_flags & EX_RUNNING) == 0)
		exinit(ifp->if_unit);
	    break;

	default:
	    error = EINVAL;
	}

    splx(s);
    return (error);
    }
/**************************************************************************
*
* exSetaddr - set ethernet address for unit
*/

LOCAL VOID exSetaddr(physaddr, unit)
    u_char *physaddr;
    int unit;

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ex_msg *bp;
    
    if (physaddr)
	{
    	xs->xs_flags |= EX_SETADDR;
    	bcopy((caddr_t)physaddr, (caddr_t)xs->xs_enaddr, 6);
	}

    if (! (xs->xs_flags & EX_RUNNING))
    	return;

    bp = exgetcbuf(xs);
    bp->mb_rqst       = LLNET_ADDRS;
    bp->mb_na.na_mask = READ_OBJ|WRITE_OBJ;
    bp->mb_na.na_slot = PHYSSLOT;
    bcopy((caddr_t)xs->xs_enaddr, (caddr_t)bp->mb_na.na_addrs, 6);

    exGiveRequest (bp, xs->devAdrs);

    bp = xs->xs_x2hnext;
    while ((bp->mb_status & MH_OWNER) == MH_EXOS)	/* poll for reply */
    	;

#ifdef	DEBUG
    printf("ex%d: reset addr %s\n", unit, ether_sprintf(bp->mb_na.na_addrs));
#endif

    /* Now, re-enable reception on phys slot. */

    bp = exgetcbuf(xs);
    bp->mb_rqst       = LLNET_RECV;
    bp->mb_nr.nr_mask = ENABLE_RCV|READ_OBJ|WRITE_OBJ;
    bp->mb_nr.nr_slot = PHYSSLOT;

    exGiveRequest (bp, xs->devAdrs);

    bp = xs->xs_x2hnext;
    while ((bp->mb_status & MH_OWNER) == MH_EXOS)	/* poll for reply */
    	;
    }
/***********************************************************************
*
* exHostAdrs - get host address as required by excelan board
*
* This routine converts the specified local address into the form
* that the excelan board wants to see it, namely converted into
* the proper vme address space and, in the VME version, with the
* address modifier in the upper byte.
*/

LOCAL char *exHostAdrs (localAdrs)
    char *localAdrs;

    {
    char *busAdrs;

    if (sysLocalToBusAdrs (exBusAdrsSpace, localAdrs, &busAdrs) != OK)
	printErr ("ex: no bus access to local adrs 0x%x\n", localAdrs);

    /* Exos 202 (vme version) requires adrs modifier code in upper byte */

    if (sysBus == VME_BUS)
	busAdrs = (char *) ((int) busAdrs | (exBusAdrsSpace << 24));

    return (busAdrs);
    }
/***********************************************************************
*
* exPromiscuous - set controller to promiscuous mode
*/

STATUS exPromiscuous (unit, on)
    int unit;
    BOOL on;	/* TRUE = turn promiscuous on, FALSE = off */

    {
    FAST struct ex_softc *xs = ex_softc[unit];
    FAST struct ex_msg *bp = exgetcbuf (xs);

    bp->mb_rqst       = LLNET_MODE;
    bp->mb_nm.nm_mask = WRITE_OBJ;
    bp->mb_nm.nm_optn = 0;
    bp->mb_nm.nm_mode = on ? MODE_PROM : MODE_PERF;

    xs->promiscuous = on;

    exGiveRequest (bp, xs->devAdrs);
    }
/***********************************************************************
*
* exGiveRequest - give request buffer to exos
*/

VOID exGiveRequest (bp, devAdrs)
    struct ex_msg *bp;		/* pointer to msg buffer to give to exos */
    struct exdevice *devAdrs;	/* device address */

    {
    bp->mb_status |= MH_EXOS;		/* give request to exos */

    if (sysBus == VME_BUS)
        devAdrs->bus.vme.xd_portb = EX_NTRUPT;
    else
        devAdrs->bus.multi.xd_portb = EX_NTRUPT;
    }
