/* trcLib.c - stack trace library */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This module provides a routine, trcStack, which traces a stack
given the current frame pointer, stack pointer, and program counter.
The resulting stack trace lists the nested routine calls and their arguments.

This module provides the low-level stack trace facility.
A higher-level symbolic stack trace, implemented on top of this facility,
is provided by the routine tt (2) in dbgLib (1).

SEE ALSO: "Debugging", dbgLib(1), tt(2)
*/

#include "UniWorks.h"
#include "symLib.h"
#include "sysSymTbl.h"

#define MAX_TRACE_DEPTH 40 /* maximum number of levels of stack to trace */

/* instruction words */

#define LINK_A6		0x4e56		/* LINK A6,... */
#define RTS		0x4e75		/* RTS */
#define JSR_ABS		0x4eb9		/* JSR abs */
#define ADD_W		0xdefc		/* ADD.W */
#define ADD_L		0xdffc		/* ADD.L */
#define ADDQ_W		0x504f		/* ADDQ.W A7 */
#define ADDQ_L		0x508f		/* ADDQ.L A7 */
#define LEA_A7		0x4fef		/* LEA $x(A7),A7 */
#define MOVE_L_A7	0x2e80		/* MOVE.L xxx,(A7) */
#define MOVE_L_A6_A7	0x2e75		/* MOVE.L (xxx,A6),A7 */

/* forward declarations */

LOCAL INSTR *trcFindCall ();
LOCAL VOID trcDefaultPrint ();
LOCAL INSTR *trcFollowBra ();

/*******************************************************************************
*
* trcStack - print trace of function calls from stack
*
* This routine provides the low-level stack trace function.
* (A higher-level symbolic stack trace built on top of this is provided
* by the routine tt (2) in dbgLib (1).)
* 
* A list of the nested routine calls that are on the stack is printed.
* Each routine call, with its parameters, will be shown.
*
* The stack being traced should be quiescent.  One should especially
* avoid trying to trace one's own stack.
*
* PRINT ROUTINE
* In order to allow symbolic or other alternative printout formats,
* the call to this routine includes the "printRtn" parameter, which
* is a routine to be called at each nesting level to print out the
* routine name and its arguments.  This routine should be declared as
* follows:
* .ne 7
* .CS
*
*   VOID printRtn (callAdrs, rtnAdrs, nargs, args)
*     INSTR *callAdrs;	/* address from which routine was called *
*     int rtnAdrs;	/* address of routine called *
*     int nargs;	/* number of arguments in call *
*     int *args;	/* pointer to arguments *
*
* .CE
*
* If a NULL printRtn is specified, then a default routine will be used
* which just prints out the call address, function address, and
* arguments as hexadecimal values.
*
* CAVEAT
* In order to do the trace, some assumptions are made.
* In general, the trace will work for all C language routines,
* and for assembly language routines that start with a LINK instruction.
* Most UniWorks assembly language routines include LINK instructions for 
* exactly this reason.
* However, routines written in other languages, strange entries into
* routines, or tasks with corrupted stacks, can make the trace very confused.
* Also, all parameters are assumed to be 32-bit quantities, so structures
* passed as parameters will be displayed as some number of long integers.
*
* .ne 14
* EXAMPLE
* The following sequence can be used
* to trace a UniWorks task given a pointer to the task's TCB:
* .CS
*
*   int dregs[8];	/* task's data registers
*   int aregs[7];	/* task's address registers
*   char *sp;		/* task's stack pointer
*   unsigned short sr;	/* task's status register
*   INSTR *pc;		/* task's program counter
*
*   taskRegsGet (taskId, dregs, aregs, &sp, &sr, &pc);
*   trcStack ((int*)aregs[6], (int*)sp, pc, (FUNCPTR)NULL);
*
* .CE
*
* SEE ALSO: tt(2)
*/

VOID trcStack (fp, sp, pc, printRtn)
    FAST int *fp;	/* stack frame pointer (A6) */
    FAST int *sp;	/* stack pointer */
    FAST INSTR *pc;	/* program counter */
    FUNCPTR printRtn;	/* routine to print single function call */

    {
    int localBuf;	/* store for long below stack pointer */

    /* use default print routine if none specified */

    if (printRtn == NULL)
	printRtn = trcDefaultPrint;

    /* since vrtx version 3 keeps stuff below the stack in a suspended
       task, we'll keep a local copy of the long word below the sp,
       and then restore it when we're done */

    localBuf = *(sp - 1);

    /* if program counter is at a LINK A6,... or RTS, 
     * or at a BRA to a LINK (but not a BRA to an RTS!),
     * adjust frame pointer */

    if ((*pc == LINK_A6) || (*pc == RTS) || (*trcFollowBra (pc) == LINK_A6))
	{
	*(sp - 1) = (int)fp;		/* make new frame pointer by */
					/* sticking old one on stack */
	fp = sp - 1;			/* and pointing to it */
	}


    /* do stack trace */

    trcStackLvl (fp, pc, 0, printRtn);

    /* restore the stuff below the stack */

    *(sp - 1) = localBuf;
    }
/************************************************************************
*
* trcStackLvl - recursive stack trace routine
*
* This routine is recursive, being called once for each level of routine
* nesting.  The maximum recursion depth is limited to 40 to prevent
* garbage stacks from causing this routine to continue unbounded.
* The "depth" parameter on the original call should be 0.
*/

LOCAL VOID trcStackLvl (fp, pc, depth, printRtn)
    FAST int *fp;	/* stack frame pointer (A6) */
    INSTR *pc;		/* program counter */
    int depth;		/* recursion depth */
    FUNCPTR printRtn;	/* routine to print single function call */

    {
    if (fp == NULL)
	return;			/* stack is untraceable */

    /* handle oldest calls first, up to MAX_TRACE_DEPTH of them */

    if ((*fp != NULL) && (depth < MAX_TRACE_DEPTH))
	trcStackLvl ((int *) *fp, (INSTR *) *(fp + 1), depth + 1, printRtn);

    (* printRtn) (trcFindCall (fp), trcFindFuncStart (fp, pc),
		  trcCountArgs ((INSTR *) *(fp + 1)), fp + 2);
    }
/*******************************************************************************
*
* trcDefaultPrint - print a function call
*
* This routine is called by trcStack to print each level in turn.
*/

LOCAL VOID trcDefaultPrint (callAdrs, funcAdrs, nargs, args)
    INSTR *callAdrs;		/* address from which function was called */
    int funcAdrs;		/* address of function called */
    FAST int nargs;		/* number of arguments in function call */
    int *args;			/* pointer to function args */

    {
    FAST int i;

    /* print call address and function address */

    printErr ("%6x: %x (", callAdrs, funcAdrs);

    /* print args */

    for (i = 0; i < nargs; ++i)
	{
	if (i != 0)
	    printErr (", ");
	printErr ("%x", args[i]);
	}

    printErr (")\n");
    }
/****************************************************************************
*
* trcFindCall - get address from which function was called
*
* RETURNS: address from which current subroutine was called, or NULL.
*/

LOCAL INSTR *trcFindCall (fp)
    int *fp;		/* frame pointer */

    {
    FAST INSTR *addr;

    /* starting at the word preceding the return address, search for a
       jsr or bsr */

    for (addr = (INSTR *)*(fp + 1) - 1; addr != NULL; --addr)
	if (((*addr & 0xffc0) == 0x4e80) || ((*addr & 0xff00) == 0x6100))
	    return (addr);		/* found it */

    return (NULL);			/* not found */
    }
/****************************************************************************
*
* trcCountArgs - find number of arguments to function
*
* RETURNS: number of arguments of function
*/

LOCAL int trcCountArgs (retaddr)
    FAST INSTR *retaddr;		/* return address of function call */

    {
    FAST INSTR inst;
    FAST int nargs;

    /* if inst is a BRA, use the target of the BRA as the retaddr */

    retaddr = trcFollowBra (retaddr);
    inst = *retaddr;		/* get the instruction */

    if (inst == ADD_W)
	nargs = *(retaddr + 1) >> 2;		/* ADD.W */
    else if (inst == ADD_L)
	nargs = *(int *)(retaddr + 1) >> 2;	/* ADD.L */
    else if (((inst & 0xF1FF) == ADDQ_W) || ((inst & 0xF1FF) == ADDQ_L))
	{
	/* get the number of bytes and divide by 4 to get the number of args */

	nargs = (inst & 0x0E00) >> 11;		/* ADDQ.L or ADDQ.W */
	if (nargs == 0)
	    nargs = 2;				/* 0 => 8 in quick mode */
	}
    else if (inst == LEA_A7)
	nargs = *(retaddr + 1) >> 2;		/* LEA $x(A7),A7 */
    else if ((inst & 0xFFC0) == MOVE_L_A7)
	nargs = 1;				/* MOVE.L xxx,(A7) */
    else if (inst == MOVE_L_A6_A7)
	nargs = 8;				/* MOVE.L (xxx,A6),A7
						   # of args unknowable */
    else
	nargs = 0;				/* no args, or unknown */

    return (nargs);
    }
/****************************************************************************
*
* trcFindFuncStart - find the starting address of a function
*
* This routine finds the starting address of a function by one of two ways.
* If the given frame pointer points to a legitimate frame pointer, then the
* long word following the frame pointer pointed to by the frame pointer should
* be the return address of the function call. Then the instruction preceding
* the return address would be the function call, and the address can be gotten
* from there, provided that the jsr was to an absolute address. If it was,
* use that address as the function address. Otherwise, search backward from the
* given pc until the begining of the function is found. Any questions?
*/

LOCAL int trcFindFuncStart (fp, pc)
    int *fp;			/* frame pointer resulting from function call */
    FAST INSTR *pc;		/* address somewhere within the function */
    {
    FAST INSTR *ip;		/* instruction pointer */
    FAST INSTR *minPc;		/* lower bound on program counter */
    int val;			/* address gotten from symbol table */
    char name[MAX_SYS_SYM_LEN];	/* string associated with val */
    UTINY type;			/* type associated with val */

    minPc = NULL;

    /* if there is a symbol table, use value from table that's <= pc as
       lower bound in search for function start */

    if (sysSymTbl != NULL)		/* is there a symbol table? */
	{
	if (symValFind (sysSymTbl, (int) pc, name, &val, &type) == OK)
	    minPc = (INSTR *) val;
	}

    if (fp != NULL)			/* another frame pointer? */
	{
	ip = trcFindCall (fp);		/* point at JSR instruction */
	if (*ip == JSR_ABS)		/* is it absolute long mode ? */
	    return (*(int *)(ip + 1));	/* return address of function called */
	}

    while ((pc >= minPc) && (*trcFollowBra (pc) != LINK_A6))
	--pc;				/* search backward for LINK A6,#xxxx */

    return ((int) pc);
    }
/*********************************************************************
*
* trcFollowBra - resolve any BRA instructions to final destination
*
* RETURNS: address that chain of branches points to.
*/

LOCAL INSTR *trcFollowBra (adrs)
    FAST INSTR *adrs;

    {
    FAST INSTR inst = *adrs;	/* 16 bit instruction at adrs */
    FAST int displacement;

    /* while instruction is a BRA, get destination adrs */

    while ((inst & 0xff00) == 0x6000)
	{
	++adrs;			/* point to word following instr */

	switch (inst & 0xff)
	    {
	    case 0:			/* 16 bit displacement */
		displacement = (short) *adrs;
		break;
	    case 0xff:			/* 32 bit displacement */
		displacement = (*adrs << 16) | *(adrs + 1);
		break;
	    default:			/* 8 bit displacement */
		displacement = (char) (inst & 0xff);

		/* check for branch to self, or to odd displacement */

		if ((displacement == 0xfe) || (displacement & 0x01))
		    return (--adrs);	/* don't follow it */

		break;
	    }

	adrs = (INSTR *) ((char *) adrs + displacement);

	inst = *adrs;		/* get the instruction */
	}

    return (adrs);
    }
