/* tiLib.c - function execution timer utility */

static char *copyright = "Copyright 1987-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*
*/

/*
TimLib provides a set of routines which may be used to 
time the execution of programs, individual routines and arbitrary groups 
of routines.  The UniWorks system clock is used as a time base to measure 
execution times.  Routines which have a short execution time relative to 
this time base may be called repetitively to establish an average execution 
time with an acceptable error percentage. 

Up to 4 routines may be specified as the group of functions to be
timed.  Additionally, sets of up to 4 routines may be specified as pre 
or post functions which are executed before and after the timed routines 
respectively.
tiPre and tiPost are used to specify pre and post functions,
while tiFunc is used to specify the routines to be timed.  

The routine timeF (2) is used to time one iteration of a function.
If no argument list is given, timeF uses the functions specified in the
tiPre, tiPost and tiFunc calls.
If arguments are supplied, they are used in lieu of the
functions previously specified.  The routine timeFN (2) may be used to iterate
function calls; like timeF, arguments supplied in the timeFN call are used
instead of previously defined functions.

EXAMPLES

timeF may be used to obtain the execution time of a routine:

	timeF (myFunc, myArg1, myArg2);

timeFN calls a function repeatedly until a set error tolerence is obtained:

	timeFN (myFunc, myArg1, myArg2);

tiPre, tiPost, and tiFunc may be used to specify groups of functions:

	tiPre (0, myFirstPreFunc, preArg1, preArg2);
	tiPre (1, mySecondPreFunc, preArg3);

	tiFunc (0, myFunc1, myArg1, myArg2);
	tiFunc (1, myFunc2, myArg3);
	tiFunc (2, myFunc3, myarg4, myArg5, myArg6);

	tiPost (0, myOnlyPostFunc, postArg);

	timeF ();
	 or
	timeFN ();

When timeF or timeFN are called, myFirstPreFunc and mySecondPreFunc are 
called with their respective arguments.  MyFunc1, myFunc2, and myFunc3 
are then called in sequence, and timed.  If timeFN was used, the sequence 
is called repeatedly until a 2% or better error tolerence is achieved.
Finally, myOnlyPostFunc is called with its argument.  The timing
results are reported after the post-functions are called.

The timings measure the time of execution of the routine body, without
the usual subroutine entry and exit code (usually just LINK, UNLINK, and
RTS instructions).  Also the time required to set up the arguments 
and call the routines is not included in the reported times.  
This is because these timing routines automatically calibrate themselves 
by timing the invocation of a null routine, 
and thereafter subtracting that constant overhead.
*/

#include "UniWorks.h"
#include "symLib.h"
#include "sysSymTbl.h"

#define MAX_CALLS	4		/* max functions in each category */
#define MAX_ARGS	8		/* max args to each function */

#define MAX_PERCENT	2		/* max percent error for auto-timing */
#define MAX_REPS	20000		/* max reps for auto-timing */

IMPORT ULONG tickGet ();		/* to get time in ticks */

typedef struct			/* CALL */
    {
    FUNCPTR func;			/* function to call */
    int arg [MAX_ARGS];			/* args to function */
    } CALL;

typedef CALL CALL_ARRAY [MAX_CALLS];

LOCAL char *timeFScaleText [] =
    {
    "secs",			/* scale = 0 */
    "millisecs",		/* scale = 1 */
    "microsecs",		/* scale = 2 */
    };


LOCAL VOID timeFNull ();	/* forward declaration */

LOCAL CALL_ARRAY timeFPreCalls =	/* calls to be made just before timing*/
    {
    { timeFNull },
    { timeFNull },
    { timeFNull },
    { timeFNull },
    };

LOCAL CALL_ARRAY timeFTimeCalls =	/* calls to be timed */
    {
    { timeFNull },
    { timeFNull },
    { timeFNull },
    { timeFNull },
    };

LOCAL CALL_ARRAY timeFPostCalls =	/* calls to be made just after timing */
    {
    { timeFNull },
    { timeFNull },
    { timeFNull },
    { timeFNull },
    };

LOCAL CALL_ARRAY timeFNullCalls =
    {
    { timeFNull },
    { timeFNull },
    { timeFNull },
    { timeFNull },
    };

LOCAL int overhead;		/* usecs of overhead per rep in timing test */

/*******************************************************************************
*
* timeFInit - include timeFInit
*
* This null routine is provided so the linker will pull in the UniWorks
* function execution timer utility library.
*/

VOID timeFInit ()
    {
    }
/**************************************************************************
*
* tiClear - clear timeF function calls
*
* This routine clears all the lists of functions involved in the timing.
*/

VOID tiClear ()

    {
    timeFClrArrays ();
    tiShow ();
    }
/**************************************************************************
*
* tiFunc - specify a function to be timed
*
* This routine allows alteration of the list of functions to be timed.
* Any function in the list of 4 functions can be added or deleted.
*/

VOID tiFunc (i, func, arg1)
    int i;		/* function number in list (0..3) */
    FUNCPTR func;	/* function to be added (NULL if 
			   to be deleted) */
    int arg1;		/* first of up to 10 arbitrary args
			   with which to call function */

    {
    timeFAddCall (timeFTimeCalls, i, func, &arg1);
    tiShow ();
    }
/**************************************************************************
*
* tihelp - print help screen for timeF
*
* This routine prints a help screen which shows all the available
* execution timer functions.
*
* .CS
*  timeF functions:
*  timeF       [func,[args...]]    - time single execution
*  timeFN      [func,[args...]]    - time repeated execution
*  tihelp                          - display this list
*  tiClear                         - clear all functions
*  tiFunc      i,func,[args...]    - add function to be timed
*  tiPre       i,func,[args...]    - add function to be called before timing
*  tiPost      i,func,[args...]    - add function to be called after timing
*  tiShow                          - show all functions to be called
*  
*  Notes:
*      1) timeFN will repeat calls enough times to get timing accuracy
*         to approximately 2%.
*      2) a single function can be specified on timeF and timeFN commands;
*         alternatively functions can be pre-set with tiFunc.
*      3) up to 4 functions can be pre-set with tiFunc, tiPre,
*         and tiPost i.e. i must be in range 0..3,
*      4) tiPre and tiPost allow things like locking/unlocking,
*         or raising/lowering priority before/after timing.
* .CE
*/

VOID tihelp ()

    {
    static char *helpMsg [] =
    {
    "\ntimeF functions:\n",
    "timeF       [func,[args...]]    - time single execution",
    "timeFN      [func,[args...]]    - time repeated execution",
    "tihelp                          - display this list",
    "tiClear                         - clear all functions",
    "tiFunc      i,func,[args...]    - add function to be timed",
    "tiPost      i,func,[args...]    - add function to be called after timing",
    "tiPre       i,func,[args...]    - add function to be called before timing",
    "tiShow                          - show all functions to be called",
    "\n  Notes:",
    "      1) timeFN will repeat calls enough times to get timing accuracy",
    "         to approximately 2%%.",
    "      2) a single function can be specified on timeF and timeFN commands;",
    "         alternatively functions can be pre-set with tiFunc.",
    "      3) up to 4 functions can be pre-set with tiFunc, tiPre,",
    "         and tiPost, i.e. i must be in range 0..3",
    "      4) tiPre and tiPost allow things like locking/unlocking,",
    "         or raising/lowering priority before/after timing.",

    NULL,	/* list terminator */
    };

    FAST char **ppMsg;

    for (ppMsg = helpMsg; *ppMsg != NULL; ++ppMsg)
	printf ("%s\n", *ppMsg);
    }
/**************************************************************************
*
* timeF - time single execution
*
* This routine performs the timing of a single execution of the list of
* functions to be timed.  If a function (and optional arguments) is specified
* in the call then the entire list is replace by the single specified
* function.  The time of the execution is printed. If the execution
* was so fast in comparison to the clock rate that the time is
* meaningless (error > 50%), then a warning message is printed instead. In that
* case, use timeFN (2).
*/

VOID timeF (func, arg1)
    FUNCPTR func;	/* optional - function to time */
    int arg1;		/* optional - args with which to 
			   call function */

    {
    int scale;
    int time;
    int error;
    int percent;

    /* if function specified, clear any existing functions and add this one */

    if (func != NULL)
	{
	timeFClrArrays ();
	timeFAddCall (timeFTimeCalls, 0, func, &arg1);
	}


    /* calibrate if necessary */

    if (overhead == 0)
	timeFCal ();


    /* time calls */

    timeFTime (1, timeFPreCalls, timeFTimeCalls, timeFPostCalls, 
	    &scale, &time, &error, &percent);

    if (percent > 50)
	{
	printErr ("timeF: execution time too short to be measured \
meaningfully\n");
	printErr ("       in a single execution.\n");
	printErr ("       Type \"timeFN\" to time repeated execution.\n");
	printErr ("       Type \"tihelp\" for more information.\n");
	}
    else
	printErr ("timeF: time of execution = %d +/- %d (%d%%) %s\n",
		  time, error, percent, timeFScaleText [scale]);
    }
/**************************************************************************
*
* timeFN - time repeated execution
*
* This routine times the execution of the list of functions to be timed.
* If a function (and optional arguments) is specified in the call,
* then the entire list is replaced by the single specified function.
* The list is called a variable number of times until sufficient resolution
* is achieved to establish the time with an error less than 2% (since each 
* iteration of the list may be measured to a resolution of +/- 1 clock 
* tick, repetitive timings decrease this error to 1/N ticks, where N is 
* the number of repetitions.)
*/

VOID timeFN (func, arg1)
    FUNCPTR func;	/* optional - function to time */
    int arg1;		/* optional - args with which to call 
			   function */

    {
    int scale;
    int time;
    int error;
    int nreps;
    int percent;

    /* if func specified, clear any existing functions and add this one */

    if (func != NULL)
	{
	timeFClrArrays ();
	timeFAddCall (timeFTimeCalls, 0, func, &arg1);
	}


    /* calibrate if necessary then time calls */

    if (overhead == 0)
	timeFCal ();

    timeFAutoTime (timeFPreCalls, timeFTimeCalls, timeFPostCalls, 
		   &nreps, &scale, &time, &error, &percent);

    printErr ("timeF: %d reps, time per rep = %d +/- %d (%d%%) %s\n",
	      nreps, time, error, percent, timeFScaleText [scale]);
    }
/**************************************************************************
*
* tiPost - specify a function to be called immediately after timing
*
* This routine allows alteration of  the list of functions which are to be 
* called immediately following the timing.
* Any function in the list of 4 functions can be added or deleted.  Up to
* 10 arguments may be passed to each function.
*/

VOID tiPost (i, func, arg1)
    int i;		/* function number in list (0..3) */
    FUNCPTR func;	/* function to be added (NULL if to be 
			   deleted) */
    int arg1;		/* first of up to 10 arbitrary args
			   with which to call function */

    {
    timeFAddCall (timeFPostCalls, i, func, &arg1);
    tiShow ();
    }
/**************************************************************************
*
* tiPre - specify a function to be called immediately prior to timing
*
* This routine allows alteration of  the list of functions which are to be 
* called immediately prior to the timing.
* Any function in the list of 4 functions can be added or deleted.
*/

VOID tiPre (i, func, arg1)
    int i;		/* function number in list (0..3) */
    FUNCPTR func;	/* function to be added (NULL if to be 
			   deleted) */
    int arg1;		/* first of up to 10 arbitrary args
			   with which to call function */

    {
    timeFAddCall (timeFPreCalls, i, func, &arg1);
    tiShow ();
    }
/**************************************************************************
*
* tiShow - display all established function calls
*
* This routine displays each function currently established in each of
* the lists of function calls.  These lists are established by calls
* to tiPre (2), tiFunc (2), and tiPost (2).
*/

VOID tiShow ()

    {
    printf ("\ntimeF:");
    printf ("\n    pre-calls:\n");
    timeFShowCalls (timeFPreCalls);
    printf ("\n    timed calls:\n");
    timeFShowCalls (timeFTimeCalls);
    printf ("\n    post-calls:\n");
    timeFShowCalls (timeFPostCalls);
    }

/*****************************************************************************
*
* timeFAddCall - enter a call in a call array
*/

LOCAL VOID timeFAddCall (callArray, i, func, argptr)
    CALL_ARRAY callArray;	/* array to be altered */
    int i;			/* number of function in list to 
				   be altered */
    FUNCPTR func;		/* function to add (NULL to delete) */
    int *argptr;		/* pointer to optional args to function */

    {
    int j;

    if (func == NULL)
	func = timeFNull;

    if ((0 <= i) && (i < MAX_CALLS))
	{
	callArray[i].func = func;

	for (j = 0; j < MAX_ARGS; j++)
	    callArray[i].arg[j] = argptr[j];
	}
    else
	printf ("timeF: call number must be in range 0..%d\n", MAX_CALLS - 1);
    }
/**************************************************************************
*
* timeFAutoTime - time specified function calls with automatic scaling
*
* This routine performs the specified timing, dynamically increasing
* the number of reps until the desired accuracy is achieved.
*/

LOCAL VOID timeFAutoTime (preCalls, timeCalls, postCalls, 
		       pNreps, pScale, pTime, pError, pPercent)
    CALL_ARRAY preCalls;	/* list of functions to call before timing */
    FAST CALL_ARRAY timeCalls;	/* list of functions to time */
    CALL_ARRAY postCalls;	/* list of functions to call after timing */
    int *pNreps;	/* ptr where to return number of times called */
    int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    int *pTime;		/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    FAST int reps;

    /* start with one rep then increase reps until it takes a long
     * enough interval to provide sufficient resolution */

    reps = 1;

    timeFTime (reps, preCalls, timeCalls, postCalls,
    	       pScale, pTime, pError, pPercent);

    while ((*pPercent > MAX_PERCENT) && ((reps < MAX_REPS) || (*pTime > 0)))
	{
	reps = reps * (*pPercent) / MAX_PERCENT;

	timeFTime (reps, preCalls, timeCalls, postCalls,
		   pScale, pTime, pError, pPercent);
	}

    *pNreps = reps;
    }
/**************************************************************************
*
* timeFCal - calibrate timeF by timing null functions
*
* This routine establishes the constant per rep overhead in the timing
* function.
*/

LOCAL VOID timeFCal ()

    {
    int scale;
    int time;
    int error;
    int percent;
    int nreps;

    overhead = 0;

    timeFAutoTime (timeFNullCalls, timeFNullCalls, timeFNullCalls, 
		   &nreps, &scale, &time, &error, &percent);

    overhead = time;
    }
/**************************************************************************
*
* timeFClrArrays - clear out function arrays
*/

LOCAL timeFClrArrays ()
    {
    bcopy ((char *) timeFNullCalls, (char *) timeFPreCalls,
	   sizeof (timeFNullCalls));
    bcopy ((char *) timeFNullCalls, (char *) timeFTimeCalls,
	   sizeof (timeFNullCalls));
    bcopy ((char *) timeFNullCalls, (char *) timeFPostCalls,
	   sizeof (timeFNullCalls));
    }
/**************************************************************************
*
* timeFMakeCalls - make function calls in specified call array
*/

LOCAL VOID timeFMakeCalls (calls)
    CALL_ARRAY calls;	/* list of functions to call */

    {
    FAST int i;
    FAST CALL *pCall;

    for (i = 0, pCall = &calls[0]; i < MAX_CALLS; i++, pCall++)
	{
	(* pCall->func) (pCall->arg[0], pCall->arg[1], pCall->arg[2],
			 pCall->arg[3], pCall->arg[4], pCall->arg[5],
			 pCall->arg[6], pCall->arg[7]);
	}
    }
/********************************************************************
*
* timeFNull - null routine
*
* This routine is used as a place holder for null routines in the
* timing function arrays.  It is used to guarantee a constant calling
* overhead in each iteration.
*/

LOCAL VOID timeFNull ()
    {
    }
/********************************************************************
*
* timeFScale - scale raw timing data
*/

LOCAL VOID timeFScale (ticks, reps, pScale, pTime, pError, pPercent)
    int ticks;		/* total ticks required for all reps */
    int reps;		/* number of reps performed */
    FAST int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    FAST int *pTime;	/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    FAST int scalePerSec;

    /* calculate time per rep in best scale */

    *pScale = 0;		/* start with "seconds" scale */
    scalePerSec = 1;

    *pTime = ticks * scalePerSec / sysClkGetRate () / reps;

    while ((*pScale < 2) && (*pTime < 100))
	{
	(*pScale)++;
	scalePerSec = scalePerSec * 1000;
	*pTime = ticks * scalePerSec / sysClkGetRate () / reps;
	}


    /* adjust for overhead if in microseconds scale */

    if (*pScale == 2)
	{
	*pTime -= overhead;
	if (*pTime < 0)
	    *pTime = 0;
	}


    /* calculate error */

    *pError = scalePerSec / sysClkGetRate () / reps;

    if (*pTime == 0)
	*pPercent = 100;
    else
	*pPercent = 100 * *pError / *pTime;
    }
/**************************************************************************
*
* timeFShowCalls - print specified call array
*/

LOCAL VOID timeFShowCalls (calls)
    CALL_ARRAY calls;		/* list of functions to be displayed */

    {
    char name [MAX_SYS_SYM_LEN + 1];
    int value;
    UTINY type;
    int offset;
    int i;
    int j;
    int arg;
    int ncalls = 0;

    for (i = 0; i < MAX_CALLS; i++)
	{
	if (calls[i].func != timeFNull)
	    {
	    ncalls++;

	    symValFind (sysSymTbl, (int)calls[i].func, name, &value, &type);
	    name [MAX_SYS_SYM_LEN] = EOS;
	    offset = (int) calls[i].func - value;

	    if (offset == 0)
		printf ("        %d: %s (", i, name);
	    else
		printf ("        %d: %s+%x (", i, name, offset);

	    for (j = 0; j < MAX_ARGS; j++)
		{
		if (j != 0)
		    printf (", ");

		arg = calls[i].arg[j];
		if ((-9 <= arg) && (arg <= 9))
		    printf ("%d", arg);
		else
		    printf ("0x%x", arg);
		}

	    printf (")\n");
	    }
	}

    if (ncalls == 0)
	printf ("        (none)\n");
    }
/**************************************************************************
*
* timeFTime - time a specified number of reps
*/

LOCAL VOID timeFTime (reps, preCalls, timeCalls, postCalls,
		      pScale, pTime, pError, pPercent)
    FAST int reps;		/* number of reps to perform */
    CALL_ARRAY preCalls;	/* list of functions to call before timing */
    FAST CALL_ARRAY timeCalls;	/* list of functions to be timed */
    CALL_ARRAY postCalls;	/* list of functions to call after timing */
    int *pScale;	/* ptr where to return scale:
			 *   0 = secs, 1 = millisecs, 2 = microsecs */
    int *pTime;		/* ptr where to return time per rep in above units */
    int *pError;	/* ptr where to return error margin in above units */
    int *pPercent;	/* ptr where to return percent error (0..100) */

    {
    int start;
    int end;
    FAST int i;

    timeFMakeCalls (preCalls);

    start = (int) tickGet ();

    for (i = 0; i < reps; i++)
	timeFMakeCalls (timeCalls);

    end = (int) tickGet ();

    timeFMakeCalls (postCalls);

    timeFScale (end - start, reps, pScale, pTime, pError, pPercent);
    }
