/* remLib.c - remote command library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*#define	DEBUG	/**/

/* 
modification history
--------------------
*/

/*
The library remLib supplies several routines found under BSD 4.2 (rcmd and 
rreservport), and several other routines useful for accessing files
via ethernet.

INCLUDE FILE
remLib.h

SEE ALSO
inetLib(1), "UniWorks Network"
*/

#include "UniWorks.h"
#include "socket.h"
#include "in.h"
#include "remLib.h"
#include "inetLib.h"
#include "strLib.h"

#define MAX_RETRY_CONNECT 5

LOCAL char remUser [MAX_IDENTITY_LEN];
LOCAL char remPasswd [MAX_IDENTITY_LEN];

int remLastResvPort = IPPORT_RESERVED - 1;

/*******************************************************************************
*
* rcmd - execute a shell command on a remote machine
*
* This routine executes a command on a remote machine, using the remote
* shell daemon (rshd) on the remote system.  It is analogous to the Unix
* rcmd (3) call.
* 
* RETURNS
*  socket descriptor if remote shell daemon accepts, or
*  ERROR if remote command failed.
*
* SEE ALSO: BSD 4.2 manual entry for rcmd
*/

int rcmd (host, remotePort, localUser, remoteUser, cmd, fd2p)
    char *host; 	/* host name or inet address */
    int remotePort;	/* remote port to connect to (rshd) */
    char *localUser;	/* local user name */
    char *remoteUser;	/* remote user name */
    char *cmd;		/* command */
    int *fd2p;		/* if this pointer is non-zero, stderr socket
			 * is opened and socket descriptor is filled in */

    {
    int sd;
    struct sockaddr_in sin;
    struct sockaddr_in mySin;
    int mySinLen;
    char c;
    int lport = IPPORT_RESERVED - 1;
    int nTries = 0;
    int stdErrPort;
    int stdErrSocket;
    char stringBuf [20];

#ifdef	DEBUG

	printf("rcmd (%s, %d, %s, %s, %s, 0x%x)\n",host,remotePort,
		localUser,remoteUser,cmd,fd2p);
#endif	DEBUG

    sin.sin_family = AF_INET;
    sin.sin_port = remotePort;
    if (((sin.sin_addr.s_addr = remGetHostByName (host)) == ERROR) &&
        ((sin.sin_addr.s_addr = inet_addr (host)) == ERROR))
	{
	return (ERROR);
	}

    do
	{
	sd = rresvport (&lport);
	if (sd == ERROR)
	    return (ERROR);

	if (connect (sd, (SOCKADDR *) &sin, sizeof (sin)) == ERROR) 
	    {
	    close (sd);
	    lport--;
	    }
	else
	    break;
	}
    while (++nTries <= MAX_RETRY_CONNECT);

    if (nTries > MAX_RETRY_CONNECT)
	return (ERROR);

    if (fd2p == 0) 
        {
        send (sd, "", 1, 0);
        lport = 0;
        } 
    else 
        {
	stdErrPort = --lport;

	stdErrSocket = rresvport (&stdErrPort);
	if (stdErrSocket == ERROR)
	    {
	    close (sd);
	    return (ERROR);
	    }
	listen (stdErrSocket, 1);

	sprintf (stringBuf, "%d", stdErrPort);
	send (sd, stringBuf, strlen (stringBuf) + 1, 0);

	/* wait for rshd to connect */
	mySinLen = sizeof (mySin);
	*fd2p = accept (stdErrSocket, (SOCKADDR *)&mySin, &mySinLen);  
	if (*fd2p == ERROR)
	    {
	    close (sd);
	    close (stdErrSocket);
	    return (ERROR);
	    }
	close (stdErrSocket);
	}

    send (sd, localUser, strlen(localUser) + 1, 0);
    send (sd, remoteUser, strlen(remoteUser) + 1, 0);

    if (send (sd, cmd, strlen(cmd) + 1, 0) <= 0)
	{
	close (sd);
	if (fd2p != NULL)
	    close (*fd2p);
	return (ERROR);
	}

    /* bsd documentation for rshd is incorrect - null byte is actually 
       received on stdin socket */

    if (recv (sd, &c, 1, 0) <= 0)
	{
	close (sd);
	if (fd2p != NULL)
	    close (*fd2p);
	return (ERROR);
	}

    if (c != 0)
	{
	/* error will come in on stdin socket */
	while (recv (sd, &c, 1, 0) == 1) 
	    {
	    write (STD_ERR, &c, 1);
	    if (c == '\n')
		break;
	    }

	errnoSet (S_remLib_RSH_ERROR);
	close (sd);

	if (fd2p != NULL)
	    close (*fd2p);

	return (ERROR);
	}
    
    return (sd);
    }
/*******************************************************************************
*
* rresvport - open a socket with a privileged port bound to it
*
* Analogous to the Unix rresvport (3) command.
*
* RETURNS
*  Socket descriptor, or
*  ERROR if couldn't open socket, or
*  all ports in use.
*
* SEE ALSO: BSD 4.2 manual entry for rresvport
*/

int rresvport (alport)
    FAST int *alport;	/* port number to initially try */

    {
    struct sockaddr_in sin;
    int sd;

#ifdef	DEBUG
	printf("rresvport(%d)\n",*alport);
#endif	DEBUG

    sin.sin_family = AF_INET;
    sin.sin_addr.s_addr = 0;

    if ((sd = socket (AF_INET, SOCK_STREAM, 0)) == ERROR)
	return (ERROR);

    *alport = remLastResvPort;

    for (;;) 
        {
        (*alport)--;

        if (*alport == IPPORT_RESERVED - 400)	/* approx lower bound */
	    *alport = IPPORT_RESERVED - 1;

	if (*alport == remLastResvPort)
            {
	    errnoSet (S_remLib_ALL_PORTS_IN_USE);
            return (ERROR);
            }

        sin.sin_port = htons ((u_short)*alport);

        if (bind (sd, (SOCKADDR *) &sin, sizeof (sin)) != ERROR)
	    {
	    remLastResvPort = *alport;
            return (sd);
	    }
        }
    }
/*******************************************************************************
*
* remGetCurId - get current user name and password
*
* This routine passes back the user name and password currently being used
* for remote host access priveleges.
* Either pointer may be NULL and the corresponding item will not be returned.
*
* SEE ALSO: iam(2), whoami(2)
*/

VOID remGetCurId (user, passwd)
    char *user;		/* where to return current user name */
    char *passwd;	/* where to return current password */

    {
    if (user != NULL)
	strcpy (user, remUser);

    if (passwd != NULL)
	strcpy (passwd, remPasswd);

#ifdef	DEBUG
	printf("remGetCurId(%s, %s)\n",user,passwd);
#endif	DEBUG
    }
/*******************************************************************************
*
* remCurIdSet - set remote user name and password
*
* The user name given here determines the access privileges available
* on the remote machine.  It should, of course, exist in /etc/passwd on
* that machine.
*
* Either pointer may be NULL in which case the corresponding item is set
* to the NULL string.
*
* The maximum length of the user name and the password is MAX_IDENTITY_LEN
* (defined in remLib.h).
*
* RETURNS
*  OK,   or
*  ERROR if name or password too long.
*
* SEE ALSO: iam(2), whoami(2)
*/

STATUS remCurIdSet (newUser, newPasswd)
    char *newUser;	/* user name to use on remote */
    char *newPasswd;	/* password to use on remote (NULL = none) */

    {
    if (((newUser != NULL) && (strlen (newUser) > MAX_IDENTITY_LEN-1)) ||
        ((newPasswd != NULL) && (strlen (newPasswd) > MAX_IDENTITY_LEN-1)))
	{
	errnoSet (S_remLib_IDENTITY_TOO_BIG);
	return (ERROR);
	}

    if (newUser == NULL)
	remUser[0] = EOS;
    else
	strcpy (remUser, newUser);

    if (newPasswd == NULL)
	remPasswd[0] = EOS;
    else
	strcpy (remPasswd, newPasswd);

    return (OK);
    }
