/* hostLib.c - host table subroutine library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*#define	DEBUG	/**/

/* 
modification history
--------------------
*/

/*
DESCRIPTION
This module provides routines to access the network host database.

SEE ALSO
inetLib(1), "UniWorks Network"
*/

#include "UniWorks.h"
#include "types.h"
#include "in.h"
#include "memLib.h"
#include "semLib.h"
#include "strLib.h"
#include "inetLib.h"
#include "lstLib.h"
#include "hostLib.h"

typedef struct hostname_struct
    {
    struct hostname_struct *link;
    char *name;
    } HOSTNAME;

typedef struct
    {
    NODE node;
    HOSTNAME hostName;
    struct in_addr netAddr;
    } HOSTENTRY;

LOCAL LIST hostList;
LOCAL SEM_ID hostListSemId;

/*******************************************************************************
*
* hostTblInit - initialize network host database table
*
* This routine must be called to initialize the host list data structure
* used by routines throughout this module.  It is called usually in
* usrConfig(1), before any other routines in this module.
*/

VOID hostTblInit ()

    {
    static BOOL done;	/* FALSE = not done, TRUE = already initialized */

    if (!done)
	{
	done = TRUE;
	lstInit (&hostList);
	hostListSemId = semCreate ();
	semGive (hostListSemId);
	}
    }
/*******************************************************************************
*
* remAddHost - add a host to the host table
*
* This command is used to add a hostname to the UniWorks host table.  This must
* be called before sockets on the remote host are opened, or files on the
* remote host are accessed via netDrv (3).
*
* The host table has one entry per internet address.
* More than one name may be used for an address.
* Additional host names are added as aliases.
*
* RETURNS:
*  OK, or
*  ERROR if host table full,
*  host name already entered,
*  invalid inet address, or
*  out of memory
*/

STATUS remAddHost (hostName, hostAddr)
    char *hostName;	/* host name */
    char *hostAddr;	/* host address in standard internet format */

    {
    HOSTNAME *pHostNamePrev;	/* pointer to previous host name entry */
    FAST HOSTNAME *pHostName;	/* pointer to host name entry */
    FAST HOSTENTRY *pHostEntry;
    struct in_addr netAddr;	/* network address */

#ifdef	DEBUG
	printf("remAddHost(%s, %s)\n",hostName,hostAddr);
#endif	DEBUG

    if ((netAddr.s_addr = inet_addr (hostAddr)) == ERROR)
	return (ERROR);
    
    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	if (pHostEntry->netAddr.s_addr == netAddr.s_addr)
	    {
	    /* host internet address already in table, add name as an alias */

	    for (pHostName = &pHostEntry->hostName;
		 pHostName != NULL;
		 pHostName = pHostName->link)
		{
		/* make sure name is not already used for this address */

		if (strcmp (pHostName->name, hostName) == 0)
		    {
		    semGive (hostListSemId);
		    errnoSet (S_hostLib_HOST_ALREADY_ENTERED);
		    return (ERROR);
		    }
	
		pHostNamePrev = pHostName;
		}
	    
	    /* name not used for this address, add it as an alias */

	    pHostNamePrev->link = (HOSTNAME *) malloc (sizeof (HOSTNAME));

	    if (pHostNamePrev->link == NULL ||
		remFillHostName (pHostNamePrev->link, hostName) == ERROR)
		{
		semGive (hostListSemId);
		return (ERROR);
		}

	    semGive (hostListSemId);
	    return (OK);
	    }
	}

    /* host name and internet address not in host table, add new host */

    if ((pHostEntry = (HOSTENTRY *)malloc (sizeof (HOSTENTRY))) == NULL ||
	remFillHostName (&pHostEntry->hostName, hostName) == ERROR)
	{
	semGive (hostListSemId);
	return (ERROR);
	}

    pHostEntry->netAddr = netAddr;

    lstAdd (&hostList, &pHostEntry->node);

    semGive (hostListSemId);
    return (OK);
    }
/*******************************************************************************
*
* remGetHostByName - look up a host in the host table
*
* This routine returns the internet address of a host that has been added
* to the UniWorks host table by remAddHost (2).
*
* RETURNS
*  Internet Address (as an int), or
*  ERROR - if host unknown.
*/

int remGetHostByName (name)
    char *name;		/* name of host */

    {
    FAST HOSTNAME *pHostName;
    FAST HOSTENTRY *pHostEntry;
    int retAddr = ERROR;

#ifdef	DEBUG
	printf("remGetHostByName(%s)\n",name);
#endif	DEBUG

    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	/* check official name */

	if (strcmp (pHostEntry->hostName.name, name) == 0)
	    {
	    retAddr = (int)pHostEntry->netAddr.s_addr;
	    break;
	    }

	/* check aliases */

	for (pHostName = pHostEntry->hostName.link;
	     pHostName != NULL;
	     pHostName = pHostName->link)
	    {
	    if (strcmp (pHostName->name, name) == 0)
		{
		retAddr = (int)pHostEntry->netAddr.s_addr;

		/* force termination of outer loop */
		pHostEntry = (HOSTENTRY *)lstLast (&hostList);
		break;
		}
	    }
	}

    if (retAddr == ERROR)
	errnoSet (S_hostLib_UNKNOWN_HOST);

    semGive (hostListSemId);
    return (retAddr);
    }
/*******************************************************************************
*
* remShowHosts - show the host table
*
* Prints a list of remote hosts along with their internet addresses and aliases.
*/

VOID remShowHosts ()

    {
    char inetBuf [INET_ADDR_LEN];
    FAST HOSTNAME *pHostName;
    FAST HOSTENTRY *pHostEntry;

    printf ("hostname         inet address       aliases\n");
    printf ("--------         ------------       -------\n");

    semTake (hostListSemId);

    for (pHostEntry = (HOSTENTRY *)lstFirst (&hostList);
	 pHostEntry != NULL;
	 pHostEntry = (HOSTENTRY *)lstNext (&pHostEntry->node))
	{
	inet_ntoa_b (pHostEntry->netAddr, inetBuf);
	printf ("%-16s %-18s", pHostEntry->hostName.name, inetBuf);

	for (pHostName = pHostEntry->hostName.link;
	     pHostName != NULL;
	     pHostName = pHostName->link)
	    {
	    printf ("%s ", pHostName->name);
	    }

	printf ("\n");
	}

    semGive (hostListSemId);
    }
/*******************************************************************************
*
* remFillHostName - fill in host name
*
* RETURNS: OK or ERROR if out of memory
*/

LOCAL STATUS remFillHostName (pHostName, newHostName)
    FAST HOSTNAME *pHostName;
    char *newHostName;

    {
    FAST char *pName = malloc ((unsigned) (strlen (newHostName) + 1));

    if (pName == NULL)
	return (ERROR);

    strcpy (pName, newHostName);
    pHostName->name = pName;
    pHostName->link = NULL;

    return (OK);
    }
