/* dbgLib.c - debugging aids */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This module provides UniWorks primary interactive debugging aids.
It provides the following facilities:

  - task breakpoints,
  - task single-stepping.

In addition, it provides the "glue" necessary to provide enhanced 
use of other UniWorks modules, including:

  - symbolic disassembly (via dsmLib),
  - symbolic task stack tracing (via trcLib),
  - enhanced shell abort and exception handling (via tyLib and excLib).

This module is optional.  It may be eliminated from production
systems simply by removing the call to dbgInit (see below).

INITIALIZATION
To initialize the debugging facilities provided by this package,
the routine dbgInit must be called.
DbgInit must be called before any other routines in this package are called.
This is normally done by the root task, usrRoot (2), in usrConfig (1).

BREAKPOINTS
You can set breakpoints in tasks with the b (2) command.  Breakpoints may
be set so that any task hitting a breakpoint will hit it, or so that
only a specific task will hit it.  Multiple breakpoints, for different
tasks, may be set at the same address.  The breakpoints are cleared
with bd (2) and bdall (2).

When a breakpoint is hit, the task hitting it is suspended and a message
is printed on the console.  At this point, you can examine the task, do
a trace on it (with tt (2)) to find out how it got where it is, change
variables used by the task, delete the task, etc.  If you examine the
task status at this point (by the i (2) command), you will find that it
is suspended.  The instruction at the breakpoint address has not yet
been executed.

When you wish the task to continue executing, use the c (2) command.
At this point, the instruction that had contained the breakpoint is executed,
and the task will then continue.  The breakpoint remains in until it is
explicitly removed.

UNBREAKABLE TASKS
When a task is "unbreakable" it ignores all breakpoints.
Tasks can be spawned "unbreakable" by specifying the VX_UNBREAKABLE task
option.  Tasks can subsequently be set "unbreakable" or "breakable" by setting
or resetting the VX_UNBREAKABLE task option with the taskOptionsSet(2) routine.
Several UniWorks tasks are spawned unbreakable, including the shell,
the exception support task (excTask), and several network related tasks.

DISASSEMBLER AND STACK TRACER
The routine l (2) provides a symbolic disassembler, using the low-level
disassembly routines in dsmLib.
The routine tt (2) provides a symbolic stack tracer, using the low-level
stack trace routines in trcLib.

SHELL ABORT AND EXCEPTION HANDLING
This package includes enhanced support for the shell in a debugging
environment.
The terminal "abort" function is set to restart the shell.
This function is invoked when the "abort" key is
typed on a terminal for which the OPT_ABORT option has been set.
The "abort" key is by default CONTROL-C.
See tySetAbort(2) and tyAbortFunc(2).

DEFAULT TASK
Many routines in this package take a task id as an argument.
In all such cases, if the task id argument is missing or zero,
the last task id referenced is used.
This package uses the routine taskIdDefault(1) to set and get
the "last referenced task id", as do many UniWorks routines in other modules.

EXCLIB
DbgLib uses the facilities of excLib, including excTask,
to support breakpoints and single-stepping and additional exception
handling functions.

SEE ALSO
excLib (1), excALib (1), tySetAbort(2), tyAbortFunc(2), excSetExceptRtn(2)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "iv68k.h"
#include "esf68k.h"
#include "ioLib.h"
#include "lstLib.h"
#include "memLib.h"
#include "symLib.h"
#include "taskLib.h"
#include "sysSymTbl.h"


/* interrupt driver routines from dbgALib.s */

IMPORT dbgBpStub ();		/* breakpoint interrupt driver */
IMPORT dbgTraceStub ();		/* trace interrupt driver */
IMPORT TCBX *taskTcbX ();	/* get task tcbX */

#define	LINK		0x4e50
#define	LINK_MASK	0xfff8
#define	RTS		0x4e75
#define	RTS_MASK	0xffff
#define	JSR		0x4e80
#define	JSR_MASK	0xffc0
#define	BSR		0x6100
#define	BSR_MASK	0xff00
#define	DSM(addr,inst,mask)	((*addr & mask) == inst)

#define TRAP0_INST	0x4e40	/* opcode for trap 0 */
#define	TRACE_BIT	0x8000	/* trace bit in 68K SR */

#define ALL		0	/* breakpoint applies to all tasks */
#define LAST		0	/* continue applies to task that last broke */
#define MAX_ARGS	8	/* max args to task level call */

#define CONTINUE	0	/* trace exception is due to a continue */
#define SINGLE_STEP	1	/* trace exception is due to a single-step */
#define STEP_FROM_BREAK	2	/* trace due to a single-step from breakpoint */

#define BRK_NORM	1	/* normal breakpoint type */
#define BRK_SO		2	/* step-over breakpoint type */
#define BRK_NORM_QUIET	3	/* normal breakpoint type, no console output */

typedef struct		/* BRKENTRY: breakpoint table entry */
    {
    NODE node;		/* link list node */
    INSTR *addr;	/* breakpoint address */
    short code;		/* code that goes at breakpoint address */
    int   task;		/* task for which breakpoint valid (0 = all tasks) */
    int   count;	/* pass count */
    short type;		/* kind of breakpoint */
    } BRKENTRY;

#if (CPU==MC68000)
#define TRACE_ESF ESF68K
#endif
#if (CPU==MC68010)
#define TRACE_ESF ESF0
#endif
#if (CPU==MC68020)
#define TRACE_ESF ESF2
#endif
#if (CPU==MC68030)
#define TRACE_ESF ESF2
#endif

#if (CPU==MC68000)
#define BREAK_ESF ESF68K
#else
#define BREAK_ESF ESF0
#endif

/* Macro's are used by routines executing at interrupt level (task switch),
 * so that fatal breakpoints are avoided.
 */
#define	LSTFIRST(list)	(list).node.next
#define	LSTNEXT(node)	(node).next

IMPORT int shellTaskId;
IMPORT ULONG taskIdCurrent;
#ifdef	is68k
IMPORT	BOOL	cdbQuietFlag;	/* TRUE when cdb is connected, want quiet */
#endif	is68k

/* local variables */

#ifdef	is68k
FUNCPTR	dbgBreakNotifyRtn;	/* if not NULL, call this routine whenever */
#else	is68k
LOCAL FUNCPTR dbgBreakNotifyRtn;/* if not NULL, call this routine whenever
				 * a breakpoint occurs.  This is normally
				 * set by a source level debugger module
				 */
#endif	is68k
LOCAL LIST dbgBrkList;		/* list of active breakpoints */
LOCAL LIST dbgFreeList;		/* list of free breakpoint entries */
LOCAL BRKENTRY *resumeBreak;	/* breakpoint from which task is proceeding */
LOCAL BRKENTRY *interruptBreak = NULL;	/* breakpoint hit at interupt level */
LOCAL unsigned short interruptSR;       /* old SR value before interrupt brk */

LOCAL int resumeTask   = NONE;	/* task proceeding from breakpoint */
LOCAL int sstepTask    = NONE;	/* single-stepping task */
LOCAL BOOL sstepQuiet  = FALSE;	/* single step quietly (to console output) */
LOCAL int traceCause   = NONE;	/* cause of trace exception */

LOCAL INSTR *nextToDsm = NULL;	/* instruction to disassemble from */
LOCAL int dfltCount    = 10;	/* number of instructions to disassemble */

LOCAL int breakInst;		/* opcode of break instruction */

/* forward declarations */

LOCAL BRKENTRY *dbgBrkDelete ();
LOCAL BRKENTRY *dbgBrkGet ();
LOCAL VOID dbgTaskSwitch ();
LOCAL VOID dbgTyAbort ();
LOCAL VOID dbgTlBreak ();
LOCAL VOID dbgTlSnglStep ();
LOCAL VOID dbgTlTyAbort ();
LOCAL INSTR *dbgList ();
LOCAL VOID dbgPrintAdrs ();
LOCAL VOID dbgBreakHook ();
VOID dbgPrintCall ();

/*******************************************************************************
*
* dbghelp - print dbgLib help menu
*
* This routine prints a help display on standard out, which gives
* short descriptions of some of the debug utilities:
*
* .CS
*  dbghelp                        Print this list
*  dbgInit                        Install debug facilities
*  b                              Display breakpoints
*  b         addr[,task[,count]]  Set breakpoint
*  bd        addr[,task]          Delete breakpoint
*  bdall     [task]               Delete all breakpoints
*  c         [task[,addr]]        Continue from breakpoint
*  cret      [task]               Continue to subroutine return
*  s         [task[,addr]]        Single step
*  so        [task]               Single step/step over subroutine
*  l         [adr[,nInst]]        List disassembled memory
*  tt        [task]               Do stack trace on task
* .CE
*
* SEE ALSO: "Debugging"
*/

VOID dbghelp ()

    {
    static char *help_msg [] = {
    "dbghelp                         Print this list",
    "dbgInit                         Install debug facilities",
    "b                               Display breakpoints",
    "b         addr[,task[,count]]   Set breakpoint",
    "bd        addr[,task]           Delete breakpoint",
    "bdall     [task]                Delete all breakpoints",
    "c         [task[,addr]]         Continue from breakpoint",
    "cret      [task]                Continue to subroutine return",
    "s         [task[,addr]]         Single step",
    "so        [task]                Single step/step over subroutine",
    "l         [adr[,nInst]]         List disassembled memory",
    "tt        [task]                Do stack trace on task",
    NULL
    };
    int i;

    for (i = 0; help_msg [i] != NULL; i++)
	printf ("%s\n", help_msg [i]);
    }
/*******************************************************************************
*
* dbgInit - initialize debug package
*
* This routine installs the debug package.  This involves:
*
*	- setting the interrupt vectors of the breakpoint
*	  and trace hardware interrupts to the appropriate
*	  debug interrupt handlers,
*
*	- adding the debug task switch routine to the kernel 
*	  context switch call out,
*
*	- setting the terminal "abort" function to restart
*	  the shell,
*
*	- setting the exception handling extension to
*	  trace and restart the shell if the exception
*	  occurred in the shell task.
*
* WHEN TO CALL
* It is usually desirable to install the debugging facilities
* as early as possible in system initialization, if the debugging facilities
* are to be included at all.  It should not, however, be called until 
* the pipe driver, the ty driver used for logging, the exception handler
* library (excLib (1)), and message logging library (logLib (1)) have
* been initialized.
*
* RETURNS: OK, or ERROR if unable to install switch routine
*/

STATUS dbgInit (breakpointTrapNum)
    int breakpointTrapNum;	/* trap number for breakpoints */

    {
    lstInit (&dbgBrkList);
    lstInit (&dbgFreeList);

    tyAbortFunc (dbgTyAbort);		/* set abort routine */

    /* add debug task switch routine */

    if (taskSwitchHookAdd (dbgTaskSwitch) != OK)
	return (ERROR);

    /* Insert the new breakpoint and trace vectors */

    intSetVec (TRAPNUM_TO_IVEC (breakpointTrapNum), dbgBpStub);
    intSetVec (IV_TRACE, dbgTraceStub);

    /* calculate breakpoint instruction */

    breakInst = TRAP0_INST + breakpointTrapNum;

    /* set break remove/install hook routine for taskOptionsSet */

    taskBpHookSet (dbgBreakHook);

    return (OK);
    }
/*******************************************************************************
*
* b - set breakpoint
*
* This routine is used to set or examine breakpoints. To see the list of
* currently active breakpoints, call it without arguments:
*
*	b
*
* The list shows the address, task and passcount of each breakpoint.
* The routines so(2) and cret(2) insert temporary breakpoints and are so marked.
*
* To set a new breakpoint, call it with at least an address:
*
*	b addr [,task [,passcount] [, quiet]]
*
* If the task is zero or missing, the breakpoint will apply to all breakable
* tasks. If the passcount is zero or missing, the breakpoint will happen every
* time it is hit.
* The breakpoint address may be specified numerically or symbolically, with
* an optional offset.
*
* If a passcount is specified, the break will not occur until the (count+1)th
* time the breakpoint is hit by an eligible task. That is, it will ignore the
* breakpoint the first 'count' times it is hit.
*
* If quiet is specified, debugging information destined for the console
* will be supressed when the breakpoint is hit.  This option is included
* for use by external source code debuggers that handle the breakpoint
* user interface themselves.
*
* Individual tasks can be "unbreakable" in which case breakpoints that
* otherwise would apply to a task are ignored.  Tasks can be spawned
* "unbreakable" by specifying the VX_UNBREAKABLE task option.
* Tasks can also be set "unbreakable" or "breakable" by setting or resetting
* the VX_UNBREAKABLE task option with the routine taskOptionsSet(2).
*
* RETURNS
*    OK, or
*    ERROR if addr is odd, or in non-existent memory, or
*    breakpoint table is full.
*
* SEE ALSO: "Debugging", taskOptionsSet (2)
*/

STATUS b (addr, superName, count, quiet)
    FAST INSTR *addr;		/* address at which to set breakpoint;
				 * 0 = display all breakpoints */
    int superName;		/* super name of task for which to set
				 * breakpoint,  0 = set all tasks */
    int count;			/* number of passes before hit */
    BOOL quiet;			/* TRUE = don't print debugging info,
				   FALSE = print debugging info */
    {
    int task = 0;

    if (superName != 0)
	task = taskIdFigure (superName);	/* convert super name to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    /* 0 address means show all breakpoints */

    if (addr == NULL) 
	{
	dbgBrkDisplay ();
	return (OK);
	}

    /* be sure address isn't odd, or beyond end of memory */

#if (CPU!=MC68020)
    if (((int)addr & 1) || (sysMemProbe ((char *)addr) != OK))
#else
    if (sysMemProbe ((char *)addr) != OK)
#endif
	{
	printf ("Illegal Breakpoint Address: $%06x\n", addr);
	return (ERROR);
	}

    /* add new breakpoint */

    if (dbgBrkAdd (addr, task, count, quiet ? BRK_NORM_QUIET : BRK_NORM) != OK)
	{
	printf ("Breakpoint table is full.\n");
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* bd - delete breakpoint
*
* This routine is used to delete a specific breakpoint. To use it type:
*
* 	bd addr [,task]
*
* If task is missing or zero, the breakpoint will be removed for all tasks.
* If the breakpoint applies to all tasks, removing it for only a single
* task will be ineffective.  It must be removed for all tasks, and then set
* for just those tasks desired.  Temporary breakpoints inserted by so (2),
* or cret (2), may be deleted.
*
* RETURNS
*  OK, or
*  ERROR if no breakpoint at specified address.
*
* SEE ALSO: "Debugging"
*/

STATUS bd (addr, superName)
    FAST INSTR *addr;	/* address of breakpoint to delete */
    FAST int superName;	/* task for which to delete breakpoint;
			 * 0 = delete for all tasks */

    {
    FAST BRKENTRY *bp;
    int task = 0;

    if (superName != 0)
	task = taskIdFigure (superName);	/* convert Sname to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    /* search breakpoint table for entry with correct address & task */

    for (bp = (BRKENTRY *) lstFirst (&dbgBrkList);
	 bp != NULL;
	 bp = (BRKENTRY *) lstNext (&bp->node))
	{
	if (bp->addr == addr && (bp->task == task || task == ALL))
	    {
	    (void) dbgBrkDelete (bp);	/* clear breakpoint table entry */
	    return (OK);
	    }
	}

    printf ("No breakpoint at $%06x", addr);

    if (task == ALL)
	printf ("\n");
    else
        printf (" for task %d\n", task);

    return (ERROR);
    }
/*******************************************************************************
*
* bdall - delete all breakpoints
*
* This routine removes all breakpoints. It is called as follows:
*
*	bdall [task]
*
* If a task is specified, all of the breakpoints that apply just to that task
* are removed.  Otherwise all breakpoints for all tasks are removed.
*
* Temporary breakpoints inserted by so (2), or cret (2), are not deleted;
* use bd (2).
*
* RETURNS: OK (always)
*
* SEE ALSO: "Debugging"
*/

STATUS bdall (superName)
    FAST int superName;	/* task for which to delete breakpoints;
			 * 0 = delete for all tasks */

    {
    FAST BRKENTRY *bp;
    int task = 0;

    if (superName != 0)
	task = taskIdFigure (superName);	/* convert Sname to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    bp = (BRKENTRY *) lstFirst (&dbgBrkList);
    /* search breakpoint table for entries to delete */

    while (bp != NULL)
	{
	if ((bp->type == BRK_NORM || bp->type == BRK_NORM_QUIET) &&
	    (bp->task == task || task == ALL))
	    {
	    bp = dbgBrkDelete (bp);	/* clear breakpoint table entry */
	    }
	else
	    bp = (BRKENTRY *) lstNext (&bp->node);
	}

    return (OK);
    }
/*******************************************************************************
*
* c - continue from breakpoint
*
* This routine is used to continue execution of a task that has stopped at a
* breakpoint. It is called as follows:
*
*	c [task [,addr]]
*
* If the task is missing or zero, the default task is assumed.
* If addr is non-zero then the program counter is changed to addr
* and the task is continued.
*
* RETURNS:
*  OK, or
*  ERROR if specified task does not exist.
*
* SEE ALSO: "Debugging"
*/

STATUS c (superName, addr)
    int superName;	/* task that should proceed from breakpoint */
    int addr;		/* address to continue at; 0 = next instruction */

    {
    int task = taskIdFigure (superName);	/* convert Sname to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    task = taskIdDefault (task);		/* set default task id */

    if (dbgCheckBreakable (task) == ERROR)
	return (ERROR);

    /* adjust pc & resume the task */

    if (addr != 0)
	dbgTaskPcSet (task, (INSTR *)addr);

    resumeTask = task;				/* mark task as resuming */

    if (taskResume (task) != OK)
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* cret - continue until return from current subroutine
*
* This routine places a breakpoint at the return address of the current
* subroutine of the given task, then continues execution of that task.
* When the breakpoint is hit, information about the task will be printed
* in the same format as in single-stepping. The breakpoint will be
* automatically removed when the breakpoint is hit, or if the task first
* hits some other breakpoint. 
*
*	cret [task]
*
* If task is missing or zero, the last task referenced is assumed.
*
* RETURNS:
*  OK, or
*  ERROR if no such task, or breakpoint table is full.
*
* SEE ALSO: so (2), "Debugging"
*/

STATUS cret (superName)
    int superName;		/* task to continue; 0 = default */

    {
    INSTR *retaddr;		/* return address of current subroutine */
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    int task = taskIdFigure (superName);	/* convert Sname to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    task = taskIdDefault (task);		/* set default task id */

    if (dbgCheckBreakable (task) == ERROR)
	return (ERROR);

    if (taskRegsGet (task, dregs, aregs, &sp, &sr, &pc) != OK)
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    /* if next instruction is a LINK or RTS, return address is on top of stack;
     * otherwise it follows saved frame pointer */

    if (DSM(pc,LINK,LINK_MASK) || DSM(pc,RTS,RTS_MASK))
	retaddr = *(INSTR **)sp;
    else
	retaddr = *((INSTR **)aregs[6] + 1);

    /* add breakpoint to breakpoint table */

    if (dbgBrkAdd (retaddr, task, 0, BRK_SO) != OK)
	{
	printf ("Breakpoint table is full.\n");
	return (ERROR);
	}

    /* resume the task */

    resumeTask = task;			/* mark task as resuming */

    if (taskResume (task) != OK)	
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* s - single step
*
* This routine single-steps a task that is stopped at a breakpoint.
*
*	s [task [,addr]]
*
* If task is missing or zero, the last task referenced is assumed.
* If addr is non-zero then the program counter is changed to addr
* and the task is stepped.
*
* SEE ALSO: "Debugging"
*/

STATUS s (superName, addr)
    int superName;	/* task to step; 0 = use default */
    int addr;		/* address to step to; 0 = next instruction */

    {
    int task = taskIdFigure (superName); /* convert super name to task id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    task = taskIdDefault (task);		/* set default task id */

    if (dbgCheckBreakable (task) == ERROR)
	return (ERROR);

    sstepTask = task;			/* mark task as single-stepping */
    sstepQuiet = FALSE;			/* output debugging info to console */

    /* adjust pc & resume the task */

    if (addr != 0)
	dbgTaskPcSet (task, (INSTR *)addr);

    if (taskResume (task) != OK)
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* so - single step, but step over subroutine
*
* This routine single-steps a task that is stopped at a breakpoint. But, if
* the next instruction is a JSR or BSR, it breaks at the instruction following
* the subroutine call instead.
*
*	so [task]
*
* If task is missing or zero, the last task referenced is assumed.
*
* SEE ALSO: "Debugging"
*/

STATUS so (superName)
    int superName;	/* task to step; 0 = use default */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    int task = taskIdFigure (superName);	/* convert super name to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    task = taskIdDefault (task);		/* set default task id */

    if (dbgCheckBreakable (task) == ERROR)
	return (ERROR);

    if (taskRegsGet (task, dregs, aregs, &sp, &sr, &pc) != OK)
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}


    /* check to see if next instruction is a JSR or BSR */

    if (DSM (pc,JSR,JSR_MASK) || DSM (pc,BSR,BSR_MASK))
	{
	/* add breakpoint to breakpoint table */

	if (dbgBrkAdd (pc + dsmNbytes (pc) / 2, task, 0, BRK_SO) != OK)
	    {
	    printf ("Breakpoint table is full.\n");
	    return (ERROR);
	    }

	resumeTask = task;		/* enable resuming from breakpoint */
	}
    else				/* ordinary single-step */
	{
	sstepTask = task;		/* mark task as single-stepping */
	sstepQuiet = FALSE;		/* turn on debugging info output */
	}

    if (taskResume (task) != OK)		/* resume the task */
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    return (OK);
    }

/*******************************************************************************
*
* dbgBrkAdd - add a breakpoint to the breakpoint table
*
* RETURNS: OK, or ERROR breakpoint table is full
*/

LOCAL STATUS dbgBrkAdd (addr, task, count, type)
    INSTR *addr;	/* address where to set breakpoint */
    int task;		/* task for which breakpoint is to be set */
    int count;		/* initial pass count */
    int type;		/* breakpoint type */

    {
    BOOL found = FALSE;
    FAST BRKENTRY *bp;
    
    bp = (BRKENTRY *) lstFirst (&dbgBrkList);
    while (bp != NULL)
	{
	/* check for pre-existing breakpoints at given address */

	if ((bp->addr == addr) && (bp->type == type))
	    {
	    if (task == ALL)		/* breakpointing all tasks */
		{
		if (!found)
		    {
		    bp->task  = ALL;	/* breakpoint applies to all tasks */
		    bp->count = count;	/* update pass count */
		    found = TRUE;	/* delete any duplicates */
		    bp = (BRKENTRY *) lstNext (&bp->node);
		    }
		else
		    bp = dbgBrkDelete (bp);	/* delete duplicate */
		}
	    else
		{
		/* see if we can use pre-existing breakpoint */

		if ((bp->task == ALL) || (bp->task == task))
		    {
		    bp->count = count;	/* use new pass count */
		    found = TRUE;
		    break;
		    }
		else
		    bp = (BRKENTRY *) lstNext (&bp->node);
		}
	    }
	else
	    bp = (BRKENTRY *) lstNext (&bp->node);
	}

    if (found)
	return (OK);			/* breakpoint already set */


    /* get a free breakpoint entry */

    taskLock ();			/* disable task switching */
    bp = (BRKENTRY *) lstGet (&dbgFreeList);
    taskUnlock ();			/* re-enable task switching */

    if (bp == NULL)
	{
	bp = (BRKENTRY *) malloc (sizeof (BRKENTRY));
	if (bp == NULL)
	    return (ERROR);
	}


    /* initialize breakpoint and add to list */

    bp->addr  = addr;			/* breakpoint address */
    bp->code  = *addr;			/* save existing code */
    bp->task  = task;			/* breakpoint's applicable task */
    bp->count = count;			/* pass count */
    bp->type  = type;			/* normal type breakpoint */

    taskLock ();			/* disable task switching */
    lstAdd (&dbgBrkList, &bp->node);	/* add bp to active list */
    taskUnlock ();			/* re-enable task switching */

    return (OK);
    }
/*******************************************************************************
*
* dbgBrkDelete - delete a breakpoint from the active list
*
* RETURNS: ptr to next breakpoint node, or NULL if at end of list
*/

LOCAL BRKENTRY *dbgBrkDelete (bp)
    BRKENTRY *bp;	/* breakpoint to delete */

    {
    BRKENTRY *nextBp = (BRKENTRY *) LSTNEXT (bp->node);

    taskLock ();			/* disable task switching */
    lstDelete (&dbgBrkList, &bp->node);	/* add bp to active list */
    lstAdd (&dbgFreeList, &bp->node);	/* add to free list */
    taskUnlock ();			/* re-enable task switching */

    return (nextBp);
    }
/*******************************************************************************
*
* dbgBrkGet - get pointer to breakpoint table entry
*
* NOTE:
* dbgBrkGet calls no other routines, so we can be reasonably sure we won't hit
* a breakpoint.
*
* RETURNS:
*    pointer to breakpoint table entry corresponding
*        to given address and task, or
*    NULL if none found.
*/

LOCAL BRKENTRY *dbgBrkGet (addr, task)
    FAST INSTR *addr;	/* breakpoint address */
    FAST int task;	/* id of task for which to find breakpoint; 0 = any */

    {
    FAST BRKENTRY *bp;			/* index into breakpoint table */
    FAST BRKENTRY *bptask = NULL;	/* breakpoint whose task is task */
    FAST BRKENTRY *bpall = NULL;	/* breakpoint whose task is ALL */

    for (bp = (BRKENTRY *) LSTFIRST (dbgBrkList);
	 bp != NULL;
	 bp = (BRKENTRY *) LSTNEXT (bp->node))
	{
	if (bp->addr == addr)
	    {
	    if ((bp->task == task) || (task == 0))
		{
		if (bp->type == BRK_SO)
		    return (bp);	/* 1st choice: step-over */
		else
		    bptask = bp;	/* 2nd choice: this task */
		}
	    else if (bp->task == ALL)
		bpall = bp;		/* 3rd choice: task = ALL */
	    }
	}

    return ((bptask != NULL) ? bptask : bpall);
    }
/*******************************************************************************
*
* dbgBrkDisplay - display breakpoints
*/

LOCAL VOID dbgBrkDisplay ()

    {
    FAST BRKENTRY *bp;
    FAST BOOL nobreaks = TRUE;		/* flag that no breakpoints are set */

    for (bp = (BRKENTRY *) lstFirst (&dbgBrkList);
	 bp != NULL;
	 bp = (BRKENTRY *) lstNext (&bp->node))
	{
	nobreaks = FALSE;

	dbgBrkPrint (bp->addr);

	if (bp->task == ALL)
	    printf ("   Task: all   Count: %2d", bp->count);
	else
	    printf ("   Task: %3d   Count: %2d", bp->task, bp->count);

	if (bp->type == BRK_NORM)
	    printf ("\n");
	else if (bp->type == BRK_NORM_QUIET)
	    printf ("   (quiet)\n");
	else
	    printf ("   (temporary)\n");
	}

    if (nobreaks)
	printf ("No breakpoints.\n");
    }
/*******************************************************************************
*
* dbgBrkPrint - print breakpoint address
*/

LOCAL VOID dbgBrkPrint (addr)
    INSTR *addr;			/* address of breakpoint */

    {
    char label[MAX_SYS_SYM_LEN+1];	/* name from symbol table goes here */
    int actVal;				/* actual value of label */
    UTINY type;
    int displacement;

    printf ("$%06x: ", addr);		/* print breakpoint address */

    /* print symbolic address and displacement, if any */

    if (symValFind (sysSymTbl, (int)addr, label, &actVal, &type) == OK)
	{
	printf ("%-15s", label);
	if ((displacement = (int)addr - actVal) != 0)
	    printf ("+$%-4x", displacement);
	else
	    printf ("%6s", "");		/* no displacement */
	}
    else
	printf ("%21s", "");		/* no symbolic address */
    }
/*******************************************************************************
*
* dbgSoDelete - delete all step-over breakpoints for given task
*/

LOCAL VOID dbgSoDelete (task)
    FAST int task;

    {
    FAST BRKENTRY *bp = (BRKENTRY *) LSTFIRST (dbgBrkList);

    while (bp != NULL)
	{
	if ((bp->task == task) && (bp->type == BRK_SO))
	    {
	    *bp->addr = bp->code;		/* restore code */
	    dbgCacheEntryClear (bp->addr); 	/* address to clear */
	    bp = dbgBrkDelete (bp);	/* remove breakpoint table entry */
	    }
	else
	    bp = (BRKENTRY *) LSTNEXT (bp->node);
	}
    }
/*******************************************************************************
*
* dbgStepQuiet - single stop quietly.
*
* dbgStepQuiet is provided for external source code debuggers which
* need to implement single step, but don't need debugging information
* printed out on the console.
*
* RETURNS: OK | ERROR
*/

STATUS dbgStepQuiet (superName)
    int superName;	/* name of task to step */

    {
    int task = taskIdFigure (superName);	/* convert Sname to id */

    if (task == ERROR)
	return (ERROR);

    task = taskIdDefault (task);	/* set default task id */

    if (dbgCheckBreakable (task) == ERROR)
	return (ERROR);

    sstepTask = task;			/* mark task as single-stepping */
    sstepQuiet = TRUE;			/* single step quietly */

    return (taskResume (task));		/* resume the task */
    }
/*******************************************************************************
*
* dbgCheckBreakable - make sure task is breakable
* 
* Prints an error message if task is unbreakable.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS dbgCheckBreakable (task)
    int task;

    {
    TCBX *pTcbX;

    if (task == ERROR)
	return (ERROR);

    pTcbX = taskTcbX (task);

    if (pTcbX == NULL)
	{
	printf ("Task %d not found.\n", task);
	return (ERROR);
	}

    if ((pTcbX->options & VX_UNBREAKABLE) != 0)
	{
	printf ("Task %d is unbreakable.\n", task == 0 ? taskIdSelf () : task);
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* dbgTaskPcSet - set task's pc
*/

LOCAL STATUS dbgTaskPcSet (task, pc)
    int task;		/* task id */
    INSTR *pc;		/* new PC */

    {
    USHORT sr;		/* status reg */		
    char *sp;		/* task's stack pointer */
    INSTR *oldPc;	/* old PC */
    int regBuf[16];	/* buffer to contain registers */

    if (taskRegsGet (task, regBuf, &regBuf[8], &sp, &sr, &oldPc) != OK)
	return (ERROR);

    taskRegsSet (task, regBuf, &regBuf[8], sp, sr, pc);
    return (OK);
    }
/*******************************************************************************
*
* dbgTaskPcGet - get task's PC
*/

LOCAL INSTR *dbgTaskPcGet (task)
    int task;

    {
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* status reg */		
    INSTR *pc;			/* task's pc */
    int regBuf[16];		/* buffer for registers */

    taskRegsGet (task, regBuf, &regBuf[8], &sp, &sr, &pc);	/* get pc */
    return (pc);
    }

/* interrupt level routines and task level helpers */

/*******************************************************************************
*
* dbgBreakpoint - handle hitting of breakpoint
*
* This routine handles the breakpoint trap.  It is called only
* from its special assembly language stub routine which is connected
* to the breakpoint trap.
*
* It prints out a bunch of pertinent information about the trap that
* occurred via excTask.
*
* NOMANUAL
*/

VOID dbgBreakpoint (info, regs)
    BREAK_ESF *info;		/* pointer to info saved on stack */
    int *regs;		        /* pointer to saved registers */

    {
    int task;			/* id of broken task goes here */
    FAST BRKENTRY *bp;		/* breakpoint table entry */

    /* break happened at interrupt level?
     * NOTE: as this is a trap, intCount is at least 1.
     */

    if (intCount () > 1)
	{
	/* ignore breakpoints hit in these situations by temporarily restoring
	 * the original instruction, executing it in trace mode, and having
	 * dbgTrace replace the breakpoint trap */

	bp = dbgBrkGet (info->pc, 0);			

       /* we must lock interrupts in order to ensure our global variables
        * interruptBreak and interruptSR don't get clobbered by another ISR 
        */
        (void) intLock ();              /* LOCK INTERRUPTS */
	interruptBreak = bp;		/* save pointer to breakpoint */
	interruptSR  = info->sr; 	/* save old SR; dbgTrace will restore */

	dbgCacheEntryClear (bp->addr);  /* address to clear */
	*bp->addr = bp->code;		/* restore original code */

	info->sr |= TRACE_BIT | 0x700;	/* set trace bit and lock interrupts */

	return;		/* first instruction will be traced by dbgTrace */
	}

    taskOptionsSet (0, VX_UNBREAKABLE, VX_UNBREAKABLE);	/* set unbreakable */

    task = taskIdSelf ();				/* who did this? */
    bp = dbgBrkGet (info->pc, task);			/* what breakpoint? */

    if ((bp != NULL) && (bp->type == BRK_SO))	/* step-over breakpoint? */
	{
	(void)dbgBrkDelete (bp);		/* delete the breakpoint */
	excToDoAdd (dbgTlSnglStep, task, info, regs, FALSE);	/* stop task */
	}

    else				/* regular breakpoint */
	{
	excToDoAdd (dbgTlBreak, info->pc, task, info, regs);
	}

    taskSuspend (0);	/* suspend ourself just in case */
    }
/*******************************************************************************
*
* dbgTrace - handle trace exception
*
* This routine handles the trace trap.  It is called only
* from its special assembly language stub routine which is connected
* to the trace trap.
*
* INTERNAL
* The trace bit should have been set by dbgTaskSwitch for a task proceeding from
* a breakpoint. Replace the BREAK in the instruction just executed.
*
* NOMANUAL
*/

VOID dbgTrace (info, regs)
    TRACE_ESF *info;		/* pointer to info saved on stack */
    int *regs;		        /* pointer to saved registers */

    {
    int oldTraceCause;

    if (interruptBreak != NULL)	/* breakpoint hit at interupt level */
	{
	/* INTERRUPTS ARE STILL LOCKED OUT */
	dbgCacheEntryClear (interruptBreak->addr);  /* address to clear */
	*interruptBreak->addr = breakInst;	/* replace BREAK */
	interruptBreak = NULL;			/* clear flag */
	info->sr = interruptSR;		/* restore interrupt level before brk */
	return;				/* slip off into the night */
	}

    oldTraceCause = traceCause;
    traceCause = NONE;

    resumeTask = NONE;
    sstepTask = NONE;

    switch (oldTraceCause)
	{
	case CONTINUE:
	case STEP_FROM_BREAK:
	    if ((resumeBreak != NULL) && (resumeBreak->addr != NULL))
		*resumeBreak->addr = breakInst;		/* replace BREAK */
	    resumeBreak = NULL;
	    if (oldTraceCause == CONTINUE)
		break;			/* continue from breakpoint */

	    /* else fall thru to single-step */

	case SINGLE_STEP:
	    taskOptionsSet (0, VX_UNBREAKABLE, VX_UNBREAKABLE);	/* unbreakable*/
	    excToDoAdd (dbgTlSnglStep, taskIdSelf (), info, regs, TRUE);	
	    taskSuspend (0);	/* suspend ourself just in case */
	    break;

	default:
	    logMsg ("dbgTrace: invalid case %d\n", oldTraceCause);
	    break;
	}
    }
/*******************************************************************************
*
* dbgTlBreak - suspend task and print breakpoint message
*/

LOCAL VOID dbgTlBreak (pc, task, info, regs)
    FAST INSTR *pc;		/* task's pc */
    FAST int task;		/* id of task that hit breakpoint */
    BREAK_ESF *info;		/* pointer to esf info saved on stack */
    int *regs;			/* pointer to buf containing saved regs */

    {
    char *sp;			/* task's stack pointer */
    FAST BRKENTRY *bp;		/* breakpoint table entry */

    taskSuspend (task);				/* stop the task immediately */
    taskIdDefault (task);			/* update default task */
    taskOptionsSet (task, VX_UNBREAKABLE, 0);	/* set task breakable again */

    sp = (char *) info + sizeof (BREAK_ESF);

    /* restore the task's registers */

    taskRegsSet (task, regs, &regs[8], sp, info->sr, pc);

    /* print breakpoint info */

    if ((bp = dbgBrkGet (pc, task)) == NULL)
	printErr ("\nUnknown breakpoint at 0x%x\n", pc);
    else if (bp->count != 0)		/* pass count not yet zero? */
	{
	--bp->count;		/* decrement pass count */
	c (task, 0);		/* continue */
	return;
	}
    else if (bp->type != BRK_NORM_QUIET)
	{
	printErr ("\nBreak at ");
	dbgBrkPrint (pc);		/* print the address */
	printErr ("   Task: %x\n", task);
	nextToDsm = pc;		/* update l's default list-from */
	}

    dbgSoDelete (task);		/* delete this task's outstanding step-overs */


    /* If there's a hook routine, call it now. */

    if (dbgBreakNotifyRtn != NULL)
#ifdef	is68k
	(*dbgBreakNotifyRtn) (task, pc);
#else	is68k
	(*dbgBreakNotifyRtn) (task);
#endif	is68k
    }
/*******************************************************************************
*
* dbgTlSnglStep - suspend task and print info about a single-stepping task
*/

LOCAL VOID dbgTlSnglStep (task, info, regs, stepBreakFlag)
    int task;
    TRACE_ESF *info;		/* pointer to esf info saved on stack */
    int *regs;			/* pointer to regs saved on stack */
    BOOL stepBreakFlag;		/* TRUE if this was a trace exception
				   FALSE if this was a SO or CRET breakpoint */

    {
    char *sp;			/* task's stack pointer */
    INSTR *pc;			/* task's pc */

    taskSuspend (task);				/* stop task immediately */
    taskIdDefault (task);			/* update default task */
    taskOptionsSet (task, VX_UNBREAKABLE, 0);	/* set task breakable again */

    if (stepBreakFlag)
	sp = (char *) info + sizeof (TRACE_ESF);
    else
	sp = (char *) info + sizeof (BREAK_ESF);

    /* restore the task's registers */

    taskRegsSet (task, regs, &regs[8], sp, info->sr, info->pc);

#ifdef	is68k
    if (!cdbQuietFlag && !sstepQuiet)
#else	is68k
    if (!sstepQuiet)
#endif	is68k
	{
	/* print the registers and disassemble next instruction */

	taskRegsShow (task);			

	pc = dbgTaskPcGet (task);
	(void) dbgList (pc, 1);	
	nextToDsm = (INSTR *) pc;
	}

    /* If there's a hook routine, call it now. */

    if (dbgBreakNotifyRtn != NULL)
	(*dbgBreakNotifyRtn) (task);
    }
/*******************************************************************************
*
* dbgTaskSwitch - system task switch routine
*
* This is the debug task switch routine.  It is installed by calling
* taskSwitchHookAdd.  It will then be called every time tasks are switched.
*
* ARGSUSED
*/

LOCAL VOID dbgTaskSwitch (oldTcbX, newTcbX)
    TCBX *oldTcbX;	/* pointer to tcbX of switch-from task */
    TCBX *newTcbX;	/* pointer to tcbX of switch-to task */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    int newId   = newTcbX->taskId;
    int options = newTcbX->options;
    FAST BRKENTRY *bp = (BRKENTRY *) LSTFIRST (dbgBrkList);

    /* if any breakpoints active; remove old task's and insert new task's */

    if (bp != NULL)
	{
	/* remove all breakpoints */

	do
	    {
	    *bp->addr = bp->code;			/* remove break */
	    dbgCacheEntryClear (bp->addr);  		/* address to clear */
	    }
	while ((bp = (BRKENTRY *) LSTNEXT (bp->node)) != NULL);

	/* if new task is not unbreakable, search breakpoint table;
	 * insert BREAKS if they apply to the new task, or to all tasks */

	if ((options & VX_UNBREAKABLE) == 0)
	    {
	    for (bp = (BRKENTRY *) LSTFIRST (dbgBrkList);
		 bp != NULL;
		 bp = (BRKENTRY *) LSTNEXT (bp->node))
		{
		if ((bp->task == newId) || (bp->task == ALL))
		    {
		    *bp->addr = breakInst;		/* insert break */
		    dbgCacheEntryClear (bp->addr);  /* address to clear */
		    }
		}
	    }
	}


    /* check for resumption of execution after breakpoint */

    if (newId == resumeTask)
	{
	if ((options & VX_UNBREAKABLE) == 0)
	    {
	    taskRegsGet (newId, dregs, aregs, &sp, &sr, &pc);
	    bp = dbgBrkGet (pc, newId);
	    if (bp != NULL)
		{
		traceCause = CONTINUE;
		resumeBreak = bp;		/* save pointer to breakpoint */
		*bp->addr = bp->code;		/* restore original code */
		dbgCacheEntryClear (bp->addr);  /* address to clear */
		taskSRSet (newId, sr | TRACE_BIT);	/* set trace bit */
		}
	    else
		{
		/* clear resumeTask so that next time s() is
		 * called we won't pass this way */

		resumeTask = NONE;
		}
	    }
	else
	    {
	    /* insurance in case task suddenly becomes unbreakable */

	    resumeTask = NONE;
	    }
	}

    /* check for single-stepping task */

    else if (newId == sstepTask)
	{
	if ((options & VX_UNBREAKABLE) == 0)
	    {
	    taskRegsGet (newId, dregs, aregs, &sp, &sr, &pc);
	    bp = dbgBrkGet (pc, newId);
	    if (bp != NULL)			/* stepping from breakpoint? */
		{
		traceCause = STEP_FROM_BREAK;
		resumeBreak = bp;		/* save pointer to breakpoint */
		*bp->addr = bp->code;		/* restore original code */
		dbgCacheEntryClear (bp->addr);  /* address to clear */
		}
	    else
		traceCause = SINGLE_STEP;

	    taskSRSet (newId, sr | TRACE_BIT);	/* set trace bit */
	    }
	else
	    {
	    /* insurance, in case task suddenly becomes unbreakable */

	    sstepTask = NONE;
	    }
	}
    }
/*******************************************************************************
*
* dbgBreakNotifyInstall - install breakpoint notification hook
*
* This routine installs a routine to be call when a breakpoint it
* is hit (in dbgTlBreak & dbgTlSnglStep).
* It is used to support remote debuggers that need to be informed when
* breakpoints are hit.
*
* NOMANUAL
*/

VOID dbgBreakNotifyInstall (breakpointRtn)
    FUNCPTR breakpointRtn;

    {
    dbgBreakNotifyRtn = breakpointRtn;
    }
/*******************************************************************************
*
* dbgTyAbort - abort the shell from interrupt level
*/

LOCAL VOID dbgTyAbort ()

    {
    excToDoAdd (dbgTlTyAbort);
    }
/*******************************************************************************
*
* dbgTlTyAbort - task level portion of shell abort
*/

LOCAL VOID dbgTlTyAbort ()

    {
    tyAbortFunc ((FUNCPTR) NULL);	/* cancel abort routine */

    /* reset the standard input and output channels in case they are hung up */

    ioctl (STD_IN,  FIOFLUSH);
    ioctl (STD_OUT, FIOFLUSH);
    ioctl (STD_ERR, FIOFLUSH);

    tt (shellTaskId);

    /* XXX shellTaskId = taskRestart (shellTaskId); XXX */
    if ((taskRestart (shellTaskId)) != ERROR)
	printErr ("shell restarted.\n");
    else
	{
	printErr ("spawning new shell.\n");

	if (shellInit (0, TRUE) == ERROR)
	    printErr ("shell spawn failed!\n");
	}

    tyAbortFunc (dbgTyAbort);		/* set abort routine */
    }


/*******************************************************************************
*
* l - print a number of disassembled instructions
*
* This routine is intended to be called from the UniWorks shell.
* It dissasembles some number of instructions and prints them on standard out.
* If the address of an instruction is entered in the system symbol table,
* the symbol will be printed as a label for that instruction.
* Also, addresses in the op-code field of instructions will be printed
* symbolically.
*
* The format of the shell call is
*
*    l [address [,count]]
*
* If address is not given, disassembly continues from the previous address.
* If count is not given, the last specified count is used (initially 10).
* As with all values entered via the shell, the address may be typed 
* symbolically.
*
* SEE ALSO: "Debugging", dsmLib(1)
*/

VOID l (binInst, count)
    INSTR *binInst;		/* Address of first instruction to
				   disassemble. If 0, continue from
				   the last instruction disassembled
				   on the last call to l */
    int count;			/* Number of instruction to disassemble.
				   If 0, use the same as the last
				   call to l. */

    {
#if (CPU!=MC68020)
    binInst = (INSTR *) ((int)binInst & ~(0x01));	/* force binInst even */
#endif

    /* set new start address and instruction count if specified */

    if ((int) binInst != 0)
	nextToDsm = binInst;

    if (count != 0)
	dfltCount = count;

    nextToDsm = dbgList (nextToDsm, dfltCount);	/* disassemble n instructions */
    }
/*******************************************************************************
*
* dbgList - disassemble and list n instructions
*
* This routine disassembles some number of instructions, using dsmLib.
* The normal user interface is through the routine 'l', rather than through
* this routine.
* 
* RETURNS: Pointer to instruction following last instruction disassembled.
*/

LOCAL INSTR *dbgList (addr, count)
    FAST INSTR *addr;
    FAST int count;

    {
    char label [MAX_SYS_SYM_LEN + 1];
    int actVal;
    UTINY type;
    FAST int i;

    for (i = 0; i < count; i++)
	{
	if ((symValFind (sysSymTbl, (int)addr, label, &actVal, &type) == OK)
	     && (actVal == (int)addr))
	    {
	    printf ("                        %s:\n", label);
	    }

	addr += dsmInst (addr, (int)addr, dbgPrintAdrs);
	}

    return (addr);
    }
/*******************************************************************************
*
* dbgPrintAdrs - A smarter address printer
*
* This routine prints addresses as symbols, if an exact match can be found
*/

LOCAL VOID dbgPrintAdrs (address)
    int address;

    {
    char label [MAX_SYS_SYM_LEN + 1];
    int actVal;
    UTINY type;

    if ((symValFind (sysSymTbl, address, label, &actVal, &type) == OK) &&
	(actVal == address))
	{
	printf ("%s", label);
	}
    else
	printf ("$%06x", address);
    }

/*******************************************************************************
*
* tt - print a stack trace of a task
*
* This routine prints a list of the nested routine calls that the specified
* task is in.
* Each routine call, with its parameters, is shown.
* This higher-level symbolic stack trace is built on top of the 
* low-level routines provided by the routines in trcLib (1).
*
* If task is missing or zero, the last task referenced is assumed.
*
* Tt can only trace the stack of some task other than itself.  For instance,
* when tt is called from the shell, it cannot trace the shell's stack.
*
* EXAMPLE
*    -> tt 30
*    routine1 (p1, p2, p3)
*    routine2 (q1, q2, q3, q4)
*    routine3 ()
*
* This would tell you that task 30 is currently in routine3, which had
* been called by routine2 (with 4 parameters), which had been called by
* routine1.
* 
* CAVEAT
* In order to do the trace, some assumptions are made.
* In general, the trace will work for all C language routines,
* and for assembly language routines that start with a LINK instruction.
* Some C compilers require specific flags to generate the LINK first.
* Most UniWorks assembly language routines include LINK instructions
* for exactly this reason.
* However, routines written in other languages, strange entries into
* routines, or tasks with corrupted stacks, can make the trace very confused.
* Also, all parameters are assumed to be 32-bit quantities, so structures
* passed as parameters will be displayed as some number of long integers.
*
* RETURNS:
*   OK, or
*   ERROR if task does not exist.
*
* SEE ALSO: "Debugging", trcLib(1), UNIX cc manual entry
*/

STATUS tt (superName)
    int superName;		/* task whose stack is to be traced */

    {
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    unsigned short sr;		/* task's status register */
    INSTR *pc;			/* task's pc */
    BOOL resumeIt = FALSE;	/* flag to remember if resuming is necessary */
    int task = taskIdFigure (superName);	/* convert super name to id */

    if (task == ERROR)
	{
	printf ("Task not found.\n");
	return (ERROR);
	}

    task = taskIdDefault (task);		/* set default task id */

    /* get caller task's id and make sure it is not the task to be traced */

    if ((task == taskIdSelf ()) || (task == 0))
	{
	trcStack ((int *) &superName - 2, (int *) &task, (INSTR *) tt + 40,
		  dbgPrintCall);
        printErr ("Sorry, traces of my own stack begin at tt ().\n");
	return (ERROR);
	}

    /* make sure the task exists */

    if (taskIdVerify (task) != OK)
	{
	printErr ("Can't trace task %d: invalid task id.\n", task);
	return (ERROR);
	}

    /* if the task is not already suspended, suspend it while we trace it */

    if (!taskIsSuspended (task))
	{
	resumeIt = TRUE;		/* we want to resume it later */
	taskSuspend (task);		/* suspend the task if need be */
	}

    /* trace the stack */

    taskRegsGet (task, dregs, aregs, &sp, &sr, &pc);
    trcStack ((int *) aregs[6], (int *) sp, pc, dbgPrintCall);

    if (resumeIt)
	taskResume (task);		/* resume task if we suspended it */

    return (OK);
    }
/*******************************************************************************
*
* dbgPrintCall - print a stack frame
*
* This routine is called by trcStack to print each level in turn.
*
* NOMANUAL	- also used by shellSigHandler
*/

VOID dbgPrintCall (callAdrs, funcAdrs, nargs, args)
    INSTR *callAdrs;		/* address from which function was called */
    int funcAdrs;		/* address of function called */
    FAST int nargs;		/* number of arguments in function call */
    int *args;			/* pointer to function args */

    {
    FAST int i;
    char name [MAX_SYS_SYM_LEN + 1];	/* name of function goes here */
    int val;				/* address of named fn goes here */
    UTINY type;				/* symbol type of name goes here */

    /* print call address and name of calling function plus offset */

    printErr ("%6x ", callAdrs);	/* print address from which called */

    if (symValFind (sysSymTbl, (int)callAdrs, name, &val, &type) == OK)
	printErr ("%-15s+%-3x: ", name, (int)callAdrs - val);
    else
	printErr ("                     : ");


    /* print function address/name */

    if ((symValFind (sysSymTbl, funcAdrs, name, &val, &type) == OK) && 
	(val == funcAdrs))
	printErr ("%s (", name);		/* print function name */
    else
	printErr ("%x (", funcAdrs);		/* print function address */


    /* print args */

    for (i = 0; i < nargs; i++)
	{
	if (i > 0)
	    printErr (", ");

	if ((symValFind (sysSymTbl, args [i], name, &val, &type) == OK) && 
	    (val == args [i]))
	    {
	    printErr ("&%s", name);		/* print argument name */
	    }
	else
	    printErr ("%x", args [i]);
	}

    printErr (")\n");
    }
/*******************************************************************************
*
* dbgBreakRemove - remove installed breakpoints
*
* This routine installs or removes all breakpoints for the current task.
* This routine is only called by taskOptionsSet when a task's unbreakable
* option is being changed. The call is made indirectly via a hook provided
* by taskLib, which is installed by dbgInit.
* This indirect linkage allows dbgLib to be omitted if desired.
*
* Note that this routine calls no other routines, so we can be reasonably
* sure that we won't hit any breakpoints.
*/

LOCAL VOID dbgBreakHook (breakable)
    BOOL breakable;

    {
    FAST BRKENTRY *bp;		/* breakpoint table entry */

    for (bp = (BRKENTRY *) LSTFIRST (dbgBrkList);
	 bp != NULL;
	 bp = (BRKENTRY *) LSTNEXT (bp->node))
	{
	if (breakable)
	    {
	    if ((bp->task == taskIdCurrent) || (bp->task == ALL))
		{
		*bp->addr = breakInst;		/* insert break */
		dbgCacheEntryClear (bp->addr);	/* address to clear */
		}
	    }
	else
	    {
	    *bp->addr = bp->code;	 	/* restore code */
	    dbgCacheEntryClear (bp->addr);	/* address to clear */
	    }
	}
    }
