/* cdbLib.c - target support library for cdb */

static char *copyright = "Copyright 1987, Wind River Systems, Inc.";

/*
modification history 
--------------------
*/

/*
DESCRIPTION
This library provides the target CPU support for cdb, Third Eye Software's
source language debugger, which runs on the host.  The two communicate
using sockets via the network.  Cdb (on the host) opens a socket to cdbDaemon
(on the target) when it is started.

Nothing here is important to the user except cdbInit, which must be called
exactly once if cdb support is required. 

There are two tasks relating to cdb.
CdbDaemon is spawned by cdbInit and runs forever.  CdbServer is spawned
by cdbDaemon whenever there is actually a connection with cdb running
on the host.  All this is automatic; the user doesn't need to spawn
anything directly.
*/

/* 
 * (C) Copyright 1987 by Integrated Solutions, Inc. 
 * (C) Copyright 1987 by Wind River Systems, Inc.
 * (C) Copyright 1984 by Third Eye Software, Inc.
 * (talk (and copyrights) is (are) cheap)
 */

/*#include <signal.h>	/**/
#include "socket.h"
#include "errno.h"
#include "in.h"
#include "lstLib.h"
#include "cdbLib.h"
#include "UniWorks.h"
#include "vrtx32.h"
#include "memLib.h"
#include "ioLib.h"
#include "xptrace.h"

#define chNull '\0'
typedef char *SBT;
#define sbNil ((SBT)0)
typedef unsigned long ADRT, *pADRT;
#define true 1
#define false 0
#define AND &&
#define OR ||

#define DEBUG	/**/

#ifdef DEBUG
#define dprint(x, y) {if (x <= vlevel) logMsg y;}
#else
#define dprint(x, y)
#endif DEBUG

#define	DLEVEL_TRACE		10
#define DLEVEL_TRACEMORE	20
#define DLEVEL_VERBOSE		30
#define DLEVEL_LOUD		40

#define	MAX_MSGS  	10	/* maximum number of msgs that can be in pipe */
#define CDB_PRIORITY	100	/* cdbServer's priority */
#define	MAX_CDB_TASKS	25

/**** move this constant to socket.h or something like that */
#define CDB_SOCKET	1101	/* cdb socket port number */

/**** do we need this as well as TASK_STACK_SIZE ???? */
#define CDB_STACK_SIZE	5000	/* cdbServer's task stack size */

#define BUFSIZE		512	/* size of read buffer */

int vlevel = 0;			/* debug print level for dprint */

BOOL vfWaitFlag = FALSE;	/* whether we're waited or not */

#define cbBufMax 1024	/* pick a number suited to the size of your system */

BOOL cdbFlag;			/* TRUE when cdb is connected */
BOOL cdbQuietFlag;		/* TRUE when cdb is connected && want quiet */
IMPORT FUNCPTR dbgBreakNotifyRtn; /* to connect cdbBreakpoint to dbgLib */

LOCAL char	vsbBuf[cbBufMax];	/* buffer where we format messages */
LOCAL SBT vsbTx = vsbBuf;

LOCAL int cdbSocket;		/* socket connected to cdb */

LOCAL LIST cdbWaitList;		/* list of tasks being waited on by cdb */
LOCAL int cdbServerId;		/* cdbServer's task id */
LOCAL int cdbDaemonId;		/* cdbDaemon's task id */

LOCAL int commandPipe;		/* pipe for sending commands to cdbServer */

/* forward declarations */

char *readLine ();
char *appendBuf ();
VOID cdbMsgTaskDeleted ();
VOID cdbMsgTaskCreated ();

/***********************************************************************
*
* TxString - send a string to cdb
*
*/

LOCAL int TxString (sb)
    SBT	sb;

    {
    dprint (DLEVEL_LOUD, ("TxRemote: %s", sb));
    return (send (cdbSocket, sb, strlen (sb), 0));
    }

/***********************************************************************
*
* RemotePanic
*
* CDB   PANIC 
* VARARGS1
*/

LOCAL int RemotePanic (sbFmt, arg1, arg2, arg3)
    SBT	sbFmt;
    int	arg1, arg2, arg3;

    {
    logMsg ("Remote Panic - ");
    logMsg (sbFmt, arg1, arg2, arg3);
    }

/***********************************************************************
*
* ValFPsb
*
* This routine eats a hex number from a string and advances
* the string pointer to the next character AFTER the first
* non-hex character.  E.g if we start with "1234|abcd",
* we will return the integer value 0x1234 and leave the string
* pointing to the 'a'.
*/

LOCAL long ValFPsb (psb)
    SBT	*psb;

    {
    int		x = 0;
    char	ch; 
    char 	*sb = *psb;

#define FHex(ch) (((ch >='0') AND (ch <= '9')) OR ((ch >= 'a') AND (ch <= 'f')))

    if ((sb == sbNil) OR (*sb == chNull))
	return(0);

    while (true)
	{
	ch = *sb++;
	if (! FHex(ch))
		break;
	x = (x * 16) + ((ch <= '9') ? (ch - '0') : ((ch - 'a') + 10));
	}
    *psb = sb;
    return (x);
    }

/***********************************************************************
*
* DecValFPsb
*
*/

LOCAL long DecValFPsb (psb)
    SBT	*psb;

    {
    int		x = 0;
    char	ch; 
    char 	*sb = *psb;

#define FDecimal(ch) ((ch >='0') AND (ch <= '9'))

    if ((sb == sbNil) OR (*sb == chNull))
	return(0);

    while (true)
	{
	ch = *sb++;
	if (! FDecimal(ch))
		break;
	x = (x * 10) + ch - '0';
	}
    *psb = sb;
    return (x);
    }

/***********************************************************************
*
* ParamFPsb
*
* This routine eats a token from a string (tokens are delimited
* by '|'), converting ascii representations of numbers to their
* binary equivalents, passing other params as is, and advancing
* the pointer to the next token
* N.B. This really needs the Shell's expression evaluator, but the Shell
* is not reentrant, and I don't want to copy all that code here ...
* When in doubt, wait for Wind River to make the Shell reentrant
*
*/

#define	NOVAL 0
#define	RVAL 1
#define	LVAL 2

LOCAL int ParamFPsb (psb, pAddr)
    SBT	*psb;
    pADRT pAddr;

    {
    SBT 	sb = *psb;
    SBT		pDelim, pArg;
    int		len;


    if ((sb == sbNil) OR (*sb == chNull))
	{
	*pAddr = (ADRT) 0;
	return NOVAL;
	}

    if (strncmp(sb, "0x", 2) == 0)
	{
	*psb +=2;
	*pAddr = (ADRT) ValFPsb(psb);
	return RVAL;
	}
    else if (FDecimal(*sb))
	{
	*pAddr = (ADRT) DecValFPsb(psb);
	return RVAL;
	}
    else
	{
	if (((pDelim = (SBT)index (sb, (char) chDelim)) == sbNil) &&
	     (pDelim = (SBT)index (sb, (char) chNull)) == sbNil)
	    return NOVAL;
	len = pDelim - sb;
	*psb = pDelim + 1;
	if ((pArg = (SBT) malloc(len + 1)) == sbNil)   /* KLUDGE: never freed */
	    return NOVAL;
	strncpy(pArg, sb, len);
	*(pArg + len) = chNull;
	*pAddr = (ADRT) pArg;
	return LVAL;
	}
    }

/***********************************************************************
*
* ReadHexBytes
*
* used with the host-to-remote block transfer command 
*
* ARGSUSED
*/

LOCAL void ReadHexBytes (pid, adr, count, sb)
    int	pid, count;
    long adr;
    SBT	sb;

    {
    char ch1;
    char ch2;
    char *sbDest;
    short val;

    sbDest = (SBT) adr;
    while (count--)
	{
	ch1 = *sb++;
	ch2 = *sb++;
	val = ((ch1 - ((ch1 > '9') ? ('a'-10) : '0')) << 4)
		+ (ch2 - ((ch2 > '9') ? ('a'-10) : '0'));
	*sbDest++ = val;
	} /* while */
    }

/***********************************************************************
*
* WriteHexBytes
*
* used with the remote-to-host block transfer command 
*/

LOCAL void WriteHexBytes (pid, adr, count)
    int	pid; 
    int count;
    long adr;

    {
    SBT	sbTx, sbTxSave; 
    SBT sbSrc;
    short val;

    sbSrc = (char *) adr;

    while (count > 1)
	{
	sbTx = (SBT) malloc (count * 2 + 100);
	if (sbTx > 0)
	    break;
	count = count / 2;
	}

    if (sbTx == (SBT) NULL)
	{
	SendStatus (ENOMEM, pid, 0);
	return;
	}

    sbTxSave = sbTx;

    sprintf(sbTx, "%x|%x|%x|", statusOk, pid, count);
    sbTx = sbTx + strlen(sbTx);
    while (count--)
	{
	val = (*sbSrc & 0xf0) >> 4;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
	val = *sbSrc++ & 0x0f;
	*sbTx++ = val + ((val > 9) ? ('a'-10) : '0');
	}

    *sbTx++ = '\n';
    *sbTx = '\0';
    TxString(sbTxSave);
    free (sbTxSave);
    }

/***********************************************************************
*
* ReadString
*
* used with the host-to-remote block transfer command 
*
* ARGSUSED
*/

LOCAL void ReadString (pid, adr, count, sb)
    int	pid, count;
    long adr;
    SBT	sb;

    {
    char *sbDest;

    sbDest = (SBT) adr;
    while (count--)
	*sbDest++ = *sb++;
    }

/***********************************************************************
*
* WriteString
*
* used with the remote-to-host block transfer command 
*/

LOCAL void WriteString (pid, adr, count)
    int	pid; 
    long adr;
    int count;

    {
    SBT	sbTx, sbTxSave; 
    SBT sbSrc;

    sbSrc = (SBT) adr;

    while (count > 1)
	{
	sbTx = (SBT) malloc (count + 100);
	if (sbTx > 0)
	    break;
	count = count / 2;
	}

    if (sbTx == (SBT) NULL)
	{
	SendStatus (ENOMEM, pid, 0);
	return;
	}

    sbTxSave = sbTx;
    sprintf(sbTx, "%x|%x|%x|", statusOk, pid, count);
    sbTx = sbTx + strlen(sbTx);
    while (count--)
	*sbTx++ = *sbSrc++;

    *sbTx++ = '\n';
    *sbTx = '\0';
    TxString(sbTxSave);
    free (sbTxSave);
    }

/***********************************************************************
*
* LocalPtrace
*/

LOCAL void LocalPtrace (pt, pid, adr, value)
    int	pt; 
    int pid; 
    long adr;
    int value;

    {
    int		   dregs [8];	/* task's data registers */
    int		   aregs [7];	/* task's address registers */
    char	   *sp;		/* task's stack pointer */
    unsigned short sr;		/* task's status register */
    INSTR	   *pc;		/* task's pc */

    /* This routine does whatever is necessary to implement
     * the ptrace(2) system call.
     */

    /* On a majority of systems, Instruction and Data Space
     * are identical.  On systems where this is not true, you
     * will need to deal with them separatly.
     *
     * In UniWorks, Instruction and Data Space are treated the same.
     */

    dprint(DLEVEL_VERBOSE, ("pt: %d  adr: %x  value: %d\n", pt, adr, value));

    switch (pt)

	{
	default:	
	    RemotePanic("Unknown ptrace request: %d", pt);	
	    break;

	case  ptReadI:
	case  ptReadD:
	    /* Read memory */
	    value = *(int *) adr;
	    break;

	case  ptWriteI:
	case  ptWriteD:
	    /* Write memory */
	    *(int *) adr = value;
	    break;

	case  ptReadUser:
	    /* Read Registers */

	    /* get tcb's register values */
	    taskRegsGet (pid, dregs, aregs, &sp, &sr, &pc);

	    switch ((int)adr)
		{
		default:	
		    RemotePanic("Unknown ptReadUser adr: %d", adr);	
		    break;

		case 0: 
		case 1: 
		case 2: 
		case 3:
		case 4: 
		case 5: 
		case 6: 
		case 7:
		    value = dregs [adr];
		    break;

		case 8:  
		case 9:  
		case 10:  
		case 11:
		case 12: 
		case 13: 
		case 14:
		    value = aregs [adr & 7];
		    break;

		case 15:
		    value = (int) sp;
		    break;

		case 16:
		    value = (int) sr;
		    break;

		case 17:
		    value = (int) pc;
		    break;
		} /* switch */
	    break;	/* case ptReadUser */

	case  ptWriteUser:
	    /* Write Registers */

	    /* get tcb's register values */
	    taskRegsGet (pid, dregs, aregs, &sp, &sr, &pc);

	    /* assign new register value */
	    switch ((int) adr)
		{
		default:	
		    RemotePanic ("Unknown ptWriteUser adr: %d", adr);	
		    break;

		case 0: 
		case 1: 
		case 2: 
		case 3:
		case 4: 
		case 5: 
		case 6: 
		case 7:
		    dregs [adr] = value;
		    break;

		case 8:  
		case 9:  
		case 10:  
		case 11:
		case 12: 
		case 13: 
		case 14:
		    aregs [adr & 7] = value;
		    break;

		case 15:
		    sp = (char *) value;
		    break;

		case 16:
		    sr = (unsigned short) value;
		    break;

		case 17:
		    pc = (INSTR *) value;
		    break;
		} /* switch */

	    /* set all registers */
	    taskRegsSet (pid, dregs, aregs, sp, sr, pc);
	    break;	/* case ptWriteUser */

	case  ptSingle:
	    /* Single Step the child */
	    if (s (pid, 0) == ERROR)
		RemotePanic ("ptrace, single step error for task %d", pid);	
	    break;

	case  ptResume:
	    /* Continue an existing child */
	    if (c (pid, 0) == ERROR)
		RemotePanic ("ptrace, continue error for task %d", pid);	
	    break;

	case  ptTerm:
	    /* cdb is going away */
	    /* Terminate the existing child */
	    td (pid);	/* delete task */
	    break;

	} /* switch */

    SendStatus (statusOk, pid, value);
    }

/***********************************************************************
*
* SendStatus
*/

LOCAL SendStatus (status, pid, value)
    int	status; 
    int pid; 
    int value;

    {
    dprint (DLEVEL_LOUD, ("SendStatus(%x|%x|%x)\n", status, pid, value));

    sprintf (vsbTx, "%x|%x|%x\n", status, pid, value);
    TxString (vsbTx);
    }

/***********************************************************************
*
* SendStateChange
*/

LOCAL SendStateChange (status, pid, addr, value)
    int	status; 
    int pid; 
    char *addr;
    int value;

    {
    sprintf (vsbTx, "%x|%x|%x|%x\n", status, pid, addr, value);
    TxString (vsbTx);
    }

extern int spTaskStackSize;

/***********************************************************************
*
* PidFExecChild
*
* Parse the argument list passed into an argv array.
* Create a child process, initialize it to a more or less virgin state,
* (i.e. clean up the signals, close the files,....) and return the pid
* to the caller.
*/

LOCAL int PidFExecChild (sb, defaultArgs)
    char *sb;
    int defaultArgs;
    {
    FUNCPTR routine;
    ADRT params [10]; 
    ADRT retVal;
    int retType;
    int i;
    int pid;
    int options;
    int priority;
    int stackSize;

    dprint(DLEVEL_TRACE,("PidFExecChild(%s,%d)\n",sb,defaultArgs));

    /* extract the routine address from the command line */
    routine = (FUNCPTR) ValFPsb (&sb);	

    /* extract up to 10 params */
    for (i = 0; i < 10; i++)
	params [i] = 0;
    priority = 100;
    options = 0;
    stackSize = spTaskStackSize;
    if (!defaultArgs)
	{
	if ((retType = ParamFPsb (&sb, &priority)) == NOVAL)
	    goto out;
	if ((retType = ParamFPsb (&sb, &pid)) == NOVAL)
	    goto out;
	if ((retType = ParamFPsb (&sb, &options)) == NOVAL)
	    goto out;
	if ((retType = ParamFPsb (&sb, &stackSize)) == NOVAL)
	    goto out;
	}
    for (i = 0; i < 10; i++)
	{
	if (ParamFPsb (&sb, &retVal) == NOVAL)
	    break;
	params [i] = retVal;
	}

    /* spawn the task and then suspend it immediately */
out:
    taskLock ();
    pid = taskSpawn ("child", priority, VX_SUPERVISOR_MODE | options, stackSize,
	    routine, 
	    params [0], params [1], params [2], params [3], params [4],
	    params [5], params [6], params [7], params [8], params [9]);
    ts (pid);
    taskUnlock ();

    dprint(DLEVEL_TRACE,("PidFExecChild, sb=%s,pid=%d\n",sb,pid));

    return (pid);
    }

/***********************************************************************
*
* cdbServer - cdb support server task
*
* This is spawned as a task by cdbDaemon.  It handles the task level of requests
* that come from cdbBreakpoint.
*/

int cdbServer ()

    {
    int	pid; 
    int status; 
    int msg; 
    int pt; 
    int value; 
    int count;
    long adr; 
    long adr2;
    int retVal;
    char *sb; 
    CDB_MSG myMsg;
    TASK_NODE *waitTask;

    /* This routine is the main command loop.
     * It acts as the Parent process to the target process.
     */

    dprint(DLEVEL_TRACE, ("Entering cdbServer\n"));

    lstInit (&cdbWaitList);

    /* don't allow breakpoints */
    taskOptionsSet (taskIdSelf(), VX_UNBREAKABLE, VX_UNBREAKABLE);


    FOREVER
	{
	if (read (commandPipe, (char *) &myMsg, sizeof (myMsg)) == ERROR)
	    {
	    logMsg ("cdbServer: error reading commandPipe: %x\n", 
		    errnoGet ());
	    return (ERROR);
	    } /* if */


	if (myMsg.request == MSG_BREAKPOINT)
	    {

	    dprint(DLEVEL_TRACE, ("cdbServer, got MSG_BREAKPOINT \n"));

	    if (vfWaitFlag)
		{
		dprint (DLEVEL_TRACEMORE, ("sending breakpoint msg...\n"));
		SendStateChange (statusStateChange, myMsg.breakPoint.pid,
					 myMsg.taskCreated.addr, CDB_BREAK);
		vfWaitFlag = FALSE;
		}
	    else
		{
		/* add the breakpoint task to the cdb wait list */
		dprint (DLEVEL_TRACEMORE,  ("adding breakpoint to wait list for task %x\n",
					myMsg.breakPoint.pid));
		waitTask = (TASK_NODE *) malloc (sizeof (TASK_NODE));
		waitTask->taskId = myMsg.breakPoint.pid;
		waitTask->addr = myMsg.breakPoint.addr;
		waitTask->cdbState = CDB_BREAK;
		waitTask->sendInfo = NULL;
		lstAdd (&cdbWaitList, (NODE *) waitTask);
		}
	    }
	else if (myMsg.request == MSG_TASK_DELETED)
	    {

	    dprint(DLEVEL_TRACE, ("cdbServer, got MSG_TASK_DELETED \n"));

	    if (vfWaitFlag)
		{
		SendStateChange (statusStateChange, myMsg.taskDeleted.pid,
					 myMsg.taskDeleted.addr, CDB_DELETED);
		vfWaitFlag = FALSE;
		}
	    else
		{
		/* add the deleted task to the cdb wait list */
		dprint (DLEVEL_TRACEMORE, ("adding taskDelete to wait list for task %x\n",
				myMsg.taskDeleted.pid));
		waitTask = (TASK_NODE *) malloc (sizeof (TASK_NODE));
		waitTask->taskId = myMsg.taskDeleted.pid;
		waitTask->addr = myMsg.taskDeleted.addr;
		waitTask->cdbState = CDB_DELETED;
		waitTask->sendInfo = NULL;
		lstAdd (&cdbWaitList, (NODE *) waitTask);
		}
	    }
	else if (myMsg.request == MSG_TASK_CREATED)
	    {

	    dprint(DLEVEL_TRACE, ("cdbServer, MSG_TASK_CREATED pid=%d, addr=0x%x\n",myMsg.taskCreated.pid,myMsg.taskCreated.addr));

	    if (vfWaitFlag)
		{
		SendStateChange (statusStateChange, myMsg.taskCreated.pid,
					 myMsg.taskCreated.addr, CDB_CREATED);
		vfWaitFlag = FALSE;
		}
	    else
		{
		/* add the deleted task to the cdb wait list */
		dprint (DLEVEL_TRACEMORE, ("adding taskCreate to wait list for task %x\n",
					myMsg.taskCreated.pid));
		waitTask = (TASK_NODE *) malloc (sizeof (TASK_NODE));
		waitTask->taskId = myMsg.taskCreated.pid;
		waitTask->addr = myMsg.taskCreated.addr;
		waitTask->cdbState = CDB_CREATED;
		waitTask->sendInfo = NULL;
		lstAdd (&cdbWaitList, (NODE *) waitTask);
		}
	    }
	else if (myMsg.request == MSG_NEWFILE)
	    {

	    dprint(DLEVEL_TRACE, ("cdbServer, got MSG_NEWFILE \n"));

		/* add the pending new file call to the wait list */
	    dprint (DLEVEL_TRACEMORE, ("adding newFile to wait list for task %x\n",
	    			myMsg.newFile.pid));
	    waitTask = (TASK_NODE *) malloc (sizeof (TASK_NODE));
	    waitTask->taskId = myMsg.newFile.pid;
	    waitTask->addr = (char *) 0;
	    waitTask->cdbState = CDB_NEWFILE;
	    waitTask->sendInfo = myMsg.newFile.sendInfo;
	    lstAdd (&cdbWaitList, (NODE *) waitTask);
	    }
	else
	    {


	    sb = myMsg.cdbCommand.cmd;
	    msg = ValFPsb (&sb);	/* get message number */
	    pid = ValFPsb (&sb);	/* get process id */

	    dprint(DLEVEL_LOUD, ("cdbServer, got MSG_CDB_COMMAND msg=%d, pid=%d\n",
			msg, pid));

	    status = statusOk;	/* to start with */

	    switch (msg)
		{
		default:
		    SendStatus (statusRetry, pid, 0);
		    break;

		case msgPtrace:
		    pt = ValFPsb (&sb);
		    adr = ValFPsb (&sb);
		    value = ValFPsb (&sb);

		    LocalPtrace (pt, pid, adr, value);
		    break;

		case msgWait:
		    waitTask = (TASK_NODE *) lstFirst (&cdbWaitList);
		    /* first parameter is blocking (0) or non-blocking (1) */
		    if ((waitTask == (TASK_NODE *) NULL) &&
						(ValFPsb (&sb) == 0))
			{
			vfWaitFlag = TRUE;
			}
		    else
			{
			if (waitTask == (TASK_NODE *) NULL)
			    {
			    SendStatus (statusOk, 0, 0);
			    }
			else
			    {
			    dprint (DLEVEL_TRACEMORE, ("wait command: list pid = %x, state = %x\n", waitTask->taskId,
						waitTask->cdbState)); 
			    if (waitTask->sendInfo)
				(*waitTask->sendInfo)(waitTask->taskId,
					waitTask->addr, waitTask->cdbState);
			    else
			    	SendStateChange (statusStateChange,
					waitTask->taskId, 
					waitTask->addr, 
					waitTask->cdbState);
			    lstDelete (&cdbWaitList, (NODE *) waitTask);
			    free ((char *) waitTask);
			    }
			}
		    break;

		case msgQuit:
		    if (pid == 0)
			{
			sprintf (vsbTx, "%x|%x\n", statusOk, pid);
			TxString (vsbTx);
			return (OK);	
			} /* if */
		    LocalPtrace (ptTerm, pid, (long) 0, 0);
		    break;


		case msgExec:
		    /* the pid returned should either be that specified by
		     * the host OR one determined by the local environment.
		     */

		    /* sb now points at "program|args" */

		    pid = PidFExecChild (sb, pid);
		    status = statusNewPid;
		    retVal = -1;

		    /* If we send a value of 0, it means that parent MUST send
		     * us msgWait.  A value of -1 says that it* must NOT send
		     * us the wait message.  What you send is based on how
		     * your PidFExecChild works.  The UNIX version needs the
		     * wait.
		     */

		    SendStatus (status, pid, retVal);
		    break;

		case msgInput:
		    /* KLUDGE - this should be fed to the program, somehow */
		    dprint(DLEVEL_TRACEMORE, ("KLUDGE - inputing: %s", sb));
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgHtoR:
		    adr = ValFPsb (&sb);
		    count = ValFPsb (&sb);
		    ReadHexBytes (pid, adr, count, sb);
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgRtoH:
		    adr = ValFPsb (&sb);
		    count = ValFPsb (&sb);
		    WriteHexBytes (pid, adr, count);
/*		    free (adr); /* */
		    break;

		case msgSbHtoR:
		    adr = ValFPsb (&sb);
		    count = ValFPsb (&sb);
		    ReadString (pid, adr, count, sb);
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgSbRtoH:
		    adr = ValFPsb (&sb);
		    count = ValFPsb (&sb);
		    WriteString (pid, adr, count);
/*		    free (adr); /* */
		    break;

		case msgMove:
		    adr = ValFPsb (&sb);
		    adr2 = ValFPsb (&sb);
		    count = ValFPsb (&sb);
		    bcopy ((char *) adr2, (char *)adr, count); 
		    SendStatus (status, pid, 0);
		    break;

		case msgSignal:
		    switch ((int) ValFPsb (&sb))
			{
			case SIGSTOP:
			    /* suspend the given task */
			    ts (pid);
			    /* put an event in the wait list */
			    waitTask = (TASK_NODE*) malloc (sizeof (TASK_NODE));
			    waitTask->taskId = pid;
			    waitTask->cdbState = SIGINT;
			    waitTask->sendInfo = NULL;
			    lstAdd (&cdbWaitList, (NODE *) waitTask);

			case SIGINT:
			case SIGKILL:
			case SIGTERM:
			    /* remove the given task */
			    td (pid);

			default:
			    break;
			} /* switch */
		    break;

		case msgLevel:

		    dprint(DLEVEL_TRACE,("msgLevel, sb=%s\n",sb));

		    /*vlevel = ValFPsb (&sb); /**/
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgDump:
		    /* Fill in with any appropriate goodies */
		    /* starting here ---->v			*/
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgBp:
		    /* set or remove? */
		    if (ValFPsb (&sb) == 0)
			{
			/* remove breakpoint */
			bd ((INSTR *) ValFPsb (&sb), pid);
			}
		    else
			{
			/* install breakpoint */
			b ((INSTR *) ValFPsb (&sb), pid, 0);
			} /* if */
		    SendStatus (statusOk, pid, 0);
		    break;

		case msgProcessList:
		    cdbSendTaskList();
		    break;

		case msgProcessName:
		    cdbSendProcName(pid);
		    break;

		case msgGetStatus:
		    cdbSendTaskStatus(pid);
		    break;
		} /* switch */
	    } /* if */
	} /* while */
    } /* cdbServer */

/*******************************************************************************
*
* cdbDaemon - cdb daemon
*
* Listens to socket connected to cdb running on Unix.
* Converts messages read from a socket into a command message and puts the
* command message in the cdb pipe.  DbgTask also writes to the cdb pipe.
* The cdb pipe feeds to the cdbServer.  Output from cdbServer goes
* directly to the cdb socket.
*/

VOID cdbDaemon ()

    {
    int 		sock;		/* socket */
    struct sockaddr_in	myAddress;
    struct sockaddr_in	cdbAddress;
    int			cdbAddrLen;
    int			i;

    cdbDaemonId = taskIdSelf ();

    /* don't allow breakpoints */
    taskOptionsSet (cdbDaemonId, VX_UNBREAKABLE, VX_UNBREAKABLE);

    /* create the cdbServer command pipe */

    if (pipeDevCreate ("/pipe/cdb", MAX_MSGS, sizeof (CDB_MSG)) == ERROR)
	{
	logMsg ("cdbDaemon: error creating /pipe/cdb");
	exit ();
	} /* if */

    if ((commandPipe = open ("/pipe/cdb", UPDATE)) < 0)
	{
	logMsg ("cdbDaemon: error opening /pipe/cdb");
	exit ();
	} /* if */

    /* cdb has not connected yet */

    cdbFlag = FALSE;
    cdbQuietFlag = FALSE;

    /* open a socket to cdb */

    if ((sock = socket ( AF_INET, SOCK_STREAM , 0)) == ERROR)
	{
	logMsg ("cdbDaemon: socket failed");
	exit ();
	}

    /* initialize socket structure */

    for (i = 0; i < sizeof (myAddress); i++)
	((char *) (&myAddress)) [i] = 0;

    myAddress.sin_family = AF_INET;
    myAddress.sin_port = CDB_SOCKET;
    myAddress.sin_addr.s_addr = 0;	/* daemon is on wrs */

    if (bind (sock, (struct sockaddr *) &myAddress, sizeof (myAddress))
	== ERROR)

	{
	logMsg ("cdbDaemon: bind failed.\n");
	logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	close (sock);
	exit ();
	}

    if (listen (sock, 1) == ERROR)
	{
	logMsg ("cdbDaemon: listen failed.");
	logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	close (sock);
	exit ();
	} /* if */

    /* 
     *  listen for cdb and pass cdb socket messages to 
     *  remote debug server in the cdb command pipe 
     */
    FOREVER
	{
	cdbAddrLen = sizeof (cdbAddress);

	/* wait for cdb to connect */

	if ((cdbSocket = accept (sock, (struct sockaddr *) &cdbAddress,
	    &cdbAddrLen)) == ERROR)

	    {
	    logMsg ("cdbDaemon: accept failed");
	    logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	    close (sock);
	    exit ();
	    }

	dprint(DLEVEL_TRACE,("cdbDaemon, cdbSocket = %d\n",cdbSocket));

	logMsg ("cdbDaemon: client accepted.\n");

	/* we are using cdb now, let excTask know */
	cdbFlag = TRUE;
	cdbQuietFlag = TRUE;

	/*
	if ((cdbServerId = vxGetFreeTaskId ()) == ERROR)
	    {
	    logMsg ("cdbDaemon: cdbServer task id error");
	    logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	    close (sock);
	    close (cdbSocket);
	    exit ();
	    }
	*/

	/* cdb needs to know whenever one of its spawned tasks is deleted.
	 *  Make UniWorks call cdbMsgTaskDeleted whenever a task is deleted.
	 */
	if (taskDeleteHookAdd (cdbMsgTaskDeleted) == ERROR)
	    {
	    RemotePanic ("taskDeleteHookAdd: task delete table full");
	    return (ERROR);
	    }

	/* spawn the cdbServer task*/
	if ((cdbServerId = taskSpawn ("cdbServer", CDB_PRIORITY,
	        VX_SUPERVISOR_MODE, CDB_STACK_SIZE, cdbServer)) == ERROR)
	    {
	    logMsg ("cdbDaemon: error spawning cdbServer");
	    logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	    close (sock);
	    close (cdbSocket);
	    exit ();
	    } /* if */

	dprint(DLEVEL_TRACE,("cdbDaemon, spawned cdbServer pid=%d\n",cdbServerId));

	/* Do not add this hook until the debug server is in place !!!! */
	if (taskCreateHookAdd (cdbMsgTaskCreated) == ERROR)
	    {
	    RemotePanic ("taskCreateHookAdd: task create table full");
	    return (ERROR);
	    }

	/* read from cdb socket and write to cdbServer 
	 *  command pipe until cdb disconnects 
	 */
	if (cdbSockToPipe () == ERROR)
	    {
	    logMsg ("cdbDaemon: error sending command to pipe");
	    logMsg ("cdbDaemon: status = %x\n", errnoGet ());
	    close (sock);
	    close (cdbSocket);
	    exit ();
	    } /* if */

	if (close (cdbSocket) == ERROR)
	    {
	    logMsg ("cdbDaemon: error closing socket");
	    close (sock);
	    exit ();
	    } /* if */

	/* flush the pipe */
	ioctl (commandPipe, FIOFLUSH, (int *) 0);

	/* don't need to tell cdb whenever a task is deleted anymore */
	if (taskDeleteHookDelete (cdbMsgTaskDeleted) == ERROR)
	    {
	    RemotePanic ("can't find cdbMsgTaskDeleted in task delete table");
	    return (ERROR);
	    } /* if */
	if (taskCreateHookDelete (cdbMsgTaskCreated) == ERROR)
	    {
	    RemotePanic ("can't find cdbMsgTaskCreated in task create table");
	    return (ERROR);
	    } /* if */

	/* free the malloc'd lists */
	lstFree (&cdbWaitList);

	/* delete cdbServer task */
	td (cdbServerId);

	/* cdb is no longer around */
	cdbFlag = FALSE;
	cdbQuietFlag = FALSE;

	logMsg ("cdbDaemon: client disconnected.\n");
	} /* FOREVER */
    } /* cdbDaemon */


/*******************************************************************************
*
* cdbSockToPipe - read messages from cdb socket and send out to cdbServer 
* 		 via a pipe
*
* cdbSockToPipe receives bytes from the cdb socket into a fixed length buffer.
* Commands are terminated with newline characters.  The buffer is scanned
* for commands.  Each command is written to cdbServer's command pipe.  If a 
* complete command runs past the end of the buffer, the rest of it is picked 
* up in successive reads until a newline is reached.
* This routine runs until cdb disconnects.
*
* RETURNS: (OK | ERROR)
*
*/

LOCAL STATUS cdbSockToPipe ()

    {
    char	recvBuf [BUFSIZE];	/* buffer for recving from socket */
    char	*p;			/* pointer to character in buf */
    char	*concatBuf;		/* holds partial commands until they are
						       complete */
    int		concatLength;		/* length of concatBuf */
    int		nLeft;			/* # of chars left to scan in recvBuf */
    char	*pLine;			/* pointer to command */
    int		lineLength;		/* length of pLine */
    CDB_MSG	cdbMsg;			/* msg written to pipe */

    p = recvBuf;
    concatBuf = (char *) NULL;
    concatLength = 0;


    FOREVER
	{
	if ((nLeft = recv (cdbSocket, recvBuf, sizeof (recvBuf), 0)) <= 0)
	    return (OK);	/* return when cdb goes away */

	p = recvBuf;
	nLeft = nLeft;

	while (nLeft > 0)
	    {

	    dprint(DLEVEL_VERBOSE,("cdbSockToPipe, got a msg from cdbx\n"));

	    /* read a line pointed to by c in recvBuf, 
	     *  a line will be newline terminated or nLeft characters
	     *  long, whichever comes first.  If it is newline terminated,
	     *  the newline will be replaced by a null character.
	     */
	    pLine = readLine (p, nLeft, &lineLength);

	    if (pLine [lineLength - 1] == '\0')
		{
		/* line returned from readLine was terminated with a newline
		 * It is a full command. Send it on to cdbServer through 
		 * the pipe.
		 */

		if (concatBuf != NULL)
		    {
		    /* partial command is in concatBuf */
		    /* append pLine to concatBuf and put in cdb message */ 
		    cdbMsg.cdbCommand.cmd = appendBuf (concatBuf, concatLength,
							pLine, lineLength);
		    }
		else
		    {
		    /* complete command is in pLine */
		    cdbMsg.cdbCommand.cmd = pLine;
		    }

		cdbMsg.request = MSG_CDB_COMMAND;

		/* write message to pipe */
		if (write (commandPipe, (char *) &cdbMsg, sizeof (cdbMsg)) != 
			sizeof (cdbMsg))
		    return (ERROR);
		/**** Do we want to set status here??? */

		concatBuf = NULL;
		concatLength = 0;
		}
	    else
		{
		/* use realloc when we get it */
		/* got to end of recvBuf, line was not terminated 
		 * This is a partial command, save it in concatBuf
		 */

		if (concatBuf != NULL)
		    {
		    /* append pLine to concatBuf */
		    concatBuf = appendBuf (concatBuf, concatLength, pLine, 
						lineLength);
		    concatLength += lineLength;
		    }
		else
		    {
		    concatBuf = pLine;
		    concatLength = lineLength;
		    }
		}
	    p += lineLength;
	    nLeft -= lineLength;
	    } /* while */
	} /* FOREVER */
    } /* cdbSockToPipe */

/*******************************************************************************
*
* appendBuf - concatinate two buffers
*
* Concatenate two buffers and return resulting buffer.  Free up the original
* buffers.
*
* RETURNS:  the number of characters in the line
*	    pLine points to an allocated buffer containing the line
*/

LOCAL char *appendBuf (buf1, l1, buf2, l2)
    char *buf1;
    int  l1;	/* length of buf1 */
    char *buf2;
    int  l2;	/* length of buf2 */

    {
    char *tempBuf;

    tempBuf = malloc (l1 + l2);
    bcopy (buf1, tempBuf, l1);
    bcopy (buf2, &tempBuf [l1], l2);
    free (buf1);
    free (buf2);
    return (tempBuf);
    }

/*******************************************************************************
*
* readLine - read a line from the buffer up to a newline or the maximum 
*	     number of characters, whichever comes first
*
* Scans buffer and puts the read line into allocated storage.
* Replaces '\n' with a null character.
*
* RETURNS: the number of characters in the line (includes null character)
*/

LOCAL char *readLine (buf, maxChars, length)
    char *buf;		/* buffer to scan */
    int maxChars;	/* maximum number of characters to scan */
    int *length;	/* pointer to line being returned */

    {
    int i = 0;
    char *pLine;

    /* look for '\n' */
    while (i < maxChars)
	{
	if (buf [i++] == '\n')
	    break;
	} /* while */

    pLine = malloc (i);
    bcopy (buf, pLine, i);
    if (pLine [i - 1] == '\n')	/* replace newline with null char */
	pLine [i - 1] = '\0';
    *length = i;
    return (pLine);			/* return length of line */
    } /* readLine */

/***************************************************************
*
* cdbBreakpoint - send breakpoint information to cdb
*
* Called by dbgLib whenever a breakpoint is hit.  Writes some info to
* a pipe, where it will be read and handled by cdbServer.
*/

VOID cdbBreakpoint (pid, pc)
    int pid;
    char *pc;

    {
    CDB_MSG	cdbMsg;

    /* if cdb is not turned on, no need to talk to it */

    if (!cdbFlag)
	return;

    /* send information to cdb via commandPipe */

    cdbMsg.request = MSG_BREAKPOINT;
    cdbMsg.breakPoint.pid = pid;
    cdbMsg.breakPoint.addr = pc;

    write (commandPipe, (char *) &cdbMsg, sizeof (cdbMsg));
    }

/***************************************************************
*
* cdbMsgTaskDeleted - tell cdb when a task is deleted
*
* Called by the UniWorks task delete hook.
*/

LOCAL VOID cdbMsgTaskDeleted (pTcbX)
    FAST TCBX *pTcbX;

    {
    CDB_MSG cdbMsg;

    cdbMsg.request = MSG_TASK_DELETED;
    cdbMsg.taskDeleted.pid = pTcbX->taskId;
    cdbMsg.taskDeleted.addr = (char *) 0;		/* not needed */

    if (write (commandPipe, (char *) &cdbMsg, sizeof (cdbMsg)) != 
	sizeof (cdbMsg))

	RemotePanic ("Pipe write error.  Task #%d deleted", pTcbX->taskId);

    }


/***************************************************************
*
* cdbMsgTaskCreated - tell cdb when a task is created
*
* Called by the UniWorks task create hook.
*/

LOCAL VOID cdbMsgTaskCreated (pTcbX)
    FAST TCBX *pTcbX;

    {
    CDB_MSG cdbMsg;

    cdbMsg.request = MSG_TASK_CREATED;
    cdbMsg.taskDeleted.pid = pTcbX->taskId;
    cdbMsg.taskDeleted.addr = (char *) pTcbX->entry;

    dprint(DLEVEL_TRACE,("taskCreated, id=%d, entry=0x%x\n",
	pTcbX->taskId,pTcbX->entry));
/*
    ts(pTcbX->taskId);
*/
#ifndef	is68k
    b ((INSTR *) pTcbX->entry, pTcbX->taskId, 0);
#endif	is68k
    if (write (commandPipe, (char *) &cdbMsg, sizeof (cdbMsg)) != 
	sizeof (cdbMsg))

	RemotePanic ("Pipe write error.  Task #%d created", pTcbX->taskId);

    }


/***************************************************************
*
* cdbSendTaskList - tell cdb all of the task id's that are
*		    cuurently active in the system
*
*/

LOCAL VOID cdbSendTaskList ()

    {
    VX_TCB	*pTcb;			/* index into chain of TCBs */
    char	*concatBuf;
    int		concatLength;
    char	*pLine;
    int		lineLength;
    char	sbBuf[32];
    int		idlist[MAX_CDB_TASKS];
    int		nTasks,i;

    sprintf (sbBuf, "%x|%x", statusOk, 0, 0);
    concatLength = strlen(sbBuf) + 1;
    concatBuf = malloc (concatLength);
    bcopy (sbBuf, concatBuf, concatLength);

    nTasks = taskIdListGet (idlist, MAX_CDB_TASKS);
    for (i=0; i < nTasks; i++) {
	pTcb = (VX_TCB *)taskTcb(idlist[i]);
	if (pTcb->tcb_id == cdbServerId)
		continue;
	sprintf (sbBuf, "|%x", pTcb->tcb_id);
	lineLength = strlen(sbBuf);
	pLine = malloc(lineLength);
	bcopy (sbBuf, pLine, lineLength);
	concatBuf = appendBuf (concatBuf, concatLength - 1, pLine, 
				lineLength + 1);
	concatLength += lineLength;
	}

    sprintf (vsbTx, "%s\n", concatBuf);
    free (concatBuf);
    dprint(DLEVEL_TRACE,("cdbSendTaskStatus, vsbTx=%s\n",vsbTx));
    TxString (vsbTx);
    }

/***************************************************************
*
* cdbSendTaskStatus - return status info for the given task
*
*/

LOCAL VOID cdbSendTaskStatus (pid)
    int pid;

    {
    VX_TCB	*pTcb;		/* index into chain of TCBs */
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    char statusString[10];	/* status string goes here */
    int		idlist[MAX_CDB_TASKS];
    int		nTasks,i;

    nTasks = taskIdListGet (idlist, MAX_CDB_TASKS);
    for (i=0; i < nTasks; i++) {
	pTcb = (VX_TCB *)taskTcb(idlist[i]);
	if (pTcb->tcb_id != pid)
	    continue;

	taskRegsGet (idlist[i], dregs, aregs, &sp, &sr, &pc);
	taskStatusString (idlist[i], statusString);

	sprintf (vsbTx, "%x|%x|%x|%x|%s|%x|%x|%x|%x|%x|%x\n",
	    statusOk,
	    pid,
	    pTcb->tcb_id,
	    pTcb->tcb_priority,
	    statusString,
	    pc,
	    sp,
	    0,		/* not used by cdb */
	    pTcb->tcb_extension->topOfStack, 
	    pTcb->tcb_extension->errorStatus,
	    0);		/* not used by cdb */

	dprint(DLEVEL_TRACE,("cdbSendTaskStatus, vsbTx=%s\n",vsbTx));
	TxString (vsbTx);
	return;
	}

    sprintf (vsbTx, "%x|%x|%x\n", ESRCH, pid, 0);
    TxString (vsbTx);
    }


/***************************************************************
*
* cdbSendProcName - return the entry address of the given task
*
*/

LOCAL VOID cdbSendProcName (pid)
    int pid;

    {
    int		idlist[MAX_CDB_TASKS];
    int		nTasks,i;
    VX_TCB	*pTcb;			/* index into chain of TCBs */

    nTasks = taskIdListGet (idlist, MAX_CDB_TASKS);
    for (i=0; i < nTasks; i++) {
	pTcb = (VX_TCB *)taskTcb(idlist[i]);
	if (pTcb->tcb_id != pid)
	    continue;

	sprintf (vsbTx, "%x|%x|%x\n", statusOk, pid,
			pTcb->tcb_extension->entry);
	dprint(DLEVEL_TRACE,("cdbSendProcName, vsbTx=%s\n",vsbTx));
	TxString (vsbTx);
	return;
	}

    sprintf (vsbTx, "%x|%x|%x\n", ESRCH, pid, 0);
    TxString (vsbTx);
    }

/**********************************************************************
*
* cdbInit - initialize cdb support
*
* This routine initializes the target side of the cdb support package.
* It spawns cdbDaemon, and sets dbgLib's dbgBreakNotifyRtn  variable,
* so cdbBreakpoint will be hit when a breakpoint occurs.
*
* This routine should be called exactly once, before the cdb package is
* used.
*
* RETURNS
*  OK,   or
*  ERROR if couldn't spawn cdbDaemon
*/

STATUS cdbInit ()

    {
    if (taskSpawn ("cdbDaemon",CDB_PRIORITY, VX_SUPERVISOR_MODE, 3000,
	cdbDaemon) == ERROR)

	return (ERROR);

    /* Set dbgLib's hook variable, so dbgBreakpoint will be called. */

    dbgBreakNotifyRtn = cdbBreakpoint;
    return (OK);
    }

LOCAL SBT *vpNames;
LOCAL unsigned long *vpAddrs = NULL;

LOCAL viname = inameNil, vinameMax;
LOCAL int vcbNames;
LOCAL SBT vsbNames = sbNil;

LOCAL VOID SendFileInfo ();
/**********************************************************************
*
* cdbNewLoad ()
*
* This routine is called every time an incremental load is started
* It tells the server what file needs to be translated and read into
* the server's symbol table
*/

VOID cdbStartLoad (maxSymbols)
    int maxSymbols;

    {

    if (!cdbFlag || maxSymbols <= 0)
	return;
    if ((vpNames = (SBT *) malloc(maxSymbols * sizeof(SBT *))) == NULL)
	{
	viname = inameNil;
	return;
	}
    if ((vpAddrs = (unsigned long *)
		malloc(maxSymbols * sizeof(unsigned long *))) == NULL)
	{
	free(vpNames);
	viname = inameNil;
	return;
	}
    viname = 0;
    vinameMax = maxSymbols;
    }
/**********************************************************************
*
* cdbSymAdd ()
*
* This routine is called every time a new symbol is added to 
* the global symbol table;
* name/address information is stuffed into a pool, waiting
* to be dumped out to the server
*/

VOID cdbSymAdd (name, value)
    SBT name;
    unsigned long value;
    {

    FAST int newmax;
    FAST SBT *pnamesTmp;
    FAST unsigned long *paddrsTmp;

    if (!cdbFlag || viname == inameNil)
	return;
    if (viname == vinameMax)
	{
	newmax = vinameMax * 3/2;
    	if ((pnamesTmp = (SBT *) realloc(vpNames, 
					newmax * sizeof(SBT *))) == NULL)
	    return;
	if ((paddrsTmp = (unsigned long *) realloc ((char *)vpAddrs,
				newmax * sizeof(unsigned long *))) == NULL)
	    return;
	vinameMax = newmax;
	vpNames = pnamesTmp;
	vpAddrs = paddrsTmp;
	}
    vpNames[viname] = name;
    vpAddrs[viname++] = value;
    }
/**********************************************************************
*
* cdbEndLoad ()
*
* This routine is called after a load is completed, so the server
* can be informed and get the name/address info for the globals
*/

VOID cdbEndLoad ()

    {
    FAST int inameTmp;
    FAST int cbname;
    FAST SBT sb;
    CDB_MSG cdbMsg;

    if (!cdbFlag || viname == inameNil)
	return;
    vcbNames = 2;		/* '\n', '\0' */
    for (inameTmp = 0; inameTmp < viname; inameTmp++)
	{
	dprint(DLEVEL_TRACEMORE, ("got %s at %x\n", vpNames[inameTmp], vpAddrs[inameTmp]));
	vcbNames += strlen(vpNames[inameTmp]) + 1;
	}
    if ((vsbNames = (SBT) malloc(vcbNames)) != NULL)
	{
	sb = vsbNames;
	for (inameTmp = 0; inameTmp < viname; inameTmp++)
	    {
	    cbname = strlen(vpNames[inameTmp]);
	    strncpy(sb, vpNames[inameTmp], cbname);
	    sb += cbname;
	    *sb++ = chDelim;
	    if (sb - vsbNames > vcbNames)
		{
		printf("overwrote string table: curr: %d, total: %d\n",
				sb - vsbNames, vcbNames);
		return;
		}
	    }
	*sb++ = '\n';
	*sb = '\0';
#ifdef	DEBUG
	for (sb = vsbNames; *sb; )
	    {
	    char tmp[100];
	    SBT foo;
	    if ((foo = (SBT)index(sb, chDelim)) == sbNil)
		break;
	    strncpy(tmp, sb, foo - sb);
	    tmp[foo - sb] = '\0';
	    dprint (DLEVEL_TRACEMORE, ("reading back %s\n", tmp));
	    sb = foo + 1;
	    }
#endif	DEBUG
	cdbMsg.request = MSG_NEWFILE;
	cdbMsg.newFile.pid = 0;
	cdbMsg.newFile.sendInfo = SendFileInfo;
	if (write (commandPipe, (char *) &cdbMsg, sizeof (CDB_MSG))
							!= sizeof (CDB_MSG))
	    RemotePanic ("Pipe write error on new file\n");
	}
    else
	{
    	free(vpAddrs);
	vpAddrs = NULL;
	}
    free(vpNames);
    }
/***********************************************************************
*
* SendFileInfo
*/

LOCAL VOID SendFileInfo (pid, addr, state)
    int pid; 
    char *addr;
    int state;

    {
    dprint (DLEVEL_VERBOSE, ("SendFileInfo: %x|%x|%x|%x|%x|%x|%x|%x\n", statusStateChange,
		pid, addr, state, vsbNames, vcbNames, vpAddrs,
		viname * sizeof (unsigned long *)));
    sprintf (vsbTx, "%x|%x|%x|%x|%x|%x|%x|%x\n", statusStateChange,
    		pid, addr, state, vsbNames, vcbNames, vpAddrs,
		viname * sizeof (unsigned long *));
    TxString (vsbTx);
    viname = inameNil;
    }
