/* qtDrv.c - ISI/Liberator SCSI tape driver */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/* 
modification history
--------------------
*/

/*
DESCRIPTION
*/

#define	SCSI_LIB

/*#define 	DEBUG			/**/
/*#define	DEBUGF		/**/

/*
 * TODO: 
	- if a command times out, should deselect target ?
 */

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "rtLib.h"
#include "semLib.h"
#include "sysLib.h"
#include "types.h"

#include "buf.h"
#include "dkbad.h"
#include "qbvar.h"

#include "openchip.h"
#include "qsvar.h"
#include "qsreg.h"
#include "qt.h"
#include "qs.h"
#include "scsi.h"
#include "mtio.h"

/*		_____________________________________________
 *  dev:	|15 ... | 8 | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |
 *		---------------------------------------------
 *		         \___cont___/\__targ___/\___lun____/
 *			  \________ctlr_______/
 *			   \_____________unit______________/
 */

/* Macros */

#define QT_CTLR(cont, targ)     (((cont) << 3) | ((targ) - 0))
#define QT_MKUNIT(ctlr, lun)	((ctlr << 3) | lun)
#define QT_UNIT(dev)	    	(dev & 0x1FF)
#define QTU_LUN(unit)	   	(unit & 7)
#define QTU_CTLR(unit)	  	(unit >> 3) 
#define QTU_TARG(unit)	  	((QTU_CTLR(unit) & 7) + 0)
#define QTU_CONT(unit)	  	(QTU_CTLR(unit) >> 3)
#define QT_MAXCNT	       	10240
#define QT_MAXBA	       	0xFFFFFFFF
#define	QT_MAXRETRY		3

/* Globals */

int	qtNum;		/* number of this driver */
char	qtiobuf[10240]; /* io buffer */

/* 
 * per-controller status struct - can only do one command at a time 
 * 				  per-controller.
 */
struct qt_softc	{
	char 			qt_buf;		/* for 1 byte commands */
	struct scsi_inq		qt_inq;		/* for inquire data */
	union scsi_sns	  	qt_sense;	/* for sense data */
	struct scsi_msen	qt_msen;/* to sense write protect during open */
	struct scsi_blklim	blklim;	/* store block size limits */
	struct buf      	qt_rbuf;   	/* used instead of rqtbuf[] */
	struct scsi_cmnd 	*qt_cmd;	/* ptr to scsi cmd struct */
	struct scsi_cmnd	qt_cmdsav;	/* for retries */
	union scsi_cdb		qt_cdb;		/* command block */
	union ihead {
		struct isihead	{
			u_short	ih_id;
			u_short	ih_fileno;
			u_short	ih_fileba;
			u_short	ih_blksz;
			u_short	ih_count;
		} qt_ih;
		char buf[512];
	}	qt_ihead; /* header block for isi tapes */
}	qtsoftc[NQT];

/*
 * per-device status struct
 */
struct qt_dev	{

	DEV_HDR	dev_hdr;	/* for io system */
	u_int 	flaqs;		/* following are defines */

#define	QTF_REWIND	0x01	/* do no rewind on close */
#define QTF_ISI		0x02	/* ISI format */
#define	QTF_SWAB	0x04	/* swap bytes at controller level */
#define	QTF_FSF		0x05	/* forward space file */

	u_short dev;		/* CONT | TARG | LUN */
	u_char	task;		/* task id */
	u_char	d_cmd;		/* save direct cmd in case theres status */
	char	retries;	/* number of retries */

	struct		{
		unsigned f_open		: 1;	/* device open */
		unsigned f_created	: 1;	/* device created */
		unsigned f_mode		: 2;	/* READ | WRITE | UPDATE */
		unsigned f_direct	: 1;	/* direct command */
		unsigned f_nodisc	: 1;	/* disallow disconnects */
		unsigned f_swcd		: 1;	/* swap command data */
		unsigned f_swmd		: 1;	/* swap media data */
		unsigned f_busy		: 1;	/* using raw buffer */
		unsigned f_want		: 1;	/* want to use raw buffer */
		unsigned f_sense	: 2;	/* times sense done on device */
		unsigned f_lastiow	: 1;	/* last i/o was a write */
		unsigned f_seeking	: 1;	/* seeking to block address */
		unsigned f_hdone	: 1;	/* isi header read/written */
		unsigned f_notraw	: 1;	/* opened as block device */
		unsigned f_reject	: 1;	/* not a valid device combo */
		unsigned f_space_type	: 1;	/* space FILE=0, RECORD=1 */
		unsigned f_blk_check	: 1;	/* got blank check */
		unsigned f_adjust	: 1;	/* adjusting position */
		unsigned f_	: 4;	/* */
	}		fbits;
#define				s_open		fbits.f_open
#define				s_created	fbits.f_created
#define				s_mode		fbits.f_mode
#define				s_direct	fbits.f_direct
#define				s_nodisc	fbits.f_nodisc
#define				s_swcd	 	fbits.f_swcd
#define				s_swmd 		fbits.f_swmd
#define				s_busy 		fbits.f_busy
#define				s_want 		fbits.f_want
#define				s_sense		fbits.f_sense
#define				s_lastiow	fbits.f_lastiow
#define				s_seeking	fbits.f_seeking
#define				s_hdone		fbits.f_hdone
#define				s_notraw	fbits.f_notraw
#define				s_reject	fbits.f_reject
#define				s_space_type	fbits.f_space_type
#define				s_blk_check	fbits.f_blk_check
#define				s_adjust	fbits.f_adjust

	int			resid;	/* most recent residual count */
	u_int			fileno; /* current file number */
	u_int			fileba; /* current ba in file */
	u_int			blksz;	/* size tape blocks (0 if variable) */
	int			count;	/* count for current op */
	u_int			roundeven;	/* bytecount rounded up */
	u_short			needreinit;	/* device needs reiniting */
	u_short			dtype;	/* brand of drive (for quirks) */
	u_short			mt_erreg; /* for ioctl status */
#define	EXABYTE	1
#define	WANQTEK	2
#define	ARCHIVE	3

}		qtdev[Nqt];


/* Externs */

extern SEMAPHORE qs_sems[];

/* Forward declarations */

int qtOpen(), qtReset(), qtClose(), qtRead(); int qtWrite(), qtIoctl();

/************************************************************************
*
* qtDrv() - called from usrConfig to start up driver.
*
*/
qtDrv()
{
	register int cont, target, ctlr, lun;

	qtNum = iosDrvInstall(qtOpen, (FUNCPTR) NULL, qtOpen, qtClose, qtRead,
		qtWrite, qtIoctl);

	/* Initialize each controller and create a device for each unit */
	for(cont=0;cont < NQT;cont++) {
		if(qsprobe(cont) != OK)
			continue;
		for(target=0;target < Nqt_targ_cont;target++) {
			ctlr = QT_CTLR(cont,target);
			for(lun=0;lun < Nqt_lun_targ;lun++) {
				qtDevCreate(QT_MKUNIT(ctlr,lun));
			}
		}

	}
	return(OK);
}

/************************************************************************
*
* qtDevCreate - check to see that unit exists and if tape device,
*		create a device 
*
*/

qtDevCreate(unit)
{
	char name[20];
	register struct qt_dev *dev;

	dev = &qtdev[unit];
	dev->flaqs = QTF_ISI | QTF_REWIND | QTF_SWAB;
	dev->s_nodisc = TRUE;
	dev->s_swmd = TRUE;
	dev->s_direct = FALSE;
	dev->s_sense = FALSE;
	dev->dev = unit;

	/* Check to see if physical unit(drive) exists */
	if(qtinq(unit) == ERROR) {
		return(ERROR);
	}

	/* Get unit info */
	if(qtinfo(unit) == ERROR)
		return(ERROR);

	sprintf(name,"/qt%d",unit);
	return(iosDevAdd((DEV_HDR *) &qtdev[unit], name, qtNum));
}

/************************************************************************
*
* qtissue - issue a command
*
*/

qtissue(unit, cmd, timeout, loud)
	register struct scsi_cmnd *cmd;
	register int loud;
{
	int stat;
	int cont, targ;
	register struct qt_dev *dev;
	register struct qt_softc *softc;
#define SHHHH(x)		{ if(loud) x;}

	cont = QTU_CONT(unit);
	targ = QTU_TARG(unit);
	dev = &qtdev[unit];
	softc = &qtsoftc[QTU_CONT(unit)];

	/*
	 * Issue the command, then process any errors. If doerr returns
	 * a 1, then issue another command.
	 */

	dev->retries = 0;
	dev->s_sense = 0;
	stat = 0;
	do {
		if (qscmdwait(cont, targ, cmd, timeout))  {
			SHHHH(logMsg("ERROR qt%d: scsi cmd 0x%x timed out\n",
				unit,softc->qt_cdb.cdb_0.cdb_0_cmd)); 
			return(ERROR);
		}
	} while ((stat = qtdoerr(unit,cmd,loud)) == 1);
	dev->s_direct = FALSE;
	return(stat);
}

/************************************************************************
*
* qtdoerr - process any errors from command
*
* RETURNS: -1 == fatal error; 1 == another command; 0 == OK
*
*/

qtdoerr(unit, cmd, loud)
	register struct scsi_cmnd *cmd;
	register int loud;
{
	int cont, lun, stat;
	int filemark;
	register struct qt_dev *dev;
	register struct qt_softc *softc;
	register union scsi_cdb *cdb;
	register struct buf *bp;
	register union scsi_sns *sns;
#define SHHHH(x)		{ if(loud) x;}

	filemark = 0;
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	dev = &qtdev[unit];
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;
	sns = &softc->qt_sense;
	bp = &softc->qt_rbuf;

	/* If inquire cmd and timeout, there isn't a target there */
	if((cdb->cdb_0.cdb_0_cmd == CMD_INQUIRY) &&
	  (cmd->qs_finstat == SFN_SLCT_TIMEOUT))
		return(0);

	/* Check for fatal errors */
	if(cmd->qs_finstat == SFN_SLCT_TIMEOUT) { 
		dev->resid = dev->count;
		SHHHH(logMsg("ERROR qt%d: could not select target\n",
		      unit)); 
		return(ERROR);
	}
	if(cmd->qs_finstat != SFN_NORMAL_END) {
		dev->resid = dev->count;
		SHHHH(logMsg("ERROR qt%d: scsi cmd 0x%x did not finish\n",
		      unit, cdb->cdb_0.cdb_0_cmd)); 
		return(ERROR);
	}
	if(cmd->qs_flaqs & SCF_FATAL_COMB) {
		dev->resid = dev->count;
		SHHHH(logMsg("ERROR qt%d: scsi cmd 0x%x failed: error 0x%x\n",
		      unit, cdb->cdb_0.cdb_0_cmd, cmd->qs_flaqs)); 
		return(ERROR);
	}
	if((cmd->qs_status[0] & SCSTAT_CHK) && dev->s_sense) {
		dev->resid = dev->count;
		dev->s_sense = FALSE;
		SHHHH(logMsg("ERROR qt%d: scsi SENSE cmd failed\n",
		      unit)); 
		return(ERROR);
	}

	/* Get sense data if it's there to be got */
	if(cmd->qs_status[0] & SCSTAT_CHK) {

#ifdef	DEBUGF
		printf("qtdoerr qt%d: getting sense info\n",unit);
#endif	DEBUGF
		dev->s_sense = TRUE;
		bcopy(cmd,&softc->qt_cmdsav,sizeof(struct scsi_cmnd));
		SET_CDB_0(cdb, CMD_SENSE, lun, 0, sizeof(union scsi_sns), 0, 0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_sense,
			sizeof(union scsi_sns),0,0);
		return(1);
	}
	dev->resid = 0;

	/* If there was sense data, check it out */
	if(dev->s_sense) {
#ifdef	DEBUGF
		printf("qtdoerr qt%d: checking sense info\n",unit);
#endif	DEBUGF
		dev->s_sense = 0;
		if(sns->sns_7.sns_7_valid)
			if(!dev->s_adjust)
				dev->resid = SCSI_HMML(sns->sns_7.sns_7_info);
#ifdef	DEBUGF
			logMsg("SENSE VALID, resid=%d\n",dev->resid);
#endif	DEBUGF
		if(sns->sns_7.sns_7_class != 7) {
			SHHHH(logMsg("ERROR qt%d: invalid sense class %d\n",
		              unit,sns->sns_7.sns_7_class));
			return(ERROR);
		}
		if(sns->sns_7.sns_7_fil_mk) {
#ifdef	DEBUGF
			printf("qtdoerr: got a filemark\n");
#endif	DEBUGF
			filemark = 1;
			dev->fileba = 0;
		}
		if(sns->sns_7.sns_7_eom) {
			/* if it is a space command then process this error
			 * a little further down.  We may have reached BOT
			 * on an exact count
			 */
			if (dev->d_cmd != CMD_SPACE) {
		    SHHHH(logMsg("ERROR qt%d: unexpected physical EOT/BOT\n",
				  unit));
			    return(ERROR);
			}
		}
		if(dev->s_notraw && sns->sns_7.sns_7_ili) {
			SHHHH(logMsg("ERROR qt%d: incorrect record lenqth\n",
		              unit));
			dev->fileba += (dev->count - dev->resid);
			return(ERROR);	
		}

		/* Check sense key */
		dev->mt_erreg = (sns->sns_7.sns_7_key << 8) |
			sns->sns_7.sns_7_err;
		switch (sns->sns_7.sns_7_key) {
		    case SNS_7_KEY_NO_SENSE: case SNS_7_KEY_RECOVERED: 
#ifdef	DEBUGF
			printf("SNS_7_KEY_NO_SENSE | SNS_7_KEY_RECOVERED\n");
#endif	DEBUGF
			break; /* go on as if no error */

		    case SNS_7_KEY_ABORT_CMD:
#ifdef	DEBUGF
			printf("SNS_7_ABORT_CMD \n");
#endif	DEBUGF
			if(dev->retries++ < QT_MAXRETRY) {
				bcopy(&softc->qt_cmdsav,cmd,
					sizeof(struct scsi_cmnd));
				return(1);
			}
		    case SNS_7_KEY_DATA_PROTECT:
			SHHHH(logMsg("qt%d: CARTRIDGE WRITE PROTECTED\n",unit));
			return (ERROR);
		    case SNS_7_KEY_UNIT_ATTN:
	SHHHH(logMsg("qt%d: DRIVE RESET OR CARTRIDGE CHANGED: reinitializing\n",
		              unit));
			qtreinit(softc, dev, cmd, cdb);
			if(dev->fileba || dev->fileno) 
				return(0);
			dev->fileba = 0;
			dev->fileno = 0;
			return (0);
		    case SNS_7_KEY_BLANK_CHECK:
#ifdef	DEBUGF
			printf("SNS_7_BLANK_CHECK \n");
#endif	DEBUGF
			filemark = 1; /* treat like a second filemark */
			dev->s_blk_check = TRUE;
			break;
		    default:
			SHHHH(logMsg("ERROR qt%d: hard error SENSE KEY %d\n", 
				unit,sns->sns_7.sns_7_key));
			return (ERROR);
		}
	}

	/* If it was a direct command, fix up file stuff */
    	if (dev->s_direct) {
	    switch (dev->d_cmd) {
	    case CMD_SPACE:
		if (dev->s_space_type) {	/* skip files */
		    if (!dev->s_adjust) {
			dev->fileba = 0;
			if(sns->sns_7.sns_7_eom && dev->resid)  {
			    if ((dev->resid != 0xffffff) && (dev->resid != 1)) {
				if (dev->count > 0)
				    dev->fileno += dev->count - dev->resid;
				else
				    dev->fileno = 0;
				return (ERROR);
			    } else {
				if (dev->count > 0)
				    dev->fileno += dev->count;
				else
				    dev->fileno += dev->count + 1;
				dev->resid = 0;
			    }
			} else {
			    if (dev->count > 0)
				dev->fileno += dev->count;
			    else
				dev->fileno += dev->count + 1;
			    if ((dev->resid != 0) && (dev->resid != 1)) 
				return (ERROR);
			    else
				dev->resid = 0;
			}
		    }
		} else {				/* space records */
		    if (!dev->s_adjust) {
#ifdef DEBUG
logMsg ("SR: eom = 0x%x, resid = %d, fileno = %d, count = %d, filemark = %d\n",
sns->sns_7.sns_7_eom, dev->resid, dev->fileno, dev->count, filemark);
#endif DEBUG
			if(sns->sns_7.sns_7_eom && dev->resid)  {
			    if ((dev->resid != 0xffffff) && (dev->resid != 1)) {
				if (dev->count > 0)
				    dev->fileba += (dev->count - dev->resid);
				else
				    dev->fileba = 0;
				return (ERROR);
			    } else {
				if (dev->count > 0)
				    dev->fileno += dev->count;
				else
				    dev->fileno += dev->count + 1;
				dev->resid = 0;
			    }
			} else {
			    if (filemark) {
				if (dev->count > 0) {
					dev->fileno++;
					dev->fileba = 0;
				} else {
					dev->fileno--;
					dev->fileba = QT_MAXBA;
				}
				if((dev->resid != 0xffffff)&&(dev->resid != 1)
				    &&(dev->resid != 0))
				    return (ERROR);
				else
				    dev->resid = 0;
			    } else {
				    dev->fileba += (dev->count - dev->resid);
			    }
			}
#ifdef DEBUG
logMsg ("SR: eom = 0x%x, resid = %d, fileno = %d, count = %d\n",
sns->sns_7.sns_7_eom, dev->resid, dev->fileno, dev->count);
#endif DEBUG
		    }
		}
		break;
	    case CMD_REZERO:
#ifdef	DEBUGF
		printf("doerr: direct cmd: CMD_REZERO\n");
#endif	DEBUGF
		dev->fileba = 0;
		dev->fileno = 0;
		break;
	    default:
#ifdef	DEBUGF
	    	printf("doerr: direct cmd default:0x%x\n",dev->d_cmd);
#endif	DEBUGF
		break;
	    } /* end switch */
	    return (0);
	}

	/* 
	 * read/write commands (or seek) - make file adjustments
	 * If ISI header, call qtiodone to make adjustments,
	 * otherwise, do it here
	 */

#ifdef	DEBUG
	printf("qtdoerr, qt%d: read/write command\n",unit);
#endif	DEBUG

	if (dev->flaqs&QTF_ISI) {
		stat = qtiodone(softc,dev,cmd,filemark);
		return(stat);
	} else if (!softc->qt_rbuf.b_flags)
		dev->s_lastiow = 1;
	if (filemark) {
		dev->fileba = 0;
		dev->fileno++;
		dev->resid = bp->b_bcount;
		return(0);
	} else {
		if (dev->blksz)
			dev->fileba += (dev->count - dev->resid);
		else
			dev->fileba++;
	}
	if (dev->s_seeking) {
		dev->s_seeking = 0;
		return (0);
	}
	if (dev->blksz)
		dev->resid *= dev->blksz;
	if (dev->s_mode == 0)
		dev->s_lastiow = 1;
	return (0);
}

/************************************************************************
*
* qtinq - check to see that unit is a tape
*
*/
qtinq(unit)
	int unit;
{
	int cont, lun;
	register struct qt_dev *dev;
	register struct qt_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;

	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	dev = &qtdev[unit];
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;

	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qt_cmd;

	SET_CDB_0(cdb, CMD_INQUIRY, lun, 0,
	    sizeof(struct scsi_inq), 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_inq,
		sizeof(struct scsi_inq),0,0);
	bzero(&softc->qt_inq, sizeof(struct scsi_inq));
	softc->qt_inq.inq_pdtype = 0xff;
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_INQUIRY;
	if(qtissue(unit, cmd, 80000, 0))
		return (ERROR);
	if (softc->qt_inq.inq_pdtype != INQ_PDT_TAPE)
		return (ERROR);
	if (!strncmp(softc->qt_inq.inq_vendor,"EXABYTE ",8))
		dev->dtype = EXABYTE;
	if (!strncmp(softc->qt_inq.inq_vendor,"WANQTEK ",8))
		dev->dtype = WANQTEK;
	if (!strncmp(softc->qt_inq.inq_vendor,"ARCHIVE ",8))
		dev->dtype = ARCHIVE;
	return (OK);
}

/************************************************************************
*
* qtinfo - get unit info
*
*/

qtinfo(unit)
	int unit;
{
	int cont, lun;
	int count;
	register struct qt_dev *dev;
	register struct qt_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;

	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	dev = &qtdev[unit];
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;

	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(QTU_CONT(unit));
	cmd = softc->qt_cmd;

	/* Get block limits info */
	SET_CDB_0(cdb, CMD_BLKLIM, lun, 0, 0, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->blklim,
		sizeof(struct scsi_blklim),0,0);
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_BLKLIM;

	if(qtissue(unit, cmd, 80000, 1)) {
		printf("        qt%d at QS%d    ",unit,QTU_CONT(unit));
		printf("    *** BLKLIM failed\n");
		return(ERROR);
	}

	/* Get msense info */
	SET_CDB_0(cdb, CMD_MSENSE, lun, 0, sizeof(struct scsi_msen), 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_msen,
		sizeof(struct scsi_msen),0,0);
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_MSENSE;
	if(qtissue(unit, cmd, 80000, 1)) {
		printf("        qt%d at QS%d    ",unit,QTU_CONT(unit));
		printf("    *** MSENSE failed\n");
		return(ERROR);
	}

	dev->blksz = SCSI_HML(softc->qt_msen.msen_bd.bd_bl);
	softc->qt_msen.msen_len = 0;
	softc->qt_msen.msen_mtype = 0;
	softc->qt_msen.msen_wprot = 0;
	softc->qt_msen.msen_rsvd = 0;
	softc->qt_msen.msen_bdl = 8;
	softc->qt_msen.msen_bd.bd_density = 0;
	SCSI_HML_SET(softc->qt_msen.msen_bd.bd_nb, 0);
	SCSI_HML_SET(softc->qt_msen.msen_bd.bd_bl, dev->blksz);
	count = 12;
	if (dev->dtype == EXABYTE) {
		/* evenbytedisconnect & parityenable */
		softc->qt_msen.msen_pd.pd_code = 6;
		count = 14;
	}
	SET_CDB_0(cdb, CMD_MSELECT, lun, 0, count, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_msen, count,0,0);
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_MSELECT;
	if(qtissue(unit, cmd, 80000, 1)) {
		printf("        qt%d at QS%d    ",unit,QTU_CONT(unit));
		printf("    *** MSELECT failed\n");
	}

	printf("        qt%d  at QS%d  ",unit,QTU_CONT(unit));
	printf("%.8s %.16s %.4s SCSI tape drive\n",
		softc->qt_inq.inq_vendor, softc->qt_inq.inq_product, 	
		softc->qt_inq.inq_rev);
	printf("                       blocklimits: %d to %d;\n",
		softc->blklim.blklim_min, softc->blklim.blklim_max);
	if (dev->blksz)
		printf("                       fixed blocksize %d;\n",
			dev->blksz);
	else
		printf("                       variable blocksize;\n");
	printf("                       density code %d\n",
		softc->qt_msen.msen_bd.bd_density);

	dev->s_created = TRUE;
	return(OK);
}

/************************************************************************
*
* qtOpen - open a device
*
*/

qtOpen(dev, name, mode)
	struct qt_dev *dev;
	char 		*name;
	int		mode;
{
	int unit, cont, lun;
	register struct qt_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;
	register struct buf *bp;

	unit = QT_UNIT(dev->dev);
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;

	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qt_cmd;

	semTake(&qs_sems[cont]);

#ifdef	DEBUG
	printf("qtOpen, dev=0x%x, unit=0x%x\n",dev,unit);
#endif	DEBUG

	/* Check error conditions */
	if(unit >= Nqt || !dev->s_created) {
		logMsg("ERROR qt%d open failed: does not exist\n",unit);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	if(dev->s_open) {
		logMsg("ERROR qt%d open failed: already open by task %d\n",
			unit, dev->task);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}

	/* Init dev thinqs */
	dev->s_open = TRUE;
	dev->s_notraw = FALSE;
	dev->s_swmd = dev->flaqs & QTF_SWAB;
	bp = &softc->qt_rbuf;
	bp->b_dev = dev->dev;
	dev->flaqs = QTF_ISI | QTF_REWIND | QTF_SWAB;

	if((dev->s_notraw && (dev->flaqs & QTF_ISI || dev->blksz != 512)) ||
	   (dev->flaqs & QTF_ISI && dev->blksz != 512)) {
		logMsg("ERROR qt%d open failed: illegal block size %d\n",
			unit, dev->blksz);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}

	SET_CDB_0(cdb, CMD_MSENSE, lun, 0, sizeof(struct scsi_msen), 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_msen, 
		sizeof(struct scsi_msen),0,0);
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_MSENSE;
	if(qtissue(unit, cmd, 80000, 1)) {
		logMsg("ERROR qt%d open failed: MSENSE failed\n",unit);
		dev->s_open = 0;
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	if(((mode == WRITE) || (mode & WRITE)) && (softc->qt_msen.msen_wprot)) { 
		logMsg("ERROR qt%d open failed: write protected \n",unit);
		dev->s_open = 0;
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	dev->task = taskIdSelf();
	dev->s_mode = mode;
	dev->s_lastiow = 0;
	semGive(&qs_sems[cont]);
	return((int) dev);
}

/************************************************************************
*
* qtClose - close the device
*
* If tape was open for writing or last operation was a write, then
* write two EOF's and backspace over the last one. Unless no-rewind
* ioctl has been issued, rewind the tape.
*
*/

qtClose(dev)
	struct qt_dev *dev;
{
	int unit, cont, lun;
	register struct qt_softc *softc;
	register struct scsi_cmnd *cmd;
	register union scsi_cdb *cdb;
	register struct buf *bp;

	unit = QT_UNIT(dev->dev);
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;

	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qt_cmd;
	bp = &softc->qt_rbuf;

	semTake(&qs_sems[cont]);

#ifdef	DEBUG
	printf("qtClose, dev=0x%x, unit=0x%x\n",dev,unit);
#endif	DEBUG

	if(!dev->s_open) {
		logMsg("ERROR qt%d close failed: unit not opened\n",unit);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	if(((dev->s_mode == WRITE) || (dev->s_mode == UPDATE)) && dev->s_lastiow) {
#ifdef	DEBUGF
		printf("qtClose, writing FILEMARK\n");
#endif	DEBUGF
		dev->fileba = 0;
		dev->fileno++;
		SET_CDB_0(cdb, CMD_FILEMARK, lun, 0, 1, 0, 0); 
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		sprintf(cmd->dbgmsg,"CMD_FILEMARK");
		dev->s_direct = TRUE;
		dev->d_cmd = CMD_FILEMARK;
		if(qtissue(unit, cmd, 80000, 1))
			logMsg("ERROR qt%d FILEMARK failed\n",unit); 
	}
	if((dev->flaqs & QTF_REWIND) && (dev->fileno || dev->fileba)) {
		dev->fileno = 0;
		dev->fileba = 0;
		SET_CDB_0(cdb, CMD_REZERO, lun, 0, 0, 0, 0); 
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		sprintf(cmd->dbgmsg,"CMD_REZERO");
		dev->s_direct = TRUE;
		dev->d_cmd = CMD_REZERO;
		if(qtissue(unit, cmd, 800000, 1)) 
			logMsg("ERROR qt%d REWIND failed\n",unit); 
	}

#ifdef	DEBUGF
	printf("after close: cnt=%d bcnt=%d resid=%d lastio=%d fno=%d fba=%d blksz=%d\n",
		dev->count,bp->b_bcount,dev->resid,dev->s_lastiow,dev->fileno,
		dev->fileba,dev->blksz);
#endif	DEBUGF

	dev->s_open = 0;
	semGive(&qs_sems[cont]);
	return(OK);
}

/************************************************************************
*
* qtRead
*
*/

qtRead(dev, buffer, nbytes)
	struct qt_dev *dev;
	char		*buffer;
	int		nbytes;
{
	if(qtstrategy(dev, buffer, nbytes, B_READ) == ERROR)
		return(ERROR);
#ifdef	DEBUG
	printf("qtRead, read=%d, fileno=%d\n",
		nbytes - dev->resid, dev->fileno);
#endif	DEBUG
	return(nbytes - dev->resid);
}

/************************************************************************
*
* qtWrite
*
*/

qtWrite(dev, buffer, nbytes)
	struct qt_dev *dev;
	char		*buffer;
	int		nbytes;
{

	if(qtstrategy(dev, buffer, nbytes, B_WRITE) == ERROR)
		return(ERROR);
#ifdef	DEBUG
	printf("qtWrite, written=%d\n",nbytes - dev->resid);
#endif	DEBUG
	return(nbytes - dev->resid);
}


/************************************************************************
*
* qtstrategy - perform I/O
*
*/

qtstrategy(dev, buffer, nbytes, flag)
	register struct qt_dev *dev;
	char		*buffer;
	int		nbytes;
{
	int unit, cont, lun;
	u_int blocksize, tmpcnt;
	register struct qt_softc *softc;
	register union scsi_cdb *cdb;
	register struct scsi_cmnd *cmd;
	register struct buf *bp;

	unit = QT_UNIT(dev->dev);
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;
	bp = &softc->qt_rbuf;

	semTake(&qs_sems[cont]);

#ifdef	DEBUG
	printf("qtstrategy, dev=0x%x, unit=0x%x, cmd=%s\n",
		dev, unit, (flag == B_READ) ? "READ":"WRITE");
#endif	DEBUG

	if(!dev->s_open) {
		logMsg("ERROR qt%d %s failed: device not open\n",
			unit, (flag == B_READ) ? "READ" : "WRITE");
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	if((dev->s_mode != UPDATE) && ((dev->s_mode == WRITE) && (flag == B_READ))){
		logMsg("ERROR qt%d READ failed: device not open for reading\n",
			unit);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}
	if((dev->s_mode != UPDATE) && ((dev->s_mode == READ) && (flag == B_WRITE))){
		logMsg("ERROR qt%d WRITE failed: device not open for writing\n",
			unit);
		semGive(&qs_sems[cont]);
		return(ERROR);
	}

	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(QTU_CONT(unit));
	cmd = softc->qt_cmd;

	dev->s_lastiow = 0;
	bp->b_flags = flag;
	if(flag == B_READ)
		bp->b_addr = (u_int) qtiobuf;
	else
		bp->b_addr = (u_int) buffer;
	bp->b_bcount = min(nbytes,QT_MAXCNT);
	if(bp->b_bcount > QT_MAXCNT) {
		dev->resid = bp->b_bcount;
		logMsg("ERROR qt%d %s failed: count %d too high \n",
			unit,(bp->b_flags & B_READ) ? "READ" : "WRITE",
			bp->b_bcount);
		semGive(&qs_sems[cont]);
		bp->b_flags |= B_ERROR;
		return(ERROR);
	}

	/* If notraw, do seek to correct block size if necessary */
	if(dev->s_notraw && (dev->count = bp->b_blkno - dev->fileba)) {
		dev->s_seeking = 1;
		SET_CDB_0(cdb, CMD_SPACE, lun, (dev->count >> 8)&0xFFFF, 
			(dev->count&0xFF), 0, 0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		dev->s_direct = TRUE;
		dev->d_cmd = CMD_SPACE;
		dev->s_space_type = FALSE; 	/* space records */
		sprintf(cmd->dbgmsg,"CMD_SPACE");

#ifdef	DEBUGF
		printcdb(cdb,6);
#endif	DEBUGF

		if(qtissue(unit, cmd, 80000, 1)) {
			logMsg("ERROR qt%d CMD_SPACE failed\n", unit);
			semGive(&qs_sems[cont]);
			bp->b_flags |= B_ERROR;
			return(ERROR);
		}
	}

	/* Get count */
	dev->roundeven = bp->b_bcount + (bp->b_bcount & 1);
	if(!(blocksize = dev->blksz))
		blocksize++;
	dev->count = ((dev->roundeven + blocksize - 1) / blocksize);

#ifdef	DEBUGF
	printf("before hdr: cnt=%d bcnt=%d resid=%d lastio=%d fno=%d fba=%d blksz=%d\n",
		dev->count,bp->b_bcount,dev->resid,dev->s_lastiow,dev->fileno,
		dev->fileba,dev->blksz);
#endif	DEBUGF

	/* If ISI format, do ISI header */
	if(dev->flaqs & QTF_ISI) {
		if(qtisiheader(softc,dev,cmd,lun)) {
		    logMsg("ERROR qt%d %s failed: error %s header\n",
			unit,(bp->b_flags & B_READ) ? "READ" : "WRITE",
			(bp->b_flags & B_READ) ? "reading" : "writing");
		    semGive(&qs_sems[cont]);
		    bp->b_flags |= B_ERROR;
		    return(ERROR);
		}
	}

#ifdef	DEBUGF
	printf("before I/O: cnt=%d bcnt=%d resid=%d lastio=%d fno=%d fba=%d blksz=%d\n",
		dev->count,bp->b_bcount,dev->resid,dev->s_lastiow,dev->fileno,
		dev->fileba,dev->blksz);
#endif	DEBUGF

	/* Do I/O */
	tmpcnt = dev->count * dev->blksz;
	SET_CDB_0(cdb, (flag == B_READ) ? CMD_READ : CMD_WRITE, 
		lun, (dev->count >> 8)|(dev->blksz?0x10000:0), 
		(dev->count&0xFF), 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,bp->b_addr,tmpcnt,0,0);
	sprintf(cmd->dbgmsg,"%s",(flag == B_READ) ? "CMD_READ" : "CMD_WRITE");

#ifdef	DEBUG
	printcdb(cdb,6);
#endif	DEBUG

	if(qtissue(unit, cmd, 80000, 1)) {
		logMsg("ERROR qt%d %s failed\n", 
			unit,(flag == B_READ) ? "READ" : "WRITE");
		semGive(&qs_sems[cont]);
		bp->b_flags |= B_ERROR;
		return(ERROR);
	}

#ifdef	DEBUGF
	printf("after I/O: cnt=%d bcnt=%d resid=%d lastio=%d fno=%d fba=%d blksz=%d\n",
		dev->count,bp->b_bcount,dev->resid,dev->s_lastiow,dev->fileno,
		dev->fileba,dev->blksz);
#endif	DEBUGF

	if(flag == B_READ)
		bcopy(bp->b_addr,buffer, nbytes);
		
	semGive(&qs_sems[cont]);
	bp->b_flags |= ~B_ERROR;
	return(OK);
}

/************************************************************************
*
* qtIoctl
*
*/

STATUS qtIoctl(dev, request, arg)
	struct qt_dev 	*dev;
	int		request;
	u_long		arg;
{
	int 				unit, cont, lun;
	register struct qt_softc 	*softc;
	register union scsi_cdb 	*cdb;
	register struct scsi_cmnd 	*cmd;
	struct mtget			*mtget;
	int				fmks,count;
	struct mtop			*mtop;

	unit = QT_UNIT(dev->dev);
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	softc = &qtsoftc[cont];
	cdb = &softc->qt_cdb;

	semTake(&qs_sems[cont]);
	softc->qt_cmd = (struct scsi_cmnd *) qsgetcmd(cont);
	cmd = softc->qt_cmd;
	dev->s_direct = TRUE;
	mtop = (struct mtop *)arg;
	/*
	count = arg;
	*/

	switch (request) {
	    case MTIOCTOP:
		switch (mtop->mt_op) {
		case MTFSF: case MTBSF:	/* Forward/Backward space file */
		case MTFSR: case MTBSR:	/* Forward/Backward space record */
		    if(qtSpace(dev,mtop->mt_op,mtop->mt_count,cmd,cdb)) {
			    semGive(&qs_sems[cont]);
			    return (ERROR);
		    }
		    semGive(&qs_sems[cont]);
		    return(dev->resid ? ERROR : OK);

		case MTREW:		/* Rewind */
		    SET_CDB_0(cdb,CMD_REZERO,lun,0,0,0,0);
		    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		sprintf(cmd->dbgmsg,"ioctl request%d, CMD_REZERO",mtop->mt_op);
		    dev->d_cmd = CMD_REZERO;
		    break;

		case MTNOREWIND:
		    if(mtop->mt_count)
			    dev->flaqs &= ~QTF_REWIND;
		    else
			    dev->flaqs |= QTF_REWIND;
		    semGive(&qs_sems[cont]);
		    return (OK);

		case MTWEOF:	/* Write EOF record */
		    SET_CDB_0(cdb, CMD_FILEMARK, lun, 0, 1, 0, 0);
		    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		sprintf(cmd->dbgmsg,"ioctl request%d, CMD_FILEMARK",mtop->mt_op);
		    dev->d_cmd = CMD_FILEMARK;
		    break;

		case MTERASE:	/* ISI: Erase entire tape */
		    SET_CDB_0(cdb,CMD_ERASE,lun,0x10000,0,0,0);
		    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		    sprintf(cmd->dbgmsg,"ioctl request%d, CMD_ERASE",mtop->mt_op);
		    dev->d_cmd = CMD_ERASE;
		    break;

		case MTRST:		/* ISI: Reset */
		    qtreset(unit);
		    semGive(&qs_sems[cont]);
		    return (OK);

		case MTNOP:		/* No operation, set status */
		    semGive(&qs_sems[cont]);
		    return (OK);

		case MTOFFL: 	/* Rewind and put offline */
		case MTRET: 	/* ISI: retension tape */
		    switch (mtop->mt_op) {
			case MTOFFL:
			    count = 0;
			    break;
			case MTRET:
			    count = 3;
			    break;
		    }
		    SET_CDB_0(cdb,CMD_STARTSTOP,lun,0,count,0,0);
		    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		    sprintf(cmd->dbgmsg,"ioctl request%d, CMD_STARTSTOP",mtop->mt_op);
		    dev->d_cmd = CMD_STARTSTOP;
		    break;

		case GTPREVENT:	/* Prevent/Allow tape removal */
		    SET_CDB_0(cdb,CMD_PREVENT,lun,0, mtop->mt_count,0,0);
		    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		    sprintf(cmd->dbgmsg,"ioctl request%d, CMD_PREVENT",mtop->mt_op);
		    dev->d_cmd = CMD_PREVENT;
		    break;

		case GTSETBKSZ:	/* Set tape block size */
		    if ((mtop->mt_count < softc->blklim.blklim_min) ||
			    (mtop->mt_count > softc->blklim.blklim_max)) {
				    logMsg("ERROR qt%d: illegal block size %d\n",
					    unit,mtop->mt_count);
				    semGive(&qs_sems[cont]);
				    return (ERROR);
		    }
		    dev->blksz = mtop->mt_count;
		    qtreinit(softc, dev, cmd, cdb);
		    semGive(&qs_sems[cont]);
		    return (0);

		case GTISI:		/* Set/Clear ISI header */
		    if(mtop->mt_count)
			    dev->flaqs |= QTF_ISI;
		    else
			    dev->flaqs &= ~QTF_ISI;
		    semGive(&qs_sems[cont]);
		    return(OK);

		default:
		    logMsg("ERROR qt%d: illegal ioctl arg 0x%x \n",unit,mtop->mt_op);
		    semGive(&qs_sems[cont]);
		    return (ERROR);
		}
		break;

	    case MTIOCGET:	/* Get tape status */
		mtget = (struct mtget *)arg;
		mtget->mt_dsreg = 0;
		mtget->mt_erreg = dev->mt_erreg;
		mtget->mt_resid = dev->resid;
		mtget->mt_fileno = dev->fileno;
		mtget->mt_blkno = dev->fileba;
		semGive(&qs_sems[cont]);
		return(0);
		break;
	    
	    case MTIOCIEOT :
	    case MTIOCEEOT :
	    default :
	       logMsg("ERROR qt%d: illegal ioctl request 0x%x \n",unit,request);
		semGive(&qs_sems[cont]);
		return (ERROR);
		break;

	}
	if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
	    logMsg("ERROR qt%d: ioctl request 0x%x failed\n",unit,mtop->mt_op);
#endif DEBUG
	    semGive(&qs_sems[cont]);
	    return(ERROR);
	}
	semGive(&qs_sems[cont]);
	return (OK);
}

qtSpace(dev,request,count,cmd,cdb)
	struct qt_dev 			*dev;
	int				request,count;
	register struct scsi_cmnd 	*cmd;
	register union scsi_cdb 	*cdb;
{
	int				fmks;
	int				unit,cont,lun;
	register struct qt_softc 	*softc;
	register union scsi_sns 	*sns;

#ifdef DEBUG
	logMsg ("qtSpace: req=%d, count=%d\n", request, count);
#endif DEBUG
	unit = QT_UNIT(dev->dev);
	cont = QTU_CONT(unit);
	lun = QTU_LUN(unit);
	softc = &qtsoftc[cont];
	sns = &softc->qt_sense;
	dev->count = count;
	fmks = 0;
	dev->s_blk_check = FALSE;
	dev->s_adjust = FALSE;

	switch (request) {
	    case MTBSF:
		dev->count = -count;
		count = -count;
		dev->s_space_type = TRUE;
		/* fall through */
	    case MTFSF:
		fmks = 0x10000;
		dev->s_space_type = TRUE;
		break;
	    case MTBSR:
		dev->count = -count;
		count = -count;
		dev->s_space_type = FALSE;
		/* fall through */
	    case MTFSR:
		dev->s_space_type = FALSE;
		break;
	}
	if (dev->dtype == WANQTEK && count < 0) {
		/* WANG doesn't support reverse! */
		logMsg("ERROR qt%d: WANG doesn't support reverse\n",
			unit);
		return(ERROR);
	}
#ifdef	DEBUGF
	printf("qtSpace, lba=0x%x, count=0x%x\n",
		(((count>>8)&0xffff)|fmks),(count&0xff));
#endif	DEBUGF
	SET_CDB_0(cdb,CMD_SPACE,lun,
		(((count>>8)&0xffff)|fmks),(count&0xff),0,0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
	sprintf(cmd->dbgmsg,"ioctl request%d, CMD_SPACE",request);
	dev->d_cmd = CMD_SPACE;
	dev->s_direct = TRUE;
	if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
		logMsg("ERROR qt%d: ioctl request 0x%x failed\n",unit,request);
#endif DEBUG
		dev->s_blk_check = FALSE;
		return(ERROR);
	}

	/* if the request was backspace file, and we didn't hit
	 * the beginning of the tape, and there was no error,
	 * then we have to FSR past the next tapemark to point
	 * to the correct file.
	 */
	if ((request == MTBSF) && !sns->sns_7.sns_7_eom) {
	    dev->s_space_type = FALSE;
	    dev->count = count = 1;
	    fmks = 0x10000;
	    SET_CDB_0(cdb,CMD_SPACE,lun,
		    (((count>>8)&0xffff)|fmks),(count&0xff),0,0);
	    SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		    SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
	    sprintf(cmd->dbgmsg,"ioctl request%d, CMD_SPACE",request);
	    dev->d_cmd = CMD_SPACE;
	    dev->s_direct = TRUE;
	    if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
		logMsg("ERROR qt%d: ioctl request 0x%x failed\n",unit,request);
#endif DEBUG
		    dev->s_blk_check = FALSE;
		    return(ERROR);
	    }
	    dev->s_adjust = FALSE;
	    dev->s_blk_check = FALSE;
	    return(OK);
	}

	/* 
	 * The following is sort of a Kludge. The Archive Viper tape drive
	 * won't let you write over the logical end of tape mark until it's
	 * read first. So if you are spacing forward to the logical end
	 * of tape and then wish to write, you must read the EOT and then
	 * you can write. The following spaces forward 1 record. If we
	 * get a blank check indicating EOT, then we're at the end. If
	 * we don't get a blank check, then we're not at the end and we
	 * back up over that record, so that we don't screw up the user.
	 */

	/*
	if((dev->s_mode != READ) && ((request == MTFSF) || (request == MTFSR))
	    	    && (dev->flaqs & QTF_ISI)) {
		    */
	if ((request == MTFSF) && (dev->s_mode != READ)) {
		fmks = 0;
		count = 1;
		dev->s_space_type = FALSE;
		dev->s_adjust = TRUE;
#ifdef	DEBUGF
		printf("qtSpace, lba=0x%x, count=0x%x\n",
			(((count>>8)&0xffff)|fmks),(count&0xff));
#endif	DEBUGF
		SET_CDB_0(cdb,CMD_SPACE,lun,
			((count>>8)&0xffff)|fmks,count&0xff,0,0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
		sprintf(cmd->dbgmsg,"CMD_SPACE forward 1 record");
		dev->d_cmd = CMD_SPACE;
		dev->s_direct = TRUE;
		if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
			logMsg("ERROR qt%d: ioctl request 0x%x failed\n",unit,request);
#endif DEBUG
			dev->s_blk_check = FALSE;
			return(ERROR);
		}
		if(dev->s_blk_check == FALSE) { 

			/* back up 1 record */
			count = ~1;
#ifdef	DEBUGF
			printf("qtSpace, lba=0x%x, count=0x%x\n",
				(((count>>8)&0xffff)|fmks),(count&0xff));
#endif	DEBUGF
			SET_CDB_0(cdb,CMD_SPACE,lun,
				(((count>>8)&0xffff)|fmks),(count&0xff),0,0);
			SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
				SFN_NOT_FINISHED,LEN_STATUS,
				&softc->qt_buf,1,0,0);
			sprintf(cmd->dbgmsg,"CMD_SPACE backward 1 record");
			dev->d_cmd = CMD_SPACE;
			dev->s_direct = TRUE;
			if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
				logMsg("ERROR qt%d: ioctl request 0x%x failed\n"
					,unit,request);
#endif DEBUG
				dev->s_blk_check = FALSE;
				return(ERROR);
			}

			/* go forward over the filemark */
			count = 1;
			dev->s_space_type = FALSE;
#ifdef	DEBUGF 
			printf("qtSpace, lba=0x%x, count=0x%x\n",
				(((count>>8)&0xffff)|fmks),(count&0xff));
#endif	DEBUGF
			SET_CDB_0(cdb,CMD_SPACE,lun,
				((count>>8)&0xffff)|fmks,count&0xff,0,0);
			SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
				SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf,1,0,0);
			sprintf(cmd->dbgmsg,"CMD_SPACE forward 1 record");
			dev->d_cmd = CMD_SPACE;
			dev->s_direct = TRUE;
			if(qtissue(unit, cmd, 800000, 1)) {
#ifdef DEBUG
				logMsg("ERROR qt%d: ioctl request 0x%x failed\n",unit,request);
#endif DEBUG
				dev->s_blk_check = FALSE;
				return(ERROR);
			}
		}
	}
	dev->s_adjust = FALSE;
	dev->s_blk_check = FALSE;
	return(OK);
}

/************************************************************************
*
* qtisiheader - set up read or write of ISI header
*
*/

qtisiheader(softc,dev,cmd,lun)
	register struct qt_softc *softc;
	register struct qt_dev	*dev;
	register struct scsi_cmnd	*cmd;
{
	register union ihead	*ih;
	register union scsi_cdb *cdb;
	register struct buf *bp;

	ih = &softc->qt_ihead;
	cdb = &softc->qt_cdb;
	bp = &softc->qt_rbuf;

	if (bp->b_flags & B_READ) {
		ih->qt_ih.ih_id = 0;
		SET_CDB_0(cdb, CMD_READ, lun, 0x10000, 1, 0, 0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_ihead,
			sizeof(union ihead),0,0);
		sprintf(cmd->dbgmsg,"isiheader CMD_READ");
		if(qtissue(QT_UNIT(dev->dev), cmd, 80000, 1))
			return(ERROR);
#ifdef	DEBUGF
		printf("read isi hdr: fno=%d fba=%d blksz=%d count=%d\n",
			ih->qt_ih.ih_fileno, ih->qt_ih.ih_fileba,
			ih->qt_ih.ih_blksz, ih->qt_ih.ih_count);
#endif	DEBUGF
			
	} else {
		ih->qt_ih.ih_id = 0xA1FE;
		ih->qt_ih.ih_fileno = dev->fileno + 1;
		ih->qt_ih.ih_fileba = dev->fileba + 1;
		ih->qt_ih.ih_blksz = bp->b_bcount;
		ih->qt_ih.ih_count = dev->count;
		SET_CDB_0(cdb, CMD_WRITE, lun, 0x10000, 1, 0, 0);
		SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
			SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_ihead,
			sizeof(union ihead),0,0);
		sprintf(cmd->dbgmsg,"isiheader CMD_WRITE");
#ifdef	DEBUGF
		printcdb(cdb,6);
#endif	DEBUGF
		if(qtissue(QT_UNIT(dev->dev), cmd, 80000, 1))
			return(ERROR);
#ifdef	DEBUGF
		printf("wrote isi hdr: fno=%d fba=%d blksz=%d count=%d\n",
			ih->qt_ih.ih_fileno, ih->qt_ih.ih_fileba,
			ih->qt_ih.ih_blksz, ih->qt_ih.ih_count);
#endif	DEBUGF
	}
	return(OK);
}

/************************************************************************
*
* qtiodone - either header or real io complete, process file stuff
*
*/
qtiodone(softc,dev,cmd,filemark)
	register struct qt_dev		*dev;
	register struct qt_softc	*softc;
	register struct scsi_cmnd 	*cmd;
{
	register union ihead		*ih;
	register struct buf 		*bp;

	bp = &softc->qt_rbuf;
 	ih = &softc->qt_ihead;

	/* Header io already done, process the real io */
	if (dev->s_hdone) {
		dev->s_hdone = 0;
		dev->resid = bp->b_bcount - ih->qt_ih.ih_blksz;
		dev->fileba++;
		if (filemark) {
			dev->fileba = 0;
			dev->fileno++;
		}
		if ((bp->b_flags & B_READ) == 0)
			dev->s_lastiow = 1;
	/* Header io complete */
	} else {
		if (bp->b_flags & B_READ) {
			if (filemark) {
				dev->resid = bp->b_bcount;
				dev->fileba = 0;
				dev->fileno++;
				return(0);
			}
			if (ih->qt_ih.ih_id != 0xA1FE) {
				logMsg("ERROR qt%d: error reading header\n",
					QT_UNIT(dev->dev));
				return(ERROR);
			}
			dev->fileno = ih->qt_ih.ih_fileno;
		}
		if ((bp->b_flags & B_READ) == 0)
			dev->s_lastiow = 1;
		dev->s_hdone = 1;
	}
	return(0);
}

qtreset()
{
}

#define QT_NO_STEPS	2
qtreinit(softc, dev, cmd, cdb)
	register struct qt_dev *dev;
	register struct qt_softc *softc;
	register struct scsi_cmnd *cmd;
	register struct scsi_cdb *cdb;
{
	register int	lun, unit, count;

	unit = QT_UNIT(dev->dev);
	lun = QTU_LUN(unit);

	/* Rewind */
	SET_CDB_0(cdb, CMD_REZERO, lun, 0, 0, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_buf, 1, 0,0);
	sprintf(cmd->dbgmsg,"qtreinit CMD_REZERO\n");
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_REZERO;
	qtissue(unit, cmd, 800000, 0);

	/* Set mode select */
	softc->qt_msen.msen_len = 0;
	softc->qt_msen.msen_mtype = 0;
	softc->qt_msen.msen_wprot = 0;
	softc->qt_msen.msen_rsvd = 0;
	softc->qt_msen.msen_bdl = 8;
	softc->qt_msen.msen_bd.bd_density = 0;
	SCSI_HML_SET(softc->qt_msen.msen_bd.bd_nb, 0);
	SCSI_HML_SET(softc->qt_msen.msen_bd.bd_bl, dev->blksz);
	count = 12;
	if (dev->dtype == EXABYTE) {
		/* evenbytedisconnect & parityenable */
		softc->qt_msen.msen_pd.pd_code = 6;
		count = 14;
	}
	SET_CDB_0(cdb, CMD_MSELECT, lun, 0, count, 0, 0);
	SET_CMD(cmd,0,cdb,sizeof(struct cdb_0),cmd->qs_status,
		SFN_NOT_FINISHED,LEN_STATUS,&softc->qt_msen, count,0,0);
	sprintf(cmd->dbgmsg,"qtreinit CMD_MSELECT\n");
	dev->s_direct = TRUE;
	dev->d_cmd = CMD_MSELECT;
	if(qtissue(unit, cmd, 80000, 0)) {
		printf("ERROR qt%d: MSELECT FAILED",unit);
		return(ERROR);
	}
	return(OK);
}

#ifdef DEBUG

qtio (fd, req, arg)
int fd, req, arg;
{
    struct mtop mtop;

    mtop.mt_op = req;
    mtop.mt_count = arg;

    ioctl (fd, MTIOCTOP, &mtop);
}
#endif DEBUG
