/* ptyDrv.c - pseudo terminal driver */

static char *copyright = "Copyright 1986-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
The pseudo terminal driver is used to provide a tty-like interface between
two processes; one process acts as a master, and simulates the 'hardware' side
of the driver (i.e. a USART), while the other process acts as the slave, which
is the normal application program that ordinarily talks to the driver.
It is used primarily in network applications.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, ptyDrv () to
initialize the driver, and ptyDevCreate () to create devices.

PTYDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    ptyDrv ()
.CE
This routine must be called before any reads, writes, or
ptyDevCreates.  Normally, it is called from usrRoot (2) in usrConfig.c.

CREATING PSEUDO-TERMINAL DEVICES
Before a pseudo-terminal can be used, it must be created.  This is done
with the ptyDevCreate call:

.CS
    STATUS ptyDevCreate (name, rdBufSize, wrtBufSize)
	char *name;		/* name of pseudo-terminal *
	int rdBufSize;		/* size of terminal read buffer *
	int wrtBufSize;		/* size of write buffer *
.CE

For instance, to create the device pair "/pty/0.M" and "/pty/0.S",
with read and write buffer sizes of 512 bytes, the proper call would be:
.CS
   ptyDevCreate ("/pty/0.", 512, 512);
.CE
When ptyDevCreate is called, two devices are created (the master
and slave device), one called nameM and the other called nameS,
which may then be opened by the master and slave processes.
Data written to the master
device may then be read on the slave device, and vice versa.  Ioctl calls
may be made either to the master or slave device, but keep in mind that they
are really one device (for aesthetic reasons, you should really only apply
them to the slave side).

IOCTL
The pty driver responds to all the same ioctl codes as a normal tty driver.
*/

#include "UniWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "wdLib.h"
#include "tyLib.h"
#include "strLib.h"

typedef struct		/* PSEUDO_DEV */
    {
    TY_DEV tyDev;
    DEV_HDR slaveDev;
    SEM_ID masterReadSemId;
    BOOL ateof;
    } PSEUDO_DEV;


/* local variables */

LOCAL int ptySlaveDrvNum;	/* driver number assigned to slave driver */
LOCAL int ptyMasterDrvNum;	/* driver number assigned to master drv */


/* forward declarations */

LOCAL int ptyMasterOpen ();
LOCAL int ptySlaveOpen ();
LOCAL int ptySlaveRead ();
LOCAL int ptyMasterRead ();
LOCAL int ptySlaveWrite ();
LOCAL int ptyMasterWrite ();
LOCAL STATUS ptySlaveClose ();
LOCAL STATUS ptyMasterClose ();
LOCAL STATUS ptySlaveIoctl ();
LOCAL STATUS ptyMasterIoctl ();
LOCAL VOID ptyMasterStartup ();

/*******************************************************************************
*
* ptyDrv - pseudo-terminal driver
*
* Called before any other routine in this module.
*/

STATUS ptyDrv ()

    {
    static BOOL done;	/* FALSE = not done, TRUE = done */
    static STATUS status;

    if (!done)
	{
	done = TRUE;

	ptySlaveDrvNum = iosDrvInstall (ptySlaveOpen, (FUNCPTR) NULL, 
					ptySlaveOpen, ptySlaveClose,
					ptySlaveRead, ptySlaveWrite, 
					ptySlaveIoctl);

	ptyMasterDrvNum = iosDrvInstall (ptyMasterOpen, (FUNCPTR) NULL, 
					ptyMasterOpen, ptyMasterClose,
					ptyMasterRead, ptyMasterWrite, 
					ptyMasterIoctl);

	status = (ptySlaveDrvNum != ERROR && ptyMasterDrvNum != ERROR) ? OK
								       : ERROR;
	}

    return (status);
    }
/*******************************************************************************
*
* ptyDevCreate - create a master/slave pair
*
* RETURNS: OK or ERROR if out of memory
*/

STATUS ptyDevCreate (name, rdBufSize, wrtBufSize)
    char *name;		/* name of pseudo-terminal */
    int rdBufSize;	/* size of terminal read buffer */
    int wrtBufSize;	/* size of write buffer */

    {
    STATUS status;
    char nameBuf [MAX_FILENAME_LENGTH];
    PSEUDO_DEV *pPseudoDev;
    
    if (ptySlaveDrvNum < 1 || ptyMasterDrvNum < 1)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    pPseudoDev = (PSEUDO_DEV *) malloc (sizeof (PSEUDO_DEV));

    if (pPseudoDev == NULL)
	return (ERROR);

    /* initialize device descriptor */

    if (tyDevInit ((TY_DEV_ID) pPseudoDev, rdBufSize, wrtBufSize,
		   ptyMasterStartup) != OK)
	{
	free ((char *)pPseudoDev);
	return (ERROR);
	}

    pPseudoDev->masterReadSemId = semCreate ();
    semGive (pPseudoDev->masterReadSemId);
    semTake (pPseudoDev->masterReadSemId);

    /* add Slave and Master devices */

    strcpy (nameBuf, name);
    strcat (nameBuf, "S");
    status = iosDevAdd (&pPseudoDev->slaveDev, nameBuf, ptySlaveDrvNum);

    if (status == OK)
	{
	strcpy (nameBuf, name);
	strcat (nameBuf, "M");
	status = iosDevAdd ((DEV_HDR *) pPseudoDev, nameBuf, ptyMasterDrvNum);
	}

    return (status);
    }
/*******************************************************************************
*
* ptyMasterOpen - open the Master side of a pseudo terminal 
*
* ARGSUSED1
*/

LOCAL int ptyMasterOpen (pPseudoDev, name, mode)
    PSEUDO_DEV *pPseudoDev;
    char *name;
    int mode;

    {
    return ((int) pPseudoDev);
    }
/*******************************************************************************
*
* ptySlaveOpen - open the Slave side of a pseudo terminal 
*
* ARGSUSED1
*/

LOCAL int ptySlaveOpen (pPseudoDev, name, mode)
    PSEUDO_DEV *pPseudoDev;
    char *name;
    int mode;

    {
    return ((int) pPseudoDev - (OFFSET(PSEUDO_DEV, slaveDev)));
    }
/*******************************************************************************
*
* ptySlaveClose -
*
* Closing the slave side should cause the master to read 0 bytes.
*/

LOCAL STATUS ptySlaveClose (pPseudoDev)
    PSEUDO_DEV *pPseudoDev;

    {
    pPseudoDev->ateof = FALSE;	/* indicate to master read of close */
    semGive (pPseudoDev->masterReadSemId);
    return (OK);
    }
/*******************************************************************************
*
* ptyMasterClose -
*
* Closing the master side will cause the slave's read to return ERROR.
*/

LOCAL STATUS ptyMasterClose (pPseudoDev)
    PSEUDO_DEV *pPseudoDev;

    {
    tyIoctl ((TY_DEV_ID) pPseudoDev, FIOCANCEL);
    return (OK);
    }
/*******************************************************************************
*
* ptySlaveRead - Slave read routine 
*
* The slave device simply calls tyRead to get data out of the ring buffer.
*
* RETURNS: whatever tyRead returns (number of bytes actually read)
*/

LOCAL int ptySlaveRead (pPseudoDev, buffer, maxbytes)
    PSEUDO_DEV *pPseudoDev;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pPseudoDev, buffer, maxbytes));
    }
/*******************************************************************************
*
* ptyMasterRead - Master read routine 
*
* The master read routine calls tyITx to empty the pseudo terminal's buffer.
*
* RETURNS: number of characters actually read
*/

LOCAL int ptyMasterRead (pPseudoDev, buffer, maxbytes)
    FAST PSEUDO_DEV *pPseudoDev;
    char *buffer;		/* where to return characters read */
    int maxbytes;

    {
    FAST int i;
    char ch;

    pPseudoDev->ateof = TRUE;

    for (i = 0; i == 0;)
	{
	while ((i < maxbytes) && (tyITx ((TY_DEV_ID) pPseudoDev, &ch) == OK))
	    buffer [i++] = ch;	

	if (!pPseudoDev->ateof)
	    break;

	if (i == 0)
	    semTake (pPseudoDev->masterReadSemId);
	}

    return (i);
    }
/*******************************************************************************
*
* ptySlaveWrite - pseudo terminal Slave write routine
*
* This routine simply calls tyWrite.
*
* RETURNS: whatever tyWrite returns (number of bytes actually written)
*/

LOCAL int ptySlaveWrite (pPseudoDev, buffer, nbytes)
    PSEUDO_DEV *pPseudoDev;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pPseudoDev, buffer, nbytes));
    }
/*******************************************************************************
*
* ptyMasterWrite - pseudo terminal Master write routine
*
* This routine calls tyIRd to put data in the pseudo terminals ring buffer.
*
* RETURNS: nbytes
*/

LOCAL int ptyMasterWrite (pPseudoDev, buffer, nbytes)
    PSEUDO_DEV *pPseudoDev;
    char *buffer;
    int nbytes;

    {
    FAST int i;

    for (i = 0; i < nbytes; i++)
	(void)tyIRd ((TY_DEV_ID) pPseudoDev, buffer [i]);

    return (i);
    }
/*******************************************************************************
*
* ptySlaveIoctl - special device control
*/

LOCAL STATUS ptySlaveIoctl (pPseudoDev, request, arg)
    PSEUDO_DEV *pPseudoDev;	/* device to control */
    int request;		/* request code */
    int arg;			/* some argument */

    {
    return (tyIoctl ((TY_DEV_ID) pPseudoDev, request, arg));
    }
/*******************************************************************************
*
* ptyMasterIoctl - special device control
*/

LOCAL STATUS ptyMasterIoctl (pPseudoDev, request, arg)
    PSEUDO_DEV *pPseudoDev;	/* device to control */
    int request;		/* request code */
    int arg;			/* some argument */

    {
    return (tyIoctl ((TY_DEV_ID) pPseudoDev, request, arg));
    }
/*******************************************************************************
*
* ptyMasterStartup - start up the Master read routine
*/

LOCAL VOID ptyMasterStartup (pPseudoDev)
    PSEUDO_DEV *pPseudoDev;

    {
    semGive (pPseudoDev->masterReadSemId);
    }
