/* rebootLib.c - reboot support library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
The rebootLib module allows for an "orderly" reboot.
The routine reboot may be called at any time to restart UniWorks.
It is called when the abort character (control-X) is typed from the shell.
Shutdown routines may be added with rebootHookAdd (2).
These are typically used to reset, or synchronize, hardware.
For example, netLib (1) adds a reboot hook to cause all network
interfaces to be reset.
Once the reboot hooks have been run, sysToMonitor (2) is
called to transfer control to the bootroms.

In the standard configuration there are three kinds of reboots.
A "warm boot" causes the system to go through the countdown sequence,
and automatically try to reboot UniWorks.
A "warm boot without autoboot" causes the system simply wait in the
bootrom monitor.
Lastly, a "cold boot" acts as if the chassis had been reset, and
autoboot will take place.

INCLUDE FILE: sysLib.h

SEE ALSO: sysLib (1), bootConfig (1)
*/

#include "UniWorks.h"
#include "memLib.h"

typedef struct
    {
    NODE node;
    FUNCPTR func;
    } REBOOT_HOOK;

LOCAL LIST rebootHookList;
LOCAL BOOL rebootHooksInitialized;

/*******************************************************************************
*
* reboot - reset network devices and call sysToMonitor
*
* This routine performs the shutdown sequence necessary before going back to
* the bootroms.  It calls a series of shutdown routines that have been added
* via rebootHookAdd (2).
*
* SEE ALSO: sysToMonitor (2), rebootHookAdd (2)
*/

VOID reboot (startType)
    int startType;	/* how the bootroms will reboot */

    {
    FAST REBOOT_HOOK *pHook;

    if (rebootHooksInitialized)
	{
	for (pHook = (REBOOT_HOOK *) lstFirst (&rebootHookList);
	     pHook != NULL;
	     pHook = (REBOOT_HOOK *) lstNext (&pHook->node))
	    {
	    (*pHook->func) (startType);
	    }
	}

    sysToMonitor (startType);
    }
/*******************************************************************************
*
* rebootHookAdd - add routine to be called at reboot 
*
* This routine adds the specified routine to a list of routines
* that get called when UniWorks is rebooted.  The routine should be
* declared as follows:
* .CS
*  VOID rebootHook (startType)
*  int startType;   /* startType is passed to all hooks *
* .CE
* Currently the only valid startTypes are BOOT_WARM_AUTOBOOT,
* and BOOT_WARM_NO_AUTOBOOT, and BOOT_COLD.
* Users may wish to add special facilities to the bootroms.
*
* RETURNS: OK, or ERROR if insufficient memory
*
* SEE ALSO: reboot (2)
*/

STATUS rebootHookAdd (rebootHook)
    FUNCPTR rebootHook;		/* routine to be at reboot */

    {
    FAST REBOOT_HOOK *pHook;

    /* initialize hook list if this is the first time */

    if (!rebootHooksInitialized)
	{
	lstInit (&rebootHookList);
	rebootHooksInitialized = TRUE;
	}

    /* allocate and initialize hook node */

    if ((pHook = (REBOOT_HOOK *) malloc (sizeof (REBOOT_HOOK))) == NULL)
	return (ERROR);

    pHook->func = rebootHook;

    /* add it to end of hook list */

    taskLock ();			/* disable task switching */
    lstAdd (&rebootHookList, &pHook->node);
    taskUnlock ();			/* re-enable task switching */

    return (OK);
    }
