/* intLib.c - interrupt subroutine library */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library provides routines to manipulate and connect to hardware
interrupts and exceptions.
Most importantly, any C language routine can be connected to any exception,
interrupt, or trap by calling the routine intConnect.
Interrupt vectors can be accessed directly by the routines
intSetVec and intGetVec.
On the 68010 and 68020, the vector base register can be accessed by the routines
intSetVecBase and intGetVecBase.
Tasks can lock and unlock interrupts by calling the routines
intLock and intUnlock.
The routines intCount and intContext can be used to determine whether
the CPU is running in an interrupt context or in a normal task context.

EXAMPLE

There are instances where it is desired to switch between one of
several routines for a particular interrupt.  The following
code fragment is one alternative.

.CS

    oldfunc = intGetVec (vector);
    newfunc = intHandlerCreate (routine, parameter);
    intSetVec (vector, newfunc);
    ...
    intSetVec (vector, oldfunc);    /* use original routine *
    ...
    intSetVec (vector, newfunc);    /* reconnect new routine *

.CE

*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "memLib.h"
#include "sysLib.h"

IMPORT intEnt ();	/* interrupt entrance stub */
IMPORT intExit ();	/* interrupt exit stub */

int intLockLevel = 7;	/* interrupt level to set to "lock-out" interrupts */

int intCnt = 0;		/* Counter to tell if we are at interrupt (non-task) */
			/* level.  Bumped and de-bumped by intALib */


#define	HI_WORD(w)		(short)(((int)(w) & 0xffff0000) >> 16)
#define	LO_WORD(w)		(short)((int)(w) & 0x0000ffff)

LOCAL USHORT intConnectCode []	=	/* intConnect stub */
    {
/*
*0x00  4EB9 kkkk kkkk	jsr	_intEnt  	  * tell kernel
*0x06  48E7 E0C0	movem.l	d0-d2/a0-a1,-(a7) * save regs
*0x0a  2F3C pppp pppp	move.l	#parameter,-(a7)  * push param
*0x10  4EB9 rrrr rrrr	jsr	routine		  * call "C"
*0x16  588F		addq.l  #4,a7             * pop param
*0x18  4CDF 0307	movem.l (a7)+,d0-d2/a0-a1 * restore regs
*0x1c  4EF9 kkkk kkkk	jmp     _intExit          * exit via kernel
*/
     0x4eb9, 0x0000, 0x0000,	/* _intEnt filled in at runtime */
     0x48E7, 0xE0C0,
     0x2F3C, 0x0000, 0x0000,	/* parameter filled in at runtime */
     0x4EB9, 0x0000, 0x0000,	/* routine to be called filled in at runtime */
     0x588f,
     0x4cdf, 0x0307,
     0x4ef9, 0x0000, 0x0000,	/* _intExit filled in at runtime */
    };


/* forward declarations */

FUNCPTR	intHandlerCreate ();
FUNCPTR	*intGetVecBase ();
FUNCPTR	intGetVec ();


/*******************************************************************************
*
* intConnect - connect C routine to hardware interrupt
*
* This routine connects the specified C routine to the specified
* interrupt vector.  When an interrupt occurs that vectors through
* the specified address, the routine will be called with the specified
* parameter.  The routine will be invoked in supervisor mode at interrupt
* level.  A proper "C" environment will have been established,
* the necessary registers saved, and the stack set up.
*
* The routine can be any normal "C" code, except that it must not
* invoke certain operating system functions.
*
* This routine simply calls intHandlerCreate(2) and intSetVec(2).
* It is the address of the handler returned by intHandlerCreate(2)
* that actually gets put in the interrupt vector.
*
* RETURNS:
*    OK or
*    ERROR if unable to build interrupt handler
*
* SEE ALSO: intHandlerCreate(2), intSetVec(2)
*/

STATUS intConnect (vector, routine, parameter)
    FUNCPTR *vector;		/* interrupt vector to attach to */
    FUNCPTR routine;		/* routine to be called */
    int parameter;		/* parameter to be passed to routine */

    {
    FUNCPTR intDrvRtn = intHandlerCreate (routine, parameter);

    if (intDrvRtn == NULL)
	return (ERROR);

    /* make vector point to synthesized code */

    intSetVec (vector, intDrvRtn);

    return (OK);
    }
/*******************************************************************************
*
* intHandlerCreate - construct interrupt handler for C routine
*
* This routine builds an interrupt handler around the specified C routine.
* This interrupt handler is then suitable for connecting to a
* specific vector address with intSetVec(2).
* The routine will be invoked in supervisor mode at interrupt level.
* A proper "C" environment will have been established,
* the necessary registers saved, and the stack set up.
*
* The routine can be any normal "C" code, except that it must not
* invoke certain operating system functions.
*
* IMPLEMENTATION:
* This routine builds an interrupt handler of the following form in
* allocated memory.
*
* .CS
*  0x00  4EB9 kkkk kkkk	jsr	_intEnt  	  * tell kernel
*  0x06  48E7 E0C0	movem.l	d0-d2/a0-a1,-(a7) * save regs
*  0x0a  2F3C pppp pppp	move.l	#parameter,-(a7)  * push param
*  0x10  4EB9 rrrr rrrr	jsr	routine		  * call "C"
*  0x16  588F		addq.l  #4,a7             * pop param
*  0x18  4CDF 0307	movem.l (a7)+,d0-d2/a0-a1 * restore regs
*  0x1c  4EF9 kkkk kkkk	jmp     _intExit          * exit via kernel
* .CE
*
* RETURNS:
*    pointer to new interrupt handler, or
*    NULL if out of memory
*/

FUNCPTR intHandlerCreate (routine, parameter)
    FUNCPTR routine;		/* routine to be called */
    int parameter;		/* parameter to be passed to routine */

    {
    FAST USHORT *pCode;		/* pointer to newly synthesized code */

    pCode = (USHORT *)malloc (sizeof (intConnectCode));

    if (pCode != NULL)
	{
	/* copy intConnectCode into new code area */

	bcopy ((char *)intConnectCode, (char *)pCode, sizeof (intConnectCode));

	/* set the addresses & instructions */

	pCode [1]  = HI_WORD (intEnt);
	pCode [2]  = LO_WORD (intEnt);
	pCode [6]  = HI_WORD (parameter);
	pCode [7]  = LO_WORD (parameter);
	pCode [9] = HI_WORD (routine);
	pCode [10] = LO_WORD (routine);
	pCode [15] = HI_WORD (intExit);
	pCode [16] = LO_WORD (intExit);
	}

    return ((FUNCPTR)pCode);
    }
/*******************************************************************************
*
* intContext - determine if we are in interrupt context or task context
*
* This routine returns true if and only if we are executing in
* interrupt context and NOT in a meaningful task context.
*/

BOOL intContext ()

    {
    return (intCnt > 0);
    }
/*******************************************************************************
*
* intCount - get current interrupt nesting depth
*
* This routine returns the number of interrupts that are currently nested.
*/

int intCount ()

    {
    return (intCnt);
    }
/*******************************************************************************
*
* intLock - lock out interrupts
*
* This routine is used to disable interrupts.  The interrupt level is
* set to the lock-out level (level 7 by default).  The routine returns
* the previous interrupt level, and this should be restored by a call
* to intUnlock(2).
*
* IMPORTANT CAVEAT
* VRTX does not recognize that interrupts are locked out if you are 
* executing from user state.  Therefore, if you make a VRTX call while locked
* out that can cause a queue scan (a semGive, for instance), VRTX
* may reschedule you out, and the task that comes in will run without
* interrupts locked out.
*
* To avoid this, you should either not make such VRTX calls while
* locked out, or lock out task switching by calling vxLock/vxUnlock in addition
* to intLock/intUnlock.
*
* EXAMPLE
*.CS
*    oldLevel = intLock ();
*    ... work with interrupts locked out ...
*    intUnlock (oldLevel);
*.CE
*
* RETURNS: Old interrupt level.
*
* SEE ALSO: intUnlock(2), vxLock(2)
*/

int intLock ()
    {
    return (intSet (intLockLevel));
    }
/*******************************************************************************
*
* intUnlock - cancel effect of intLock
*
* This routine is used to re-enable interrupts that have been disabled by
* intLock(2).  Use the level obtained from the preceding intLock(2) call.
*
* SEE ALSO: intLock (2)
*/

VOID intUnlock (level)
    int level;			/* level to which to restore interrupts */

    {
    intSet (level);
    }
/*******************************************************************************
*
* intSetVecBase - set the vector base address
*
* This routine sets the vector base address.  The CPU's vector base
* register is set to the specified value, and subsequent calls to 
* intGetVec/inVecSet will use this base address.
* The vector base address is initially 0, until changed by calls to this
* routine.
*
* NOTE:
* The 68000 does not have a vector base register
* so this routine is a NOP for 68000 systems.
*
* SEE ALSO: intGetVecBase (2), intGetVec (2), intSetVec (2)
*/

VOID intSetVecBase (baseAddr)
    FUNCPTR *baseAddr;	    /* new vector base address */

    {
    if (sysCpu == MC68000)
	return;
    intSetVBR (baseAddr);	/* set the actual vector base register */
    }
/*******************************************************************************
*
* intGetVecBase - get the vector base address
*
* This routine returns the current vector base address that has been set
* via the intSetVecBase(2) routine.
*
* RETURNS: current vector base address
*
* SEE ALSO: intSetVecBase(2)
*/

FUNCPTR *intGetVecBase ()

    {
    if (sysCpu == MC68000)
	return(0);
    return ((FUNCPTR *)intGetVBR());
    }
/******************************************************************************
*
* intSetVec - set CPU vector
*
* This routine sets the specified exception/interrupt vector to the
* specified address.  The vector is specified as an offset into the
* CPU's vector table.  This vector table starts, by default, at address
* 0, but, on 68010 and 68020 CPU's, may be set to start at any address
* via the intSetVecBase(2) routine.
*
* SEE ALSO: intSetVecBase (2), intGetVec (2)
*/

VOID intSetVec (vector, function)
    FUNCPTR *vector;		/* vector offset */
    FUNCPTR function;		/* address to place in vector */

    {
    FUNCPTR *newVector;

    /* vector is offset by the vector base address */

    newVector = (FUNCPTR *) ((int) vector + (int) intGetVecBase ());
    *newVector = function;
    }
/*******************************************************************************
*
* intGetVec - get vector
*
* This routine returns the current value of the specified
* exception/interrupt vector.  The vector is specified as an offset into the
* CPU's vector table.  This vector table starts, by default, at address
* 0, but, on 68010 and 68020 CPU's, may be set to start at any address
* via the intSetVecBase(2) routine.
*
* RETURNS: current value of specified vector
*
* SEE ALSO: intSetVec(2), intSetVecBase(2)
*/

FUNCPTR intGetVec (vector)
    FUNCPTR *vector;	/* vector offset */

    {
    /* vector is offset by vector base address */

    return (* (FUNCPTR *) ((int) vector + (int) intGetVecBase ()));
    }
