/*
 * hddump.c - dump various parts of the RT raw disk info. 
 *	ie: config record, boot record, minidisk table, badblock table.
 *
 *		Roger Florkowski (roger@ans.net)
 */

#include <stdio.h>
#include <sys/types.h>
#include <ctype.h>
#include "hdcompat.h"
#include "hdinfo.h"

int raw;
int fd;
extern char *optarg;

main(argc, argv)
int argc;
char *argv[];
{
	int ch;
	char buf[512];

        while ((ch = getopt(argc, argv, "f:bcmsBHr")) != EOF)
	    switch((char)ch) {
		case 'f': open_hd(optarg); break;
		case 'b': brec_dump(); break;
		case 'c': config_dump(); break;
		case 's': 
		    if (read(fd, buf, 512) != 512){
			    fprintf(stderr, "short read\n");
			    perror("skip");
			    exit(1);
		    }
		    break;
		case 'm': minidisk_dump(); break;
		case 'B': bbt_dump (); break;
		case 'H': hid_dump (); break;
		case 'r': raw++; break;
		case '?':
		default:
		    printf ("%s: -f <file> -bcmBH\n", argv[0]);
		    printf ("\t-b -- dump a boot record\n");
		    printf ("\t-c -- dump a config record\n");
		    printf ("\t-m -- dump minidisk entries\n");
		    printf ("\t-B -- dump bad block entries\n");
		    printf ("\t-B -- dump HIDDEN bad block entries\n");
		    exit (1);
	    }
}

open_hd(str)
char *str;
{
    if ((fd = open(str, 0, 0)) < 0) {
	perror (str);
	exit (1);
    }
}

minidisk_dump()
{
	char buf[512];
	struct miniheader *mhdr;
	struct minidisk *md;
	int i;

	/* assume the minidisk info fits in 1 sector */
	if (!raw) {
	    if (lseek (fd, MD_DIR_LSN * 512, SEEK_SET) < 0) {
		fprintf(stderr, "bad seek\n");
		perror("CONFIG record");
		exit(1);
	    }
	}
	if (read(fd, buf, 512) != 512){
		fprintf(stderr, "short read\n");
		perror("minidisk info");
		exit(1);
	}
	printf("\n");
	mhdr = (struct miniheader *)buf;
	printf("mini_number      %d (number of entries)\n", 
		mhdr->number);
	printf("mini_level       %d (revision level)\n", 
		mhdr->level);
	printf("mini_unused      %d (index of next unused entry)\n", 
		mhdr->unused);
	printf("mini_lastused    %d (index of most recently entry)\n", 
		mhdr->lastused);
	printf("mini_first       %d (first entry)\n", mhdr->first);
	printf("mini_last        %d (last entry)\n", mhdr->last);
	printf("mini_bad_block   %d (first bad block)\n", mhdr->bad_block);
	printf("mini_bad_size    %d (number of bad blocks)\n", mhdr->bad_size);

	md = (struct minidisk *) (buf + sizeof (struct miniheader) +
		sizeof (struct minidisk) * mhdr->first);
	for (i = 0; i < mhdr->number; i++) {
	    printf("\n");
	    printf("mini_previous    %d (link to previous entry)\n", 
		md->previous);
	    printf("mini_next        %d (link to next entry)\n", 
		md->next);
	    printf("mini_name        %c%c%c%c\n", 
			md->name[0],md->name[1],md->name[2],md->name[3]);
	    printf("mini_date        %d (date of creation)\n", 
		md->date);
	    printf("mini_iodn        %d\n", md->iodn);
	    printf("mini_blocksize   %d ", md->blocksize);
	    switch (md->blocksize) {
		case 0: printf ("(512)\n"); break;
		case 1: printf ("(1024)\n"); break;
		case 3: printf ("(2048)\n"); break;
		default: printf ("(unknown)\n"); break;
	    }
	    printf("mini_type        0x%x\n", md->type);
	    printf("mini_start       %d (start block of partition)\n", 
			md->start);
	    printf("mini_size        %d (size of partition)\n", 
			md->size);
		
	    md = (struct minidisk *) (buf + sizeof (struct miniheader) +
		    sizeof (struct minidisk) * md->next);
	}
}

config_dump()
{
	struct hdconfig chdr;
	char *s;

	if (!raw) {
	    if (lseek (fd, CONFIG_REC_LSN * 512, SEEK_SET) < 0) {
		fprintf(stderr, "bad seek\n");
		perror("CONFIG record");
		exit(1);
	    }
	}
	if (read(fd,&chdr,sizeof(struct hdconfig)) != sizeof(struct hdconfig)){
		fprintf(stderr, "short read\n");
		perror("config record");
		exit(1);
	}
	printf("\n");
	printf("conf_magic	 0x%x\n", chdr.conf_magic);
	printf("conf_sectorcount %d\n", chdr.conf_sectorcount);
	printf("conf_landing     %d\n", chdr.conf_landing);
	printf("conf_interleave  %d\n", chdr.conf_interleave);
	printf("conf_sectorsize	 %d (2 == 512)\n", chdr.conf_sectsize);
	printf("conf_lastcyl	 %d\n", chdr.conf_lastcyl);
	printf("conf_lasttrack	 %d\n", chdr.conf_lasttrack);
	printf("conf_lastsect	 %d\n", chdr.conf_lastsect);
	printf("conf_precomp     0x%x (0xff == noprecomp)\n", 
		chdr.conf_precomp);
	printf("conf_status  	 0x%x\n", chdr.conf_status);
	printf("conf_maxcyl  	 %d\n", chdr.conf_maxcyl);
	printf("conf_end_of_life %d (max # bad blocks)\n", 
		chdr.conf_end_of_life);
	printf("conf_size  	 %d (approx # MEG)\n", chdr.conf_size);
	printf("conf_mfr  	 0x%x\n", chdr.conf_mfr);
	printf("conf_adapter  	 0x%x ", chdr.conf_adapter);
	switch (chdr.conf_adapter) {
		case 0x00: printf ("(AT)\n"); break;
		case 0x01: printf ("(ESDI)\n"); break;
		case 0x02: printf ("(EESDI)\n"); break;
		default: printf ("(unknown)\n"); break;
	}
	printf("conf_srn     	 %d\n", chdr.conf_srn);
	printf("conf_label   	 0x%x\n", chdr.conf_label);
	printf("conf_hidmax  	 %d (max # hidden defects)\n",
			chdr.conf_hidmax);
#if 0
	printf("conf_name     	 %02x %02x %02x %02x %02x %02x %02x %02x\n", 
		chdr.conf_name[0], chdr.conf_name[1], 
		chdr.conf_name[2], chdr.conf_name[3],
		chdr.conf_name[4], chdr.conf_name[5], 
		chdr.conf_name[6], chdr.conf_name[7]);

#else
	printf ("conf_name	 ");
	s = chdr.conf_name;
	while (isprint (*s))
	    printf ("%c", *s++);
	printf ("\n");
#endif
}

brec_dump()
{
	struct boothdr bhdr;

	if (!raw) {
	    if (lseek (fd, IPL_REC_LSN * 512, SEEK_SET) < 0) {
		fprintf(stderr, "bad seek\n");
		perror("CONFIG record");
		exit(1);
	    }
	}
	if (read(fd, &bhdr, sizeof(struct boothdr)) != sizeof(struct boothdr)) {
		fprintf(stderr, "short read\n");
		perror("boot record");
		exit(1);
	}
	printf("\n");
	printf("boot_check	 %x\n", bhdr.boot_check);
	printf("boot_lastcyl	 %d\n", bhdr.boot_lastcyl);
	printf("boot_lasttrack	 %d\n", bhdr.boot_lasttrack);
	printf("boot_lastsect	 %d\n", bhdr.boot_lastsect);
	printf("boot_sectorsize	 %d\n", bhdr.boot_sectorsize);
	printf("boot_interleave	 %d\n", bhdr.boot_interleave);
	printf("boot_sectorcount %d\n", bhdr.boot_sectorcount);
	printf("boot_formatdate	 %x\n", bhdr.boot_formatdate);
	printf("boot_cyl	 %d (cyl # of boot program)\n", bhdr.boot_cyl);
	printf("boot_track	 %d (track # of boot program)\n", 
		bhdr.boot_track);
	printf("boot_sector	 %d (sector # of boot program)\n", 
		bhdr.boot_sector);
	printf("boot_length	 %d (length of boot program)\n", 
		bhdr.boot_length);
	printf("boot_entry	 %d (offset into boot program to start at)\n", 
		bhdr.boot_entry);
	printf("boot_vrmminidisk %x\n", bhdr.boot_vrmminidisk);
	printf("boot_llp	 %x\n", bhdr.boot_llp);
	printf("boot_vrmlength	 %x\n", bhdr.boot_vrmlength);
	printf("boot_ibma	 %02x %02x %02x %02x\n", bhdr.boot_ibma[0],
	  bhdr.boot_ibma[1], bhdr.boot_ibma[2], bhdr.boot_ibma[3]);
	printf("boot_reserved1	 %02x %02x %02x %02x\n", bhdr.boot_reserved1[0],
	  bhdr.boot_reserved1[1], bhdr.boot_reserved1[2], 
	  bhdr.boot_reserved1[3]);
	printf("boot_reserved2	 %02x %02x %02x\n", bhdr.boot_reserved2[0],
	  bhdr.boot_reserved2[1], bhdr.boot_reserved2[2]);
}

bbt_dump()
{
    bad_block_dir bbthdr, *bbtp;
    bad_dirent *bdp;
    char *s;
    int size, i;

    if (!raw) {
	if (lseek (fd, BAD_DIR_LSN * 512, SEEK_SET) < 0) {
	    fprintf(stderr, "bad seek\n");
	    perror("BBT record");
	    exit(1);
	}
    }
    if (read(fd, &bbthdr, sizeof(bad_block_dir)) != sizeof(bad_block_dir)) {
	fprintf(stderr, "short read\n");
	perror("BBT record");
	exit(1);
    }

    printf("\n");
    printf("BBT_id       ");
    s = bbthdr.id;
    while (isprint (*s))
	printf ("%c", *s++);
    printf ("\n");
    printf("num_bad_blks %d\n", bbthdr.total);

    if (strncmp (bbthdr.id, "DEFECT", 6)) {
	printf ("NO BAD BLOCK TABLE\n");
	return;
    }

    size = sizeof (bad_block_dir) + bbthdr.total * sizeof (bad_dirent);
    bbtp = (bad_block_dir *) malloc (size);
    if (lseek (fd, BAD_DIR_LSN * 512, SEEK_SET) < 0) {
	fprintf(stderr, "bad seek\n");
	perror("BBT record");
	exit(1);
    }
    if (read(fd, bbtp, size) != size) {
	fprintf(stderr, "short read\n");
	perror("BBT records");
	exit(1);
    }
    for (bdp = &bbtp -> entry[0], i = 0; i < bbtp -> total; i++, bdp ++) {
	printf ("reason         %x\n", bdp->reason);
	printf ("reason        	0x%x ", bdp->reason);
	switch (bdp->reason) {
	    case MANUFACT_DEF: printf ("(manufacturer)\n"); break;
	    case SURFACE_SCAN_DEF: printf ("(surface scan)\n"); break;
	    case SYSTEM_DEF: printf ("(system)\n"); break;
	    case MANUFACT_TEST_DEF: printf ("(manufacturing test)\n"); break;
	    default: printf ("(unknown)\n"); break;
	}
	printf ("bad_lsn        %d\n", bdp->bad_lsn);
	printf ("reloc_lsn      %d\n", bdp->reloc_lsn);
    }
    free (bbtp);
}

hid_dump()
{
    hidden_defect_dir hidhdr, *htp;
    hidden_entry *hep;
    config_rec confrec;
    config_rec *cr = &confrec;
    char *s;
    int size, i;

    /* read the config record... (defines that follow need it) */
    if (!raw) {
	if (lseek (fd, CONFIG_REC_LSN * 512, SEEK_SET) < 0) {
	    fprintf(stderr, "bad seek\n");
	    perror("CONFIG record");
	    exit(1);
	}
    }
    if (read(fd, cr, sizeof(config_rec)) != sizeof(config_rec)) {
	fprintf(stderr, "short read\n");
	perror("CONFIG record");
	exit(1);
    }

    /* seek to and read the hidden bbt */
    if (lseek (fd, HID_DIR1_LSN * 512, SEEK_SET) < 0) {
	fprintf(stderr, "bad seek\n");
	perror("HIDDEN defect record");
	exit(1);
    }
    if (read(fd, &hidhdr, sizeof(hidden_defect_dir)) 
	    != sizeof(hidden_defect_dir)) {
	fprintf(stderr, "short read\n");
	perror("HIDDEN defect record");
	exit(1);
    }

    printf("\n");
    printf("HID_id       ");
    s = hidhdr.id;
    while (isprint (*s))
	printf ("%c", *s++);
    printf ("\n");
    printf("num_bad_blks %d\n", hidhdr.total);

    if (strncmp (hidhdr.id, "HIDDEN", 6)) {
	printf ("NO HIDDEN BAD BLOCK TABLE\n");
	return;
    }

    size = sizeof (hidden_defect_dir) + hidhdr.total * sizeof (hidden_entry);
    htp = (hidden_defect_dir *) malloc (size);
    if (lseek (fd, HID_DIR1_LSN * 512, SEEK_SET) < 0) {
	fprintf(stderr, "bad seek\n");
	perror("HIDDEN DEFECT record");
	exit(1);
    }
    if (read(fd, htp, size) != size) {
	fprintf(stderr, "short read\n");
	perror("HIDDEN DEFECT records");
	exit(1);
    }
    for (hep = &htp -> entry[0], i = 0; i < htp -> total; i++, hep ++) {
	printf ("cyl %d head %d sect %d\n", 
		hep -> cyl, 
		hep -> head, 
		hep -> sector);
    }
    free (htp);
}
