#!/bin/sh
#
# Copyright (c) 1990,1991, OPEN SOFTWARE FOUNDATION, INC.
# ALL RIGHTS RESERVED
#
#  book build
#
#  synopsis:  book-format [-n|-t|-x] [-d|-p] [-c] [-Ttypesetter]
#		[body] [toc] [index] [gloss] [piece] [all]
#
#	see below for fuller explaination.
#
Version="\$Revision: 1.19 $, \$Date: 91/12/19 11:37:28 $, \$Author: bowe $";

DEFAULT_DTE_LIBDIR=/project/tools/libdata

if [ -z "$DTE_LIBDIR" ] ; then DTE_LIBDIR=$DEFAULT_DTE_LIBDIR ; fi
if [ -z "$OUTPUT" ] ; then OUTPUT=Output ; fi
if [ -z "$FMT_PIECE" ] ; then FMT_PIECE="fmt-piece" ; fi
if [ -z "$SYNC_DIR" ] ; then SYNC_DIR="Sync" ; fi

doNroff=false
doTroff=false
doExpress=false
formatType="d"
Form=draft
subdir=draft
Language=PostScript
typeset=""
verbose=false

doBody=false
doToc=false
doIndex=false
doGloss=false
doAll=false
glossArgs=""
bookName=""
#doCopyright=false
doSML=""
SMLspecified=false
FMTspecified=false
ROFFspecified=false
defines=""
formatOption=-t

descFile=Description
Piece=""
doNothing=true
extraArgs=""
doODE=false
doPrintOrder=false

# look at personal defaults
if [ -f $HOME/.dterc ] ; then
    . $HOME/.dterc
    export DTE_LIBDIR DTE_OUTFILTER TYPESETTER OUTPUT
    export TROFF NROFF TBL PIC NEQN EQN RMIFDEF
fi

# parse command line

while [ $# -gt 0 ] ; do
    case "$1" in

	-n)	Language=ASCII
		formatOption=-n
		doNroff=true
		ROFFspecified=true
		subsubdir=ascii
		;;
	-t)	Language=PostScript
		formatOption=-t
		doNroff=false
		doTroff=true
		ROFFspecified=true
		;;
	-x)	Language=Express
		formatOption=-x
		doNroff=false
		doExpress=true
		ROFFspecified=true
		subsubdir=dito
		;;
	-draft | -d)
		Form=draft
		formatType=d
		FMTspecified=true
		subdir=draft
		;;
	-pub* | -p)
		Form=pub
		formatType=p
		FMTspecified=true
		subdir=publish
		;;
	-nosml)	doSML=""
		SMLspecified=true
		;;
	-sml)	doSML="-sml"
		SMLspecified=true
		;;
	-T*)	typeset=$1
		TYPESETTER=`echo $1 | sed "s/-T//"`
		;;
	-libdir | -root)
		shift
		DTE_LIBDIR=$1
		;;
	-out*)	shift
		OUTPUT=$1
		;;
	-alt)	shift
		extraArgs="$extraArgs -alt $1"
		;;
	-des*)	shift
		descFile=$1
		extraArgs="$extraArgs -des $descFile"
		;;
	-book)	shift
		bookName=$1
		;;
	-D*)	defines="$defines $1" ;;
	-U*)	defines="$defines $1" ;;
	-showif*)
		extraArgs="$extraArgs -showifdef"
		;;
	-v)	verbose=true
		extraArgs="$extraArgs -v"
		;;
	-mac)	extraArgs="$extraArgs -mac \"$2\""
		shift
		;;
	-ODE)	doODE=true ;;
	body)	doNothing=false
		doBody=true
		;;
	toc)	doNothing=false
		doToc=true
		;;
	index)	doNothing=false
		doIndex=true
		;;
	gloss*)	doNothing=false
		doGloss=true
		;;
	all | -all)
		doNothing=false
		doBody=true
		doToc=true
		doIndex=true
		doGloss=true
		doAll=true
		;;
	-version)
		echo "`basename $0`: $Version"
		exit 0
		;;
	-printorder)
		doPrintOrder=true
		extraArgs="$extraArgs -printorder"
		;;
	-h*)	doNothing=true
		break
		;;
	*)	doNothing=false
		Piece="$Piece $1"
		;;
    esac
    shift
done

# ________________________________________________________________________

if `$doNothing`; then
    cat << E_O_F

synopsis:  book-format [-n|-t|-x] [-d|-p] [-Ttypesetter] [-des <file>]
	[-libdir <dir>] [-output <dir>] [-D<symbol> ...] [-showifdef]
	[body] [toc] [index] [gloss] [piece ...] [all]

    -t		Build with troff (default)
    -n		Build with nroff
    -x		Build with express
    -sml	Build with sml macros
    -nosml	Build without sml macros (default)
    -draft|-d	Draft formatting (default)
    -pub|-p	Publish formatting
    -mac <macs>	include macros <macs> after others.
    -T<typesetter>
		Use the named typesetter (don't postprocess) (default "ps")

    -out <dir>	Set directory for output to <dir>
    -des <file>	Specify alternate Description file to <file>
    -alt <dir>	Search directory <dir> for files not found in local directory
    -libdir <dir> Set root directory of DTE to <dir>
    -book <book>  Specify name of book to screen by when formatting glossary

    -D<sybmol>	Process #ifdef/#else/#endif (cpp-like), defining <symbol>.
    -showifdef	Embolden #ifdef/#else/#endif in the output.  Turns off other
		#ifdef processing.

    -printorder	Name the output files so that 'lpr *' will print them in order

    body	Builds the document body
    toc		Builds the document table of contents
    index	Builds the document index
    gloss	Builds the document glossary
    piece	Builds only the named pieces (names from Description file)
    all		Builds everything (body, toc, glossary, index)

E_O_F
    exit 1
fi

# ________________________________________________________________________

# working locations and format control places

if [ ! -d "$DTE_LIBDIR" ] ; then
    echo "DTE_LIBDIR is set to $DTE_LIBDIR, a non-existent directory."
    exit 1
fi

if [ ! -r $descFile ] ; then
    echo "Can't find book Description file ($descFile)"
    exit 1
fi

#PATH=$DTE_LIBDIR/bin:$DTE_LIBDIR/tools/bin:/usr/local/bin:$PATH
PATH=/usr/local/bin:$PATH
export DTE_LIBDIR PATH

# ________________________________________________________________________

if $doODE ; then
    D=`pwd | sed -e 's:/src/:/obj/:'`
    if [ ! -d $D ] ; then
	mkdir `basename $D` >/dev/null 2>&1
	if mkdir $D ; then
	    echo "Could not create $D"
	    exit 1
	fi
    fi
    OUTPUT="$D/$OUTPUT"
fi

export OUTPUT

# ________________________________________________________________________

# look thru the description file for extra instructions

if [ -z "$bookName" ] ; then
    bookName="`egrep '^##.*BOOK' $descFile | head -1 | sed -e 's/^.*BOOK//'`"
fi

if [ -z "$defines" ] ; then
    defines="`egrep '^##.*DEFINE' $descFile | sed -e 's/^.*DEFINE//'`"
fi

if [ -n "$bookName" ] ; then glossArgs="-book $bookName" ;fi

if [ -n "$defines" ] ; then extraArgs="$extraArgs $defines" ; fi

#  These can be used if writer NEVER wants a glossary, toc, index when
#  doing "book-format all".  (saves processing time).
if egrep '^##.*NO GLOSS' $descFile >/dev/null ; then
    doGloss=false
    if $verbose ; then echo "Will skip glossary. (per $descFile)" ; fi
fi
if egrep '^##.*NO TOC' $descFile >/dev/null ; then
    doToc=false
    if $verbose ; then echo "Will skip toc. (per $descFile)" ; fi
fi
if egrep '^##.*NO INDEX' $descFile >/dev/null ; then
    doIndex=false
    if $verbose ; then echo "Will skip index. (per $descFile)" ; fi
fi

if `$SMLspecified` ; then
    true
else
    if egrep '^##[ 	]*NO *SML' $descFile >/dev/null ; then
	doSML=""
	a=" NOT"
    elif egrep '^##[ 	]*SML' $descFile >/dev/null ; then
	doSML="-sml"
	a=""
    fi
    if $verbose ; then echo "Will$a use SML macros. (per $descFile)" ; fi
fi

if $FMTspecified ; then
    true
else
    if egrep '^##[ 	]*DRAFT' $descFile >/dev/null ; then
	Form=draft ; formatType=d
    elif egrep '^##[ 	]*PUBLISH' $descFile >/dev/null ; then
	Form=pub   ; formatType=p
    fi
    if $verbose ; then
	echo "Formatting with '$Form' format. (per $descFile)"
    fi
fi

if $ROFFspecified ; then
    true
else
    if egrep '^##[ 	]*TROFF' $descFile >/dev/null ; then
	Language=PostScript ; formatOption=-t ; doNroff=false ; doTroff=true
    elif egrep '^##[ 	]*NROFF' $descFile >/dev/null ; then
	Language=ASCII      ; formatOption=-n ; doNroff=true
    elif egrep '^##[ 	]*EXPRESS' $descFile >/dev/null ; then
	Language=Express    ; formatOption=-x ; doNroff=false ; doExpress=true
    fi
    if $verbose ; then
	echo "Using '$Language' and '$formatOption'. (per $descFile)"
    fi
fi

if egrep '^##.*PRINTORDER' $descFile >/dev/null ; then
    doPrintOrder=true
    extraArgs="$extraArgs -printorder"
    if $verbose ; then echo "Using print ordering (per $descFile)" ; fi
fi

# ________________________________________________________________________

if [ -z "$subsubdir" ] ; then
    if [ -n "$TYPESETTER" ] ; then subsubdir=$TYPESETTER
    else subsubdir="ps"			: default typesetter...
    fi
fi

if [ ! -d $OUTPUT ] ; then mkdir $OUTPUT ; fi
if [ ! -d $OUTPUT/$subdir ] ; then mkdir $OUTPUT/$subdir ; fi
if [ ! -d $OUTPUT/$subdir/$subsubdir ] ; then mkdir $OUTPUT/$subdir/$subsubdir ; fi

# ________________________________________________________________________
#
#	the body stuff

#  If doing "all", remove sync files so that if writer added/removed a
#  file, things won't get screwed up.  A safty catch.
if $doAll ; then rm -f $SYNC_DIR/$formatType$formatOption.* ; fi

echo ""
echo "Doing '$Form' formatting into $Language"
echo ""

#  See if requested to do entire body.

if $doBody; then
    count=1
    maxcount=`sed -e '/^#/d' -e '/^$/d' $descFile | wc -l`

    while [ $count -le $maxcount ] ; do
	if $verbose ; then
	    echo "_______________________________"
	    date
	    echo args to ${FMT_PIECE}: $formatOption -$formatType $doSML \
			$extraArgs $count $nroffOptions $typeset
	fi
	${FMT_PIECE} $formatOption -$formatType $doSML $extraArgs $count \
			"$nroffOptions" $typeset
	# $doCopyright REMOVED, following $count
	count=`expr $count + 1`
    done
fi

#  See if requested to do a single piece.

if [ "$Piece" != "" ]; then
    for P in $Piece ; do
	one='$1'	: HACK
	PieceNum=`sed -e '/^#/d' -e '/^$/d' $descFile | \
		    awk "$one ~ /^${P}$/ {print NR}"`
	if [ `echo $PieceNum | wc -w` -gt 1 ] ; then
	    echo "There is more than one piece with the same name, $Piece,"
	    echo "in the Description file ($descFile):"
	    grep $Piece $descFile
	    exit 1
	fi
	if [ "$PieceNum" -gt 0 -a "$PieceNum" -lt 5000 ]; then
	    if $verbose ; then
		echo args to ${FMT_PIECE}: $formatOption -$formatType $doSML\
			$extraArgs $PieceNum $nroffOptions $typeset
	    fi
	    ${FMT_PIECE} $formatOption -$formatType $doSML $extraArgs \
			$PieceNum "$nroffOptions" $typeset
	    # $doCopyright REMOVED, following $PieceNum
	else
	    echo "can't find piece $Piece"
	    exit 2
	fi
    done
fi

# ________________________________________________________________________

# Do Table of Contents, Index, Glossary, if requested

if $doIndex && $atOSF; then
    fmt-index $formatOption -$formatType $typeset $extraArgs
fi

if $doGloss && $atOSF; then
    fmt-gloss $formatOption -$formatType $typeset $glossArgs $extraArgs
fi

if $doToc && $atOSF ; then
    fmt-toc $formatOption -$formatType $typeset $extraArgs
fi

if $verbose ; then date ; fi
echo ""
