/*
 * 5799-WZQ (C) COPYRIGHT IBM CORPORATION  1986,1987,1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */
/* $Header:if_un.c 12.0$ */
/* $ACIS:if_un.c 12.0$ */
/* $Source: /ibm/acis/usr/sys/ca_atr/RCS/if_un.c,v $ */

#ifndef lint
static char    *rcsid = "$Header:if_un.c 12.0$";
#endif lint

/*
 * Ungermann-Bass NICps/2 (Ethernet) Adapter (4.3)
 * This driver uses code on the PC and FCI (Function Call Interface) code that
 * is on the Adapter itself
 * We pass the PC control blocks (ZCB's) that tell the FCI what we want to do
 *
 * NOTE: while the name was changed from if_ub.c to if_un.c not all function
 *	and variable names have yet been changed too.
 *
 * Implementation notes:
 *	-	we have a number of ZCB's (control blocks) that we send to
 *		the pc code (via cbcb_req) with various commands. We have
 *		two transmit ZCB's, one receive ZCB, and one misc ZCB.
 *	-	for xmits, we copy the data output of the mbuf's chain that
 *		we are given into a buffer (one of two) in the softc structure.
 *		we then send a xmit command to the PS/2 and it copies the ZCB
 *		into the single PS/2 xmit ZCB and then copies the data
 *		into PS/2 memory and then calls the FCI with the xmit ZCB.
 *		the PS/2 code then returns. Later, when an interrupt arrives
 *		it is vectored to the FCI interrupt routine (which was taken
 *		over by the FCI code) which then sets the appropriate status
 *		in the xmit ZCB. Later the main loop detects that the status
 *		has changed and copies the ZCB back to ROMP memory and sends
 *		an interrupt to indicate the completion of the operation.
 *	-	for rcv's, we've issued a rcv enable, which is passed to the
 *		FCI with a ZCB in PS/2 memory. Later, after an interrupt the
 *		change in status in the received ZCB is picked up out of the
 *		main loop and the data is copied back to ROMP memory. Then
 *		an interrupt is generated (via rompint3) and the romp interrupt
 *		routine picks up the packet and then sends another rcv enable
 *		to allow more packets to be received.
 *	-	control commands (init, cancel receive etc). are similar except that
 *		a temporary ZCB is used, and the PS/2 code waits until the operation
 *		completes before copying back the status, so these operations
 *		are essentially synchronous.
 */

#include "un.h"
#if NUN > 0

#include "../machine/pte.h"

#include "param.h"
#include "systm.h"
#include "mbuf.h"
#include "buf.h"
#include "protosw.h"
#include "socket.h"
#include "vmmac.h"
#include "ioctl.h"
#include "errno.h"

#include "../net/if.h"
#include "../net/netisr.h"
#include "../net/route.h"

#ifdef INET
#include "../netinet/in.h"
#include "../netinet/in_systm.h"
#include "../netinet/in_var.h"
#include "../netinet/ip.h"
#include "../netinet/if_ether.h"
#endif INET

#ifdef NS
#include "../netns/ns.h"
#include "../netns/ns_if.h"
#endif NS

#include "../machine/io.h"
#include "if_unreg.h"
#include "../machineio/ioccvar.h"
#include "time.h"
#include "kernel.h"
#include "../machine/debug.h"
#include "../ca_atr/pcif.h"

#define UBIWATCH	30
#define UBOWATCH	30

int 
ubprobe(), ubattach();

caddr_t         ubstd[] = {(caddr_t) 0x00001550, (caddr_t) 0x00001554,
			   (caddr_t) 0x00001558, (caddr_t) 0x0000155c, 0};

struct iocc_device *ubinfo[NUN];

int 
ubint(), ubinit(), ubioctl(), uboutput(), ubreset(), ubwatch();
int ubsuspend();

struct iocc_driver undriver =
{ubprobe, 0, ubattach, 0, ubstd, "un", ubinfo,
 /*     int   csr chanrel flags 		suspend */
 0, 0, ubint, 0,    0,     DRIVER_SUSPEND,   ubsuspend };

struct mbuf    *ubget();

#ifdef DEBUG
#define SET_COMMAND(var, command) \
	if ((ubdebug&04)) printf("un: setting command = var (%x)\n",var); \
	var = command
#else
#define SET_COMMAND(var, command) var = command
#endif

/*
 * Ethernet software status per adapter.
 */
struct ub_softc {
	struct arpcom   us_ac;	/* generic network interface stuff */
#define	us_if	us_ac.ac_if	/* ifnet struct */
#define	us_addr	us_ac.ac_enaddr	/* hardware (i.e. ethernet) address */
	short           us_oactive;	/* 1 => output active */
	short           us_xbuf;/* in-use xmt buf (if output active) */
	short           us_xfull[2];	/* 1 => a full xmt buf */
	ZCB             zcb_tmp;/* zcb to attach and config down */
	char            ub_rcvbuf[MAX_PAC_LEN];	/* 1 receive buffer per
						 * adapter */
	ZCB             zcb_r;	/* 1 receive zcb per adapter */
	char            ub_xmtbuf[2][MAX_PAC_LEN];	/* 2 transmit buffers
							 * per adapter */
	ZCB             zcb_t[2];	/* 2 transmit zcb's per adapter */
	int             ubextra_ints;	/* Interrupt count for stray handling */
	short		us_owatch;	/* output watch timer */
	short		us_iwatch;	/* input watch timer */
	short		us_watch;	/* watchdog running */
}               ub_softc[NUN];

#ifdef DEBUG
int            ubdebug = 0;
#endif DEBUG

/*
 *  ubprobe - This adapter can only be at interrupt 3
 */
ubprobe(p)
	register caddr_t p;
{
	register struct ubdevice *addr = (struct ubdevice *) p;

	set_pcvec_map(UBIRQ, 1);	/* defensive tell pc to allow ints */
#ifdef notdef
	OUT(&addr->ub_csr, UB_GSFTINT);	/* generate software interrupt */
	PROBE_DELAY(200000);		/* allow .2 seconds for interrupt */
	OUT(&addr->ub_csr, 0);		/* turn off interrupt */
	return (PROBE_OK);
#else
	return (PROBE_NOINT);		/* don't attempt hardware interrupt */
#endif
}

/*
 *  ubattach - make the interface available to the network software
 *  (if the auto-configuration software determines that the interface
 *  exists).  The system will initialize the interface when it is
 *  ready to accept packets.
 */
ubattach(iod)
	register struct iocc_device *iod;
{
	register struct ub_softc *us = &ub_softc[iod->iod_unit];
	register struct ifnet *ifp = &us->us_if;
	register struct ubdevice *addr = (struct ubdevice *) iod->iod_addr;
	register int    i;
	ZCB            *zcbp;
	int	old_window = get_512_window();
	unsigned int ub_win = get_pc_cb(UBENT);
	struct ubpc *ub_pc = (struct ubpc *)(set_512_window(ub_win)
								+ pcif_512_fw);
	char volatile *ub_test;
	char *ub_get_addr;

	ifp->if_unit = iod->iod_unit;
	ifp->if_name = "un";
	ifp->if_mtu = ETHERMTU;

	zcbp = &us->zcb_tmp;

	if (ub_win) {
		ub_pc[iod->iod_unit].new = 1;
		ub_test = &(ub_pc[iod->iod_unit].tmp_status);
		ub_get_addr = &(ub_pc[iod->iod_unit].ub_addr[0]); 
		*ub_test = 0;
	} else { 
		ub_test = &zcbp->header.z_result;
		ub_get_addr = &zcbp->zcbs.z_stat_id[0];
	}

	/*
	 * Read the ethernet address off the board. First we have to do an
	 * INIT command (send the PC an init zcb) and then we do a STATUS
	 * command (send the PC a status zcb) so that we can get the address.
	 * We will then save the address. 
	 */
	zcbp = &us->zcb_tmp;
	ubinitcmd(zcbp,addr,ifp,ub_test);

	/*
	 * If the init command takes to long we time out and return so we
	 * don't hang the system If this happens, there is most likely a
	 * hardware problem with the adapter. 
	 */
	if (*ub_test == 0) {
		printf("un%d: init timed out - not attached\n", iod->iod_unit);
		return;
	}
	if (*ub_test != 0x18 && *ub_test != 0x1b) {
		printf("un%d: not init'ed\n", iod->iod_unit);
	}
	zcbp->header.z_options_lb = LOWBYTE(ifp->if_unit);
	zcbp->header.z_options_ub = HIGHBYTE(ifp->if_unit);
	SET_COMMAND(zcbp->header.z_command,  UB_STAT);
	*ub_test = 0;
	zcbp->header.z_result = 0;
	if (pc_req(CB_UBREQ, zcbp, UBENT) < 0) {
		panic("pc req fail, help!");
	}
	for (i = 100; *ub_test == 0 && i > 0; --i)
		delay(1);

	/*
	 * If the init command takes to long we time out and return so we
	 * don't hang the system If this happens, there is most likely a
	 * hardware problem with the adapter. 
	 */
	if (*ub_test == 0) {
		printf("un%d didn't init - not attached\n", iod->iod_unit);
		return;
	}
	for (i = 0; i < ETH_ADDR_SIZE; i++) {
		us->us_addr[i] = ub_get_addr[i];
	}
	set_512_window(old_window);
	printf("un%d: ethernet address ", ifp->if_unit);
	ubprintethaddr(us->us_addr);
	printf("\n");
	ifp->if_init = ubinit;
	ifp->if_ioctl = ubioctl;
	ifp->if_output = uboutput;
	ifp->if_reset = ubreset;
	ifp->if_flags = IFF_BROADCAST;
	if_attach(ifp);
	DEBUGF(ubdebug, printf("un%d: attached\n", iod->iod_unit));
}

/*
 * send an ub initialize command
 */
ubinitcmd(zcbp,addr,ifp,ub_test)
	ZCB            *zcbp;
	struct ubdevice *addr;
	struct ifnet *ifp;
	char volatile *ub_test;
{
	register int    i;
	SET_COMMAND(zcbp->header.z_command,  UB_INIT);
	zcbp->zcbi.z_max_xmt_len_lb = LOWBYTE(MAX_PAC_LEN);
	zcbp->zcbi.z_max_xmt_len_ub = HIGHBYTE(MAX_PAC_LEN);
	zcbp->zcbi.z_num_xmt_bfrs_lb = LOWBYTE(2);
	zcbp->zcbi.z_num_xmt_bfrs_ub = HIGHBYTE(2);
	zcbp->zcbi.z_max_rcv_size_lb = LOWBYTE(MAX_PAC_LEN);
	zcbp->zcbi.z_max_rcv_size_ub = HIGHBYTE(MAX_PAC_LEN);
	zcbp->zcbi.z_num_rcv_bfrs_lb = LOWBYTE(15);
	zcbp->zcbi.z_num_rcv_bfrs_ub = HIGHBYTE(15);
	/*
	 * The "acquired modes" field of the ZCB is not needed here so we use
	 * it to pass the 80286 the bus address of the adapter and have the
	 * PC read the POS registers and save the RAM address. 
	 */
	DEBUGF(ubdebug, printf("port addr is %x\n", addr);
	)
		zcbp->header.z_options_lb = LOWBYTE(ifp->if_unit);
	zcbp->header.z_options_ub = HIGHBYTE(ifp->if_unit);
	zcbp->zcbi.z_acq_modes_lb = LOWBYTE(addr);
	zcbp->zcbi.z_acq_modes_ub = HIGHBYTE(addr);
	if (pc_req(CB_UBREQ, zcbp, UBENT) < 0) {
		panic("pc req fail, help!");
	}
	for (i = 100; *ub_test == 0 && i > 0; --i)
		delay(1);

	DEBUGF(ubdebug, printf("un%d: init result=%x status=%x\n", ifp->if_unit, *ub_test, zcbp->header.z_status));
}

ubsuspend(iod, idr, how)
	register struct iocc_device *iod;
	struct iocc_driver *idr;
{
	int unit = iod->iod_unit;
	register struct ub_softc *us = &ub_softc[unit];
	register struct ifnet *ifp = &us->us_if;
	ZCB            *zcbp = &us->zcb_tmp;
	int	old_window = get_512_window();
	unsigned int ub_win = get_pc_cb(UBENT);
	struct ubpc *ub_pc = (struct ubpc *)(set_512_window(ub_win)
								+ pcif_512_fw);
	char volatile *ub_test;


	if (ub_win) {
		ub_test = &(ub_pc[unit].tmp_status);
		ub_pc[iod->iod_unit].new = 1;
		*ub_test = 0;
	} else { 
		ub_test = &zcbp->header.z_result;
	}

	if (how == SUSPEND_START) {
		if (ub_softc[unit].us_if.if_flags & IFF_RUNNING)
			ubzap(unit);
	} else if (how == SUSPEND_DONE) {
		ubinitcmd(zcbp,(struct ubdevice *)iod->iod_addr,ifp,ub_test);
		if (ifp->if_flags & IFF_RUNNING) {
			ifp->if_flags &= ~IFF_RUNNING;	/* pretend not running */
			ubinit(unit);
		}
	}
		
	set_512_window(old_window);
}

int ubhardreset = 0;

/*
 *  ubreset - reset interface
 *		only called from ubareset which is only called on halt
 *		when returning to DOS.
 */
ubreset(unit)
	register unsigned int unit;
{
	register struct iocc_device *iod;
	register struct ubdevice *addr = (struct ubdevice *) iod->iod_addr;

	if (unit < NUN && (iod = ubinfo[unit]) != 0 && iod->iod_alive != 0) {
		if (ub_softc[unit].us_if.if_flags & IFF_RUNNING) {
			ub_softc[unit].us_if.if_flags &= ~IFF_RUNNING;
			DEBUGF(ubdebug, printf("un%d: reset\n", unit));
			ubzap(unit);
		}
		if (ubhardreset) {
			DEBUGF(ubdebug, printf("un%d: reset hardware\n", unit));
			delay(100);				/* make sure it gets done */
			OUT(&addr->ub_csr, UB_RESET);		/* reset the hardware */
			delay(10);				/* make sure it gets done */
			OUT(&addr->ub_csr, 0);			/* reset the reset */
		}
#ifdef notdef
		ubinit(unit);
#endif
	}
}

/*
 * watch routine. We check to see if we've not gotten any transmit or 
 * receive interrupts lately and attempt to restart if necessary.
 * We assume that on an ethernet with at least one other active machine
 * that we will get at least one broadcast packet per timeout period.
 */

ubwatch(arg)
caddr_t arg;
{
	register int    unit = (int) arg;
	register struct ub_softc *us = &ub_softc[unit];
	register struct ifnet *ifp = &us->us_if;

	if ((ifp->if_flags & IFF_RUNNING) != 0) {
		if (us->us_owatch && --us->us_owatch==0) {
			printf("un%d: lost output interrupt - calling ubstart\n", unit);
			ubstart(us, us->us_xbuf);
		}
		if (us->us_iwatch && --us->us_iwatch==0) {
			DEBUGF(ubdebug,
		 printf("un%d: lost input interrupt?? - calling ubenrcv\n",
						 unit);
			);
			ubenrcv(us, unit);
		}
		timeout(ubwatch, (caddr_t) unit, hz);
	} else
		us->us_watch = 0;	/* if not running don't watch it */
}

/*
 *  ubinit - initialize interface, enable packet reception, start any
 *  pending writes
 */
ubinit(unit)
	register int    unit;
{
	register struct ub_softc *us = &ub_softc[unit];
	register struct ifnet *ifp = &us->us_if;
	register int    s;
	register char  *bp;
	ZCB            *zcbp;

	if (ifp->if_addrlist == (struct ifaddr *) 0) {
		/* no address */
		return;
	}
	if ((ifp->if_flags & IFF_RUNNING) == 0) {

		s = splimp();
		if (!us->us_watch) {
			timeout(ubwatch, (caddr_t) unit, hz);
			us->us_watch++;		/* timer running */
		}
		/* re-program ethernet address here if XNS supported */
		/* turn adapter on here */
		us->us_oactive = 0;
		ifp->if_flags |= IFF_RUNNING;
		/*
		 * turn on packet reception here - We do that by sending the
		 * FCI a receive zcb 
		 */
		bp = ((char *) (&us->ub_rcvbuf));
		zcbp = &us->zcb_r;
		SET_COMMAND(zcbp->header.z_command,  UB_RCV);
		zcbp->header.z_result = 0;
		zcbp->header.z_status = 1;
		zcbp->header.z_options_lb = LOWBYTE(unit);
		zcbp->header.z_options_ub = HIGHBYTE(unit);
		zcbp->zcbr.z_rcv_bfr_addr_4th = FRTHBYTE(bp);
		zcbp->zcbr.z_rcv_bfr_addr_3rd = THRDBYTE(bp);
		zcbp->zcbr.z_rcv_bfr_addr_2nd = SCNDBYTE(bp);
		zcbp->zcbr.z_rcv_bfr_addr_1st = FRSTBYTE(bp);
		if (pc_req(CB_UBREQ, zcbp, UBENT) < 0) {
			panic("pc req fail, help!");
		}
		if (ifp->if_snd.ifq_head) {	/* anything on send queue */
			struct mbuf    *m;

			IF_DEQUEUE(&ifp->if_snd, m);
			ubput(us, m, 0, unit);
			ubstart(us, 0);
			if (ifp->if_snd.ifq_head) {
				IF_DEQUEUE(&ifp->if_snd, m);
				ubput(us, m, 1, unit);
			}
		}
		splx(s);
	}
	DEBUGF(ubdebug, printf("un%d: init'ed\n", unit));
}

/*
 *  ubstart - start output from one of the adapter's 2 transmit buffers
 */
ubstart(us, xbuf)
	register struct ub_softc *us;
	register int    xbuf;
{

	us->us_oactive = 1;
	us->us_xbuf = xbuf;
	us->us_owatch = UBOWATCH;	/* set watchdog timer */
	if (pc_req(CB_UBREQ, (&us->zcb_t[xbuf]), UBENT) < 0) {
		panic("pc req fail, help!");
	}
}

/*
 *  ubint - interrupt handler.  find the cause of the interrupt and
 *  dispatch an appropriate handler routine.
 */
ubint(unit,irq,info)
	register int    unit;
{
	register struct ub_softc *us = &ub_softc[unit];
	register char   status;
	int	old_window = get_512_window();
	unsigned int ub_win = get_pc_cb(UBENT);
	struct ubpc *ub_pc = (struct ubpc *)(set_512_window(ub_win)
								+ pcif_512_fw);
#ifndef notdef
	int	rc = 1;

	switch(IRQ_GET_INFO(info)) {
	case UB_RCV_INTR:
		{
		register u_short length;
		register u_long	 addr;

		status = ub_pc[unit].rcv_status;
		if (status == 0) {
			DEBUGF(ubdebug&0x02,printf("un%d:stray rcv intr\n",unit););
			rc = 1;
			break;
		    }
		length = ub_pc[unit].receive_length;
		addr = ub_pc[unit].receive_addr;
		ub_pc[unit].rcv_status = 0;
		bcopy((char *)(set_512_window(addr)+pcif_512_fw),
			(char *)(&us->ub_rcvbuf),length);
		ubrint(unit, us, status, length);
		pc_copy_in += length;
		rc = 0;
		}
		break;
	case UB_XMIT_INTR:
		status = ub_pc[unit].xmit_status;
		if (status == 0) {
			DEBUGF(ubdebug&0x02,printf("un%d,stray xmit intr\n",unit););
			rc = 1;
			break;
		}
		ub_pc[unit].xmit_status = 0;
		ubxint(unit, us, status);
		rc = 0;
		break;
	default:
		printf("un%d: Illegal intr field in ub_pc (%d)\n",unit,
					 		IRQ_GET_INFO(info));
	case 0:
		{
		register int    intcount = 0;
		register int length;

		DEBUGF(ubdebug&0x02,printf("un%d:stray unknown intr\n",unit););
		if (us->zcb_r.header.z_result != 0) {
			/* determine the length of the packet */
			length = (us->zcb_r.zcbr.z_rcv_data_len_ub << 8 |
	       			us->zcb_r.zcbr.z_rcv_data_len_lb);
			status = us->zcb_r.header.z_result;
			us->zcb_r.header.z_result = 0;
			ubrint(unit, us, status, length);
			intcount++;
			rc = 0;
		}
		if (us->zcb_t[us->us_xbuf].header.z_result != 0) {
			status = us->zcb_t[us->us_xbuf].header.z_result;
			us->zcb_t[us->us_xbuf].header.z_result = 0;
			ubxint(unit, us, status);
			intcount++;
			rc = 0;
		}
		if (intcount == 0 && us->ubextra_ints >= 1) {
			/*
			 * if interrupt appears to be a stray, but we have
			 * previously processed ZCBs for which we have not
			 * received interrupts, claim the interrupt (and 
			 * decrement the count of expected interrupts) 
		 	 */
			rc = 0;
			us->ubextra_ints--;
		}
		if (intcount >= 2) {
			/*
			 * handled 2 ZCB's with 1 interrupt, so keep track of extra
			 * interrupt that we should claim later 
			 */
			us->ubextra_ints++;
		}
		}
		break;
	}
#else
#ifdef UN_STRAY_BUG
	register int    rc = 0;
#else
	register int    rc = 1;
#endif
	register int    intcount = 0;


	if (us->zcb_r.header.z_result != 0) {
		status = us->zcb_r.header.z_result;
		us->zcb_r.header.z_result = 0;
		ubrint(unit, us, status);
		intcount++;
		rc = 0;
	}
	if (us->zcb_t[us->us_xbuf].header.z_result != 0) {
		status = us->zcb_t[us->us_xbuf].header.z_result;
		us->zcb_t[us->us_xbuf].header.z_result = 0;
		ubxint(unit, us, status);
		intcount++;
		rc = 0;
	}
	if (intcount == 0 && us->ubextra_ints >= 1) {
		/*
		 * if interrupt appears to be a stray, but we have previously
		 * processed ZCBs for which we have not received interrupts,
		 * claim the interrupt (and decrement the count of expected
		 * interrupts) 
		 */
		rc = 0;
		us->ubextra_ints--;
	}
	if (intcount >= 2) {
		/*
		 * handled 2 ZCB's with 1 interrupt, so keep track of extra
		 * interrupt that we should claim later 
		 */
		us->ubextra_ints++;
	}
#endif
	return (rc);
}

/*
 *  ubrint - interrupt handler for packet reception.
 *
 *  log error if the result field was not correct. If the return code was
 *  UB_CMD_CAN (command canceled) we chuckit but we don't add one to the error
 *  log. Examine packet to determine type, if can't determine packet length
 *  from type, drop packet. otherwise decapsulate packet based on type and 
 *  pass to an appropriate higher-level input routine.
 */

int             maxethlen = 0;

ubrint(unit, us, result, len)
	int             unit;
	register struct ub_softc *us;
	register char   result;
	register int    len;
{
	register struct ifnet *ifp = &us->us_if;
	register struct ether_header *eh;
	register struct mbuf *m;
	int             resid;
	int             off;
	struct ifqueue *inq;
	u_short         type;

	us->us_iwatch = 0;		/* got an interrupt */
	if (result != UB_RCV_COMP_SUC) {
		if (result != UB_CMD_CAN) {
			us->us_if.if_ierrors++;
		}
		goto chuckit;
	}
	us->us_if.if_ipackets++;
	if (len >= maxethlen) {
		maxethlen = len;
	}
	if (len > MAX_PAC_LEN) {
		printf("un%d: huge packet!\n", unit);
		goto chuckit;
	}
	/*
	 * Process the packet 
	 */
	eh = (struct ether_header *) & us->ub_rcvbuf;
	DEBUGF(ubdebug & 0x2,
	       {
		char            cbuf[6];
		printf(" from = ");
		bcopy(eh->ether_shost, cbuf, sizeof(cbuf));
		ubprintethaddr(cbuf);
		printf("  to = ");
		bcopy(eh->ether_dhost, cbuf, sizeof(cbuf));
		ubprintethaddr(cbuf);
		printf(" ");
	}
	)
		len -= sizeof(struct ether_header);
	type = ntohs((u_short) eh->ether_type);
	/*
	 * The ETHERTYPE_NTRAILER packet types starting at ETHERTYPE_TRAIL
	 * have (type - ETHERTYPE_TRAIL) * 512 bytes of data followed by a
	 * type field and then a (variable length) header 
	 */
	if (type >= ETHERTYPE_TRAIL &&
	    type < ETHERTYPE_TRAIL + ETHERTYPE_NTRAILER) {
		off = (type - ETHERTYPE_TRAIL) * 512;
		if (off >= ETHERMTU) {
			goto chuckit;
		}
		type = ntohs(*(u_short *) ((caddr_t) (eh + 1) + off));
		resid = ntohs(*(u_short *) ((caddr_t) (eh + 1) + off + 2));
		if (off + resid > len) {
			goto chuckit;
		}
		len = off + resid;
	} else {
		off = 0;
	}
	if (len == 0) {
		goto chuckit;
	}
	/*
	 * pull packet off interface.  if off is non-zero, the packet has a
	 * trailing "header".  unget will move this header to the front, but
	 * we still have to remove the type and length fields from the front
	 * of the data. 
	 */
	m = ubget((char *) eh, len, off, &us->us_if);
	if (m != 0) {
		if (off) {
			struct ifnet   *ifp2;

			/*
			 * bcopy is used since word moves must be on 4 byte
			 * boundaries on the RT PC 
			 */
			bcopy(mtod(m, char *), (char *) &ifp2, sizeof(ifp2));
			m->m_off += 2 * sizeof(u_short);
			m->m_len -= 2 * sizeof(u_short);
			bcopy((char *) &ifp2, mtod(m, char *), sizeof(ifp2));
		}
		switch (type) {
#ifdef INET
		case ETHERTYPE_IP:
 		    {
			int s;
			DEBUGF(ubdebug & 0x2, printf("ip packet\n");
			)
			schednetisr(NETISR_IP);
			s = splimp();
			inq = &ipintrq;
			if (IF_QFULL(inq)) {
				DEBUGF(ubdebug & 0x2, printf(" qfull\n");
				)
					IF_DROP(inq);
				m_freem(m);
			} else {
				IF_ENQUEUE(inq, m);
				DEBUGF(ubdebug & 0x2, printf(" queued\n");
				)
			}
			splx(s);
			break;
		     }

		case ETHERTYPE_ARP:
			DEBUGF(ubdebug & 0x2, printf("arp packet\n");
			)
				arpinput(&us->us_ac, m);	/* arpinput frees m */
			break;
#endif INET
#ifdef NS
		case ETHERTYPE_NS:
			DEBUGF(ubdebug & 0x2, printf("ns packet\n");
			)
				schednetisr(NETISR_NS);
			inq = &nsintrq;
			break;
#endif NS
		default:
			DEBUGF(ubdebug & 0x2, printf("unknown packet\n");
			)
				m_freem(m);
			break;
		}
	}
chuckit:
	/*
	 * If the board is down we do not want to send the FCI another
	 * receive zcb 
	 */
	if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) == (IFF_UP | IFF_RUNNING)) {
		ubenrcv(us, unit);
	}
	return;
}

/*
 *  ubxint -  interrupt handler for transmit ready
 */
ubxint(unit, us, result)
	register int    unit;
	register struct ub_softc *us;
	register char   result;
{
	register int    next_buf;

	if (result != UB_XMT_COMP_SUC) {
		us->us_if.if_oerrors++;
	}
	us->us_if.if_opackets++;
	us->us_owatch = 0;		/* got an interrupt */
	us->us_xfull[us->us_xbuf] = 0;
	next_buf = 1 - us->us_xbuf;
	/* if the next xmt buffer is full then start transmitting */
	if (us->us_xfull[next_buf]) {
		ubstart(us, next_buf);
		if (us->us_if.if_snd.ifq_head) {
			struct mbuf    *m;

			IF_DEQUEUE(&us->us_if.if_snd, m);
			ubput(us, m, 1 - next_buf, unit);
		}
	} else {
		us->us_oactive = 0;
	}
}

/*
 *  uboutput - ethernet output routine.  encapsulate a packet of type
 *  family for the local net.  use trailer local net encapsulation if
 *  the number of bytes in the mbufs after the first is a multiple of
 *  512.
 */
uboutput(ifp, m0, dst)
	register struct ifnet *ifp;
	register struct mbuf *m0;
	register struct sockaddr *dst;
{
	int             type;
	int             s;
	int             error;
	char            edst[ETH_ADDR_SIZE];
	struct in_addr  idst;
	register struct ub_softc *us = &ub_softc[ifp->if_unit];
	register struct mbuf *m = m0;
	register struct ether_header *eh;
	int             off;
	struct mbuf    *m_get();
	int             usetrailers;

	if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING)) {
		error = ENETDOWN;
		goto bad;
	}
	switch (dst->sa_family) {

#ifdef INET
	case AF_INET:
		idst = ((struct sockaddr_in *) dst)->sin_addr;
		if (!arpresolve(&us->us_ac, m, &idst, edst, &usetrailers)) {
			/* not resolved */
			return (0);
		}
		off = ntohs((u_short) mtod(m, struct ip *)->ip_len) - m->m_len;
		if (usetrailers && off > 0 && (off & 0x1ff) == 0 &&
		    m->m_off >= MMINOFF + 2 * sizeof(u_short)) {
			type = ETHERTYPE_TRAIL + (off >> 9);
			m->m_off -= 2 * sizeof(u_short);
			m->m_len += 2 * sizeof(u_short);
			*mtod(m, u_short *) = htons((u_short) ETHERTYPE_IP);
			*(mtod(m, u_short *) +1) = htons((u_short) m->m_len);
			/*
			 * Packet to be sent with trailer, move first packet
			 * (control information) to end of chain. 
			 */
			while (m->m_next)
				m = m->m_next;
			m->m_next = m0;
			m = m0->m_next;
			m0->m_next = 0;
			m0 = m;
		} else {
			type = ETHERTYPE_IP;
		}
		break;
#endif INET
#ifdef NS
	case AF_NS:
		bcopy((caddr_t) & (((struct sockaddr_ns *) dst)->sns_addr.x_host),
		      (caddr_t) edst, sizeof(edst));
		type = ETHERTYPE_NS;
		off = 0;
		break;
#endif NS
	case AF_UNSPEC:
		eh = (struct ether_header *) dst->sa_data;
		bcopy((char *) eh->ether_dhost, (caddr_t) edst, sizeof(edst));
		type = eh->ether_type;
		break;
	default:
		printf("un%d: can't handle af%d\n", ifp->if_unit,
		       dst->sa_family);
		error = EAFNOSUPPORT;
		goto bad;
	}
	/*
	 * Add local net header.  If no space in first mbuf, allocate
	 * another. 
	 */
	if (m->m_off > MMAXOFF ||
	    MMINOFF + sizeof(struct ether_header) > m->m_off) {
		m = m_get(M_DONTWAIT, MT_HEADER);
		/*
		 * Note:  m_get, m_freem etc. guard against concurrent
		 * updates to the list of free mbufs. 
		 */
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		m->m_next = m0;
		m->m_off = MMINOFF;
		m->m_len = sizeof(struct ether_header);
	} else {
		m->m_off -= sizeof(struct ether_header);
		m->m_len += sizeof(struct ether_header);
	}
	eh = mtod(m, struct ether_header *);
	bcopy((caddr_t) edst, (caddr_t) eh->ether_dhost, sizeof(edst));
	bcopy((caddr_t) us->us_addr, (caddr_t) eh->ether_shost,
	      sizeof(eh->ether_shost));
	eh->ether_type = htons((u_short) type);
	/*
	 * queue packet for transmission.  if there is an empty transmit
	 * buffer on the adapter, use it. 
	 */
	s = splimp();
	if (IF_QFULL(&ifp->if_snd)) {
		IF_DROP(&ifp->if_snd);
		error = ENOBUFS;
		goto qfull;
	}
	if (us->us_xfull[0] == 0 || us->us_xfull[1] == 0) {	/* empty xmt buf */
		int             next_buf;

		if (us->us_xfull[0] == 0) {
			next_buf = 0;
		} else {
			next_buf = 1;
		}
		ubput(us, m, next_buf, ifp->if_unit);
		if (us->us_oactive == 0) {
			ubstart(us, next_buf);
		}
	} else {
		IF_ENQUEUE(&ifp->if_snd, m);
	}
	splx(s);
	return (0);
qfull:
	m0 = m;
	splx(s);
bad:
	m_freem(m0);
	return (error);
}

/*
 * ubenrcv - send the PC another receive zcb to give to the FCI
 */
ubenrcv(us, unit)
	struct ub_softc *us;
	int             unit;
{
	register int    s = splimp();
	ZCB            *zcbp;


	us->us_iwatch = UBIWATCH;	/* set input timeout */
	zcbp = &us->zcb_r;
	zcbp->header.z_options_lb = LOWBYTE(unit);
	zcbp->header.z_options_ub = HIGHBYTE(unit);
	if (pc_req(CB_UBREQ, zcbp, UBENT) < 0) {
		panic("pc req fail, help!");
	}
	splx(s);
}

/*
 *  ubput -  copy packet from an  mbuf chain to one of the adapter's softc
 *  transmit buffers.  the packet is extended to the minimum legal
 *  size if necessary.  the extra bytes could be zeroed out to improve
 *  security but are not to maximize performance.
 */
ubput(us, m, xbuf, unit)
	struct ub_softc *us;
	register struct mbuf *m;
	register int    xbuf;
	int             unit;
{
	register struct mbuf *mp;
	register char  *bp;
	int             total_len = 0;
	ZCB            *zcbp;

	/* bp will point to the appropriate softc transmit buffers */
	bp = ((char *) (&us->ub_xmtbuf[xbuf][0]));
	for (mp = m; mp; mp = mp->m_next) {
		register unsigned len = mp->m_len;

		bcopy(mtod(mp, char *), bp, len);
		bp += len;
		total_len += len;
		/*
		 * we put the address of the transmit buffer in the zcb down
		 * below so we have to make sure that it is pointing to the
		 * front of the buffer 
		 */
	}
	bp = ((char *) (&us->ub_xmtbuf[xbuf][0]));
	us->us_xfull[xbuf] = 1;	/* mark buffer full */
	zcbp = &us->zcb_t[xbuf];
	DEBUGF(ubdebug&2, printf("unput: packet length = %d\n", total_len);
	)
		SET_COMMAND(zcbp->header.z_command,  UB_XMT_FR);
	zcbp->header.z_status = 1;
	zcbp->header.z_result = 0;
	zcbp->header.z_options_lb = LOWBYTE(unit);
	zcbp->header.z_options_ub = HIGHBYTE(unit);
	zcbp->zcbt.z_xmt_data_len_lb = LOWBYTE(total_len);
	zcbp->zcbt.z_xmt_data_len_ub = HIGHBYTE(total_len);
	/*
	 * The 4th byte is the least sign. byte of a 4 byte thing the 1st is
	 * the most sign. byte 
	 */
	zcbp->zcbt.z_xmt_data_addr_4th = FRTHBYTE(bp);
	zcbp->zcbt.z_xmt_data_addr_3rd = THRDBYTE(bp);
	zcbp->zcbt.z_xmt_data_addr_2nd = SCNDBYTE(bp);
	zcbp->zcbt.z_xmt_data_addr_1st = FRSTBYTE(bp);
	m_freem(m);
}

/*
 *  ubget - copy packet from adapter's softc receive buffers
 *  into a chain of mbufs
 */

struct mbuf    *
ubget(ubbuf, totlen, off0, ifp)
	char           *ubbuf;
	register int    totlen;
	int             off0;
	struct ifnet   *ifp;
{
	register struct mbuf *m;
	struct mbuf    *top = 0;
	register struct mbuf **mp = &top;
	register int    off = off0;
	register int    len;
	register char  *cp;

	cp = ubbuf + sizeof(struct ether_header);
	while (totlen > 0) {
		char           *mcp;

		MGET(m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			goto bad;
		if (off) {	/* trailer exists */
			len = totlen - off;
			cp = ubbuf + sizeof(struct ether_header) + off;
		} else
			len = totlen;
		if (ifp)
			len += sizeof(ifp);
		if (len >= NBPG) {
			MCLGET(m);
			if (m->m_len == CLBYTES)
				m->m_len = len = MIN(len, CLBYTES);
			else
				m->m_len = len = MIN(MLEN, len);
		} else {
			m->m_len = len = MIN(MLEN, len);
			m->m_off = MMINOFF;
		}
		mcp = mtod(m, char *);
		if (ifp) {
			/* prepend ifp to first mbuf */
			/*
			 * bcopy is used since since word moves must be on 4
			 * byte boundaries on the RT PC 
			 */
			bcopy((char *) &ifp, mcp, sizeof(ifp));
			mcp += sizeof(ifp);
			len -= sizeof(ifp);
			ifp = (struct ifnet *) 0;
		}
		bcopy(cp, mcp, len);
		cp += len;
		*mp = m;
		mp = &m->m_next;
		if (off == 0) {
			totlen -= len;
			continue;
		}
		off += len;
		if (off == totlen) {
			cp = ubbuf + sizeof(struct ether_header);
			off = 0;
			totlen = off0;
		}
	}
	return (top);
bad:
	m_freem(top);
	return (0);
}

/*
 *  ioctl - process an ioctl request.
 */
ubioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	register int    cmd;
	register caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *) data;
	register int    s = splimp();
	register int    error = 0;

	switch (cmd) {
	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;

		switch (ifa->ifa_addr.sa_family) {
#ifdef INET
		case AF_INET:
			ubinit(ifp->if_unit);	/* before arpwhohas */
			((struct arpcom *) ifp)->ac_ipaddr =
				IA_SIN(ifa)->sin_addr;
			arpwhohas((struct arpcom *) ifp, &IA_SIN(ifa)->sin_addr);
			break;
#endif INET
#ifdef NS
		case AF_NS:
			{
				struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);
				struct ub_softc *us = &ub_softc[ifp->if_unit];

				if (ns_nullhost(*ina))
					ina->x_host = *(union ns_host *) (us->us_addr);
				else {
					ifp->if_flags &= ~IFF_RUNNING;
					bcopy((caddr_t) ina->x_host.c_host,
					      (caddr_t) us->us_addr, sizeof(us->us_addr));
					/*
					 * the ubinit will set the hardware
					 * address since the IFF_RUNNING flag
					 * is off 
					 */
				}
				ubinit(ifp->if_unit);
				break;
			}
#endif NS
		default:
			ubinit(ifp->if_unit);
			break;
		}
		break;
	case SIOCSIFFLAGS:
		if ((ifp->if_flags & IFF_UP) == 0 && ifp->if_flags &
		    IFF_RUNNING) {
			ubzap(ifp->if_unit);
			ifp->if_flags &= ~IFF_RUNNING;
		} else if (ifp->if_flags & IFF_UP && (ifp->if_flags &
						      IFF_RUNNING) == 0)
			ubinit(ifp->if_unit);
		break;
	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}

/*
 *  ubzap - send a cancel receive's zcb to the PC 
 *  
 */
ubzap(unit)
	register int    unit;
{
	register struct ub_softc *us = &ub_softc[unit];
	ZCB            *zcbp;
	int	old_window = get_512_window();
	unsigned int ub_win = get_pc_cb(UBENT);
	struct ubpc *ub_pc = (struct ubpc *)(set_512_window(ub_win)
								+ pcif_512_fw);
	char *ub_test;


	zcbp = &us->zcb_tmp;

	if (ub_win) {
		ub_test = &(ub_pc[unit].tmp_status);
		*ub_test = 0;
	} else { 
		ub_test = &zcbp->header.z_result;
	}

	DEBUGF(ubdebug, printf("un%d: Zap the board\n", unit));
	SET_COMMAND(zcbp->header.z_command,  UB_CAN_RCV);
	zcbp->header.z_result = 0;
	zcbp->header.z_options_lb = LOWBYTE(unit);
	zcbp->header.z_options_ub = HIGHBYTE(unit);
	if (pc_req(CB_UBREQ, zcbp, UBENT) < 0) {
		panic("pc req fail, help!");
	}
	while (*ub_test == 0) {
		delay(10);
	}
	if (*ub_test != UB_CAN_COMP) {
		DEBUGF(ubdebug, printf("un%d: Receives not Cancelled\n", unit));
	}
	set_512_window(old_window);
}

/*
 *  ubprintethaddr - print an ethernet address
 */
ubprintethaddr(p)
	register char  *p;
{
	register int    i;

	for (i = 0; i < ETH_ADDR_SIZE; i++) {
		if (i != 0)
			printf(":");
		printf("%x", *p++);
	}
}
#endif NUN > 0
