#   Syntax10.Scn.Fnt  _
   _
  MODULE V24; (* supporting flow control - gri 5.5.92 *)

  IMPORT SYSTEM, Kernel;

  CONST
    N = 1024;
    M = 64;

    ICU = 0FFFE0CH; UART = 0FFFD40H; (* Ceres-1 *)
    (* ICU = 0FFFF9004H; UART = 0FFFFC000H; Ceres-2 *)

    ISR = UART + 4*5; (* Interrupt Status Register *)
    MRB = UART + 4*8; (* Mode Register B *)
    SRB = UART + 4*9; (* Status Register B *)
    RHRB = UART + 4*11; (* Rx Holding Register B *)
    IPR = UART + 4*13; (* Input Port Register *)

    ACR = UART + 4*4; (* Auxiliary Control Register *)
    IMR = UART + 4*5; (* Interrupt Mask Register *)
    CSRB = UART + 4*9; (* Clock Select Register B *)
    CRB = UART + 4*10; (* Command Register B *)
    THRB = UART + 4*11; (* Tx Holding Register B *)
    SOPR = UART + 4*14; (* Set Output Port Register *)
    ROPR = UART + 4*15; (* Reset Output Port Register *)

    XON = 11X; XOFF = 13X;

  VAR
    On, NoFC, TxOn, RxOff: BOOLEAN;
    in, out: INTEGER;
    buf: ARRAY N OF CHAR;

  PROCEDURE +Int;
  BEGIN
    SYSTEM.GET(RHRB, buf[in]);
    IF NoFC THEN in := (in+1) MOD N
    ELSIF buf[in] = XOFF THEN TxOn := FALSE
    ELSIF buf[in] = XON THEN TxOn := TRUE
    ELSE in := (in + 1) MOD N;
      IF (out-in) MOD N < M THEN (* buffer almost full *)
        IF in = out THEN in := (in-1) MOD N END; (* buffer full: ignore character *)
        REPEAT UNTIL SYSTEM.BIT(SRB, 2); (* send XOFF *)
        SYSTEM.PUT(THRB, XOFF); RxOff := TRUE
      END
    END
  END Int;

  PROCEDURE Start* (CSR, MR1, MR2: CHAR);
  BEGIN
    On := TRUE; in := 0; out := 0; Kernel.InstallIP(Int, 2);
    SYSTEM.PUT(CRB, 30X); (* reset transmitter B *)
    SYSTEM.PUT(CRB, 20X); (* reset receiver B *)
    SYSTEM.PUT(CSRB, CSR); (* set in/out baud rate *)
    SYSTEM.PUT(CRB, 15X); (* reset MR pointer & enable Tx & Rx *)
    SYSTEM.PUT(MRB, MR1); (* set char length & parity mode *)
    SYSTEM.PUT(MRB, MR2); (* set channel mode & stop bit length *)
    SYSTEM.PUT(IMR, 20X); (* enable RxRDY interrupt *)
    SYSTEM.PUT(ICU, 1AH)
  END Start;

  PROCEDURE SetOP* (s: SET);
  BEGIN SYSTEM.PUT(SOPR, s)
  END SetOP;

  PROCEDURE ClearOP* (s: SET);
  BEGIN SYSTEM.PUT(ROPR, s)
  END ClearOP;

  PROCEDURE IP* (n: INTEGER): BOOLEAN;
  BEGIN RETURN SYSTEM.BIT(IPR, n)
  END IP;

  PROCEDURE SR* (n: INTEGER): BOOLEAN;
  BEGIN RETURN SYSTEM.BIT(SRB, n)
  END SR;

  PROCEDURE Available* (): INTEGER;
  BEGIN RETURN (in-out) MOD N
  END Available;

  PROCEDURE Receive* (VAR x: SYSTEM.BYTE);
  BEGIN
    REPEAT UNTIL in # out;
    x := buf[out]; out := (out+1) MOD N;
    IF ~NoFC & RxOff & ((in-out) MOD N < M) THEN (* buffer allmost empty *)
      REPEAT UNTIL SYSTEM.BIT(SRB, 2); (* send XON *)
      SYSTEM.PUT(THRB, XON); RxOff := FALSE
    END
  END Receive;

  PROCEDURE Send* (x: SYSTEM.BYTE);
  BEGIN
    IF On THEN
      REPEAT UNTIL (NoFC OR TxOn) & SYSTEM.BIT(SRB, 2); (* wait for TxRDY *)
      SYSTEM.PUT(THRB, x) (* send byte *)
    ELSE (* set flow control *) NoFC := ORD(x) = 0; TxOn := TRUE; RxOff := FALSE
    END
  END Send;

  PROCEDURE Break*;
    VAR i: LONGINT;
  BEGIN
    SYSTEM.PUT(CRB, 60X); (* start break *) i := 100000;
    REPEAT DEC(i) UNTIL i = 0;
    SYSTEM.PUT(CRB, 70X) (* stop break *)
  END Break;

  PROCEDURE Stop*;
  BEGIN
    SYSTEM.PUT(IMR, 0); (* disable all DUART interrupts *)
    SYSTEM.PUT(ICU, 3AH);
    On := FALSE
  END Stop;

BEGIN On := FALSE; NoFC := TRUE; SYSTEM.PUT(ACR, 90X) (* select baud rate set 2 *)
END V24.
