#   Syntax10.Scn.Fnt       MODULE SchemeMachine;	(*mf 11.2.89/11.5.92*)

	(* Virtual Machine, Scheme Interpreter.  This Module capsules the internal representation
		 of Scheme Objects and contains the Finite State Machine for evaluating Scheme expressions. *)

	CONST
		(* Scheme Types and Commands. *)
			tBits*=67108864;	(* Bits 28 through 31 of an object are reserved for a tag indicating the object's type. *)
			cBits=65536;	(* Bits 0 through 15 reference the cell memory array. *)

			(* Constant Types. *)
			constT*=0;	nil*=0; false*=1; true*=2; emptyVec*=3; emptyStrg*=4;
			symT*=-1; symBits=65536;
			pProcT*=-2;	pProcBits=512;
			numT*=-3;	maxNum=tBits DIV 2-1; zero=numT*tBits;
			charT*=-4;	charBits=256;
			portT*=-5; markT*=-6; errorT*=-7;
			undefT*=-8;	undefined*=undefT*tBits;
			(* Dynamic Types. *)
			pairT*=1;	MaxCells=32767; illRef=errorT*tBits; moved=-1;
			envT*=2; uProcT*=3; contT*=4; frameT*=contT;
			vecT*=5; strgT*=6;	maxArrInx=1024; lexLt=0; lexEq=1; lexGt=2;
			(* Commands. *)
			andC*=7; beginC*=8; popC*=9; popCmd*=popC*tBits; ifC*=10; lambdaC*=11; letC*=12;
			orC*=13; setLocC*=14; setGlobC*=15; globalC*=-9; local0C*=- 10; localC*=- 11; pcallC*=16; pcall0C*=17;

		(* Scheme Errors. *)
			errOvfl*=16; (* Overflow. *)	errUnfl*=17; (* Underflow. *)	errDiv0*=18; (* Division by zero. *)
			errOOM*=19; (* Out of Memory. *)	errOORng*=20; (* Reference out of Range. *)
			errSyntx*=21; (* Syntax Error. *)	errNotImpl*=22; (* Implementation Restriction. *)
			errArTooMny*=23; (* Too many arguments passed. *)	errArTooFew*=24; (* Not enough arguments passed. *)
			errListX*=25; (* Proper List expected. *)	errProcX*=26; (* Procedure expected. *)
			errListEnd*=27; (* End of list reached. *)


		(* Primitive Procedure Classes. *)
			clNiladic*=0; clMonadic*=1; clDyadic*=2; clGeneral*=3;

		(* Primitive Procedure OpCodes. *)
			(* Niladic Primitives. *)
			ppCurrInPrt*=0; ppCurrOutPrt*=1;

			(* Monadic Primitives. *)
			ppCaaaar*=0; ppCaaadr*=1; ppCaadar*=2; ppCaaddr*=3; ppCadaar*=4; ppCadadr*=5;
			ppCaddar*=6; ppCadddr*=7; ppCdaaar*=8; ppCdaadr*=9; ppCdadar*=10; ppCdaddr*=11;
			ppCddaar*=12; ppCddadr*=13; ppCdddar*=14; ppCddddr*=15;
			ppCaaar*=16; ppCaadr*=17; ppCadar*=18; ppCaddr*=19;
			ppCdaar*=20; ppCdadr*=21; ppCddar*=22; ppCdddr*=23;
			ppCaar*=24; ppCadr*=25; ppCdar*=26; ppCddr*=27; ppCar*=28; ppCdr*=29;
			ppPairP*=30; ppNullP*=31; ppLength*=32; ppRev*=33; ppLastPair*=34;
			ppNot*=35; ppBoolP*=36;
			ppSymP*=37; ppSymToStr*=38; ppStrToSym*=39;
			ppNumP*=40; ppCmxP*=41; ppRealP*=42; ppRatP*=43; ppIntP*=44; ppZeroP*=45;
			ppPosP*=46; ppNeg*=47; ppOddP*=48; ppEvenP*=49; ppExP*=50; ppInexP*=51; ppAbs*=52;
			ppChP*=53; ppChToInt*=54; ppIntToCh*=55;
			ppStrP*=56; ppStrLength*=57; ppStrToList*=58; ppListToStr*=59;
			ppVecP*=60; ppMkVec*=61; ppVecLength*=62; ppVecToList*=63; ppListToVec*=64;
			ppProcP*=65; ppCallCC*=66;
			ppInPrtP*=67; ppOutPrtP*=68; ppRead*=69; ppReadCh*=70; ppEofObjP*=71; ppNl*=72;
			ppLoad*=73;

			(* Dyadic Primitives. *)
			ppEqvP*=0; ppEqP*=1; ppEqualP*=2;
			ppCons*=3; ppSetCarX*=4; ppSetCdrX*=5; ppApp*=6; ppListTail*=7; ppListRef*=8;
			ppMemq*=9; ppMemv*=10; ppMember*=11; ppAssq*=12; ppAssv*=13; ppAssoc*=14;
			ppEq*=15; ppLt*=16; ppGt*=17; ppLe*=18; ppGe*=19;
			ppMax*=20; ppMin*=21; ppAdd*=22; ppMult*=23; ppSub*=24;
			ppDiv*=25; ppQuot*=26; ppRem*=27; ppMod*=28;
			ppChEq*=29; ppChLt*=30; ppChGt*=31; ppChLe*=32; ppChGe*=33;
			ppStrRef*=34; ppStrEq*=35; ppStrLt*=36; ppStrGt*=37; ppStrLe*=38; ppStrGe*=39; ppStrApp*=40;
			ppVecRef*=41;
			ppApply*=42; ppMap*=43; ppForEach*=44;
			ppCWInF*=45; ppCWOutF*=46;

			(* General Primitives with more than two, a varying or an arbitrary number of arguments. *)
			ppList*=0; ppSubstr*=1; ppVec*=2; ppVecSetX*=3; ppWrite*=4; ppDispl*=5; ppWriteCh*=6;


		(* States of the Evaluator State Machine. *)
			stAppNoArg=0; stAppMon=1; stAppDy=2; stAppGen=3;
			stAppU=4; stAppCont=5; stDyAccFirst=6; stGenAccArg=7; stGenAccLast=8; stEvalArgs=9;
			stSeqCont=10; stIfCont=11; stAndCont=12; stOrCont=13; stSetGlobCont=14; stSetLocCont=15;
			stMapCont=16; stMapEnd=17; eval=18; stReturn=19;


	TYPE
		Obj*=LONGINT;	(* Object. *)
		SType*=SHORTINT;	(* Scheme Data Type. *)
		
		Cells=POINTER TO CellsDescr;
		CellsDescr=RECORD r: ARRAY MaxCells OF Obj END;

		ObjProc*=PROCEDURE(obj: Obj): Obj;
		GCGlobalCountProc*=PROCEDURE(): INTEGER;


	VAR
		rand: LONGINT;	(* Used by Random Number Generator. *)

		sp, mrk: Obj;	(* Stack pointer and current mark. *)
		ca, cd, oldCa, oldCd, globalVa: Cells;	(* Two sets of CAR and CDR arrays and Global Environment. *)
		free: INTEGER;	(* Index to next available cell in <ca, cd> arrays. *)

		globalCount*: GCGlobalCountProc;	(* Installed Procedure to query how many objects are defined globally. *)
		symToString*, stringToSym*: ObjProc;	(* Procedures to be installed by the Scanner. *)
		
	(* Evaluator Registers. *)
		exp: Obj;	(* Expression to be evaluated. *)
		env: Obj;	(* Environment in which evaluation is to be performed. *)
		val: Obj;	(* Value obtained by evaluating expression in environment. *)
		unev: Obj;	(* Unevaluated statements in statement sequence. *)

		fun: Obj;	(* Function to be applied. *)
		first: Obj;	(* First pair of argument list or first parameter of dyadic function. *)
		last: Obj;	(* Last pair of argument list. *)
		aux: Obj;	(* Auxiliary Register. *)
		trap: Obj;	(* Set to 'nil' if all is o.k. *)
		numR: Obj;	(* General-Purpose-Register for scalar values.  The contents of this register are not relocated. *)

		cont: Obj;	(* Continuation to be passed to next evaluator function. *)
		state: Obj;	(* Next evaluator function. *)


(* Garbage Collection and Storage Allocation. **********************************************************)

	PROCEDURE RELOC(VAR entry: Obj);	(* Relocate cell pointed at by 'entry' and update 'entry' field. *)
		VAR ref: INTEGER; t: SType; tsize: INTEGER;
	BEGIN
		IF	entry >= pairT*tBits	THEN	t:=SHORT(SHORT(entry DIV tBits)); ref:=SHORT(entry);
			IF	oldCa.r[ref]=moved 	THEN	entry:=entry-ref+oldCd.r[ref]	(* Copy forwarding address. *)
			ELSE	(* Relocate object and enter broken heart and forwarding address at old location. *)
				ca.r[free]:=oldCa.r[ref]; cd.r[free]:=oldCd.r[ref]; oldCa.r[ref]:=moved; oldCd.r[ref]:=free;
				tsize:=SHORT(((entry MOD tBits)+cBits) DIV (cBits*2));
				entry:=entry-ref+free; INC(free); INC(ref);
				WHILE	tsize > 1	DO	ca.r[free]:=oldCa.r[ref]; cd.r[free]:=oldCd.r[ref]; INC(free); INC(ref); DEC(tsize)	END
			END
		END
	END RELOC;

	PROCEDURE GC(amnt: INTEGER): BOOLEAN;	(* Attempt Garbage Collection and return TRUE if 'amnt' cells are free. *)
		VAR gc, inx, scan: INTEGER; tmp: Cells;
	BEGIN	tmp:=oldCa; oldCa:=ca; ca:=tmp; tmp:=oldCd; oldCd:=cd; cd:=tmp;	(* Flip memory partitions. *)
		free:=1; scan:=1; RELOC(exp); RELOC(env); RELOC(val); RELOC(unev);	(* Relocate registers. *)
		RELOC(fun); RELOC(first); RELOC(last); RELOC(aux); RELOC(trap); RELOC(sp);
		inx:=0; gc:=globalCount(); WHILE	inx < gc	DO	RELOC(globalVa.r[inx]); INC(inx)	END;
		WHILE	scan < free	DO	RELOC(ca.r[scan]); RELOC(cd.r[scan]); INC(scan)	END;
		IF	free < MaxCells-amnt	THEN	RETURN TRUE	ELSE	trap:=errorT*tBits+errOOM; state:=stReturn; RETURN FALSE	END
	END GC;

	PROCEDURE MAV(amnt: INTEGER): BOOLEAN;	(* Is there enough memory available to allocate 'amnt' cells. *)
	BEGIN	RETURN free < MaxCells-amnt
	END MAV;

	PROCEDURE RLC(VAR entry: LONGINT): BOOLEAN;	(* Relocate cell pointed at by 'entry' and update 'entry' field. *)
	BEGIN	RELOC(entry); RETURN TRUE
	END RLC;

	PROCEDURE PUT(car, cdr: Obj);
	BEGIN	ca.r[free]:=car; cd.r[free]:=cdr; INC(free)
	END PUT;


(* Traps. ****************************************************************************************)

	PROCEDURE ResetTrap*();
	BEGIN	trap:=nil
	END ResetTrap;

	PROCEDURE Trap*(errno: SHORTINT; arg: Obj);
	BEGIN	state:=stReturn;
		IF	MAV(1) OR (GC(1)&RLC(arg))
		THEN	PUT(errorT*tBits+errno, arg); trap:=pairT*tBits+2*cBits-1+free
		ELSE	trap:=errorT*tBits+errno	END;
	END Trap;


(* Type Predicates and Type Checks. *******************************************************************)

	PROCEDURE TAG*(obj: Obj): SHORTINT;	(* Return the type decription tag of the object 'obj'. *)
	BEGIN	RETURN SHORT(SHORT(obj DIV tBits))
	END TAG;

	PROCEDURE TP(t: SType; obj: Obj): BOOLEAN;	(* Type Predicate.  Return TRUE if 'obj' is of type 't'. *)
	BEGIN	RETURN SHORT(SHORT(obj DIV tBits))=t
	END TP;

	PROCEDURE TCHK(t: SType; obj: Obj): BOOLEAN;	(* Return TRUE only if 'obj' is of type 't', else trap. *)
	BEGIN	IF	SHORT(SHORT(obj DIV tBits))=t	THEN	RETURN TRUE	ELSE	Trap(8+t, obj); RETURN FALSE	END
	END TCHK;

	PROCEDURE Num0CHK(obj: Obj): BOOLEAN;	(* Return TRUE only if 'obj' is not equal to the number zero, else trap. *)
	BEGIN	IF	obj # zero	THEN	RETURN TRUE	ELSE	Trap(errDiv0, nil); RETURN FALSE	END
	END Num0CHK;

	PROCEDURE NumInxCHK(obj: Obj): BOOLEAN;	(* Return TRUE only if 'obj' is a valid array index, else trap. *)
	BEGIN
		IF	(SHORT(SHORT(obj DIV tBits))=numT)&(obj MOD tBits < maxArrInx)
		THEN	RETURN TRUE	ELSE	Trap(errOORng, nil); RETURN FALSE	END
	END NumInxCHK;


(* Scalar Types. ***********************************************************************************)

	PROCEDURE Rand(): LONGINT;	(* 'Minimal Standard' Random Number Generator:  S. K. Park, K. W. Miller, Comm. ACM V31N10, Oct 88 *)
		CONST m=2147483647; q=127773; VAR lo, hi, test: LONGINT;
	BEGIN	hi:=rand DIV q; lo:=rand MOD q; test:=16807*lo-2836*hi;
		IF	test > 0	THEN	rand:=test	ELSE	rand:=test+m	END;	RETURN rand
	END Rand;

	PROCEDURE NewSym*(name: INTEGER): Obj;	(* Return a new Symbol object. *)
	BEGIN	RETURN symT*tBits+(Rand() MOD 1024)*symBits+name
	END NewSym;

	PROCEDURE SymName*(sym: Obj): INTEGER;	(* Return the location of 'sym's name in the symbol table. *)
	BEGIN	RETURN SHORT(sym (*MOD symBits*))
	END SymName;


	PROCEDURE NewNum*(val: LONGINT): Obj;	(* Return a new number object.  Trap if the number is out of range. *)
	BEGIN
		IF	val > maxNum	THEN	Trap(errOvfl, nil); RETURN zero
		ELSIF	val < -maxNum	THEN	Trap(errUnfl, nil); RETURN zero
		ELSE	RETURN numT*tBits+(val MOD tBits)	END
	END NewNum;

	PROCEDURE NumVal*(num: Obj): LONGINT;	(* Return value of 'num'. *)
	BEGIN	num:=num MOD tBits; IF	num > maxNum	THEN	RETURN num-tBits	ELSE	RETURN num	END
	END NumVal;


	PROCEDURE NewChar*(ch: CHAR): Obj;	(* Return a new character object. *)
	BEGIN	RETURN charT*tBits+ORD(ch)
	END NewChar;

	PROCEDURE CharCh*(char: Obj): CHAR;	(* Return character value of 'char'. *)
	BEGIN	RETURN CHR(SHORT(SHORT(char (*MOD charBits*))))
	END CharCh;


	PROCEDURE NewPProc*(class, op: SHORTINT): Obj;	(* Return a new Primitive Procedure object. *)
	BEGIN	RETURN pProcT*tBits+class*pProcBits+op
	END NewPProc;

	PROCEDURE PProcClass(pProc: Obj): SHORTINT;	(* Return the procedure class of 'pProc'. *)
	BEGIN	RETURN SHORT(SHORT((pProc MOD tBits) DIV pProcBits))
	END PProcClass;

	PROCEDURE PProcOp(pProc: Obj): SHORTINT;	(* Return the operation code of 'pProc'. *)
	BEGIN	RETURN SHORT(SHORT(pProc MOD pProcBits))
	END PProcOp;


(* Dynamic Scheme Data Types. **********************************************************************)

	PROCEDURE CONS*(car, cdr: Obj): Obj;	(* Return a new Pair CONStructed from 'car' and 'cdr'. *)
	BEGIN
		IF	MAV(1) OR (GC(1)&RLC(car)&RLC(cdr))
		THEN	PUT(car, cdr); RETURN pairT*tBits+2*cBits-1+free	ELSE	RETURN illRef	END
	END CONS;

	PROCEDURE TCONS*(t: SType; car, cdr: Obj): Obj;	(* Return a new object of type 't' constructed from 'car' and 'cdr'. *)
	BEGIN
		IF	MAV(1) OR (GC(1)&RLC(car)&RLC(cdr))
		THEN	PUT(car, cdr); RETURN t*tBits+2*cBits-1+free	ELSE	RETURN illRef	END
	END TCONS;

	PROCEDURE RCONS*(pair, obj: Obj): Obj;	(* Reverse-CONS 'obj' to the CDR of 'pair'. *)
	BEGIN
		IF	MAV(1) OR (GC(1)&RLC(pair)&RLC(obj))	THEN
			PUT(obj, nil); cd.r[SHORT(pair)]:=pairT*tBits+2*cBits-1+free; RETURN pairT*tBits+2*cBits-1+free
		ELSE	RETURN illRef	END
	END RCONS;

	PROCEDURE CAR*(pair: Obj): Obj;	(* Return the CAR of 'pair'. *)
	BEGIN	RETURN ca.r[SHORT(pair)]
	END CAR;

	PROCEDURE CDR*(pair: Obj): Obj;	(* Return the CDR of 'pair'. *)
	BEGIN	RETURN cd.r[SHORT(pair)]
	END CDR;

	PROCEDURE SETCAR*(pair: Obj; obj: Obj);	(* Set the CAR of 'pair' to 'obj'. *)
	BEGIN	ca.r[SHORT(pair)]:=obj
	END SETCAR;

	PROCEDURE SETCDR*(pair: Obj; obj: Obj);	(* Set the CDR of 'pair' to 'obj'. *)
	BEGIN	cd.r[SHORT(pair)]:=obj
	END SETCDR;

	PROCEDURE TCONS4*(t: SType; car1, cdr1, car2, cdr2: Obj): Obj;	(* Typed-cons vector*4 (used for commands). *)
	BEGIN
		IF	MAV(2) OR (GC(2)&RLC(car1)&RLC(cdr1)&RLC(car2)&RLC(cdr2))	THEN
			PUT(car1, cdr1); PUT(car2, cdr2); RETURN t*tBits+4*cBits-2+free
		ELSE	RETURN illRef	END
	END TCONS4;


	PROCEDURE NewArrFill(t: SType; len: INTEGER; cnstEl: Obj): Obj;	(* Return a new array of type 't'. *)
		VAR cells: INTEGER; arr: Obj;
	BEGIN	cells:=(len+1) DIV 2;
		IF	MAV(cells) OR GC(cells)	THEN	arr:=t*tBits+len*cBits+free;
			WHILE	cells > 0	DO	PUT(cnstEl, cnstEl); DEC(cells)	END;	RETURN arr
		ELSE	RETURN illRef	END
	END NewArrFill;

	PROCEDURE NewArrList*(t: SType; len: INTEGER; valLst: Obj): Obj;	(* Return a new array of type 't'. *)
		VAR cells: INTEGER; arr: Obj;
	BEGIN	cells:=(len+1) DIV 2;
		IF	MAV(cells) OR (GC(cells)&RLC(valLst))	THEN	arr:=t*tBits+len*cBits+free;
			WHILE	len > 0	DO	ca.r[free]:=ca.r[SHORT(valLst)]; valLst:=cd.r[SHORT(valLst)]; DEC(len);
				IF	len > 0	THEN	cd.r[free]:=ca.r[SHORT(valLst)]; valLst:=cd.r[SHORT(valLst)]; DEC(len)
				ELSE	cd.r[free]:=nil	END;
				INC(free)
			END;	RETURN arr
		ELSE	RETURN illRef	END
	END NewArrList;

	PROCEDURE ArrLen*(arr: Obj): INTEGER;	(* Return the size of array 'arr'. *)
	BEGIN	RETURN SHORT((arr MOD tBits) DIV cBits)
	END ArrLen;

	PROCEDURE ArrRef*(arr: Obj; el: INTEGER): Obj;	(* Return element 'el' of array 'arr'. *)
	BEGIN
		IF	(el >= 0)&(el < ArrLen(arr)) 	THEN
			IF	ODD(el)	THEN	RETURN cd.r[SHORT(arr)+(el DIV 2)]
			ELSE	RETURN ca.r[SHORT(arr)+(el DIV 2)]	END
		ELSE	Trap(errOORng, arr); RETURN illRef	END
	END ArrRef;

	PROCEDURE ArrSet(arr: Obj; el: INTEGER; val: Obj);	(* Set element 'el' of array 'arr' to 'val'. *)
	BEGIN
		IF	(el >= 0)&(el < ArrLen(arr)) 	THEN
			IF	ODD(el)	THEN	cd.r[SHORT(arr)+(el DIV 2)]:=val
			ELSE	ca.r[SHORT(arr)+(el DIV 2)]:=val	END
		ELSE	Trap(errOORng, arr)	END
	END ArrSet;

	PROCEDURE ArrLst(arr: Obj): Obj;	(* Return a list of array elements. *)
		VAR len, cells: INTEGER; lst: Obj;
	BEGIN	len:=ArrLen(arr); cells:=len DIV 2;
		IF	MAV(len) OR (GC(len)&RLC(arr))	THEN	lst:=pairT*tBits+2*cBits+free;
			WHILE	cells > 0	DO
				PUT(ca.r[SHORT(arr)], pairT*tBits+2*cBits+1+free); PUT(cd.r[SHORT(arr)], pairT*tBits+2*cBits+1+free);
				DEC(cells); INC(arr)
			END;
			IF	ODD(len)	THEN	PUT(ca.r[SHORT(arr)], nil)	ELSE	cd.r[free-1]:=nil	END;
			RETURN lst
		ELSE	RETURN illRef	END
	END ArrLst;


(* Variable References. *****************************************************************************)

	PROCEDURE NewLocRef*(lvl: SHORTINT; ref: INTEGER): Obj;
	BEGIN	IF	lvl=0	THEN	RETURN local0C*tBits+ref	ELSE	RETURN localC*tBits+lvl*cBits+ref	END
	END NewLocRef;

	PROCEDURE NewGlobRef*(ref: INTEGER): Obj;
	BEGIN	RETURN globalC*tBits+ref
	END NewGlobRef;


	PROCEDURE SetGlobal*(ref, obj: Obj);
	BEGIN	globalVa.r[SHORT(ref)]:=obj
	END SetGlobal;


(* Stack Operations. *******************************************************************************)

	PROCEDURE PSH1*(obj: Obj);
	BEGIN	IF	MAV(1) OR (GC(1)&RLC(obj))	THEN	PUT(obj, sp); sp:=frameT*tBits+2*cBits-1+free	END
	END PSH1;

	PROCEDURE POP1*(VAR obj: Obj);
	BEGIN	obj:=ca.r[SHORT(sp)]; sp:=cd.r[SHORT(sp)]
	END POP1;

	PROCEDURE POP2(VAR obj2, obj1: Obj);	(* PSH2 is performed by the operation PSH3(o1, o2, nil) *)
	BEGIN	obj1:=ca.r[SHORT(sp)]; obj2:=cd.r[SHORT(sp)]; sp:=cd.r[SHORT(sp+1)]
	END POP2;

	PROCEDURE PSH3(obj1, obj2, obj3: Obj);
	BEGIN
		IF	MAV(2) OR (GC(2)&RLC(obj1)&RLC(obj2)&RLC(obj3))
		THEN	PUT(obj1, obj2); PUT(obj3, sp); sp:=frameT*tBits+4*cBits-2+free	END
	END PSH3;

	PROCEDURE POP3(VAR obj3, obj2, obj1: Obj);
	BEGIN	obj1:=ca.r[SHORT(sp)]; obj2:=cd.r[SHORT(sp)]; obj3:=ca.r[SHORT(sp+1)]; sp:=cd.r[SHORT(sp+1)]
	END POP3;

	PROCEDURE POP4(VAR obj4, obj3, obj2, obj1: Obj);
	BEGIN	obj1:=ca.r[SHORT(sp)]; obj2:=cd.r[SHORT(sp)]; obj3:=ca.r[SHORT(sp+1)]; obj4:=cd.r[SHORT(sp+1)]; sp:=cd.r[SHORT(sp+2)]
	END POP4;

	PROCEDURE PSH5(obj1, obj2, obj3, obj4, obj5: Obj);	(* PSH4 is performed by the operation PSH5(o1, o2, o3, o4, nil) *)
	BEGIN
		IF	MAV(3) OR (GC(3)&RLC(obj1)&RLC(obj2)&RLC(obj3)&RLC(obj4)&RLC(obj5))
		THEN	PUT(obj1, obj2); PUT(obj3, obj4); PUT(obj5, sp); sp:=frameT*tBits+6*cBits-3+free	END
	END PSH5;

	PROCEDURE POP5(VAR obj5, obj4, obj3, obj2, obj1: Obj);
	BEGIN
		obj1:=ca.r[SHORT(sp)]; obj2:=cd.r[SHORT(sp)]; obj3:=ca.r[SHORT(sp+1)];
		obj4:=cd.r[SHORT(sp+1)]; obj5:=ca.r[SHORT(sp+2)]; sp:=cd.r[SHORT(sp+2)];
	END POP5;


(* Marks are required to compare and print cyclic structures. ************************************************)

	PROCEDURE NewMark*();	(* Generate a new mark. *)
	BEGIN	mrk:=mrk+2;	IF	mrk > markT*tBits+tBits-2	THEN	mrk:=markT*tBits+1	END
	END NewMark;

	PROCEDURE Mark*(obj: Obj);	(* Mark an object. *)
	BEGIN	IF	oldCa.r[SHORT(obj)]=mrk	THEN	INC(oldCa.r[SHORT(obj)])	ELSE	oldCa.r[SHORT(obj)]:=mrk	END
	END Mark;

	PROCEDURE Unmark*(obj: Obj);	(* Unmark an object. *)
	BEGIN	oldCa.r[SHORT(obj)]:=nil
	END Unmark;

	PROCEDURE Marked*(obj: Obj): BOOLEAN;	(* Check if object has been marked once. *)
	BEGIN	IF	oldCa.r[SHORT(obj)]=mrk	THEN	RETURN TRUE	ELSE	RETURN FALSE	END
	END Marked;

	PROCEDURE Mark2d*(obj: Obj): BOOLEAN;	(* Check if object has been marked twice. *)
	BEGIN	IF	oldCa.r[SHORT(obj)]=mrk+1	THEN	RETURN TRUE	ELSE	RETURN FALSE	END
	END Mark2d;


(* Structural Equality (as opposed to referential equality represented by pointer value identity). *******************)

	PROCEDURE EQUAL(o1, o2: Obj): BOOLEAN;	(* Return TRUE only if 'o1' and 'o2' are 'EQUAL?'. *)
		VAR e: BOOLEAN; tsize: INTEGER;
	BEGIN
		IF	o1=o2	THEN	RETURN TRUE
		ELSIF	(o1 >= pairT*tBits)&((o1 DIV cBits)=(o2 DIV cBits))&~Marked(o1)&~Marked(o2)	THEN
			Mark(o1); Mark(o2); tsize:=SHORT(((o1 MOD tBits)+1) DIV (cBits*2)); e:=TRUE;
			WHILE	e&(tsize > 0)	DO
				e:=e&EQUAL(ca.r[SHORT(o1)], ca.r[SHORT(o2)])&EQUAL(cd.r[SHORT(o1)], cd.r[SHORT(o2)]); INC(o1); INC(o2); DEC(tsize)
			END;	Unmark(o1); Unmark(o2);	RETURN e
		ELSE	RETURN FALSE	END
	END EQUAL;


(* Miscellaneous Procedures. ***********************************************************************)

	PROCEDURE CxxR(op, lvl: SHORTINT; obj: Obj): Obj;	(* CAR and CDR combinations. *)
	BEGIN
		WHILE	(lvl > 0)&TCHK(pairT, obj)	DO
			IF	ODD(op)	THEN	obj:=cd.r[SHORT(obj)]	ELSE	obj:=ca.r[SHORT(obj)]	END; op:=op DIV 2; DEC(lvl)
		END;	RETURN obj
	END CxxR;

	PROCEDURE Boolean(bool: BOOLEAN);	(* Set value register to a boolean object according to parameter. *)
	BEGIN	IF	bool	THEN	val:=true	ELSE	val:=false	END
	END Boolean;

	PROCEDURE StrComp(s1, s2: Obj): SHORTINT;	(* String Comparison. *)
		VAR l1, l2, inx: INTEGER;
	BEGIN
		l1:=ArrLen(s1); l2:=ArrLen(s2); inx:=0;
		WHILE	(inx < l1)&(inx < l2)&(ArrRef(s1, inx)=ArrRef(s2, inx))	DO	INC(inx)	END;
		IF	(inx=l1)&(l1=l2)	THEN	RETURN lexEq
		ELSIF	(inx=l1) OR ((inx # l2)&(ArrRef(s1, inx) < ArrRef(s2, inx)))	THEN	RETURN lexLt
		ELSE	RETURN lexGt	END
	END StrComp;

	PROCEDURE StrApp(s1, s2: Obj): Obj;	(* String Append. *)
		VAR l1, l2, inx, i: INTEGER; strg: Obj;
	BEGIN
		l1:=ArrLen(s1); l2:=ArrLen(s2); inx:=0; strg:=NewArrFill(strgT, l1+l2, nil);
		i:=0;	WHILE	i < l1	DO	ArrSet(strg, inx, ArrRef(s1, i)); INC(i); INC(inx)	END;
		i:=0;	WHILE	i < l2	DO	ArrSet(strg, inx, ArrRef(s2, i)); INC(i); INC(inx)	END;
		RETURN strg
	END StrApp;

	PROCEDURE Count(plist: Obj): INTEGER;	(* Count number of elements in proper list. *)
		VAR n: INTEGER;
	BEGIN	NewMark(); n:=0;
		WHILE	~Marked(plist)&TP(pairT, plist)	DO	Mark(plist); plist:=CDR(plist); INC(n)	END;
		IF	Marked(plist)	THEN	RETURN 32767
		ELSIF	last # nil	THEN	Trap(errListX, plist); RETURN n	ELSE	RETURN n	END
	END Count;


(* ***********************************THE STATE MACHINE*****************************************)
(* Instruction Sequencing. **************************************************************************)

	PROCEDURE SeqCont();	(* Advance to next instruction in instruction sequence. *)
	BEGIN	state:=eval; POP2(unev, env); unev:=cd.r[SHORT(unev)]; exp:=ca.r[SHORT(unev)];
		IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	(* cont=stSeqCont *)	END
	END SeqCont;

	PROCEDURE IfCont();	(* Choose between consequence or alternative according to outcome of test. *)
	BEGIN	state:=eval; POP3(unev, env, cont);
		IF	(val=false) OR (val=nil)	THEN	exp:=ca.r[SHORT(unev+1)]	ELSE	exp:=cd.r[SHORT(unev)]	END
	END IfCont;

	PROCEDURE AndCont();	(* Return result of test or true. *)
	BEGIN	POP1(cont); state:=cont;	IF	(val # false)&(val # nil)	THEN	val:=true	END
	END AndCont;

	PROCEDURE OrCont();	(* Return result of test or false. *)
	BEGIN	POP1(cont); state:=cont;	IF	val=nil	THEN	val:=false	END
	END OrCont;


(* Changing variables. ******************************************************************************)

	PROCEDURE SetGlobCont();	(* Store a global variable. *)
		VAR ref: Obj;
	BEGIN	POP2(cont, ref); state:=cont; globalVa.r[SHORT(ref)]:=val
	END SetGlobCont;

	PROCEDURE SetLocCont();	(* Store a local variable. *)
		VAR ref: Obj; l: SHORTINT; r: INTEGER;
	BEGIN
		POP3(cont, env, ref); state:=cont; l:=SHORT(SHORT((ref MOD tBits) DIV cBits)); r:=SHORT(ref);
		WHILE	l > 0	DO	env:=ca.r[SHORT(env)]; DEC(l)	END;
		IF	ODD(r)	THEN	cd.r[SHORT(env)+(r DIV 2)]:=val	ELSE	ca.r [SHORT(env)+(r DIV 2)]:=val	END
	END SetLocCont;


(* Mapping Functions. ******************************************************************************)

	PROCEDURE MapNxt();	(* Map the next argument. *)
		VAR t: SType; c: SHORTINT;
	BEGIN
		IF	TCHK(pairT, unev)	THEN
			IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stMapEnd; PSH3(first, last, fun)
			ELSE	PSH5(first, last, fun, unev, nil); cont:=stMapCont	END;	(* No need for saving env! *)
			PSH1(cont); t:=TAG(fun);
			IF	t=pProcT	THEN	c:=PProcClass(fun);
				IF	c=clMonadic	THEN	state:=stAppMon; val:=ca.r[SHORT(unev)]; PSH1(fun)
				ELSIF	c=clGeneral	THEN	state:=stAppGen; first:=CONS(ca.r[SHORT(unev)], nil); numR:=1
				ELSE	Trap(errNotImpl, nil)	END
			ELSIF	t=uProcT	THEN	state:=stAppU; first:=CONS(ca.r[SHORT(unev)], nil); numR:=1
			ELSIF	t=contT	THEN	sp:=fun; POP1(cont); state:=cont; val:=ca.r[SHORT(unev)]
			ELSE	Trap(errProcX, fun)	END
		END
	END MapNxt;

	PROCEDURE MapCont();
	BEGIN	POP4(unev, fun, last, first);
		IF	first=nil	THEN	first:=CONS(val, nil); last:=first	ELSE	last:=RCONS(last, val)	END;
		unev:=cd.r[SHORT(unev)]; MapNxt()
	END MapCont;

	PROCEDURE MapEnd();	(* End of Mapping. *)
	BEGIN	POP3(fun, last, first); POP1(cont); state:=cont;
		IF	first=nil	THEN	val:=CONS(val, nil); last:=first	ELSE	last:=RCONS(last, val); val:=first	END
	END MapEnd;


(* Applying Functions. *****************************************************************************)

	PROCEDURE Apply();
		VAR t: SType; c: SHORTINT;
	BEGIN	PSH1(cont);
		IF	val=nil	THEN	val:=fun; state:=stAppNoArg
		ELSIF	TCHK(pairT, val)	THEN	t:=TAG(first);
			IF	t=pProcT	THEN	fun:=first; PSH1(fun); c:=PProcClass(fun);
				IF	c=clMonadic	THEN
					IF	cd.r[SHORT(val)]=nil	THEN	val:=ca.r[SHORT(val)]; state:=stAppMon	ELSE	Trap(errArTooMny, val)	END
				ELSIF	c=clDyadic	THEN	PSH1(ca.r[SHORT(val)]); val:=cd.r[SHORT(val)];
					IF	TP(pairT, val)	THEN	
						IF	cd.r[SHORT(val)]=nil	THEN	val:=ca.r[SHORT(val)]; state:=stAppDy	ELSE	Trap(errArTooMny, val)	END
					ELSE	Trap(errArTooFew, nil)	END
				ELSE	state:=stAppGen; fun:=first; first:=val; numR:=Count(first)	END
			ELSIF	t=uProcT	THEN	state:=stAppU; fun:=first; first:=val; numR:=Count(first)
			ELSIF	t=contT	THEN
				IF	cd.r[SHORT(val)]=nil	THEN	sp:=first; POP1(cont); state:=cont	ELSE	Trap(errArTooMny, val)	END
			ELSE	Trap(errProcX, first)	END
		END
	END Apply;


(* Calling Functions with Current Continuation. **********************************************************)

	PROCEDURE CallCC();
		VAR t: SType; c, op: SHORTINT;
	BEGIN	t:=TAG(val);
		IF	t=pProcT 	THEN	c:=PProcClass(val);
			IF	c=clGeneral	THEN	PSH1(cont); first:=CONS(sp, nil); fun:=val; numR:=1; state:=stAppGen
			ELSIF	c=clMonadic	THEN	PSH1(cont); PSH1(val); val:=CDR(sp); state:=stAppMon
			ELSE	Trap(errArTooFew, nil)	END
		ELSIF	t=uProcT 	THEN	PSH1(cont); first:=CONS(sp, nil); fun:=val; numR:=1; state:=stAppU
		ELSIF	t=contT 	THEN	PSH1(cont); fun:=val; val:=sp; cont:=fun; state:=cont; sp:=cd.r[SHORT(cont)]
		ELSE	Trap(errProcX, val)	END
	END CallCC;


(* Function Application. ***************************************************************************)
(* User-Defined Functions. ************************************************************************)

	PROCEDURE AppU();	(* Apply a user-defined function with at least one argument. *)
		VAR nargs: INTEGER;
	BEGIN	nargs:=SHORT(ca.r[SHORT(fun)]);
		IF	nargs > 0	THEN
			IF	MAV((nargs+1) DIV 2) OR GC((nargs+1) DIV 2)	THEN	env:=envT*tBits+nargs*cBits+free;
				state:=eval; cont:=stSeqCont; unev:=ca.r[SHORT(fun+1)]; exp:=ca.r[SHORT(unev)];
				ca.r[free]:=cd.r[SHORT(fun+1)]; cd.r[free]:=ca.r[SHORT(first)]; first:=cd.r[SHORT(first)]; DEC(nargs); INC(free);
				IF	nargs=numR	THEN	DEC(nargs);
					WHILE	nargs > 1	DO
						ca.r[free]:=ca.r[SHORT(first)]; first:=cd.r[SHORT(first)]; cd.r[free]:=ca.r[SHORT(first)]; first:=cd.r[SHORT(first)];
						DEC(nargs); DEC(nargs); INC(free)
					END;
					IF	nargs > 0	THEN	ca.r[free]:=ca.r[SHORT(first)]; cd.r[free]:=nil; INC(free)	END;
					IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	END
				ELSIF	nargs < numR	THEN
					IF	cd.r[SHORT(fun)]=true	THEN	(* Rest-Arg. *)
						DEC(nargs); DEC(nargs); 
						WHILE	nargs > 1	DO
							ca.r[free]:=ca.r[SHORT(first)]; first:=cd.r[SHORT(first)]; cd.r[free]:=ca.r[SHORT(first)]; first:=cd.r[SHORT(first)];
							DEC(nargs); DEC(nargs); INC(free)
						END;
						IF	nargs > 0	THEN	ca.r[free]:=ca.r[SHORT(first)]; cd.r[free]:=cd.r[SHORT(first)]; INC(free)
						ELSE	ca.r[free]:=first; cd.r[free]:=nil; INC(free)	END;
						IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	END
					ELSE	Trap(errArTooMny, nil)	END
				ELSE	Trap(errArTooFew, nil)
				END
			END
		ELSE	Trap(errArTooMny, first)	END
	END AppU;


(* Primitive Functions with variable number of arguments. *************************************************)

	PROCEDURE GenAccArg();	(* Accumulate argument. *)
	BEGIN	state:=eval; POP5(unev, env, numR, last, first); INC(numR); unev:=cd.r[SHORT(unev)]; exp:=ca.r[SHORT(unev)];
		IF	first=nil	THEN	first:=CONS(val, nil); last:=first	ELSE	last:=RCONS(last, val)	END;
		IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stGenAccLast; PSH3(first, last, numR);
		ELSE	PSH5(first, last, numR, env, unev); cont:=stGenAccArg	END
	END GenAccArg;

	PROCEDURE GenAccLast();	(* Accumulate last argument. *)
	BEGIN	POP3(numR, last, first); INC(numR); POP1(fun);
		IF	first=nil	THEN	first:=CONS(val, nil); last:=first	ELSE	last:=RCONS(last, val)	END;
		IF	TAG(fun)=uProcT	THEN	state:=stAppU	ELSE	state:=stAppGen	END
	END GenAccLast;

	PROCEDURE AppGen();	(* Apply Primitive Function with variable number of arguments. *)
		VAR i, j, k: INTEGER;
	BEGIN	POP1(cont); state:=cont;
		CASE	PProcOp(fun)	OF
		| ppList:
			val:=first
		| ppSubstr:
			IF	(numR=3)&TCHK(strgT, ca.r[SHORT(first)])
					&NumInxCHK(ca.r[SHORT(cd.r[SHORT(first)])])&NumInxCHK(ca.r[SHORT(cd.r[SHORT(cd.r[SHORT(first)])])])
			THEN	i:=SHORT(NumVal(ca.r[SHORT(cd.r[SHORT(first)])])); j:=SHORT(NumVal(ca.r[SHORT(cd.r[SHORT(cd.r[SHORT(first)])])]));
				IF	i=j	THEN	val:=emptyStrg
				ELSIF	i < j	THEN
					val:=NewArrFill(strgT, j-i, nil); k:=0;
					WHILE	i < j	DO	ArrSet(val, k, ArrRef(ca.r[SHORT(first)], i)); INC(i); INC(k)	END
				ELSE	Trap(errOORng, nil)	END
			ELSIF	numR < 3	THEN	Trap(errArTooFew, nil)	ELSE	Trap(errArTooMny, nil)	END
		| ppVec:
			val:=NewArrList(vecT, SHORT(numR), first)
		| ppVecSetX:
			IF	(numR=3)&TCHK(vecT, ca.r[SHORT(first)])&NumInxCHK(ca.r[SHORT(cd.r[SHORT(first)])])	THEN
				ArrSet(ca.r[SHORT(first)], SHORT(NumVal(ca.r[SHORT(cd.r[SHORT(first)])])), ca.r[SHORT(cd.r[SHORT(cd.r[SHORT(first)])])])
			ELSIF	numR < 3	THEN	Trap(errArTooFew, nil)	ELSE	Trap(errArTooMny, nil)	END
		| ppWrite, ppDispl, ppNl, ppWriteCh:
			Trap(errNotImpl, nil)
		END
	END AppGen;


(* Monadic Functions. ***************************************************************************)

	PROCEDURE AppMon();	(* Apply a monadic function. *)
		VAR i: INTEGER; t: SType; op: SHORTINT;
	BEGIN	POP1(fun); POP1(cont); state:=cont; op:=PProcOp(fun);
		CASE op OF
		| ppNot:	Boolean((val=false) OR (val=nil))
		| ppBoolP:	Boolean((val=true) OR (val=false))
		| ppPairP:	Boolean(TP(pairT, val))
		| ppCar:	IF	TCHK(pairT, val)	THEN	val:=ca.r[SHORT(val)]	END
		| ppCdr:	IF	TCHK(pairT, val)	THEN	val:=cd.r[SHORT(val)]	END
		| ppCaar..ppCddr:	val:=CxxR(op, 2, val)
		| ppCaaar..ppCdddr:	val:=CxxR(op, 3, val)
		| ppCaaaar..ppCddddr:	val:=CxxR(op, 4, val)
		| ppNullP:	Boolean(val=nil)
		| ppLength:
			i:=0;	WHILE	TP(pairT, val)	DO	INC(i); val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=NewNum(i)	ELSE	Trap(errListX, val)	END
		| ppRev:
			first:=nil;	WHILE	TP(pairT, val)	DO	first:=CONS(ca.r[SHORT(val)], first); val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=first	ELSE	Trap(errListX, val)	END
		| ppLastPair:
			last:=errListEnd;	WHILE	TP(pairT, val)	DO	last:=val; val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=last	ELSE	Trap(errListX, val)	END
		| ppSymP:	Boolean(TP(symT, val))
		| ppSymToStr:	IF	TCHK(symT, val)	THEN	val:=symToString(val)	END
		| ppStrToSym:	IF	TCHK(strgT, val)	THEN	val:=stringToSym(val)	END
		| ppNumP:	Boolean(TP(numT, val))
		| ppCmxP, ppRealP, ppRatP:	val:=false
		| ppIntP:	Boolean(TP(numT, val))
		| ppZeroP:	Boolean(TCHK(numT, val)&(NumVal(val)=0))
		| ppPosP:	Boolean(TCHK(numT, val)&(NumVal(val) > 0))
		| ppNeg:	Boolean(TCHK(numT, val)&(NumVal(val) < 0))
		| ppOddP:	Boolean(TCHK(numT, val)&ODD(NumVal(val)))
		| ppEvenP:	Boolean(TCHK(numT, val)&~ODD(NumVal(val)))
		| ppExP:	Boolean(TCHK(numT, val))
		| ppInexP:	Boolean(~TCHK(numT, val))
		| ppAbs:	IF	TCHK(numT, val)	THEN	val:=NewNum(ABS(NumVal(val)))	END
		| ppChP:	Boolean(TP(charT, val))
		| ppChToInt:	IF	TCHK(charT, val)	THEN	val:=NewNum(ORD(CharCh(val)))	END
		| ppIntToCh:	IF	TCHK(numT, val)	THEN	val:=NewChar(CHR(NumVal(val)))	END
		| ppStrP:	Boolean(TP(strgT, val))
		| ppStrLength  :	IF	TCHK(strgT, val)	THEN	val:=NewNum(ArrLen(val))	END
		| ppStrToList:	IF	TCHK(strgT, val)	THEN	val:=ArrLst(val)	END
		| ppListToStr:
			IF	val=nil	THEN	val:=emptyVec
			ELSE	NewMark(); i:=0; first:=val;
				WHILE	~Marked(first)&TP(pairT, first)&TCHK(charT, ca.r[SHORT(first)])
				DO	Mark(first); first:=cd.r[SHORT(first)]; INC(i)	END;
				IF	(Marked(first) OR (first # nil))	THEN	Trap(errListX, val)	ELSE	val:=NewArrList(vecT, i, val)	END
			END
		| ppVecP:	Boolean(TP(vecT, val) OR (val=emptyVec))
		| ppMkVec:	IF	NumInxCHK(val)	THEN	val:=NewArrFill(vecT, SHORT(NumVal(val)), nil)	END
		| ppVecLength:	IF	TCHK(vecT, val)	THEN	val:=NewNum(ArrLen(val))	END
		| ppVecToList :	IF	val=emptyVec	THEN	val:=nil	ELSIF	TCHK(vecT, val)	THEN	val:=ArrLst(val)	END
		| ppListToVec:
			IF	val=nil	THEN	val:=emptyVec
			ELSE	NewMark(); i:=0; first:=val;
				WHILE	~Marked(first)&TP(pairT, first)	DO	Mark(first); first:=cd.r[SHORT(first)]; INC(i)	END;
				IF	(Marked(first) OR (first # nil))	THEN	Trap(errListX, val)	ELSE	val:=NewArrList(vecT, i, val)	END
			END
		| ppProcP:	t:=TAG(val); Boolean((t=pProcT) OR (t=uProcT) OR (t=contT))
		| ppCallCC:	CallCC()
		| ppInPrtP, ppOutPrtP, ppRead, ppReadCh, ppEofObjP, ppLoad:	Trap(errNotImpl, nil)
		END
	END AppMon;


(* Dyadic Functions. ********************************************************************************)

	PROCEDURE DyAccFirst();	(* Accumulate first argument of dyadic function. *)
	BEGIN	state:=eval; POP2(unev, env); first:=val; PSH1(first); unev:=cd.r[SHORT(unev)]; exp:=ca.r[SHORT(unev)];
		IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stAppDy	ELSE	Trap(errArTooMny, unev)	END
	END DyAccFirst;

	PROCEDURE AppDy();	(* Apply dyadic function. *)
	BEGIN	POP1(first); POP1(fun); POP1(cont); state:=cont;	(* First parameter is in first, second in val. *)
		CASE	PProcOp(fun)	OF
		| ppEqvP:	Boolean(first=val)
		| ppEqP:	Boolean(first=val)
		| ppEqualP:	NewMark(); Boolean(EQUAL(first, val))
		| ppCons:	val:=CONS(first, val)
		| ppSetCarX:	IF	TCHK(pairT, first)	THEN	SETCAR(first, val)	END
		| ppSetCdrX:	IF	TCHK(pairT, first)	THEN	SETCDR(first, val)	END
		| ppApp:	
			IF	TCHK(pairT, first)	THEN	aux:=CONS(ca.r[SHORT(first)], nil); last:=aux; first:=cd.r[SHORT(first)];
				WHILE	TP(pairT, first)	DO	last:=RCONS(last, ca.r[SHORT(first)]); first:=cd.r[SHORT(first)]	END;
				IF	first=nil	THEN	SETCDR(last, val); val:=aux	ELSE	Trap(errListX, first)	END
			END
		| ppListTail:
			IF	TCHK(numT, val)	THEN	numR:=NumVal(val);
				WHILE	(numR > 0)&TP(pairT, first)	DO	DEC(numR); first:=cd.r[SHORT(first)]	END;
				IF	numR=0	THEN	val:=first	ELSIF	first=nil	THEN	Trap(errListEnd, nil)	ELSE	Trap(errListX, first)	END
			END
		| ppListRef:
			IF	TCHK(numT, val)	THEN	numR:=NumVal(val);
				WHILE	(numR > 0)&TP(pairT, first)	DO	DEC(numR); first:=cd.r[SHORT(first)]	END;
				IF	first=nil	THEN	Trap(errListEnd, nil)	ELSIF	(numR=0)&TCHK(pairT, first)	THEN	val:=ca.r[SHORT(first)]	END
			END
		| ppMemq, ppMemv:
			WHILE	TP(pairT, val)&(first # ca.r[SHORT(val)])	DO	val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=false	ELSIF	TP(pairT, val)	THEN	val:=cd.r[SHORT(val)]	ELSE	Trap(errListX, val)	END
		| ppMember :
			NewMark();	WHILE	TP(pairT, val)&~EQUAL(first, ca.r[SHORT(val)])	DO	val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=false	ELSIF	TP(pairT, val)	THEN	val:=cd.r[SHORT(val)]	ELSE	Trap(errListX, val)	END
		| ppAssq, ppAssv:
			WHILE	TP(pairT, val)&TP(pairT, ca.r[SHORT(val)])&(first # ca.r[SHORT(ca.r[SHORT(val)])])	DO	val:=cd.r[SHORT(val)]	END;
			IF	val=nil	THEN	val:=false	ELSIF	TCHK(pairT, val)&TCHK(pairT, ca.r[SHORT(val)])	THEN	val:=ca.r[SHORT(val)]	END
		| ppAssoc:
				NewMark();
				WHILE	TP(pairT, val)&TP(pairT, ca.r[SHORT(val)])&~EQUAL(first, ca.r[SHORT(ca.r[SHORT(val)])])	DO	val:=cd.r[SHORT(val)]	END;
				IF	val=nil	THEN	val:=false	ELSIF	TCHK(pairT, val)&TCHK(pairT, ca.r[SHORT(val)])	THEN	val:=ca.r[SHORT(val)]	END
		| ppEq:	Boolean(TCHK(numT, first)&TCHK(numT, val)&(first=val))
		| ppLt:	Boolean(TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) < NumVal(val)))
		| ppGt:	Boolean(TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) > NumVal(val)))
		| ppLe:	Boolean(TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) <= NumVal(val)))
		| ppGe:	Boolean(TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) >= NumVal(val)))
		| ppMax:	IF	TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) > NumVal(val))	THEN	val:=first	END
		| ppMin:	IF	TCHK(numT, first)&TCHK(numT, val)&(NumVal(first) < NumVal(val))	THEN	val:=first	END
		| ppAdd:	IF	TCHK(numT, first)&TCHK(numT, val)	THEN	val:=NewNum(NumVal(first)+NumVal(val))	END
		| ppMult:	IF	TCHK(numT, first)&TCHK(numT, val)	THEN	val:=NewNum(NumVal(first)*NumVal(val))	END
		| ppSub:	IF	TCHK(numT, first)&TCHK(numT, val)	THEN	val:=NewNum(NumVal(first)-NumVal(val))	END
		| ppDiv:	IF	TCHK(numT, first)&TCHK(numT, val)&Num0CHK(val)	THEN	val:=NewNum(NumVal(first) DIV NumVal(val))	END
		| ppQuot:	IF	TCHK(numT, first)&TCHK(numT, val)&Num0CHK(val)	THEN	val:=NewNum(NumVal(first) DIV NumVal(val))	END
		| ppRem:	IF	TCHK(numT, first)&TCHK(numT, val)&Num0CHK(val)	THEN	val:=NewNum(NumVal(first) MOD NumVal(val))	END
		| ppMod:
				IF	TCHK(numT, first)&TCHK(numT, val)&Num0CHK(val)	THEN	numR:=NumVal(val); first:=NumVal(first); val:=first DIV numR;
					IF	((first < 0)&(numR > 0)) OR ((first > 0)&(numR  < 0))	THEN	val:=val+numR	END;
					val:=NewNum(val)
				END
		| ppChEq:	Boolean(TCHK(charT, first)&TCHK(charT, val)&(first=val))
		| ppChLt:	Boolean(TCHK(charT, first)&TCHK(charT, val)&(first < val))
		| ppChGt:	Boolean(TCHK(charT, first)&TCHK(charT, val)&(first > val))
		| ppChLe:	Boolean(TCHK(charT, first)&TCHK(charT, val)&(first <= val))
		| ppChGe:	Boolean(TCHK(charT, first)&TCHK(charT, val)&(first >= val))
		| ppStrRef:
				IF	first=emptyStrg	THEN	Trap(errOORng, val)
				ELSIF	TCHK(strgT, first)&NumInxCHK(val) 	THEN	val:=ArrRef(first, SHORT(NumVal(val)))	END
		| ppStrEq:	Boolean(TCHK(strgT, first)&TCHK(strgT, val)&(StrComp(first, val)=lexEq))
		| ppStrLt:	Boolean(TCHK(strgT, first)&TCHK(strgT, val)&(StrComp(first, val)=lexLt))
		| ppStrGt:	Boolean(TCHK(strgT, first)&TCHK(strgT, val)&(StrComp(first, val)=lexGt))
		| ppStrLe:	Boolean(TCHK(strgT, first)&TCHK(strgT, val)&(StrComp(first, val) # lexGt))
		| ppStrGe:	Boolean(TCHK(strgT, first)&TCHK(strgT, val)&(StrComp(first, val) # lexLt))
		| ppStrApp:
				IF	first # emptyStrg	THEN
					IF	val=emptyStrg	THEN	val:=first
					ELSIF	TCHK(strgT, first)&TCHK(strgT, val)	THEN	val:=StrApp(first, val)	END
				END
		| ppVecRef:	
				IF	first=emptyVec	THEN	Trap(errOORng, val)
				ELSIF	TCHK(vecT, first)&NumInxCHK(val) 	THEN	val:=ArrRef(first, SHORT(NumVal(val)))	END
		| ppApply:	Apply()
		| ppMap, ppForEach:	PSH1(cont); fun:=first; unev:=val; first:=nil; last:=nil; MapNxt()
		| ppCWInF, ppCWOutF:	Trap(errNotImpl, nil)
		END
	END AppDy;


(* Continuations. *******************************************************************************)

	PROCEDURE AppCont();	(* Application of a Continuation. *)
	BEGIN	POP1(fun); sp:=fun; POP1(cont); state:=cont
	END AppCont;


(* Setting up Function Application. *****************************************************************)

	PROCEDURE AppNoArg();	(*Application of a Function without Arguments. *)
		VAR t: SType; c, o: SHORTINT;
	BEGIN	t:=TAG(val);	(* Register 'val' contains function to be applied. *)
		IF	t=pProcT 	THEN	c:=PProcClass(val); o:=PProcOp(val); POP1(cont); state:=cont;
			IF	c=clGeneral	THEN
				IF	o=ppList	THEN	val:=nil	ELSIF	o=ppVec	THEN	val:=emptyVec	ELSE	Trap(errArTooFew, nil)	END
			ELSIF	c=clMonadic	THEN
				IF	o=ppRead	THEN	Trap(errNotImpl, nil)
				ELSIF	o=ppReadCh	THEN	Trap(errNotImpl, nil)
				ELSIF	o=ppNl	THEN	Trap(errNotImpl, nil)
				ELSE	Trap(errArTooFew, nil)	END
			ELSIF	c=clNiladic	THEN	Trap(errNotImpl, nil)
			ELSE	Trap(errArTooFew, nil)	END
		ELSIF	t=uProcT	THEN
			IF	SHORT(ca.r[SHORT(val)])=0	THEN
				state:=eval; unev:=ca.r[SHORT(val+1)]; cont:=stSeqCont; exp:=ca.r[SHORT(unev)];
				IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	END
			ELSE	Trap(errArTooFew, nil)	END
		ELSIF	t=contT	THEN	Trap(errArTooFew, nil)
		ELSE	Trap(errProcX, val)
		END
	END AppNoArg;


	PROCEDURE EvalArgs ();	(* Application of a Function (dispatches to argument evaluation). *)
		VAR t: SType; c: SHORTINT;
	BEGIN	state:=eval; POP2(unev, env); fun:=val; PSH1(fun); t:=TAG(fun); exp:=ca.r[SHORT(unev)];
		IF	t=pProcT 	THEN	c:=PProcClass(fun);
			IF	c=clMonadic	THEN
				IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stAppMon 	ELSE	Trap(errArTooMny, unev)	END
			ELSIF	c=clDyadic	THEN
				IF	cd.r[SHORT(unev)]=nil	THEN	Trap(errArTooFew, nil)	ELSE	PSH3(env, unev, nil); cont:=stDyAccFirst	END
			ELSE	first:=nil; last:=nil; numR:=0;	(* clGeneral *)
				IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stGenAccLast; PSH3(first, last, numR)
				ELSE	PSH5(first, last, numR, env, unev); cont:=stGenAccArg	END
			END
		ELSIF	t=uProcT	THEN	first:=nil; last:=nil; numR:=0;
			IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stGenAccLast; PSH3(first, last, numR)
			ELSE	PSH5(first, last, numR, env, unev); cont:=stGenAccArg	END
		ELSIF	t=contT	THEN	IF	cd.r[SHORT(unev)]=nil	THEN	cont:=stAppCont 	ELSE	Trap(errArTooMny, unev)	END
		ELSE	Trap(errProcX, fun)
		END
	END EvalArgs;


(* The EVAL state of the Evaluator *****************************************************************)

	PROCEDURE EvalDsp();	(* Evaluation Dispatching Procedure. *)
		VAR l: SHORTINT; r, e: INTEGER;
	BEGIN
		CASE	exp DIV tBits	OF
		| constT, pairT, symT, numT, charT, vecT, strgT:	val:=exp; state:=cont	(* Self evaluating. *)
		| andC:	exp:=ca.r[SHORT(exp)]; PSH1(cont); cont:=stAndCont	(* Last Clause of AND. *)
		| beginC:	(* Instruction Sequence. *)
			unev:=exp; PSH1(cont); cont:=stSeqCont; exp:=ca.r[SHORT(unev)];	
			IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	END
		| popC:	state:=cont	(* Result of a conditional test expression is result of the complete conditional expression. *)
		| ifC:	unev:=exp; exp:=ca.r[SHORT(exp)]; PSH3(cont, env, unev); cont:=stIfCont	(* Conditional expression. *)
		| lambdaC:	val:=TCONS4(uProcT, ca.r[SHORT(exp)], ca.r[SHORT(exp+1)], cd.r[SHORT(exp)], env); state:=cont	(* Lambda. *)
		| letC:	(* Generation of a new scope. *)
			e:=(SHORT(ca.r[SHORT(exp)])+1) DIV 2;
			IF	MAV(e) OR (GC(e))	THEN
				PUT(env, undefined); env:=envT*tBits-1+e*2*cBits+free; DEC(e);
				WHILE	e > 0	DO	PUT(undefined, undefined); DEC(e)	END;
				unev:=cd.r[SHORT(exp)]; PSH1(cont); cont:=stSeqCont; exp:=ca.r[SHORT(unev)];
				IF	cd.r[SHORT(unev)]=nil	THEN	POP1(cont)	ELSE	PSH3(env, unev, nil)	END
			ELSE	Trap(errOOM, nil)
			END;
		| orC:	exp:=ca.r[SHORT(exp)]; PSH1(cont); cont:=stOrCont	(* Last Clause of OR. *)
		| setLocC:	PSH3(ca.r[SHORT(exp)], env, cont); exp:=cd.r[SHORT(exp)]; cont:=stSetLocCont	(* state=eval *)	(* Set! local variable. *)
		| setGlobC:	PSH3(ca.r[SHORT(exp)], cont, nil); exp:=cd.r[SHORT(exp)]; cont:=stSetGlobCont	(* state=eval *)	(* Set! global variable. *)
		| pcallC:	unev:=cd.r[SHORT(exp)]; exp:=ca.r[SHORT(exp)]; PSH1(cont); PSH3(env, unev, nil); cont:=stEvalArgs	(* Application. *)
		| pcall0C:	exp:=ca.r[SHORT(exp)]; PSH1(cont); cont:=stAppNoArg	(* Application, no arguments. *)
		| globalC:	val:=globalVa.r[SHORT(exp)]; state:=cont	(* Global variable reference. *)
		| local0C:	(* Local variable reference in top scope. *)
				r:=SHORT(exp); state:=cont;
				IF	ODD(r)	THEN	val:=cd.r[SHORT(env)+(r DIV 2)]
				ELSE	val:=ca.r[SHORT(env)+(r DIV 2)]	END
		| localC:	(* Local variable reference. *)
				l:=SHORT(SHORT((exp MOD tBits) DIV cBits)); r:=SHORT(exp); e:=SHORT(env); state:=cont;
				WHILE	l > 0	DO	e:=SHORT(ca.r[e]); DEC(l)	END;
				IF	ODD(r)	THEN	val:=cd.r[e+(r DIV 2)]	ELSE	val:=ca.r[e+(r DIV 2)]	END
		END
	END EvalDsp;


(* Scheme Instruction Sequencing *****************************************************************)

	PROCEDURE Eval*(obj: Obj): Obj;
	BEGIN
		IF	trap#nil	THEN	RETURN trap	END;
		exp:=obj; env:=nil; val:=nil; fun:=nil; first:=nil; last:=nil; unev:=nil; aux:=nil; cont:=stReturn; state:=eval; trap:=nil;
		LOOP
			CASE	state	OF
			| stAppNoArg:	AppNoArg()
			| stEvalArgs:	EvalArgs()
			| stSeqCont:	SeqCont()
			| stIfCont:	IfCont()
			| stSetGlobCont:	SetGlobCont()
			| stSetLocCont:	SetLocCont()
			| stGenAccArg:	GenAccArg()
			| stAppGen:	AppGen()
			| stGenAccLast:	GenAccLast()
			| stAppU:	AppU()
			| stAppMon:	AppMon()
			| stDyAccFirst :	DyAccFirst()
			| stAppDy:	AppDy()
			| stAppCont: AppCont()
			| stAndCont:	AndCont()
			| stOrCont:	OrCont()
			| stMapCont:	MapCont()
			| stMapEnd:	MapEnd()

			| eval:	EvalDsp()
			| stReturn:	EXIT
			END
		END;
		IF	trap#nil	THEN	RETURN trap	ELSE	RETURN val	END
	END Eval;


BEGIN
	rand:=1; sp:=illRef; mrk:=markT*tBits; trap:=nil;
	NEW(ca); NEW(cd); NEW(oldCa); NEW(oldCd); NEW(globalVa); free:=1;
	ca.r[0]:=illRef; cd.r[0]:=illRef; oldCa.r[0]:=illRef; oldCd.r[0]:=illRef
END SchemeMachine.