/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	nfs_gateway.c,v $
 * Revision 2.3  89/08/02  08:01:35  jsb
 * 	Replaced AFS_KALLOCs with osi_Allocs.
 * 	[89/07/31  18:21:02  jsb]
 * 
 * Revision 2.2  89/06/03  15:31:24  jsb
 * 	This file contains versions of nfs routines that have been modified
 * 	for gateway support, allowing gateway support without modifying
 * 	nfs source files. (This is mostly for the ITC's benefit.)
 * 	[89/06/02  14:35:15  jsb]
 * 
 */
/*
 *	File:	nfs_gateway.c
 *	Author:	Joseph S. Barrera III
 *
 *	Copyright (C) 1989, Joseph S. Barrera III
 *
 *	Support for afs service to nfs-only machines.
 *
 */
/* @(#)nfs.h		1.2 86/11/17 NFSSRC */
/* @(#)nfs_xdr.c	1.6 87/03/02 NFSSRC */
/* @(#)nfs_server.c	1.6 87/03/02 NFSSRC */
/*
 * Copyright (c) 1986 by Sun Microsystems, Inc.
 */

#include <afs/param.h>
#ifdef	AFS_GATEWAY
#define NFSSERVER
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/buf.h>
#ifdef	AFS_MACH_ENV
#include <sys/inode.h>
#else	AFS_MACH_ENV
#include <vfs/vfs.h>
#include <vfs/vnode.h>
#include <vfs/pathname.h>
#endif	AFS_MACH_ENV
#include <sys/uio.h>
#include <sys/file.h>
#include <sys/socketvar.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <rpc/auth.h>
#include <rpc/auth_unix.h>
#include <rpc/svc.h>
#include <rpc/xdr.h>
#include <nfs/nfs.h>
#include <sys/mbuf.h>

/*
 * rpc service program version range supported
 */
#define VERSIONMIN	2
#define VERSIONMAX	2

struct vnode	*fhtovp();
struct file	*getsock();

#ifdef	AFS_VFS40
/*
 *  VFS40 directory reading stuff may well work with afs.
 *  Thus we probably don't need to fixup the directory reading routines.
 */
#define	afs_rfs_readdir_fixup()
#else	AFS_VFS40
/*
 *  define new versions of:
 *
 *	struct nfsrdok
 *	struct nfsrddirres
 *	xdr_putrddirres()
 *	rfs_readdir()
 *	rfs_rddirfree()
 *
 *  which work with afs directories. Afs_rfs_readdir_fixup then replaces
 *  the old versions with the new ones.
 */

/*
 * NFS_OK part of readdir result. (AFS_GATEWAY: new rdok_offlist field)
 */
struct afs_nfsrdok {
	u_long	rdok_offset;		/* next offset (opaque) */
	u_long	rdok_size;		/* size in bytes of entries */
	bool_t	rdok_eof;		/* true if last entry is in result*/
	struct direct *rdok_entries;	/* variable number of entries */
	u_long	*rdok_offlist;		/* list of offsets for each entry */
};

/*
 * Readdir result. (AFS_GATEWAY: use afs_nfsrdok instead of nfsrdok)
 */
struct afs_nfsrddirres {
	u_long		rd_bufsize;	/* size of client request (not xdr'ed)*/
	enum nfsstat	rd_status;
	union {
		struct afs_nfsrdok rd_rdok_u;
	} rd_u;
};

#define rd_offlist	rd_u.rd_rdok_u.rdok_offlist

/*
 *  xdr_putrddirres.
 *
 *  AFS_GATEWAY: use offlist to send correct offsets for each directory
 *  entry. The code used to calculate offsets by summing record lengths,
 *  which doesn't work with afs directories.
 */

#define	nextdp(dp)	((struct direct *)((int)(dp) + (dp)->d_reclen))
#undef DIRSIZ
#define DIRSIZ(dp)	(sizeof(struct direct) - MAXNAMLEN + (dp)->d_namlen)

/*
 * ENCODE ONLY
 */
bool_t
afs_xdr_putrddirres(xdrs, rd)
	XDR *xdrs;
	struct afs_nfsrddirres *rd;
{
	struct direct *dp;
	char *name;
	int size;
	int xdrpos;
	u_int namlen;
	u_long offset;
#ifdef	AFS_GATEWAY
	u_long *offlist;
#endif	AFS_GATEWAY
	bool_t true = TRUE;
	bool_t false = FALSE;

#ifdef NFSDEBUG
	dprint(nfsdebug, 6, "afs_xdr_putrddirres: %s size %d offset %d\n",
	    xdropnames[(int)xdrs->x_op], rd->rd_size, rd->rd_offset);
#endif
	if (xdrs->x_op != XDR_ENCODE) {
		return (FALSE);
	}
	if (!xdr_enum(xdrs, (enum_t *)&rd->rd_status)) {
		return (FALSE);
	}
	if (rd->rd_status != NFS_OK) {
		return (TRUE);
	}

	xdrpos = XDR_GETPOS(xdrs);
#ifdef	AFS_GATEWAY
	offlist = rd->rd_offlist;
	if (offlist == NULL) {
		offset = rd->rd_offset;
	}
	for (size = rd->rd_size, dp = rd->rd_entries;
#else	AFS_GATEWAY
	for (offset = rd->rd_offset, size = rd->rd_size, dp = rd->rd_entries;
#endif	AFS_GATEWAY
	     size > 0;
	     size -= dp->d_reclen, dp = nextdp(dp) ) {
		if (dp->d_reclen == 0 || DIRSIZ(dp) > dp->d_reclen) {
#ifdef NFSDEBUG
			dprint(nfsdebug, 2, "afs_xdr_putrddirres: bad directory\n");
#endif
			return (FALSE);
		}
#ifdef	AFS_GATEWAY
		if (offlist) {
			offset = *offlist++;
		} else {
			offset += dp->d_reclen;
		}
#else	AFS_GATEWAY
		offset += dp->d_reclen;
#endif	AFS_GATEWAY
#ifdef NFSDEBUG
		dprint(nfsdebug, 10,
		    "afs_xdr_putrddirres: entry %d %s(%d) %d %d %d %d\n",
		    dp->d_fileno, dp->d_name, dp->d_namlen, offset,
		    dp->d_reclen, XDR_GETPOS(xdrs), size);
#endif
		if (dp->d_fileno == 0) {
			continue;
		}
		name = dp->d_name;
		namlen = dp->d_namlen;
		if (!xdr_bool(xdrs, &true) ||
		    !xdr_u_long(xdrs, &dp->d_fileno) ||
		    !xdr_bytes(xdrs, &name, &namlen, NFS_MAXNAMLEN) ||
		    !xdr_u_long(xdrs, &offset) ) {
			return (FALSE);
		}
		if (XDR_GETPOS(xdrs) - xdrpos >= rd->rd_bufsize) {
			rd->rd_eof = FALSE;
			break;
		}
	}
	if (!xdr_bool(xdrs, &false)) {
		return (FALSE);
	}
	if (!xdr_bool(xdrs, &rd->rd_eof)) {
		return (FALSE);
	}
	return (TRUE);
}

int
afs_rfs_readdir(rda, rd)
	struct nfsrddirargs *rda;
	register struct afs_nfsrddirres  *rd;
{
	int error;
	struct iovec iov;
	struct uio uio;
	register struct vnode *vp;

	/* afs_rfs_rddirfree examines this, so make sure it's always set */
	rd->rd_offlist = 0;
#ifdef	NFSDEBUG
	dprint(nfsdebug, 4, "afs_rfs_readdir fh %x %x %d count %d\n",
	    rda->rda_fh.fh_fsid.val[0], rda->rda_fh.fh_fsid.val[1],
	    rda->rda_fh.fh_len, rda->rda_count);
#endif
	vp = fhtovp(&rda->rda_fh);
	if (vp == NULL) {
		rd->rd_entries = NULL;
		rd->rd_status = NFSERR_STALE;
		return;
	}
#ifdef	AFS_MACH_ENV
	if (vp->v_type != ITYPE_AFS) {
#else	AFS_MACH_ENV
	if (vp->v_op != afs_ops) {
#endif	AFS_MACH_ENV
		VN_RELE(vp);
		rd->rd_offlist = 0;
		return rfs_readdir(rda, rd);
	}
	/*
	 * check cd access to dir.  we have to do this here because
	 * the opendir doesn't go over the wire.
	 */
	error = VOP_ACCESS(vp, VEXEC, u.u_cred);
	if (error) {
		goto bad;
	}

	/*
	 * Allocate data for entries and offset list.
	 * This will be freed by afs_rfs_rddirfree.
	 */
	rd->rd_bufsize = (u_int)rda->rda_count;
	rd->rd_entries = (struct direct *) osi_Alloc(rd->rd_bufsize);
	rd->rd_offlist = (u_long *) osi_Alloc(rd->rd_bufsize);
	/*
	 *  Set up uio and iov.
	 */
	iov.iov_base = (caddr_t)rd->rd_entries;
	iov.iov_len = rda->rda_count;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_seg = UIO_SYSSPACE;
	uio.uio_offset = rda->rda_offset;
	uio.uio_resid = rda->rda_count;
	/*
	 *  Read the directory and set return values.  If we get an
	 *  error, or there was nothing left, we set rd->rd_size = 0.
	 *  Otherwise, set size appropriately, and set eof to FALSE.
	 *  In either case, jump to bad, which cleans up and
	 *  returns properly.
	 */
	error = afs_readdir_with_offlist(vp, &uio, u.u_cred, rd->rd_offlist);
	if (error || uio.uio_resid == rda->rda_count) {
		rd->rd_size = 0;
	} else {
		rd->rd_size = rda->rda_count - uio.uio_resid;
		rd->rd_eof = FALSE;
	}
bad:
	rd->rd_status = puterrno(error);
	VN_RELE(vp);
#ifdef	NFSDEBUG
	dprint(nfsdebug, 5, "afs_rfs_readdir: returning %d\n", error);
#endif
}

afs_rfs_rddirfree(rd)
	struct afs_nfsrddirres *rd;
{
	if (rd->rd_offlist) {
		osi_Free((caddr_t)rd->rd_entries, (u_int)rd->rd_bufsize);
		osi_Free((caddr_t)rd->rd_offlist, (u_int)rd->rd_bufsize);
	} else {
		rfs_rddirfree(rd);
	}
}

extern struct rfsdisp {
	int	  (*dis_proc)();	/* proc to call */
	xdrproc_t dis_xdrargs;		/* xdr routine to get args */
	int	  dis_argsz;		/* sizeof args */
	xdrproc_t dis_xdrres;		/* xdr routine to put results */
	int	  dis_ressz;		/* size of results */
	int	  (*dis_resfree)();	/* frees space allocated by proc */
} rfsdisptab[][RFS_NPROC];


void
afs_rfs_readdir_fixup()
{
	struct rfsdisp *readdirvec;

	readdirvec = &rfsdisptab[NFS_VERSION - VERSIONMIN][RFS_READDIR];
	readdirvec->dis_proc = afs_rfs_readdir;
	readdirvec->dis_xdrres = afs_xdr_putrddirres;
	readdirvec->dis_ressz = sizeof(struct afs_nfsrddirres);
	readdirvec->dis_resfree = afs_rfs_rddirfree;
}
#endif	AFS_VFS40

extern int nobody;
extern int nfs_portmon;
extern int rfssize;
extern struct {
	int	ncalls;		/* number of calls received */
	int	nbadcalls;	/* calls that failed */
	int	reqs[32];	/* count for each request */
} svstat;
extern int nullfree();

void
afs_rfs_dispatch(req, xprt)
	struct svc_req *req;
	register SVCXPRT *xprt;
{
	int which;
	int vers;
	caddr_t	*args = NULL;
	caddr_t	*res = NULL;
	register struct rfsdisp *disp;
	struct authunix_parms *aup;
	register int i;
	struct ucred *tmpcr;
	struct ucred *newcr = NULL;
	int error;

	svstat.ncalls++;
	error = 0;
	which = req->rq_proc;
	if (which < 0 || which >= RFS_NPROC) {
#ifdef	NFSDEBUG
		dprint(nfsdebug, 2,
		    "afs_rfs_dispatch: bad proc %d\n", which);
#endif
		svcerr_noproc(req->rq_xprt);
		error++;
		goto done;
	}
	vers = req->rq_vers;
	if (vers < VERSIONMIN || vers > VERSIONMAX) {
#ifdef	NFSDEBUG
		dprint(nfsdebug, 2,
		    "afs_rfs_dispatch: bad vers %d low %d high %d\n",
		    vers, VERSIONMIN, VERSIONMAX);
#endif
		svcerr_progvers(req->rq_xprt, (u_long)VERSIONMIN,
		    (u_long)VERSIONMAX);
		error++;
		goto done;
	}
	vers -= VERSIONMIN;
	disp = &rfsdisptab[vers][which];

	/*
	 * Clean up as if a system call just started
	 */
	u.u_error = 0;

	/*
	 * Allocate args struct and deserialize into it.
	 */
	args = (caddr_t *)rfsget();
	bzero((caddr_t)args, (u_int)rfssize);
	if ( ! SVC_GETARGS(xprt, disp->dis_xdrargs, args)) {
		svcerr_decode(xprt);
		error++;
		goto done;
	}

	/*
	 * Check for unix style credentials
	 */
	if (req->rq_cred.oa_flavor != AUTH_UNIX && which != RFS_NULL) {
		svcerr_weakauth(xprt);
		error++;
		goto done;
	}

	if (nfs_portmon) {
		/*
		* Check for privileged port number
		*/
       	static count = 0;
		if (ntohs(xprt->xp_raddr.sin_port) >= IPPORT_RESERVED) {
			svcerr_weakauth(xprt);
			if (count == 0) {
				printf("NFS request from unprivileged port, ");
				printf("source IP address = %x\n",
					xprt->xp_raddr.sin_addr.s_addr);
			}
			count++;
			count %= 256;
			error++;
			goto done;
		}
	}


	/*
	 * Set uid, gid, and gids to auth params
	 */
	if (which != RFS_NULL) {
		aup = (struct authunix_parms *)req->rq_clntcred;
		newcr = crget();
#ifdef	AFS_GATEWAY
		afs_gateway_set_hostaddr(newcr,xprt->xp_raddr.sin_addr.s_addr);
#endif	AFS_GATEWAY
		if (aup->aup_uid == 0) {
			/*
			 * root over the net becomes other on the server (uid -2)
			 */
			newcr->cr_uid = (uid_t)nobody;
		} else {
			newcr->cr_uid = (uid_t)aup->aup_uid;
		}
		newcr->cr_gid = (gid_t)aup->aup_gid;
		for (i=0; i < aup->aup_len; i++)
			newcr->cr_groups[i] = (gid_t)aup->aup_gids[i];
		for (; i < NGROUPS; i++)
			newcr->cr_groups[i] = NOGROUP;
		tmpcr = u.u_cred;
		u.u_cred = newcr;
	}

	/*
	 * Allocate results struct.
	 */
	res = (caddr_t *)rfsget();
	bzero((caddr_t)res, (u_int)rfssize);

	svstat.reqs[which]++;

	/*
	 * Call service routine with arg struct and results struct
	 */
	(*disp->dis_proc)(args, res, req);

done:
	/*
	 * Free arguments struct
	 */
	if (!SVC_FREEARGS(xprt, disp->dis_xdrargs, args) ) {
		error++;
	}
	if (args != NULL) {
		rfsput((struct rfsspace *)args);
	}

	/*
	 * Serialize and send results struct
	 */
	if (!error) {
		if (!svc_sendreply(xprt, disp->dis_xdrres, (caddr_t)res)) {
			error++;
		}
	}

	/*
	 * Free results struct
	 */
	if (res != NULL) {
		if ( disp->dis_resfree != nullfree ) {
			(*disp->dis_resfree)(res);
		}
		rfsput((struct rfsspace *)res);
	}
	/*
	 * restore original credentials
	 */
	if (newcr) {
		u.u_cred = tmpcr;
#ifdef	AFS_GATEWAY
		afs_gateway_unset_hostaddr(newcr);
#endif	AFS_GATEWAY
		crfree(newcr);
	}
	svstat.nbadcalls += error;
}

/*
 * afs_nfs_svc: exactly the same as nfs_svc, except that we register
 * afs_rfs_dispatch instead of rfs_dispatch. We could avoid duplicating this
 * code if we were willing to live with race conditions (but we aren't).
 */
/*
 * NFS Server system call.
 * Does all of the work of running a NFS server.
 * sock is the fd of an open UDP socket.
 */
afs_xnfs_svc()
{
	struct a {
		int     sock;
	} *uap = (struct a *)u.u_ap;
	struct vnode	*rdir;
	struct vnode	*cdir;
	struct socket   *so;
	struct file	*fp;
	SVCXPRT *xprt;
	u_long vers;
	int	error;
 
	fp = getsock(uap->sock);
	if (fp == 0) {
		u.u_error = EBADF;
		return;
	}
	so = (struct socket *)fp->f_data;

	/*
	 *	  Allocate extra space for this socket, to minimize
	 *	lost requests for NFS.  We don't want everyone to do
	 *	this, so do it here, rather than in udp_usrreq().
	 */

	error = soreserve(so, 18000, 18000 + 2 *(sizeof(struct sockaddr)));
	if (error)	{
		u.u_error = error;
		return;
	}
	 
	/*
	 * Be sure that rdir (the server's root vnode) is set.
	 * Save the current directory and restore it again when
	 * the call terminates.  rfs_lookup uses u.u_cdir for lookupname.
	 */
	rdir = ITOV(u.u_rdir);
	cdir = ITOV(u.u_cdir);
	if (ITOV(u.u_rdir) == (struct vnode *)0) {
		u.u_rdir = u.u_cdir;
	}
	xprt = svckudp_create(so, NFS_PORT);
	for (vers = VERSIONMIN; vers <= VERSIONMAX; vers++) {
#ifdef	AFS_GATEWAY
		(void) svc_register(xprt, NFS_PROGRAM, vers, afs_rfs_dispatch,
#else	AFS_GATEWAY
		(void) svc_register(xprt, NFS_PROGRAM, vers, rfs_dispatch,
#endif	AFS_GATEWAY
		    FALSE);
	}
	if (setjmp(&u.u_qsave)) {
		for (vers = VERSIONMIN; vers <= VERSIONMAX; vers++) {
			svc_unregister(NFS_PROGRAM, vers);
		}
		SVC_DESTROY(xprt);
		u.u_error = EINTR;
	} else {
#ifdef	AFS_GATEWAY
		afs_rfs_readdir_fixup();
#endif	AFS_GATEWAY
		svc_run(xprt);  /* never returns */
	}
	u.u_rdir = VTOI(rdir);
	u.u_cdir = VTOI(cdir);
}
#endif	AFS_GATEWAY
