/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_pioctl.c,v $
 * Revision 2.9  89/08/02  07:59:52  jsb
 * 	Call afs_GCUserData every sixteenth time PSetTokens is called
 * 	(to allow garbage to be collected as quickly as it is generated).
 * 	[89/08/01  14:55:16  jsb]
 * 
 * Revision 2.8  89/06/24  23:58:31  jsb
 * 	Newer ITC sources.
 * 	[89/06/24  23:41:15  jsb]
 * 
 * Revision 2.7  89/06/03  15:28:57  jsb
 * 	Merged with newer ITC sources; totally redid gateway support.
 * 	[89/05/27  11:50:32  jsb]
 * 
 * Revision 2.6  89/04/22  15:14:55  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987, 1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

#include <afs/param.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef	AFS_AUX_ENV
#include <sys/mmu.h>
#include <sys/seg.h>
#include <sys/sysmacros.h>
#include <sys/signal.h>
#include <sys/errno.h>
#endif
#if	!defined(AFS_IBM_ENV) || !defined(sys_rt_r3)
#include <sys/time.h>
#endif	AFS_IBM_ENV
#ifdef	AFS_AIX_ENV
#include <sys/errno.h>
#else
#include <sys/kernel.h>
#endif
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/uio.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vfs.h>
#include <afs/gfs_vnode.h>
#else
#ifdef	AFS_MACH_ENV
#include <vfs/vfs.h>
#include <vfs/vnode.h>
#include <sys/inode.h>
#else	AFS_MACH_ENV
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV
#ifdef	AFS_AIX_ENV
#include <afs/aix_vfs.h>
#endif
#include <netinet/in.h>
#include <sys/mbuf.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <sys/ioctl.h>

#include <afs/osi.h>
#include <rx/rx.h>

#include <afs/lock.h>
#include <afs/volerrors.h>
#include <afsint/afsint.h>
#include <afs/afs.h>
#include <afs/prs_fs.h>
#include <afs/dir.h>

struct VenusFid afs_rootFid;
long afs_waitForever=0;
short afs_waitForeverCount = 0;
extern struct osi_dev cacheDev;
extern long afs_cacheBlocks;
extern struct cell *afs_cells;
extern long afs_origCacheBlocks;
extern char *afs_indexFlags; /* only one: is there data there? */
extern struct unixuser *afs_users[NUSERS];
extern struct server *afs_servers[NSERVERS];
extern struct afs_lock afs_xserver;
extern struct afs_lock afs_xcell;
extern struct afs_lock afs_xconn;
extern struct afs_lock afs_xuser;
extern struct afs_lock afs_xdcache;		    /* lock: alloc new disk cache entries */
extern long afs_mariner, afs_marinerHost;

static int PBogus(), PSetAcl(), PGetAcl(), PSetTokens(), PGetVolumeStatus();
static int PSetVolumeStatus(), PFlush(), PNewStatMount(), PGetTokens(), PUnlog();
static int PCheckServers(), PCheckVolNames(), PCheckAuth(), PFindVolume();
static int PViceAccess(), PSetCacheSize(), Prefetch();
static int PRemoveCallBack(), PNewCell(), PListCells(), PRemoveMount();
static int PMariner(), PGetUserCell(), PGetWSCell(), PGetFileCell();
static int PVenusLogging(), PNoop(), PSetCellStatus(), PGetCellStatus();
static int PFlushVolumeData();
#ifdef	AFS_GATEWAY
extern int PSetSysName(), PSetClientContext();
#endif	AFS_GATEWAY

static int (*(pioctlSw[]))() = {
    PBogus,			/* 0 */
    PSetAcl,			/* 1 */
    PGetAcl,			/* 2 */
    PSetTokens,			/* 3 */
    PGetVolumeStatus,		/* 4 */
    PSetVolumeStatus,		/* 5 */
    PFlush,			/* 6 */
    PBogus,			/* 7 */
    PGetTokens,			/* 8 */
    PUnlog,			/* 9 */
    PCheckServers,		/* 10 */
    PCheckVolNames,		/* 11 */
    PCheckAuth,			/* 12 */
    PBogus,			/* 13 -- used to be quick check time */
    PFindVolume,		/* 14*/
    PBogus,			/* 15 -- prefetch is now special-cased; see pioctl code! */
    PBogus,			/* 16 -- used to be testing code */
    PNoop,			/* 17 -- used to be enable group */
    PNoop,		    	/* 18 -- used to be disable group */
    PBogus,			/* 19 -- used to be list group */
    PViceAccess,		/* 20 */
    PUnlog,			/* 21 -- unlog *is* unpag in this system */
    PBogus,			/* 22 -- used to be fast getwd */
    PBogus,			/* 23 -- used to be waitforever */
    PSetCacheSize,		/* 24 */
    PRemoveCallBack,		/* 25 -- flush only the callback */
    PNewCell,			/* 26 */
    PListCells,		    	/* 27 */
    PRemoveMount,		/* 28 -- delete mount point */
    PNewStatMount,		/* 29 -- new style mount point stat */
    PGetFileCell,		/* 30 -- get cell name for input file */
    PGetWSCell,			/* 31 -- get cell name for workstation */
    PMariner,			/* 32 - set/get mariner host */
    PGetUserCell,		/* 33 -- get cell name for user */
    PVenusLogging,		/* 34 -- Enable/Disable logging */
    PGetCellStatus,		/* 35 */
    PSetCellStatus,		/* 36 */
    PFlushVolumeData,		/* 37 -- flush all data from a volume */
#ifdef	AFS_GATEWAY
    PSetSysName,		/* 38 -- set system name */
    PSetClientContext,		/* 39 -- set hostaddr, cred; for pagd */
#endif	AFS_GATEWAY
};

/* stolen from vice.h */
#ifdef MM_XXX
#ifdef __STDC__
#define _AFSIOCTL(id)  ((unsigned int ) _IOW('V', id, struct afs_ioctl))
#else
#define _AFSIOCTL(id)  ((unsigned int ) _IOW(V, id, struct afs_ioctl))
#endif
#else
#define _AFSIOCTL(id)  ((unsigned int ) _IOW(V, id, struct afs_ioctl))
#endif MM_XXX
#define _VALIDAFSIOCTL(com) (com >= _AFSIOCTL(0) && com <= _AFSIOCTL(255))

HandleIoctl(avc, afile, acom, adata)
    register struct vcache *avc;
    struct file *afile;
    register long acom;
    struct afs_ioctl *adata; {
    register long code;

    code = 0;

    switch(acom & 0xff) {
	case 1:
	    avc->states |= CSafeStore;
	    break;

	/* case 2 used to be abort store, but this is no longer provided,
	    since it is impossible to implement under normal Unix.
	*/
	
	case 3: {
	    /* return the name of the cell this file is open on */
	    register struct cell *tcell;
	    register long i;
	    
	    tcell = afs_GetCell(avc->fid.Cell);
	    if (tcell) {
		i = strlen(tcell->cellName) + 1;    /* bytes to copy out */
		if (i > adata->out_size) {
		    /* 0 means we're not interested in the output */
		    if (adata->out_size != 0) code = EFAULT;
		}
		else {
		    /* do the copy */
		    code = copyout(tcell->cellName, adata->out, i);
		}
	    }
	    else code = ENOTTY;
	}
	break;

	default:
	    code = EINVAL;
	    break;
    }
    return code;		/* so far, none implemented */
}


#ifdef	AFS_AIX_ENV
/* For aix we don't temporarily bypass ioctl(2) but rather do our thing directly in the vnode layer call, VNOP_IOCTL; thus afs_ioctl is now called from afs_gn_ioctl. */
afs_ioctl(cmd, arg)
int	cmd;
int	arg;
{
    struct afs_ioctl data;
    struct vcache *tvc;
    int error = 0;

    if (((cmd >> 8) & 0xff) == 'V') {
	/* This is a VICEIOCTL call */
	error = copyin(arg, (caddr_t) &data, sizeof(data));
	if (error)
	    return(error);
	error = HandleIoctl(tvc, (struct file *)0/*Not used*/, cmd, &data);
	return(error);
    } else {
	/* No-op call; just return. */
	return(ENOTTY);
    }
}

#else

/* unlike most calls here, this one uses u.u_error to return error conditions,
    since this is really an intercepted chapter 2 call, rather than a vnode
    interface call.
*/
afs_xioctl () {
    struct a {
	int fd;
	int com;
	caddr_t cmarg;
    } *uap;
    register struct file *fd;
    struct vcache *tvc;
    struct afs_ioctl data;
    int ioctlDone;
    register int cmd;
    
    ioctlDone = 0;
    uap = (struct a *)u.u_ap;

    fd = getf(uap->fd);
    if (!fd) return;

    cmd = uap->com;

    /* first determine whether this is any sort of vnode */
    if (fd->f_type == DTYPE_VNODE) {
	/* good, this is a vnode; next see if it is an AFS vnode */
	tvc = (struct vcache *) fd->f_data;	/* valid, given a vnode */
#ifdef	AFS_MACH_ENV
	if (tvc && tvc->v.v_type == ITYPE_AFS) {
#else	AFS_MACH_ENV
	if (tvc && tvc->v.v_op == afs_ops) {
#endif	AFS_MACH_ENV
	    /* This is an AFS vnode */
	    if (((cmd >> 8) & 0xff) == 'V') {
		/* this is a VICEIOCTL call */
		u.u_error = copyin(uap->cmarg, (caddr_t) &data, sizeof (data));
		if (u.u_error)
		    return;
		u.u_error = HandleIoctl(tvc, fd, cmd, &data);
		ioctlDone = 1;
	    }
	}
    }
    if (!ioctlDone) ioctl();
    return;
}
#endif	AFS_AIX_ENV

afs_pioctl() {
    extern rmt_ioctl1();
    struct a {
	char	*path;
	int	cmd;
	caddr_t cmarg;
	int	follow;
    } *uap;
    struct afs_ioctl data;
    int com;
    struct vnode *vp;

    uap = (struct a *) u.u_ap;
    if (uap->follow) uap->follow = 1;	/* compat. with old venus */
    com = uap->cmd;
    if (! _VALIDAFSIOCTL(com)) {
	u.u_error = EINVAL;
	return;
    }
    u.u_error = copyin(uap->cmarg, (caddr_t) &data, sizeof (data));
    if (u.u_error)
    	return;
    if ((com & 0xff) == 15) {
	/* special case prefetch so entire pathname eval occurs in helper process.
	    otherwise, the pioctl call is essentially useless */
	return Prefetch(uap->path, &data, uap->follow, u.u_cred);
    }
    if (uap->path) {
	u.u_error = gop_lookupname(uap->path, AFS_UIOUSER, uap->follow,  (struct vnode **) 0, &vp);
	if (u.u_error)
	    return(u.u_error);
    }
    else vp = (struct vnode *) 0;

    /* now make the call if we were passed no file, or were passed an AFS file */
#ifdef	AFS_MACH_ENV
    if (!vp || vp->v_type == ITYPE_AFS) {
#else	AFS_MACH_ENV
    if (!vp || vp->v_op == afs_ops) {
#endif	AFS_MACH_ENV
	u.u_error = HandlePioctl(vp, com, &data, uap->follow, u.u_cred);
    }
    else u.u_error = EINVAL;	/* not in /afs */
    if (vp) VN_RELE(vp);	/* put vnode back */
}

HandlePioctl(avc, acom, ablob, afollow, acred)
    register struct vcache *avc;
    long acom;
    struct ucred *acred;
    register struct afs_ioctl *ablob;
    int afollow; {
    struct vrequest treq;
    register long code;
    register long function;
    long inSize, outSize;
    char *inData, *outData;
    if (afs_debug & AFSDEB_GENERAL) afs_dp("in afs_pioctl (%x), com %d\n", avc, acom);
    
    afs_InitReq(&treq, acred);
    function = acom & 0xff;
    if (function >= (sizeof(pioctlSw) / sizeof(char *))) {
	return EINVAL;	/* out of range */
    }
    inSize = ablob->in_size;
    if (inSize >= PIGGYSIZE) return E2BIG;
    inData = osi_AllocSendSpace();
    if (inSize > 0) {
	code = copyin(ablob->in, inData, inSize);
    }
    else code = 0;
    if (code) {
	osi_FreeSendSpace(inData);
	return code;
    }
    outData = osi_AllocSendSpace();
    outSize = 0;
    code = (*pioctlSw[function])(avc, function, &treq, inData, outData, inSize, &outSize);
    osi_FreeSendSpace(inData);
    if (code == 0 && ablob->out_size > 0) {
	if (outSize > ablob->out_size) outSize = ablob->out_size;
	if (outSize >= PIGGYSIZE) code = E2BIG;
	else code = copyout(outData, ablob->out, outSize);
    }
    osi_FreeSendSpace(outData);
    return afs_CheckCode(code, &treq);
}

static PSetAcl(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long code;
    struct conn *tconn;
    struct AFSOpaque acl;
    struct AFSVolSync tsync;
    struct AFSFetchStatus OutStatus;

    if (!avc) return EINVAL;
    if ((acl.AFSOpaque_len = strlen(ain)+1) > 1000) return EINVAL;
    acl.AFSOpaque_val = ain;
    do {
	tconn = afs_Conn (&avc->fid, areq);
	if (tconn) {
	    code = RXAFS_StoreACL(tconn->id, (struct AFSFid *) &avc->fid.Fid, &acl, &OutStatus, &tsync);
	}
	else code = -1;
    } while (afs_Analyze(tconn, code, &avc->fid, areq));
    /* now we've forgotten all of the access info */
    avc->callback = 0;
    return code;
};

static PGetAcl(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    struct AFSOpaque acl;
    struct AFSVolSync tsync;
    struct AFSFetchStatus OutStatus;
    long code;
    struct conn *tconn;

    if (!avc) return EINVAL;
    acl.AFSOpaque_val = aout;
    do {
	tconn = afs_Conn(&avc->fid, areq);
	if (tconn) {
	    *aout = 0;
	    code = RXAFS_FetchACL(tconn->id, (struct AFSFid *) &avc->fid.Fid, &acl, &OutStatus, &tsync);
	}
	else code = -1;
    } while (afs_Analyze(tconn, code, &avc->fid, areq));
    if (code == 0) *aoutSize = (acl.AFSOpaque_len == 0 ? 1 : acl.AFSOpaque_len);
    return code;
}

static PNoop() {
    return 0;
}

static PBogus() {
    return EINVAL;
}

static PGetFileCell(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct cell *tcell;

    if (!avc) return EINVAL;
    tcell = afs_GetCell(avc->fid.Cell);
    if (!tcell) return ESRCH;
    strcpy(aout, tcell->cellName);
    *aoutSize = strlen(aout) + 1;
    return 0;
}

static PGetWSCell(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct cell *tcell, *cellOne;

    ObtainReadLock(&afs_xcell);
    cellOne = (struct cell *) 0;
    for(tcell = afs_cells; tcell; tcell=tcell->next) {
	if (tcell->states & CPrimary) break;
	if (tcell->cell == 1) cellOne = tcell;
    }
    ReleaseReadLock(&afs_xcell);
    if (!tcell)	{	    /* no primary cell, use cell #1 */
	if (!cellOne) return ESRCH;
	tcell = cellOne;
    }
    strcpy(aout, tcell->cellName);
    *aoutSize = strlen(aout) + 1;
    return 0;
}

static PGetUserCell(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long i;
    register struct unixuser *tu;
    register struct cell *tcell;

    /* return the cell name of the primary cell for this user */
    i = UHash(areq->uid);
    ObtainWriteLock(&afs_xuser);
    for(tu = afs_users[i]; tu; tu = tu->next) {
#ifdef	AFS_GATEWAY
	if (tu->uid == areq->uid && tu->primary && tu->hostaddr == areq->hostaddr) {
#else	AFS_GATEWAY
	if (tu->uid == areq->uid && tu->primary) {
#endif	AFS_GATEWAY
	    tu->refCount++;
	    ReleaseWriteLock(&afs_xuser);
	    break;
	}
    }
    if (tu) {
	tcell = afs_GetCell(tu->cell);
	afs_PutUser(tu);
	if (!tcell) return ESRCH;
	else {
	    strcpy(aout, tcell->cellName);
	    *aoutSize =	strlen(aout)+1;	    /* 1 for the null */
	}
    }
    else {
	ReleaseWriteLock(&afs_xuser);
	*aout = 0;
	*aoutSize = 1;
    }
    return 0;
}

static PSetTokens(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    long i;
    register struct unixuser *tu;
    struct ClearToken clear;
    register struct cell *tcell;
    char *stp;
    int stLen;
    long flag;
#ifdef	AFS_MACH_ENV
    static int callcount = 0;
#endif

    bcopy(ain, &i, sizeof(long));
    ain += sizeof(long);
    stp	= ain;	/* remember where the ticket is */
    stLen = i;
    ain	+= i;	/* skip over ticket */
    bcopy(ain, &i, sizeof(long));
    ain += sizeof(long);
    if (i != sizeof(struct ClearToken)) {
	return EINVAL;
    }
    bcopy(ain, &clear, sizeof(struct ClearToken));
    if (clear.AuthHandle == -1)	clear.AuthHandle = 999;	/* more rxvab compat crap */
    ain += sizeof(struct ClearToken);
    if (ainSize != 2*sizeof(long) + sizeof(struct SecretToken) + sizeof(struct ClearToken)) {
	/* still stuff left?  we've got primary flag and cell name.  Set these */
	bcopy(ain, &flag, sizeof(long));		/* primary id flag */
	ain += sizeof(long);			/* skip id field */
	/* rest is cell name, look it up */
	tcell = afs_GetCellByName(ain);
	if (tcell) {
	    i = tcell->cell;
	}
	else {
	    return ESRCH;
	}
    }
    else {
	/* default to cell 1, primary id */
	flag = 1;		/* primary id */
	i = 1;		/* cell number */
	tcell = afs_GetCell(1);
	if (!tcell) return EIO;
    }

    /* now we just set the tokens */
#ifdef	AFS_GATEWAY
    tu = afs_GetUser(areq->uid,	i, areq->hostaddr); /* i has the cell # */
#else	AFS_GATEWAY
    tu = afs_GetUser(areq->uid,	i); /* i has the cell # */
#endif	AFS_GATEWAY
    tu->vid = clear.ViceId;
    if (tu->stp != (char *) 0) {
	osi_Free (tu->stp, tu->stLen);
    }
    tu->stp = (char *) osi_Alloc(stLen);
    tu->stLen = stLen;
    bcopy(stp, tu->stp, stLen);
    tu->ct = clear;
    tu->states |= UHasTokens;
    tu->states &= ~UTokensBad;
    afs_SetPrimary(tu, flag);
    tu->tokenTime =osi_Time();
    afs_ResetUserConns(tu);
    afs_PutUser(tu);
#ifdef	AFS_MACH_ENV
    if (++callcount > 16) {
	afs_GCUserData(); /* don't let garbage pile up too quickly */
	callcount = 0;
    }
#endif
    return 0;
}

static PGetVolumeStatus(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    char volName[32];
    char offLineMsg[256];
    char motd[256];
    register struct conn *tc;
    register long code;
    struct VolumeStatus volstat;
    register char *cp;
    char *Name, *OfflineMsg, *MOTD;

    if (!avc) return EINVAL;
    Name = volName;
    OfflineMsg = offLineMsg;
    MOTD = motd;
    do {
	tc = afs_Conn(&avc->fid, areq);
	if (tc)
	    code = RXAFS_GetVolumeStatus(tc->id, avc->fid.Fid.Volume, &volstat,
					&Name, &OfflineMsg, &MOTD);
	else code = -1;
    } while (afs_Analyze(tc, code, &avc->fid, areq));
    if (code) return code;
    /* Copy all this junk into msg->im_data, keeping track of the lengths. */
    cp = aout;
    bcopy(&volstat, cp, sizeof(VolumeStatus));
    cp += sizeof(VolumeStatus);
    strcpy(cp, volName);
    cp += strlen(volName)+1;
    strcpy(cp, offLineMsg);
    cp += strlen(offLineMsg)+1;
    strcpy(cp, motd);
    cp += strlen(motd)+1;
    *aoutSize = (cp - aout);
    return 0;
}

static PSetVolumeStatus(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    char volName[32];
    char offLineMsg[256];
    char motd[256];
    register struct conn *tc;
    register long code;
    struct AFSFetchVolumeStatus volstat;
    struct AFSStoreVolumeStatus storeStat;
    register char *cp;

    if (!avc) return EINVAL;
    /* Copy the junk out, using cp as a roving pointer. */
    cp = ain;
    bcopy(cp, &volstat, sizeof(AFSFetchVolumeStatus));
    cp += sizeof(AFSFetchVolumeStatus);
    strcpy(volName, cp);
    cp += strlen(volName)+1;
    strcpy(offLineMsg, cp);
    cp +=  strlen(offLineMsg)+1;
    strcpy(motd, cp);
    storeStat.Mask = 0;
    if (volstat.MinQuota != -1) {
	storeStat.MinQuota = volstat.MinQuota;
	storeStat.Mask |= AFS_SETMINQUOTA;
    }
    if (volstat.MaxQuota != -1) {
	storeStat.MaxQuota = volstat.MaxQuota;
	storeStat.Mask |= AFS_SETMAXQUOTA;
    }
    do {
	tc = afs_Conn(&avc->fid, areq);
	if (tc)
	    code = RXAFS_SetVolumeStatus(tc->id, avc->fid.Fid.Volume,
					&storeStat, volName, offLineMsg, motd);
	else code = -1;
    } while (afs_Analyze(tc, code, &avc->fid, areq));
    if (code) return code;
    /* we are sending parms back to make compat. with prev system.  should
      change interface later to not ask for current status, just set new status */
    cp = aout;
    bcopy(&volstat, cp, sizeof(VolumeStatus));
    cp += sizeof(VolumeStatus);
    strcpy(cp, volName);
    cp += strlen(volName)+1;
    strcpy(cp, offLineMsg);
    cp += strlen(offLineMsg)+1;
    strcpy(cp, motd);
    cp += strlen(motd)+1;
    *aoutSize = cp - aout;
    return 0;
}

static PFlush(avc, afun, areq, ain, aout, ainSize, aoutSize)
    register struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {

    if (!avc) return EINVAL;
    ObtainWriteLock(&avc->lock);
    avc->states	&= ~CStatd;	/* next reference will re-stat cache entry */
    /* now find the disk cache entries */
    afs_TryToSmush(avc);
    ReleaseWriteLock(&avc->lock);
    return 0;
}

static PNewStatMount(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long code;
    register struct vcache *tvc;
    register struct dcache *tdc;
    struct VenusFid tfid;
    long offset, len;

    if (!avc) return EINVAL;
    code = afs_VerifyVCache(avc, areq);
    if (code) return code;
    if (vType(avc) != VDIR) {
	return ENOTDIR;
    }
    tdc = afs_GetDCache(avc, 0, areq, &offset, &len, 1);
    if (!tdc) return ENOENT;
    code = dir_Lookup(&tdc->f.inode, ain, &tfid.Fid);
    if (code) {
	afs_PutDCache(tdc);
	return code;
    }
    tfid.Cell = avc->fid.Cell;
    tfid.Fid.Volume = avc->fid.Fid.Volume;
    afs_PutDCache(tdc);	    /* we're done with the data */
    tvc = afs_GetVCache(&tfid, areq);
    if (!tvc) return ENOENT;
    if (vType(tvc) != VLNK) {
	afs_PutVCache(tvc);
	return EINVAL;
    }
    ObtainWriteLock(&tvc->lock);
    code = afs_HandleLink(tvc, areq);
    if (code == 0) {
	if (tvc->linkData) {
	    /* we have the data */
	    strcpy(aout, tvc->linkData);
	    *aoutSize = strlen(tvc->linkData)+1;
	}
	else code = EIO;
    }
    ReleaseWriteLock(&tvc->lock);
    afs_PutVCache(tvc);
    return code;
}

static PGetTokens(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct cell *tcell;
    register long i;
    register struct unixuser *tu;
    register char *cp;
    long iterator;
    int newStyle;

    /* weird interface.  If input parameter is present, it is an integer and
	we're supposed to return the parm'th tokens for this unix uid.
	If not present, we just return tokens for cell 1.
	If counter out of bounds, return EDOM.
	If no tokens for the particular cell, return ENOTCONN.
	Also, if this mysterious parm is present, we return, along with the
	tokens, the primary cell indicator (a long 0) and the cell name
	at the end, in that order.
    */
    if (newStyle = (ainSize > 0)) {
	bcopy(ain, &iterator, sizeof(long));
    }
    i = UHash(areq->uid);
    ObtainReadLock(&afs_xuser);
    for(tu = afs_users[i]; tu; tu=tu->next) {
	if (newStyle) {
#ifdef	AFS_GATEWAY
	    if (tu->uid == areq->uid && (tu->states & UHasTokens) && tu->hostaddr == areq->hostaddr) {
#else	AFS_GATEWAY
	    if (tu->uid == areq->uid && (tu->states & UHasTokens)) {
#endif	AFS_GATEWAY
		if (iterator-- == 0) break;	/* are we done yet? */
	    }
	}
	else {
#ifdef	AFS_GATEWAY
	    if (tu->uid == areq->uid && tu->cell == 1 && tu->hostaddr == areq->hostaddr) break;
#else	AFS_GATEWAY
	    if (tu->uid == areq->uid && tu->cell == 1) break;
#endif	AFS_GATEWAY
	}
    }
    if (tu) tu->refCount++;
    ReleaseReadLock(&afs_xuser);
    if (!tu) {
	return EDOM;
    }
    if ((tu->states & UHasTokens) == 0) {
	afs_PutUser(tu);
	return ENOTCONN;
    }
    /* use iterator for temp */
    cp = aout;
    iterator = tu->stLen;	/* for compat, we try to return 56 byte tix if they fit */
    if (iterator < 56) iterator	= 56;	/* # of bytes we're returning */
    bcopy(&iterator, cp, sizeof(long));
    cp += sizeof(long);
    bcopy(tu->stp, cp, tu->stLen);	/* copy out st */
    cp += iterator;
    iterator = sizeof(struct ClearToken);
    bcopy(&iterator, cp, sizeof(long));
    cp += sizeof(long);
    bcopy(&tu->ct, cp, sizeof(struct ClearToken));
    cp += sizeof(struct ClearToken);
    if (newStyle) {
	/* put out primary id and cell name, too */
	iterator = tu->primary;
	bcopy(&iterator, cp, sizeof(long));
	cp += sizeof(long);
	tcell = afs_GetCell(tu->cell);
	if (tcell) {
	    strcpy(cp, tcell->cellName);
	    cp += strlen(tcell->cellName)+1;
	}
	else *cp++ = 0;
    }
    *aoutSize = cp - aout;
    afs_PutUser(tu);
    return 0;
}

static PUnlog(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long i;
    register struct unixuser *tu;

    i = UHash(areq->uid);
    ObtainWriteLock(&afs_xuser);
    for(tu=afs_users[i]; tu; tu=tu->next) {
#ifdef	AFS_GATEWAY
	if (tu->uid == areq->uid && tu->hostaddr == areq->hostaddr) {
#else	AFS_GATEWAY
	if (tu->uid == areq->uid) {
#endif	AFS_GATEWAY
	    tu->vid = UNDEFVID;
	    tu->states &= ~UHasTokens;
	    /* security is not having to say you're sorry */
	    bzero(&tu->ct, sizeof(struct ClearToken));
	    afs_ResetUserConns(tu);
	}
    }
    ReleaseWriteLock(&afs_xuser);
    return 0;
}

static PMariner(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    long newHostAddr;
    long oldHostAddr;
    
    if (afs_mariner)
	bcopy(&afs_marinerHost, &oldHostAddr, sizeof(long));
    else
	oldHostAddr = 0xffffffff;   /* disabled */
    
    bcopy(ain, &newHostAddr, sizeof(long));
    if (newHostAddr == 0xffffffff) {
	/* disable mariner operations */
	afs_mariner = 0;
    }
    else if (newHostAddr) {
	afs_mariner = 1;
	afs_marinerHost = newHostAddr;
    }
    bcopy(&oldHostAddr, aout, sizeof(long));
    *aoutSize = sizeof(long);
    return 0;
}

static PCheckServers(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register char *cp;
    register int i;
    register struct server *ts;
    long temp;

    if (ainSize	> 0) {	/* compat, of course */
	bcopy(ain, &temp, sizeof(long));
    }
    else temp = 0;
    if (temp == 0) {
	afs_CheckServers(1);	/* check down servers */
	afs_CheckServers(0);	/* check up servers */
    }
    /* now return the current down server list */
    cp = aout;
    ObtainReadLock(&afs_xserver);
    for(i=0;i<NSERVERS;i++) {
	for(ts = afs_servers[i]; ts; ts=ts->next) {
	    if (ts->isDown) {
		bcopy(&ts->host, cp, sizeof(long));
		cp += sizeof(long);
	    }
	}
    }
    ReleaseReadLock(&afs_xserver);
    *aoutSize = cp - aout;
    return 0;
}

static PCheckVolNames(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    afs_CheckRootVolume();
    afs_CheckVolumeNames();
    return 0;
}

static PCheckAuth(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register int i;
    register struct server *ts;
    register struct conn *tc;
    register struct unixuser *tu;
    long retValue;

    retValue = 0;
#ifdef	AFS_GATEWAY
    tu = afs_GetUser(areq->uid,	1, areq->hostaddr);	/* check local cell authentication */
#else	AFS_GATEWAY
    tu = afs_GetUser(areq->uid,	1);	/* check local cell authentication */
#endif	AFS_GATEWAY
    if (!tu) retValue = EACCES;
    else {
	/* we have a user */
	ObtainReadLock(&afs_xserver);
	ObtainReadLock(&afs_xconn);
	/* any tokens set? */
	if ((tu->states	& UHasTokens) == 0) retValue = EACCES;
	/* all connections in cell 1 working? */
	for(i=0;i<NSERVERS;i++) {
	    for(ts = afs_servers[i]; ts; ts=ts->next) {
		for(tc = ts->conns; tc; tc=tc->next) {
		    if (tc->user == tu && (tu->states & UTokensBad)) retValue = EACCES;
		}
	    }
	}
	ReleaseReadLock(&afs_xserver);
	ReleaseReadLock(&afs_xconn);
	afs_PutUser(tu);
    }
    bcopy(&retValue, aout, sizeof(long));
    *aoutSize = sizeof(long);
    return 0;
}

static Prefetch(apath, adata, afollow, acred)
char *apath;
struct afs_ioctl *adata;
int afollow;
struct ucred *acred; {
    register char *tp;
    register long code;
    long bufferSize;

    if (!apath) return EINVAL;
    tp = osi_AllocSendSpace();
#if (defined(AFS_AUX_ENV) || defined(AFS_AIX_ENV))
    code = 0;
    /* Check error code possibility here? */
    bufferSize = bcopyin(apath, tp, 1024);
#else
    code = copyinstr(apath, tp, 1024, &bufferSize);
#endif
    if (code) {
	osi_FreeSendSpace(tp);
	return code;
    }
    if (afs_BBusy()) {	/* do this as late as possible */
	osi_FreeSendSpace(tp);
	return EWOULDBLOCK;	/* pretty close */
    }
    afs_BQueue(BOP_PATH, 0, 0, tp, 0);
    return 0;
}

static PFindVolume(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct volume *tvp;
    register struct server *ts;
    register long i;
    register char *cp;
    
    if (!avc) return EINVAL;
    tvp = afs_GetVolume(&avc->fid, areq);
    if (tvp) {
	cp = aout;
	for(i=0;i<MAXHOSTS;i++) {
	    ts = tvp->serverHost[i];
	    if (!ts) break;
	    bcopy(&ts->host, cp, sizeof(long));
	    cp += sizeof(long);
	}
	*aoutSize = cp - aout;
	afs_PutVolume(tvp);
	return 0;
    }
    return ENODEV;
}

static PViceAccess(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long code;
    long temp;
    
    if (!avc) return EINVAL;
    bcopy(ain, &temp, sizeof(long));
    code = afs_VerifyVCache(avc, areq);
    if (code) return code;
    code = afs_AccessOK(avc,temp, areq);
    if (code) return 0;
    else return EACCES;
}

static PSetCacheSize(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    long newValue;
    
    if (!afs_suser()) return EACCES;
    bcopy(ain, &newValue, sizeof(long));
    if (newValue == 0) afs_cacheBlocks = afs_origCacheBlocks;
    else afs_cacheBlocks = newValue;
    afs_CheckSize(0);
    return 0;
}

static PRemoveCallBack(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct conn *tc;
    register long code;
    struct AFSCallBack CallBacks_Array[1];
    struct AFSCBFids theFids;
    struct AFSCBs theCBs;

    if (!avc) return EINVAL;
    if (avc->states & CRO) return 0;	/* read-only-ness can't change */
    ObtainWriteLock(&avc->lock);
    theFids.AFSCBFids_len = 1;
    theCBs.AFSCBs_len = 1;
    theFids.AFSCBFids_val = (struct AFSFid *) &avc->fid.Fid;
    theCBs.AFSCBs_val = CallBacks_Array;
    CallBacks_Array[0].CallBackType = DROPPED;
    if (avc->callback) {
	do {
	    tc = afs_Conn(&avc->fid, areq);
	    if (tc) code = RXAFS_GiveUpCallBacks(tc->id, &theFids, &theCBs);
	    /* don't set code on failure since we wouldn't use it */
	} while (afs_Analyze(tc, code, &avc->fid, areq));
	avc->callback = 0;
    }
    ReleaseWriteLock(&avc->lock);
    return 0;
}

static PNewCell(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    /* create a new cell */
    long cellHosts[MAXHOSTS];	/* about 8 */
    register struct cell *tcell;
    register long code;
    
    if (!afs_suser()) return EACCES;
    bcopy(ain, cellHosts, MAXHOSTS*sizeof(long));
    tcell = afs_NewCell(ain + MAXHOSTS*sizeof(long), cellHosts, 0);
    if (tcell) code = 0;
    else code = EINVAL;
    return code;
}

static PListCells(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    long whichCell;
    register struct cell *tcell;
    register long i;
    register char *cp;

    bcopy(ain, &whichCell, sizeof(long));
    ObtainReadLock(&afs_xcell);
    for(tcell=afs_cells; tcell; tcell=tcell->next) {
	if (whichCell == 0) break;
	whichCell--;
    }
    if (tcell) {
	cp = aout;
	bzero(cp, MAXHOSTS*sizeof(long));
	for(i=0;i<MAXHOSTS;i++) {
	    if (tcell->cellHosts[i] == 0) break;
	    bcopy(&tcell->cellHosts[i]->host, cp, sizeof(long));
	    cp += sizeof(long);
	}
	cp = aout + MAXHOSTS*sizeof(long);
	strcpy(cp, tcell->cellName);
	cp += strlen(tcell->cellName)+1;
	*aoutSize = cp - aout;
    }
    ReleaseReadLock(&afs_xcell);
    if (tcell) return 0;
    else return EDOM;
}

static PRemoveMount(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register long code;
    long offset, len;
    register struct conn *tc;
    register struct dcache *tdc;
    struct AFSFetchStatus OutDirStatus;
    struct AFSVolSync tsync;

    /* "ain" is the name of the file in this dir to remove */

    if (!avc) return EINVAL;
    code = afs_VerifyVCache(avc, areq);
    if (code) return code;
    if (vType(avc) != VDIR) return ENOTDIR;
    tdc	= afs_GetDCache(avc, 0,	areq, &offset,	&len, 1);	/* test for error below */
    ObtainWriteLock(&avc->lock);
    do {
	tc = afs_Conn(&avc->fid, areq);
	if (tc) {
	    code = RXAFS_RemoveFile(tc->id, (struct AFSFid *) &avc->fid.Fid, ain, &OutDirStatus, &tsync);
	}
	else code = -1;
    } while (afs_Analyze(tc, code, &avc->fid, areq));
    if (code) {
	if (tdc) afs_PutDCache(tdc);
	ReleaseWriteLock(&avc->lock);
	return afs_CheckCode(code, areq);
    }
    if (tdc) {
	/* we have the thing in the cache */
	if (afs_LocalHero(avc, tdc, &OutDirStatus, 1)) {
	    /* we can do it locally */
	    code = dir_Delete(&tdc->f.inode, ain);
	    if (code) tdc->f.versionNo = -1;	/* surprise error -- invalid value */
	    /* DFEntryMod set by local hero */
	}
	afs_PutDCache(tdc);	/* drop ref count */
    }
    ReleaseWriteLock(&avc->lock);
    return 0;    
}

static PVenusLogging(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    register char *ain;
    char *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    long inputValue, outValue;
    long afsdeb, log;
    extern long afs_debug, LogFileInUse;

    bcopy(ain, &inputValue,sizeof(long));
    afsdeb = (inputValue >> 24) & 0xff;
    log = inputValue & 0xff;
    if (afsdeb != 99) afs_debug = afsdeb;
    if (log == 1) StartLogFile();
    else if (log == 0) {
	EndLogFile();
	afs_debug = 0;	/* Automatic shuting down with "off" */
    }
    else if (log == 99) {	
	outValue = (afs_debug << 24) + (LogFileInUse & 0xff);
	bcopy(&outValue, aout, sizeof(long));
	*aoutSize = sizeof(long);
    }
    return 0;
}

static PGetCellStatus(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct cell *tcell;
    long temp;
    
    tcell = afs_GetCellByName(ain);
    if (!tcell) return ENOENT;
    temp = tcell->states;
    bcopy(&temp, aout, sizeof(long));
    *aoutSize = sizeof(long);
    return 0;
}

static PSetCellStatus(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */ {
    register struct cell *tcell;
    long temp;
    
    if (!afs_suser()) return EACCES;
    tcell = afs_GetCellByName(ain+2*sizeof(long));
    if (!tcell) return ENOENT;
    bcopy(ain, &temp, sizeof(long));
    tcell->states = (tcell->states & CPrimary) | (temp & ~CPrimary);
    if (temp & CPrimary) tcell->states |= CPrimary;
    return 0;
}

static PFlushVolumeData(avc, afun, areq, ain, aout, ainSize, aoutSize)
struct vcache *avc;
int afun;
struct vrequest *areq;
char *ain, *aout;
long ainSize;
long *aoutSize;	/* set this */ {
    register long i;
    register struct dcache *tdc;
    long cell, volume;

    volume = avc->fid.Fid.Volume;  /* who to zap */
    cell = avc->fid.Cell;
    ObtainWriteLock(&afs_xdcache);  /* needed if you're going to flush any stuff */
    for(i=0;i<afs_cacheFiles;i++) {
	if (!(afs_indexFlags[i] & IFEverUsed)) continue;	/* never had any data */
	tdc = afs_GetDSlot(i, (struct dcache *) 0);
	if (tdc->refCount <= 1) {    /* too high, in use by running sys call */
	    if (tdc->f.fid.Fid.Volume == volume && tdc->f.fid.Cell == cell) {
		afs_FlushDCache(tdc);
	    }
	}
	tdc->refCount--;	/* bumped by getdslot */
    }
    ReleaseWriteLock(&afs_xdcache);
    return 0;
}

#ifdef	AFS_GATEWAY
/*
 *  We require root for local sysname changes, but not for remote
 *  (since we don't really believe remote uids anyway).
 */
static PSetSysName(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */
{
    char *nameptr, name[128];
    int namelen, code;

    if (areq->hostaddr == 0 && !afs_suser()) return EACCES;
    nameptr = *((char **)ain);
    ain += sizeof(char *);
    namelen = *((int *)ain);
    if (namelen >= 127) {
	return EINVAL;
    }
    code = copyin(nameptr, name, namelen + 1);
    if (code) {
	return code;
    }
    name[127] = 0;
    afs_gateway_set_sysname(areq->hostaddr, name);
    return 0;
}

/*
 *  Set u.u_cred to correspond to a user with given <hostaddr, uid, g0, g1>.
 *  This allows a server running as root to provide pioctl service to gateway
 *  clients by using this call to `become' the client. We can't really
 *  emulate this with setre[ug]id+setgroups because we can't trust setgroups
 *  to not mutilate the groups (since some afs implementations store the
 *  pag there). (Even if we could do it that way, it would be way too gross.)
 *
 *  We map uid 0 to nobody to (1) match the mapping that the nfs server does
 *  and (2) ensure that the suser() calls in the afs code fail.
 */
PSetClientContext(avc, afun, areq, ain, aout, ainSize, aoutSize)
    struct vcache *avc;
    int afun;
    struct vrequest *areq;
    char *ain, *aout;
    long ainSize;
    long *aoutSize;	/* set this */
{
    unsigned long hostaddr;
    int uid, g0, g1;
    extern int nobody;

    if (!afs_suser()) return EACCES;
    hostaddr = *((unsigned long *)ain);
    ain += sizeof(hostaddr);
    uid = *((unsigned long *)ain);
    ain += sizeof(uid);
    g0 = *((unsigned long *)ain);
    ain += sizeof(g0);
    g1 = *((unsigned long *)ain);
    ain += sizeof(g1);
    if (uid == 0) {
	uid = nobody;
    }
    crfree(u.u_cred);
    u.u_cred = crget();
    afs_gateway_set_hostaddr(u.u_cred, hostaddr);
    u.u_cred->cr_uid = uid;
#ifdef	AFS_AIX_ENV
    /@ I dont know how AIX works, so this is a guess @/
    u.u_cred->cr_grplst[0] = g0;
    u.u_cred->cr_grplst[1] = g1;
    u.u_cred->cr_ngrps = 2;
#else	AFS_AIX_ENV
    u.u_cred->cr_groups[0] = g0;
    u.u_cred->cr_groups[1] = g1;
    u.u_cred->cr_groups[2] = NOGROUP;
#endif	AFS_AIX_ENV
    return 0;
}
#endif	AFS_GATEWAY
