/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_osifile.c,v $
 * Revision 2.11  90/02/09  12:33:22  berman
 * 	Changed VOP_RDWR() in osi_Write() to use osi_cred. from jsb
 * 	[90/02/09            berman]
 * 
 * Revision 2.10  89/09/05  20:40:01  jsb
 * 	Added code in osi_Read and osi_Write to retry whenever receiving
 * 	EIO errors, since afs in general is not prepared to deal gracefully
 * 	with errors from these routines. Also remember the last non-zero error
 * 	returned from each of these routines.
 * 	[89/09/05  20:29:46  jsb]
 * 
 * Revision 2.9  89/08/09  09:47:58  jsb
 * 	Added support for afs resource pausing.
 * 	[89/08/09  09:34:37  jsb]
 * 
 * Revision 2.8  89/08/02  07:59:22  jsb
 * 	Use osi_Zalloc instead of osi_Alloc for osi_files.
 * 	[89/07/31  18:14:45  jsb]
 * 
 * Revision 2.7  89/06/24  23:58:23  jsb
 * 	Newer ITC sources.
 * 	[89/06/24  23:41:04  jsb]
 * 
 * Revision 2.5.1.1  89/06/02  22:00:15  jsb
 * 	Merged with newer ITC sources.
 * 
 * Revision 2.6  89/06/03  13:41:59  rpd
 * 	Merged with newer ITC sources.
 * 	[89/06/02  22:00:15  jsb]
 * 
 * Revision 2.5  89/04/22  15:14:29  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987, 1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

#include <afs/param.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef	AFS_AUX_ENV
#ifdef	PAGING
#include <sys/mmu.h>
#include <sys/seg.h>
#include <sys/page.h>
#include <sys/region.h>
#endif
#include <sys/sysmacros.h>
#include <sys/signal.h>
#include <sys/errno.h>
#endif
#include <sys/systm.h>
#if	!defined(AFS_IBM_ENV) || !defined(sys_rt_r3)
#include <sys/time.h>
#endif	AFS_IBM_ENV
#ifdef	AFS_AIX_ENV
#include <sys/errno.h>
#include <sys/file.h>
#include <sys/fullstat.h>
#include <sys/vattr.h>
#include <afs/aix_vfs.h>
#else
#include <sys/kernel.h>
#endif
#if	(!defined(AFS_AIX_ENV) && !defined(AFS_AUX_ENV) && !defined(AFS_VFS40))
#include <sys/cmap.h>	
#endif
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/uio.h>
#include <sys/buf.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vfs.h>
#include <afs/gfs_vnode.h>
#else
#ifdef	AFS_MACH_ENV
#include <vfs/vfs.h>
#include <vfs/vnode.h>
#include <sys/inode.h>
#else	AFS_MACH_ENV
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV
#include <netinet/in.h>
#include <sys/mbuf.h>
#include <sys/proc.h>
#include <afs/osi.h>
#include <afs/lock.h>

#if	(!defined(AFS_AIX_ENV) && !defined(AFS_VFS40) && !defined(AFS_MACH_ENV))
#if defined(vax) || defined(ibm032) || defined(mips)
#ifndef	ibm032
#include <sys/ipc.h>
#include <sys/shm.h>
#endif
#include <sys/vm.h>
#include <sys/map.h>
#include <machine/pte.h>
#endif vax || ibm032 || mips
#ifdef ibm032
#include <machine/mmu.h>
#endif ibm032
#endif	/*(!defined(AFS_AIX_ENV) && !defined(AFS_VFS40) && !defined(AFS_MACH_ENV))*/

extern osi_zone_t afs_osifile_zone;

static init=0;
struct ucred osi_cred;
struct vattr tvattr;
struct afs_lock afs_xosi;	/* lock is for tvattr */
int eio_retry_count = 5;
int osi_Read_error = 0, osi_Write_error = 0;


struct osi_file *osi_UFSOpen(adev, ainode)
    register struct osi_dev *adev;
    long ainode; {
    register struct inode *ip;
    register struct osi_file *afile;
#ifdef	AFS_AIX_ENV
    extern struct vfs *rootvfs;
    struct vnode *vp = (struct vnode *)0;
#endif
    if (!init) {
	bzero(&osi_cred, sizeof(struct ucred));
	Lock_Init(&afs_xosi);
	crhold(&osi_cred);	/* don't let it evaporate, since it is static */
	init = 1;
    }
    afile = (struct osi_file *) osi_Zalloc(afs_osifile_zone);
    u.u_error = 0;
#ifdef	AFS_AIX_ENV
    ip = (struct inode *) igetinode((dev_t) adev->dev, rootvfs, (ino_t)ainode, &vp);
#else
    ip = (struct inode *) igetinode((dev_t) adev->dev, (ino_t)ainode);
#endif
    if (u.u_error)
	    return (struct osi_file *) 0;
#ifdef	AFS_AIX_ENV
    afile->vnode = vp;		    /* Save the vnode pointer for the inode ip; also ip is already prele'ed in igetinode */
#else
    iunlock(ip);
    afile->vnode = ITOV(ip);
#endif
    afile->offset = 0;
    afile->proc = (int (*)()) 0;
    return afile;
}

osi_Stat(afile, astat)
    register struct osi_file *afile;
    register struct osi_stat *astat; {
    register long code;

    ObtainWriteLock(&afs_xosi);
#ifdef	AFS_AIX_ENV
    /* Note that in aix we don't use the credentials (osi_cred) */
    code = VNOP_GETATTR(afile->vnode, &tvattr, FL_STAT_REV, FULL_STAT);
#else
    code = VOP_GETATTR(afile->vnode, &tvattr, &osi_cred);
#endif
    if (code == 0) {
	astat->size = tvattr.va_size;
	astat->blksize = tvattr.va_blocksize;
	astat->mtime = tvattr.va_mtime.tv_sec;
	astat->atime = tvattr.va_atime.tv_sec;
    }
    ReleaseWriteLock(&afs_xosi);
    return code;
}

osi_Close(afile)
    register struct osi_file *afile; {
    VN_RELE(afile->vnode);
    osi_Zfree(afs_osifile_zone, afile);
    return 0;
}

osi_Truncate(afile, asize)
    register struct osi_file *afile;
    long asize; {
    struct ucred *oldCred;
    register long code;

    ObtainWriteLock(&afs_xosi);
#ifdef	AFS_AIX_ENV
    code = VNOP_FTRUNC(afile->vnode, FWRITE|FSYNC, asize, (caddr_t)0);
#else
    VATTR_NULL(&tvattr);
    tvattr.va_size = asize;
    /* note that this credential swapping crap is only necessary because
	of ufs's references directly to u.u_cred instead of to
	credentials parameter.  Probably should fix ufs some day. */
    oldCred = u.u_cred;	    /* remember old credentials pointer  */
    u.u_cred = &osi_cred;   /* temporarily use superuser credentials */
    code = VOP_SETATTR(afile->vnode, &tvattr, &osi_cred);
    u.u_cred = oldCred;	    /* restore */
#endif
    ReleaseWriteLock(&afs_xosi);
    return code;
}

/* Generic read interface */
osi_Read(afile, aptr, asize)
    register struct osi_file *afile;
    char *aptr;
    long asize; {
    unsigned int resid;
    register long code;
#ifdef	AFS_MACH_ENV
    int retry_count;

    retry_count = eio_retry_count;
    do {
	code = gop_rdwr(UIO_READ, afile->vnode, (caddr_t) aptr, asize,
			afile->offset, AFS_UIOSYS, IO_UNIT, &resid);
	if (code == EIO && retry_count > 0) {
	    printf("afs: osi_Read got EIO, retrying...\n");
	    if (retry_count < eio_retry_count) osi_Wait(1000, (char *) 0);
	}
    } while (code == EIO && retry_count-- > 0);
    if (code == EIO) printf("afs: osi_Read returning EIO!\n");
#else
#ifdef	AFS_AIX_ENV
    /* Note the difference in the way the afile->offset is passed (see comments in gop_rdwr() in afs_aix_subr.c for comments) */
    code = gop_rdwr(UIO_READ, afile->vnode, (caddr_t) aptr, asize, (off_t)&afile->offset,
		  AFS_UIOSYS, NULL, &resid);
#else
    code = gop_rdwr(UIO_READ, afile->vnode, (caddr_t) aptr, asize, afile->offset,
		  AFS_UIOSYS, IO_UNIT, &resid);
#endif
#endif
    if (code == 0) {
	code = asize - resid;
	afile->offset += code;
    }
    else {
	osi_Read_error = u.u_error = code;
	code = -1;
    }
    return code;
}

/* Generic write interface */
osi_Write(afile, aptr, asize)
    register struct osi_file *afile;
    char *aptr;
    long asize; {
    unsigned int resid;
    register long code;
#ifdef	AFS_MACH_ENV
    struct uio uio;
    struct iovec iov;
    int retry_count;

    retry_count = eio_retry_count;
    do {
	iov.iov_base = (caddr_t) aptr;
	iov.iov_len = asize;
	uio.uio_iov = &iov;
	uio.uio_iovcnt = 1;
	uio.uio_offset = (off_t) afile->offset;
	uio.uio_segflg = AFS_UIOSYS;
	uio.uio_resid = asize;
	do {
	    code = VOP_RDWR(afile->vnode, &uio, UIO_WRITE, IO_UNIT, &osi_cred);
	} while (afs_fspause(0, code));
	if (code == EIO && retry_count > 0) {
	    printf("afs: osi_Write got EIO, retrying...\n");
	    if (retry_count < eio_retry_count) osi_Wait(1000, (char *) 0);
	}
    } while (code == EIO && retry_count-- > 0);
    if (code == EIO) printf("afs: osi_Write returning EIO!\n");
    resid = uio.uio_resid;
#else
#ifdef	AFS_AIX_ENV
    /* Note the difference in the way the afile->offset is passed (see comments in gop_rdwr() in afs_aix_subr.c for comments) */
    code = gop_rdwr(UIO_WRITE, afile->vnode, (caddr_t) aptr, asize, (off_t)&afile->offset,
		  AFS_UIOSYS, NULL, &resid);
#else
    code = gop_rdwr(UIO_WRITE, afile->vnode, (caddr_t) aptr, asize, afile->offset,
		  AFS_UIOSYS, IO_UNIT, &resid);
#endif
#endif
    if (code == 0) {
	code = asize - resid;
	afile->offset += code;
    }
    else {
	osi_Write_error = u.u_error = code;
	code = -1;
    }
    if (afile->proc) {
	(*afile->proc)(afile, code);
    }
    return code;
}


/* BARF CITY: this work should be handled by physstrat in ca/machdep.c.
    This routine written from the RT NFS port strategy routine.
    It has been generalized a bit, but should still be pretty clear. */
int osi_MapStrategy(aproc, bp)
	int (*aproc)();
	register struct buf *bp;
{
	long returnCode;

#if     (!defined(AFS_AIX_ENV) && !defined(AFS_VFS40) && !defined(AFS_MACH_ENV))
#if defined(vax) || defined(ibm032) || defined(mips)
	if (bp->b_flags & B_PHYS) {
		register int npte;
		register int n;
		register long a;
		register struct pte *pte, *kpte;
		caddr_t va;
		int o;
		caddr_t saddr;
		struct proc *rp;
		unsigned v;

#ifdef ibm032
		struct hatipt_entry *ipte;
		u_int vp;
		long saddrtag;	/* puke */
#endif ibm032
#if defined(vax) || defined(mips)
		if (!(bp->b_flags & B_PAGET)) {
#endif
		/*
		 * Buffer's data is in userland, or in some other
		 * currently inaccessable place. We get a hunk of
		 * kernel address space and map it in.
		 */
		v = btop(bp->b_un.b_addr);
		o = (int)bp->b_un.b_addr & PGOFSET;
		npte = btoc(bp->b_bcount + o);
#if defined(vax) || defined(mips)
		rp = bp->b_flags&B_DIRTY ? &proc[2] : bp->b_proc;
		if (bp->b_flags & B_UAREA)
		    pte = &rp->p_addr[v];
		else if ((bp->b_flags & B_SMEM)  &&
			 ((bp->b_flags & B_DIRTY) == 0))
		    pte = ((struct smem *)rp)->sm_ptaddr+v;
		else
		    pte = vtopte(rp, v);
#else
		pte = vtopte(bp->b_proc, btop(bp->b_un.b_addr));
#endif vax || mips
		while ((a = rmalloc(kernelmap, (long)clrnd(npte))) == NULL) {
			kmapwnt++;
			sleep((caddr_t)kernelmap, PSWP+4);
		}
		kpte = &Usrptmap[a];
#if defined(vax) || defined(mips)
		for (n = npte; n--; kpte++, pte++)
#ifdef mips
			*(int *)kpte = (*(int *)pte & PG_PFNUM);
#endif mips
#ifdef vax
			*(int *)kpte = PG_NOACC | (*(int *)pte & PG_PFNUM);
#endif vax
		va = (caddr_t)kmxtob(a);
#ifdef mips
		vmaccess(&Usrptmap[a], va, npte, DO_CACHE);
#endif mips
#ifdef vax
		vmaccess(&Usrptmap[a], va, npte);
#endif vax
#endif vax || mips
#ifdef ibm032
		va = (caddr_t)kmxtob(a);
		vp = btop(va);
		for (n = npte; n; kpte++, pte++, vp++, n--) {
			*(int *)kpte = *(int *)pte & PG_PFNUM;
			ipte = &MMU_HATIPT[*(int *)kpte];
			if (n == npte)
				saddrtag = ipte->key_addrtag;
			else if (saddrtag != ipte->key_addrtag)
				printf("AFS temp map confusion\n");
			ipte->key_addrtag = MMU_SID_SYSTEM << MMU_VPAGE_BITS;
			mapin(kpte, vp, (u_int)(PG_V|PG_KW | *(int *)kpte), 1);
		}
#endif ibm032
		saddr = bp->b_un.b_addr;
		bp->b_un.b_addr = va + o;
#if defined(vax) || defined(mips)
		}
#endif
		/*
		 * do the io
		 */
		returnCode = (*aproc)(bp);
		/*
		 * Release kernel maps
		 */
#if defined(vax) || defined(mips)
		if (!(bp->b_flags & B_PAGET)) {
#endif
		bp->b_un.b_addr = saddr;
		kpte = &Usrptmap[a];
#if defined(vax) || defined(mips)
		for (n = npte; n-- ; kpte++)
#ifdef mips
			*(int *)kpte = 0;
#endif  mips
#ifdef vax
			*(int *)kpte = PG_NOACC;
#endif vax
#endif vax || mips
#ifdef ibm032
		for (n = npte; n--; kpte++, vp++) {
			ipte = &MMU_HATIPT[*(int *)kpte & PG_PFNUM];
			mapout(kpte, 1);
			*(int *)kpte = PG_NOACC; /* who cares? */
			ipte->key_addrtag = saddrtag;
		}
#endif ibm032
		rmfree(kernelmap, (long)clrnd(npte), a);
#if defined(vax) || defined(mips)
		}
#endif
	} else
#endif vax || ibm032 || mips

#endif	/*(!defined(AFS_AIX_ENV) && !defined(AFS_VFS40) && !defined(AFS_MACH_ENV))*/
	    returnCode = (*aproc) (bp);

	return returnCode;
}

shutdown_osifile()
{
    init = 0;
}
