/* 
 * Mach Operating System
 * Copyright (c) 1989 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log:	afs_istuff.c,v $
 * Revision 2.5  89/09/11  08:06:39  rvb
 * 	Looks like we added retry if iget fails.
 * 
 * 
 * Revision 2.4  89/06/24  23:58:10  jsb
 * 	Newer ITC sources.
 * 	[89/06/24  23:40:48  jsb]
 * 
 * Revision 2.3  89/06/03  15:27:37  jsb
 * 	Merging with newer ITC sources.
 * 	[89/05/26  19:23:40  jsb]
 * 
 * Revision 2.2  89/04/22  15:14:08  gm0w
 * 	Updated to RX version.
 * 	[89/04/14            gm0w]
 * 
 */
/*
 * P_R_P_Q_# (C) COPYRIGHT IBM CORPORATION 1987, 1988
 * LICENSED MATERIALS - PROPERTY OF IBM
 * REFER TO COPYRIGHT INSTRUCTIONS FORM NUMBER G120-2083
 */

#include <afs/param.h>
#include <sys/types.h>
#include <sys/param.h>
#ifdef	AFS_AUX_ENV
#ifdef	PAGING
#include <sys/mmu.h>
#include <sys/seg.h>
#include <sys/page.h>
#endif
#include <sys/sysmacros.h>
#include <sys/signal.h>
#include <sys/errno.h>
#endif
#include <sys/systm.h>
#if	!defined(AFS_IBM_ENV) || !defined(sys_rt_r3)
#include <sys/time.h>
#endif	AFS_IBM_ENV
#ifdef	AFS_AIX_ENV
#include <sys/errno.h>
#include <afs/aix_vfs.h>
#else
#include <sys/kernel.h>
#endif
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/protosw.h>
#include <sys/dir.h>
#include <sys/user.h>
#include <sys/file.h>
#include <sys/uio.h>
#ifdef	AFS_GFS_ENV
#include <afs/gfs_vfs.h>
#include <afs/gfs_vnode.h>
#else
#ifdef	AFS_MACH_ENV
#include <sys/inode.h>
#include <sys/mount.h>
#else	AFS_MACH_ENV
#include <sys/vfs.h>
#include <sys/vnode.h>
#include <ufs/inode.h>
#include <ufs/mount.h>
#endif	AFS_MACH_ENV
#endif	AFS_GFS_ENV
#include <sys/buf.h>
#include <netinet/in.h>
#ifdef	notdef
#include <sys/mbuf.h>	/* Can we do without this? */
#endif	
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <afs/osi.h>
#include <rx/rx.h>

#include <afs/lock.h>
#include <afs/volerrors.h>
#include <afs/voldefs.h>
#include <afsint/afsint.h>
#include <afs/afs.h>
#ifndef	AFS_MACH_ENV
#include <afs/auxinode.h>
#endif	AFS_MACH_ENV

extern int eio_retry_count;

#ifdef	AFS_AUX_ENV
#define	ino_t	ino_tl
#endif

#ifndef	AFS_AIX_ENV
/* trygets and getinode aren't needed by aix */

#ifndef	AFS_GFS_ENV
#define	V_TO_I(x)   (x)

#if	(defined(AFS_IBM_ENV) && defined(sys_rt_r3)) || defined(AFS_TRYGETFS_ENV)
extern struct fs *trygetfs();
#else	/* AFS_IBM_ENV */
#ifdef AFS_AUX_ENV
extern struct filsys *trygetfs();
#else	/* AFS_AUX_ENV */
struct fs *
 trygetfs(dev)
 	dev_t dev;
 {
 	register struct mount *mp;
 
 	mp = getmp(dev);
 	if (mp == NULL) {
 		return(NULL);
 	}
 	return (mp->m_bufp->b_un.b_fs);
 }
#endif /* AFS_AUX_ENV */
#endif /* AFS_IBM_ENV */
#endif /* AFS_GFS_ENV */

struct inode *
getinode(dev, inode)
dev_t dev;
ino_t inode;
{
#ifdef	AFS_GFS_ENV
    register struct mount *mp;

    GETMP(mp, dev);
    if (mp == NULL || (mp == (struct mount *) MSWAPX)) {
	u.u_error = ENXIO;
	return((struct inode *)0);
    }
    return(gfs_gget(dev, mp, inode, 0));
#else
#ifdef	AFS_AUX_ENV
    register struct filsys *fs;
#else
#ifdef	AFS_VFS40
    struct inode *ip;
#endif
    register struct fs *fs;
#endif	AFS_AUX_ENV
#ifdef	AFS_MACH_ENV
    int retry_count, old_error;
    struct inode *ip;
#endif	AFS_MACH_ENV

    fs = trygetfs(dev);
    if (fs == NULL) {
	u.u_error = ENXIO;
	return((struct inode *)0);
    }
#ifdef	AFS_VFS40
    /* In order to get more details on the returned error from iget we should change the way igetinode and getinode interact someday... */
    if (iget(dev, fs, inode, &ip))
	return (struct inode *)0;
    return (ip);
#else
#ifdef	AFS_MACH_ENV
    retry_count = eio_retry_count;
    old_error = u.u_error;
    for (;;) {
	ip = iget(dev, fs, inode);
	if (ip) {
	    u.u_error = old_error;
	    return ip;
	}
	if (! retry_count) {
	    printf("afs: iget failed (%d); getinode returns 0!\n", u.u_error);
	    return ip;
	}
	printf("afs: iget failed (%d), retrying...\n", u.u_error);
	if (retry_count-- < eio_retry_count) osi_Wait(1000, (char *) 0);
    }
#else
    return(iget(dev, fs, inode));
#endif	AFS_MACH_ENV
#endif
#endif	AFS_GFS_ENV
}
#endif	AFS_AIX_ENV

#ifdef	AFS_AIX_ENV
long aux_debug=0;
static struct afs_lock afs_xparts;
static struct afs_lock afs_xiparams;

struct vfs *
devtovfs(dev)
dev_t dev;
{
    extern struct vfs	*rootvfs;
    struct vfs		*vfsp;
    struct inode	*ip;

	for (vfsp = rootvfs; vfsp != NULL; vfsp = vfsp->vfs_next)
		if (vfsp->vfs_mntd != NULL && vfsp->vfs_type == MNT_AIX
		     && (vfsp->vfs_flag & VFS_DEVMOUNT)) { 
			ip = VTOIP(vfsp->vfs_mntd);
			if (brdev(ip->i_dev) == brdev(dev))
				return (vfsp);
		}
	return (struct vfs *)0;
}
#endif

/* get an existing inode.  Common code for iopen, iread/write, iinc/dec. */
/* Also used by rmt_remote to support passing of inode number from venus */
struct inode *
#ifdef	AFS_AIX_ENV
igetinode(dev, vfsp, inode, vpp)
        struct vfs *vfsp;
	struct vnode **vpp;  /* vnode associated with the inode */
#else
igetinode(dev, inode)
#endif
	dev_t dev;
	ino_t inode;
{
	register struct inode *ip;
#ifdef	AFS_AIX_ENV
	extern struct inode *iget();
#endif

#ifdef	AFS_AIX_ENV
	ip = iget(dev, inode);
#else
	ip = getinode(dev, inode);
#endif
	if (ip == NULL) {
	    u.u_error = ENOENT;		/* Well... */
	    return;
	}
	if (ip->i_mode == 0) {
	    /* Not an allocated inode */
	    iforget(ip);	    
	    u.u_error = ENOENT;
	    return;
	}
	if (ip->i_nlink == 0 || (ip->i_mode&IFMT) != IFREG) {
	    iput(ip);
	    u.u_error = ENOENT;
	    return;
	}
#ifdef	AFS_AIX_ENV
	if (vfsp)
	    u.u_error = iptovp(vfsp, ip, vpp);
#endif
	return ip;
}


/*
 * Really horrible fudge to allow us to drop an inode we got with iget but
 * which isn't really allocated.
 */
iforget(ip)
struct inode *ip;
{
   extern struct inode *ifreeh, **ifreet;

#ifdef	AFS_MACH_ENV
	CLEAR_INODE_FLAGS(ip);
#else	AFS_MACH_ENV
	ip->i_flag = 0;
#endif	AFS_MACH_ENV
	if (ifreeh) {
		*ifreet = ip;
		ip->i_freeb = ifreet;
	} else {
		ifreeh = ip;
		ip->i_freeb = &ifreeh;
	}
	ip->i_freef = NULL;
	ifreet = &ip->i_freef;
}


/*
 * icreate system call -- create an inode
 */
icreate()
{
	register struct a {
		int	dev;
		int	near_inode;
		int	param1;
		int	param2;
		int	param3;
		int	param4;
	} *uap = (struct a *) u.u_ap;
	struct inode *ip, *newip;
	extern struct inode *ialloc();
#ifdef	AFS_VFS40
	register int err;
#endif
#ifdef	AFS_AIX_ENV
	struct vnode *vp = (struct vnode *)0;
	extern struct vfs *rootvfs;
	register struct vfs *vfsp;
#endif
	if (!suser())
		return;
#if	(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))
	printf("'ICreate' not supported on AUX/AIX client kernels yet\n");
#else
#ifdef	AFS_AIX_ENV
	if (aux_debug) printf("icreate-1: dev=%d, p1=%d, p2=%d, p3=%d, p4=%d\n", uap->dev, uap->param1, uap->param2, uap->param3, uap->param4);
	newip = ialloc((dev_t)uap->dev, IFREG, 1);
	if (u.u_error) {
	    return;
	}
	newip->i_flag |= IACC|IUPD|ICHG|ISYN;
	newip->i_gid = -2;	/* Put special gid flag */
	if ((vfsp = devtovfs((dev_t)uap->dev)) == 0) {
	    printf("Dev=%d not mounted!!; quitting\n", uap->dev);
	    u.u_error =	ENODEV;	/* Yuck */
	    return;
	}
	u.u_error = iptovp(vfsp, newip, &vp); 
	plock(newip);	/* Since iptovp prele's it */
	afs_CreateAuxEntry(newip, VICEMAGIC, uap->param1, uap->param2, uap->param3, uap->param4);
	u.u_rval1 = newip->i_number;
	prele(newip);	/* Must be unlocked before VNOP_RELE */
	if (aux_debug) printf("icreate-e: u_rval1=%d\n", u.u_rval1);
	VNOP_RELE(vp);	/* We need this, don't we? */
#else
	ip = getinode((dev_t)uap->dev, 2);
	if (ip == NULL) {
		u.u_error = ENOENT; /* Well... */
		return;
	}
#ifdef	AFS_VFS40
	if (err	= ialloc(ip, 0,	0, &newip)) {	
	    u.u_error = err;
	    return;
	}
#else
	newip = ialloc(ip, 0, 0);
#endif
	iput(ip);
	if  (newip == NULL)
	    return;
#if defined(AFS_SUN_ENV) || defined(AFS_MACH_ENV)
#ifdef	AFS_VFS40
	newip->i_flag |= IACC|IUPD|ICHG;
#else
	imark(newip, IACC|IUPD|ICHG);
#endif
#else 
	newip->i_flag |= IACC|IUPD|ICHG;
#endif	/*defined(AFS_SUN_ENV) || defined(AFS_MACH_ENV)*/
	newip->i_nlink = 1;
	newip->i_uid = 0;
	newip->i_gid = -2;
	newip->i_mode = IFREG;
#ifndef	AFS_GFS_ENV
#ifdef	AFS_MACH_ENV
	newip->i_vnode.v_mode = VREG;
#else	AFS_MACH_ENV
	newip->i_vnode.v_type = VREG;
#endif	AFS_MACH_ENV
#endif
	V_TO_I(newip)->i_vicemagic = VICEMAGIC;
	V_TO_I(newip)->i_vicep1 = uap->param1;
	V_TO_I(newip)->i_vicep2 = uap->param2;
	V_TO_I(newip)->i_vicep3 = uap->param3;
	V_TO_I(newip)->i_vicep4 = uap->param4; 
	u.u_r.r_val1 = newip->i_number;
	iput(newip);
#endif	AFS_AIX_ENV
#endif	/*(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))*/
	return;
}


/*
 * iopen system call -- open an inode for reading/writing
 * Restricted to super user.
 * Any IFREG files.
 */
iopen()
{
	register struct a {
		int	dev;
		int	inode;
#ifdef	AFS_AUX_ENV
		int	usrmod;
#else
		int	usermode;
#endif
	} *uap = (struct a *) u.u_ap;
	struct file *fp;
	register struct inode *ip;
	struct vnode *vp = (struct vnode *)0;
	extern struct fileops vnodefops;
#ifdef	AFS_AIX_ENV
	register struct vfs *vfsp;
#endif

	if (!suser())
		return;
#if	(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))
	printf("'Iopen' not supported on AIX client kernel yet\n");
	/* Note that most calls (i.e. igetinode, iunlock) are ready; it's just that f_ops aren't used in the aix kernel and since we don't use this call in the client side, we let alone for now... */
#else
#ifdef	AFS_AIX_ENV
	if (aux_debug) printf("iopen-1: dev=%d, inode=%d, umode=%x\n", uap->dev, uap->inode, uap->usermode);
	if ((vfsp = devtovfs((dev_t)uap->dev)) == 0) {
	    printf("Dev=%d not mounted!!; quitting\n", uap->dev);
	    u.u_error =	ENODEV;	/* Yuck */
	    return;
	}
	ip = igetinode((dev_t)uap->dev, vfsp, (ino_t)uap->inode, &vp);
	if (u.u_error) {
	    return;
	}
	if (u.u_error = falloc(vp, (uap->usermode-FOPEN)&FMASK, &fp)) {
	    VNOP_RELE(vp);
	    return;
	}
	if (aux_debug) printf("iopen-e:\n");
	prele(ip);
#else
	ip = igetinode((dev_t)uap->dev, (ino_t)uap->inode);
	if (u.u_error)
		return;
	fp = falloc();
	if (fp == NULL) {
		iput(ip);
		return;
	}
	iunlock(ip);
#ifdef	AFS_AUX_ENV
	fp->f_flag = (uap->usrmod-FOPEN)&FMASK;
#else
	fp->f_flag = (uap->usermode-FOPEN)&FMASK;
#endif
	fp->f_type = DTYPE_VNODE;
	fp->f_ops = &vnodefops;
	vp = ITOV(ip);
	fp->f_data = (caddr_t)vp;
#endif	AFS_AIX_EN
#endif	/*(defined(AFS_AIX_ENV) && defined(AFS_CLIENT))*/
	return;
}


/*
 * Support for iinc() and idec() system calls--increment or decrement
 * count on inode.
 * Restricted to super user.
 * Only VICEMAGIC type inodes.
 */
iincdec(amount)
	int amount;
{
	register struct a {
		int	dev;
		int	inode;
		long	inode_p1;
	} *uap = (struct a *) u.u_ap;
	register struct inode *ip;
#if	(defined(AFS_AIX_ENV) && (defined(AFS_COMMON) || defined(AFS_SERVER)))
	struct vnode *vp;
	register struct vauxinode *auxp;
	extern struct vauxinode *afs_GetAuxInode();
#endif

	if (!suser())
		return;
#if	(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))
	printf("'Iinc/Idec' not supported on AIX/AUX client kernels yet..\n");
	return;
#else
#ifdef	AFS_AIX_ENV
	if (aux_debug) printf("iincdec-1: dev=%d, inode=%d, inod_p1=%d, amnt=%d\n", uap->dev, uap->inode, uap->inode_p1, amount);
	ip = igetinode((dev_t)uap->dev, (struct vfs *)0, (ino_t)uap->inode, &vp);
	if (u.u_error) {
	    return;
	}
	auxp = afs_GetAuxInode(ip);
	/* Note that auxp will next be returned as null! */
	if (auxp->aux_dimage.aux_magic != VICEMAGIC)
	    u.u_error = EPERM;
	else if (auxp->aux_dimage.aux_param1 != uap->inode_p1)
	    u.u_error = ENXIO;
	else
	    ip->i_nlink += amount;
	afs_PutAuxInode(auxp);
	if (ip->i_nlink == 0) {
	    /* We now must remove the cached auxp entry and also clean-up its disk image since the file is deleted */
	    afs_PutAuxInode(auxp);  /* Since refCount is still 1!! */	    
	    ObtainWriteLock(&afs_xiparams);
	    afs_FlushAuxCache(auxp, 1);
	    ReleaseWriteLock(&afs_xiparams);
	}
	if (aux_debug) printf("iincdec-e:u.u_err=%d, n_nlink=%d\n", u.u_error, ip->i_nlink);
#else
	ip = igetinode((dev_t)uap->dev, (ino_t)uap->inode);
	if (u.u_error)
		return;
	if (V_TO_I(ip)->i_vicemagic != VICEMAGIC)
		u.u_error = EPERM;
	else if (V_TO_I(ip)->i_vicep1 != uap->inode_p1)
	    u.u_error = ENXIO;
	else
		ip->i_nlink += amount;
	if (ip->i_nlink == 0)
		V_TO_I(ip)->i_vicemagic = 0;
#endif	AFS_AIX_ENV
	ip->i_flag |= ICHG;
	iput(ip);
#endif	/*(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))*/
}

iinc() {
    iincdec(1);
}

idec() {
    iincdec(-1);
}


/*
 * Support for iread/iwrite system calls.
 * Restricted to super user.
 * Only inodes with owner and group == -1.
 * NB:  VICEMAGIC inodes default to this owner and group.
 */
ireadwrite(rw)
{
	register struct a {
		int		dev;
		int		inode;
		long		inode_p1; 
		unsigned int 	offset;
		char		*cbuf;
		unsigned int 	count;
	} *uap = (struct a *) u.u_ap;
	register struct inode *ip;
	unsigned int resid;
#ifdef	AFS_AIX_ENV
	struct vnode *vp;
#if	(defined(AFS_COMMON) || defined(AFS_SERVER))
	register struct vauxinode *auxp;
	extern struct vauxinode *afs_GetAuxInode();
	register struct vfs *vfsp;
#endif
#else
	daddr_t	db[NDADDR], ib[NIADDR];
#endif
	int size;

	if (!suser())
		return;
#if	(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))
	printf("'Iread/Iwrite' not supported on AIX/AUX client kernels yet!\n");
	return;
#else
#ifdef	AFS_AIX_ENV
	if (aux_debug) printf("irdwrite-1: rw=%d, dev=%d, inode=%d, count=%d\n", rw, uap->dev, uap->inode, uap->count);
	if ((vfsp = devtovfs((dev_t)uap->dev)) == 0) {
	    printf("Dev=%d not mounted!!; quitting\n", uap->dev);
	    u.u_error =	ENODEV;	/* Yuck */
	    return;
	}
	ip = igetinode((dev_t)uap->dev, vfsp, (ino_t)uap->inode, &vp);
	if (u.u_error) {
	    return;
	}
        plock(ip);
	auxp = afs_GetAuxInode(ip);
	if (auxp->aux_dimage.aux_param1 != uap->inode_p1) {
	        prele(ip);
		VNOP_RELE(vp);
	        u.u_error = ENXIO;
		return;
	}
	size = ip->i_size;
	prele(ip);
	u.u_error = gop_rdwr(rw, vp, (caddr_t)uap->cbuf, uap->count, (off_t)&uap->offset, AFS_UIOUSER, NULL, &resid);
	/* How about if we got an error from gop_rdwr??? */
	u.u_rval1 = uap->count - resid;
	afs_PutAuxInode(auxp);
	if (aux_debug) printf("irdwrite-e: uerr=%d, u_rval1=%d\n", u.u_error, u.u_rval1);
	/* Ignore case that doesn't write the inode if it didn't change for the time being; it should be done at some point */
	VNOP_RELE(vp);
#else
	ip = igetinode((dev_t)uap->dev, (ino_t)uap->inode);
	if (u.u_error)
		return;
	if (V_TO_I(ip)->i_vicep1 != uap->inode_p1) {
	        iput(ip);
	        u.u_error = ENXIO;
		return;
	}
	bcopy(V_TO_I(ip)->i_db, db, sizeof db);
	bcopy(V_TO_I(ip)->i_ib, ib, sizeof ib);
	size = ip->i_size;
	resid = 0;
	iunlock(ip);
	u.u_error = rdwri(
	    rw, ip, (caddr_t) uap->cbuf, uap->count, uap->offset, 0, &resid);
	u.u_r.r_val1 = uap->count - resid;
#ifndef	AFS_GFS_ENV
	ilock(ip);
#endif
	if (size == ip->i_size && bcmp(V_TO_I(ip)->i_db,db,sizeof db) == 0
	    && bcmp(V_TO_I(ip)->i_ib,ib,sizeof ib) == 0) {
		/* Don t write out the inode if it hasn t really changed.
		   We don t care about inode dates in file server files */
		ip->i_flag &= ~(IUPD|IACC|ICHG);
	}
	iput(ip);
#endif	AFS_AIX_ENV
#endif	/*(defined(AFS_AUX_ENV) || (defined(AFS_AIX_ENV) && defined(AFS_CLIENT)))*/
}

#ifdef	AFS_AIX_ENV
/* Since there is a conflict with an aix 'iread' call, we rename this one for aix; we don't really use it anyway! */
oiread()
#else
iread()
#endif
{
	ireadwrite(UIO_READ);
}

iwrite()
{
    	ireadwrite(UIO_WRITE);
}


#ifdef	AFS_AUX_ENV
/* Somehow this silly routine doesn't exist in AUX */
char *
strcpy(s1, s2)
	register char *s1, *s2;
{
	register char *os1;

	os1 = s1;
	while (*s1++ = *s2++)
		;
	return (os1);
}
#endif

#if	(defined(AFS_AIX_ENV) && (defined(AFS_COMMON) || defined(AFS_SERVER)))

#define	NHAUXINO    128	    /* Must be a power of 2 */
struct vauxinode *aux_table[NHAUXINO];
#define	AHash(ainode)   ((int)(ainode) & (NHAUXINO-1))
#define	QTOA(e)	    ((struct vauxinode *)(((char *) (e)) - (((char *)(&(((struct vauxinode *)(e))->aux_lruq))) - ((char *)(e)))))
#define	AUXHSIZE    300

static struct afs_auxheader auxheader;
static struct afspart *afs_parts = 0;
static struct afs_q AUXLRU;
static struct vauxinode *FreeAuxList = 0;
static int auxInitialized = 0;

aux_init() {
    register int i;
    register struct vauxinode *auxp;

    if (auxInitialized) {
	if (aux_debug) printf("Already initialized: exiting aux_init\n");
	return;
    }
    auxInitialized++;
    Lock_Init(&afs_xiparams);
    Lock_Init(&afs_xparts);
    auxp = (struct vauxinode *) osi_Alloc(AUXHSIZE * sizeof(struct vauxinode));
    bzero(auxp, AUXHSIZE * sizeof(struct vauxinode));
    FreeAuxList = (struct vauxinode *) &auxp[0];
    for (i=0; i < AUXHSIZE; i++)
	auxp[i].aux_lruq.next = (struct afs_q *) (&auxp[i+1]);
    auxp[AUXHSIZE-1].aux_lruq.next = (struct afs_q *) 0;
    for (i=0; i < NHAUXINO;i++) aux_table[i] = NULL;
    QInit(&AUXLRU);
}

static struct afspart *afs_FindPartByDev(dev)
    register long dev;
{
    register struct afspart *tc;

    ObtainWriteLock(&afs_xparts);
    for(tc = afs_parts; tc; tc = tc->next) {
	if (tc->cdev.dev == dev) {
	    ReleaseWriteLock(&afs_xparts);
	    if (aux_debug) printf("afs_FindPartByDev: Found afspart entry=%x\n", tc);	    
	    return tc;
	}
    }
    ReleaseWriteLock(&afs_xparts);
    if (aux_debug) printf("afs_FindPartByDev: Didn't Find afspart entry\n");
    return (struct afspart *) 0;
}    
    

static struct afspart *afs_GetNewPart(dev)
    register long dev;
{
    register struct afspart *tc;

    ObtainWriteLock(&afs_xparts);
    for(tc = afs_parts; tc; tc = tc->next) {
	if (tc->cdev.dev == dev) {
	    if (aux_debug) printf("afs_GetNewPart: Found afspart entry in %x\n", tc);	    
	    break;
	}
    }
    if (!tc) {
	tc = (struct afspart *) osi_Alloc(sizeof (struct afspart));
	if (aux_debug) printf("afs_GetNewPart: Creating a new one in %x\n", tc);
	tc->next = afs_parts;
	afs_parts = tc;
	tc->cdev.dev = dev;
	tc->fileInode =	0;  /* To be set in afs_InitAuxVolfile */
	tc->maxIndex = 0;

    }
    ReleaseWriteLock(&afs_xparts);
    return tc;
}

/* This should be called from an external volume related routine; it should be called first before any of the other i-calls are used... */
afs_InitAuxVolFile(afile)
    register char *afile; 
{
    register long code;
    struct osi_stat tstat;
    register struct osi_file *tfile;
    register struct afspart *tc;
    struct vnode *filevp;
    int goodFile;

    if (!auxInitialized) aux_init();
    if (aux_debug) printf("In afs_InitAuxVolFile(afile=%s)\n", afile);
    code = gop_lookupname(afile, AFS_UIOSYS, 0, (struct vnode *) 0, &filevp);
    if (code) return ENOENT;
    tc = afs_GetNewPart(VTOI(filevp)->i_dev);
    tc->fileInode = VTOI(filevp)->i_number;	
    if (aux_debug == 2) printf("tc=%x, cdev=%d, fileinode=%d\n", tc, tc->cdev.dev, tc->fileInode);
    tc->maxIndex = 0; /* Reinitialize it just in case */
    VNOP_RELE(filevp);
    tfile = osi_UFSOpen(&tc->cdev, tc->fileInode);
    if (!tfile) panic("initcacheinfo");
    osi_Stat(tfile, &tstat);
    tc->modTime = tstat.mtime;
    code = osi_Read(tfile, &auxheader, sizeof(auxheader));
    goodFile = 0;
    if (code == sizeof(auxheader)) {
	/* read the header correctly */
	if (auxheader.magic == VICEMAGIC)
	    goodFile = 1;
    }
    if (!goodFile) {
	/* write out a good file label; be careful here so that we don't truncate the file for a silly reason. We should probably revisit this code! */
	if (aux_debug) printf("afs_InitAuxVolFile: creating new label\n");
	auxheader.magic = VICEMAGIC;
	osi_Seek(tfile, 0);
	osi_Write(tfile, &auxheader, sizeof(auxheader));
	/* now truncate the rest of the file, since it may be arbitrarily wrong */
	osi_Truncate(tfile, sizeof(struct afs_auxheader));
    }
    osi_Close(tfile);
    return 0;
}

/*static*/ long afs_auxcount = 0;	/* In case there're leaks */
static struct vauxinode *afs_GetAuxSlot()
{
    register struct vauxinode *auxp;

    if (!auxInitialized) aux_init();
    if (!FreeAuxList) afs_GetDownAux(5);
    if (!FreeAuxList) {
	afs_auxcount++;
	auxp = (struct vauxinode *) osi_Alloc(sizeof (struct vauxinode));
    } else {
	auxp = FreeAuxList;
	FreeAuxList = (struct vauxinode *) auxp->aux_lruq.next;
    }
    bzero(&auxp->aux_dimage, sizeof(struct dauxinode));
    QAdd(&AUXLRU, &auxp->aux_lruq);
    if (auxp->aux_lruq.prev == &auxp->aux_lruq) panic("afs_GetAuxSlot: lruq!");
    return auxp;
}

static int afs_CreateAuxEntry(ip, magic, param1, param2, param3, param4)
    register struct inode *ip;
{
    register long i;
    register struct vauxinode *auxp;

    ObtainWriteLock(&afs_xiparams);
    auxp = afs_GetAuxSlot();
    ReleaseWriteLock(&afs_xiparams);
    if (auxheader.magic != magic)
	panic("Bad magic number=%d(should be=%d)\n", magic,auxheader.magic);
    auxp->aux_dimage.aux_magic = magic;
    auxp->aux_dimage.aux_param1 = param1;
    auxp->aux_dimage.aux_param2 = param2;
    auxp->aux_dimage.aux_param3 = param3;
    auxp->aux_dimage.aux_param4 = param4;
    /* Add it to the proper hash entry */
    i = AHash(ip->i_number);
    auxp->aux_next = aux_table[i];
    aux_table[i] = auxp;
    auxp->aux_dev = ip->i_dev;
    auxp->aux_afspart  = afs_FindPartByDev(auxp->aux_dev);
    if (auxp->aux_afspart == 0)
	panic("No afspart entry for the dev =%d yet!\n", auxp->aux_dev);
    auxp->aux_ino = ip->i_number;
    auxp->aux_refCount = 1; 
    /* For now also flush the created entry into the Auxvolfile; a better way would be to flush these guys periodically (similar to std buffer algorithms) but this will suffice for the time being... */
    afs_WriteAuxInode(auxp,1,0);   
}
	
afs_GetDownAux(anumber)
    int anumber; 
{
    register struct afs_q *tq, *uq;
    register struct vauxinode *auxp;

    if (CheckLock(&afs_xiparams) != -1) panic("getdownAux lock");
    for(tq = AUXLRU.prev; tq != &AUXLRU && anumber > 0; tq = uq) {
	auxp = QTOA(tq);
	uq = QPrev(tq);
	if (auxp->aux_refCount == 0) {
	    afs_FlushAuxCache(auxp,0);
	    anumber--;
	}
    }
}


afs_FlushAuxCache(auxp, zeroit)
    register struct vauxinode *auxp; 
    register int zeroit;    /* if set, the disk image is zeroed... */
{
    register long i;
    register struct vauxinode **uvc, *wvc;

    if (auxp->aux_refCount != 0) {
	if (aux_debug) printf("afs_FlushAuxCache: refCount = %d!!\n", auxp->aux_refCount);
	return EBUSY;
    }
    /* pull the entry out of the lruq and put it on the free list */
    QRemove(&auxp->aux_lruq);
    /* remove entry from the hash chain */
    i = AHash(auxp->aux_ino);
    uvc = &aux_table[i];
    for(wvc = *uvc; wvc; uvc = &wvc->aux_next, wvc = *uvc) {
	if (auxp == wvc) {
	    *uvc = auxp->aux_next;
	    break;
	}
    }
    if (!wvc) {
	panic("flushAuxCache: auxp=%x (ino=%d) not in hash table...", auxp, auxp->aux_ino);
    }
    /* put the entry in the free list and free the callback */
    afs_WriteAuxInode(auxp,1, zeroit);
    auxp->aux_lruq.next = (struct afs_q *) FreeAuxList;
    FreeAuxList = auxp;
    return 0;
}


struct vauxinode *afs_GetAuxInode(ip)
    register struct inode *ip;
{
    register long i;
    register struct vauxinode *auxp;

    i = AHash(ip->i_number);
    ObtainWriteLock(&afs_xiparams);
    for(auxp = aux_table[i]; auxp; auxp = auxp->aux_next) {
	if (auxp->aux_ino == ip->i_number && auxp->aux_dev == ip->i_dev) {
	    QRemove(&auxp->aux_lruq);
	    QAdd(&AUXLRU, &auxp->aux_lruq);
	    auxp->aux_refCount++;
	    ReleaseWriteLock(&afs_xiparams);	   
	    return auxp;
	}
    }
    /* Get it from the disk */
    auxp = afs_GetAuxSlot();	
    auxp->aux_next = aux_table[i];
    aux_table[i] = auxp;
    auxp->aux_dev = ip->i_dev;
    auxp->aux_afspart  = afs_FindPartByDev(auxp->aux_dev);
    if (auxp->aux_afspart == 0)
	panic("afs_GetAuxInode: No afspart entry for the dev =%d yet!\n", auxp->aux_dev);
    auxp->aux_ino = ip->i_number;
    if (auxp->aux_dev != auxp->aux_afspart->cdev.dev) printf("aux_dev=%d, part->cdev.dev=%d\n", auxp->aux_dev, auxp->aux_afspart->cdev.dev);
    auxp->aux_refCount = 1;
    ReleaseWriteLock(&afs_xiparams);
    afs_ReadAuxInode(auxp);
    return auxp;
}

static afs_PutAuxInode(auxp)
    register struct vauxinode *auxp;
{
    --auxp->aux_refCount;
}

static int afs_ReadAuxInode(auxp)
    register struct vauxinode *auxp;
{
    register struct osi_file *tfile;
    register long code;

    if (aux_debug) printf("In afs_ReadAuxInode(auxp=%x, dev=%d, fInode=%d, ino=%d)\n", auxp, auxp->aux_afspart->cdev.dev, auxp->aux_afspart->fileInode, auxp->aux_ino);
    tfile = osi_UFSOpen(&auxp->aux_afspart->cdev, auxp->aux_afspart->fileInode);
    if (!tfile) panic("open dauxinode file");
    osi_Seek(tfile, sizeof(struct dauxinode) * (int)auxp->aux_ino + sizeof(struct afs_auxheader));
    code = osi_Read(tfile, (char *)&(auxp->aux_dimage), sizeof(struct dauxinode));
    osi_Close(tfile);
    if (code != sizeof(struct dauxinode)) panic("read dauxinode file(%d != %d", code, sizeof(struct dauxinode));
}

static int afs_WriteAuxInode(auxp, atime, zeroit)
    register struct vauxinode *auxp;
    register long atime, zeroit;
{
    register struct osi_file *tfile;
    struct dauxinode dummy;
    register long code;

    if (aux_debug) printf("In afs_WriteAuxInode(auxp=%x, dev=%d, fInode=%d)\n", auxp, auxp->aux_afspart->cdev.dev, auxp->aux_afspart->fileInode);
    if (atime) auxp->aux_dimage.aux_modTime = osi_Time();
    tfile = osi_UFSOpen(&auxp->aux_afspart->cdev, auxp->aux_afspart->fileInode);
    if (!tfile) panic("open dauxinode file");
    osi_Seek(tfile, sizeof(struct dauxinode) * (int)auxp->aux_ino + sizeof(struct afs_auxheader));
    if (zeroit)	{
	/* We zero the entry usually when the representing inode is been unlinked */
	bzero(&dummy, sizeof(struct dauxinode));
	code = osi_Write(tfile, (char *)&dummy, sizeof(struct dauxinode));
    } else {	
	code = osi_Write(tfile, (char *)&(auxp->aux_dimage), sizeof(struct dauxinode));
    }
    osi_Close(tfile); 
    if (code != sizeof(struct dauxinode)) panic("write dauxinode file");
}

/* AFSOPs 30-40 are reserved for the aix server; the following opcodes should move in afs.h */
#define	AFSOP_AUXINIT	30
#define	AFSOP_AUXFILE	31
#define	AFSOP_AUXDEBON	32
#define	AFSOP_AUXDEBOFF	33

#ifdef	AFS_SERVER
afs_call() {
    afs_auxcall();
}
#endif

/* It only supports a few calls */
afs_auxcall () {
    register struct a {
	    long parm;
	    long parm2;
	    long parm3;
	    long parm4;
	    long parm5;
	    long parm6;
    } *uap = (struct a *)u.u_ap;
    long code;

    if (!afs_suser()) {	/* only root can run this code */
	u.u_error = EACCES;
	return;
    }
    if (aux_debug) printf("afs_call-> op=%d\n", uap->parm);
    if (uap->parm == AFSOP_AUXINIT) {
	aux_init();
    } else if (uap->parm == AFSOP_AUXFILE) {
	char tbuffer[256];
	long bufferSize, code;

	/* Check error code possibility here? */
	bufferSize = bcopyin(uap->parm2, tbuffer, 256);
	if (bufferSize == -1) return -1;
	tbuffer[255] = 0;	/* null-terminate the name */
	/* we now have the cache dir copied in.  Call the cache init routines */
	if (aux_debug) printf("afs_call(op=2): file=%s\n", tbuffer);
	code = afs_InitAuxVolFile(tbuffer);
	return code;
    } else if (uap->parm == AFSOP_AUXDEBON) {
	printf("afs_call(op=3): setting aux_debug flag on!\n");
	aux_debug = 3;
	/* Print interesting structures.... */
	tmpdbg_parttbl();
	tmpdbg_auxtbl();
    } else if (uap->parm == AFSOP_AUXDEBOFF) {
	printf("afs_call(op=4): setting aux_debug flag off!\n");
	aux_debug = 0;
    }	
    return 0;
}

/* This to be deleted soon... */
tmpdbg_parttbl()
{
    register struct afspart *tc;
    int count = 0;

    if (aux_debug) printf("DUMP of al entries in afs_parts list\n");
    for(tc = afs_parts; tc; tc = tc->next) {
	if (aux_debug) printf("tc=%x, cdev=%d, fileinode=%d\n", tc, tc->cdev.dev, tc->fileInode);
	count++;
    }
    if (aux_debug) printf("Total # partition entries=%d\n", count);
 }

/* This to be deleted soon... */
tmpdbg_auxtbl()
{
    register struct vauxinode *auxp;
    register int i, count = 0;

    for (i=0; i < NHAUXINO; i++) {
	for (auxp = aux_table[i]; auxp; auxp = auxp->aux_next) {
	    if (aux_debug) printf("auxp=%x,dev=%d, ino=%d, rcnt=%d, p1=%d, p2=%d\n",auxp, auxp->aux_dev, auxp->aux_ino, auxp->aux_refCount, auxp->aux_dimage.aux_param1, auxp->aux_dimage.aux_param2);
	    count++;
	}
    }
    if (aux_debug) printf("Total # entries=%d\n", count);   
}

#ifdef	AFS_SERVER
/* Some extra handling is needed when calling the aix's version of the local RDWR module, particularly */
/* the usage of the uio structure to the lower routine. Note of significant importance to the AFS port is the */
/* offset in/out parameter, which in two cases returns a new value back. The cases are: (1) when it's a FIFO */
/* file (it's reset to zero) which we don't yet support in AFS and (2) in a regular case when we write */
/* (i.e. rw == UIO_WRITE) and are in appending mode (i.e. FAPPEND bit on) where offset is set to the */
/* file's size.None of these cases apply to us currently so no problems occur; caution if things change! */
int
gop_rdwr(rw, vp, base, len, offset, segflg, unit, aresid)
enum uio_rw	rw;
struct vnode	*vp;
caddr_t		base;
off_t		*offset;
int		len, segflg;
int		*aresid;
int		unit;	    /* Ignored */
{
    struct uio	uio_struct;
    struct iovec uiovector;
    register int code;

    /* Set up the uio structure */
    uiovector.iov_base = (caddr_t) base;
    uiovector.iov_len = len;

    uio_struct.uio_iov = &uiovector;
    uio_struct.uio_iovcnt = 1;
    uio_struct.uio_offset = *offset;
    uio_struct.uio_segflg = AFS_UIOSYS;
    uio_struct.uio_resid = len;
    uio_struct.uio_fmode = (rw == UIO_READ ? FREAD : FWRITE);

    code = VNOP_RDWR(vp, rw, (rw == UIO_READ ? FREAD : FWRITE), (off_t)offset, &uio_struct, NULL, NULL, -1);
    *aresid = uio_struct.uio_resid;
    return code;
}
#endif	AFS_SERVER
#endif /*(defined(AFS_AIX_ENV) && defined(AFS_SERVER))*/
