/* 
 * Mach Operating System
 * Copyright (c) 1990 Carnegie-Mellon University
 * Copyright (c) 1989 Carnegie-Mellon University
 * Copyright (c) 1988 Carnegie-Mellon University
 * All rights reserved.  The CMU software License Agreement specifies
 * the terms and conditions for use and redistribution.
 */
/*
 * HISTORY
 * $Log$
 */

/*
 *	Apple Macintosh II Mach (macmach)
 *
 *	File: mac2/pmap.c
 *	Author: David E. Bohman II (CMU macmach)
 */

/*
 * Manages physical address maps.
 *
 * In addition to hardware address maps, this
 * module is called upon to provide software-use-only
 * maps which may or may not be stored in the same
 * form as hardware maps.  These pseudo-maps are
 * used to store intermediate results from copy
 * operations to and from address spaces.
 *
 * Since the information managed by this module is
 * also stored by the logical address mapping module,
 * this module may throw away valid virtual-to-physical
 * mappings at almost any time.  However, invalidations
 * of virtual-to-physical mappings must be done as
 * requested.
 *
 * In order to cope with hardware architectures which
 * make virtual-to-physical map invalidates expensive,
 * this module may delay invalidate or reduced protection
 * operations until such time as they are actually
 * necessary.  This module is given full information as
 * to which processors are currently using which maps,
 * and to when physical maps must be made correct.
 */

#include <cpus.h>
#include <ramdisk.h>
#include <mach_xp.h>

#include <machine/vm_types.h>

#include <mach/boolean.h>
#include <kern/thread.h>
#include <kern/zalloc.h>
#include <kern/lock.h>

#include <mach/vm_param.h>
#include <mach/vm_prot.h>

#include <vm/pmap.h>
#include <vm/vm_map.h>
#include <vm/vm_kern.h>
#include <vm/vm_page.h>
#include <vm/vm_user.h>

#include <mac2/pmap_kmap.h>

/*
 * Private data structures.
 */

/*
 * For each vm_page_t, there is a list of all currently
 * valid virtual mappings of that page.  An entry is
 * a pv_entry_t; the list is the pv_head_table.
 */

typedef struct pv_entry {
	struct pv_entry	*next;		/* next pv_entry */
	pmap_t		pmap;		/* pmap where mapping lies */
	vm_offset_t	va;		/* virtual address for mapping */
} *pv_entry_t;

#define	PV_ENTRY_NULL	((pv_entry_t) 0)

pv_entry_t	pv_head_table;		/* array of entries, one per page */
zone_t		pv_list_zone;		/* zone of pv_entry structures */

/*
 * Each entry in the pv_head_table is locked by a bit in the
 * pv_lock_table.  The lock bits are accessed by the physical
 * address of the page they lock.
 */

char	*pv_lock_table;		/* pointer to array of bits */
#define	pv_lock_table_size(n)	(((n)+BYTE_SIZE-1)/BYTE_SIZE)

/*
 * First and last physical addresses that we maintain any information
 * for.  Initialized to zero so that pmap operations done before
 * pmap_init won't touch any non-existent structures.
 */
vm_offset_t	vm_first_phys = (vm_offset_t) 0;
vm_offset_t	vm_last_phys  = (vm_offset_t) 0;

/*
 * Index into pv_head table, its lock bits, and the modify bits
 * starting at vm_first_phys.
 */

#define	pa_index(pa)	(atop(pa - vm_first_phys))

#define	pai_to_pvh(pai)		(&pv_head_table[pai])
#define	lock_pvh_pai(pai)	(bit_lock(pai, pv_lock_table))
#define	unlock_pvh_pai(pai)	(bit_unlock(pai, pv_lock_table))

/*
 * Array of modify & reference bits, one byte
 * per physical page.
 */
char	*pmap_page_attributes;
#define	PG_MOD	    0x01	/* page is modified */
#define	PG_REF	    0x02	/* page has been referenced */

/*
 * Page table pointer tables are allocated
 * in non-pageable kernel virtual memory.
 * Each ptptbl is 1024 bytes, so 8 of them fit
 * in a page.
 */
struct pt_root_page {
    queue_chain_t	link;		/* link in free queue */
    vm_offset_t		page;		/* ptr to base of page */
    int			free_count;	/* number of free ptptbls in page */
    unsigned char	free_map;	/* bitmap of free ptptbls in page */
};
typedef struct pt_root_page	*ptrt_t;
#define PT_ROOT_PAGE_NULL	((ptrt_t)0)

zone_t	ptrt_zone;

queue_head_t	pt_root_page_queue_free; /* list of above pages that contain
					  * at least one free ptptbl
					  */
simple_lock_data_t
    		pt_root_page_lock; /* lock for above */

/*
 * pt_root_page manipulation macros.
 */

/*
 * Given a ptr to a ptptbl, return the
 * pt_root_page index of the ptptbl.
 */
#define get_pt_root_page_index(ptptbl) \
	(((vm_offset_t)(ptptbl)&(MAC2_PGBYTES-1))>>10)

/*
 * Given the base address of a pt_root_page
 * and a pt_root_page index of a ptptbl, return
 * the address of the ptptbl.
 */
#define get_ptptbl_addr(page, n) ((page)|((n)<<10))

/*
 * Inline expansions for special
 * PMMU functions
 */
#include <machine/pmmu_inline.c>

/*
 * Character bitmap allocation routines (inline expansions).
 *
 * The bit offsets are numbered as follows:
 *
 *	7 6 5 4 3 2 1 0
 *	M	      L
 *	S	      S
 *	B	      B
 */
#include <machine/bitmap_inline.c>

/*
 * Initialize a bitmap.

void
init_bitmap_byte(bp);
    unsigned char *bp;

	movl sp@(4),a0
	bfset a0@{#0:#8}
*/

/*
 * Allocate an element in a bitmap.
 * returns -1 if no free elements,
 * otherwise the bit offset.

int
alloc_bitmap_byte(bp);
    unsigned char *bp;

	movl sp@(4),a0
	bfffo a0@{#0:#8},d1
	bne 0f
	movl #-1,d0
	bra 1f
0:	bfclr a0@{d1:#1}
	moveq #7,d0
	subl d1,d0
1:
*/

/*
 * Free an element in a bitmap.
 * returns -1 if the element
 * was already free, 0 otherwise.
 * NB: the element number is
 * not checked for range errors.

int
free_bitmap_byte(bp, n);
    unsigned char *bp;
    int n;

	movl sp@(4),a0
	movl sp@(8),d1
	moveq #7,d0
	subl d1,d0
	bfset a0@{d0:#1}
	beq 0f
	movl #-1,d0
	bra 1f
0:	clrl d0
1:
*/

/*
 * Physical memory is reserved for page pointer tables.
 * They are allocated one page at a time when necessary.
 */
struct pt_page {
	queue_chain_t	link;		/* link in active or free queues */
	pmap_t		pmap;		/* pmap this page is allocated to */
	vm_offset_t	va;		/* first virtual address mapped
					 * by this page
					 */
	int		use_count;	/* number of mappings in use */
	int		wired_count;	/* number of mappings wired in */
};
typedef	struct pt_page	*ptpg_t;
#define	PT_PAGE_NULL	((ptpg_t)0)

ptpg_t	ptpg_array;			/* pointer to array of ptpg_t
					   for each pt page allocated */

queue_head_t	pt_page_queue_active;	/* list of all active pt pages */
queue_head_t	pt_page_queue_free;	/* list of all free pt pages */

simple_lock_data_t
		pt_page_queue_lock;	/* both queues are locked either
					   by this lock, or by a write-
					   lock on the pmap system lock. */

vm_offset_t	first_pt_pa;		/* physical address of first pt page */
int		first_pt_pfn;		/* page number for above */

/*
 * Count of pt pages allocated at
 * boot.  Can be patched.
 */
int		pt_page_alloc_count = 30;

/*
 * pt_page manipulation macros.
 */

/*
 * Given a ptr to a pp, return
 * a ptr to the corresponding
 * pt_page.
 */
#define	get_pt_page(pp)		(pfn_to_pt_page(mac2_btop(pp)))

/*
 * Given the pfn of a page table,
 * return a ptr to the corresponding
 * pt_page.
 */
#define	pfn_to_pt_page(pfn)	(&ptpg_array[(pfn) - first_pt_pfn])

/*
 * Given a ptr to a pt_page, return
 * the corresponding pfn.
 */
#define	pt_page_to_pfn(ptpg)	(first_pt_pfn + ((ptpg) - ptpg_array))

/*
 * Given a ptr to a pt_page, return
 * a ptr to the corresponding page table.
 */
#define	get_pt_addr(ptpg)	(mac2_ptob(pt_page_to_pfn(ptpg)))

/*
 * Return the size of memory
 * mapped by one page-pointer table.
 */
#define	pt_page_map_size	(mac2_ptob(MAC2_PGBYTES/sizeof(pp_t)))

/*
 * Address translation tree.
 *
 * An address translation tree contains 2 levels.
 * The top level is called the page-table-pointer table (ptptbl).
 * It consists of short format table descriptors, and is indexed
 * by the top 8 bits of the virtual address (256 entries).
 * Each of these page-table-pointers (ptp) can point to a
 * page-pointer table (pptbl).  Each pptbl consists of short format
 * page descriptors and is indexed by the next lower 11 bits of
 * of the virtual address (2048 entries).  Each pptbl consists
 * of 8192 bytes (a machine dependent memory page), and is
 * allocated as a pt_page by the pmap module.  Each of these
 * page-pointers (pp) can point to a physical memory page, which
 * is indexed by the lower 13 bits of the virtual address.
 *
 * The kernel has it's own address translation tree which is
 * defined by the kernel_pmap, and pointed to by the PMMU
 * supervisor root pointer (SRP).  Thus the address space
 * for the kernel is 2^32 bytes.
 * 
 * Each task has it's own address translation tree which is
 * defined by it's pmap, and pointed to by the PMMU
 * cpu root pointer (CRP) while a thread owned by the task is executing.
 * The address space for each task is also 2^32 bytes.
 */

/*
 * Address Translation tree walking
 * primatives (inline expansions).
 *
 * NB: these primatives DO NOT check
 * for zero pointers or invalid ptp's or pp's.
 */
#include <machine/pmap_inline.c>

/*
 * Given a ptr to a page-table-pointer table and
 * an offset, return a ptr to the corresponding
 * page-table-pointer.

ptp_t	*ptptbl_to_ptp(p, v);
    ptp_t	*p;
    vm_offset_t	v;

	movl	sp@(4), a0
	movl	sp@(8), d0
	bfextu	d0{#0:#8}, d0
	lea	a0@(0,d0:L:4), a0
	movl	a0, d0
*/

/*
 * Given a ptr to a page-table-pointer and
 * an offset, return a ptr to the corresponding
 * page-pointer.

pp_t	*ptp_to_pp(p, v);
    ptp_t	*p;
    vm_offset_t	v;

	movl	sp@(4), a0
	movl	sp@(8), d0
	bfextu	a0@{#0:#19}, d1
	movl	d2, sp@-
	moveq	#0, d2
	bfins	d1, d2{#0:#19}
	movl	d2, a0
	movl	sp@+, d2
	bfextu	d0{#8:#11}, d0
	lea	a0@(0,d0:L:4), a0
	movl	a0, d0
*/

/*
 * Given a ptr to a page-table-pointer, return
 * a ptr to the corresponding page-pointer table.

pp_t	*ptp_to_pptbl(p);
    ptp_t	*p;

	movl	sp@(4), a0
	bfextu	a0@{#0:#19}, d1
	moveq	#0, d0
	bfins	d1, d0{#0:#19}
*/

/*
 * Given a ptr to a page-pointer table and
 * an offset, return a ptr to the corresponding
 * page-pointer.

pp_t	*pptbl_to_pp(p, v);
    pp_t	*p;
    vm_offset_t	v;

	movl	sp@(4), a0
	movl	sp@(8), d0
	bfextu	d0{#8,#11}, d0
	lea	a0@(0,d0:L:4), a0
	movl	a0, d0
*/

/*
 * Given a ptr to a ptp, return the corresponding
 * virtual offset.

vm_offset_t ptp_to_va(p);
    ptp_t   *p;

	movl	sp@(4), d0
	bfextu	d0{#22:#8}, d1
	moveq	#0, d0
	bfins	d1, d0{#0:#8}
*/

/*
 *	Statistics
 */
int	pt_page_active_count	= 0;
int	pt_page_free_count	= 0;
int	pt_page_wired_count	= 0;
int	pt_page_max_active	= 0;
int	pt_page_max_wired	= 0;
int	pt_root_page_alloc_count= 0;
int	pt_root_page_free_count	= 0;

/*
 *	Locking Protocols:
 *
 *	There are two structures in the pmap module that need locking:
 *	the pmaps themselves, and the per-page pv_lists (which are locked
 *	by locking the pv_lock_table entry that corresponds to the pv_head
 *	for the list in question.)  Most routines want to lock a pmap and
 *	then do operations in it that require pv_list locking -- however
 *	pmap_remove_all and pmap_copy_on_write operate on a physical page
 *	basis and want to do the locking in the reverse order, i.e. lock
 *	a pv_list and then go through all the pmaps referenced by that list.
 *	To protect against deadlock between these two cases, the pmap_lock
 *	is used.  There are three different locking protocols as a result:
 *
 *  1.  pmap operations only (pmap_extract, pmap_access, ...)  Lock only
 *		the pmap.
 *
 *  2.  pmap-based operations (pmap_enter, pmap_remove, ...)  Get a read
 *		lock on the pmap_lock (shared read), then lock the pmap
 *		and finally the pv_lists as needed [i.e. pmap lock before
 *		pv_list lock.]
 *
 *  3.  pv_list-based operations (pmap_remove_all, pmap_copy_on_write, ...)
 *		Get a write lock on the pmap_lock (exclusive write); this
 *		also guaranteees exclusive access to the pv_lists.  Lock the
 *		pmaps as needed.
 *
 *	At no time may any routine hold more than one pmap lock or more than
 *	one pv_list lock.  Because interrupt level routines can allocate
 *	mbufs and cause pmap_enter's, the pmap_lock and the lock on the
 *	kernel_pmap can only be held at splvm.
 */

#define	SPLVM(spl)	{ spl = splvm(); }
#define	SPLX(spl)	{ splx(spl); }

#define	PMAP_READ_LOCK(pmap, spl)	SPLVM(spl)
#define	PMAP_WRITE_LOCK(spl)		SPLVM(spl)
#define	PMAP_READ_UNLOCK(pmap, spl)	SPLX(spl)
#define	PMAP_WRITE_UNLOCK(spl)		SPLX(spl)
#define	PMAP_WRITE_TO_READ_LOCK(pmap)

#define	LOCK_PVH(index)
#define	UNLOCK_PVH(index)

/*
 * CPU Cache managment.
 */
#define PMAP_FLUSH_CPU_CACHES(pmap)			\
{							\
    if ((pmap) == kernel_pmap || (pmap)->cpus_using)	\
	flush_cpu_caches();				\
}

/*
 * Address Translation Cache (ATC) managment.
 */

extern is68030;

/*
 * Flush ATC entries corresponding to the
 * given pmap and address range (start inclusive, end exclusive).
 */
#define	PMAP_UPDATE_ATC(pmap, s, e)			\
{							\
    if ((pmap) == kernel_pmap) {			\
        ATC_INVALIDATE_SUPR_RANGE((s), (e));		\
    }							\
    else if ((pmap)->cpus_using) {			\
	if ((pmap)->alt_root == 0) {			\
	    ATC_INVALIDATE_USER_RANGE((s), (e));	\
	}						\
	else {						\
	    if ((pmap)->flags&PMAP_F_24BIT) {		\
		ATC_INVALIDATE_USER_ALL();		\
		ATC_INVALIDATE_MAP((pmap)->root);	\
	    }						\
	    else {					\
		ATC_INVALIDATE_USER_RANGE((s), (e));	\
		ATC_INVALIDATE_MAP((pmap)->alt_root);	\
	    }						\
	}						\
    }							\
    else {						\
        ATC_INVALIDATE_MAP((pmap)->root);		\
	if ((pmap)->alt_root)				\
	    ATC_INVALIDATE_MAP((pmap)->alt_root);	\
    }							\
}

/*
 * This constant defines the maximum number
 * of entries that will be flushed without
 * flushing all of the supervisor or current user
 * entries from the ATC.
 */
#define	ATC_PAGE_INVAL_MAX 8

/*
 * Invalidate the given range of (global) kernel virtual
 * addresses from the ATC.
 */
#define	ATC_INVALIDATE_SUPR_RANGE(s, e)				\
{								\
    if ((s) < (e) &&						\
	(e) - (s) < ATC_PAGE_INVAL_MAX*MAC2_PGBYTES) {		\
	register vm_offset_t _va;				\
\
	if (is68030) {						\
	    for (_va = (s); _va < (e); _va += MAC2_PGBYTES)     \
	        pmmu_flush_supr(_va);				\
	}							\
	else {							\
	    for (_va = (s); _va < (e); _va += MAC2_PGBYTES)     \
	        pmmu_flush_supr_shared(_va);			\
	}							\
    }								\
    else {						        \
	if (is68030)						\
	    pmmu_flush_supr_all();				\
	else							\
	    pmmu_flush_supr_shared_all();			\
    }								\
}

/*
 * Invalidate the given range of (current) user virtual
 * addresses from the ATC.  Used when the range
 * is in the address translation tree currently
 * pointed at by the CRP.
 */
#define	ATC_INVALIDATE_USER_RANGE(s, e)				\
{								\
    if ((s) < (e) &&						\
	(e) - (s) < ATC_PAGE_INVAL_MAX*MAC2_PGBYTES) {		\
	register vm_offset_t _va;				\
\
        for (_va = (s); _va < (e); _va += MAC2_PGBYTES)		\
	    pmmu_flush_user(_va);				\
    }								\
    else {							\
	pmmu_flush_user_all();					\
    }								\
}

/*
 * Unconditionally invalidate all (current) user virtual
 * addresses from the ATC.
 */
#define ATC_INVALIDATE_USER_ALL()	\
    pmmu_flush_user_all();

/*
 * Invalidate all of the ATC entries for
 * the tree which is rooted at physical
 * address r.  Used when the address
 * translation tree is not the active
 * one pointed at by the CRP.
 */
#define	ATC_INVALIDATE_MAP(r)	    \
if (!is68030) {			    \
    PMMU_RP_reg _rp;		    \
				    \
    _rp.limit = PMMU_RP_LIMIT;	    \
    _rp.valid = PMMU_VALID_RP;	    \
    _rp.phys = (long)(r);	    \
    pmmu_flush_map(&_rp);	    \
}

/*
 * Translate a MACH vm_prot_t into
 * PMMU write-protect value.
 */
#define	pmmu_prot(prot)		(((prot)&VM_PROT_WRITE)?PMMU_RW:PMMU_WP)

/*
 * The kernel pmap is statically
 * allocated.
 */
struct pmap	kernel_pmap_store;
pmap_t		kernel_pmap;

/*
 * Zone used to allocate
 * pmap structures.
 */
zone_t		pmap_zone;

/*
 * Page-table-pointer tables
 * exist 1-to-1 with pmaps and
 * are created at pmap creation time.
 */
#define		pmap_ptptbl_size	(256*sizeof(ptp_t))

/*
 * Number of page-pointers required to
 * map one machine indepedent page.
 * Calculated at initialization.
 */
int		pps_per_vm_page;

/*
 * Has pmap_init completed?
 */
boolean_t	pmap_initialized = FALSE;

/*
 * Amount of space reserved in the kernel pmap
 * to be used for allocating kernel virtual memory
 * once the system is up.
 */
vm_size_t   kernel_alloc_space = (48*1024*1024);    /* 48 Meg */

/*
 * Given an offset and a pmap, compute the address of the
 * pp.  If the address is invalid with respect to the pmap
 * then PP_NULL is returned.
 */
inline
pp_t *
pmap_pp(pmap, va)
    register pmap_t	    pmap;
    register vm_offset_t    va;
{
    register ptp_t	    *ptp;

    ptp = pmap->root;
    if (ptp == PTP_NULL)
	return (PP_NULL);

    ptp = ptptbl_to_ptp(ptp, va);
    if (ptp->valid == PMMU_INVALID)
	return (PP_NULL);

    return (ptp_to_pp(ptp, va));
}

/*
 * External form of the PMAP_UPDATE_ATC() macro.
 */
pmap_update_atc(pmap, start, end)
register pmap_t pmap;
register vm_offset_t start, end;
{
    PMAP_UPDATE_ATC(pmap, start, end);
}

static vm_prot_t pmmu_prot_map[8];

static
pmmu_prot_map_init()
{
    register comb;

    for (comb = 0; comb < 8; comb++)
	switch (comb) {
	  case VM_PROT_NONE:
	  case VM_PROT_EXECUTE:
	  case VM_PROT_WRITE:
	  case VM_PROT_WRITE | VM_PROT_EXECUTE:
	    pmmu_prot_map[comb] = VM_PROT_NONE;
	    break;

	  case VM_PROT_READ:
	  case VM_PROT_READ | VM_PROT_EXECUTE:
	    pmmu_prot_map[comb] = VM_PROT_READ;
	    break;

	  case VM_PROT_READ | VM_PROT_WRITE:
	  case VM_PROT_READ | VM_PROT_WRITE | VM_PROT_EXECUTE:
	    pmmu_prot_map[comb] = VM_PROT_READ | VM_PROT_WRITE;
	    break;
	}
}

/*
 * Map memory in kernel pmap at initialization.
 * The physical addresses being mapped are not managed
 * and are never unmapped.  pptbl space must already have
 * been allocated in the kernel_pmap by pmap_alloc_kernel_pptbl().
 */
vm_offset_t
pmap_map(va, sphys, ephys, prot)
    register vm_offset_t	va;
    register vm_offset_t	sphys;
    register vm_offset_t	ephys;
    register int		prot;
{
    register size;

    size = PAGE_SIZE;
    while (sphys < ephys) {
	pmap_enter(kernel_pmap, va, sphys, prot, FALSE);
	pmap_resident_count(kernel_pmap)--; /* XXX */
	va += size;
	sphys += size;
    }
    return (va);
}

/*
 * Allocate page-pointer tables in the kernel_pmap from
 * physical memory starting at *aphys_free, according to
 * the indicated offset and size.
 */
void
pmap_alloc_kernel_pptbl(aphys_free, va, size)
    vm_offset_t	    *aphys_free;
    vm_offset_t	    va;
    vm_size_t	    size;
{
    register ptp_t	    *ptp;
    register vm_offset_t    pa;
    register		    nptpgs;

    nptpgs = (size+pt_page_map_size-1)/pt_page_map_size;
    ptp = ptptbl_to_ptp(kernel_pmap->root, va);
    for (; nptpgs-- > 0; ptp++) {
	if (ptp->valid == PMMU_VALID_PTP)	/* pptbl already allocated */
	    continue;
	pa = *aphys_free;
	*aphys_free += MAC2_PGBYTES;
	bzero((caddr_t)pa, MAC2_PGBYTES);
	ptp->pfn = mac2_btop(pa);
	ptp->valid = PMMU_VALID_PTP;
    }
}

/*
 * Bootstrap the system enough to run with virtual memory.
 * Create kernel_pmap by mapping kernel text and data, physical
 * memory, IO and NuBus slots as virt == phys. Called with
 * mapping OFF, enables kernel mapping before returning.
 * page_size must already be set.
 *
 *	Parameters:
 *	aphys_free	ptr to pa of first available physical page
 *	aphys_last	ptr to pa of last available physical page
 *	avirt_free	ptr to va of first available virtual page address
 *	avirt_last	ptr to va of last available virtual page address
 *
 *	&start		start of kernel text
 *	&etext		end of kernel text
 */
void
pmap_bootstrap(aphys_free, aphys_last, avirt_free, avirt_last)
    vm_offset_t	    *aphys_free;	/* IN/OUT */
    vm_offset_t	    *aphys_last;	/* IN/OUT */
    vm_offset_t	    *avirt_free;	/* OUT */
    vm_offset_t	    *avirt_last;	/* OUT */
{
    register ptp_t	*ptp;
    register pp_t	*pp;
    vm_offset_t		va;
    PMMU_RP_reg		SRP;
    PMMU_TC_reg		TC;
    extern int		start, etext;
#if	RAMDISK
    extern		ramd_size;
#endif

    pmmu_prot_map_init();

    /*
     *	Set pps_per_vm_page for general use.
     */
    pps_per_vm_page = PAGE_SIZE / MAC2_PGBYTES;

    /*
     *	The kernel's pmap is statically allocated so we don't
     *	have to use pmap_create, which is unlikely to work
     *	correctly at this part of the boot sequence.
     */
    kernel_pmap = &kernel_pmap_store;

    simple_lock_init(&kernel_pmap->lock);

    /*
     * Allocate a page tbl ptr table
     * for the kernel pmap.
     */
    kernel_pmap->root = (ptp_t *)*aphys_free;
    *aphys_free += MAC2_PGBYTES;

    /*
     * zero it.
     */
    bzero((caddr_t)kernel_pmap->root, MAC2_PGBYTES);

    kernel_pmap->ref_count = 1;

    /*
     * Allocate enough page ptr table
     * space to map all of physical
     * memory as virt == phys.
     */
#if	RAMDISK
    pmap_alloc_kernel_pptbl(aphys_free, VM_MIN_KERNEL_ADDRESS,
			    mem_size + ramd_size);
#else
    pmap_alloc_kernel_pptbl(aphys_free, VM_MIN_KERNEL_ADDRESS, mem_size);
#endif

    /*
     *	Map low system memory		RW
     *	Map kernel text			RO
     *	Map kernel data-end of memory	RW
     */
    va = pmap_map(VM_MIN_KERNEL_ADDRESS, 0, &start,
		  VM_PROT_READ|VM_PROT_WRITE);
    va = pmap_map(va, &start, round_page(&etext), VM_PROT_READ);
#if	RAMDISK
    va = pmap_map(va, round_page(&etext), mem_size + ramd_size,
		  VM_PROT_WRITE|VM_PROT_READ);
#else
    va = pmap_map(va, round_page(&etext), mem_size,
		  VM_PROT_WRITE|VM_PROT_READ);
#endif

    /*
     * Allocate enough extra kernel_pmap
     * space to map an extra kernel_alloc_space bytes.
     */
    pmap_alloc_kernel_pptbl(aphys_free, va, kernel_alloc_space);

    /*
     * Set virtual_avail and virtual_end
     * to describe the virtual memory space
     * that the kernel may allocate from.
     */
    *avirt_free = va;
    *avirt_last = va + kernel_alloc_space;

    /*
     * Enter special virt == phys
     * mappings into the kernel pmap.
     */
#define	pmap_map_kernel_special(addr, size, prot) \
    {								\
	pmap_alloc_kernel_pptbl(aphys_free, (addr), (size));	\
	pmap_map((addr), (addr), (addr) + (size), (prot));	\
    }

#ifdef	PMAP_ROM_MAP_ADDR
    pmap_map_kernel_special(PMAP_ROM_MAP_ADDR,
			    PMAP_ROM_MAP_SIZE,
			    PMAP_ROM_MAP_PROT);
#endif

#ifdef	PMAP_IO_MAP_ADDR
    pmap_map_kernel_special(PMAP_IO_MAP_ADDR,
			    PMAP_IO_MAP_SIZE,
			    PMAP_IO_MAP_PROT);
#endif

#ifdef	PMAP_IOX_MAP_ADDR
    pmap_map_kernel_special(PMAP_IOX_MAP_ADDR,
			    PMAP_IO_MAP_SIZE,
			    PMAP_IO_MAP_PROT);
#endif

#ifdef	PMAP_SLOT_MAP_ADDR
    pmap_map_kernel_special(PMAP_SLOT_MAP_ADDR,
			    PMAP_SLOT_MAP_SIZE,
			    PMAP_SLOT_MAP_PROT);
#endif

#undef	pmap_map_kernel_special

    /*
     *	Allocate some amount of physical memory to use for
     *	user page tables.  The amount allocated is set at
     *  compile time in the kernel data segment, and can
     *  be patched in the kernel a.out file.
     */
    first_pt_pa = *aphys_free;
    *aphys_free += round_page(mac2_ptob(pt_page_alloc_count));
    first_pt_pfn = mac2_btop(first_pt_pa);

    /*
     *	Don't forget to zero it.
     */
    bzero((caddr_t) first_pt_pa, *aphys_free - first_pt_pa);

    /*
     * Load the SRP with the root of the kernel
     * translation tree.
     */
    SRP.limit = PMMU_RP_LIMIT;
    SRP.valid = (PMMU_RP_SG|PMMU_VALID_RP);
    SRP.phys = (unsigned long)kernel_pmap->root;
    pmmu_set_srp(&SRP);

    /*
     * Load the TC register to
     * set the mapping parameters
     * and enable mapping.
     */
    *((unsigned long *)&TC) = 0;
    TC.enable = TC.sre = 1;
    TC.pagesize = PMMU_PAGESIZE;
    TC.addrsize = PMMU_IS_32b;
    TC.tia = PMMU_TIA;
    TC.tib = PMMU_TIB;
    pmmu_set_tc(&TC);
}

/*
 * Initialize the pmap module.
 * Called by vm_init, to initialize any structures that the pmap
 * system needs to map virtual memory.
 */
void
pmap_init(phys_start, phys_end)
    vm_offset_t	phys_start, phys_end;
{
    register long		npages;
    register vm_offset_t	addr;
    register vm_size_t		s;

    /*
     *	Allocate memory for the pv_head_table and its lock bits,
     *	the page attribute byte array, and the pt_page table.
     */
    npages = atop(phys_end - phys_start);
    s = (vm_size_t) (sizeof(struct pv_entry) * npages
    			+ pv_lock_table_size(npages)
			+ npages
			+ pt_page_alloc_count * sizeof(struct pt_page));

    s = round_page(s);
    addr = (vm_offset_t) kmem_alloc(kernel_map, s);

    ptpg_array = (ptpg_t) addr;
    addr = (vm_offset_t)(ptpg_array + pt_page_alloc_count);

    pv_head_table = (pv_entry_t) addr;
    addr = (vm_offset_t) (pv_head_table + npages);

    pv_lock_table = (char *) addr;
    addr = (vm_offset_t) (pv_lock_table + pv_lock_table_size(npages));

    pmap_page_attributes = (char *) addr;

    /*
     *	Create the zone of pmaps, pt root pages and
     *  the physical-to-virtual entries.
     */
    s = (vm_size_t) sizeof (struct pmap);
    pmap_zone = zinit(s, 400*s, MAC2_PGBYTES, FALSE, "pmap"); /* XXX */
    s = (vm_size_t) sizeof (struct pt_root_page);
    ptrt_zone = zinit(s, 50*s, MAC2_PGBYTES, FALSE, "ptrt"); /* XXX */
    s = (vm_size_t) sizeof (struct pv_entry);
    pv_list_zone = zinit(s, 10000*s, MAC2_PGBYTES, FALSE, "pv_list"); /* XXX */

    /*
     * Initialize the pt root page free queue.
     */
    simple_lock_init(&pt_root_page_lock);
    queue_init(&pt_root_page_queue_free);

    /*
     * Initialize the pt page queues.
     */
    simple_lock_init(&pt_page_queue_lock);
    queue_init(&pt_page_queue_free);
    queue_init(&pt_page_queue_active);
    {
	register ptpg_t	    ptpg;
	register int	    i;

	for (i = 0, ptpg = ptpg_array; i < pt_page_alloc_count; i++, ptpg++) {
	    enqueue_tail(&pt_page_queue_free, (queue_entry_t) ptpg);
	}
    }
    pt_page_free_count = pt_page_alloc_count;

    /*
     *	Only now, when all of the data structures are allocated,
     *	can we set vm_first_phys and vm_last_phys.  If we set them
     *	too soon, the kmem_alloc above will try to use these
     *	data structures and blow up.
     */
    vm_first_phys = phys_start;
    vm_last_phys = phys_end;

    pmap_initialized = TRUE;
}

/*
 * The physical address space is dense... there are no holes.
 * All addresses provided to vm_page_startup() are valid.
 */
boolean_t
pmap_valid_page(p)
	vm_offset_t	p;
{
	return (TRUE);
}

/*
 * Allocate a free pt page.
 * The affected pmap must be locked.
 * Returns NULL if there are no free pt pages.
 */
ptpg_t
pt_free_page_alloc(pmap, va)
    register pmap_t	pmap;
    vm_offset_t		va;
{
    register ptpg_t	ptpg;

    simple_lock(&pt_page_queue_lock);

    if (queue_empty(&pt_page_queue_free)) {
	simple_unlock(&pt_page_queue_lock);
	return (PT_PAGE_NULL);
    }

    ptpg = (ptpg_t) dequeue_head(&pt_page_queue_free);
    pt_page_free_count--;

    if (ptpg->pmap != PMAP_NULL || ptpg->va != (vm_offset_t) 0)
	panic("ptpg in use!");

    va &= ~(pt_page_map_size - 1);

    ptpg->pmap = pmap;
    ptpg->va = va;

    enqueue_tail(&pt_page_queue_active, (queue_entry_t) ptpg);
    pt_page_active_count++;

    if (pt_page_active_count > pt_page_max_active)
	pt_page_max_active = pt_page_active_count;

    simple_unlock(&pt_page_queue_lock);

    return (ptpg);
}

/*
 * Allocate a pt page.  May grab one from another pmap if
 * there are no free pages.
 * Must be called with the pmap system locked for write.
 */
ptpg_t
pt_page_alloc(pmap, va)
    pmap_t		pmap;
    vm_offset_t		va;
{
    register ptpg_t	ptpg;

    if (queue_empty(&pt_page_queue_free)) {
	pmap_t		old_pmap;
	vm_offset_t	old_va;
	pp_t		*spp, *epp;

	/*
	 * no free pt pages.  Recycle one.
	 */
	ptpg = (ptpg_t) queue_first(&pt_page_queue_active);
	if (ptpg == PT_PAGE_NULL)
	    panic("not enough pt pages!");

	/*
	 * free the pt page
	 */
	old_pmap = ptpg->pmap;
	old_va = ptpg->va;
	spp = pmap_pp(old_pmap, old_va);
	epp = spp + MAC2_PGBYTES/sizeof(pp_t);
	simple_lock(&old_pmap->lock);
	PMAP_UPDATE_ATC(old_pmap, old_va, old_va + pt_page_map_size);
	pmap_remove_range(old_pmap, old_va, spp, epp, TRUE);
	/*
	 * calls pt_page_free, which clears the pt page
	 * and puts it at the head of the free queue.
	 */
	simple_unlock(&old_pmap->lock);
    }
    return (pt_free_page_alloc(pmap, va));
}
	
/*
 * Free a pt page.
 * The mappings in the pt page must have already been freed.
 * The pmap system needs only to be locked for read.
 */
void
pt_page_free(ptpg)
    register ptpg_t	ptpg;
{
    /*
     *	Clear the pt group fields.
     */
    ptpg->pmap = PMAP_NULL;
    ptpg->va = (vm_offset_t) 0;
    ptpg->use_count = 0;
    ptpg->wired_count = 0;

    /*
     *	Put it on the free list.
     */
    simple_lock(&pt_page_queue_lock);

    remqueue(&pt_page_queue_active, (queue_entry_t) ptpg);
    pt_page_active_count--;

    enqueue_head(&pt_page_queue_free, (queue_entry_t) ptpg);
    pt_page_free_count++;

    simple_unlock(&pt_page_queue_lock);
}

/*
 * Wire down a pt page so it cannot be reused.
 */
void
pt_page_wire(ptpg)
    ptpg_t	ptpg;
{
    simple_lock(&pt_page_queue_lock);

    remqueue(&pt_page_queue_active, (queue_entry_t) ptpg);

    pt_page_active_count--;
    pt_page_wired_count++;

    if (pt_page_wired_count > pt_page_max_wired)
	pt_page_max_wired = pt_page_wired_count;

    simple_unlock(&pt_page_queue_lock);
}

/*
 * Unwire a pt page.
 */
void
pt_page_unwire(ptpg)
    ptpg_t	ptpg;
{
    simple_lock(&pt_page_queue_lock);

    enqueue_tail(&pt_page_queue_active, (queue_entry_t) ptpg);

    pt_page_wired_count--;
    pt_page_active_count++;

    simple_unlock(&pt_page_queue_lock);
}

/*
 * Allocate a pt root and
 * return the pt root page
 * structure pointer and
 * the pt root physical address.
 */
pt_root_alloc(aptrt, aroot)
register ptrt_t *aptrt;
register vm_offset_t *aroot;
{
    register ptrt_t ptrt;
    register ndx;

    simple_lock(&pt_root_page_lock);

    /*
     * Find a pt root page with a free
     * pt root.  If there are none on the
     * free queue, have to allocate another.
     */
    if (queue_empty(&pt_root_page_queue_free)) {
	simple_unlock(&pt_root_page_lock);

	ptrt = (ptrt_t)zalloc(ptrt_zone);
	if (ptrt == PT_ROOT_PAGE_NULL)
	    panic("pt_root_alloc: zalloc failed");

	ptrt->page = kmem_alloc(kernel_map, MAC2_PGBYTES);
	if (ptrt->page == 0)
	    panic("pt_root_alloc: kmem_alloc failed");
/* XXX */
	ptrt->free_count = 8;
/* XXX */
	init_bitmap_byte(&ptrt->free_map);

	simple_lock(&pt_root_page_lock);

	pt_root_page_alloc_count++;

	enqueue_head(&pt_root_page_queue_free, (queue_entry_t)ptrt);
	pt_root_page_free_count++;
    }
    else
	ptrt = (ptrt_t)queue_first(&pt_root_page_queue_free);

    /*
     * Allocate a pt root from this
     * pt root page.  If we are taking
     * the last one, remove from free
     * queue.
     */
    if (--ptrt->free_count == 0) {
	remqueue(&pt_root_page_queue_free, (queue_entry_t)ptrt);
	pt_root_page_free_count--;
    }
    ndx = alloc_bitmap_byte(&ptrt->free_map);
    if (ndx < 0)
	panic("pmap_ptptbl_alloc: bitmap full");

    simple_unlock(&pt_root_page_lock);

    /*
     * Return a pointer to the
     * pt root page structure and the
     * physical pt root address.
     */
    *aptrt = ptrt;
    *aroot = (vm_offset_t)pmap_extract(kernel_pmap,
				      get_ptptbl_addr(ptrt->page, ndx));

    /*
     * Clear the pt root.
     */
    bzero(*aroot, pmap_ptptbl_size);
}    

/*
 * Free a pt root.
 */
pt_root_free(ptrt, root)
register ptrt_t ptrt;
register vm_offset_t root;
{
    register ndx;

    /*
     * Calculate index from
     * pt root address.
     */
    ndx = get_pt_root_page_index(root);

    simple_lock(&pt_root_page_lock);

    /*
     * Free pt root.  If we are deallocating
     * the last one in the pt root page and
     * there are other pt root pages with free
     * pt roots, free this one.
     */
    if (++ptrt->free_count == 8 && !queue_empty(&pt_root_page_queue_free)) {
	remqueue(&pt_root_page_queue_free, (queue_entry_t)ptrt);
	pt_root_page_free_count--;

	kmem_free(kernel_map, ptrt->page, MAC2_PGBYTES);
	zfree(ptrt_zone, ptrt);
	pt_root_page_alloc_count--;
    }
    else {
	if (ptrt->free_count == 1) {
	    enqueue_head(&pt_root_page_queue_free, (queue_entry_t)ptrt);
	    pt_root_page_free_count++;
	}
	if (free_bitmap_byte(&ptrt->free_map, ndx) < 0)
	    panic("pmap_ptptbl_free: bitmap already free");
    }

    simple_unlock(&pt_root_page_lock);
}

/*
 * Setup an alternate root page
 * that simulates 24 bit addressing.
 */
pmap_do_24bit_map(pmap)
register pmap_t	pmap;
{
    register ptpg_t	ptpg;
    register ptp_t	*ptp;
    register	ipl;

    PMAP_READ_LOCK(pmap, ipl);

    ptp = pmap->root;

    /*
     * Allocate a page table if
     * none exists.
     */
    if (ptp->valid == PMMU_INVALID) {
	ptpg = pt_free_page_alloc(pmap, (vm_offset_t)0);
	if (ptpg == PT_PAGE_NULL) {
	    PMAP_READ_UNLOCK(pmap, ipl);
	    PMAP_WRITE_LOCK(ipl);
	    ptpg = pt_page_alloc(pmap, (vm_offset_t)0);
	    PMAP_WRITE_TO_READ_LOCK(pmap);
	}
	ptp->pfn = pt_page_to_pfn(ptpg);
	ptp->valid = PMMU_VALID_PTP;
    }
    else
	ptpg = pfn_to_pt_page(ptp->pfn);

    ptpg->use_count++;

    /*
     * Wire down the page table
     * since it will be referenced
     * from two root pages, one
     * of which is not known about
     * by the rest of the code.
     */
    if (ptpg->wired_count++ == 0)
	pt_page_wire(ptpg);

    /*
     * Setup the mappings.
     */
    {
	register ptp_t	*eptp;
	ptp_t	template;

	template = *ptp;
	ptp = pmap->alt_root;
	eptp = ptp+(pmap_ptptbl_size/sizeof (ptp_t));
	for (; ptp < eptp;)
	    *ptp++ = template;
    }

    PMAP_READ_UNLOCK(pmap, ipl);
}

/*
 * Remove references on behalf
 * of alternate root.
 */
pmap_undo_24bit_map(pmap)
register pmap_t	pmap;
{
    register ptpg_t	ptpg;
    register ptp_t	*ptp;
    register	ipl;

    PMAP_READ_LOCK(pmap, ipl);

    ptp = pmap->alt_root;
    ptpg = pfn_to_pt_page(ptp->pfn);

    if (--ptpg->wired_count == 0)
	pt_page_unwire(ptpg);

    if (--ptpg->use_count == 0)
	pt_page_free(ptpg);

    PMAP_READ_UNLOCK(pmap, ipl);    
}

/*
 * Create and return a physical map.
 *
 * If the size specified for the map
 * is zero, the map is an actual physical
 * map, and may be referenced by the
 * hardware.
 *
 * If the size specified is non-zero,
 * the map will be used in software only, and
 * is bounded by that size.
 */
pmap_t
pmap_create(size)
    vm_size_t		size;
{
    register pmap_t	pmap;

    /*
     *	A software use-only map doesn't even need a map.
     */
    if (size != 0)
	return(PMAP_NULL);

    pmap = (pmap_t) zalloc(pmap_zone);
    if (pmap == PMAP_NULL)
	panic("pmap_create: cannot allocate a pmap");

    /*
     * Initialize some variables
     */
    simple_lock_init(&pmap->lock);
    pmap->ref_count = 1;
    pmap->flags = 0;
    pmap->cpus_using = 0;
    pmap->alt_root = 0;

    /*
     *	Initialize statistics.
     */
    pmap->stats.resident_count = 0;
    pmap->stats.wired_count = 0;

    /*
     * Allocate a pt root (ptptbl)
     * for the pmap.
     */
    pt_root_alloc(&pmap->ptrt, &pmap->root);

    return(pmap);
}

/*
 * Retire the given physical map from service.
 * Should only be called if the map contains
 * no valid mappings.
 */
void
pmap_destroy(pmap)
    register pmap_t	pmap;
{
    register int	c, ipl;

    if (pmap == PMAP_NULL)
	return;

    SPLVM(ipl);

    simple_lock(&pmap->lock);
    c = --pmap->ref_count;
    simple_unlock(&pmap->lock);

    SPLX(ipl);

    if (c != 0)
	return;	/* pmap still in use */

    if (pmap_resident_count(pmap))
	printf("pmap_destroy: pmap %x %d mappings still exist\n", pmap, pmap_resident_count(pmap));

    if (pmap->alt_root) {
	ATC_INVALIDATE_MAP(pmap->alt_root);
	pmap_undo_24bit_map(pmap);
	pt_root_free(pmap->alt_ptrt, pmap->alt_root);
    }

    ATC_INVALIDATE_MAP(pmap->root);
    pt_root_free(pmap->ptrt, pmap->root);

    zfree(pmap_zone, (vm_offset_t)pmap);
}

/*
 * Add a reference to the specified pmap.
 */
void
pmap_reference(pmap)
    register pmap_t	pmap;
{
    register		ipl;

    if (pmap != PMAP_NULL) {
	SPLVM(ipl);

	simple_lock(&pmap->lock);
	pmap->ref_count++;
	simple_unlock(&pmap->lock);

	SPLX(ipl);
    }
}

/*
 * Remove a range of page-pointers.
 * The entries given are the first (inclusive)
 * and last (exclusive) pointers for the VM pages.
 * start is the va for the first page.
 *
 * The pmap must be locked.
 * The range must lie entirely within one pt page.
 * This is NOT checked.
 * Assumes that the pt page exists.
 *
 */
pmap_remove_range(pmap, start, spp, epp, free_if_empty)
    pmap_t			pmap;
    vm_offset_t			start;
    pp_t			*spp;
    pp_t			*epp;
    boolean_t			free_if_empty;
{
    register pp_t		*pp;
    ptp_t			*ptp;
    int				pfn;
    int				num_removed, num_unwired;
    register int		pai;
    register vm_offset_t	va;
    vm_offset_t			pa;

    num_removed = 0;
    num_unwired = 0;

    va = start;
    for (pp = spp; pp < epp; pp += pps_per_vm_page, va += PAGE_SIZE) {

	if (pp->valid == PMMU_INVALID)
	    continue;

	pfn = pp->pfn;

	num_removed++;
	if (pp->wired)
	    num_unwired++;

	pa = mac2_ptob(pfn);
	if (pa < vm_first_phys || pa >= vm_last_phys) {
	    /*
	     *	Outside range of managed physical memory.
	     */
	    continue;
	}

	pai = pa_index(pa);
	LOCK_PVH(pai);

	/*
	 * Get the referenced & modified bits.
	 */
	{
	    register int	i;
	    register pp_t	*p;

	    p = pp;
	    for (i = pps_per_vm_page; i-- > 0; p++) {
		if (p->mod)
		    pmap_page_attributes[pai] |= PG_MOD;
		if (p->ref)
		    pmap_page_attributes[pai] |= PG_REF;
	    }
	}

	/*
	 * Remove the mapping from the pvlist for
	 * this physical page.
	 */
	{
	    register pv_entry_t	pv_h, prev, cur;

	    pv_h = pai_to_pvh(pai);
	    if (pv_h->pmap == PMAP_NULL)
		panic("pmap_remove: null pv_list!");

	    if (pv_h->va == va && pv_h->pmap == pmap) {
		/*
		 * Header is the pv_entry.  Copy the next one
		 * to header and free the next one (we cannot
		 * free the header)
		 */
		cur = pv_h->next;
		if (cur != PV_ENTRY_NULL) {
		    *pv_h = *cur;
		    zfree(pv_list_zone, (vm_offset_t) cur);
		} else {
		    pv_h->pmap = PMAP_NULL;
		}
	    } else {
		cur = pv_h;
		do {
		    prev = cur;
		    if ((cur = prev->next) == PV_ENTRY_NULL)
			panic("pmap-remove: mapping not in pv_list!");
		} while (cur->va != va || cur->pmap != pmap);
		prev->next = cur->next;
		zfree(pv_list_zone, (vm_offset_t) cur);
	    }
	}
	UNLOCK_PVH(pai);
    }

    /*
     *	Zero the PPs to remove the mappings.
     */
    bzero((caddr_t) spp, (epp-spp)*sizeof(pp_t));

    /*
     *	Update the counts
     */
    pmap->stats.resident_count -= num_removed;
    pmap->stats.wired_count -= num_unwired;

    if (pmap != kernel_pmap) {
	register ptpg_t	    ptpg;

	/*
	 * See whether this pt page can be freed.
	 */
	ptpg = get_pt_page(spp);
	ptpg->use_count -= num_removed;
	ptpg->wired_count -= num_unwired;
	if (num_unwired > 0 && ptpg->wired_count == 0)
	    pt_page_unwire(ptpg);
	if (ptpg->use_count == 0 && free_if_empty) {
	    ptp = ptptbl_to_ptp(pmap->root, start);
	    ptp->valid = PMMU_INVALID;
	    pt_page_free(ptpg);
	}
    }
}

/*
 * Remove the given range of addresses
 * from the specified pmap.
 *
 * It is assumed that the start and end are properly
 * rounded to the page size.
 */
void
pmap_remove(pmap, start, end)
    pmap_t		pmap;
    vm_offset_t		start, end;
{
    register ptp_t	*sptp, *eptp;
    register pp_t	*spp, *epp;
    int			ipl;

    if (pmap == PMAP_NULL || pmap->root == PTP_NULL)
	return;

    PMAP_READ_LOCK(pmap, ipl);

    /*
     *	Invalidate the translation buffer first
     */
    PMAP_UPDATE_ATC(pmap, start, end);

    /*
     * Flush the CPU caches if warranted
     */
    PMAP_FLUSH_CPU_CACHES(pmap);

    /*
     * calculate the start and end
     * page-table-pointers
     */
    sptp = ptptbl_to_ptp(pmap->root, start);
    eptp = ptptbl_to_ptp(pmap->root, end);
    if (sptp == eptp) {
	/*
	 * range is in one
	 * page-pointer table
	 */
	if (sptp->valid == PMMU_VALID_PTP) {
	    spp = ptp_to_pp(sptp, start);
	    epp = ptp_to_pp(sptp, end);
	    if (spp < epp)
		pmap_remove_range(pmap, start, spp, epp, TRUE);
	}
    } else {
	/*
	 * range crosses page-pointer table
	 * boundary
	 */
	if (sptp->valid == PMMU_VALID_PTP) {
	    /*
	     * do first (maybe partial) pptbl
	     */
	    spp = ptp_to_pp(sptp, start);
	    epp = (pp_t *)mac2_round_page(spp+1);
	    pmap_remove_range(pmap, start, spp, epp, TRUE);
	}
	for (sptp++; sptp < eptp; sptp++) {
	    /*
	     * do all complete pptbls
	     */
	    if (sptp->valid == PMMU_INVALID)
		continue;
	    start = ptp_to_va(sptp);
	    spp = ptp_to_pptbl(sptp);
	    epp = (pp_t *)mac2_round_page(spp+1);
	    pmap_remove_range(pmap, start, spp, epp, TRUE);
	}
	if (sptp->valid == PMMU_VALID_PTP) {
	    /*
	    * do last (maybe partial) pptbl
	    */
	    start = ptp_to_va(sptp);
	    spp = ptp_to_pptbl(sptp);
	    epp = ptp_to_pp(sptp, end);
	    if (spp < epp)
		pmap_remove_range(pmap, start, spp, epp, TRUE);
	}
    }

    PMAP_READ_UNLOCK(pmap, ipl);
}

/*
 * Removes this physical page from all pmaps
 * in which it resides.  Reflects back modify
 * bits to the pager.
 */
void
pmap_remove_all(pa)
    vm_offset_t			pa;
{
    register pv_entry_t		pv_h;
    register pp_t		*pp;
    int				pai;
    register vm_offset_t	va;
    register pmap_t		pmap;
    int				ipl;

    if (pa < vm_first_phys || pa >= vm_last_phys) {
	/*
	 *	Not a managed page.
	 */
	return;
    }

    /*
     *	Lock the pmap system first, since we will be changing
     *	several pmaps.
     */
    PMAP_WRITE_LOCK(ipl);

    /*
     *	Walk down PV list, removing all mappings.
     *	We don't have to lock the pv_head, since
     *  we have the entire pmap system.
     */
    pai = pa_index(pa);
    pv_h = pai_to_pvh(pai);

    while ((pmap = pv_h->pmap) != PMAP_NULL) {
	va = pv_h->va;

	/*
	 *	Lock the pmap to block pmap_extract and similar routines.
	 */
	simple_lock(&pmap->lock);

	pp = pmap_pp(pmap, va);

	if (pp == PP_NULL)
	    panic("pmap_remove_all: no page-ptr tbl");

	if (mac2_ptob(pp->pfn) != pa)
	    panic("pmap_remove_all: page_ptr doesn't point to page!");

	pmap->stats.resident_count--;
	if (pp->wired)
	    panic("pmap_remove_all removing a wired page");

	/*
	 *	Tell CPU using pmap to invalidate its TLB
	 */
	PMAP_UPDATE_ATC(pmap, va, va + PAGE_SIZE);

	/*
	 * Flush cpu caches if warranted
	 */
	PMAP_FLUSH_CPU_CACHES(pmap);

	/*
	 *	Remove the pv_entry from the pv_list.
	 */
	{
	    register pv_entry_t	cur;

	    if ((cur = pv_h->next) != PV_ENTRY_NULL) {
		*pv_h = *cur;
		zfree(pv_list_zone, (vm_offset_t) cur);
	    } else
		pv_h->pmap = PMAP_NULL;
	}

	/*
	 *	Remove the mapping, collecting any referenced & modified bits.
	 */
	{
	    register pp_t   *p;
	    register int    i;

	    p = pp;
	    for (i = pps_per_vm_page; i-- > 0; p++) {
		if (p->mod)
		    pmap_page_attributes[pai] |= PG_MOD;
		if (p->ref)
		    pmap_page_attributes[pai] |= PG_REF;
		*(unsigned long *)p = 0;
	    }
	}

	/*
	 *	Free the pt page if it is no longer needed.
	 */
	if (pmap != kernel_pmap) {
	    register ptpg_t	ptpg;
	    register ptp_t	*ptp;

	    ptpg = get_pt_page(pp);
	    if (--ptpg->use_count == 0) {
		pt_page_free(ptpg);
		/*
		 * Invalidate the ptp.
		 */
		ptp = ptptbl_to_ptp(pmap->root, va);
		ptp->valid = PMMU_INVALID;
	    }
		    
	}
	simple_unlock(&pmap->lock);
    }

    PMAP_WRITE_UNLOCK(ipl);
}

boolean_t
pmap_verify_free(pa)
    vm_offset_t	pa;
{
    pv_entry_t	pv_h;
    int		pai;
    int		ipl;
    boolean_t	result;

    if (!pmap_initialized)
	return(TRUE);

    if (pa < vm_first_phys || pa >= vm_last_phys)
	return(FALSE);

    PMAP_WRITE_LOCK(ipl);

    pai = pa_index(pa);
    pv_h = pai_to_pvh(pai);

    result = (pv_h->pmap == PMAP_NULL);
    PMAP_WRITE_UNLOCK(ipl);
    
    return (result);
}

/*
 * Remove write privileges from all
 * pmaps for this physical page.
 */
void
pmap_copy_on_write(pa)
    vm_offset_t		pa;
{
    register pv_entry_t	pv_e;
    register pp_t	*pp;
    register int	i;
    int			ipl;

    /*
     *	Lock the entire pmap system, since we may be changing
     *	several maps.
     */
    PMAP_WRITE_LOCK(ipl);

    pv_e = pai_to_pvh(pa_index(pa));
    if (pv_e->pmap == PMAP_NULL) {
	PMAP_WRITE_UNLOCK(ipl);
	return;		/* no mappings */
    }
    /*
     *	Run down the list of mappings to this physical page,
     *	disabling write privileges on each one.
     */
    while (pv_e != PV_ENTRY_NULL) {
	pmap_t		pmap;
	vm_offset_t	va;

	pmap = pv_e->pmap;
	va = pv_e->va;

	simple_lock(&pmap->lock);

	pp = pmap_pp(pmap, va);

	if (pp == PP_NULL)
	    panic("pmap_copy_on_write: no page-ptr tbl");

	/*
	 * Ask cpus using pmap to invalidate their ATC
	 */
	PMAP_UPDATE_ATC(pmap, va, va + PAGE_SIZE);

	for (i = pps_per_vm_page; i-- > 0; pp++)
	    pp->prot = PMMU_WP;

	simple_unlock(&pmap->lock);

	pv_e = pv_e->next;
    }

    PMAP_WRITE_UNLOCK(ipl);
}

/*
 * Set the physical protection on the
 * specified range of this pmap as requested.
 */
void
pmap_protect(pmap, start, end, prot)
    pmap_t		pmap;
    vm_offset_t		start, end;
    vm_prot_t		prot;
{
    register ptp_t	*sptp, *eptp;
    register pp_t	*spp, *epp;
    register int	mprot;
    int		ipl;

    if (pmap == PMAP_NULL || pmap->root == PTP_NULL)
	return;

    /*
     * Convert the requested MACH
     * protection value into one
     * that we can implement.
     */
    prot = pmmu_prot_map[prot];

    /*
     * VM_PROT_NONE means remove
     * mappings.
     */
    if (prot == VM_PROT_NONE) {
	pmap_remove(pmap, start, end);
	return;
    }

    mprot = pmmu_prot(prot);

    /*
     * Due to the XP changes,
     * this routine can now only
     * reduce permissions.
     */
    if (mprot == PMMU_RW)
	return;

    SPLVM(ipl);
    simple_lock(&pmap->lock);

    /*
     *	Invalidate the translation buffer first
     */
    PMAP_UPDATE_ATC(pmap, start, end);

    /*
     * calculate the start and end
     * page-table-pointers
     */
    sptp = ptptbl_to_ptp(pmap->root, start);
    eptp = ptptbl_to_ptp(pmap->root, end);
    if (sptp == eptp) {
	/*
	 * range is in one
	 * page-pointer table
	 */
	if (sptp->valid == PMMU_VALID_PTP) {
	    spp = ptp_to_pp(sptp, start);
	    epp = ptp_to_pp(sptp, end);
	    for (; spp < epp; spp++)
		spp->prot = mprot;
	}
    } else {
	/*
	 * range crosses page-pointer table
	 * boundary
	 */
	if (sptp->valid == PMMU_VALID_PTP) {
	    /*
	     * do first (maybe partial) pptbl
	     */
	    spp = ptp_to_pp(sptp, start);
	    epp = (pp_t *)mac2_round_page(spp+1);
	    for (; spp < epp; spp++)
		spp->prot = mprot;
	}
	for (sptp++; sptp < eptp; sptp++) {
	    /*
	     * do all complete pptbls
	     */
	    if (sptp->valid == PMMU_INVALID)
		continue;
	    spp = ptp_to_pptbl(sptp);
	    epp = (pp_t *)mac2_round_page(spp+1);
	    for (; spp < epp; spp++)
		spp->prot = mprot;
	}
	if (sptp->valid == PMMU_VALID_PTP) {
	    /*
	     * do last (maybe partial) pptbl
	     */
	    spp = ptp_to_pptbl(sptp);
	    epp = ptp_to_pp(sptp, end);
	    for (; spp < epp; spp++)
		spp->prot = mprot;
	}
    }

    simple_unlock(&pmap->lock);
    SPLX(ipl);
}

/*
 * Insert the given physical page pa at
 * the specified virtual address va in the
 * target pmap with the protection requested.
 *
 * If specified, the page will be wired down, meaning
 * that the related page_ptr can not be reclaimed.
 *
 * NB:  This is the only routine which MAY NOT lazy-evaluate
 * or lose information.  That is, this routine must actually
 * insert this page into the given pmap NOW.
 */
void
pmap_enter(pmap, va, pa, prot, wired)
    register pmap_t		pmap;
    vm_offset_t			va;
    register vm_offset_t	pa;
    vm_prot_t			prot;
    boolean_t			wired;
{
    register pp_t		*pp;
    register ptp_t		*ptp;
    pv_entry_t			pv_e;
    int				ipl, mprot;
    ptpg_t			ptpg;
    vm_offset_t			old_pa;

    if (pmap == PMAP_NULL)
	return;

    /*
     * Convert the requested MACH
     * protection value into one
     * that we can implement.
     */
    prot = pmmu_prot_map[prot];

    /*
     * VM_PROT_NONE means remove
     * the mapping.
     */
    if (prot == VM_PROT_NONE) {
	pmap_remove(pmap, va, va + PAGE_SIZE);
	return;
    }

    mprot = pmmu_prot(prot);

    /*
     *	Must allocate a new pvlist entry while we're unlocked;
     *	zalloc may cause pageout (which will lock the pmap system).
     *	If we determine we need a pvlist entry, we will unlock
     *	and allocate one.  Then we will retry, throwing away
     *	the allocated entry later (if we no longer need it).
     */
    pv_e = PV_ENTRY_NULL;

Retry:
    PMAP_READ_LOCK(pmap, ipl);

    /*
     * check to see if this pmap
     * has a ptptbl.
     */
    if (pmap->root == PTP_NULL)
	panic("pmap_enter: no page-table-pointer table");

    /*
     * check to see if the
     * page-pointer table exists.
     * if not, have to get one.
     */
    pp = pmap_pp(pmap, va);
    if (pp == PP_NULL) {
	if (pmap == kernel_pmap)
	    panic("pmap_enter: kernel pmap too small");

	ptpg = pt_free_page_alloc(pmap, va);
	if (ptpg == PT_PAGE_NULL) {
	    /*
	     *	No free pt pages.  Must get the write lock
	     *	on the pmap system to grab one from another pmap.
	     */
	    PMAP_READ_UNLOCK(pmap, ipl);
	    PMAP_WRITE_LOCK(ipl);
	    ptpg = pt_page_alloc(pmap, va);
	    PMAP_WRITE_TO_READ_LOCK(pmap);
	}
	ptp = ptptbl_to_ptp(pmap->root, va);
	ptp->pfn = pt_page_to_pfn(ptpg);
	ptp->valid = PMMU_VALID_PTP;
	pp = ptp_to_pp(ptp, va);
    } else if (pmap == kernel_pmap)
	ptpg = PT_PAGE_NULL;
    else
	ptpg = get_pt_page(pp);

    /*
     *	Special case if the physical page is already mapped
     *	at this address.
     */
    old_pa = mac2_ptob(pp->pfn);
    if ((pp->valid == PMMU_VALID_PP) && (old_pa == pa)) {
	/*
	 *	May be changing its wired attribute or protection
	 */
	if (wired && !pp->wired) {
	    pmap->stats.wired_count++;
	    if (ptpg && ptpg->wired_count++ == 0)
		pt_page_wire(ptpg);
	} else if (!wired && pp->wired) {
	    pmap->stats.wired_count--;
	    if (ptpg && --ptpg->wired_count == 0)
		pt_page_unwire(ptpg);
	}

	/*
	 * Set protection and wire as indicated.
	 */
	{
	    register int i;
	    register pp_t *p = pp;
	    pp_t template;

	    PMAP_UPDATE_ATC(pmap, va, va + PAGE_SIZE);
	    for (i = pps_per_vm_page; i > 0; i--) {
		template = *p;
		template.prot = mprot;
		if (wired)
		    template.wired = 1;
		else
		    template.wired = 0;
		*p++ = template;
	    }
	}
    } else {
	/*
	 *	Remove old mapping from the PV list if necessary.
	 */
	if (pp->valid == PMMU_VALID_PP) {
	    /*
	     *	Invalidate the translation buffer,
	     *	then remove the mapping.
	     */
	    PMAP_UPDATE_ATC(pmap, va, va + PAGE_SIZE);

	    /*
	     * Flush the CPU caches if warranted
	     */
	    PMAP_FLUSH_CPU_CACHES(pmap);

	    /*
	     *	Don't free the pt page if removing last
	     *	mapping - we will immediately replace it.
	     */
	    pmap_remove_range(pmap, va, pp, pp + pps_per_vm_page, FALSE);
	}

	if (pa >= vm_first_phys && pa < vm_last_phys) {
	    /*
	     *	Enter the mapping in the PV list for this
	     *	physical page.
	     */
	    register int		pai;
	    register pv_entry_t		pv_h;

	    pai = pa_index(pa);
	    LOCK_PVH(pai);
	    pv_h = pai_to_pvh(pai);

	    if (pv_h->pmap == PMAP_NULL) {
		/*
		 * No mappings yet
		 */
		pv_h->va = va;
		pv_h->pmap = pmap;
		pv_h->next = PV_ENTRY_NULL;
	    } else {
		/*
		 *	Add new pv_entry after header.
		 */
		if (pv_e == PV_ENTRY_NULL) {
		    UNLOCK_PVH(pai);
		    PMAP_READ_UNLOCK(pmap, ipl);
		    /*
		     * This isn't correct either: if zalloc blocks in
		     * kmem_alloc, we return with our CPU in the active
		     * set but ipl raised.  We then can deadlock
		     * grabbing the zone lock.
		     */
		    SPLVM(ipl);
		    pv_e = (pv_entry_t) zalloc(pv_list_zone);
		    SPLX(ipl);
		    goto Retry;
		}
		pv_e->va = va;
		pv_e->pmap = pmap;
		pv_e->next = pv_h->next;
		pv_h->next = pv_e;
		/*
		 *	Remember that we used the pvlist entry.
		 */
		pv_e = PV_ENTRY_NULL;
	    }
	    UNLOCK_PVH(pai);
	}

	/*
	 * Count the mapping.
	 */
	pmap->stats.resident_count++;
	if (ptpg)
	    ptpg->use_count++;
	if (wired) {
	    pmap->stats.wired_count++;
	    if (ptpg && ptpg->wired_count++ == 0)
		pt_page_wire(ptpg);
	}

	/*
	 * Create the entry.
	 */
	{
	    register int i;
	    register pp_t *p = pp;
	    pp_t template;

	    *((unsigned long *)&template) = 0;
	    template.pfn = mac2_btop(pa);
	    if (wired)
		template.wired = 1;
	    if (pa >= PMAP_IO_MAP_ADDR)
		template.ci = 1;
	    template.prot = mprot;
	    template.valid = PMMU_VALID_PP;
	    for (i = pps_per_vm_page; i > 0; i--) {
		*p++ = template;
		template.pfn++;
	    }
	    PMAP_UPDATE_ATC(pmap, va, va + PAGE_SIZE);
	}
    }

    PMAP_READ_UNLOCK(pmap, ipl);

    if (pv_e != PV_ENTRY_NULL) {
	SPLVM(ipl);
	zfree(pv_list_zone, (vm_offset_t) pv_e);
	SPLX(ipl);
    }
}

/*
 * Change the wiring attribute for a
 * pmap/virtual-address	pair.
 *
 * The mapping must already exist in the pmap.
 */
void
pmap_change_wiring(pmap, va, wired)
    register pmap_t	pmap;
    vm_offset_t		va;
    boolean_t		wired;
{
    register pp_t	*pp;
    register int	i;
    int			ipl;
    ptpg_t		ptpg;

    /*
     *	We must grab the pmap system lock because we may
     *	change a pt_page queue.
     */
    PMAP_READ_LOCK(pmap, ipl);

    if ((pp = pmap_pp(pmap, va)) == PP_NULL)
	panic("pmap_change_wiring: no page-ptr tbl");

    if (pmap == kernel_pmap)
	ptpg = PT_PAGE_NULL;
    else
	ptpg = get_pt_page(pp);

    if (wired && !pp->wired) {
	/*
	 * wiring down mapping
	 */
	pmap->stats.wired_count++;
	if (ptpg && ptpg->wired_count++ == 0)
	    pt_page_wire(ptpg);
    }
    else
    if (!wired && pp->wired) {
	/*
	 * unwiring mapping
	 */
	pmap->stats.wired_count--;
	if (ptpg && --ptpg->wired_count == 0)
	    pt_page_unwire(ptpg);
    }

    for (i = pps_per_vm_page; i > 0; i--)
	(pp++)->wired = wired;

    PMAP_READ_UNLOCK(pmap, ipl);
}

/*
 * 	Extract the physical page address associated
 * 	with the given map/virtual_address pair.
 */
vm_offset_t
pmap_extract(pmap, va)
    register pmap_t	pmap;
    vm_offset_t		va;
{
    register pp_t	    *pp;
    register vm_offset_t    pa;
    int			    ipl;

    SPLVM(ipl);

    simple_lock(&pmap->lock);

    if ((pp = pmap_pp(pmap, va)) == PP_NULL)
	pa = (vm_offset_t)0;
    else {
	pa = mac2_ptob(pp->pfn);
	pa |= (va&(MAC2_PGBYTES-1));	/* offset within page */
    }

    simple_unlock(&pmap->lock);

    SPLX(ipl);

    return (pa);
}

/*
 * Copy the range specified by src_addr/len
 * from the source map to the range dst_addr/len
 * in the destination map.
 *
 * This routine is only advisory and need not do anything.
 */
void
pmap_copy(dst_pmap, src_pmap, dst_addr, size, src_addr)
    pmap_t		dst_pmap;
    pmap_t		src_pmap;
    vm_offset_t		dst_addr;
    vm_size_t		size;
    vm_offset_t		src_addr;
{
}

/*
 * Require that all active physical maps contain no
 * incorrect entries NOW.  [This update includes
 * forcing updates of any address map caching.]
 *
 * Generally used to insure that a thread about
 * to run will see a semantically correct world.
 */
void
pmap_update()
{
}

/*
 * Garbage collects the physical map system for
 * pages which are no longer used.
 * Success need not be guaranteed -- that is, there
 * may well be pages which are not referenced, but
 * others may be collected.
 *
 * Called by the pageout daemon when pages are scarce.
 */
void
pmap_collect(pmap)
    pmap_t		pmap;
{
}

/*
 * Bind the given pmap to the given
 * processor.
 */
void
pmap_activate(pmap, th, cpu)
    register pmap_t	pmap;
    thread_t		th;
    int			cpu;
{
    PMAP_ACTIVATE(pmap, th, cpu);
}


/*
 * Indicates that the given physical map is no longer
 * in use on the specified processor.
 */
void
pmap_deactivate(pmap, th, cpu)
    pmap_t		pmap;
    thread_t		th;
    int			cpu;
{
    PMAP_DEACTIVATE(pmap, th, cpu);
}

/*
 * Return the pmap handle for the kernel.
 */
pmap_t
pmap_kernel()
{
    return (kernel_pmap);
}

/*
 * pmap_zero_page zeros the specified (machine independent)
 * page.  This routine is written entirely in assembly language,
 * see mac2/phys.s.
 */
#if	0
pmap_zero_page(pa)
    register vm_offset_t	pa;
{
}
#endif	0

/*
 * pmap_copy_page copies the specified (machine independent)
 * pages.  This routine is written entirely in assembly language,
 * see mac2/phys.s.
 */
#if	0
pmap_copy_page(src, dst)
    vm_offset_t		    src, dst;
{
}
#endif	0

/*
 * Make the specified pages (by pmap, offset)
 * pageable (or not) as requested.
 *
 * A page which is not pageable may not take
 * a fault; therefore, its page table entry
 * must remain valid for the duration.
 *
 * This routine is merely advisory; pmap_enter
 * will specify that these pages are to be wired
 * down (or not) as appropriate.
 */
pmap_pageable(pmap, start, end, pageable)
    pmap_t	pmap;
    vm_offset_t	start;
    vm_offset_t	end;
    boolean_t	pageable;
{
}

/*
 * Give the kernel read-only access to the specified address.
 * This is used to detect stack overflows.  It is assumed that the
 * address specified is the last possible kernel stack address.
 * Therefore, we round up to the nearest machine dependent page.
 */
void pmap_redzone(pmap, va)
   pmap_t		pmap;
    vm_offset_t		va;
{
    pp_t		*pp;
    int			ipl;

    va = mac2_round_page(va);

    SPLVM(ipl);

    simple_lock(&pmap->lock);

    if ((pp = pmap_pp(pmap, va)) != PP_NULL)
	pp->prot = PMMU_WP;
	
    PMAP_UPDATE_ATC(pmap, va, va+PAGE_SIZE);

    simple_unlock(&pmap->lock);

    SPLX(ipl);
}

/*
 * Clear the modify bits on the specified physical page.
 */
void
pmap_clear_modify(pa)
    register vm_offset_t	pa;
{
    if (pa >= vm_first_phys && pa < vm_last_phys)
	pmap_clear_page_attrib(pa, PG_MOD);
}

/*
 * Return whether or not the specified physical page is modified
 * by any pmaps.
 */
boolean_t
pmap_is_modified(pa)
    register vm_offset_t	pa;
{
    if (pa >= vm_first_phys && pa < vm_last_phys)
	return (pmap_check_page_attrib(pa, PG_MOD));
    else
	return (FALSE);
}

/*
 * Clear the reference bit on the specified physical page.
 */
void
pmap_clear_reference(pa)
    vm_offset_t			pa;
{
    if (pa >= vm_first_phys && pa < vm_last_phys)
	pmap_clear_page_attrib(pa, PG_REF);
}

/*
 * Return whether or not the specified physical page is referenced
 * by any physical maps.
 */
boolean_t
pmap_is_referenced(pa)
    vm_offset_t			pa;
{
    if (pa >= vm_first_phys && pa < vm_last_phys)
	return (pmap_check_page_attrib(pa, PG_REF));
    else
	return (FALSE);
}

/*
 * Clear the specified page attributes both in the
 * pmap_page_attributes table and the address translation
 * tables.  Note that we DO have to flush the entries from
 * the ATC since the PMMU uses the bits in the ATC to
 * determine whether it has to write the bits out to memory.
 */
pmap_clear_page_attrib(pa, attrib)
    register vm_offset_t	pa;
    register int		attrib;
{
    pv_entry_t			pv_h;
    pmap_t			pmap;
    vm_offset_t			va;
    register int		pai, i;
    register pp_t		*pp;
    int				ipl;

    pai = pa_index(pa);
    pmap_page_attributes[pai] &= ~attrib;

    pv_h = pai_to_pvh(pai);
    PMAP_WRITE_LOCK(ipl);
    while ((pmap = pv_h->pmap) != PMAP_NULL) {
	va = pv_h->va;

	simple_lock(&pmap->lock);

	PMAP_UPDATE_ATC(pmap, va, va+PAGE_SIZE);

	pp = pmap_pp(pmap, va);
	if (pp != PP_NULL) {
	    for (i = pps_per_vm_page; i-- > 0; pp++) {
		if (attrib & PG_MOD)
		    pp->mod = 0;
		if (attrib & PG_REF)
		    pp->ref = 0;
	    }
	}

	simple_unlock(&pmap->lock);
	if ((pv_h = pv_h->next) == PV_ENTRY_NULL)
	    break;
    }
    PMAP_WRITE_UNLOCK(ipl);
}

/*
 * Check for the specified attributes for the
 * physical page.  if all bits are true in
 * the pmap_page_attributes table, we can trust
 * it.  otherwise, we must check the address
 * translation tables ourselves.  Note that we
 * DO NOT have to flush the entry from the ATC
 * before looking at the address translation
 * table since the ATC is write-through for the bits.
 */
boolean_t
pmap_check_page_attrib(pa, attrib)
    register vm_offset_t	pa;
    register int		attrib;
{
    pv_entry_t			pv_h;
    pmap_t			pmap;
    vm_offset_t			va;
    register int		pai, i;
    register pp_t		*pp;
    int				ipl;

    pai = pa_index(pa);
    if ((pmap_page_attributes[pai] & attrib) == attrib)
	return (TRUE);

    pv_h = pai_to_pvh(pai);
    PMAP_WRITE_LOCK(ipl);
    while ((pmap = pv_h->pmap) != PMAP_NULL) {
	va = pv_h->va;

	simple_lock(&pmap->lock);

	pp = pmap_pp(pmap, va);
	if (pp != PP_NULL) {
	    for (i = pps_per_vm_page; i-- > 0; pp++) {
		if (pp->mod)
		    pmap_page_attributes[pai] |= PG_MOD;
		if (pp->ref)
		    pmap_page_attributes[pai] |= PG_REF;
	    }
	    if ((pmap_page_attributes[pai] & attrib) == attrib) {
		simple_unlock(&pmap->lock);
		PMAP_WRITE_UNLOCK(ipl);
		return (TRUE);
	    }
	}

	simple_unlock(&pmap->lock);

	if ((pv_h = pv_h->next) == PV_ENTRY_NULL)
	    break;
    }
    PMAP_WRITE_UNLOCK(ipl);

    return (FALSE);
}

/*
 * Dummy routine to satisfy external reference.
 */
void
pmap_update_interrupt()
{
    /* should never be called. */
    panic("pmap_update_interrupt");
}

/*
 * Lower the permission for all mappings to a given page.
 */
void
pmap_page_protect(pa, prot)
    vm_offset_t	pa;
    vm_prot_t	prot;
{
    switch (prot) {
	case VM_PROT_READ:
	case VM_PROT_READ|VM_PROT_EXECUTE:
	    pmap_copy_on_write(pa);
	    break;

	case VM_PROT_ALL:
	    break;

	default:
	    pmap_remove_all(pa);
	    break;
    }
}
