/*----------------------------------------------------------------------------
/ pdisk.c - tables for physical disk parameters and init routines
/
/ The algorithms are table driven so supporting a new disk merely
/ involves changing the disk parameter table and (possibly) the timing
/ register table. Entries in the table must be unique in their cylinders-heads-
/ sectors values.
/ If the tmregs is expanded make sure the last entry is null.
/
/ (c) Copyright 1988 ARIX Corp.  San Jose, CA
/---------------------------------------------------------------------------*/

#include "disktest.h"
#include "hsdterror.h"
#include "devcmd.h"


/* media defect record as she is written at the disk factory */

struct orig_medf {
	unsigned short	cyl;
	unsigned char	head;
	unsigned char	dummy;		/* must be == 0 */
	struct defpos	defpos[4];
	unsigned char	defpattern[4];	/* must be [0] == 0xf0 */
};

/* physical disk parameters */

struct dparmplate {
	ushort	cyldisk;		/* # of cyls per disk */
	uchar	headcyl;		/* # of heads per cylinder */
	uchar	sechead;		/* # of sectors per track */
	ushort	bytesec;		/* # of data bytes per sector */
	ushort	trbps;			/* # of bits + gap per sector */
	ushort	ltrbps;			/* # of bits + gap last sector */
	char	trtype;			/* timing register set to use */
	char	autosz;			/* ok to poke non-existent heads */
	char	gap;			/* recommended disk interlace gap */
	char    need_more_status;	/* send out for more status */
	short   extra_stat;		/* the additional status received */
	char	*diskname;		/* human readable disk type */
};

/* controller timing register definitions */

struct trplate {
	char tr[16];
	char *trname;
};

/* timing register definitions */

struct trplate tmregs[] = {		/* timing register values */
	{0x08, 0x0b, 0x00, 0x07,	/* standard SMD (CDC515, ...) */
	 0x01, 0x03, 0x00, 0x0f,
	 0x01, 0x00, 0x00, 0x1b,
	 0x00, 0x00, 0x00, 0x00},
	"standard SMD",

	{0x00, 0x0c, 0x00, 0x07,	/* Fujitsu Eagle 475M/689MB */
	 0x02, 0x01, 0x00, 0x0f,
	 0x03, 0x00, 0x00, 0x13,
	 0x00, 0x00, 0x00, 0x00},
	"Fuji Eagle",

	{0x08, 0x0e, 0x00, 0x07,	/* Extended SMD Fuji 8" 337MB */
	 0x01, 0x03, 0x00, 0x12,
	 0x03, 0x00, 0x00, 0x1b,
	 0x00, 0x00, 0x00, 0x00},
	"extended SMD",

	{0x08, 0x0b, 0x00, 0x07,	/* Hitachi SMD */
	 0x01, 0x03, 0x00, 0x0f,
	 0x02, 0x00, 0x00, 0x1b,
	 0x00, 0x00, 0x00, 0x00},
	"Hitachi SMD",

	{0}				/* spare--must be last entry */
};

#define TR_STD	0
#define	TR_EAGLE 1
#define	TR_EXT	2
#define TR_HIT  3

int	tmrend = {sizeof(tmregs) / sizeof(struct trplate) - 1};
int	tmridx;			/* index for current set of timing regs */

/* timing register values for reading media defect info
 * each non-zero value replaces its corresponding value in tmregs
 */
struct trplate mdtmmask = {
	0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00,
	0x00, 0x06, 0x00, 0xff
};
	
/* disk parameter table
 * to support new disks add entries to this table and/or tmregs
 */
struct dparmplate dptab[] = {
   /*cyls hds  secs  bps  bit/s   lbps+g tm regs autosz gap status extra_stat 
name */
    {589,  7,  18, 1024, 0x2390, 0x2370, TR_STD,   YES, 4,  NO,     0x23, "Fuji M2312 (84MB)"},
    {823, 10,  18, 1024, 0x2390, 0x2370, TR_STD,   YES, 4,  YES,    0x25,"Fuji M2322 (168MB)"},
    {842, 20,  25, 1024, 0x2338, 0x22c0, TR_EAGLE, YES, 7,   NO,    0,   "Fuji Eagle M2351(475MB)"},
    {842, 20,  36, 1024, 0x2390, 0x2350, TR_EAGLE, YES, 7,   NO,    0,   "Fuji Eagle M2361(689MB)"},
    {711, 24,  18, 1024, 0x2304, 0x22bc, TR_STD,   YES, 7,   NO,    0,   "CDC PA5G1 (340MB)"},
    {711, 24,  26, 1024, 0x2436, 0x2436, TR_STD,   NO,  7,   NO,    0,   "CDC PA5N1 (515MB)"},
    {823, 10,  36, 1024, 0x2390, 0x2350, TR_EXT,   YES, 7,   NO,    0x2e,"Fuji M2333 (337MB)"},
    {411, 19,  18, 1024, 0x2304, 0x226c, TR_STD,   YES, 4,   NO,    0,   "CDC 9764 (140MB)"},
    {823, 19,  18, 1024, 0x2304, 0x226c, TR_STD,   YES, 4,   NO,    0,   "CDC 9766 (280MB)"},
    {823, 5,   18, 1024, 0x2304, 0x226c, TR_STD,   YES, 4,   NO,    0,   "CDC 9762 (80MB)"},
    {624, 27,  36, 1024, 0x2390, 0x2350, TR_EXT,   YES, 7,   NO,    0,   "Fuji M2344 (690MB)"},
    {823, 10,  18, 1024, 0x2460, 0x2420, TR_HIT,   YES, 4,   YES,   0x4, "Hitachi DK5125-17(172MB)"},
    {745, 27,  36, 1024, 0x2390, 0x2350, TR_EXT,   YES, 7,   NO,    0,   "Fuji M2372 (824MB)"},
};

#define	DPTEND (sizeof(dptab)/sizeof(struct dparmplate))

char	wbuf[SECTSIZ+4];			/* long aligned buffer */
char	*wb = {&wbuf[3]};

/********************************************************************!HDR!*****
 *
 * inittmreg(b)
 *
 * initialize timing registsers 
 *
 * input:
 *
 *	pointer to sector0 structure
 *
 * description:
 *
 * 	find a set of timing registers which reads sector 0.
 * 	the code looks odd because HSDT does its own checking the first time
 * 	sector 0 is read.  If that read succeeds the TRs are in sector0->gap.
 * 	
 ********************************************************************!RDH!*****/
inittmreg(b)
struct sector0 *b;
{	struct trplate *p;
	short gotsec0, nodata;

	int i;
	char *j;

	tmridx = 0;
	wb = (char *)((int)wb & ~3);	/* point to long word */
	if (readdata(b, 0, 0, 0))	/* read sector 0      */
		gotsec0 = YES;		/* successful read */
	else	{
		if (errno == DER_NRDY) {		/* read failure */
            printf("Device not ready\n");
			longjmp(MenuBuffer);
		}
		gotsec0 = NO;
	}
	/*
	 * if we read sector 0 succesfully
	 *	 find timing register set that matches
	 */ 
	if (gotsec0) {
		for (p=tmregs; tmridx < tmrend; tmridx++, p++) {
			if (bcmp(p, b->gap, 16) != 16)
				continue;
			settmreg();
			return(1);
		}
	}
	/* 
	 * if timing reg set not found we fall through to this Point. this code 
	 * does look wierD
	 *
	 */
	for (p=tmregs; tmridx < tmrend; tmridx++, p++) {
		settmreg();
		if (readid(b, 0, 0, 0)) {
			if (readdata(b, 0, 0, 0)) {
                xprintf("Read ok timreg set %d\n", tmridx);
				return(1);
			}
			else
				return(0);
		}
	}
	/*
	 * still haven't found em
	 */
	if (tmridx == tmrend) {
		bcopy(b->gap, tmregs[tmridx].tr, 16);
		settmreg();
		if (!readid(b, 0, 0, 0))
			return(0);
		tmrend++;
		if (readdata(b, 0, 0, 0)) {
            xprintf("Timreg from sec0 set\n");
			return(1);
		}
		return(0);
	}
    xprintf("Can't find good timing regs\n");
	return(0);
}

/********************************************************************!HDR!*****
 * copytmregs()
 *
 ********************************************************************!RDH!*****/
copytmregs(sp)
struct sector0 *sp;
{	bcopy(tmregs[tmridx].tr, sp->gap, 16);
}

/********************************************************************!HDR!*****
 *
 * hsized()
 *
 * take a heuristic approach to finding the physical disk parameters.
 * the idea is to use physical device commands to ascertain disk parms.
 * if we can read sector 0 with a known set of timing registers tmridx has
 * that index already.
 * if media defect info can be read find number of cyls and heads and make a
 * guess at the disk type. Then go back and verify number of sectors.
 * To avoid hanging up the hardware the automatic head sizing can be turned off
 * and we won't try to read from what are thought to be non-existant heads.
 *
 ********************************************************************!RDH!*****/
hsized(sp, writeok)
struct sector0 *sp;
{	struct trplate *p;
	struct devq rq;
	int i, j, m;
	char havetreg, triedsec;

	xprintf("hsized(%s)\n", writeok?"writeok" : "readonly");
	/* find a set of timing registers which reads either
	   a media defect record or a regular data record
	 */
	p = &tmregs[tmridx];
	havetreg = TRUE;
	if (tmridx == tmrend) {
		havetreg = FALSE;
		for (tmridx=0; tmridx < tmrend; tmridx++) {
			setmdtmreg();
			if (readmd(wb, 0, 0) || readmd(wb, 1, 1) || readmd(wb, 2, 2)) {
				break;
			}
		}
	}
	if (tmridx == tmrend) {
        printf("Can't read either data or media defects using known timing registers\n");
		return(0);
	}
	settmreg();

	/* at this point we have a set of timing registers
	 * now attempt to discover cyls, heads and sectors/track
	 * use a binary search to find the last cylinder
	 */
	if (sp->cyldisk && sp->headcyl && sp->sechead && dptsrch(0, sp, &i) > 0)
		autosize = dptab[i].autosz ? 1 : -1;
	else if (!autosize)
        autosize = getyn("Do you want to automatically size the disk") ? 1 : -1;
	rqsetup(&rq, 0, 0, 0, 0);
	rq.q_cmd = SEEKCYL;
	for (i=0,j=0x400; i < j; ) {
		m = (i + j) >> 1;
		rq.q_devun.pdisk.cyl = m;
		rq.rc1 = rq.rc2 = 0;
		if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
			if (rq.rc1 != DER_SEEK) {
				deverr(rq.rc1, rq.rc2);
				if (rq.rc1 == DER_DEVCK)
					exit(1);
				return(FALSE);
			}
			j = m;
		}
		else	i = m + 1;
	}
	sp->cyldisk = i;

	/* search for last head */
	m = (autosize==1) ? 256 : sp->headcyl;
	for (i=0; i < m; i++){
		if (havetreg) {
		    if (readid(wb, 0, i, 0))
			continue;
		}
		if (errno == DER_RDYCHG || errno == DER_ABNOM)
			break;
		setmdtmreg();
		j = readmd(wb, 0, i);
		if (errno == DER_DEVCK || errno == DER_ABNOM)
			break;
		settmreg();
		if (j)
			continue;
		if (!havetreg || !writeok)
			break;
	     	if (!writid(wb, 0, i, 0) || !readid(wb, 0, i, 0))
			break;
	}
	if (errno == DER_ABNOM) {
		deverr(errno, 0);
        printf("You must reset the system and rerun disktest.\n");
        printf("Do not select the auto-size option when formatting this disk.\n");
		exit(1);
	}
	settmreg();
	sp->headcyl = i;
	/* now figure number of sectors.
	   format sectors until wrap around occurs
	   N.B. when wrap occurs check i-1th sector as well as the ith sector--
	   some drives have enough room for a short sector at the end of the
	   track.
	 */
	triedsec = 0;
   secsearch:
	sp->sechead = 0;
	if (havetreg) {
		for (i=0; i<256; i++) {
			if (readid(wb, 0, 0, i))
				continue;
			if (writeok) {
			    writid(wb, 0, 0, i);
			    if (!writc0(wb, 0, 0, i) || !readid(wb, 0, 0, 0)) {
				clearid(wb, 0, 0, i);
				writid(wb, 0, 0, 0);
				writc0(wb, 0, 0, 0);
				writc0(wb, 0, 0, i-1);
				if (!readid(wb, 0, 0, 0)) {
				    writid(wb, 0, 0, 0);
				    writc0(wb, 0, 0, 0);
				    clearid(wb, 0, 0, i-1);
				    i--;
				}
			    	break;
			    }
			}
			else
				break;
		}
		if (i > 1)
			sp->sechead = i;
	}
	if ( (j=dptsrch(0, sp, &i)) == 0) {
		j = sp->sechead;
		if (havetreg)
			sp->sechead = 0;
		if (dptsrch(0, sp, &i) < 0 && dptsrch(i, sp, &i)) {
		    if (tmridx != dptab[i].trtype)
			havetreg = NO;
		}
		else	{
			sp->sechead = j;
			return(NO);
		}
	}
	if (!havetreg) {
		havetreg = YES;
		tmridx = dptab[i].trtype;
		settmreg();
		goto secsearch;
	}
	sp->trkbpsg = dptab[i].trbps;
	sp->ltrkbpsg = dptab[i].ltrbps;
	tmridx = dptab[i].trtype;



	MthrBlk.rec_interlace = dptab[i].gap;
	copytmregs(sp);

	return(TRUE);
}

/********************************************************************!HDR!*****
 *
 * dptsrch()
 *
 * search for disk with matching parameters
 * return 0 if not found, -1 if more than one entry matches, 1 if unique
 *
 ********************************************************************!RDH!*****/
dptsrch(i, sp, ip)
unsigned i;
register struct sector0 *sp;
int *ip;
{	int j;
	struct devq rq;

	j = -1;
	for (; i<sizeof(dptab)/sizeof(dptab[0]); i++) {
		if (dptab[i].cyldisk == sp->cyldisk
		    && dptab[i].headcyl == sp->headcyl
		    && (dptab[i].sechead==sp->sechead || sp->sechead==0)){
			if (j == -1)
				j = i;
			else
				break;
		}
	}
	*ip = 0;
	if (j == -1)
		return(0);
	*ip = j;
	if (i < sizeof(dptab)/sizeof(dptab[0])){
		if(dptab[i].need_more_status == YES){
	    /* At this point we cannot be sure of the disk type
	       so we must request additional status information
	    */
			rqsetup(&rq, 0, 0, 0, 0);
			rq.q_cmd = NEWSTAT;
			rq.q_count = 0;
			if (devreq(SlotPtr, &rq) || rq.rc1  || rq.rc2 ) {
				errno = rq.rc1 ? rq.rc1 : rq.rc2;
				return(0);
			}
			if(rq.q_count == dptab[i].extra_stat)
				*ip = i;
			return(1);
		}

		return(-1);
	}
	else	return(1);
}

/********************************************************************!HDR!*****
 *
 * readdata()
 *
 *	returns YES (1) if devreq() succeeds
 *	returns NO (1) and sets global errno if failure
 ********************************************************************!RDH!*****/
readdata(b, c, h, s)
long *b;
{	struct devq rq;

	rqsetup(&rq, b, c, h, s);
	rq.q_cmd = PREADPS;
	rq.q_count = sectsiz;
	if (devreq(SlotPtr, &rq) || rq.rc1  || rq.rc2 || rq.q_count!=sectsiz) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	return(YES);
}

/********************************************************************!HDR!*****
 * readid()
 * 	
 * read sector id header
 *
 ********************************************************************!RDH!*****/
readid(idp, c, h, s)
register struct rdsecid *idp;
{
	struct devq rq;
	int cc;


	rqsetup(&rq, idp, c, h, s);
	rq.q_cmd = READID;
	rq.q_count = 1;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	if (idp->flags != 0x1900 || c != idp->cyl || h != idp->head || s != idp->sector)
		return(NO);
	return(YES);
}

/********************************************************************!HDR!*****
 *
 * readmd()
 *
 * read a media defect record from the track.
 * timing registers must be modified to do this
 * 	
 ********************************************************************!RDH!*****/
readmd(mp, c, h)
struct orig_medf *mp;
{	struct devq rq;
	char mdtr[16], *p;
	int i;

	rqsetup(&rq, mp, c, h, 0);
	rq.q_cmd = MEDEFTRK;
	rq.q_count = 1;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	if (mp->dummy != 0 || mp->defpattern[0] != 0xf0)
		return(NO);
	if ((mp->cyl & 0x7fff) != c || mp->head != h)
		return(NO);
	/* if defpos[N]!=0 then defpos[N-1] ... defpos[0] better be != 0 */
	for (p=0, i=4; i--; ) {
		if (mp->defpos[i].pos || mp->defpos[i].len)
			p = (char *)mp;
		else if (p)
			return(NO);
	}
	return(YES);
}

writid(idp, c, h, s)
register struct wrsecid *idp;
{	struct devq rq;

	bzero(idp, sizeof(struct wrsecid));
	idp->fmtf = 0;
	idp->cylhi = (c >> 8) & 0xff;
	idp->cyllo = c & 0xff;
	idp->head = h;
	idp->sec = s;
	rqsetup(&rq, idp, c, h, s);
	rq.q_cmd = WRITEID;
	rq.q_count = 1;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	return(YES);
}

/* clear a sector header
 */
clearid(idp, c, h, s)
register struct wrsecid *idp;
{	struct devq rq;

	bzero(idp, sizeof(struct wrsecid));
	idp->fmtf = 0xff;
	idp->cylhi = 0x0f;
	idp->cyllo = 0xff;
	rqsetup(&rq, idp, c, h, s);
	rq.q_cmd = WRITEID;
	rq.q_count = 1;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	return(YES);
}

writdata(p, c, h, s)
char *p;
{	struct devq rq;
	rqsetup(&rq, p, c, h, s);
	rq.q_cmd = PWRITPS;
	rq.q_count = sectsiz;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	return(YES);
}

/********************************************************************!HDR!*****
 * writc0()
 *
 * special version of write data for sectors in cylinder 0
 * HSDT doesn't need a nice sector 0 to execute this command
 *
 ********************************************************************!RDH!*****/
writc0(p, c, h, s)
char *p;
{	struct devq rq;
	rqsetup(&rq, p, c, h, s);
	rq.q_cmd = PWRITS0;
	rq.q_count = sectsiz;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		errno = rq.rc1 ? rq.rc1 : rq.rc2;
		return(NO);
	}
	return(YES);
}

/********************************************************************!HDR!*****
 * settmreg()
 * 	
 * 	issue a set timing register command
 *
 ********************************************************************!RDH!*****/
settmreg()
{	struct devq rq;

	rqsetup(&rq, tmregs[tmridx].tr, 0, 0, 0);	/* init req. struct. */
	rq.q_cmd = SETIME;				/* set time reg. flag */
	rq.q_count = 0;					/* byte cound ? */
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		deverr(rq.rc1, rq.rc2);
		return(NO);
	}
	return(YES);
}

/********************************************************************!HDR!*****
 * setmdtmreg()
 * 	
 *
 ********************************************************************!RDH!*****/
setmdtmreg()
{	struct devq rq;
	char *p;
	int i;

	p = tmregs[tmridx].tr;
	for (i=0; i<16; i++, p++)
		wb[i] = mdtmmask.tr[i] ? mdtmmask.tr[i] : *p;
	rqsetup(&rq, wb, 0, 0, 0);
	rq.q_cmd = SETIME;
	rq.q_count = 0;
	if (devreq(SlotPtr, &rq) || rq.rc1 || rq.rc2) {
		deverr(rq.rc1, rq.rc2);
		return(NO);
	}
	return(YES);
}

/********************************************************************!HDR!*****
 * rqsetup()
 * 	
 * 	get ready to send a request 
 *
 ********************************************************************!RDH!*****/
rqsetup(rp, b, c, h, s)
struct devq *rp;
{
	bzero(rp, sizeof(struct devq));	/* clear */
	rp->q_devtype = DTDISK;
	rp->q_devnum = DriveNum;
	rp->q_devun.pdisk.cyl = c;
	rp->q_devun.pdisk.head = h;
	rp->q_devun.pdisk.sector = s;
	rp->q_count = 0;
	rp->q_extdtb = XDDFLT;
	rp->q_priority = 0;
	rp->q_flag = 1;			/* no automatic retries */
	rp->q_mem = (char *)b;		/* transfer area */
	rp->rc1 = 0;
	rp->rc2 = 0;
	errno = 0;
}

/********************************************************************!HDR!*****
 * printdtype()
 * 	
 * print the disk manufacturer name, and layout
 *
 ********************************************************************!RDH!*****/
printdtype(sp)
register struct sector0 *sp;
{	struct dparmplate *dp;
	struct devq rq;
	int i;
	char *p;

	p = "unknown disk";
	for (dp=dptab; dp < &dptab[DPTEND]; dp++) {
		if (dp->cyldisk == sp->cyldisk
		    && dp->headcyl == sp->headcyl
		    && dp->sechead == sp->sechead) {
			p = dp->diskname;
			if(dp->need_more_status == YES){
	    /* At this point we cannot be sure of the disk type
	       so we must request additional status information
	    */
				rqsetup(&rq, 0, 0, 0, 0);
				rq.q_cmd = NEWSTAT;
				rq.q_count = 0;
				if (devreq(SlotPtr, &rq) || rq.rc1  || rq.rc2 ) {
					errno = rq.rc1 ? rq.rc1 : rq.rc2;
					break;
				}
				if(rq.q_count != dp->extra_stat)
					continue;
			}
			break;
		}
	}
	printf("\nc%dd%d: %s\n", UnitNum >> 4, DriveNum, p);
	printf("%d(0x%x) cylinders\n", sp->cyldisk, sp->cyldisk);
	printf("%d(0x%x) heads\n", sp->headcyl, sp->headcyl);
	printf("%d(0x%x) sectors/track\n", sp->sechead, sp->sechead);
	i = sp->sechead * sp->headcyl * sp->cyldisk;
	printf("%d(0x%x) sectors/disk\n", i, i);
	printf("%d(0x%x) bytes/sector\n", sp->bytesec, sp->bytesec);
}

/********************************************************************!HDR!*****
 * gettmreg()
 * 	
 *
 ********************************************************************!RDH!*****/
gettmreg(p)
unsigned *p;
{	int i, b[4], r;

	tmrend = sizeof(tmregs) / sizeof(tmregs[0]) - 1;
	for (i=0; i < tmrend; i++)
		printf("%c: %s\n", i+'a', tmregs[i].trname);
	printf("%c: new timing register values\n", i+'a');
	do	{
        printf("Enter a-%c for timing register set: ", i+'a');
		tmridx = getreply() - 'a';
	} while (tmridx < 0 || tmridx > i);
	if (tmridx == tmrend) {
		for (i=0; i<4; i++)
			b[i] = p[i];
		printf("enter timing register values(4 hex bytes at a time)\n");
		bcopy(p, b, 16);
		do	{
			printf("current values: 0x%8x %8x %8x %8x\n", b[0],
					b[1], b[2], b[3]);
			for (i=0; i<4; i++) {
				printf("+%d ", i<<2);
				b[i] = g1parm("0x%x? ", b[i], YES);
			}
        } while (!getyn("Ok"));
		bcopy(b, tmregs[tmrend].tr, 16);
		tmrend++;
	}
	bcopy(tmregs[tmridx].tr, p, 16);
}

/********************************************************************!HDR!*****
 * getdparm()
 * 	
 * get disk parameters from the human
 *
 ********************************************************************!RDH!*****/
getdparm(sp)
struct sector0 *sp;
{	int t, r;
	struct sector0 sx;
	struct dparmplate *dpp;
again:
	for (t='a', dpp=dptab; dpp < &dptab[sizeof(dptab)/sizeof(*dpp)]; t++, dpp++)
		printf("%c: %s\n", t, dpp->diskname);
	printf("%c: new disk\n", t);
	do	{
        printf("Enter a-%c for your disk type--> ", t);
		r = getreply();
	} while (r < 'a' || r > t);
	if (r < t) {
		dpp = &dptab[r-'a'];
		sp->cyldisk = dpp->cyldisk;
		sp->headcyl = dpp->headcyl;
		sp->sechead = dpp->sechead;
		sp->bytesec = dpp->bytesec;
		sp->trkbpsg = dpp->trbps;
		sp->ltrkbpsg =  dpp->ltrbps;
		tmridx = dpp->trtype;
		bcopy(tmregs[tmridx].tr, sp->gap, 16);
		if (!hsized(&sx, YES)
		    || sx.cyldisk != sp->cyldisk || sx.headcyl != sp->headcyl
		    || sx.sechead != sp->sechead) {
			printf("controller returns:\n");
			printdtype(&sx);
            printf("Type specified does not agree with physical disk\n");
            if (getyn("Do you still want to format the drive"))
				return;
            printf("Check the drive settings.\n");
            if (getyn("Do you want to try again"))
				goto again;
			else
				longjmp(fabort);
		}
		return;
	}
	while (1) {
		gettmreg(sp->gap);
		settmreg();
		printf("enter all values in hex\n<cr> means leave value as is\n");
		sp->cyldisk = g1parm("cyls %d(0x%x)? ", sp->cyldisk, NO);
		sp->headcyl = g1parm("heads %d(0x%x)? ", sp->headcyl, NO);
		sp->sechead = g1parm("sectors/track %d(0x%x)? ", sp->sechead, NO);
		sp->bytesec = SECTSIZ;
		/*     = g1parm("bytes/sector %d(0x%x)? ", sp->bytesec, NO);
		 */
		printf("bytes/sector %d(0x%x)\n", sp->bytesec, sp->bytesec);
		sp->trkbpsg = g1parm("bits/sector+gap %d(0x%x)? ", sp->trkbpsg, NO);
		sp->ltrkbpsg = g1parm("bits/lastsector+gap %d(0x%x)? ", sp->ltrkbpsg, NO);
		bcopy(sp, &sx, sizeof(sx));
		hsized(&sx, YES);
		if (sx.cyldisk != sp->cyldisk || sx.headcyl != sp->headcyl
		    || sx.sechead != sp->sechead) {
            printf("Controller returns:\n");
			printdtype(&sx);
			sp->cyldisk = sx.cyldisk;
			sp->headcyl = sx.headcyl;
			sp->sechead = sx.sechead;
            printf("Controller values will be used\n");
		}
        if (getyn("Ok"))
			return;
        if (getyn("Quit"))
			longjmp(fabort);
	}
}

	
g1parm(t, i, zerook)
{	char b[CBUFSIZ], *bp;
	int ii;

	while (1) {
		bp = b;
		printf(t, i, i);
        if (gets(b, CBUFSIZ) < 0)
			continue;
		if (*bp == 0)
			break;
		if (getlong(&bp, &ii) == 0) {
			if (ii==0 && !zerook) {
                printf("Value must be non-zero\n");
				continue;
			}
			i = ii;
			break;
		}
		prhuh();
	}
	return(i);
}

/********************************************************************!HDR!*****
 * bcmp()
 *	
 *	compare l bytes at location a with l bytes a location b
 *	return address of first location that differs
 *
 ********************************************************************!RDH!*****/
bcmp(a, b, l)
register char *a, *b;
register int l;
{
    char *aa = a;

    while (l-- && *a == *b++)
		a++;
	return(a - aa);
}

/*--------------------------------- End of pdisk.c -------------------------*/
