/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) swap.c: version 23.1 created on 11/15/90 at 13:27:03	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)swap.c	23.1	11/15/90 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 *	swap.c  (lives in /etc/swap)
 *		Swap administrative interface.
 *		Used to add/delete/list swap devices in use by paging.
 *		Use the sysarix() ARIXSWPI command.
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/swap.h>
#include <sys/sysarix.h>
#include <sys/sysmacros.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <sys/immu.h>
#include <fcntl.h>

char		*cmd_list[] = { "list", "add", "delete" };

extern int	errno;
extern int	optind;
extern char	*optarg;

main(argc,argv)
register int	argc;
register char	**argv;
{
register int	c, nblks, did_one, err;
register char	cmd = SI_LIST;

	nblks = did_one = err = 0;
	while ((c = getopt(argc, argv, "a:d:n:l")) != EOF) {
		switch (c) {
		case 'a':	/* add a swap device */
			++did_one;
			cmd = SI_ADD;
			if (c = add_del(cmd,optarg,nblks))
				err = c;
			break;

		case 'd':	/* delete a swap device */
			++did_one;
			cmd = SI_DEL;
			if (c = add_del(cmd,optarg,nblks))
				err = c;
			break;

		case 'l':	/* list swap devices */
			++did_one;
			cmd = SI_LIST;
			if (c = swap_list())
				err = c;
			break;

		case 'n':	/* set nblks (size of the device in blocks) */
			nblks = getnumarg(optarg);
			break;

		default:
			fprintf(stderr,"swap: unknown flag '%c'\n",c);
			exit (altusage());
		}
	}
	/*
	 * if there are extra arguments, repeat the last ADD/DEL command
	 */
	if ((cmd != SI_LIST) && (optind < argc)) {
		for (; optind < argc; optind++) {
			if (c = add_del(cmd,argv[optind],nblks))
				err = c;
		}
	}
	else
		if (did_one == 0)
			err = swap_list();
	exit (err);
}

/*
 * add_del -- add or delete a swap device, returns non-zero on error
 */
add_del(cmd,devnm,nblks)
register char	cmd, *devnm;
register int	nblks;
{
register int	ret;
swpi_t		swpi;			/* structure for sysarix	*/

	swpi.si_cmd   = cmd;
	swpi.si_buf   = devnm;
	swpi.si_swplo = 0;
	swpi.si_nblks = nblks;
	if ((ret = sys_arix(&swpi)) < 0)
		return (errno);
	fprintf(stderr,"swap: %s of %s is done (Swap Index = %d).\n",
		cmd_list[swpi.si_cmd],swpi.si_buf,ret);
	return (0);
}

/*
 * swap_list -- print a list of swap devices
 */
swap_list()
{
register int	i;
register swpt_t	*sp;
swpt_t		swpt[MSFILES];		/* SI_LIST buffer		*/
swpi_t		swpi;			/* structure for sysarix	*/
char	*path, *GetPath();

	sp = swpt;
	swpi.si_cmd   = SI_LIST;
	swpi.si_buf   = (char *)sp;
	swpi.si_swplo = 0;
	swpi.si_nblks = 0;
	if (sys_arix(&swpi) < 0)
		return (errno);
	InitPath();
	printf("Idx Path               Device    NumPgs    FreePgs     NextPg\n");
	for (i = 0; i < MSFILES; i++, sp++) {
		if (sp->st_ucnt == NULL)
			continue;		/* not in use */
		sp->st_dev &= 0x7fff; /* Take off internal not-minored bit */
		path = GetPath(sp->st_dev);
		if (path == NULL) {
			path = "??????";
		}
		printf("%2d  %-18.18s  %04x %10d %10d %10d",
			i,
			path,
			sp->st_dev,
			sp->st_npgs,
			sp->st_nfpgs,
			(sp->st_next - sp->st_ucnt));
		if (sp->st_flags & ST_OPEN)
			printf(" (being opened)");
		if (sp->st_flags & ST_INDEL)
			printf(" (being deleted)");
		(void) fputc('\n', stdout);
	}
	return (0);
}

/*
 * sys_arix -- error trapped sysarix call
 */
sys_arix(si)
register swpi_t	*si;
{
register int	ret;
register char	*p;
extern char	*sys_errlist[];

	if ((ret = sysarix(ARIXSWPI,si)) >= 0)
		return (ret);
	switch (errno) {
	case EINVAL:
		p = (si->si_cmd == SI_DEL) ? "Not a swap device" :
			sys_errlist[EINVAL];
		break;
	case EPERM:
		p = "You must be root to add or delete swap devices";
		break;
	case ENOTBLK:
		p = "The pathname must refer to a block device";
		break;
	case ENOMEM:
		p = (si->si_cmd == SI_ADD) ?
"Could not allocate enough memory for the swap map" :
"Can't delete last swap device or delete wouldn't leave enough swap space";
		break;
	case EEXIST:
		p = "The device is already a swap device";
		break;
	case ENOSPC:
		p = "The maximum number of swap devices are already active";
		break;
	default:
		p = sys_errlist[errno];
	}
	fprintf(stderr,"swap: %s %s failed!\n***** %s!\n",
		cmd_list[si->si_cmd],(si->si_buf ? si->si_buf : ""),p);
	return (-1);
}

/*
 * getnumarg -- return a numeric argument greater than zero,
 *		returns less than zero on error
 */
getnumarg(narg)
register char	*narg;
{
register long	n;
char		*p;		/* don't declare as a register	*/
extern long	strtol();

	p = narg;
	n = strtol(narg, &p, 0);
	if (p == narg) {
		fprintf(stderr,"swap: argument '%s' must be numeric.\n",
			narg);
		exit (2);
	}
	if (n < 0) {
		fprintf(stderr,"swap: argument '%d' must positive.\n",n);
		exit (3);
	}
	return (n);
}

/*
 * altusage -- print a usage message and exit
 */
altusage()
{
	register char	**cp;
	static char	*msg[] = {
"\nusage:  swap [-l] [-n <nblks>] [-a <swap_dev> | -d <swap_dev>]...\n",
"Swap adds, deletes, or lists system swap devices.  If no arguments are",
"supplied, it will list.\n",
"-l -- Lists the current swap devices.",
"-a -- Add one or more swap devices.",
"-d -- Delete one or more swap devices.",
"<swap_dev> is the pathname of a block device node which will be, or has been,",
"      used as a swap device.  Any previous contents of the device have",
"      already been destroyed or will be destroyed.",
"-n -- Sets the number of blocks in the device [add only] (nblks is set",
"      automatically by default, or if <nblks> is zero).",
NULL
	};
	for (cp = msg; *cp != NULL; cp++)
		fprintf(stderr,"%s\n",*cp);
	return (1);
}

typedef struct SwapDirs {
	char	*sd_name;	/* Name of directory to search.	*/
	int	sd_fd;		/* File descriptor.		*/
} SwapDirs_t;

SwapDirs_t	SwapDirs[] = {
	{"/dev/dsk"},
	{"/dev/stripe"},
	{"/dev"},
};

typedef struct direct	dir_t;

#define	NbrSwapDirs	(sizeof(SwapDirs) / sizeof(SwapDirs_t))
#define	MAXPATH	200
#define	DirBufSz	(BSIZE / sizeof(dir_t))

dir_t	DirBuf[DirBufSz];

char	PathBuf[MAXPATH];

/*
 * Initialization for the path search routine.
 */
InitPath()
{
	register int	i;

	for (i = 0; i < NbrSwapDirs; i++) {
		SwapDirs[i].sd_fd = open(SwapDirs[i].sd_name,O_RDONLY);
		if (SwapDirs[i].sd_fd < 0) {
			fprintf(stderr,"Couldn't open directory %s",
				SwapDirs[i].sd_name);
			fprintf(stderr," - not being searched.\n");
			perror("swap: InitPath");
		}
	}
	return (0);
}

/*
 * Get path name corresponding to a device.
 */
char *
GetPath(dev)
register int	dev;
{
register int	sdndx;
register char	*path;
char		*SearchDir();

	for (sdndx = 0; sdndx < NbrSwapDirs; sdndx++) {
		/*
		 * Skip any directories which we couldn't open.
		 */
		if (SwapDirs[sdndx].sd_fd < 0)
			continue;
		/*
		 * Change to directory we want to search.
		 */
		if (chdir(SwapDirs[sdndx].sd_name) < 0) {
			fprintf(stderr,"Couldn't chdir to %s",
				SwapDirs[sdndx].sd_name);
			fprintf(stderr," - directory not searched.\n");
			perror("swap: GetPath");
			SwapDirs[sdndx].sd_fd = -1;
			continue;
		}
		/*
		 * Position at start of directory.
		 */
		if (lseek(SwapDirs[sdndx].sd_fd, 0, 0) != 0 )
			perror("Couldn't lseek");
		if (path = SearchDir(&SwapDirs[sdndx],dev))
			return (path);
	}
	return (NULL);
}

char *
SearchDir(sdp, dev)
register SwapDirs_t	*sdp;
register int		dev;
{
register int	i, limit;
register dir_t	*dp;
struct stat	statbuf;

	while (limit = read(sdp->sd_fd, DirBuf, DirBufSz)) {
		limit /= sizeof(dir_t);
		dp = (dir_t *)DirBuf;
		for (i = 0; i < limit; i++, dp++) {
			if (dp->d_ino == 0)
				continue;
			sprintf(PathBuf,"%.14s",dp->d_name);
			if (stat(PathBuf,&statbuf) < 0) {
				fprintf(stderr,"Could not stat %s.n",PathBuf);
				perror("swap: SearchDir");
				continue;
			}
			if (statbuf.st_rdev != dev)
				continue;
			if ((statbuf.st_mode & S_IFMT) != S_IFBLK)
				continue;
			strcpy(PathBuf,sdp->sd_name);
			strcat(PathBuf,"/");
			sprintf(&PathBuf[strlen(PathBuf)],"%.14s",dp->d_name);
			return (PathBuf);
		}
	}
	return (NULL);
}
