/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) syscheck.c: version 1.1 created on 5/17/90 */
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)syscheck.c	1.1	5/17/90 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/* syscheck - Allows users to read or set labels, privileges, modes,
	      ownerships, and verify file sizes

	      This utility takes data lines, one at time, from stdin,
	      and either verifies security issues (-c) or sets them (-s)

	      Data line should be of the form:
	      Permissions file_size uid gid labels privileges file_name

	      ie

	      -rwsrwxr-x 23996 2 4 TRUSTED P_FSYS:P_MKNOD /tmp/mic

	      Multiple labels & privileges should be seperated by a ':'

	      The -g option will take data lines, one at a time, from stdin,
	      and generate a new tcblist to stdout, correcting the file
	      sizes.  All other options will be ignored.
*/

#include "syscheck.h"
#include <stdio.h>
#include <fcntl.h>
#include <sys/priv.h>
#include <sys/mls.h>
#include <sys/types.h>
#include <sys/stat.h>

#define	MODE_LEN	11
#define MAX_DATA_LINE_SIZE 1024

unsigned int	flags = 0, exit_status = 0;

main (argc, argv)
int	argc;
char	**argv;
{
	extern	char	*optarg;
	unsigned int	opt;

	if (argc < 2)

		usage ();

	while ( (opt = getopt (argc, argv, "gscdlp?") ) != EOF)

		switch (opt) {

			case 's':	/* Set security */

				flags = (flags & ~GET_SEC) | SET_SEC;
				break;

			case 'c':	/* Get security */

				flags = (flags & ~SET_SEC) | GET_SEC;
				break;

			case 'l':	/* Set labels */

				flags |= SET_LABEL;
				break;

			case 'p':

				flags |= SET_PRIV;
				break;

			case 'd':

				flags |= SET_DAC;
				break;

			case 'g':

				flags |= GENERATE;
				break;

			case '?':
			default:

				usage ();
		}

	/* If not security administrator, exit */
	if ( ! priv (P_SEC, geteuid ()) ) {

		fprintf ( stderr, "permission denied\n");
		exit (1);
	}


	if ( flags & GENERATE )

			generate_list ();	/* Generates a new tcb list, with 
						   corrected file sizes.  This routine
						   won't return*/
	else if ( flags & GET_SEC )

			check_security ();	/* This routine won't return */

	else if (flags & SET_SEC)

			set_security ();	/* This routine won't return */

	else {

		fprintf (stderr, "syscheck: Unknown flags set\n");
		exit (1);
	}
}


/* generate_list - Reads current tcb list, stats all regular files, and
		   updates their sizes.  Add a checksum routine here
		   also once that has been implemented. */
int
generate_list ()
{
	char	data_line [MAX_DATA_LINE_SIZE], mode [MODE_LEN];
	syscheck_t	sys_chk;
	struct	stat	st;
	int	args;

	while ( gets (data_line) != (char *)0 ) {

		args = load_sys_chk_struct ( &sys_chk, data_line, mode );
		if ( args == EOF ) {

			fprintf (stderr, "syscheck: Unable to parse data line\nline = %s\n\n", data_line);
			break;
		}

		else {

			if ( stat ( sys_chk.path, &st ) == -1 ) {

				fprintf (stderr, "syscheck: unable to stat file %s\n", sys_chk.path);
				exit_status = 1;
			}

			else {

				fprintf (stdout, "%s	%d	%d	%d	%s	%s	%s\n", mode,
					 st.st_size, sys_chk.uid, sys_chk.gid,
					 sys_chk.label, sys_chk.privs, sys_chk.path);
			}
		}
	}

	exit ( exit_status );
}
/* check_security - reads in data from stdin, and verifies that each file named
		    exists and has appropriate DACS, privileges, and labels */
int
check_security ()
{
	char		data_line [MAX_DATA_LINE_SIZE];
	syscheck_t	sys_chk;

	while ( gets (data_line) != (char *)0 ) {

		if ( parse (data_line, &sys_chk)  != 0 ) {

			exit_status = 1;
		}

		else {

			/* if -1 is returned, file is not accessable,
			   so don't bother going further */
			if ( check_dac (&sys_chk) != -1) {
	
				check_label (&sys_chk);
				check_priv (&sys_chk);
			}
		}
	}
	exit (exit_status);
}


/* set_security - takes data lines one at a time from stdin, and tries
		  to set DAC, privileges, and labels, as specified by
		  the command line options 
*/

int
set_security ()
{
	char		data_line [MAX_DATA_LINE_SIZE];
	syscheck_t	sys_chk;

	while ( gets (data_line) != (char *)0 ) {

		if ( parse (data_line, &sys_chk)  != 0 ) {

			exit_status = 1;
		}

		else {

			if ( flags & SET_DAC ) {

				set_dac (&sys_chk);
			}

			if ( flags & SET_LABEL ) {

				set_label (&sys_chk);
			}

			if ( flags & SET_PRIV ) {

				set_priv (&sys_chk);
			}
	
		}
	}

	exit (exit_status);
}


/* check_dac - checks Discretionary Access Controls (permissions, uid,
	       gid, and file size) */
int
check_dac (sys_chk)
syscheck_t	*sys_chk;
{
	struct	stat stats;

	if ( stat (sys_chk->path, &stats) == -1 ){

		fprintf (stderr, "syscheck: Unable to stat () file %s\n", 
			 sys_chk->path);
		exit_status = 1;
		return (-1);
	}

	else {

		if ( stats.st_mode != sys_chk->mode ) {

			fprintf (stderr,"syscheck: modes mismatch, file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}

		if (stats.st_uid != sys_chk->uid ){

			fprintf (stderr, "syscheck: uid mismatch, file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}

		if (stats.st_gid != sys_chk->gid ) {

			fprintf (stderr, "syscheck: gid mismatch, file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}
/*
		if ( ( (stats.st_mode & S_IFMT) == S_IFREG) &&  
		     (stats.st_size != sys_chk->size) ) {

			fprintf (stderr, "syscheck: size mismatch, file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}
*/
	}

	return (0);
}


/* check_label - verifies label on a file is the correct one */
int
check_label (sys_chk)
syscheck_t	*sys_chk;
{
	slabel_t	file_label, req_label;

	/* Initialize structures to 0 */
	memset ( (char *)&file_label, 0, sizeof (slabel_t));
	memset ( (char *)&req_label, 0, sizeof (slabel_t));

	if ( ! cvt_sym_slabel (sys_chk->label, &req_label) ) {

		fprintf (stderr, "syscheck: undeciperable label = %s\n",
			sys_chk->label);
		fprintf (stderr, "          file = %s\n", sys_chk->path);

	} 

	else if ( get_file_slabel (sys_chk->path, &file_label) == -1 ) {

		fprintf (stderr,
			"syscheck: unable to retrieve label for file %s\n",
			 sys_chk->path);
	}
	
	else if ( ! mls_equ (&req_label, &file_label) ) {

		fprintf (stderr,
			"syscheck: label mismatch for file %s\n",sys_chk->path);
	}

	else

		return (0);	/* Everythings cool */

	exit_status = 1;
	return (-1);	/* failed, no biscuit! */
}


/* check_priv - verifies that a file's privileges are correct */
int
check_priv (sys_chk)
syscheck_t	*sys_chk;
{
	unsigned int	file_priv = 0, req_priv = 0;

	if ( ! cvt_sym_to_priv (sys_chk->privs, &req_priv, ',') ){

		fprintf (stderr, "syscheck: undecipherable privilege = %s\n",
			 sys_chk->privs);
		fprintf (stderr, "          file = %s\n", sys_chk->path);
	}

	else if ( get_file_priv (sys_chk->path, &file_priv) == -1 ) {

		fprintf (stderr, "syscheck: can't get privileges for file %s\n",
			 sys_chk->path);
	}

	else if ( file_priv != req_priv ){

		fprintf (stderr, "syscheck: privilege mismatch for file %s\n",
			 sys_chk->path);
	}

	else {

		return (0);
	}

	exit_status = 1;
	return (-1);
}

		
/* parse - takes a line of data, and stuffs the syscheck structure with
	   the appropriate data, or returns -1
*/
int
parse (data_line, sys_chk)
char	*data_line;
syscheck_t	*sys_chk;
{
	int	args;
	char	mode [MODE_LEN];  /* Other routines require the mode unparsed */

	args = load_sys_chk_struct ( sys_chk, data_line , mode );

	if ( args == EOF ) {

		fprintf (stderr, "syscheck: Unable to parse data line\nline = %s\n\n", data_line);
	}

	else if ( ! (sys_chk->mode = getmode (mode)) ) {

		fprintf (stderr, "syscheck: Unable to parse modes\nline = %s\n\n", data_line);
	}

	else {

		return (0);
	}

	exit_status = 1;
	return (-1);
}


/* GETMODE:  */
/* Expected ascii string: */
/*    [d,b,c,p,-] - file type */
/*    [r,-][w,-][x,-,s,S] - owner permissions (last field "s" is set uid)*/
/*    [r,-][w,-][x,-,s,l] - group permissions (last field "s" is set group id)*/
/*    [r,-][w,-][x,-,t,T] - other (world) permissions (last field is set */
/*				sticky bit */
/* (In this case, each field must have a value, the choice is within the  */
/*  brackets).  Example:   -rwsr-xr-t */
/* note that "s or t" impiles "x" in the corresponding field */
/* note that "S, l or T" impiles NO "x" in the corresponding field */
/* This display is identical to that shown by "ls -l" */
/* getmode returns "0" if error, otherwise returns the mode. */


int
getmode(str)
char str[MODE_LEN];
{
	int mode=0;

#ifndef S_IRGRP	/* 5.3 stat.h defines these, 5.2 stat.h does not */
#define	S_IRGRP	00040		/* read permission: group */
#define	S_IWGRP	00020		/* write permission: group */
#define	S_IXGRP	00010		/* execute permission: group */
#define	S_IROTH	00004		/* read permission: other */
#define	S_IWOTH	00002		/* write permission: other */
#define	S_IXOTH	00001		/* execute permission: other */
#endif

	if (strlen(str) != MODE_LEN-1)	/* don't count terminating null here */

		return (0);

	switch(str[0]){
		case '-':
			mode |= S_IFREG; /* regular */
			break;
		case 'd':
		case 'c':
		case 'b':
		case 'p':
		default:
			return(0);
	}
	
	if (str[1] == 'r')	/* owner modes */
		mode |= S_IREAD;
	else if (str[1] != '-')
		return(0);

	if (str[2] == 'w')
		mode |= S_IWRITE;
	else if (str[2] != '-')
		return(0);

	switch (str[3]){
		case 'x':
			mode |= S_IEXEC;
			break;
		case 's':
			mode |= S_ISUID;
			mode |= S_IEXEC;
			break;
		case 'S':
			mode |= S_ISUID;
			break;
		case '-':
			break;	/* okay */
		default:
			return(0);
	}

	if (str[4] == 'r')	/* group modes */
		mode |= S_IRGRP;
	else if (str[4] != '-')
		return(0);

	if (str[5] == 'w')
		mode |= S_IWGRP;
	else if (str[5] != '-')
		return(0);

	switch (str[6]){
		case 'x':
			mode |= S_IXGRP;
			break;
		case 's':
			mode |= S_ISGID;
			mode |= S_IXGRP;
			break;
		case 'l':
			mode |= S_ISGID;
			break;
		case '-':
			break;
		default:
			return(0);
	}

	if (str[7] == 'r')	/* world modes */
		mode |= S_IROTH;
	else if (str[7] != '-')
		return(0);

	if (str[8] == 'w')
		mode |= S_IWOTH;
	else if (str[8] != '-')
		return(0);

	switch (str[9]){
		case 'x':
			mode |= S_IXOTH;
			break;
		case 't':
			mode |= S_IXOTH;
			mode |= S_ISVTX;
			break;
		case 'T':
			mode |= S_ISVTX;
			break;
		case '-':
			break;
		default:
			return(0);
	}

	return (mode);
}


/* set_dac - sets permissions, uid, & gid of a file */
int
set_dac (sys_chk)
syscheck_t	*sys_chk;
{
	if ( access (sys_chk->path, 00) != 0 ){

		fprintf (stderr, "syscheck: Unable to access () file %s\n", 
			 sys_chk->path);
		exit_status = 1;
		return (-1);
	}

	else {

		if ( chmod (sys_chk->path, sys_chk->mode) == -1) {

			fprintf (stderr,"syscheck: failed to chmod file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}

		if ( chown ( sys_chk->path, sys_chk->uid, sys_chk->gid)  == -1){

			fprintf (stderr,"syscheck: failed to chown file %s\n",
				 sys_chk->path);
			exit_status = 1;
		}
	}

	return (0);
}


/* set_priv - sets appropriate privileges for a file */
int
set_priv (sys_chk)
syscheck_t	*sys_chk;
{
	unsigned int	req_priv = 0;

	if ( ! cvt_sym_to_priv (sys_chk->privs, &req_priv, ',') ){

		fprintf (stderr, "syscheck: unreadable privilege = %s\n",
			 sys_chk->privs);
	}

	else if ( set_file_priv (sys_chk->path, &req_priv) == -1) {

		fprintf (stderr, "syscheck: unable to set privilege on file %s\n", sys_chk->path);
	}

	else {

		return (0);
	}

	exit_status = 1;
	return (-1);
}


/* set_label - sets appropriate labels on a file */
int
set_label (sys_chk)
syscheck_t	*sys_chk;
{
	slabel_t	file_label;

	/* Initialize structure file_label */
	memset ( (char *)&file_label, 0, sizeof (slabel_t));

	if ( ! cvt_sym_slabel (sys_chk->label, &file_label) ) {
 
		fprintf (stderr, "syscheck: undeciperable label = %s\n", sys_chk->label);
		fprintf (stderr, "          file = %s\n", sys_chk->path);
	}

	else if ( set_file_slabel (sys_chk->path, &file_label) == -1 ) {

		fprintf (stderr, "syscheck: unable to set label on file %s\n",
			 sys_chk->path);
	}

	else {

		return (0);
	}

	exit_status = 1;
	return (-1);
}


int
load_sys_chk_struct ( sys_chk, data_line, mode )
syscheck_t	*sys_chk;
char		*data_line, *mode;
{
	int	args, uid, gid;

	args = sscanf (data_line,"%s %d %d %d %s %s %s", mode, &(sys_chk->size),
		       &uid, &gid, &sys_chk->label[0], &sys_chk->privs[0], 
		       &sys_chk->path[0] );

	/* Required because of a deficiency in sscanf, which doesn't
	   allow for shorts */
	sys_chk->uid = (uid_t)uid;
	sys_chk->gid = (gid_t)gid;

	return (args);
}


int
usage ()
{
	fprintf (stderr, "Usage: syscheck -c (checks system integrity, errors to stderr) < tcb_file\n");
	fprintf (stderr, "       syscheck -s[dlp] < tcb_file (sets DACs, labels, \n");
	fprintf (stderr, "                                      privileges)\n");
	fprintf (stderr, "       syscheck -g < old_tcb > new_tcb (resizes all files)\n");

	exit (1);
}
