/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) auditfmt.c: version 23.5 created on 7/18/91 at 11:15:39	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)auditfmt.c	23.5	7/18/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/* Security Package Command:
 *
 *	auditfmt -- format security trace file
 *
 *      Usage: auditfmt [ -RVSTLrvstlfi ] [-M map ] [-E eventnumber] [-m # bytes for arena 0] [ file ]
 *
 *	Auditfmt formats a security trace file so that it is readable by
 *	the security administrator.  The "-r" format is "raw", in which
 *	the basic information is displayed in blank separated fields, one
 *	line per record, suitable for use by awk, sort, join, sed, cut,
 *	paste, etc.  The "-v" format is default, the output is presented
 *	in verbose mode for easier visual inspection.  The "-s" format
 *	presents only information directly of interest to the security
 *	officer.  Default is the verbose ("-v") format.
 */


#include <stdio.h>
#include <time.h>
#include <termio.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <sys/tty.h>
#include <sys/utsname.h>
#include <sys/synch.h>
#include <sys/mls.h>
#include <sys/priv.h>
#include <sys/audit.h>

#define FALSE	0
#define TRUE	1
#define NAMSIZ 12
#define MEM_DEFAULT BUFSIZ*20      /* Default size for memory arena 0 */
#define NEG_ONE 0xffff
#define IPCMSG	49
#define IPCSHM	52
#define IPCSEM	53

/*MBM this is ulgy but it is there */
#define IFE(ch)	if ( (event_is_flagged)  ? \
		     (ch == flagged_event) : \
	             ( ( (ch < 32) ? \
		           (mask[0] & (1<<ch)) : \
		           (mask[1] & (1<<(ch-32))) )) )

#define AIFE(ch) if (((ch < 32)?(mask[0] & (1<<ch)) : (mask[1] & (1<<(ch-32)))))

#define MKUSER(A)	!strncmp((A), "MKUSER", 6)
#define MKGRP(A)	!strncmp((A), "MKGRP", 5)


sat_hdr_t	h;	/* audit trail header */

sat_fork_t	fb;	/* fork buffer */
sat_exec_t	eb;	/* exec buffer */
sat_exit_t	exb;	/* exit buffer */
sat_setuid_t	suidb;	/* setuid buffer */
sat_setgid_t	sgidb;	/* setgid buffer */
sat_setsid_t	ssidb;  /* set session id */
sat_setsupp_t	supgb;	/* supplemental group buffer */
sat_setrole_t	rolb;	/* process role change */
sat_priv_t	privb;	/* process privilege changes */
sat_label_t	lablb;	/* process label changes */
sat_chroot_t	chrtb;	/* process change root */
sat_ctty_t	cttyb;	/* change a process ctty */
sat_kill_t	killb;	/* send signal to process */

sat_access_t	accssb;	/* file access status */
sat_mknod_t	mkndb;	/* mknod/mkfifo  */
sat_pipe_t	pipeb;	/* create a pipe */
sat_create_t	creatb;	/* reserved for file creation events */
sat_link_t	linkb;	/* link new name to a file */
sat_rename_t	renamb;	/* file rename */
sat_unlink_t	unlnkb;	/* file unlink */
sat_chmod_t	chmodb;	/* change mode of a file */
sat_chown_t	chownb;	/* change owner of a file */
sat_chgrp_t	chgrpb;	/* change files group */
sat_filepriv_t	fprivb;	/* change effective privlige */
sat_filelabel_t	flablb;	/* change a files label */
sat_acl_t	aclb;	/* change files acl */

sat_mount_t	mountb;	/* mount */
sat_umount_t	umountb;	/* audit unmount request */
sat_syserr_t	serrb;	/* failed system call */
sat_clock_t	clckb;	/* system clock changes */

sat_ipccreat_t	ipcreatb;	/* ipcreate function */
sat_ipcaccess_t	ipcaccssb;	/* ipc access record */
sat_ipcrm_t	ipcrmb;		/* ipc remove function */
sat_ipclabel_t	ipclablb;	/* reserved for B1 */
sat_ipcdac_t 	ipcdacb;	/* change ipc mode owner group */
sat_acl_t	ipcaclb;	/* change ipc acl */

sat_login_t	loginb;		/* login/logout events */
sat_audit_t 	auditb;		/* audit control events */
sat_usrinfo_t	usrinfob;	/* Changes to user_auth */
sat_devinfo_t	devinfob;	/* Changes to dev_auth */
sat_lp_t	lpb;	/* line printer */
sat_cron_t	cronb;	/* cron */
sat_slink_t slinkb;	/*symbolic link*/
char	uucpb;	/* uucp */
char ulibuf[512];	/* buffer passed from User Level Interface */


int	ch, size, cpu, pid;     /* key attributes of record from header */
uint 	seq;
time_t 	event_time;

int 	*rcnt;      /* rcnt is the sequence counter for each event.  */
int fuid;				/* uid of flagged user */

time_t basetime; /* if >0, time of boot in seconds since 00:00:00 1/1/70 GMT */
time_t skiptime; /* if >0, lbolt format time to skip to before printing */

/* format flags */
int raw = 0;
int verbose = 0;	/* print verbose output */
int sensitive = 0;
int realtime = 1;	/* default mm/dd/yy hh:mm:ss format time */
int lbolttime = 0;
int userflagged = 0;	/* default is output for all users */
int event_is_flagged = 0;	/* default is output for all events */
int flagged_event;

char user[NAMSIZ+1];      /* name of user associated with current process */

char hdr[80]="", rawhdr[80]="";		/* static header string arrays used by 
								fmt_hdr et al */

int follow = 0;		/* default is NOT to keep waiting for file to grow */
int mask[2];		/* bits on enable printing of corresponding event */
int initial_mask[2];	/* events on when SAT file was created */



/* file for initializing tables  (default: stdin) */
FILE *mapf = stdin;		/* system mapping file */

FILE *ctrltty;			/* controling tty in interactive mode */
int interactive = 0;		/* default, not interactive mode */
struct termio tty_norm;		/* structure to hold starting tty settings */
struct termio tty_cbreak;	/* structure to hold cbreak-like tty settings */

/* Note - This structure is also defined in auditfmt.h */
struct session {
	ushort	nlink;		/* number of processes in this session */
	ushort	uid;		/* owner of this session */
	ushort  ttydev;		/* tty this session was started on */
	ushort	pid;		/* process group leader at login */
	ushort	state;		/* state of the session */
	time_t	tstart;		/* event start time */
	time_t	tfinish;	/* session finished time */
};

/* Note - This structure is also defined in auditfmt.h */
struct proc_entry {
	struct session *session_data;   /* common session data */
	priv_t priv;
	ushort euid, auid, ruid;	/* this process's user permissions */
	ushort egid, rgid;  		/* this process's group permissions */
	ushort role;
	ushort groups[NGROUPS_MAX];	/* Supplement groups */
};

/* session state flags */
#define S_NEW	0	/* forked from init, no exec yet */
#define S_GETTY	1	/* getty seen */
#define S_LOGIN	2	/* login seen */
#define S_START	4	/* started session */
#define S_STARTED 8	/* started session reported */
#define S_STOP	16	/* terminated session */

struct proc_entry *proc[MAXPID];  /* active procs indexed by pid */

char *ipc_ops[4] = {
	"message queue", "shared memory", "semaphore", "unknown ipc"
};

main(argc, argv)
int argc;
char *argv[];
{
 
int rt;
fpos_t posn;
	int c, errflg=0;
	int indx;
	char name[NAMSIZ+1];
	slabel_t	blank_label;
	extern char *optarg;
	extern int optind;
	char *devname;
	char *do_unmount();
	int memory_arena0;  /* Memory to allocated to arena 0 */
	void malloc_init();
	void validate_init();
	int result;

	blank_label.level = 0;
	blank_label.catlst[0] = '\0';

	memory_arena0 = MEM_DEFAULT;

	while ((c = getopt(argc, argv, "RVSTLrvstlfiM:N:E:m:")) != EOF)
		switch (c) {
		case 'R':		/* raw only mode */
			raw = 1;
			verbose = 0;
			sensitive = 0;
			break;
		case 'V':		/* verbose only mode */
			raw = 0;
			verbose = 1;
			sensitive = 0;
			break;
		case 'S':		/* sensitive only mode */
			raw = 0;
			verbose = 0;
			sensitive = 1;
			break;
		case 'T':		/* time in real time format only */
			realtime = 1;
			lbolttime = 0;
			break;
		case 'L':		/* time in lbolt format only */
			realtime = 0;
			lbolttime = 1;
			break;
		case 'r':
			raw ^= 1;
			break;
		case 'v':
			verbose ^= 1;
			break;
		case 's':
			sensitive ^= 1;
			break;
		case 't':
			realtime ^= 1;
			break;
		case 'l':
			lbolttime ^= 1;
			break;
		case 'f':
			follow ^= 1;
			break;
		case 'i':			/* try to go interactive */
			if ((ctrltty = fopen("/dev/tty", "rw")) == NULL) {
				fprintf(stderr,"auditfmt: cannot open /dev/tty\n");
				errflg++; 
			}
			else 
				if ( isatty(fileno(ctrltty)) ) {
					interactive_setup();
				} 
				else {
					fprintf(stderr,"auditfmt: in interactive mode stdout must be a tty\n");
					errflg++;
				}
			break;
		case 'm':			/* arena[0] memory allocation */
			memory_arena0 = atoi(optarg);
			if ( memory_arena0 <= 0) {
				fprintf(stderr,
				    "auditfmt: illegal value \"%s\" for -m option\n",
				    optarg);
				errflg++; 
			}
			break;
		case 'M':			/* alternate mapping file provided */
			if ((mapf = fopen(optarg, "r")) == NULL) {
				fprintf(stderr,"auditfmt: cannot open %s\n",optarg);
				errflg++; 
			}
			break;
		case 'N':			/* save Name of suspect user */
			userflagged = 1;
			strncpy(name, optarg, NAMSIZ);
			break;
		case 'E':			/* save number of event */
			event_is_flagged = 1;
			interactive = 0;        /* sorry can't be both */
			if (((flagged_event = atoi(optarg)) >= AUD_MAX_DEF)
			   || (flagged_event < 0)){
				fprintf(stderr,"auditfmt: Bad event number %d\n"
				   ,flagged_event);
				errflg++; 
			}
			break;
		default:
			errflg++;
		}
	if ( argv[optind] != NULL )
		if (freopen(argv[optind], "r", stdin) == NULL ) {
			fprintf(stderr,"auditfmt: cannot open %s\n",argv[optind]);
			errflg++; 
		}
	if ( errflg ) {
		fprintf(stderr,"Usage: auditfmt [ -RVSTLrvstlfi ] [-M map ] [-E event number] [-m # bytes for arena 0] [ file ]\n");
		get_out(2);
	}

	/*
		if no options are used - default to verbose.
	*/

	if ( (argc >= 2) && ( !raw ) && 
	    ( !sensitive ) && ( !verbose ) ){
		verbose = 1;
	}

	malloc_init(memory_arena0);
	validate_init();

	rcnt = (int *) s_malloc((sizeof(int) * AUD_MAX_EVT),1);

	initialize();	/* read init files to set up number -> name tables */


 	if (userflagged)     /* tracing a particular user -- get his number */
 		if ((fuid = getuid(name)) <= -1) {		/* not found */
 			fprintf(stderr, "auditfmt: no user with name %s\n"
			        , name);
 			get_out(3); 
 		}
	else 
		printf("Selecting records for user <%s> only\n",name);

	if (event_is_flagged)
		printf("Selecting records for event %d only\n",flagged_event);
		
 
	/* skip any records left over form previous SAT */
 
	result = fread((char *)&h, 1, sizeof(h), stdin);
	if ( result == 0 )
	{
		printf("no audit records!\n");
		exit();
	}
 
 	if (h.event != AUD_AUDIT )
 		skip_to_SAT_start();
 	
	fread((char *)&auditb, 1, h.size, stdin);
 	checksequence(h.event, (uint) h.seq);
	for (indx=0; indx < AUD_MAX_EVT; indx++)
		rcnt[indx] = -1 ;

 
 
 	event_time = h.time;
	basetime = 0;    /*basetime is set in the first AUD_CLOCK record */
 	mask[0] = initial_mask[0] = auditb.mask[0];
 	mask[1] = initial_mask[1] = auditb.mask[1];
 
 	/* print events recorded in Audit Trail */
 
	if (raw)
 		printf(" %o %o\n",initial_mask[1], initial_mask[0]);
		
	if (verbose || sensitive) { 
		printf("Events recorded in Audit Trail: ");
 		for ( indx = 0; indx < AUD_MAX_DEF; indx++ )
 			AIFE(indx) printf("%d ", indx);
		printf("\n");
	}

 	/* checkpoint initial data structures to facilitate restarts */

 	if ( interactive ) {
 		init_chkpt_files();
 		checkpoint();
 	}
 
retry:
	fgetpos(stdin, &posn);
	rt = fread((char *)&h, 1, sizeof(h), stdin);
	if(rt != sizeof(h)) {
		if (follow) {
			fsetpos(stdin, &posn);
			goto retry;
		}
	}
	else
	{
insync:
		if (!validate(&h))
			re_sync(&h);

		ch   = h.event;
		size = h.size;
		pid  = h.pid;
		event_time = h.time;
		seq = (uint) h.seq;
		cpu = h.cpu;

		checkskiptime(event_time);
		fmt_hdr(ch,size,event_time,cpu,seq);

		if ( (proc[pid] == NULL) && (ch != AUD_FORK) ){
			makeup_session(pid,0);
		}

		switch ( ch ) {

		case AUD_FORK :
			fread((char *)&fb, 1, size, stdin);
			checksequence(ch, seq);
			if ( fb.ppid == 1 ){		/* if forked from init */
				create_session(&fb);
			}
			else {
				if (proc[fb.ppid] == NULL )
					makeup_session(fb.ppid,0);
				link_session(&fb);
			}
			IFE(ch) print_fork(&fb);
			break;
		case AUD_EXEC :
			fread((char *)&eb, 1, size, stdin);
			checksequence(ch, seq);
			update_session(&eb);
			IFE(ch) print_exec(&eb);
			if ( procstate(S_START))
				/* record that this login has been reported */
				proc[pid]->session_data->state = S_STARTED;
			break;
		case AUD_EXIT :
			fread((char *)&exb, 1, size, stdin);
			checksequence(ch, seq);
			if ((exb.status&0377) != 0177 && proc[pid] != NULL )
				stop_session();	/* if real exit, not stop */
			IFE(ch) print_exit(&exb);
			if ((exb.status&0377) != 0177 && proc[pid] != NULL )
				free_session();	/* if real exit, not stop */
			break;
		case AUD_SETUID :
			fread((char *)&suidb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_setuid(&suidb);
			proc[pid]->euid = suidb.euid;
			proc[pid]->ruid = suidb.ruid;
			proc[pid]->auid = suidb.auid;
			break;
		case AUD_SETGID :
			fread((char *)&sgidb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_setgid(&sgidb);
			proc[pid]->egid = sgidb.egid;
			proc[pid]->rgid = sgidb.rgid;
			break;
		case AUD_SETSID :
			fread((char *)&ssidb, 1, size, stdin);
			checksequence(ch, seq);
			/*MBM  make printsession print a goodby for the session if if this is the
  last process in the session start_own_session will remove it if nessary*/
			IFE(ch) print_setsid(&ssidb);
			start_own_session(&ssidb);
			break;
		case AUD_SUPPL :
			fread((char *)&supgb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_supp(&supgb);
			for(indx=0;(indx < NGROUPS_MAX) && 
			    ((ushort) supgb.groups[indx] != NEG_ONE); indx++)
				proc[pid]->groups[indx] = supgb.groups[indx];
			if (indx < NGROUPS_MAX) 
				proc[pid]->groups[indx] = NEG_ONE;
			break;
		case AUD_ROLE :
			fread((char *)&rolb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_setrole(&rolb);
			proc[pid]->role = rolb.new_role;
			break;
		case AUD_PRIV :
			fread((char *)&privb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_priv(&privb);
			proc[pid]->priv.real = privb.priv.real;
			proc[pid]->priv.eff = privb.priv.eff;
			proc[pid]->priv.mask = privb.priv.mask;
			break;
		case AUD_LABEL :
			fread((char *)&lablb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_label(&lablb);
			break;
		case AUD_CHROOT :
			fread((char *)&chrtb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_chroot(&chrtb);
			break;
		case AUD_CTTY :
			fread((char *)&cttyb, 1, size, stdin);
			checksequence(ch, seq);
			if( cttyb.sessid != proc[pid]->session_data->pid)
				printf("ERR: bad session %d for process %d session %d\n"
				    , cttyb.sessid, pid, proc[pid]->session_data->pid);
			else
				proc[pid]->session_data->ttydev = cttyb.ctty;
			IFE(ch) print_ctty(&cttyb);
			break;
		case AUD_KILL :
			fread((char *)&killb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_kill(&killb);
			break;

		case AUD_ACCSR :
		case AUD_ACCSW :
		case AUD_ACCFR :
		case AUD_ACCFW :
			fread((char *)&accssb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_access(&accssb, ch);
			break;

		case AUD_MKNOD :
			fread((char *)&mkndb, 1, size, stdin);
			checksequence(ch, seq);
			file_mknd(mkndb.dev,mkndb.ino,mkndb.dino,mkndb.name,0,&blank_label);
			IFE(ch) print_mknod(&mkndb);
			break;
		case AUD_PIPE :
			fread((char *)&pipeb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_pipe(&pipeb);
			break;
		case AUD_CREAT :
			fread((char *)&creatb, 1, size, stdin);
			checksequence(ch, seq);
			file_build(creatb.dev,creatb.ino,creatb.dino,creatb.name,0,
 			           &blank_label,0,"");
			IFE(ch) print_create(&creatb);
			break;
		case AUD_LINK :
			fread((char *)&linkb, 1, size, stdin);
			checksequence(ch, seq);
			file_link(linkb.dev,linkb.ino,linkb.dino,linkb.name,0,&linkb.label);
			IFE(ch) print_link(&linkb);
			break;
		case AUD_RENAME :
			fread((char *)&renamb, 1, size, stdin);
			checksequence(ch, seq);
			file_rename(renamb.dev,renamb.ino,renamb.dino,renamb.name,&blank_label);
			IFE(ch) print_rename(&renamb);
			break;
		case AUD_UNLINK :
			fread((char *)&unlnkb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_unlink(&unlnkb);
			file_ulrm(unlnkb.dev,unlnkb.ino,unlnkb.dino,unlnkb.name);
			break;
		case AUD_CHMOD :
			fread((char *)&chmodb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_chmod(&chmodb);
			break;
		case AUD_CHOWN :
			fread((char *)&chownb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_chown(&chownb);
			break;
		case AUD_CHGRP :
			fread((char *)&chgrpb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_chgrp(&chgrpb);
			break;
		case AUD_FILEPRIV :
			fread((char *)&fprivb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_filepriv(&fprivb);
			file_priv(&fprivb);
			break;
		case AUD_FILELABEL :
			fread((char *)&flablb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_filelabel(&flablb);
			file_label(&flablb);
			break;
		case AUD_ACL :
			fread((char *)&aclb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_acl(&aclb);
			break;


		case AUD_MOUNT :
			fread((char *)&mountb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_mount(&mountb);
			break;
		case AUD_UMOUNT :
			fread((char *)&umountb, 1, size, stdin);
			checksequence(ch, seq);
			devname = do_unmount((int)umountb.fsdev,
			    (int)umountb.mptdev,(int)umountb.mptptino);
			IFE(ch) print_unmount(devname, &umountb);
			s_free(devname, 1);
			break;
		case AUD_SYSERR :
			fread((char *)&serrb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_syserr(&serrb);
			break;
		case AUD_CLOCK :
			fread((char *)&clckb, 1, size, stdin);
			checksequence(ch, seq);

	/*If basetime == 0 then we are at the top of the sat file
	  so we don't want to print the record, just set the time*/

			if ( basetime == 0 ) 
				basetime = clckb.time - event_time/HZ;
			else {
				checkbasetime(event_time, clckb.time);
				IFE(ch) print_clock(&clckb);
			}
			break;



		case AUD_IPCREATE :
			fread((char *)&ipcreatb, 1, size, stdin);
			checksequence(ch, seq);
			add_key(&ipcreatb,proc[pid]->egid);
			IFE(ch) print_ipccreat(&ipcreatb);
			break;
		case AUD_IPCACCESS :
			fread((char *)&ipcaccssb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_ipcaccs(&ipcaccssb);
			break;
		case AUD_IPCRM :
			fread((char *)&ipcrmb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_ipcrm(&ipcrmb);
			rm_key(&ipcrmb);
			break;
		case AUD_IPCLABEL :
			fread((char *)&ipclablb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_ipclabel(&ipclablb);
			break;
		case AUD_IPCDAC :
			fread((char *)&ipcdacb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_ipcdac(&ipcdacb);
			break;
		case AUD_IPCACL :
			fread((char *)&ipcaclb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_acl(&ipcaclb);
			break;


		case AUD_LOGIN :
			fread((char *)&loginb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_login(&loginb);
			break;
		case AUD_AUDIT :
			fread((char *)&auditb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_audit(&auditb);
			if  (auditb.mode == AUDMASK){
				mask[0] = auditb.mask[0];
				mask[1] = auditb.mask[1];
 	printf("mask[1] - %o, mask[0] - %o\n",initial_mask[1], initial_mask[0]);
			}
			/* clear all sequence nums on a start */
			else if (auditb.mode == AUDSTART)
				for (indx=0; indx < AUD_MAX_EVT; indx++)
					rcnt[indx] = -1 ;
			break;
		case AUD_USRINFO :
			fread((char *)&usrinfob, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_usrinfo(&usrinfob);
			break;
		case AUD_DEVINFO :
			fread((char *)&devinfob, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_devinfo(&devinfob);
			break;
		case AUD_ULI :
			fread((char *)ulibuf, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_uli(ulibuf);
			break;
		case AUD_LP :
			fread((char *)&lpb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_lp(&lpb);
			break;
		case AUD_CRON :
			fread((char *)&cronb, 1, size, stdin);
			checksequence(ch, seq);
			IFE(ch) print_cron(&cronb);
			break;
		case AUD_SLINK :
			fread((char *)&slinkb, 1, size, stdin);
			checksequence(ch, seq);
			file_build(slinkb.dev,slinkb.ino,slinkb.dino,slinkb.name,0,
			         slinkb.label,slinkb.path_length,slinkb.target);
			IFE(ch) print_slink(&slinkb);
			break;
		/*MBM  these are all new and undefined */
		case AUD_UUCP :
		case AUD_TCP :
		case AUD_NFS :
		case AUD_RFS :
		default:
			re_sync(&h); 
			goto insync; 
		}
		if ( interactive ) checkcmds();
		goto retry;
	}

	if ( follow ) {
		sleep(1);
		goto retry;
	}
	fprintf(stderr, "End-Of-File\n");
	get_out(0);
}

checksequence(ch,seq)
int ch;
uint seq;
{
	register int *p = &rcnt[ch];
	if (*p == -1) *p=seq;
	if (*p != seq) {
		printf("%s SECURITY TRACE RECORDS LOST!  (%d records on event %d)\n", hdr, seq - *p, ch);
		*p=seq;
	}
	*p=(++(*p))&0377;
}

checkbasetime(lbolt, time)
time_t lbolt;		/* time since boot in ticks (1/HZth of a second) */
time_t time;		/* time in seconds since 00:00:00 1/1/70 */
{
	time_t difference = time-basetime-lbolt/HZ;
	if (difference > 1 || difference < -1 )
	{	/* system clock reset!!! */
		printf("%sCLOCK CHANGED!  (%ld seconds difference)\n", hdr, difference);
		basetime = time-lbolt/HZ;
	}
}

int mask_save[2];  /* needs to be outside of checkskiptime so that status
			reporter and event toggle can access */
checkskiptime(lt)
time_t lt;
{
	if ( skiptime == 0 ) return;	/* no skip requested */
	if ( (mask_save[0] == 0)  &&
	     (mask_save[1] == 0) ) {	/* if skip not started, start it... */
		mask_save[0] = mask[0];	/* ...by saving current mask, */
		mask_save[1] = mask[1];	/* ...by saving current mask, */
		mask[0] = 0;		/* and disabling all printing */
		mask[1] = 0;		/* and disabling all printing */
	}
	if ( skiptime <= lt ) {		/* if skip finished... */
		mask[0] = mask_save[0];	/* restore printing */
		mask[1] = mask_save[1];	/* restore printing */
		mask_save[0] = 0;		/* prepare for next time */
		mask_save[1] = 0;		/* prepare for next time */
		skiptime = 0;
	}
}


char *flag(flg)
int flg;
{
	switch (flg) {
	case 2:
	case 3:
		return("used ROOT permission");
	}
	return("");
}

/*
 * fmt_hdr ( int ch, int size, time_t ltime, int cpu, uint seq )
 *
 * format headers for use by print functions
 */

fmt_hdr(ch, size, ltime, cpu, seq)
int ch,size,cpu;
uint seq;
time_t ltime;
{
	register char *p;

	if ( raw )
	{
		p = rawhdr;
		p += sprintf(p, "%d %d %lu %d %u ", ch, size, ltime, cpu, seq);
	}
	if ( verbose || sensitive )
	{
		p = hdr;
		if ( lbolttime  ){

			register int t,hr,min,sec;

			t = ltime;
			hr=t/(3600*HZ); 
			t=t-(3600*HZ)*hr;
			min=t/(60*HZ); 
			t=t-(60*HZ)*min;
			sec=t/HZ; 
			t=t-HZ*sec;
			p += sprintf(p, "Time : %.2d:%.2d:%.2d.%.2d ", 
			    hr, min, sec, (int) t);

		}
		if ( realtime && basetime ){
			time_t time;
			struct tm *tm;
			char buf[100];

			time = (ltime/HZ) + basetime;
			tm = localtime(&time);

			ascftime(buf,"%a %b %d %T %Y",tm);

			p += sprintf(p, "Date/Time : %s",buf);

		}
	}
}

initialize()	/* sets up the data structures used to map numbers to names */
{
	int version, c;

	if (fread(&version,4,1,mapf) != 1){
		perror("Auditfmt: ");
		exit (1);
	}
	
	if (version != AUDIT_TRAIL_VERSION ){
		fprintf(stderr,"Wrong version of auditfmt for this audit trail\n");
		fprintf(stderr,"auditfmt version = %d, audit trail version = %d\n",
			AUDIT_TRAIL_VERSION,version);
		exit (1);
	}

	date_sys();  /*check the date, and system name*/

	initkeys();
	skipto("users:",mapf);
	username_init(mapf);
	skipto("roles:",mapf);
	roles_init(mapf);
	skipto("groups:",mapf);
	gname_init(mapf);
	skipto("ttys:",mapf);
	ttyname_init(mapf);
	skipto("filesystem:",mapf);
	fseek(mapf, (long) -sizeof("filesystem:"), 1); /* back over marker */
	filename_init(mapf);
	skipto("data:\n",stdin);
}

date_sys()
{
	char date[30];
	struct utsname nm;

	fread(date,30,1,mapf);
	printf("AUDIT DATE: %s\n",date);

	fscanf(mapf,"%s %s %s %s %s\n",nm.sysname, nm.nodename, 
	    nm.release, nm.version, nm.machine);
	printf("AUDIT SYSTEM:%s %s %s %s %s\n"
	    ,nm.sysname, nm.nodename, nm.release, nm.version, nm.machine);
}


skipto(str,f) /* skip in stream "f" to position right after marker "str" */
char *str;
FILE *f;
{
	static char buf[80];
	int l = strlen(str);
	while ((fgets(buf,l+1,f) != NULL) &&
	    strncmp(str, buf, l)) /* skip to marker */;
	if (strncmp(str, buf, l)) {
		fprintf(stderr, "skipto: marker '%s' not found\n", str);
		get_out(1);
	}
	return 0;
}

skip_to_SAT_start()
{
	int indx;
	char *supp_groups;

	if(!validate(&h))
		re_sync(&h);

	while (h.event != AUD_AUDIT) {
		size = h.size;
		pid = h.pid;
		seq = h.seq;
		cpu = h.cpu;
		sprintf(rawhdr, "%d %d %lu %d %u ", h.event, size, h.time,cpu,seq);
		fprintf(stderr, "AUD record left over from previous audit trail:\n\t");

		switch ( h.event ) {
		case AUD_FORK :
			fread((char *)&fb, 1, size, stdin);
			fprintf(stderr, "%s%d %d\n", rawhdr, pid, (int) fb.ppid);
			break;
		case AUD_EXEC :
			fread((char *)&eb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %u %u %u %u %u %u %s\n", rawhdr,
			    pid, (int) eb.ctty,
			    (int) eb.ruid, (int) eb.rgid, (int) eb.euid,
			    (int) eb.egid, (int) eb.dev, (int) eb.ino,
			    eb.name);
			break;
		case AUD_EXIT :
			fread((char *)&exb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %d\n", rawhdr,
			    pid,
			    (int) exb.flag, (int) exb.status);
			break;
		case AUD_SETUID :
			fread((char *)&suidb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u\n", rawhdr,
			    pid,
			    (uint) suidb.ruid, (uint) suidb.euid);
			break;
		case AUD_SETGID :
			fread((char *)&sgidb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u\n", rawhdr,
			    pid,
			    (uint) sgidb.rgid, (uint) sgidb.egid);
			break;
		case AUD_SETSID :
			fread((char *)&ssidb, 1, size, stdin);
			fprintf(stderr, "%s %d %o %u %u %u % \n", rawhdr, pid, 
			    (uint)ssidb.mode, (uint)ssidb.old_sessid,  
			    (uint)ssidb.old_pgrp, (int)ssidb.old_ctty, 
			    (int) ssidb.new_ctty);
			break;
		case AUD_SUPPL :
			fread((char *)&supgb, 1, size, stdin);
			{

				for(indx=0;(indx < NGROUPS_MAX) && 
				    ((ushort)supgb.groups[indx] != NEG_ONE);indx++){
					supp_groups += sprintf(supp_groups,"%d",supgb.groups[indx]);
					if((indx+1 < NGROUPS_MAX) && 
					    ((ushort)supgb.groups[indx+1] != NEG_ONE))
						supp_groups += sprintf(supp_groups, ",");
					else
						supp_groups += sprintf(supp_groups, "");
				}
				fprintf(stderr, "%s %d %s\n", rawhdr, pid, supp_groups);
			}
			break;
		case AUD_ROLE :
			fread((char *)&rolb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u %u %u\n", rawhdr, pid,
			    (uint) rolb.new_role, (uint) (rolb.priv).real, 
			    (uint) (rolb.priv).eff, (uint) (rolb.priv).mask);
			break;
		case AUD_PRIV :
			fread((char *)&privb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u %u\n", rawhdr, pid,
			    (uint) privb.priv.real,
			    (uint) privb.priv.eff,
			    (uint) privb.priv.mask);
			break;
		case AUD_LABEL :
			fread((char *)&lablb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %s %u %s\n", rawhdr, pid,
			    (uint)(lablb.minlabel).level,(lablb.minlabel).catlst,
			    (uint)(lablb.maxlabel).level,(lablb.maxlabel).catlst);
			break;
		case AUD_CHROOT :
			fread((char *)&chrtb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u\n", rawhdr, pid,
			    (uint) chrtb.dev,
			    (uint) chrtb.ino);
			break;
		case AUD_CTTY :
			fread((char *)&cttyb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %d\n", rawhdr, pid,
			    (uint)cttyb.sessid, (int)cttyb.ctty);
			break;
		case AUD_KILL :
			fread((char *)&killb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u\n", rawhdr, pid,
			    (uint) killb.from_pid, (uint) killb.signal);
			break;


		case AUD_ACCSR :
		case AUD_ACCSW :
		case AUD_ACCFR :
		case AUD_ACCFW :
			fread((char *)&accssb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %u\n", rawhdr,
			    pid, accssb.mode, accssb.type,
			    (int) accssb.dev, (int) accssb.ino);
			break;
		case AUD_MKNOD :
			fread((char *)&mkndb, 1, size, stdin);
			fprintf(stderr, "%s%d %o %u %u %u %.14s\n", rawhdr,
			    pid, (int) mkndb.mode,
			    (int) mkndb.dev, (int) mkndb.ino,
			    (int) mkndb.dino, mkndb.name);
			break;
		case AUD_PIPE :
			fread((char *)&pipeb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u\n", rawhdr, pid,
			    (uint) pipeb.dev, (uint) pipeb.ino);
			break;
		case AUD_CREAT :
			fread((char *)&creatb, 1, size, stdin);
			break;
		case AUD_LINK :
			fread((char *)&linkb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %.14s\n", rawhdr,
			    pid, (int) linkb.dev, 
			    (int) linkb.ino, (int) linkb.dino, linkb.name);
			break;
		case AUD_RENAME :
			fread((char *)&renamb, 1, size, stdin);
			break;
		case AUD_UNLINK :
			fread((char *)&unlnkb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %.14s\n", rawhdr, pid,
			    (int) unlnkb.dev, (int) unlnkb.ino,
			    (int) unlnkb.dino, unlnkb.name);
			break;
		case AUD_CHMOD :
			fread((char *)&chmodb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %u %u\n", rawhdr,
			    pid, (int) chmodb.dev, (int) chmodb.ino,
			    (int) chmodb.old_mode, (int) chmodb.new_mode);
			break;
		case AUD_CHOWN :
			fread((char *)&chownb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %u %u\n", rawhdr,
			    pid, (int) chownb.dev, (int) chownb.ino, 
			    (int) chownb.old_uid, (int) chownb.new_uid);
			break;
		case AUD_CHGRP :
			fread((char *)&chgrpb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %u\n", rawhdr, pid,
			    (int) chgrpb.dev, (int) chgrpb.ino,
			    (int) chgrpb.old_gid, (int) chgrpb.new_gid);
			break;
		case AUD_FILEPRIV :
			fread((char *)&fprivb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %u %u %u\n", rawhdr, pid,
			    (int)fprivb.dev, (uint)fprivb.ino, fprivb.old_priv,
			    fprivb.new_priv);
			break;
		case AUD_FILELABEL :
			fread((char *)&flablb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %u %u %s %u %s\n", rawhdr, pid,
			    (int)flablb.dev, (uint)flablb.ino, 
			    (flablb.old_label).level, (flablb.old_label).catlst,
			    (flablb.new_label).level, (flablb.new_label).catlst);
			break;
		case AUD_ACL :
			fread((char *)&aclb, 1, size, stdin);
			/*MBM  finish the acl stuff */
			fprintf(stderr, "%s%d %u Problem with ACl record\n", 
			    rawhdr, pid, aclb.aino);
			break;


		case AUD_MOUNT :
			fread((char *)&mountb, 1, size, stdin);
			fprintf(stderr, "%s %d %d %d %d %d %d\n", rawhdr, pid,
			    (int) mountb.mtptdev, (int) mountb.mtptino,
			    (int) mountb.fsdev, (int) mountb.fsrootdev, 
			    (int) mountb.fsrootino);
			break;
		case AUD_UMOUNT :
			fread((char *)&umountb, 1, size, stdin);
			fprintf(stderr, "%s %d %d %d %d %d %d\n", rawhdr,pid,
			    (int) mountb.mtptdev, (int) mountb.mtptino,
			    (int) mountb.fsdev, (int) mountb.fsrootdev, 
			    (int) mountb.fsrootino);
			break;
		case AUD_SYSERR :
			fread((char *)&serrb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %d\n", rawhdr, pid,
			    (int) serrb.sysno, (int) serrb.errno);
			break;
		case AUD_CLOCK :
			fread((char *)&clckb, 1, size, stdin);
			fprintf(stderr, "%s%d %lu\n", rawhdr, pid, clckb.time);
			break;



		case AUD_IPCREATE :
			fread((char *)&ipcreatb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %d %d %d %o\n", rawhdr,
			    pid, ipcreatb.id, ipcreatb.key, ipcreatb.uid,
			    (int) ipcreatb.sysno, (int) ipcreatb.mode);
			break;
		case AUD_IPCACCESS :
			fread((char *)&ipcaccssb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %d %d\n", rawhdr,
			    pid, ipcaccssb.id,
			    (int) ipcaccssb.sysno, (int) ipcaccssb.mode);
			break;
		case AUD_IPCRM :
			fread((char *)&ipcrmb, 1, size, stdin);
			fprintf(stderr, "%s%d %d %d\n", rawhdr, 
			    pid, ipcrmb.id, (int) ipcrmb.sysno);
			break;
		case AUD_IPCLABEL :
			fread((char *)&ipclablb, 1, size, stdin);
			break;
		case AUD_IPCDAC :
			fread((char *)&ipcdacb, 1, size, stdin);
			fprintf(stderr, "%s%d %u %u %u %u\n", rawhdr,
			    pid, (int) ipcdacb.id, (int) ipcdacb.sysno,
			    (int) ipcdacb.old_uid, (int) ipcdacb.new_uid,
			    (int) ipcdacb.old_gid, (int) ipcdacb.new_gid,
			    (int) ipcdacb.old_mode, (int) ipcdacb.new_mode);
			break;
		case AUD_IPCACL :
			fread((char *)&ipcaclb, 1, size, stdin);
			fprintf(stderr, "%s%d %u Problem with ACl record\n", 
			    rawhdr, pid,ipcaclb.aino);
			break;


		case AUD_LOGIN :
			fread((char *)&loginb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %d %s %u \n", rawhdr,pid,
			    loginb.mode, (int) loginb.tty, loginb.uname, 
			    loginb.flag);
			break;
		case AUD_AUDIT :
			fread((char *)&auditb, 1, size, stdin);
			fprintf(stderr, "%s %d %u %u %u %u\n", rawhdr,pid,
			    auditb.mode, auditb.version, auditb.mask[0],
			    auditb.mask[1]);
			break;
		case AUD_USRINFO :
			fread((char *)&usrinfob, 1, size, stdin);
			fprintf(stderr, "%s %u %u %s\n", rawhdr,pid,
			    usrinfob.mode, usrinfob.uname);
			break;
		case AUD_DEVINFO :
			fread((char *)&devinfob, 1, size, stdin);
			fprintf(stderr, "%s %u %u %s\n", rawhdr,pid,
			    devinfob.mode, devinfob.ttyn);
			break;
		case AUD_ULI :
			fread((char *)ulibuf, 1, size, stdin);
			fprintf(stderr, "%s %d %s\n", rawhdr,pid,ulibuf);
			break;
		case AUD_LP :
			fread((char *)&lpb, 1, size, stdin);
			fprintf(stderr, "%s %u %u %u %u %d %u %d %u %s\n", 
				rawhdr,pid,lpb.mode, lpb.uid, lpb.gid, 
				(int)lpb.file_dev, lpb.file_ino, 
				(int)lpb.pr_dev, lpb.pr_ino, lpb.pname);
			break;
		case AUD_CRON :
			fread((char *)&cronb, 1, size, stdin);
			fprintf(stderr, "%s %d %d %u\n", rawhdr,pid,
				cronb.mode, cronb.uid, (int) cronb.file_dev,
				cronb.file_ino);
			break;
		case AUD_SLINK :
			fread((char *)&slinkb, 1, size, stdin);
			break;
		case AUD_UUCP :
			fread((char *)&uucpb, 1, size, stdin);
			fprintf(stderr, "%s %d %s\n", rawhdr,pid,&uucpb);
			break;
		case AUD_TCP :
		case AUD_NFS :
		case AUD_RFS :
			fread(ulibuf, 1, size, stdin);
			fprintf(stderr, "%s%d %d %.*s\n", rawhdr,
			    (int) ulibuf[0], pid, size - 1,  &ulibuf[1]);
			break;

		default:
			if(re_sync(&h))
				continue;
		}
		fread((char *) &h, 1, sizeof(h), stdin);
		if (!validate(&h))
			re_sync(&h);
	}
}

get_out(rc)
int rc;
{
	time_t ltime;
	struct tm *tm;

	if ( interactive ) {   /* clean up the tty */
		if (ioctl(fileno(ctrltty),TCSETA,&tty_norm) == -1){
			perror("Auditfmt:");
			exit(1);
		}
	}
	if (event_time > 0) {
		ltime = (event_time/HZ) + basetime;
		tm = localtime(&ltime);
		printf("exiting at SAT time %.2d/%.2d/%.2d %.2d:%.2d:%.2d\n",
		    tm->tm_mon+1, tm->tm_mday, tm->tm_year,
		    tm->tm_hour, tm->tm_min, tm->tm_sec);
	}
	exit(rc);
}



/* start_own_session(
   This routine is used when the process changed sessions or process groups.
   The process no longer has a controling tty at this point.  The controling 
   tty will be set when the process opens a tty, and a AUD_CTTY record is 
   written*/

start_own_session(sid)
sat_setsid_t sid;
{
	register struct session *p;

	p = proc[pid]->session_data;

	if (!((p->nlink)--)){	/*if we were the last process in the sess*/
		s_free(p,1);	/*free up the old session */

	}

	proc[pid]->session_data = p = (struct session *) alloc(sizeof(struct session));
	p->nlink=1;
	p->uid = proc[pid]->ruid;
	p->ttydev = NULL;         /*not ctty at this time*/
	p->pid = pid;             /*pid = session id*/
	p->state = S_STARTED;
	p->tstart = event_time;

}

create_session(fb)
sat_fork_t *fb;
{
	register struct session *p;
	register struct proc_entry *q;


	if ( proc[pid] != NULL ) {
		warn("create_session: reusing uncleared pid");
		s_free(proc[pid], 1);
	}

	q = proc[pid] = (struct proc_entry *) alloc(sizeof(struct proc_entry));
	q->euid = q->ruid = q->egid = q->rgid = 0;

	q->auid = 0;
	q->groups[0] = NEG_ONE;

	q->priv.real = 0x0;
	q->priv.eff = 0x0;
	q->priv.mask = 0;

	p = (struct session *) alloc(sizeof(struct session));
	proc[pid]->session_data = p;

	p->pid = pid;
	p->tstart = event_time;
	p->tfinish = 0;
	p->uid = 0;
	p->state = S_NEW;
	p->nlink = 1;

}

update_session(eb)
sat_exec_t *eb;
{
	register struct session *p;
	int indx;

	if ( (proc[pid] != NULL) && 
	    ((p = (proc[pid]->session_data)) != NULL) ) {
		switch ( p->state )
		{
		case S_NEW:
			if (( strncmp(eb->name,"getty",DIRSIZ) == 0 ) ||
			    ( strncmp(eb->name,"uugetty",DIRSIZ) == 0))
				p->state = S_GETTY;
			break;
		case S_GETTY:
			if ( strncmp(eb->name,"login",DIRSIZ) == 0 )
			{
				p->ttydev = eb->ctty;
				p->state = S_LOGIN;
			}
			break;
		case S_LOGIN:
			if ( eb->ruid != 0 || strncmp(eb->name,"sh",DIRSIZ))
			{
				p->uid = eb->ruid;
				p->tstart = event_time;
				p->state = S_START;

				proc[pid]->auid = eb->auid;
				proc[pid]->priv.real = eb->priv.real;
				proc[pid]->priv.eff = eb->priv.eff;
				proc[pid]->priv.mask = eb->priv.mask;
			}
			break;
		case S_STARTED:
			proc[pid]->auid = eb->auid;
			proc[pid]->priv.real = eb->priv.real;
			proc[pid]->priv.eff = eb->priv.eff;
			proc[pid]->priv.mask = eb->priv.mask;
			break;

		}
		proc[pid]->euid = eb->euid;
		proc[pid]->ruid = eb->ruid;
		proc[pid]->egid = eb->egid;
		proc[pid]->rgid = eb->rgid;

		for(indx=0;indx < NGROUPS_MAX;indx++){
			proc[pid]->groups[indx] = eb->groups[indx];
		}

	}
}

link_session(fb)
sat_fork_t *fb;
{					/* forked from other session */
	register struct proc_entry *p,*q;
	int indx;

	if ( proc[pid] != NULL ) {
		warn("link_session: reusing uncleared pid");
		s_free(proc[pid], 1);
	}
	p = proc[pid]
	    = (struct proc_entry *) alloc(sizeof(struct proc_entry));
	q = proc[fb->ppid];
	p->session_data = q->session_data;
	q->session_data->nlink++;
	p->euid = q->euid;
	p->ruid = q->ruid;
	p->egid = q->egid;
	p->rgid = q->rgid;

	p->auid = q->auid;
	p->priv.real = 0;
	p->priv.eff = 0;
	p->priv.mask = 0;
	for(indx=0;indx < NGROUPS_MAX;indx++){
		p->groups[indx] = q->groups[indx];
	}
}


stop_session()
{
	if ( proc[pid] && (proc[pid]->session_data->pid == pid) )
		/* if pgrp leader */
		if ( procstate(S_STARTED) )	/* and session started */
		{
			proc[pid]->session_data->tfinish = event_time;
			proc[pid]->session_data->state = S_STOP;
		}
}

free_session()
{
	if ( proc[pid] ) {
		--(proc[pid]->session_data->nlink);
		if ( proc[pid]->session_data->nlink == 0 ){
			s_free(proc[pid]->session_data, 1);
		}
		s_free(proc[pid], 1);
		proc[pid] = NULL;
	}
}

/* make up session data for a session which should exist, but does
   not.  This can happen, for instance, at the beginning of the
   audit trail. */

makeup_session(pid,auid)
int pid;
int auid;
{
	register struct proc_entry *p;
	register struct session *r;
	int indx;

	if ( proc[pid] != NULL ) {
		warn("makeup_session: session data already exists");
		return;
	}
	p = proc[pid]
	    = (struct proc_entry *) alloc(sizeof(struct proc_entry));

	r = p->session_data =
	    (struct session *)alloc(sizeof(struct session));

	r->pid = pid;
	r->tstart = event_time;
	r->tfinish = 0;
	r->uid = 0;
	r->state = S_STARTED;
	r->nlink = 1;
	p->euid = p->ruid = p->egid = p->rgid = 0;

	p->priv.real = 0x0;
	p->priv.eff = 0x0;
	p->priv.mask	= 0;

	p->auid = auid;
	p->groups[0] = NEG_ONE;
}

procstate(state)
int state;	/* if proc[pid] exists, return true if in specified state */
{
	if ( proc[pid] == NULL ) return(FALSE);
	return(proc[pid]->session_data->state == state);
}

static warn(str)
char *str;
{
	fprintf(stderr,"auditfmt: %s\n",str);
}

char *
ipc_op(n)
unchar n;
{
	switch (n) {
	case IPCMSG:
		return(ipc_ops[0]);
	case IPCSHM:
		return(ipc_ops[1]);
	case IPCSEM:
		return(ipc_ops[2]);
	default:
		return(ipc_ops[3]);

	}
}







