/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) auditmap.c: version 25.1 created on 12/2/91 at 17:16:22	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)auditmap.c	25.1	12/2/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
#include "auditmap.h"
#include <sys/audit.h>

/* list head pointer
 */
struct idlist *list;
struct idlist *pre;

/*
 * Description:
 * auditmap read all of the mounted file systems and the system files:
 *		 /etc/security/user_auth, /etc/group, 
 *		 /dev,  and  /etc/security/roles. 
 * and prints header information in the audit file
 *
 * The information is used by auditfmt to process a given security audit
 * trail.  This information is :
 *		1) current date, and uname(1) info
 *
 * 		2) from /etc/security/user_auth -
 *
 *		3) from /etc/group  - 
 *			the group id, group name
 *
 *		4) from /etc/security/roles  - 
 *			a reldef_t for each role 
 *
 *		4) from /dev -
 *			all character special devices in the /dev directory
 *			Note: no sub-directories are examined.
 *
 * Auditmap upon finishing a data file prints on the console the totals.  After
 * the above files have been examined and the information recorded. Auditmap
 * will print filesystem name maps.  This functionality is accomplished by
 * afsmap.  Auditmap will call afsmap for every mounted
 * filesystem.  The mounted filesystems are found by reading "/etc/mnttab".
 *
 * Exit conditions:
 *
 *	6 :	Insufficent memory to allocated needed data structures.
 *
 */
extern FILE  *audF;
extern FILE  *conF;

auditmap()
{
	FILE	*out;
	DIR	*fd;
	int	child_status;
	int version;
	int	fm = 0, total = 0, i = 0, oflag = 0, eor_mkr = -1;
	ino_t	eofs = ~0;    /*end of file system */
	char	*filename;
	char	*progname;
	void 	qsort();

	/* Various pointer to structures used to read information
	 */
	usrauth_t	*ua;
	roldef_t	*rl,*save_rl;
	struct group	*gr;
	struct dirent	*de;
	struct stat	*st;
	struct mnttab	*mt;

	out = audF;

	/* Malloc space for data structures
	 */
	de = (struct dirent *) malloc(sizeof(struct dirent));
	st = (struct stat *)   malloc(sizeof(struct stat));
	mt = (struct mnttab *) malloc(sizeof(struct mnttab));

	if ( (de == (struct dirent *) 0) || (st == (struct stat *)   0) ||
	     (mt == (struct mnttab *) 0) )
	{
		fprintf(conF, "%s: Insufficent memory\n", progname);
		exit(6);
	}

	/*  Print out the audit trail version
	 */
	version = AUDIT_TRAIL_VERSION;
	fwrite(&version,4,1,out);

	/* Get and print out the date
	 */
	print_date(out);

	/* Get and print out system info
	 */
	print_sysinfo(out);



	/*Print greeting on stderr, so they know whats happening
	 */
	fprintf(conF,"\nAudit Trail Starting Up: Header information being written\n");

	/* Open the user_uath file
	 */
	setuaent();  

	fprintf(conF, "   ...user name map ");
	fprintf(out,    "users:");

	/* Initialize list
	 */
	pre = list = (struct idlist *) 0;

	/* Loop thru user_uath file, removing duplicate uid's and sorting
	 * in numerical order in a linked list.
	 */
	while ( (ua = getuaent()) != (usrauth_t *) 0 )
		make_list(ua, USER);

	/* Print out users information
	 */
	total = print_list(out, USER);

	/* Print out totals
	 */
	fwrite(&eor_mkr, sizeof(int),1,out);
	fprintf(out, "%s  ( %d uids )\n","end_of_users", total);
	fprintf(conF, "(%d uids) complete.\n", total);

	/* Close user_uath file
	 */
	enduaent();

	

	/* Open role file
	 */
	setrlent();

	fprintf(conF, "   ...role name map ");
	fprintf(out, "roles:");

	/* Initialize list
	 */
	pre = list = (struct idlist *) 0;

	/* Loop thru role file, 
	 */
	while ( (rl = getrolent()) != (roldef_t *) 0 ){
		save_rl = (roldef_t *) malloc (sizeof(roldef_t));
		*save_rl = *rl;     /*we need to save the whole structure*/
		make_list(save_rl, ROLE);
	}

	/* Print out roles
	 */
	total = print_list(out, ROLE);

	strcpy(save_rl->r_name,"marker");
	save_rl->r_rolenumber = -1;
	fwrite(save_rl,sizeof(roldef_t),1,out);
	fprintf(out, "%s  ( %d roles )\n","end_of_roles", total);
	fprintf(conF, "(%d roles) complete.\n", total);

	/* Close role file
	 */
	endrlent();


	
	/* Open group file
	 */
	setgrent();

	fprintf(conF, "   ...group name map ");
	fprintf(out, "groups:");

	/* Initialize list
	 */
	pre = list = (struct idlist *) 0;

	/* Loop thru group file, removing duplicate gid's and sorting
	 */
	while ( (gr = getgrent()) != (struct group *) 0 )
		make_list(gr, GROUP);

	/* Print out groups
	 */
	total = print_list(out, GROUP);

	fwrite(&eor_mkr, sizeof(int),1,out);
	fprintf(out, "%s  ( %d gids )\n","end_of_groups", total);
	fprintf(conF, "(%d gids) complete.\n", total);

	/* Close group file
	 */
	endgrent();

	/* Open /dev
	 */
	fd = opendir(DEV_DIRECTORY);

	/* Change to /dev to make stats work
	 */
	chdir(DEV_DIRECTORY);

	fprintf(conF, "   ...tty name map ");
	fprintf(out, "ttys:");

	/* Initialize list
	 */
	pre = list = (struct idlist *) 0;

	/* Read the /dev directory and add to the list only
	 * those files that:
	 *	have a i-number != 0 and are character special devices
	 */

	while ( (de = readdir(fd)) != (struct dirent *) 0 )
	{
		if ( de->d_ino == 0 )
			continue;

		stat(de->d_name, st);

		i = (st->st_mode&S_IFMT);

		if ( i != S_IFCHR )
			continue;
		else
			make_list(de, TTYS);
	}

	/* Close /dev
	 */
	closedir(fd);

	/* Print out character devices
	 */
	total = print_list(out, TTYS);

	fwrite(&eor_mkr, sizeof(int),1,out);
	fprintf(out,"%s\n","end_of_ttys");
	fprintf(conF, "(%d devices) complete.\n", total);






	/* Print all filesystems currently mounted
	 * This is determined by reading /etc/mnttab.
	 */
	fm = open(MOUNT_TABLE, O_RDONLY);

	/* Read entries in the mnttab and print filesystem maps
	 * for each entry by using sfsmap.
	 */
	while ( read(fm, mt, sizeof(struct mnttab)) == sizeof(struct mnttab) )
	{
		fprintf(conF, "   ...name map for %s ", mt->mt_dev);
		fprintf(out, "filesystem:");
		fflush(out);

		afsmap(mt->mt_dev, mt->mt_filsys);

		fwrite(&eofs,sizeof(ino_t),1,out); 
		fprintf(out,"end_of_filesystem\n");
		fflush(out);
        }

	/* Close /etc/mnttab
	 */
	close(fm);

	/* All done now print out end of map line "date:"
	 * fflush & close file pointers, and return.
	 */
	fprintf(out,"data:\n");

	fflush(out);

	free(de);	            /*free up our junk */
	free(st);
	free(mt);

	return 0;
}

void
print_date(out)
FILE *out;
{
	long t;
	char date[30];

	/* Get current time in seconds
	 */
	time(&t);

	/* Format date as "Mon Sep 29 12:30:29 EST 1987"
	 */
	cftime(date, "%a %b %d %T %Z %Y", &t);

	/* print date to file pointer
	 */
	fwrite(date,30,1,out);

	return;
}


void
print_sysinfo(out)
FILE *out;
{
	struct utsname nm;

	/* Get system name information see uname(1)
	 */
	uname(&nm);
	
	/* Print info to file pointer
	 */
	fprintf(out, "%s %s %s %s %s\n",
		nm.sysname, nm.nodename, nm.release, nm.version, nm.machine);

	return;
}

