/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) mirror.c: version 25.1 created on 12/2/91 at 16:38:22	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)mirror.c	25.1	12/2/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

#ident	"@(#)src:mirror.c	25.1"

/*
 * Mirror Device Utility
 * 
 * This utility will create, open, close or delete a mirror device; will
 * initialize kernel data structures from MIRROR_TAB 
 *
 * The utility may open and read information from a configuration-information 
 * file.  
*/

#include <signal.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include "sys/ioctl.h"
#include <sys/buf.h>
#include "sys/vd.h"
#include <sys/ino.h>	
#include <mnttab.h>
#include <sys/fs/s5dir.h>

#define	MIRROR_TAB	"/etc/mirrortab"	/* mirror device data table */
#define HELP_FILE	"/etc/mirror.help"	/* mirror help file */
#define LOCK_TIMEOUT	30			/* locking attempt timeout */

static int	input_mode;	/* M_PROMPT or M_FILE or M_CMDLINE */	
#define	M_PROMPT	1	/* Prompt mode flag, => no config file */
#define	M_FILE		2	/* Config file mode flag */
#define	M_CMDLINE	4

#define	MAXNM		64	/* max length of device name */

/* Options (cmd selected = one of these ) */
#define	SHOW	1	/* '-s': Show MIRROR_TAB entry on a mirrored device */
#define	OPEN	2	/* '-o': Open a mirrored device */
#define	CLOSE	3	/* '-c': Close a mirrored device */
#define	DELETE	4	/* '-d': Delete (Clear) a mirrored device */
#define	MIRROR	5	/* '-m': Mirror (Initial set-up) a mirrored device */
#define	INIT	6	/* '-i': Init kernel for all mirrors in etc/mirrortab */
			/*	compress mirrortab and thus mr_data */
#define	SHOWALL	7	/* '-l': Show all entries */
#define	REMIRROR 8	/* '-r': remirror a disk, source chosen by -P and -S */

#define BAD_SEEK	(-999999L)

#define dprintf		if (debugflag) eprintf

#define	min(A, B)	((A) > (B) ? (B) : (A))
#define bzero(p, size)	memset(p, 0, size)
#define EQ(a, b)	(strcmp(a, b) == 0)

extern char	*optarg;
extern int	optind;
extern int	opterr;

extern int	errno, sys_nerr;
extern char	*sys_errlist[];

extern char	*memset(), *strdup();
extern unsigned	alarm(), sleep();
extern off_t	lseek();
extern void	exit();

static int	mt_fd;
static int	pri_fd;		/* primary mirrored device file descriptor */
static char	pri_name[MAXNM]; /* primary mirrored device name (char dev) */
static char	mirror_tab[] = MIRROR_TAB;
static char	help_file[] = HELP_FILE;
static char	dev_rdsk[] = "/dev/rdsk/";
static char	dev_r[] = "/dev/r";

static int	no_copy_mode;	/* '-n': don't copy disk		*/
static int	auto_mode;	/* '-a': automatic copy mode		*/
static int	debugflag;	/* '-D': undocumented debug mode	*/
static int	repair_mode;	/* set while repairing a mirror		*/
static int	copy_src = -1;	/* dir for remirrors, set by -P & -S flag */

typedef struct mirror_element { /* each entry in MIRROR_TAB = mirror struct. */
	mirror_t     md_kern_data;
	char         pd_name[MAXNM];	/* primary name */
	char         sd_name[MAXNM];	/* secondary name */
} mir_dev_struct; /* mirror entry structure - one entry per mirrored device */
 
static mir_dev_struct	entry; 	
static off_t		ent_offset = BAD_SEEK;

#define kern_data (&entry.md_kern_data)
 
typedef struct mirror_component {  /* used only in setting up mirror entry */
	int	cdev;
	int	bdev;
	int	fd;
	int	cyl_size;
	int	disk_size;
	char 	*name;
	char 	*prompt;
} mrcomp_t ;
 
static struct stat 	ld_chrstat; /* blk/chr stat for primary */
static struct stat 	ld_blkstat;
 
static char	*config_file;	/* the configuration file name, if any */
static FILE	*config_fptr;	/* Configuration file file pointer */

static char	*prog_name;	/* this program name, used in err messages */
 
static int	nargs;		/* a copy of argc, used by get_string */
static char	**argvals;	/* a copy of argv, used by get_string */
static int	cmd;

static char	tok_sep[] = " \n\t"; /* strtok(3) separator string */

void		die();
char		*get_root_dev();/* used for WAIT_MIRROR_ERROR ioctl */

main(argc,argv)
int	argc; 
char	**argv;
{
	init(argc, argv);
 
 	switch (cmd) {
 		case INIT:	init_md();
                            	break;
 		case MIRROR:	config_md();
                             	break;
 		case OPEN:	open_md();
                             	break;
		case CLOSE:	close_md();
                            	break;
  		case DELETE:	delete_md();
                            	break;
		case SHOW:	show_md();
                            	break;
		case SHOWALL:	show_all();
                            	break;
		case REMIRROR:	remirror_md();
				break;
        }
	die(0);
	/*NOTREACHED*/
}

/*
 * pathend -- return the last component of a pathname
 */

char *
pathend(path)
char	*path;
{
	char	*p;

	if (p = strrchr(path, '/'))
		return (p + 1);
	return (path);
}

/*
 * init -- check out arguments and set things up
 */

init(argc, argv)
int	argc; 
char	**argv;
{
	int	uncool = 0;
	int	cmd_selected = 0;
	int	c;
 
	setbuf(stdout, (char *)NULL);
	prog_name = pathend(argv[0]);
	nargs = argc;
	argvals = argv;
 
	input_mode = M_PROMPT;
	cmd = SHOWALL;			/* default command */
 
	while ((c = getopt(argc, argv, "Damnocdsilf:rPS")) != EOF)
 		switch (c) {
 			case 'f' :
 				input_mode = M_FILE;
 				config_file = optarg;
 				break;
 			case 'm' :
 				cmd = MIRROR;
 				cmd_selected++;
 				break;
 			case 'c' :
 				cmd = CLOSE;
 				cmd_selected++;
 				break;
 			case 'o' :
 				cmd = OPEN;
 				cmd_selected++;
 				break;
 			case 'd' :
 				cmd = DELETE;
 				cmd_selected++;
 				break;
 			case 's' :
 				cmd = SHOW;
 				cmd_selected++;
 				break;
 			case 'a' :
 				auto_mode++;
 				break;
 			case 'n' :
 				no_copy_mode++;
 				break;
 			case 'i' :
 				cmd = INIT;
 				cmd_selected++;
 				break;
 			case 'l' :
 				cmd = SHOWALL;
 				cmd_selected++;
 				break;
 			case 'D' :
 				debugflag++;
 				break;
			case 'r' :
 				cmd = REMIRROR;
 				cmd_selected++;
 				break;
			case 'P' :
				copy_src = MR_COPY_P_TO_S;
				break;
			case 'S' :
				copy_src = MR_COPY_S_TO_P;
				break;
 			case '?' :
 				uncool++;
 		}
 	if (uncool || optind == 1 || cmd_selected > 1) 
 		usage();
 
 	if (optind < argc) {
 		if (input_mode == M_FILE)
 			usage();
 		input_mode = M_CMDLINE;
 	}
 
	/* get name of primary device from user;  open MIRROR_TAB */ 
	/* if it exists and get copy of the entry for the */
	/* primary device (if an entry exists). */
 
 	if (cmd != INIT && cmd != SHOWALL)
 		init_globals();

	signal(SIGINT,die);	/* make sure mirrortab is left unlocked */
	signal(SIGHUP,die);
	signal(SIGQUIT,die);
}

/*
 * prompt -- print the prompt with up to two arguments if mode is M_PROMPT
 */
/*VARARGS1*/
prompt(fmt, a, b)
char	*fmt;
{
	if (input_mode == M_PROMPT) 
		printf(fmt, a, b);
}

/*
 * eprintf -- print a string with up to three arguments on stderr (see dprintf)
 */
/*VARARGS1*/
eprintf(fmt, a, b, c)
char	*fmt;
{
	fprintf(stderr, fmt, a, b, c);
}

/*
 * err - generic error reporting routine.
 *	prints to stderr the program name, a format string with up to 4
 *	arguments, along with errno (if non-zero)
 */
/*VARARGS1*/
void
err(fmt, a, b, c, d)
char	*fmt;
{
	int	err_num = errno;

	fprintf(stderr, "%s : ", prog_name);
	fprintf(stderr, fmt, a, b, c, d);
        if (err_num > 0 && err_num < sys_nerr)
		fprintf(stderr, " (%s)\n", sys_errlist[err_num]);
	else if (err_num)
		fprintf(stderr, " (error number %d)\n", err_num);
	else
		fputc('\n', stderr);
}

/* init_globals: */
/* 	get name of primary device from user;  open configuration file */
/* 	if one is specified, and open MIRROR_TAB if it exists (create it */
/*	if it doesn't), and get a copy of the entry for the */
/* 	primary device (if an entry exists). */

init_globals()
{
	char *s;

	dprintf("init_globals :\n");
	/* 
	 *  set input_mode and open configuration file if specified 
	 */
	if (input_mode == M_FILE) {
		config_fptr = fopen(config_file, "r");
		if (config_fptr == NULL) {
			err("Cannot open config file %s", config_file);
			die(1);
		}
	}

	s = "\nEnter the character device for the primary mirrored disk -> "; 
	pri_fd = get_dev(&ld_chrstat, &ld_blkstat, pri_name, s);
}

/*
 * entry_dump -- dump a mirror_t to stderr if we have nowhere else to put it
 */

entry_dump(mp)
register mirror_t	*mp;
{
	fprintf(stderr,
	  "Error: flags=%#x, pri=[0x%04x], sec=[0x%04x], valid=[0x%04x]\n",
	  mp->m_flags, mp->m_pri, mp->m_sec, mp->m_valid_dev);
	fprintf(stderr,
	  "\topentyp=0x%08x, cyl_size=%d, percent=%d, size=%u, copy_blk=%d\n",
	  mp->m_opentyp, mp->m_cyl_size, mp->m_ppercent, mp->m_size,
	  mp->m_cp_blk);
}

/*	Init_md will become the mirror daemon.  After initializing
 *	the kernel from /etc/mirrortab it will sleep waiting for mirrored
 *	slices to become invalid.  When a mirror becomes invalid the
 *	ioctl WAIT_MIRROR_ERROR will return the mirror struct for the
 *	mirror that failed.  It is then replaced in /etc/mirrortab.
 */

init_md()
{
	int		root_fd;
	mirror_t	mdata;
	char 		root_dev_name[MAXNM];

	dprintf("init_md :\n");

	(void) close(0);		/* won't be needing stdin */
	pri_fd = -1;

	if (!get_mt_fd())
		init_entries();
	
	(void) close(mt_fd);
	
	signal(SIGCLD, SIG_IGN);

	switch (fork()) {
	case -1:
		die(1);			/* fork failed */
	case 0:
		break;			/* child */
	default:
		die(0);			/* parent exits */
	}

	/* child code */

	(void) setpgrp();	/* now immune to keyboad generated interrupts */
	signal(SIGHUP, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	(void) chdir("/");

	sprintf(root_dev_name, "%s%.*s", dev_rdsk,
	  sizeof(root_dev_name) - sizeof(dev_rdsk), pathend(get_root_dev()));

	if ((root_fd = open(root_dev_name, O_RDONLY)) < 0) {
		err("Cannot open root device: %s", root_dev_name);
		die(1);
	}

	for (; ; (void) close(mt_fd)) {
        	if (ioctl(root_fd, WAIT_MIRROR_ERROR, &mdata) == -1) {
        		err("Ioctl to WAIT_MIRROR_ERROR failed.");
			die(1);
		}
		dprintf("mirror: {error daemon} ioctl returned\n");

        	if (get_mt_fd()) {
			errno = 0;
			err("{error daemon} %s was missing!", mirror_tab);
			entry_dump(&mdata);
			continue;
		}
		if (dev_get_entry(mdata.m_pri)) {
			errno = 0;
			err("{error daemon} no mirror entry for [0x%04x]!",
			  mdata.m_pri);
			entry_dump(&mdata);
			continue;
		}
		if ((pri_fd = open(entry.pd_name, O_RDONLY)) < 0) {
			err("{error daemon} can't open %s", entry.pd_name);
			entry_dump(&mdata);
			continue;
		}
		strcpy(pri_name, entry.pd_name);
		redo_entry();
		(void) close(pri_fd);
		pri_fd = -1;
	}
}


/*
 * help -- find a %%str in HELP_FILE and print it to stderr
 */

/*VARARGS1*/
help(str, arg)
register char	*str, *arg;
{
	register FILE	*fp;
	register char	*p;
	char		line[2 * MAXNM];

	if ((fp = fopen(help_file, "r")) == NULL) {
		err("Can't open help file %s to find %%%%%s", help_file, str);
		return;
	}

	/*
	 * find %%str
	 */

	for (;;) {
		if (fgets(line, sizeof(line), fp) == NULL) {
			(void) fclose(fp);
			err("Didn't find %%%%%s in help file", str);
			return;					/* EOF */
		}
		if (line[0] != '%' || line[1] != '%')
			continue;				/* not %% */
		if ((p = strtok(line + 2, tok_sep)) == NULL)
			continue;				/* no str */
		if (!EQ(p, str))
			continue;				/* no match */
		break;
	}

	/*
	 * print the text until the next %% or EOF
	 */

	while (fgets(line, sizeof(line), fp)) {
		if (line[0] == '%' && line[1] == '%')
			break;					/* %% */
		if (arg)
			fprintf(stderr, line, arg);
		else
			fputs(line, stderr);
	}

	(void) fclose(fp);
}


open_md()
{
	dprintf("open_md :\n");

	(void) get_mt_fd();

        if (name_get_entry(pri_name)) {
		printf("Error - mirror is not configured - use -m option.\n");
		die(1);
        }

	if (ioctl(pri_fd, LOCK_MIRROR_DISK, 0) < 0) {
		redo_entry();
		err("Ioctl to open mirrored disk failed.");
		die(1);
	} 
	redo_entry();
}

/*	copy disk will fork a child that will attempt to copy one disk  *
 *	slice to another.  If the copy fails the mirror will be marked	*
 *	invalid.  							*
 */

static void
copy_disk(from_pri)
int	from_pri;
{
	int	errnum;

	dprintf("copy_disk :\n");

	signal(SIGCLD, SIG_IGN);

	switch (fork()) {
	case -1:
		err("copy_disk: fork failed");
		die(1);			/* fork failed */
	case 0:
		break;			/* child */
	default:
		return;			/* parent returns */
	}

	/* child code */

	(void) setpgrp();	/* now immune to keyboad generated interrupts */
	signal(SIGHUP, SIG_IGN);
	signal(SIGINT, SIG_IGN);
	signal(SIGQUIT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);
	(void) chdir("/");

       	if (ioctl(pri_fd, RE_MIRROR_DISK, from_pri) < 0) {
		errnum = errno;
		redo_entry();
		errno = errnum;
		err("Ioctl to copy mirrored disk %s failed.", pri_name);
		die(1);
	}  
	redo_entry();
	die(0);
}

/*
 * config_md - configure an unconfigured mirrored disk.
 *             uses: GLOBAL - kern_data;
 *             in: nothing
 *             out: nothing 
 */
config_md()
{
	mrcomp_t	pri,
			sec;
	int 		i;
	int		new_mt;
	char		buf[MAXNM];

	dprintf("config_md :\n");

        prompt("\nConfiguring %s\n", pri_name);

	new_mt = get_mt_fd();

	if (!name_get_entry(pri_name)) {
		printf("Device %s is already mirrored.\n", pri_name);
		die(1);
	}

	memset((char *)&entry, 0, sizeof(entry));
	strcpy(entry.pd_name, pri_name);
	pri.name = entry.pd_name;
	pri.fd = pri_fd;
	pri.cdev = ld_chrstat.st_rdev;
	pri.bdev = ld_blkstat.st_rdev;

        if ((pri.cyl_size = ioctl(pri.fd, GET_CYLINDER_SIZE, 0)) < 0) {
        	err("Cannot get cylinder size of %s", pri.name);
                die(1);
        }
        if ((pri.disk_size = ioctl(pri.fd, GET_DISK_SIZE, 0)) < 0) {
        	err("Cannot get size of %s", pri.name);
                die(1);
        }

	sec.prompt =
	  "Enter the character device for the secondary component -> ";
	sec.name = entry.sd_name; 
	get_comp(&sec);
	while (EQ(pri.name, sec.name)) {
		printf("\nError -  You cannot mirror a disk onto itself.\n\n");
		get_comp(&sec);
	}

	if (pri.disk_size > sec.disk_size)
		help("pri_larger_than_sec", NULL);

	if (pri.disk_size < sec.disk_size)
		help("pri_smaller_than_sec", NULL);

	kern_data->m_flags = MR_OPEN;
	kern_data->m_pri = pri.cdev;
	kern_data->m_sec = sec.cdev;
        kern_data->m_cyl_size = min(pri.cyl_size, sec.cyl_size);
        kern_data->m_size = min(pri.disk_size, sec.disk_size);
	kern_data->m_valid_dev = pri.cdev;

	for (;;) {
        	get_string(buf, "Enter primary device read percentage -> ");

		dprintf("value returned by get_string = %s\n", buf);

		if (sscanf(buf, "%u", &kern_data->m_ppercent) <= 0 ||
		    (kern_data->m_ppercent != 0 &&
		     kern_data->m_ppercent != 50 &&
		     kern_data->m_ppercent != 100)) {
			if (input_mode != M_PROMPT) {
				errno = 0;
				err("Invalid read percentage: %d -- exiting",
				  kern_data->m_ppercent);
				die(1);
			}
			prompt("Invalid read percentage: %s\n", buf);
			prompt("Valid values are 0, 50 or 100 percent\n");
			continue;
		}
		break;			/* ok */
	}

	(void) close(sec.fd);
	sec.fd = -1;

	errno = 0;

	if (no_copy_mode) {
		printf("No Copy: the mirror components may not match\n");
		kern_data->m_flags |= MR_VALID;
        	if (ioctl(pri_fd, CREATE_MIRROR_DISK, kern_data) < 0) {
        		err("Ioctl to set mirrored disk failed.");
			if (new_mt || errno == EINVAL)		/* help them */
				help("mirror_not_configured", NULL);
               		die(1);
	        } 
		put_entry();
		redo_entry();
	} 
	else {
		printf("Mirror %s to %s?\n", pri.name, sec.name);

		printf(" DEL if wrong:  ");

		for (i = 5; i >= 0; i--) {
			printf("%d\b", i);
			fflush(stdout);
			sleep(1);
		}
		printf("\n");

        	if (ioctl(pri_fd, CREATE_MIRROR_DISK, kern_data) < 0) {
        		err("Ioctl to create mirrored disk failed.");
			if (new_mt || errno == EINVAL)		/* help them */
				help("mirror_not_configured", NULL);
               		die(1);
	        } 

		put_entry();
		redo_entry();

        	copy_disk(MR_COPY_P_TO_S);
	}
}

get_comp(comp)
	mrcomp_t *comp;
{
	struct stat 	chrstat, blkstat;

	dprintf("get_comp : \n");

        comp->fd = get_dev(&chrstat, &blkstat, comp->name, comp->prompt);
	comp->cdev = chrstat.st_rdev;
	comp->bdev = blkstat.st_rdev;
        if ((comp->cyl_size = ioctl(comp->fd, GET_CYLINDER_SIZE, 0)) == -1) {
        	err("Cannot get cylinder size of %s",comp->name); 
                die(1);
        }
        if ((comp->disk_size = ioctl(comp->fd, GET_DISK_SIZE, 0)) == -1) {
        	err("Cannot get disk size of %s", comp->name);
                die(1);
        }
}

#if 0
/*
 * yorn -- gets a yes or no answer, returns non-zero for yes
 */

yorn(prmpt)
char	*prmpt;
{
	char	*p, answer[MAXNM];

	fputs(prmpt, stdout);
	get_string(answer, " (y/n) -> ");
	p = strtok(answer, tok_sep);
	return (p && (p[0] == 'y' || p[0] == 'Y'));
}
#endif


delete_md()
{
	dprintf("delete_md: \n");

        if (ioctl(pri_fd, DELETE_MIRROR_DISK, kern_data) < 0 && errno != ENXIO){
        	err("Ioctl to delete mirrored disk failed.");
		die(1);
	}

	(void) get_mt_fd();

	if (name_get_entry(pri_name)) {
		printf("Device %s is not in %s\n", pri_name, mirror_tab);
		die(1);
	}
	clear_entry();
}

/*
 * get_mirror_disk -- do the GET_MIRROR_DISK ioctl on pri_fd into kern_data
 */

get_mirror_disk()
{
        if (ioctl(pri_fd, GET_MIRROR_DISK, kern_data) < 0) {
        	err("Ioctl to get the mirrored disk data for %s failed.",
		  pri_name);
		return (1);
	}
	return (0);
}

close_md()
{
	dprintf("close_md: \n");

	(void) get_mt_fd();

	if (name_get_entry(pri_name)) {
		err("Device %s is not in %s", pri_name, mirror_tab);
		die(1);
	}

        if (get_mirror_disk())
		die(1);

	if (!mr_open(kern_data)) {
		errno = 0;
		err("%s is not open - cannot close", pri_name);
		die(1);
        }

	if (ioctl(pri_fd, LOCK_MIRROR_DISK, 1) < 0) {
		redo_entry(); 
        	err("Ioctl to close mirrored disk failed.");
		die(1);
        }
	redo_entry(); 

	dprintf("Mirrored disk successfully closed\n");
}

/* 
 * get_mt_fd - If it exists, open the current mirrored device data table.
 *              SIDE EFFECTS: Opens a new file for the mirror device data if
 *                           none exist, fills in global mt_fd
 *              returns: 1 if a new table, 0 if table exists.
 *              exit on: Any problem creating or opening "MIRROR_TAB".
 */
int
get_mt_fd()
{
	int	new_mirrortab = 0;
	int	i, modes, rw;

	dprintf("get_mt_fd :\n");

	rw = (cmd == SHOW || cmd == SHOWALL) ? O_RDONLY : O_RDWR;

	for (i = 10; --i >= 0; ) {
		modes = rw | O_SYNC;
		if ((mt_fd = open(mirror_tab, modes)) >= 0)
			break;
		if (errno != ENOENT)
			break;
		modes = rw | O_CREAT | O_EXCL | O_SYNC;
		if ((mt_fd = open(mirror_tab, modes, 0600)) >= 0) {
			new_mirrortab = 1;
			break;
		}
		if (errno != EEXIST)
			break;
	}
	if (mt_fd < 0) {
		err("open of %s with modes %#o failed", mirror_tab, modes);
		die(1);
	}

        return(new_mirrortab);
}

/*
 * get_dev - prompt user and gets device.
 */
get_dev(chrstat, blkstat, name, prompt_ptr)
struct stat *chrstat, *blkstat;
char *name;
char *prompt_ptr;
{
	int fd;

	dprintf("get_dev : input_mode = %d \n", input_mode);

	while (1) {
		get_string(name, prompt_ptr);
                if ( !check_dev(chrstat, blkstat, name, &fd) ) 
			return(fd);
		/* 
		 *    check_dev returned error 
		 */
		if ( input_mode == M_FILE ) 
			die(1);
		if ( input_mode == M_CMDLINE )
			input_mode = M_PROMPT;
	}
}

/*
 * check_dev - in: device name, status buffer
 *             out: filled status buffer
 *             returned: 1 if not sucessful, 0 on success
 */
int
check_dev(chrstat, blkstat, chrname, chrfd)
struct	stat	*chrstat, *blkstat;
char	*chrname;
int	*chrfd;
{
	char	blkname[MAXNM];

	dprintf("check_dev : devname = %s\n",chrname);

	if (stat(chrname, chrstat) < 0) {
		err("cannot stat character device: %s", chrname);
		return(1);
	}
	if (*chrname != '/') {
		errno = 0;
		err("device names must be absolute pathnames: %s", chrname);
		fputs("\tExample:  /dev/rdsk/c0d1s4\n", stderr);
		return(1);
	}
	if (strncmp(chrname, dev_r, sizeof(dev_r) - 1)) {
		errno = 0;
		err("All mirror device pathnames must start with '%s': %s",
		  dev_r, chrname);
		fputs("\tExample:  /dev/rdsk/c0d1s4\n", stderr);
		return(1);
	}
	strcpy(blkname, "/dev/");
	strcat(blkname, chrname + sizeof(dev_r) - 1);

	if (stat(blkname, blkstat) < 0) {
		err("cannot stat block device: %s", blkname);
		return(1);
	}

	if ((chrstat->st_mode & S_IFMT) != S_IFCHR) {
		errno = 0;
		err("%s not a character special device.", chrname);
		return(1);
	}
	if ((blkstat->st_mode & S_IFMT) != S_IFBLK) {
		errno = 0;
		err("%s not a block special device.", blkname);
		return(1);
	}
	if (blkstat->st_rdev != chrstat->st_rdev) {
		errno = 0;
		err("block device (%s) and char device (%s) don't match",
		  blkname, chrname);
		fprintf(stderr, "\tblock=[%d, %d] char=[%d, %d]\n", 
		  major(blkstat->st_rdev), minor(blkstat->st_rdev),
		  major(chrstat->st_rdev), minor(chrstat->st_rdev));
		return(1);
	}
	if ((*chrfd = open(chrname, O_RDONLY)) < 0) {
		if (repair_mode && errno == EBUSY && EQ(chrname, entry.sd_name))
			return (0);		/* we expect this */
		err("cannot open character device %s", chrname);
        	return(1);
        }
	return (0);
}
                 
/*
 * get_string - Setup a string from user input of config file input.
 *             in: a string of at least MAXNM bytes and a prompt
 *             out: nada
 *             exit on: 'q' as the first letter of any command.
 *	cute stuff : if in M_PROMPT input_mode and there are extra command line args,
 *		use them.
 */
get_string(str, prmpt)
	char *str;
	char *prmpt; 
{
	char	*p, s[MAXNM];

	dprintf("get_string :\n");

	fflush(stdout);
	*str = '\0';
	switch (input_mode) {
		case M_CMDLINE:
			if (optind < nargs) {
				strcpy(s,argvals[optind++]);
				if (optind >= nargs)
					input_mode = M_PROMPT;
				break;
			}
			input_mode = M_PROMPT;
			/* fall through */
		case M_PROMPT:	
			fputs(prmpt, stdout);
			fflush(stdout);
			if (fgets(s, sizeof(s), stdin) == NULL)
				die(1);
			if(s[0] == 'q' || s[0] == 'Q')
				die(1);
			break;
		case M_FILE:
			if (fgets(s, sizeof(s), config_fptr) == NULL)
				return;
	}
	if (p = strchr(s, '\n'))
		*p = '\0';		/* remove any newline */

		/* create a pointer to str as needed */
	strcpy(str, s);
}

void
die(n)
int n;
{
        exit(n);
}

/*
 * remirror_md -- copy one side of a mirror to the other, if needed
 *		if -P or -S was specified, always copy, else only if invalid
 */

remirror_md()
{
	dprintf("remirror_md :\n");

	(void) get_mt_fd();

        if (name_get_entry(pri_name)) {
		printf("Error - mirror is not configured - use -m option.\n");
		die(1);
        }
	if (get_mirror_disk())
		die(1);

	if (copy_src < 0) {
		if (mr_valid(kern_data)) {
			prompt("%s does not need remirroring\n", entry.pd_name);
			exit(0);
		}
		copy_src = (kern_data->m_valid_dev == kern_data->m_sec) ?
		  MR_COPY_S_TO_P : MR_COPY_P_TO_S;
	}
	copy_disk(copy_src);
}

show_md()
{

	dprintf("show_md: \n");

	(void) get_mt_fd();

	if (name_get_entry(pri_name)) {
		printf("Device %s is not in %s\n", pri_name, mirror_tab);
		die(1);
	}
	unlock_tab();		/* don't leave MIRROR_TAB locked during show */

	pri_fd = open(entry.pd_name, O_RDONLY);
	show_entry();
}

/*
 * mr_name -- returns the name of a given device
 */

static char *
mr_name(m, dev)
mirror_t	*m;
dev_t		dev;
{
	if (dev == m->m_pri)
		return ("primary");
	else if (dev == m->m_sec)
		return ("secondary");
	return ("?????");
}

show_entry()
{
	register mirror_t	*datap;
	mirror_t		kdata;

	printf("\nMirrored slice %s : ", entry.pd_name);

        if (ioctl(pri_fd, GET_MIRROR_DISK, &kdata) < 0) {
		if (errno != ENXIO)
		        err("Ioctl to get the mirrored disk data failed."); 

		printf("   (Status from %s only)", mirror_tab);
		datap = kern_data;
	}
	else
		datap = &kdata;
	printf("\n");
	dprintf(" %s flags = 0x%x\n", mirror_tab, kern_data->m_flags);

	if (mr_open(datap)) {
		if (mr_valid(datap))
			printf("  Device is open and valid.\n");
		else
			printf("  Device is open and invalid.\n");
	} else { 
		if (mr_valid(datap)) {
			if (mr_open(kern_data))	/* data is out of sync */
				printf(
"  Device was not closed properly and will need to be remirrored.\n");
			else
				printf("  Device is closed and valid.\n");
		}
		else
			printf("  Device is closed and invalid.\n");
	}

	if (mr_copying(datap))
	        printf("  Disk is being remirrored from %s (now on block %d)\n",
		  mr_name(datap, datap->m_valid_dev), datap->m_cp_blk);
	else if (!mr_valid(datap))
		printf("  Valid device is %s [0x%04x]\n", 
		  mr_name(datap, datap->m_valid_dev),
		  datap->m_valid_dev);

	if (datap->m_flags &
	    (MR_PRI_ERROR | MR_SEC_ERROR | MR_DELAYED_ERR | MR_ABORT_COPY)) {
		if (datap->m_flags & MR_PRI_ERROR)
			printf("  (Primary had I/O Error)");
		if (datap->m_flags & MR_SEC_ERROR)
			printf("  (Secondary had I/O Error)");
		if (datap->m_flags & MR_DELAYED_ERR)
			printf("  (Delayed I/O Error)");
		if (datap->m_flags & MR_ABORT_COPY)
			printf("  (Abort Copy)");
		printf("\n");
	}
	if (datap->m_flags & MR_DO_ALL_ERRS) {
		printf("  Will cause these errors:");
		if (datap->m_flags & MR_DO_RP_ERR)
			printf("  primary-read");
		if (datap->m_flags & MR_DO_RS_ERR)
			printf("  secondary-read");
		if (datap->m_flags & MR_DO_WP_ERR)
			printf("  primary-write");
		if (datap->m_flags & MR_DO_WS_ERR)
			printf("  secondary-write");
		printf("\n");
	}
	printf("  Primary component   [0x%04x] = %s\n",
	  datap->m_pri, entry.pd_name);
	printf("  Secondary component [0x%04x] = %s\n",
	  datap->m_sec, entry.sd_name);
	printf("  Mirrored Disk Size           = %d\n", datap->m_size);
	printf("  Mirrored Cylinder Size       = %d\n", datap->m_cyl_size);
	printf("  Primary read percentage      = %d\n\n", datap->m_ppercent);

	if (debugflag)
		print_mirror_t(datap);
}

/*
 * comp_mir -- compare mirror data from MIRROR_TAB and kernel
 */

comp_mir(tab_dat, kern_dat, name)
uint	tab_dat, kern_dat;
char	*name;
{
	if (tab_dat != kern_dat) {
		printf("Warning: %s data does not match kernel data:\n",
		  mirror_tab);
		printf("	%s data : ", mirror_tab);
		printf(" %s	= %d\n", name, tab_dat);
		printf("	kernel data : ");
		printf(" %s	= %d\n", name, kern_dat);
	} else 
		printf(" %s	= %d\n", name, kern_dat);
}


/*
 * for debugging, this will print what MIRROR_TAB thinks the kernel 
 * should have.
 */
print_mirror_t(kdatap)
mirror_t	*kdatap;
{
	comp_mir(mr_open(kern_data), mr_open(kdatap), "mr_open flag");
	comp_mir(mr_valid(kern_data), mr_valid(kdatap), "mr_valid flag");
	comp_mir(mr_copying(kern_data), mr_copying(kdatap), "copying flag");
	comp_mir(kern_data->m_opentyp.ot_all_opens,
	  kdatap->m_opentyp.ot_all_opens, "m_opentyp");
	comp_mir(kern_data->m_valid_dev, kdatap->m_valid_dev, "m_valid_dev");
	comp_mir(kern_data->m_pri, kdatap->m_pri, "m_pri");
	comp_mir(kern_data->m_sec, kdatap->m_sec, "m_sec");
	comp_mir(kern_data->m_ppercent, kdatap->m_ppercent, "m_ppercent");
}

/*
 * usage -- print a usage message and die
 */

usage()
{
	help("usage", prog_name);
	die(1);
}

/*
 * seek -- do a lseek on mt_fd with error checking
 */

off_t
seek(offset, whence)
off_t	offset;
int	whence;
{
	off_t	ret;

	if ((ret = lseek(mt_fd, offset, whence)) < 0) {
		err("lseek(%d, %ld, %d) failed!", mt_fd, offset, whence);
		die(1);
	}
	return (ret);
}

/*
 * name_get_entry -- find an entry in MIRROR_TAB given its primary device name
 *	Returns zero with MIRROR_TAB locked on success, else zero and unlocked
 */

name_get_entry(name)
char *name;
{
	(void) seek(0L, SEEK_SET);

	lock_tab();

        while (read(mt_fd, (char *)&entry, sizeof(entry)) == sizeof(entry))
		if (EQ(entry.pd_name, name)) {
			ent_offset = seek(-(off_t)sizeof(entry), SEEK_CUR);
			return (0);
		}

	unlock_tab();

	bzero((caddr_t)&entry, sizeof(entry));
	ent_offset = BAD_SEEK;

	return(1);
}

/*
 * dev_get_entry -- find an entry in MIRROR_TAB given its primary device number
 *	Returns zero with MIRROR_TAB locked on success, else !=0 and unlocked
 */

dev_get_entry(dev)
dev_t dev;
{
	(void) seek(0L, SEEK_SET);

	lock_tab();

        while (read(mt_fd, (char *)&entry, sizeof(entry)) == sizeof(entry))
		if (entry.md_kern_data.m_pri == dev) {
			ent_offset = seek(-(off_t)sizeof(entry), SEEK_CUR);
			return (0);
		}

	unlock_tab();

	bzero((caddr_t)&entry, sizeof(entry));
	ent_offset = BAD_SEEK;

	return(1);
}

put_entry()
{
	mir_dev_struct	tmpent; 	
	int		found = 0;

	(void) seek(0L, SEEK_SET);

	lock_tab();

        while (read(mt_fd, (char *)&tmpent, sizeof(tmpent)) == sizeof(entry))
		if (EQ(tmpent.pd_name, entry.pd_name)) {
			found++;
			(void) seek(-(off_t)sizeof(entry), SEEK_CUR);
			break;
		}

	if (!found) {	/* use first available null entry */
		(void) seek(0L, SEEK_SET);

		while (read(mt_fd, (char *)&tmpent, sizeof(tmpent)) == 
				sizeof(entry))
			if (!tmpent.pd_name[0]) {
				(void) seek(-(off_t)sizeof(entry), SEEK_CUR);
				break;
			}
	}

	if (write(mt_fd, (char *)&entry, sizeof(entry)) != sizeof(entry)) {
		err("error writing entry into %s", mirror_tab);
		die(1);
	}

	unlock_tab();
}

/*
 * redo_entry -- update the kernel data in the entry
 */

redo_entry()
{
	mir_dev_struct	tmpent; 	
	int		found = 0;

	(void) seek(0L, SEEK_SET);

	lock_tab();

        while (read(mt_fd, (char *)&tmpent, sizeof(tmpent)) == sizeof(entry))
		if (EQ(tmpent.pd_name, entry.pd_name)) {
			found++;
			(void) seek(-(off_t)sizeof(entry), SEEK_CUR);
			break;
		}

	if (!found) {
		unlock_tab();
		return;
	}

	entry = tmpent;
	if (get_mirror_disk() == 0 &&
	    write(mt_fd, (char *)&entry, sizeof(entry)) != sizeof(entry)) {
		err("error writing entry into %s", mirror_tab);
		die(1);
	}

	unlock_tab();
}


clear_entry()
{
	mir_dev_struct	tmpent; 	

	dprintf("clear_entry: \n");
	(void) seek(0L, SEEK_SET);

	lock_tab();

        while (read(mt_fd, (char *)&tmpent, sizeof(tmpent)) == sizeof(entry)) {
		if (!EQ(tmpent.pd_name, entry.pd_name))
			continue;
		(void) seek(-(off_t)sizeof(entry), SEEK_CUR);
		bzero((caddr_t)&entry, sizeof(entry));
		ent_offset = BAD_SEEK;

		if (write(mt_fd, (char *)&entry, sizeof(entry)) !=
								sizeof(entry)) {
			err("error clearing entry in %s", mirror_tab);
			die(1);
		}
		break;
	}
	unlock_tab();
}

init_entries()
{
	register int	copy_dir;

	dprintf("init_entries :\n");

	lock_tab();

        while (read(mt_fd, (char *)&entry, sizeof(entry)) == sizeof(entry)) {
		if (!entry.pd_name[0]) 
			continue;

		if ((pri_fd = open(entry.pd_name, O_RDONLY)) < 0) {
			err("cannot open primary device %s", entry.pd_name);
			continue;
		}

		/*
		 * open | valid | copying | copy direction
		 * ---------------------------------------
		 *   0  |   0   |    0    | Valid -> Invalid
		 *   0  |   0   |    1    | should be impossible
		 *   0  |   1   |    0    | no copy
		 *   0  |   1   |    1    | should be impossible
		 *   1  |   0   |    0    | V -> I
		 *   1  |   0   |    1    | V -> I
		 *   1  |   1   |    0    | arbitrarily Primary -> Secondary
		 *   1  |   1   |    1    | V -> I
		 */
		copy_dir = (kern_data->m_valid_dev == kern_data->m_sec) ?
		  MR_COPY_S_TO_P : MR_COPY_P_TO_S;
		if (mr_valid(kern_data)) {
			if (mr_open(kern_data)) {
				kern_data->m_flags &= ~MR_VALID;
				if (!mr_copying(kern_data)) {
					kern_data->m_valid_dev =
					  kern_data->m_pri;
					copy_dir = MR_COPY_P_TO_S;
				}
			}
			else
				copy_dir = -1;		/* no copy needed */
		}
		/* else, m_valid_dev already set */

		if (ioctl(pri_fd, INIT_MIRROR_DISK, kern_data) < 0)
			err("Ioctl to initialize kernel data failed for %s",
			  entry.pd_name);
		redo_entry();

		if (copy_dir >= 0 && auto_mode) {
			strcpy(pri_name, entry.pd_name);
			copy_disk(copy_dir);
		}

		(void) close(pri_fd);
		pri_fd = -1;
		lock_tab();
	}
	unlock_tab();
}

show_all()
{
	dprintf("show_all :\n");

        (void) get_mt_fd();

        while (read(mt_fd, (char *)&entry, sizeof(entry)) == sizeof(entry)) {
		if (entry.pd_name[0]) {
			pri_fd = open(entry.pd_name, O_RDONLY);
			show_entry();
		}
	}
}

/*
 * lock_alrm -- SIGALRM catcher for lock_tab
 */

void
lock_alrm()
{
	err("file lock attempt timed out!");
	die(1);
}

/*
 * lock_tab -- lock all of mt_fd
 */

lock_tab()
{
	struct flock	l;

	l.l_type = (cmd == SHOW || cmd == SHOWALL) ? F_RDLCK : F_WRLCK;
	l.l_whence = SEEK_SET;
	l.l_start = 0L;
	l.l_len = 0L;

	signal(SIGALRM, lock_alrm);
	alarm(LOCK_TIMEOUT);

	if (fcntl(mt_fd, F_SETLKW, &l) < 0) {
		err("cannot lock %s", mirror_tab);
		die(1);
	}

	alarm(0);
}

/*
 * unlock_tab -- unlock mt_fd
 */

unlock_tab()
{
	struct flock	l;

	l.l_type = F_UNLCK;
	l.l_whence = 0;
	l.l_start = 0L;
	l.l_len = 0L;

	if (fcntl(mt_fd, F_SETLK, &l) < 0) {
		err("cannot unlock %s", mirror_tab);
		die(1);
	}
}

/*
 * get_root_dev -- returns the name of root's device from /dev/dsk
 */

char *
get_root_dev()
{
	int		fd;
	struct direct	dbuf;
	struct stat	S;
	static char	dir_name[] = "/dev/dsk/";
	static char	devnam[sizeof(dir_name) + sizeof(dbuf.d_name) + 1];

	if (stat("/", &S) < 0) {
		err("get_root_dev: stat of '/' failed!");
		die(1);
	}
	if ((fd = open(dir_name, O_RDONLY)) < 0) {
		err("get_root_dev: open of %s failed!", dir_name);
		die(1);
	}
	if (chdir(dir_name) < 0) {
		err("get_root_dev: chdir to %s failed!", dir_name);
		die(1);
	}
	if (!dsearch(fd, S.st_dev, &dbuf)) {
		err("get_root_dev: can't find root device in %s", dir_name);
		die(1);
	}
	(void) close(fd);
	strcpy(devnam, dir_name);
	strncat(devnam, dbuf.d_name, sizeof(dbuf.d_name));
	return(devnam);
}

dsearch(ddir, fno, dirp)
int			ddir;
dev_t			fno;
register struct direct	*dirp;
{
	struct stat	S;

	while (read(ddir, (char *)dirp, sizeof(*dirp)) == sizeof(*dirp)) {
		if (!dirp->d_ino)
			continue;
		if (stat(dirp->d_name, &S) < 0) {
			err("dsearch: cannot stat %.*s",
			  sizeof(dirp->d_name), dirp->d_name);
			return(0);
		}
		if ((fno == S.st_rdev) &&
		    ((S.st_mode & S_IFMT) == S_IFBLK) &&
		    (strcmp(dirp->d_name, "swap") != 0))
			return(1);
	}
	return(0);
}
