/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) getty.c: version 25.5 created on 3/4/92 at 21:22:53	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)getty.c	25.5	3/4/92 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/* See README.ses for a description of changes from 86-90 */

/*	getty - sets up speed, various terminal flags, line discipline,	*/
/*	and waits for new prospective user to enter name, before	*/
/*	calling "login".						*/
/*									*/
/*	Usage:								*/
/*	getty [-h] [-t time] line [speed_label]                         */
/*      [[-O][-S] [-F] [-s strmod] [-f] [-E variable=value] strmod ...]	*/
/*									*/
/*	-h says don't hangup by dropping carrier during the		*/
/*		initialization phase.  Normally carrier is dropped to	*/
/*		make the dataswitch release the line.			*/
/*	-t says timeout after the number of seconds in "time" have	*/
/*		elapsed even if nothing is typed.  This is useful	*/
/*		for making sure dialup lines release if someone calls	*/
/*		in and then doesn't actually login in.			*/
/*	"line" is the device in "/dev" and is required			*/
/*	"speed_label" is a pointer into the "/etc/gettydefs"		*/
/*			where the definition for the speeds and		*/
/*			other associated flags are to be found.		*/
/*			If omitted then the first entry in the		*/
/*			"/etc/gettydefs" file is used.			*/
/*	-S tells getty to save current terminal setting and restore it	*/
/*	   after ALL of the specified modules have been pushed.		*/
/*	-F if specified in front of a particular stream module, will	*/
/*	   force its push even if the module has already been pushed.	*/
/*	   This option has effect until '-f' which turns it off.	*/
/*	-f if specified will turn off '-F' effect.			*/
/*	-s if specified in front of a particular stream module, will	*/
/*	   save current terminal settings before pushing the module	*/
/*	   and will restore the settings after the push is done.	*/
/*	   Any occurance of this argument will turn OFF	the -S		*/
/*	   argument.                                               	*/
/*									*/
/*	"strmod" is the name of a stream module to be pushed on open.	*/
/*		 The modules are pushed in the order supplied on the	*/
/*		 cmd line.  The module name must have spelling and	*/
/*		 capitalization exactly	as the corresponding stream	*/
/*		 module name as defined in the kernel.			*/
/*									*/
/*	Usage:  getty -c gettydefs_like_file				*/
/*									*/
/*	The "-c" flag is used to have "getty" check a gettydefs file.	*/
/*	"getty" parses the entire file and prints out its findings so	*/
/*	that the user can make sure that the file contains the proper	*/
/*	information.							*/
/*									*/
/*      -E if specified, defines the environment variables to be used   */
/*         The variable argument must have the format variable=value    */
/*         If the variable  is redifined by the user in the login promt */
/*         the user definition will have priority.  Multiple -E options */
/*         can be specified.                                            */
/*      -O beginning of the optional arguments.  This option must be    */
/*          used if the optional speed is not specified and streams     */
/*          modules to be pushed are presented and no - options are     */
/*          used before the firststream module listed.                  */
/*                                                                      */

#include	<stdio.h>
#include	<sys/types.h>
#include	<sys/stropts.h>	/* mu0	*/
#include	<termio.h>
#include	<fcntl.h>
#include	<sys/gctio.h>		/* extended gcp tty ioctl sw0	*/
#include	<signal.h>
#include	<sys/stat.h>
#include	<utmp.h>
#include	<sys/crtctl.h>
#include	<sys/utsname.h>
#include	<ctype.h>
#include	<errno.h>

#ifdef SECON
#include <sys/synch.h>
#include <sys/security.h>
#include <sys/priv.h>
#include <sys/mls.h>
#include <sys/audit.h>
#include <sys/sakioctl.h>
#include <sys/termios.h>
#include <auth.h>
#endif

#define		TRUE		1
#define		FALSE		0
#define		FAILURE		-1

#define		SUCCESS		0
#define		ID		1
#define		IFLAGS		2
#define		FFLAGS		3
#define		MESSAGE		4
#define		NEXTID		5

#define		ACTIVE		1
#define		FINISHED	0

#define		ABORT		0177		/* Delete */
#define		QUIT		('\\'&037)	/* ^\ */
#define		ERASE		'#'
#define		BACKSPACE	'\b'
#define		KILL		'@'

#ifndef TIMEOUT
#define TIMEOUT 0
#endif

#ifdef OSS
/*	The following three characters are the standard OSS erase,	*/
/*	kill, and abort characters.					*/

#define		STDERASE	'_'
#define		STDKILL		'$'
#define		STDABORT	'&'
#endif	/* OSS */

#define		control(x)	(x&037)

#define		GOODNAME	1
#define		NONAME		0
#define		BADSPEED	-1

#ifndef		fioctl
#define		fioctl(x,y,z)	ioctl(fileno(x),y,z)
#endif

#define CONSOLE "/dev/console"

struct Gdef {
	char	*g_id;	/* identification for modes & speeds */
	struct termio	g_iflags;	/* initial terminal flags */
	struct termio	g_fflags;	/* final terminal flags */
	char	*g_message;	/* login message */
	char	*g_nextid;	/* next id if this speed is wrong */
};

#define	MAXIDLENGTH	15	/* Maximum length the "g_id" and "g_nextid" \
				 * strings can take.  Longer ones will be \
				 * truncated. \
				 */
#define MAXMESSAGE	79	/* Maximum length the "g_message" string \
				 * can be.  Longer ones are truncated. \
				 */

/*	Maximum length of line in /etc/gettydefs file and the maximum	*/
/*	length of the user response to the "login" message.		*/

#define	MAXLINE		255
#define	MAXARGS		64	/* Maximum number of arguments that can be \
				 * passed to "login" \
				 */

struct Symbols {
	char		*s_symbol;	/* Name of symbol */
	unsigned	s_value	;	/* Value of symbol */
};

/*	The following four symbols define the "SANE" state.		*/

#define	ISANE	(BRKINT|IGNPAR|ISTRIP|ICRNL|IXON)
#define	OSANE	(OPOST|ONLCR)
#define	CSANE	(CS7|PARENB|CREAD)
#define	LSANE	(ISIG|ICANON|ECHO|ECHOK)

/*	Modes set with the TCSETAW ioctl command.			*/

struct Symbols imodes[] = {
	"IGNBRK",	IGNBRK,
	"BRKINT",	BRKINT,
	"IGNPAR",	IGNPAR,
	"PARMRK",	PARMRK,
	"INPCK",	INPCK,
	"ISTRIP",	ISTRIP,
	"INLCR",	INLCR,
	"IGNCR",	IGNCR,
	"ICRNL",	ICRNL,
	"IUCLC",	IUCLC,
	"IXON",	IXON,
	"IXANY",	IXANY,
	"IXOFF",	IXOFF,
	NULL,	0
};

struct Symbols omodes[] = {
	"OPOST",	OPOST,
	"OLCUC",	OLCUC,
	"ONLCR",	ONLCR,
	"OCRNL",	OCRNL,
	"ONOCR",	ONOCR,
	"ONLRET",	ONLRET,
	"OFILL",	OFILL,
	"OFDEL",	OFDEL,
	"NLDLY",	NLDLY,
	"NL0",	NL0,
	"NL1",	NL1,
	"CRDLY",	CRDLY,
	"CR0",	CR0,
	"CR1",	CR1,
	"CR2",	CR2,
	"CR3",	CR3,
	"TABDLY",	TABDLY,
	"TAB0",	TAB0,
	"TAB1",	TAB1,
	"TAB2",	TAB2,
	"TAB3",	TAB3,
	"BSDLY",	BSDLY,
	"BS0",	BS0,
	"BS1",	BS1,
	"VTDLY",	VTDLY,
	"VT0",	VT0,
	"VT1",	VT1,
	"FFDLY",	FFDLY,
	"FF0",	FF0,
	"FF1",	FF1,
	NULL,	0
};

struct Symbols cmodes[] = {
	"B0",		B0,
	"B50",	B50,
	"B75", 	B75,
	"B110",	B110,
	"B134",	B134,
	"B150",	B150,
	"B200",	B200,
	"B300",	B300,
	"B600",	B600,
	"B1200",	B1200,
	"B1800",	B1800,
	"B2400",	B2400,
	"B4800",	B4800,
	"B9600",	B9600,
 	"B19200",	B19200,
 				/* the following aliases should be deleted  */
 				/* when cleaned out of the source */
 	"EXTA",	EXTA,		/* same as B19200 */
 	"EXTB",	EXTB,		/* same as B38400 */
 				/**/
	"CS5",	CS5,
	"CS6",	CS6,
	"CS7",	CS7,
	"CS8",	CS8,
	"CSTOPB",	CSTOPB,
	"CREAD",	CREAD,
	"PARENB",	PARENB,
	"PARODD",	PARODD,
	"HUPCL",	HUPCL,
	"CLOCAL",	CLOCAL,
	NULL,	0
};

struct Symbols lmodes[] = {
	"ISIG",	ISIG,
	"ICANON",	ICANON,
	"XCASE",	XCASE,
	"ECHO",	ECHO,
	"ECHOE",	ECHOE,
	"ECHOK",	ECHOK,
	"ECHONL",	ECHONL,
	"NOFLSH",	NOFLSH,
	NULL,	0
};

/*	Terminal types set with the LDSETT ioctl command.		*/

struct Symbols terminals[] = {
	"none",		TERM_NONE,
#ifdef	TERM_V10
	"vt100",		TERM_V10,
#endif
#ifdef	TERM_H45
	"hp45",		TERM_H45,
#endif
#ifdef	TERM_C10
	"c100",		TERM_C100,
#endif
#ifdef	TERM_TEX
	"tektronix",	TERM_TEX,
	"tek",		TERM_TEX,
#endif
#ifdef	TERM_D40
	"ds40-1",		TERM_D40,
#endif
#ifdef	TERM_V61
	"vt61",		TERM_V61,
#endif
#ifdef	TERM_TEC
	"tec",		TERM_TEC,
#endif
	NULL,		0
};

/*	Line disciplines set by the TIOCSETD ioctl command.		*/

#ifndef	LDISC0
#define	LDISC0	0
#endif
#ifndef	LDISC1
#define	LDISC1	1
#endif
#ifndef	LDISC2
#define	LDISC2	2
#endif
#ifndef	LDISC3
#define	LDISC3	3
#endif
#ifndef	LDISC4
#define	LDISC4	4
#endif

struct Symbols linedisc[] = {
	"LDISC0",	LDISC0,
	"LDISC1",	LDISC1,
	"LDISC2",	LDISC2,
	"LDISC3",	LDISC3,
	"LDISC4",	LDISC4,
	NULL,		0
};

/*	If the /etc/gettydefs file can't be opened, the following	*/
/*	default is used.						*/

struct Gdef DEFAULT = {
	"default",
	{
		ICRNL,0,B300+CREAD+HUPCL,0,
		LDISC0,
			{
				ABORT,QUIT,ERASE,KILL,'\0','\0','\0','\0'
			},
	},
	{
		ICRNL,OPOST+ONLCR+NLDLY+TAB3,B300+CS7+CREAD+HUPCL,
		ISIG+ICANON+ECHO+ECHOE+ECHOK,
		LDISC0,
			{
				ABORT,QUIT,ERASE,KILL,'\0','\0','\0','\0'
			},
	},
	"LOGIN: ",
	"default"
};

#ifndef	DEBUG
char	*CTTY		=	"/dev/syscon";
#else
char	*CTTY		=	"/dev/sysconx";
#endif

char	*ISSUE_FILE	=	"/etc/issue";
char	*GETTY_DEFS	=	"/etc/gettydefs";

int 	check = {
	FALSE
};
char	*checkgdfile;		/* Name of gettydefs file during
				 * check mode.
				 */
struct gctio		i_gctio, f_gctio; /* sw0 */


#define	STRFLG_SAVE	0x01	/* save tty state before a push		*/
#define	STRFLG_FORCE	0x02	/* force push of already pushed module	*/

typedef	struct	strlisttab	/* mu1					*/
	{
		char	*name;	/* module name				*/
		int	flags;	/* see above defines for definitions	*/
	} strlisttab_t;

strlisttab_t	strlisttab[64];	/* be liberal and don't check bounds	*/

int		strsaveall;	/* save tty settings for all modules	*/

char		*defstrname = "ldterm";	/* default module to push	*/

char *env_var[MAXARGS];
#ifdef SECON
/* JTOF - i hate globals, but these are for the timedout routine
 *        so he can tell the user what he's doing
 */
uint		sakset;
char		ttyn[PATH_MAX];
ttyauth_t	*ttya;
char lockmsg[] = "\n\
**********************************************\n\
                    SORRY                     \n\
         The console has been locked !!       \n\
 (Only the security administrator may log in) \n\
**********************************************\n";
#endif    /* SECON */



main(argc,argv,envp)
register int	argc;
register char	**argv, **envp;
{
	char *line;
	register struct Gdef *speedef;
	char oldspeed[MAXIDLENGTH+1],newspeed[MAXIDLENGTH+1];
	extern struct Gdef *find_def();
	int termtype,lined;
	extern char *ISSUE_FILE,*GETTY_DEFS;
	extern int check;
	int hangup,timeout;
	extern char *checkgdfile;
	extern struct Symbols *search(),terminals[],linedisc[];
	extern int timedout();
	register struct Symbols *answer;
	char user[MAXLINE],*largs[MAXARGS],*ptr,buffer[MAXLINE];
	int indvar = 0;  /* index for -E args */
	FILE *fp;
	FILE *fdup();
#ifdef SECON
	ttyauth_t *ta, *opendevauth();
#endif    /* SECON */
	struct utsname utsname;
	struct termcb termcb;
	struct termio termio;
	static char clrscreen[2] = {
		ESC,CS
	};
	register strlisttab_t	*strlistptr = strlisttab;
	register int		strforce;
	int timeoutset = 0;

	signal(SIGINT,SIG_IGN);
	signal(SIGQUIT,SIG_DFL);

	hangup = TRUE;
	timeout = TIMEOUT;

	/* take the option '-t', '-c', '-h' and the line */
	while(--argc && **++argv == '-'){
	      for(ptr= *argv + 1; *ptr; ptr++) switch(*ptr){
		case 'c' :
			/* Check a "gettydefs" file mode. */
			signal(SIGINT,SIG_DFL);
			if(--argc == 0) { 
				fprintf(stderr,
				"Check Mode Usage: getty -c gettydefs-like-file\n");
				exit(1);
			}
			check = TRUE;
			checkgdfile = *++argv;

	/* Attempt to open the check gettydefs file. */
			if((fp = fopen(checkgdfile,"r")) == NULL) {
				fprintf(stderr,"Cannot open %s\n",checkgdfile);
				exit(1);
			}
			fclose(fp);

/* Call "find_def" to check the check file.  With the "check" flag */
/* set, it will parse the entire file, printing out the results. */
			find_def(NULL);
			exit(0);

			case 'h':
			       hangup = FALSE;
			       break;
 
			case 't':
			if(isdigit(*++ptr)) {
				sscanf(ptr,"%d",&timeout);

/* Advance "ptr" so that it is pointing to the last digit of the */
/* timeout argument. */
				while(isdigit(*++ptr));
				ptr--;
			} else if(--argc) {
						if(isdigit(*(ptr = *++argv)))
						sscanf(ptr,"%d",&timeout);
				else error("getty: timeout argument invalid. \"%s\"\n\r", *argv);
			}
			timeoutset = 1;
			break;
                 default:
			fprintf(stderr,"Illegal option -%c\n", *ptr );
                        error("getty: no terminal line specified.\n\r");
			sleep(20);
                        exit(1);
          }  /* end of switch */
        } /* endwhile argc && argv= '-' */
		if(argc < 1 ) 
                { 
                        error("getty: no terminal line specified.\n\r");
			sleep(20);
			exit(1);
	        }else line = *argv;     

	        lined = LDISC0;      /* put a defautl in linediscipline */
	        termtype = TERM_NONE;  /* put no terminal type specified */
	        if(argc > 1) {
			 --argc;
			 ++argv;
			 if (**argv == '-') {
			     speedef = find_def(NULL); /* no speed specified */
			 } else {
                            /* check the speed specified */
			    if((speedef = find_def(*argv)) == NULL) {
				error("getty: unable to find %s in \"%s\".\n\r",
                                *argv,GETTY_DEFS);
				/* use the default instead */
				speedef = find_def(NULL);
			    }
			argv++;
	                }
       }
        /* treat the optional arguments for stream modules and env.vars */
        /* the maximun number of streams modules is 64 */
               for( ; --argc > 0; argv++ )
               {
	          if(**argv == '-')
                  {
		    switch(*(*argv+1))
                    {
		    case 'O':
			  if( --argc < 1) 
                          {
			    error("getty: no stream module specified.\n\r");
			    sleep(20);
			    exit(1);
			  } else   {
				   strlistptr->name = *++argv;
				   strlistptr->flags = strforce;
				   strlistptr++;
				   }
			   break;
		     case 'E':
			  if(--argc < 1) 
                          {
			    error("getty: no environment variable specified.\n\r");
			    sleep(20);
			    exit(1);
			  } else {
				    /* We nee to stick in a -E so it is passed to
					* login when we exec it. 
					*/
				    if (!indvar) 
						env_var[indvar++] = "-E"; 
				    env_var[indvar++] = *++argv;
				  }
                          break;
		     case 'F':
			   strforce = STRFLG_FORCE;
			   break;
		     case 'f':
			  strforce = 0;
			  break;
		     case 'S':
			  strsaveall = 1;
			  break;
		     case 's':
			  if(--argc < 1) 
                          {
			    error("getty: no stream module specified.\n\r");
			    sleep(20);
			    exit(1);
			  } else {
				     strlistptr->name= *++argv;
				     strlistptr->flags = STRFLG_SAVE | strforce;
				     strlistptr++;
				  }
                          break;
		     default :
			   break;
		    }                                 /* end switch */
                } else   {
			 strlistptr->name = *argv;
			 strlistptr->flags = strforce;
			 strlistptr++;
			 }
	      }                                     /* end looping for */

/* Perform "utmp" accounting. */
	account(line);

/* Attempt to open standard input, output, and error on specified line. */
	chdir("/dev");
#ifdef SECON
	/* export name for msg routine */
	ta = opendevauth(line);
	ttya = ta;
	openline(line,speedef,termtype,lined,hangup,ta);
	/* JTOF - for now, exempt the console cause all the
	 *        deamons must be started from there
	 */
	if ( strcmp(line, "console" )) {
		kill_cterm_spfrs(line);
		/* kill_good_spfrs(line); */
	}
#else    /* SECON */
	openline(line,speedef,termtype,lined,hangup);

/* Loop until user is successful in requesting login. */
#endif    /* SECON */
	for(;;) {

/* If there is no terminal type, just advance a line. */
		if(termtype == TERM_NONE) {

/* A bug in the stdio package requires that the first output on */
/* the newly reopened stderr stream be a putc rather than an */
/* fprintf. */
			putc('\r',stderr);
			putc('\n',stderr);

/* If there is a terminal type, clear the screen with the common */
/* crt language.  Note that the characters have to be written in */
/* one write, and hence can't go through standard io, which is */
/* currently unbuffered. */
		} else write(fileno(stderr),clrscreen,sizeof(clrscreen));

/* If getty is supposed to die if no one logs in after a */
/* predetermined amount of time, set the timer. */
#ifdef SECON
		if(timeoutset && timeout && sakset) {
#else
		if(timeout) {
#endif
			signal(SIGALRM,timedout);
			alarm(timeout);
		}

#ifdef SYS_NAME
/* Generate a message with the system identification in it. */
		if (uname(&utsname) != FAILURE) {
			sprintf(buffer,"%.9s\r\n", utsname.nodename);

#ifdef	UPPERCASE_ONLY
/* Make all the alphabetics upper case. */
			for (ptr= buffer; *ptr;ptr++) *ptr = tolower(*ptr);
#endif	/* UPPERCASE_ONLY */
			fputs(buffer,stderr);
		}

/* Print out the issue file. */
		if ((fp = fopen(ISSUE_FILE,"r")) != NULL) {
			while ((ptr = fgets(buffer,sizeof(buffer),fp)) != NULL) {
				fputs(ptr,stderr);

/* In "raw" mode, a carriage return must be supplied at the end of */
/* each line. */
				putc('\r',stderr);
			}
			fclose(fp);
		}
#endif	/* SYS_NAME */

#ifdef SECON
		/* This *must* be the console since opendevauth() would */
		/* have locked a terminal with ta_status. */
		/* If the console is locked, print a message. */
		if ( ta && ta->ta_status )
			fprintf(stderr, "%s", lockmsg);
#endif

/* Print the login message. */
		fprintf(stderr,"%s",speedef->g_message);

/* Get the user's typed response and respond appropriately. */
		switch(getname(user,&termio)) {
		case GOODNAME:
			if (timeout) alarm(0);
/* If a terminal type was specified, keep only those parts of */
/* the gettydef final settings which were not explicitely turned */
/* on when the terminal type was set. */
			if (termtype != TERM_NONE) {
				termio.c_iflag |= (ISTRIP|ICRNL|IXON|IXANY)
					| (speedef->g_fflags.c_iflag
						& ~(ISTRIP|ICRNL|IXON|IXANY));
				termio.c_oflag |= (OPOST|ONLCR)
					| (speedef->g_fflags.c_oflag
						& ~(OPOST|ONLCR));
				termio.c_cflag = speedef->g_fflags.c_cflag;
				termio.c_lflag = (ISIG|ICANON|ECHO|ECHOE|ECHOK)
					| (speedef->g_fflags.c_lflag
						& ~(ISIG|ICANON|ECHO|ECHOE|ECHOK));
			} else {
				termio.c_iflag |= speedef->g_fflags.c_iflag;
				termio.c_oflag |= speedef->g_fflags.c_oflag;
				termio.c_cflag |= speedef->g_fflags.c_cflag;
				termio.c_lflag |= speedef->g_fflags.c_lflag;
			}
			termio.c_line = lined;
			fioctl(stdin,TCSETAW,&termio);

			fioctl(stdin,GCTIO_SET,&f_gctio); /* sw0 */

/* Parse the input line from the user, breaking it at white spaces. */
			largs[0] = "login";
			parse(user,&largs[1],MAXARGS-1);
                        
/* Exec "login". */

#ifndef	DEBUG
 			execve("/bin/login",largs,envp);
			exit(1);
#else
			exit(0);
#endif

/* If the speed supplied was bad, try the next speed in the list. */
		case BADSPEED:

/* Save the name of the old speed definition incase new one is */
/* bad.  Copy the new speed out of the static so that "find_def" */
/* won't overwrite it in the process of looking for new entry. */
			strcpy(oldspeed,speedef->g_id);
			strcpy(newspeed,speedef->g_nextid);
			if ((speedef = find_def(newspeed)) == NULL) {
				error("getty: pointer to next speed in entry %s is bad.\n\r",
				oldspeed);

/* In case of error, go back to the original entry. */
				if((speedef = find_def(oldspeed)) == NULL) {

/* If the old entry has disappeared, then quit and let next "getty" try. */
					error("getty: unable to find %s again.\n\r",
						oldspeed);
					exit(1);
				}
			}

/* Setup the terminal for the new information. */
			setupline(speedef,termtype,lined);
			break;

/* If no name was supplied, not nothing, but try again. */
		case NONAME:
			break;
		}
	}
}

account(line)
char *line;
{
	register int ownpid;
	register struct utmp *u;
	extern struct utmp *getutent(), *pututline();
	register FILE *fp;

/* Look in "utmp" file for our own entry and change it to LOGIN. */
	ownpid = getpid();

	while ((u = getutent()) != NULL || (!u && errno == EINVAL) ) {
		
		/* This is if the utmp file is garbage, we must remove
		 * it, or getty will never work .
		 */
		if (!u)  {
			unlink(UTMP_FILE);
			continue;
		}

		/* Is this our own entry? */
		if (u->ut_type == INIT_PROCESS && u->ut_pid == ownpid) {
			strncpy(u->ut_line,line,sizeof(u->ut_line));
			strncpy(u->ut_user,"LOGIN",sizeof(u->ut_user));
			u->ut_type = LOGIN_PROCESS;

			/* Write out the updated entry. */
			pututline(u);
			break;
		}
	}

/* If we were successful in finding an entry for ourself in the */
/* utmp file, then attempt to append to the end of the wtmp file. */
	if (u != NULL && (fp = fopen(WTMP_FILE,"r+")) != NULL) {
		fseek(fp,0L,2);	/* Seek to end of file */
		fwrite(u,sizeof(*u),1,fp);
		fclose(fp);
	}

/* Close the utmp file. */
	endutent();
}

/*	"search" scans through a table of Symbols trying to find a	*/
/*	match for the supplied string.  If it does, it returns the	*/
/*	pointer to the Symbols structure, otherwise it returns NULL.	*/

struct Symbols *search(target,symbols)
register char *target;
register struct Symbols *symbols;
{

/* Each symbol array terminates with a null pointer for an */
/* "s_symbol".  Scan until a match is found, or the null pointer */
/* is reached. */
	for (;symbols->s_symbol != NULL; symbols++)
		if (strcmp(target,symbols->s_symbol) == 0) return(symbols);
	return(NULL);
}

error(format,arg1,arg2,arg3,arg4)
char *format;
int arg1,arg2,arg3,arg4;
{
	register FILE *fp;

	if ((fp = fopen(CTTY,"w")) == NULL)
		return;
	fprintf(fp,format,arg1,arg2,arg3,arg4);
	fflush(fp);
	fclose(fp);
}

#ifdef SECON
openline(line,speedef,termtype,lined,hangup,ta)
register char *line;
register struct Gdef *speedef;
int termtype,lined,hangup;
ttyauth_t *ta;
#else    /* SECON */
openline(line,speedef,termtype,lined,hangup)
register char *line;
register struct Gdef *speedef;
int termtype,lined,hangup;
#endif    /* SECON */
{
	register FILE *fpin,*fp;
	struct stat statb;
	int fd, tmp_fd;
	extern int errno;
	struct termio termio;	/* mu1	*/
	struct termcb termcb;	/* mu1	*/
	unsigned char timer;	/* MUST be unsigned mu1	*/
	register strlisttab_t	*strlistptr;	/* mu1	*/
	register int		strfind, stronesuccess;	/* mu2 */

	close(0);
	close(1);
	close(2);
	fclose(stdin);
	fclose(stdout);
	fclose(stderr);

#ifdef SECON
	protect(line,ta);
#else    /* SECON */
	/* Change the ownership of the terminal line to root and set 
	 * the protections to only allow root to read the line. 
	 */
	stat(line,&statb);
	chown(line,0,statb.st_gid);
	chmod(line,0622);
#endif    /* SECON */

#ifdef _POSIX_SOURCE
	setsid();
#else
	setpgrp();
#endif

/* setup initial timer values now, instead of after fopen completes  eb0 */
	if ((fd=open(line,O_RDWR|O_NDELAY)) == -1) {
		error("getty: cannot open \"%s\". errno: %d\n\r",line,errno);
		sleep(20);
		exit(1);
	}
	ioctl(fd,GCTIO_SET,&i_gctio); /* eb0 */
	close(fd);

/* Attempt to open the line.  It should become "stdin".  If not, */
/* then close. */
	if (fopen(line,"r+") == NULL) {
		error("getty: cannot open \"%s\". errno: %d\n\r",line,errno);
		sleep(20);
		exit(1);
	}
	fdup(stdin);
	fdup(stdin);
	setbuf(stdin,NULL);
	setbuf(stdout,NULL);
	setbuf(stderr,NULL);

/* Unless getty is being invoked by ct, make sure that DTR has been */
/* dropped and reasserted */
	if (hangup) hang_up_line();

	/*
	 *	mu1
	 *	Push all stream modules specified on invocation line.
	 *	Do this before calling setupline() so that the streams
	 *	module responds to the setupline() settings.
	 */
	strlistptr = &strlisttab[0];
	if (strlistptr->name == NULL) {
	} else	{
		stronesuccess = 0;
		if (strsaveall) {
			fioctl(stdin,LDGETT,&termcb);
			fioctl(stdin,TCGETA,&termio);
		}
		while (strlistptr->name != NULL) {
			strfind = fioctl(stdin,I_FIND,strlistptr->name);
			if (strfind < 0) {
		error("getty: stream module '%s' not supported (errno=%d)\n\r",
					strlistptr->name,errno);
				strlistptr++;
				continue;
			}
			if (strfind > 0) {
			error("getty: stream module '%s' is already pushed\n\r",
					strlistptr->name);
				if (!(strlistptr->flags & STRFLG_FORCE)) {
					strlistptr++;
					continue;
				}
		error("getty: force another push of stream module '%s'\n\r",
					strlistptr->name);
			}
			if (strlistptr->flags & STRFLG_SAVE) {
				fioctl(stdin,LDGETT,&termcb);
				fioctl(stdin,TCGETA,&termio);
				strsaveall = 0;	/* just in case	*/
			}
			if (fioctl(stdin,I_PUSH,strlistptr->name) < 0) {
		error("getty: can't push stream module '%s' (errno=%d)\n\r",
					strlistptr->name,errno);
				strlistptr++;
				continue;
			}
			stronesuccess++;
			if (strlistptr->flags & STRFLG_SAVE) {
				fioctl(stdin,LDSETT,&termcb);
				fioctl(stdin,TCSETAF,&termio);
				/*
				 *	Pause very, very briefly while
				 *	terminal settles.
				 */
				for (timer = 0; ++timer != 0; );
			}
			strlistptr++;
		}
		if (strsaveall && stronesuccess) {
			fioctl(stdin,LDSETT,&termcb);
			fioctl(stdin,TCSETAF,&termio);
			/*
			 *	Pause very, very briefly while terminal settles.
			 */
			for (timer = 0; ++timer != 0; );
		}
	}

/* Set the terminal type and line discipline. */
	setupline(speedef,termtype,lined);

#ifdef SECON
	setsak(line, ta);
#endif    /* SECON */
}

#ifdef HANGUP

hang_up_line()
{
	struct termio termio;

	fioctl(stdin,TCGETA,&termio);
	termio.c_cflag &= ~CBAUD;
	termio.c_cflag |= B0;
	fioctl(stdin,TCSETAF,&termio);
	sleep(1);
}
#else
hang_up_line()
{
}
#endif	/* HANGUP */

timedout()
{
#ifdef SECON
int i, fd;
char buf[PATH_MAX];

	if ( sakset ) {
		/* catch all signals to be sent */
		for (i=0;i<MAXSIG;i++)
			signal( i, SIG_IGN );
		signal( SIGSAK, SIG_DFL );
		msg( "\r\n" );
		msg( "*************** Security Timeout *****************\r\n" );
		msg( "Use Secure Attention Key sequence to restart login\r\n" );
		msg( "**************************************************\r\n" );
		while (1)
			sleep(999);
	}
	else
		exit(1);
#else    /* SECON */
	exit(1);
#endif    /* SECON */
}

setupline(speedef,termtype,lined)
register struct Gdef *speedef;
int termtype,lined;
{
	struct termio termio;
	struct termcb termcb;
	unsigned short timer;

/* Set the terminal type to "none", which will clear all old */
/* special flags, if a terminal type was set from before. */
	termcb.st_flgs = 0;
	termcb.st_termt = TERM_NONE;
	termcb.st_vrow = 0;
	fioctl(stdin,LDSETT,&termcb);
	termcb.st_termt = termtype;
	fioctl(stdin,LDSETT,&termcb);

/* Get the current state of the modes and such for the terminal. */
	fioctl(stdin,TCGETA,&termio);
	if (termtype != TERM_NONE) {

/* If there is a terminal type, take away settings so that */
/* terminal is "raw" and "no echo".  Also take away the orginal */
/* speed setting. */
		termio.c_iflag = 0;
		termio.c_cflag &= ~(CSIZE|PARENB|CBAUD);
		termio.c_cflag |= CS8|CREAD|HUPCL;
		termio.c_lflag &= ~(ISIG|ICANON|ECHO|ECHOE|ECHOK);

/* Add in the speed. */
		termio.c_cflag |= (speedef->g_iflags.c_cflag & CBAUD);
	} else {
		termio.c_iflag = speedef->g_iflags.c_iflag;
		termio.c_oflag = speedef->g_iflags.c_oflag;
		termio.c_cflag = speedef->g_iflags.c_cflag;
		termio.c_lflag = speedef->g_iflags.c_lflag;
	}

/* Make sure that raw reads are 1 character at a time with no */
/* timeout. */
	termio.c_cc[VMIN] = 1;
	termio.c_cc[VTIME] = 0;

/* Add the line discipline. */
	termio.c_line = lined;

	fioctl(stdin,TCSETAF,&termio);

/* Pause briefly while terminal settles. */
	for(timer=0; ++timer != 0;);
}

/*	"getname" picks up the user's name from the standard input.	*/
/*	It makes certain						*/
/*	determinations about the modes that should be set up for the	*/
/*	terminal depending upon what it sees.  If it sees all UPPER	*/
/*	case characters, it sets the IUCLC & OLCUC flags.  If it sees	*/
/*	a line terminated with a <linefeed>, it sets ICRNL.  If it sees	*/
/*	the user using the "standard" OSS erase, kill, abort, or line	*/
/*	termination characters ( '_','$','&','/','!' respectively)	*/
/*	it resets the erase, kill, and end of line characters.		*/

int getname(user,termio)
char *user;
struct termio *termio;
{
	register char *ptr,c;
	register int rawc;
	int upper,lower;

/* Get the previous modes, erase, and kill characters and speeds. */
	fioctl(stdin,TCGETA,termio);

/* Set the flags to 0 and the erase and kill to the standard */
/* characters. */
	termio->c_iflag &= ICRNL;
	termio->c_oflag = 0;
	termio->c_cflag = 0;
	termio->c_lflag &= ECHO;
	for (ptr= (char*)termio->c_cc; ptr < (char*)&termio->c_cc[NCC];)
		*ptr++ = NULL;
	termio->c_cc[VINTR] = ABORT;
	termio->c_cc[VQUIT] = QUIT;
	termio->c_cc[VERASE] = ERASE;
	termio->c_cc[VKILL] = KILL;
	termio->c_cc[VEOF] = ('D'&037);
	termio->c_cc[VSWTCH] = CSWTCH;
	ptr = user;
	upper = 0;
	lower = 0;
	do {

/* If it isn't possible to read line, exit. */
		if ((rawc = getc(stdin)) == EOF) exit(0);

/* If a null character was typed, return 0. */
		if ((c = (rawc & 0177)) == '\0') return(BADSPEED);

/* Echo the character if ECHO is off. */
		if( (termio->c_lflag&ECHO) == 0 )
			putc(rawc,stdout);
#ifdef OSS
		if (c == ERASE || c == BACKSPACE) {

/* Store this character as the "erase" character. */
			termio->c_cc[VERASE] = c;
#else
		if (c == ERASE) {
#endif	/* OSS */

/* If there is anything to erase, erase a character. */
			if (ptr > user) --ptr;
		}

#ifdef OSS
		else if (c == STDERASE) {
			if (ptr > user) --ptr;

/* Set up the "standard OSS" erase, kill, etc. characters. */
			termio->c_cc[VINTR] = STDABORT;
			termio->c_cc[VERASE] = STDERASE;
			termio->c_cc[VKILL] = STDKILL;
			termio->c_cc[VEOL] = '/';
			termio->c_cc[VEOL2] = '!';
		}
#endif	/* OSS */

/* If the character is a kill line or abort character, reset the */
/* line. */
		else if (c == KILL || c == ABORT || c == control('U')) {
			ptr = user;
			fputs("\r\n",stdout);

/* Make sure the erase, kill, etc. are set to the UNIX standard. */
			termio->c_cc[VINTR] = ABORT;
			termio->c_cc[VERASE] = ERASE;
			termio->c_cc[VKILL] = KILL;
			termio->c_cc[VEOL] = '\0';
			termio->c_cc[VEOL2] = '\0';
		}

#ifdef OSS
		else if (c == STDKILL || c == STDABORT) {
			ptr = user;

/* Set up the "standard OSS" erase, kill, etc. characters. */
			termio->c_cc[VINTR] = STDABORT;
			termio->c_cc[VERASE] = STDERASE;
			termio->c_cc[VKILL] = STDKILL;
			termio->c_cc[VEOL] = '/';
			termio->c_cc[VEOL2] = '!';
		}
#endif	/* OSS */

/* If the character is lower case, increment the flag for lower case. */
		else if (islower(c)) {
			lower++;
			*ptr++ = c;
		}

/* If the character is upper case, increment the flag. */
		else if (isupper(c)) {
			upper++;
			*ptr++ = c;
		}

/* Just store all other characters. */
		else *ptr++ = c;
	}

/* Continue the above loop until a line terminator is found or */
/* until user name array is full. */

#ifdef OSS
	while (c != '\n' && c != '\r' && c != '/' && c != '!' &&
	    ptr < (user + MAXLINE));
#else
	while (c != '\n' && c != '\r'
	    && ptr < (user + MAXLINE));
#endif	/* OSS */

/* Remove the last character from name. */
	*--ptr = '\0';
	if (ptr == user) return(NONAME);

#ifdef OSS
/* If the line was terminated with one of the printing OSS line */
/* termination characters or is a <cr>, add a <newline>. */
	if (c == '/' || c == '!') {
		putc('\n',stdout);

/* Set up the "standard OSS" erase, kill, etc. characters. */
		termio->c_cc[VINTR] = STDABORT;
		termio->c_cc[VERASE] = STDERASE;
		termio->c_cc[VKILL] = STDKILL;
		termio->c_cc[VEOL] = '/';
		termio->c_cc[VEOL2] = '!';
	} else
#endif	/* OSS */
		if (c == '\r') putc('\n',stdout);

/* If the line terminated with a <lf>, put ICRNL and ONLCR into */
/* into the modes. */
	if (c == '\r') {
		termio->c_iflag |= ICRNL;
		termio->c_oflag |= ONLCR;

/* When line ends with a <lf>, then add the <cr>. */
	} else putc('\r',stdout);

/* Set the upper-lower case conversion switchs if only upper */
/* case characters were seen in the login and no lower case. */
/* Also convert all the upper case characters to lower case. */

	if (upper > 0 && lower == 0) {
		termio->c_iflag |= IUCLC;
		termio->c_oflag |= OLCUC;
		termio->c_lflag |= XCASE;
		for (ptr=user; *ptr; ptr++)
			if (*ptr >= 'A' && *ptr <= 'Z' ) *ptr += ('a' - 'A');
	}
	return(GOODNAME);
}

/*	"find_def" scans "/etc/gettydefs" for a string with the		*/
/*	requested "id".  If the "id" is NULL, then the first entry is	*/
/*	taken, hence the first entry must be the default entry.		*/
/*	If a match for the "id" is found, then the line is parsed and	*/
/*	the Gdef structure filled.  Errors in parsing generate error	*/
/*	messages on the system console.					*/

struct Gdef *find_def(id)
char *id;
{
	register struct Gdef *gptr;
	register char *ptr,c;
	FILE *fp;
	int i,input,state,size,rawc,field;
	char oldc,*optr,quoted(),*gdfile;
	char line[MAXLINE+1];
	static struct Gdef def;
	extern struct Gdef DEFAULT;
	static char d_id[MAXIDLENGTH+1],d_nextid[MAXIDLENGTH+1];
	static char d_message[MAXMESSAGE+1];
	extern char *GETTY_DEFS;
	extern char *getword(),*fields(),*speed();
	extern int check;
	extern char *checkgdfile;
	static char *states[] = {
		"","id","initial flags","final flags","message","next id"
	};

/* Decide whether to read the real /etc/gettydefs or the supplied */
/* check file. */
	if (check) gdfile = checkgdfile;
	else gdfile = GETTY_DEFS;

/* Open the "/etc/gettydefs" file.  Be persistent. */
	for (i=0; i < 3;i++) {
		if ((fp = fopen(gdfile,"r")) != NULL) break;
		else sleep(3);	/* Wait a little and then try again. */
	}

/* If unable to open, complain and then use the built in default. */
	if (fp == NULL) {
		error("getty: can't open \"%s\".\n\r",gdfile);
		return(&DEFAULT);
	}

/* Start searching for the line with the proper "id". */
	input = ACTIVE;
	do {
		for(ptr= line,oldc='\0'; ptr < &line[sizeof(line)] &&
		    (rawc = getc(fp)) != EOF; ptr++,oldc = c) {
			c = *ptr = rawc;

/* Search for two \n's in a row. */
			if (c == '\n' && oldc == '\n') break;
		}

/* If we didn't end with a '\n' or EOF, then the line is too long. */
/* Skip over the remainder of the stuff in the line so that we */
/* start correctly on next line. */
		if (rawc != EOF && c != '\n') {
			for (oldc='\0'; (rawc = getc(fp)) != EOF;oldc=c) {
				c = rawc;
				if (c == '\n' && oldc != '\n') break;
			}
			if (check) fprintf(stdout,"Entry too long.\n");
		}

/* If we ended at the end of the file, then if there is no */
/* input, break out immediately otherwise set the "input" */
/* flag to FINISHED so that the "do" loop will terminate. */
		if (rawc == EOF) {
			if (ptr == line) break;
			else input = FINISHED;
		}

/* If the last character stored was an EOF or '\n', replace it */
/* with a '\0'. */
		if (*ptr == (EOF & 0377) || *ptr == '\n') *ptr = '\0';

/* If the buffer is full, then make sure there is a null after the */
/* last character stored. */
		else *++ptr == '\0';
		if (check) fprintf(stdout,"\n**** Next Entry ****\n%s\n",line);

/* If line starts with #, treat as comment */
		if(line[0] == '#') continue;

/* Initialize "def" and "gptr". */
		gptr = &def;
		gptr->g_id = (char*)NULL;
		gptr->g_iflags.c_iflag = 0;
		gptr->g_iflags.c_oflag = 0;
		gptr->g_iflags.c_cflag = 0;
		gptr->g_iflags.c_lflag = 0;
		gptr->g_fflags.c_iflag = 0;
		gptr->g_fflags.c_oflag = 0;
		gptr->g_fflags.c_cflag = 0;
		gptr->g_fflags.c_lflag = 0;
		gptr->g_message = (char*)NULL;
		gptr->g_nextid = (char*)NULL;

/* Initialize gctio's   sw0 */
		i_gctio.gt_tdcd = 0; /* sw0 */
		i_gctio.gt_tact = 0; /* sw0 */
		f_gctio.gt_tdcd = 0; /* sw0 */
		f_gctio.gt_tact = 0; /* sw0 */

/* Now that we have the complete line, scan if for the various */
/* fields.  Advance to new field at each unquoted '#'. */
		for (state=ID,ptr= line; state != FAILURE && state != SUCCESS;) {
			switch(state) {
			case ID:

/* Find word in ID field and move it to "d_id" array. */
				strncpy(d_id,getword(ptr,&size),MAXIDLENGTH);
				gptr->g_id = d_id;

/* Move to the next field.  If there is anything but white space */
/* following the id up until the '#', then set state to FAILURE. */
				ptr += size;
				while (isspace(*ptr)) ptr++;
				if (*ptr != '#') {
					field = state;
					state = FAILURE;
				} else {
					ptr++;	/* Skip the '#' */
					state = IFLAGS;
				}
				break;

/* Extract the "g_iflags" */
			case IFLAGS:
				if ((ptr = fields(ptr,&gptr->g_iflags,&i_gctio)) == NULL) { /* sw0 */
					field = state;
					state = FAILURE;
				} else {
				  	gptr->g_iflags.c_iflag |= ICRNL; /*ca0*/
					if((gptr->g_iflags.c_cflag & CSIZE) == 0)
						gptr->g_iflags.c_cflag |= CS8;
					gptr->g_iflags.c_cflag |= CREAD|HUPCL;
					gptr->g_iflags.c_lflag &= ~(ISIG|ICANON
						|XCASE|ECHOE|ECHOK);
					ptr++;
					state = FFLAGS;
				}
				break;

/* Extract the "g_fflags". */
			case FFLAGS:
				if ((ptr = fields(ptr,&gptr->g_fflags,&f_gctio)) == NULL) { /* sw0 */
					field = state;
					state = FAILURE;
				} else {

/* Force the CREAD mode in regardless of what the user specified. */
					gptr->g_fflags.c_cflag |= CREAD;
					ptr++;
					state = MESSAGE;
				}
				break;

/* Take the entire next field as the "login" message. */
/* Follow usual quoting procedures for control characters. */
			case MESSAGE:
				for (optr= d_message; (c = *ptr) != '\0'
				    && c != '#';ptr++) {

/* If the next character is a backslash, then get the quoted */
/* character as one item. */
					if (c == '\\') {
						c = quoted(ptr,&size);
/* -1 accounts for ++ that takes place later. */
						ptr += size - 1;
					}

/* If there is room, store the next character in d_message. */
					if (optr < &d_message[MAXMESSAGE])
						*optr++ = c;
				}

/* If we ended on a '#', then all is okay.  Move state to NEXTID. */
/* If we didn't, then set state to FAILURE. */
				if (c == '#') {
					gptr->g_message = d_message;
					state = NEXTID;

/* Make sure message is null terminated. */
					*optr++ = '\0';
					ptr++;
				} else {
					field = state;
					state = FAILURE;
				}
				break;

/* Finally get the "g_nextid" field.  If this is successful, then */
/* the line parsed okay. */
			case NEXTID:

/* Find the first word in the field and save it as the next id. */
				strncpy(d_nextid,getword(ptr,&size),MAXIDLENGTH);
				gptr->g_nextid = d_nextid;

/* There should be nothing else on the line.  Starting after the */
/* word found, scan to end of line.  If anything beside white */
/* space, set state to FAILURE. */
				ptr += size;
				while (isspace(*ptr)) ptr++;
				if (*ptr != '\0') {
					field = state;
					state = FAILURE;
				} else state = SUCCESS;
				break;
			}
		}

/* If a line was successfully picked up and parsed, compare the */
/* "g_id" field with the "id" we are looking for. */
		if (state == SUCCESS) {

/* If there is an "id", compare them. */
			if (id != NULL) {
				if (strcmp(id,gptr->g_id) == 0) {
					fclose(fp);
					return(gptr);
				}

/* If there is no "id", then return this first successfully */
/* parsed line outright. */
			} else if (check == FALSE) {
				fclose(fp);
				return(gptr);

/* In check mode print out the results of the parsing. */
			} else {
				fprintf(stdout,"id: %s\n",gptr->g_id);
				fprintf(stdout,"initial flags:\niflag- %o oflag- %o cflag- %o lflag- %o\n",
					gptr->g_iflags.c_iflag,
					gptr->g_iflags.c_oflag,
					gptr->g_iflags.c_cflag,
					gptr->g_iflags.c_lflag);
				if(i_gctio.gt_tdcd || i_gctio.gt_tact) /* sw0 */
				fprintf(stdout,"  tdcd - %d  tact- %d\n",
					i_gctio.gt_tdcd, i_gctio.gt_tact);
				else fprintf(stdout,"\n");
				fprintf(stdout,"final flags:\niflag- %o oflag- %o cflag- %o lflag- %o\n",
					gptr->g_fflags.c_iflag,
					gptr->g_fflags.c_oflag,
					gptr->g_fflags.c_cflag,
					gptr->g_fflags.c_lflag);
				if(f_gctio.gt_tdcd || f_gctio.gt_tact) /* sw0 */
				fprintf(stdout,"  tdcd - %d  tact- %d\n",
					f_gctio.gt_tdcd, f_gctio.gt_tact);
				else fprintf(stdout,"\n");
				fprintf(stdout,"message: %s\n",gptr->g_message);
				fprintf(stdout,"next id: %s\n",gptr->g_nextid);
			}

/* If parsing failed in check mode, complain, otherwise ignore */
/* the bad line. */
		} else if (check) {
			*++ptr = '\0';
			fprintf(stdout,"Parsing failure in the \"%s\" field\n\
%s<--error detected here\n",
				states[field],line);
		}
	} while (input == ACTIVE);

/* If no match was found, then return NULL. */
	fclose(fp);
	return(NULL);
}

char *getword(ptr,size)
register char *ptr;
int *size;
{
	register char *optr,c;
	char quoted();
	static char word[MAXIDLENGTH+1];
	int qsize;

/* Skip over all white spaces including quoted spaces and tabs. */
	for (*size=0; isspace(*ptr) || *ptr == '\\';) {
		if (*ptr == '\\') {
			c = quoted(ptr,&qsize);
			(*size) += qsize;
			ptr += qsize+1;

/* If this quoted character is not a space or a tab or a newline */
/* then break. */
			if (isspace(c) == 0) break;
		} else {
			(*size)++;
			ptr++;
		}
	}

/* Put all characters from here to next white space or '#' or '\0' */
/* into the word, up to the size of the word. */
	for (optr= word,*optr='\0'; isspace(*ptr) == 0 &&
	    *ptr != '\0' && *ptr != '#'; ptr++,(*size)++) {

/* If the character is quoted, analyze it. */
		if (*ptr == '\\') {
			c = quoted(ptr,&qsize);
			(*size) += qsize;
			ptr += qsize;
		} else c = *ptr;

/* If there is room, add this character to the word. */
		if (optr < &word[MAXIDLENGTH+1] ) *optr++ = c;
	}

/* Make sure the line is null terminated. */
	*optr++ = '\0';
	return(word);
}

/*	"quoted" takes a quoted character, starting at the quote	*/
/*	character, and returns a single character plus the size of	*/
/*	the quote string.  "quoted" recognizes the following as		*/
/*	special, \n,\r,\v,\t,\b,\f as well as the \nnn notation.	*/

char quoted(ptr,qsize)
char *ptr;
int *qsize;
{
	register char c,*rptr;
	register int i;

	rptr = ptr;
	switch(*++rptr) {
	case 'n':
		c = '\n';
		break;
	case 'r':
		c = '\r';
		break;
	case 'v':
		c = '\013';
		break;
	case 'b':
		c = '\b';
		break;
	case 't':
		c = '\t';
		break;
	case 'f':
		c = '\f';
		break;
	default:

/* If this is a numeric string, take up to three characters of */
/* it as the value of the quoted character. */
		if (*rptr >= '0' && *rptr <= '7') {
			for (i=0,c=0; i < 3;i++) {
				c = c*8 + (*rptr - '0');
				if (*++rptr < '0' || *rptr > '7') break;
			}
			rptr--;

/* If the character following the '\\' is a NULL, back up the */
/* ptr so that the NULL won't be missed.  The sequence */
/* backslash null is essentually illegal. */
		} else if (*rptr == '\0') {
			c = '\0';
			rptr--;

/* In all other cases the quoting does nothing. */
		} else c = *rptr;
		break;
	}

/* Compute the size of the quoted character. */
	(*qsize) = rptr - ptr + 1;
	return(c);
}

/*	"fields" picks up the words in the next field and converts all	*/
/*	recognized words into the proper mask and puts it in the target	*/
/*	field.								*/

char *fields(ptr,termio,gctio) /* sw0 */
register char *ptr;
struct termio *termio;
struct gctio *gctio; /* sw0 */
{
	extern struct Symbols imodes[],omodes[],cmodes[],lmodes[];
	extern struct Symbols *search();
	register struct Symbols *symbol;
	char *word,*getword();
	int size, value;
	extern int check;

	termio->c_iflag = 0;
	termio->c_oflag = 0;
	termio->c_cflag = 0;
	termio->c_lflag = 0;

	gctio->gt_tdcd = 0; /* sw0 */
	gctio->gt_tact = 0; /* sw0 */

	while (*ptr != '#' && *ptr != '\0') {

/* Pick up the next word in the sequence. */
		word = getword(ptr,&size);

/* If there is a word, scan the two mode tables for it. */
eval:		if (*word != '\0') {

/* If the word is the special word "SANE", put in all the flags */
/* that are needed for SANE tty behavior. */
			if (strcmp(word,"SANE") == 0) {
				termio->c_iflag |= ISANE;
				termio->c_oflag |= OSANE;
				termio->c_cflag |= CSANE;
				termio->c_lflag |= LSANE;
			} else if (strcmp(word,"TDCD") == 0) { /* sw0 */
				ptr += size; /* sw0 */
				word = getword(ptr,&size); /* sw0 */
				value = atoi(word); /* sw0 */
				if (value < 1 || value > 18000) /* sw0 */
					goto eval; /* sw0 */
				gctio->gt_tdcd = value; /* sw0 */
				gctio->gt_tdcd_sig = SIGHUP;
			} else if (strcmp(word,"TACT") == 0) { /* sw0 */
				ptr += size; /* sw0 */
				word = getword(ptr,&size); /* sw0 */
				value = atoi(word); /* sw0 */
				if (value < 1 || value > 18000) /* sw0 */
					goto eval; /* sw0 */
				gctio->gt_tact = value; /* sw0 */
				gctio->gt_tact_sig = SIGHUP;
			} else if ((symbol = search(word,imodes)) != NULL)
				termio->c_iflag |= symbol->s_value;
			else if ((symbol = search(word,omodes)) != NULL)
				termio->c_oflag |= symbol->s_value;
			else if ((symbol = search(word,cmodes)) != NULL)
				termio->c_cflag |= symbol->s_value;
			else if ((symbol = search(word,lmodes)) != NULL)
				termio->c_lflag |= symbol->s_value;
			else if (check) fprintf(stdout,"Undefined: %s\n",word);
		}

/* Advance pointer to after the word. */
		ptr += size;
	}

/* If we didn't end on a '#', return NULL, otherwise return the */
/* updated pointer. */
	return(*ptr != '#' ? NULL : ptr);
}

/*	"parse" breaks up the user's response into seperate arguments	*/
/*	and fills the supplied array with those arguments.  Quoting	*/
/*	with the backspace is allowed.					*/

parse(string,args,cnt)
char *string,**args;
int cnt;
{
	register char *ptrin,*ptrout;
	int i;
	extern char quoted();
	int qsize;

	for (i=0; i < cnt; i++) 
		args[i] = (char *)NULL;

	for (ptrin = ptrout = string,i=0; *ptrin != '\0' && i < cnt; i++) {

/* Skip excess white spaces between arguments. */
		while(*ptrin == ' ' || *ptrin == '\t') {
			ptrin++;
			ptrout++;
		}
/* Save the address of the argument if there is something there. */
		if (*ptrin == '\0') break;
		else args[i] = ptrout;

/* Span the argument itself.  The '\' character causes quoting */
/* of the next character to take place (except for '\0'). */
		while (*ptrin != '\0') {

/* Is this the quote character? */
			if (*ptrin == '\\') {
				*ptrout++ = quoted(ptrin,&qsize);
				ptrin += qsize;

/* Is this the end of the argument?  If so quit loop. */
			} else if (*ptrin == ' ' || *ptrin == '\t') {
				ptrin++;
				break;

/* If this is a normal letter of the argument, save it, advancing */
/* the pointers at the same time. */
			} else *ptrout++ = *ptrin++;
		}

/* Null terminate the string. */
		*ptrout++ = '\0';

                if(i == 0)    /* insert environ. vars after user's name */
				i = cpy(env_var, args, i, MAXARGS-1);

	}
}

FILE *fdup(fp)
register FILE *fp;
{
	register int newfd;
	register char *mode;

/* Dup the file descriptor for the specified stream and then */
/* convert it to a stream pointer with the modes of the original */
/* stream pointer. */
	if ((newfd = dup(fileno(fp))) != FAILURE) {

/* Determine the proper mode.  If the old file was _IORW, then */
/* use the "r+" option, if _IOREAD, the "r" option, or if _IOWRT */
/* the "w" option.  Note that since none of these force an lseek */
/* by "fdopen", the dupped file pointer will be at the same spot */
/* as the original. */
		if (fp->_flag & _IORW) mode = "r+";
		else if (fp->_flag & _IOREAD) mode = "r";
		else if (fp->_flag & _IOWRT) mode = "w";

/* Something is wrong, close dupped descriptor and return NULL. */
		else {
			close(newfd);
			return(NULL);
		}

/* Now have fdopen finish the job of establishing a new file pointer. */
		return(fdopen(newfd,mode));
	} else return(NULL);
}


cpy(in,out,count,max)
char **in;
char **out;
int count;
int max;
{
	while (*in && (count < max )) {
		*++out = *in++;
		count++;
	}
	return (count);
}




/* From this point on there is only SECURE stuff.  Please put any
 * non-secure additions above this point 
 */

#ifdef SECON
/*
 *	opendevauth   opent the authorization databases for the 
 *		       terminal. If everything is okay, then we
 *		       will return a valid ta ptr, els NULL is 
 *		       returned. All other security relevant routines
 *		       must check the validity of this ptr...if it's
 *		       NULL we should skip the processing!
 */

ttyauth_t *opendevauth(line)
char *line;
{
ttyauth_t *ta, *gettanam();

	strcpy( ttyn, "/dev/" );
	strcat( ttyn, line );
	/* create the name to search for in the devauth database */	
	if ( access(TTYAUTH, 0))
		return (ttyauth_t *) 0;
	if (!(ta = gettanam( ttyn ))) {
		/* since the auth database is present and we don't know
		 * about the tty, lock it up now! 
		 */
		lockterm(ttyn); 
		return 0;
	}
	endtaent();
	/* make sure terminal isn't supposed to be locked...it it is
	 * invoke the lock stuff now
	 */
	if ( ta->ta_status )
		lockterm(ttyn);
	return ta;
}

/*
 *	lockterm   lock the terminal passwd on by name 
 */	
lockterm(fname)
char *fname;
{
int i;

	/* Never lock the console. The sec_adm may want in. */
	if ( strcmp(fname, CONSOLE) == 0 )
		return;

	/* catch all signals just in case the lock utility isn't
	 * present or errored out
 	 */ 
	for ( i=0; i<MAXSIG; i++ )
		signal( i, SIG_IGN );

	close(0);
	close(1);
	close(2);
	
	/* become our own session leaders to we can get the 
	 * terminal as the controlling terminal on the very 
	 * next open 
	 */

#ifdef _POSIX_SOURCE
	setsid();
#else
	setpgrp();
#endif
	open(fname, O_RDWR|O_NDELAY );
	execl( "/etc/lock", "lock", "-l", "-t", fname, (char *) 0 );
	while (TRUE) {
		error( "can't exec /etc/lock\n\r" );
			sleep( 999 );
		}
}

/*
 *	protect   make sure the terminal is protected till login 
 *		  hands it off to the user. This means that the 
 *		  device will be marked as SECURED
 */
protect(line,ta)
char *line;
ttyauth_t *ta;
{
struct stat statb;

	stat(line,&statb);
	chown(line,0,statb.st_gid);
	chmod(line,0622);
	if ( ta ) {
		/* JTOF - if this fails, we probably want to write and 
		 * audit record right here. ADD for next release 
		 */
		set_file_slabel( line, &ta->ta_maxlabel ); 
		/* killspoofs(line); */
	}
}


/* 
 * 	setsak  do the secure attention key processing, including the
 *	        download of the keys to the device. This will only
 *	        work for modules pushed on the local acdb/acrw and
 *		console drivers 
 */
setsak(line, ta)
char *line;
ttyauth_t *ta;
{
char buf[80];
int fd;

	/* if there isn't a devauth database don't do 
	 * the sak stuff, cause there ain't no sak information
	 * to do it with
	 */
	if ( ta == NULL )
		return;

	/* if the ttyauth file sak keys are not set
 	 * then there is nothing to do so return
 	 */ 
	if ( ta->ta_sak.flags == 0 )
		return 0;

	if ((fd=open(line, O_WRONLY|O_NDELAY)) < 0 ) {
		error( "can't open device /dev/%s\n\r", line );
		return -1;
	}
	if ( ioctl(fd, TCSAK_SET, &ta->ta_sak ) != -1 ||
	     ioctl(fd, TCSAK_GET, &ta->ta_sak ) != -1 && ta->ta_sak.flags )
		sakset = 1;
	close(fd);
}

/*
 *	msg print a message to the terminal being opened 
 */
msg(format,arg1,arg2,arg3,arg4)
char *format;
int arg1,arg2,arg3,arg4;
{
register FILE *fp;

	if ((fp = fopen(ttyn,"w")) == NULL)
		return;
	fprintf(fp,format,arg1,arg2,arg3,arg4);
	fflush(fp);
	fclose(fp);
}

/*
 *
 */
kill_good_spfrs(line)
char *line;
{
int fd, status, cnt; 

	if (!strcmp( line, "console" ))
		return 0;

	/* open the display so we can set SIGSAK */
	if ((fd=open(ttyn, O_WRONLY|O_NDELAY)) < 0 ) {
		error( "can't open device %s\n\r", ttyn );
		return 0;
	}

	cnt = 0;
	while ( kernref( fd ) > 1 ) {
		close( fd );

		/* we don't want to spin in infinite loops */
		if ( cnt++ > 6 ) {
			lockterm(ttyn);
			break;
		}
		/* JTOF - kill everyone - if can't exec for some reason
		   kill line rather than letting users in */
		if ( fork() )  {
			while (wait( &status )!= -1);
			/* reopen port */
			if ((fd=open(ttyn, O_WRONLY|O_NDELAY)) < 0 ) {
				error( "can't open device %s\n\r", ttyn );
				return 0;
			}
		}
		else {
			if ( execl( "/etc/fuser", "fuser", 
				"-ku", ttyn, 0 ) == -1) {

			/* lock the line if we can't kill everyone */
			error( "can't exec fuser - locking line\n\r" );
			lockterm(ttyn);
			}
		}
	}
	close(fd);

}

/*
 *	kill_cterm_spfrs - check to see if we've got the console as the 
 *		       controlling console. If we don't well kill all
 *		       the guys that do 
 */
kill_cterm_spfrs(line)
char *line;
{
int fd;
	if ((fd = open("/dev/tty", O_RDONLY)) == -1) {
		error("\ngetty: %s failed to become control terminal.\n\r",
		      line );
	        error("getty: killing all references to /dev/%s\n\r", line );
		close(0);
		close(1);
		close(2);
		open("/dev/null", O_RDONLY);
		open(CTTY,O_RDONLY);
		dup(1);
		if ( execl( "/etc/fuser", "fuser", "-ku", line, 0 ) == -1) {
			error( "getty: cannot clear line /dev/%s\n\r", line );
			sleep(10);
			exit(1);
		}
	}
	close(fd);
}
#endif
