/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) dsetscsi.c: version 25.1 created on 12/2/91 at 14:16:37	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)dsetscsi.c	25.1	12/2/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*----------------------------------------------------------------------------
/ dsetscsi.c - scsi module for 'dsetup' program
/
/ April 27, 1988    Collected by Craig J. Kim
/ Extracted from dset.c, dset1.c, dset2.c, and rvio.c for SCSI
/
/ - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
/ NOTE:  dsetup makes use of setjmp(), longjmp() pair to recover from serious
/   error conditions.  setjmp() is executed once at the top of dsetup()
/   function and longjmp() is called from many error handling functions.
/
/   All SCSI functions are prefixed with 'sc_' to distinguish from their
/   counter functions for SMD.
/---------------------------------------------------------------------------*/

#ifdef SADEBUG
#define DBG(statement)  statement
#else
#define DBG(statement)  /* do nothing */
#endif

#include "disksect.h"

#if defined(UNIXSTANDALONE) | defined(RO_UNIX)
#include <stdio.h>
#include <ctype.h>
#endif

#include "dsetup.h"
#include "diskconfig.h"
#include "dk.h"

#ifndef TRUE
#define TRUE    1
#define FALSE   0
#endif

extern int check_drive();               /* check if drive unit num is ok */
extern int gstr();                      /* get a string from user input */
extern int gnum();                      /* get a number from user input */
extern int gdrive();                    /* get a drive unit num from user input */
extern void pversion();                 /* print program version number */
extern int q_cmd();                     /* quit program */
extern int u_cmd();                     /* undo */
extern int close();                     /* close a file */
extern void warn();                     /* print a warning message */
extern void error();                    /* print a error message and longjump */
extern void confirm();                  /* ask for confirmation, if no, longjmp */
extern checkldi();                      /* check if valid logical disk num */
extern checkldi0();                     /* check if valid logical disk num */
extern struct confld *setupwritei();    /* */
extern writelasti();                    /* */
extern insertld();                      /* insert a logical disk */
extern removeld();                      /* remove a logical disk */
extern int isscsi();                    /* is controller SCSI type */
extern pdfstat();                       /* get physical disk status */
extern roundtrack();                    /* */
extern pdwrite();                       /* write to physical disk */
extern ioctlck();                       /* physical disk ioctl */
extern pdreadany();                     /* read physical disk */
extern pdfread();                       /* read physical disk */
extern strcmp();                        /* string comparsion routine */
extern memcpy();                        /* memory copy */
extern memset();                        /* memory initialization */
extern reperr();                        /* call error() or warn() */
extern char *strlwr();                  /* convert to lower case */

/*---------------------------------------------------- forwards ------------*/
int scsi_dsetup();                      /* this is main */
static void sc_setupselectclear();      /* clear setup - release mem */
static void sc_setupapplyclear();       /* */
static void sc_setupsave();             /* */
static void sc_a_cmd();
static void sc_c_cmd();
static void sc_d_cmd();
static void sc_l_n_cmd();
static void sc_l_o_cmd();
static void sc_l_p_cmd();
static void sc_l_t_cmd();
static void sc_l_c_cmd();
static void sc_m_p_cmd();
static void sc_m_r_cmd();
static void sc_m_s_cmd();
static void sc_m_t_cmd();
static void sc_n_cmd();
static void sc_q_cmd();
static void sc_r_cmd();
static void sc_s_cmd();
static void sc_w_cmd();
static void sc_printhelp();

static void sc_setdisk();                  /* get disk info */
static int  sc_configit();
static void sc_modifyldtype();
static void sc_setldsize();
static int  sc_checkusersetup();
static void sc_printphysdisk();
static void sc_printsetup();
static void sc_printld();
static void sc_setupldtype();
static void sc_setupld();

static void sc_writeconfig();
static void sc_applyconfig();
static void sc_applystart();
static void sc_printapplysetup();
static void sc_printapplyld();
static int  sc_checkapplysetup();
static void sc_readphys0();
static void sc_rvwriteconfig();

/*---------------------------------------------------- globals -------------*/
extern jmp_buf errorbuf;                /* so we can bail us out */
extern struct setup oldsetup;           /* as found on disk */
extern struct setup prevsetup;          /* undo with this one */
extern struct setup newsetup;           /* working copy */
extern struct setup tempsetup;          /* newsetup just before any changes */

extern int debugflag;                   /* can toggle with z command */

extern char cmdbuf[200];                /* i hope there is a limit */
extern char *pcmd;                      /* command pointer */

extern int warn_level_flag;
extern int scsi_dtc;                    /* scsi type controller */

extern struct conftable gConf[];
extern char *ascldtype[];
extern char *brief_ascldtype[];

/*---------------------------------------------------- statics -------------*/
static int scsi_ldtype[] = {
    LD_SWAP,
#ifndef S3000
    LD_PWF,
#endif
    LD_USERSPARE    };

/*---------------------------------------------------- scsi_dsetup() -------*/
int scsi_dsetup(drv)
int drv;
{
    char cmdchar1[10], cmdchar2[10];
    int cmdint1, cmdint2, cmdint3;
    int help_request = FALSE;

    while (1) {
        if (setjmp(errorbuf)) {         /* the error subr comes here */
            xprintf("errorjump ");
            if (tempsetup.flag & SET_USED) {
                printf("No change was made.\n");
                if (debugflag)
                    sc_l_n_cmd();
                if (tempsetup.flag & SET_FIG)
                    sc_setupapplyclear(&tempsetup);
                if (newsetup.flag & SET_FIG)
                    sc_setupapplyclear(&newsetup);
                newsetup = tempsetup;
                tempsetup.flag = 0;
            }
            drv = -1;
        }

        if (drv >= 0) {
            sc_s_cmd(drv);
            drv = -1;
        }

        if (help_request) {             /* display help then accept a command */
            sc_printhelp(cmdbuf);
            help_request = FALSE;
        }
        else {
            printf("dsetup-> ");        /* this is our prompt */
            gets(cmdbuf);               /* see what user says */
        }
        pcmd = strlwr(cmdbuf);          /* we'll need to parse */
        if (!gstr(cmdchar1))            /* didn't type anything */
            continue;

        switch (cmdchar1[0]) {          /* we'll look at only first char */
            case '?':                   /* (?) Help! */
                help_request = TRUE;
                break;
#ifndef RO_UNIX
            case 'a':                   /* (a) apply a setup */
                sc_a_cmd();
                break;
            case 'c':                   /* (c) pick a configuration */
                if (!gnum(&cmdint1))
                    cmdint1 = -1;
                sc_c_cmd(cmdint1);
                break;
            case 'd':                   /* (d) delete a logical disk */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    cmdint2 = cmdint1 - 1;
                sc_d_cmd(cmdint1, cmdint2);
                break;
#endif
            case 'l':                   /* (l) one of a variety of list commands */
                if (!gstr(cmdchar2))
                    cmdchar2[0] = 'n';  /* default to list new */
                switch (cmdchar2[0]) {
                    case 'c':           /* (l c) list predefined configurations */
                        sc_l_c_cmd();
                        break;
                    case 'n':           /* (l n) list new configuration */
                        sc_l_n_cmd();
                        break;
                    case 'o':           /* (l o) list old configuration */
                        sc_l_o_cmd();
                        break;
                    case 'p':           /* (l p) list physical property */
                        sc_l_p_cmd();
                        break;
                    case 't':           /* (l t) list possible logical disk types */
                        sc_l_t_cmd();
                        break;
                    default:            /* huh? */
                        goto parserr;
                }
                break;
#ifndef RO_UNIX
            case 'm':                   /* (m) one of a variety of modify commands */
                if (!gstr(cmdchar2))    /* nothing follows 'm' */
                    goto parserr;
                switch (cmdchar2[0]) {
                    case 'p':           /* (m p) reset the protection */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        sc_m_p_cmd(cmdint1);
                        break;
                    case 'r':           /* (m r) modify reserved area size */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            cmdint2 = -1;
                        sc_m_r_cmd(cmdint1, cmdint2);
                        break;
                    case 's':           /* (m s) modify size */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            goto parserr;
                        /* the following not quite right, non numeric garbage
                            will look like a nonexistent thing, sigh */
                        if (!gnum(&cmdint3))
                            cmdint3 = -1;
                        sc_m_s_cmd(cmdint1,cmdint2,cmdint3);
                        break;
                    case 't':           /* (m t) modify type */
                        if (!gnum(&cmdint1))
                            goto parserr;
                        if (!gnum(&cmdint2))
                            goto parserr;
                        sc_m_t_cmd(cmdint1, cmdint2);
                        break;
                    default:            /* huh? */
                        goto parserr;
                }
                break;
            case 'n':                   /* (n) select config 0 */
                sc_n_cmd(0);
                break;
#endif
            case 'q':                   /* (q) quit */
                if (q_cmd())
                    return(-1);
                break;
#ifndef RO_UNIX
            case 'r':                   /* (r) remove a logical disk */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    cmdint2 = cmdint1-1;
                sc_r_cmd(cmdint1,cmdint2);
                break;
#endif
            case 's':                   /* (s) select physical disk */
                if (!gdrive(&cmdint1))
                    goto parserr;
                if (check_drive(cmdint1))
                    return(cmdint1);
                break;
#ifndef RO_UNIX
            case 'u':                   /* (u) undo */
                u_cmd();
                break;
#endif
            case 'v':                   /* (v) print current version */
                pversion();
                printf("\n");
                break;
#ifndef RO_UNIX
            case 'w':                   /* (w) calculate and write it out */
                sc_w_cmd();
                break;
            case 'x':                   /* (x) split a disk, spread the table out */
                if (!gnum(&cmdint1))
                    goto parserr;
                if (!gnum(&cmdint2))
                    goto parserr;
                sc_m_s_cmd(cmdint1,cmdint2,cmdint1);
                break;
#endif
            case 'z':                   /* (z) hidden debug command */
                debugflag ^= 1;
                printf("Debug %s\n", debugflag ? "on" : "off");
                break;
            default:                    /* huh? */
                goto parserr;
        }
        if (tempsetup.flag & SET_USED)
            prevsetup = tempsetup;
        tempsetup.flag = 0;
        continue;
parserr:
        printf("Unrecognized command.  Type '?' for help.\n\n");
    }
}

/*---------------------------------------------------- sc_setupselectclear() -*/
static void sc_setupselectclear(setup)
struct setup *setup;
{
    if (setup->fd)                      /* close related file */
        close(setup->fd);
    if (setup->memPhysdisk)             /* free any used memory */
        memfree(setup->memPhysdisk);
    if (setup->memBad)
        memfree(setup->memBad);
    if (setup->memSpare)
        memfree(setup->memSpare);
    if (setup->memSkip)
        memfree(setup->memSkip);
    if (setup->flag & SET_FIG)          /* has it been changed in anyway */
        sc_setupapplyclear(setup);

    memset(setup, 0, sizeof (*setup));  /* clean up the structure */
}

/*---------------------------------------------------- sc_setupapplyclear() */
static void sc_setupapplyclear(setup)
struct setup *setup;
{
    setup->flag &= ~(SET_FIG | SET_WRIT);/* clear apply and write flags */
}

/*---------------------------------------------------- sc_setupsave() ------*/
static void sc_setupsave(d, s)
struct setup *d, *s;
{
    if (!(s->flag & SET_USED))
        error("no disk is selected");

    *d = *s;                            /* copy structure here */
    d->flag |= SET_USED;
    if (!(warn_level_flag++)) {
        struct sector0 *pd;

        pd = PD0(s);
        if (pd->DSETUP_LEVEL < LEVEL_1_DSETUP)
            warn("the disk tables are at old version level, compensating");
    }
}

/*---------------------------------------------------- sc_a_cmd() ------------
/ apply new configuration
/---------------------------------------------------------------------------*/
static void sc_a_cmd()
{
    sc_setupsave(&tempsetup, &newsetup);/* copy newsetup to tempsetup */
    sc_applyconfig(&newsetup);          /* let newsetup be the current */
    newsetup.flag |= SET_MOD;           /* set modify bit */
}

/*---------------------------------------------------- sc_c_cmd() -----------*/
static void sc_c_cmd(i)
{
    register struct sector0 *pd;        /* misspelling this causes compiler dump */

    pd = 0;
    if (i == -1) {
        if (IS_SYS0(&newsetup)) {
            if (newsetup.cTotalSect < SMALL_LIMIT)
                i = CONF_SMALL0;
            else if (newsetup.cTotalSect < MEDIUM_LIMIT)
                i = CONF_MED0;
            else
                i = CONF_LARGE0;
        }
        else {
            if (newsetup.cTotalSect < SMALL_LIMIT)
                i = CONF_SMALL;
            else if (newsetup.cTotalSect < MEDIUM_LIMIT)
                i = CONF_MED;
            else
                i = CONF_LARGE;
        }
        printf("Choosing configuration %d\n", i);
    }

    if (i < 0 || i >= NUM_DISKCONFS)
        error("%d is not an available configuration", i);
    sc_setupsave(&tempsetup, &newsetup);

    pd = PD0(&newsetup);                /* set pointer, non zero is flag */
    if (i == 0 && (newsetup.flag & SET_OLD)) {
        pd->pd_ldnum = 0;               /* force it to be treated as new */
        printf("New configuration, if BOOTIMAGE desired, must re-run ldsa.\n");
        warn("Logical drive data will no longer be accessable!");
        confirm();
        newsetup.confRsv.flag = 0;
        newsetup.flag = SET_USED;
    }
    else if (newsetup.flag & (SET_OLD | SET_MOD)) {
        warn("Logical drive data will no longer be accessable");
        confirm();
        sc_setupapplyclear(&newsetup);
    }

    if (!sc_configit(&newsetup, gConf[i].pconf))
        error("selected configuration doesn't fit on disk");
    newsetup.flag |= SET_MOD | SET_NEW;

    /*-----------------------------------------------------------------
    / the following because of a earlier bug in the stored format of
    / the spare table.  By turning off SET_OLD the old table is not
    / read.  Since the table is not read the reserved area must be
    / unprotected so that all the sectors will be accounted for.
    /----------------------------------------------------------------*/

    if (pd && pd->DSETUP_LEVEL < LEVEL_1_DSETUP) {
        newsetup.confRsv.flag &= ~CFLD_PROTECT;
        newsetup.flag &= ~ SET_OLD;
        warn("the disk tables are at old version level, compensating");
    }
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_d_cmd() -----------*/
static void sc_d_cmd(p1,p2)
{
    sc_m_s_cmd(p1, 0, p2);
}

/*---------------------------------------------------- sc_l_c_cmd() ----------
/ list predefined configurations to be used with 'c' command
/---------------------------------------------------------------------------*/
static void sc_l_c_cmd()
{
    register int i;
    register int count;
    struct defconf *conf;
    int *p;

    printf("id  name                  detail\n");
    printf("--  --------------------  ------------------------------------\n");
    for (i = 0; i < NUM_DISKCONFS; i++) {
        printf("%2d  %20s", i, gConf[i].name);
        conf = gConf[i].pconf;
        printf("  Reserved:%d    leftover disks { %s:%d }\n     ",
                conf->cRsv, brief_ascldtype[conf->normaltype], conf->normal);
        for (count = 0, p = conf->defined;
                count < conf->definedcount;
                count++, p += 2)
            printf(" %s:%d", brief_ascldtype[p[1]], *p);
        printf("\n");
    }
}

/*---------------------------------------------------- sc_l_n_cmd() --------*/
static void sc_l_n_cmd()
{
    if (!(newsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    sc_printapplysetup(&newsetup, debugflag);
}

/*---------------------------------------------------- sc_l_o_cmd() --------*/
static void sc_l_o_cmd()
{
    if (!(oldsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    sc_printapplysetup(&oldsetup);
}

/*---------------------------------------------------- sc_l_p_cmd() --------*/
static void sc_l_p_cmd()
{
    if (!(newsetup.flag & SET_USED)) {
        warn("there is nothing to list");
        return;
    }
    sc_printphysdisk(&newsetup);
}

/*---------------------------------------------------- sc_l_t_cmd() ----------
/ list possible logical disk types to used with 'm t' command
/---------------------------------------------------------------------------*/
static void sc_l_t_cmd()
{
    register int i;

    printf("\n");
    for (i = 0; i < sizeof (scsi_ldtype) / sizeof (int); i++)
        printf("   %3d.  %s\n", i + 1, ascldtype[scsi_ldtype[i]]);
    printf("\n");
}

/*---------------------------------------------------- sc_m_p_cmd() --------*/
static void sc_m_p_cmd(p1)
{
    struct confld *ld;

    sc_setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    ld = &newsetup.confld[p1];
    if (!(ld->flag & CFLD_PROTECT))
        error("Logical disk %d is not protected", p1);
    ld->flag &= ~CFLD_PROTECT;
    sc_checkusersetup(&newsetup, warn, errorbuf);
    newsetup.flag |= SET_MOD;
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_m_r_cmd() --------*/
static void sc_m_r_cmd(p1, p2)
{
    sc_setupsave(&tempsetup, &newsetup);
    if (newsetup.confld[-1].flag & CFLD_PROTECT)
        error("Reserved area is protected");
    sc_setldsize(&newsetup, RV_LOGDR, p1, p2);
    newsetup.flag |= SET_MOD;
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_m_s_cmd() --------*/
static void sc_m_s_cmd(p1,p2,p3)
{
    sc_setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);

    sc_setldsize(&newsetup, p1, p2, p3);
    newsetup.flag |= SET_MOD;
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_m_t_cmd() --------*/
static void sc_m_t_cmd(p1, p2)
{
    sc_setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected.", p1);
    if (p2 < 1 || p2 > (sizeof (scsi_ldtype) / sizeof (int)))
        error("Unknown logical disk type %d.", p2);
    sc_modifyldtype(&newsetup, p1, scsi_ldtype[p2 - 1]);
    newsetup.flag |= SET_MOD;
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_n_cmd() ------------
/ (n) select configuration #0
/---------------------------------------------------------------------------*/
static void sc_n_cmd()
{
    sc_c_cmd(0);
}

/*---------------------------------------------------- sc_r_cmd() ----------*/
static void sc_r_cmd(p1,p2)
{
    sc_setupsave(&tempsetup, &newsetup);
    checkldi0(p1);
    if (newsetup.confld[p1].flag & CFLD_PROTECT)
        error("Logical disk %d is protected", p1);
    sc_setldsize(&newsetup,p1,0,p2);
    removeld(&newsetup,p1);
    newsetup.flag |= SET_MOD;
    sc_applyconfig(&newsetup);
}

/*---------------------------------------------------- sc_s_cmd() ------------
/ select a disk
/---------------------------------------------------------------------------*/
static void sc_s_cmd(pd)
int pd;                 /* disk number in binary "ccccdddd" format */
{
    struct icbdev dev;

    dev.dev_num = pd;                   /* save physical unit number */
    if (newsetup.flag & SET_MOD) {
        warn("Configuration has not been written to selected disk");
        confirm();
    }
    sc_setupselectclear(&newsetup);     /* clear working area */
    sc_setdisk(&newsetup, &dev);        /* go fill working area with current */
    if (!(newsetup.flag & SET_NEW))     /* is it new? */
        oldsetup = newsetup;

    printf("SCSI Controller %d, drive %d selected.\n", pd >> 4, pd & 0x0f);
    warn_level_flag = 0;
}

/*---------------------------------------------------- sc_w_cmd() ------------
/ write the setup to disk (Caution: no undo - permanent)
/---------------------------------------------------------------------------*/
static void sc_w_cmd()
{
    struct icbdev dev;

    dev = newsetup.dev;                 /* save device number */
    prevsetup.flag = 0;                 /* there is no undo */
    if (!(newsetup.flag & SET_FIG))
        sc_applyconfig(&newsetup);
    sc_writeconfig(&newsetup);          /* write to disk (permanent) */
    sc_setupselectclear(&newsetup);     /* clear structure */
    sc_s_cmd(dev.dev_num);              /* start over */
}

/*---------------------------------------------------- sc_printhelp() ------*/
static void sc_printhelp(cmd)
char *cmd;
{
    static char *helpmsg[] = {
    "SELECT Commands ----------------------------------------------------",
    "  s <#>            Select physical drive",
#ifndef RO_UNIX
    "  c [#]            Select logical disk configuration",
#endif
    "LIST Commands ------------------------------------------------------",
    "  l c              List predefined Configurations, use with 'c'",
    "  l [n]            List New configuration",
    "  l o              List Old configuration",
    "  l p              List physical property",
    "  l t              List possible logical disk Types, use with 'm t'",
#ifndef RO_UNIX
    "MODIFY Commands ----------------------------------------------------",
    "  m p <ld>         Modify/reset protection of <ld>",
    "  m s <ld1> <size> [ld2]  Mod Size of ld1, difference from ld2",
    "  m t <ld> <type>  Mod Type of ld1 to <type>",
#endif
    "OTHER Commands -----------------------------------------------------",
#ifndef RO_UNIX
    "  x <ld> <size>    Split ld into two chunks, expand the ld table",
    "  d <ld1> [ld2]    Delete <ld1>, sectors to <ld2>",
#endif
    "  q                Quit",
#ifndef RO_UNIX
    "  r <ld1> [ld2]    Remove <ld1>, as delete except collapse table",
    "  u                Undo the previous change",
    "  w                Write the setup to disk (no undo)",
#endif
    0 };
    register char **p, *cp;

    *cmd = '\0';
    for (p = helpmsg; *p; p++)
        if (**p == '*') {               /* page break */
            printf("\nPress [Return] for next page or enter a command-> ");
            gets(cmd);
            printf("\n");
            cp = cmd;
            while (isspace(*cp))        /* skip leading spaces */
                cp++;
            if (*cp)                    /* user types in a command */
                return;
        }
        else
            printf("%s\n", *p);

    printf("\n");
}

/*---------------------------------------------------- sc_setdisk() ----------
/ collects all necessary disk information:  sector 0, bad block list, spare
/ block list,
/---------------------------------------------------------------------------*/
static void sc_setdisk(setup, dev)
register struct setup *setup;
struct icbdev *dev;     /* device address of disk to operate on */
{
    struct sector0 *pd;
    register struct confld *ld;
    register struct logtype *logdrive;

    DBG(printf("Entered sc_setdisk()\n");)
    memset((char *) setup, 0, sizeof (*setup));

    setup->dev = *dev;                  /* read sector 0 */
    DBG(printf("Calling sc_readphys0()\n");)
    sc_readphys0(setup);                /* setup->memPhysdisk */
    DBG(printf("Calling pdfstat()\n");)
    pdfstat(setup->fd, &setup->pdstat);
    pd = PD0(setup);                    /* pd = setup->memPhysdisk->p */
    setup->cTotalSect = pd->cyldisk * pd->headcyl * pd->sechead;

    DBG(printf("Calling checknewsetup()\n");)
    if (checknewsetup(setup)) {         /* freshly formatted */
        setup->flag = SET_USED | SET_NEW;
        sc_configit(setup, gConf[0].pconf);
    }
    else {                              /* setup current information */
        register int i, avail = 0;

        setup->flag = SET_USED | SET_OLD;
        ld = &setup->confRsv;           /* reserved area */
        ld->flag = CFLD_OLD | CFLD_PROTECT;
        ld->type = LD_USERSPARE;
        /* start is 0 */
        ld->cSect = pd->rv_size;        /* sector counts */
        ld->cAssignSect = ld->cSect;    /* final number of sectors */
        ld = setup->confld;             /* first logical disk (0) */
        logdrive = pd->logdrive;        /* logical disk type */
        for (i = 0; i < LOGDR; i++, ld++, logdrive++) {
            ld->type = logdrive->ld_type;
            if (ld->type) {
                if (i >= pd->pd_ldnum)  /* too many logical disks */
                    errxit("funny format, some ld past ldnum");/* run dsck */
                ld->flag = (CFLD_OLD | CFLD_PROTECT);
                ld->logdriveindex = i;  /* remember position in old table */
                ld->start = logdrive->ld_strt;
                ld->cSect = logdrive->ld_size;
                ld->cAssignSect = ld->cSect;
                avail += ld->cSect;     /* total sector counts */
            }
        }
        setup->cAvail = avail;
    }
    setup->cSpareGuess = 0;
}

/*---------------------------------------------------- sc_configit() ---------
/ take 'conf' and apply it to the physical disk size.
/ watch out for the last disk.
/ if there is only one disk then its done by default.
/ this routine assumes an honest conf.
/---------------------------------------------------------------------------*/
static int sc_configit(setup, conf)
register struct setup *setup;
register struct defconf *conf;
{
    register int i, c, *p;
    register struct confld *d;
    register int tsize;

    if (!setup->memPhysdisk)
        error("no physical disk selected");

    setup->defaultldtype = conf->normaltype;
    d = &setup->confRsv;                /* reserved area */
    d->flag &= ~CFLD_PROTECT;           /* don't protect */
    d->type = LD_USERSPARE;             /* spare */
                                        /* reserved area size in # of sectors */
    d->cSect = roundtrack(setup, conf->cRsv);
    setup->cAvail = setup->cTotalSect - d->cSect;

    /* set up a big logical disk */
    d = setup->confld;
    memset(d, 0, sizeof (struct confld) * LOGDR);
    sc_setupldtype(setup, d, setup->cAvail, conf->normaltype);
    if (conf->definedcount == 0)        /* this is it - only one disk */
        return(1);

    c = setup->cAvail;
    for (i = 0, p = conf->defined; i < conf->definedcount; i++, d++, p += 2) {
        sc_setupldtype(setup, d, roundtrack(setup, p[0]), p[1]);
        c -= d->cSect;
        if (c < 0)                      /* oops, won't fit */
            return(0); 
    }

    d--;                                /* point to the last logical disk */
    if (c == 0)
        goto exitok;                    /* it was lucky to just fit, probably a bug */

    if (c < MIN_LD_SIZE || i==LOGDR) {
        warn("last disk of selected configuration extended");
        d->cSect += c;
        goto exitok;
    }
                                        /* make the remains one disk */
    if (conf->normal == -1 || c < conf->normal) { 
        if (c < conf->normal)
            warn("last disk is smaller than expected");
        i++;
        (++d)->cSect = c;
        d->type = conf->normaltype;
        goto exitok;
    }

    /* fill up the rest of the disk, the last will be the biggest */
    while (c >= (tsize = roundtrack(setup, conf->normal)) && i != LOGDR) {
        sc_setupldtype(setup, ++d, tsize, conf->normaltype);
        c -= d->cSect;
        i++;
    }
    
    d->cSect += c;                      /* add in the remains */

exitok:
    sc_setupld(setup, d, d->cSect);
    return(1);
}

/*---------------------------------------------------- sc_modifyldtype() ---*/
static void sc_modifyldtype(setup, i, t)
struct setup *setup;
{
    register struct confld *ld;

    checkldi(i);
    ld = &setup->confld[i];
    if (!ld->type)
        error("cannot set type of empty ld");
    sc_setupldtype(setup, ld, ld->cSect, t);
    sc_checkusersetup(setup, warn, errorbuf);
}

/*---------------------------------------------------- sc_setldsize() ------*/
static void sc_setldsize(setup, inew, size, islack)
register struct setup *setup;
{
    register struct confld *ldnew, *ldslack;
    int lo,hi;
    int slackadjust;                    /* >0, add to ldslack, <0, remove from ldslack */
    int endcontig;

    /* three cases
     * - some disk in the middle
     * - the last disk (may be like previous case)
     * - the disk after the last disk
     */
    size = roundtrack(setup, size);
    if (inew != RV_LOGDR) {
        ldnew = setupwritei(setup,inew);
        if (inew == islack) {           /* special case, split an existing disk */
            insertld(setup, islack = inew + 1); /* make an empty ld after i */
        }
    }
    else {
        ldnew = &setup->confRsv;
        setup->cAvail = setup->cTotalSect - (size + setup->cSpareGuess);
    }
    ldslack = (islack != -1) ? setupwritei(setup,islack) : 0;

    if (!ldslack) {                     /* take the last disk of the sequence */
        islack = writelasti(setup, inew + 1, NOPASS_EMPTY | NOFREEZE_OK);
        ldslack = &setup->confld[islack];
        if (!ldslack->type) {           /* currently not used */
            memset(ldslack, 0, sizeof (*ldslack));
        }
    }
    else {
        /* check contiguous writeable logical disks between new,slack */
        if (inew < islack) {
            lo = inew;
            hi = islack;
        }
        else {
            lo = islack;
            hi = inew;
        }

        endcontig = writelasti(setup, lo + 1, PASS_EMPTY | FREEZE_OK | EMPTY_OK);
        if (endcontig < hi)
            error("non contiguous, unprotected logical drives");
    }
    if (!ldslack->type)
        ldslack->type = ldnew->type;

    if (ldnew->type) { /* adjust an existing disk */
        slackadjust = ldnew->cSect - size;
    }
    else { /* create a logical disk */
        ldnew->type = setup->defaultldtype;
        ldnew->flag = 0;
        slackadjust = -size;
    }
    sc_setupld(setup, ldnew, size);

    /* adjust the slack disk */
    sc_setupld(setup, ldslack, ldslack->cSect + slackadjust);

    sc_checkusersetup(setup, warn, errorbuf);
}

/*---------------------------------------------------- sc_checkusersetup() -*/
static int sc_checkusersetup(setup, reportproc, escapebuf)
register struct setup *setup;
jmp_buf escapebuf;
int (*reportproc)();
{
    register int i;
    register struct confld *ld;
    register int check = 0;
    int allok = TRUE;
    int tsum;
    register struct sector0 *pd = PD0(setup);
    int countSWAP = 0;
    int countPWF = 0;

    /* check out reserved first */
    ld = &setup->confRsv;
    tsum = pd->sechead * 3;
    if (ld->cSect < tsum) {
        reperr(reportproc, "Reserved are size too small, at least %d", tsum);
        allok = FALSE;
    }
    if (ld->cSect > MAX_SIZE_RV) {
        reperr(reportproc, "Reserved size too big, %d max", MAX_SIZE_RV);
        allok = FALSE;
    }
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++)
        if (ld->type) {
            if (ld->type == LD_SWAP)
                countSWAP++;
            else if (ld->type == LD_PWF)
                countPWF++;
            else if (ld->type != LD_USERSPARE)
                reperr(reportproc, "check: ld %d, unsupported disk type %s",
                        i, ascldtype[ld->type]);
            if (ld->cSect < MIN_LD_SIZE) {
                reperr(reportproc, "check: ld %d, bad size %d", i, ld->cSect);
                allok = FALSE;
            }
            if (ld->cSect % pd->sechead) {
                reperr(reportproc, "check: ld %d, bad size modulo %d",
                        i, ld->cSect);
                allok = FALSE;
            }
            check += ld->cSect;
        }
    if (countSWAP > 1)
        warn("At most one SWAP area per disk");
    if (countPWF > 1)
        warn("At most one PWRF area per disk");
    if (setup->cAvail != check) {
        allok = FALSE;
        reperr(reportproc, "available sectors (%d) != ld sizes (%d)",
                setup->cAvail,check);
    }
    if (setup->cTotalSect !=
            (tsum = pd->cyldisk
                  * pd->headcyl
                  * pd->sechead)) {
        allok = FALSE;
        reperr(reportproc, "total sect (%d) != physdisk sectors (%d)",
                setup->cTotalSect, tsum);
    }
    if (setup->cTotalSect != (tsum = check + setup->confRsv.cSect)) {
        allok = FALSE;
        reperr(reportproc, "total sect (%d) != ld + rsv (%d)",
                setup->cTotalSect, tsum);
    }
    if ((!allok) && escapebuf)
        longjmp(escapebuf,1);
    return(allok);
}

static void sc_printphysdisk(setup)
struct setup *setup;
{
    register struct sector0 *pd = PD0(setup);
    register int temp;

    temp = setup->dev.dev_num;
    printf("\n");
    printf("  Controller Type: SCSI\n");
    printf("Controller Number: %d\n", temp >> 4);
    printf("     Drive Number: %d\n", temp & 0x0f);
    printf("   Unit Specifier: c%dd%d\n", temp >> 4, temp & 0x0f);
    printf("   # of Cylinders: %d\n", pd->cyldisk);
    printf("       # of Heads: %d\n", pd->headcyl);
    printf("# of Sectors/Head: %d\n", pd->sechead);
    printf("    Reserved Area: %d sectors\n", setup->confRsv.cSect);
    printf("    Total Sectors: %d\n", setup->cTotalSect);
    printf("Available Sectors: %d\n", setup->cAvail);

    xprintf("      Setup Flags: $%x ==>", setup->flag);
    if (setup->flag & SET_USED)  xprintf("Used ");
    if (setup->flag & SET_WRIT)  xprintf("Written ");
    if (setup->flag & SET_FIG)   xprintf("Fig ");
    if (setup->flag & SET_NEW)   xprintf("New ");
    if (setup->flag & SET_OLD)   xprintf("Old ");
    if (setup->flag & SET_MOD)   xprintf("Mod");
    xprintf("\n");
    printf("\n");
}

/*---------------------------------------------------- sc_printsetup() -----*/
static void sc_printsetup(setup)
register struct setup *setup;
{
    register int i;
    register struct confld *ld;

    printf("\n  %d sectors available\n", setup->cAvail);
    printf(" ld type st    size\n");
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++)
        if (ld->type)
            sc_printld(i, ld);

    sc_checkusersetup(setup,warn,0);
}

static void sc_printld(i, ld) /* ADD VERBOSE FLAG */
struct confld *ld;
{
    printf(" %2d %4s ", i, ascldtype[ld->type]);
    if (ld->flag & CFLD_PROTECT)
        printf(" P");
    else if (ld->flag & CFLD_FREEZE)
        printf(" F");
    else
        printf("  ");
    printf(" %7d\n", ld->cSect);
}

/*---------------------------------------------------- sc_setupldtype() ----*/
static void sc_setupldtype(setup, ld, size, type)
register struct confld *ld;
{
    ld->type = type;
    sc_setupld(setup, ld, size);
}

/*---------------------------------------------------- sc_setupld() --------*/
static void sc_setupld(setup, ld, size)
struct setup *setup;
register struct confld *ld;
{
    if (!size) {
        memset(ld, 0, sizeof (*ld));
        return;
    }
    if (&setup->confRsv != ld && size < MIN_LD_SIZE) {
        error("Attempt to set logical disk %d to %d sectors (%d is smallest)",
                ld - setup->confld, size, MIN_LD_SIZE);
    }
    ld->cSect = ld->cAssignSect = size;
    if (ld->cSect == 0)
        ld->type = 0;
}

/*---------------------------------------------------- sc_writeconfig() ------
/ apply the selected configuration to disk
/ TODO: have some option to allow the user to suggest changes when
/       things don't fit
/---------------------------------------------------------------------------*/
static void sc_writeconfig(setup)
struct setup *setup;
{
    int count;

    xprintf("sc_writeconfig(),");
    if (!(setup->flag & SET_FIG))
        sc_applyconfig(setup);

    sc_rvwriteconfig(setup);

    setup->flag |= SET_WRIT;            /* mark all logical disks as protected */
}

/*---------------------------------------------------- sc_applyconfig() ----*/
static void sc_applyconfig(setup)
struct setup *setup;
{
    register int i;

    xprintf("sc_applyconfig(), ");
    setup->flag |= SET_FIG;
    sc_assign1start(setup, 1);
    for (i = 0; i < 10 && assign1start(setup, 0); i++)
        /* do nothing */ ;
    if (i >= 10)
        software_bug("sc_applyconfig", "Cannot configure after %d tries", i);
    sc_applystart(setup);
    sc_checkapplysetup(setup, warn, errorbuf);
}

/*---------------------------------------------------- sc_assign1start() -----
/ start handling protected disks
/
/ this procedure selects contiguous groups of non protected disks and passes
/ these sections of disk to assign2start
/ return 0 if no change made
/---------------------------------------------------------------------------*/
sc_assign1start(setup, firstflag)
struct setup *setup;
{
    int change;
    int i,j;
    int avail;
    register struct confld *ld;
    register int nextstart;

    change = 0;
    i = RV_LOGDR;
    nextstart = 0;
    while (i < LOGDR) {
        ld = &setup->confld[i];
        if ((ld->flag & CFLD_PROTECT) || !ld->type) {
            if (firstflag && ld->type) {
                nextstart = ld->start + ld->cSect;
            }
            i++;
            continue;
        }
        if (firstflag)
            ld->start = nextstart;
        j = writelasti(setup, i, PASS_EMPTY | EMPTY_OK | FREEZE_OK);
        avail = sc_sizesection(setup, i, j);
        xprintf("sc_assign1start: i = %d, j = %d, avail = %d, ", i, j, avail);
        change += sc_assign2start(setup, i, j, avail, firstflag);
        i = j + 1;
    }
    xprintf("change = %d\n",change);
    return(change);
}

/*---------------------------------------------------- sc_assign2start() -----
/ assign physical disk starting addresses
/ this is guessing at where to start
/
/   a clean setup is assumed.  ie. xx->start,
/   xx->cAssign, etc. are 0.
/
/ the spare handling seems awkward.  the tracks skipped by the skip track type
/ disks are not considered in calculating the spare requirements.  this is so
/ that spares can be relatively close to where the bad sectors are.
/
/ Some function which gives average distance to spare needs to be computed.
/ This function would be used to determine the exact placement and usage of
/ spare tracks.  A few extra spare tracks scattered here and there could do
/ much to guarentee a minimal distance to spare
/
/ Return 0 if no change made; else number of changes
/----------------------------------------------------------------------------*/
sc_assign2start(setup, i, j, avail, firstflag)
register struct setup *setup;
{
    register struct confld *ld;
    register int nextdiskstart;
    register int ii;
    register int change = 0;
    register int oldskip;

    /* pick an initial assignment, start with the reserved area */
    /* dummy up a reserved area logical disk */
    xprintf("sc_assign2start %d, ", firstflag);
    ii = i;
    ld = &setup->confld[ii];
    nextdiskstart = ld->start;
    for (; ii <= j; ld++, ii++)
        if (ld->type) {
            if (ld->flag & CFLD_PROTECT)
                software_bug("sc_assign2start", "Protected logical disk encountered");
            /* part of a new sequence */
            if (ld->start != nextdiskstart) {
                ld->start = nextdiskstart;
                change++;
            }
            if (firstflag)
                ld->cAssignSect = ld->cSect;
            avail -= ld->cAssignSect;
            nextdiskstart += ld->cAssignSect;
        }

    if (avail) { /* sizes didn't quite match up */
        if (avail > 0)
            ii = writelasti(setup, i, PASS_EMPTY | FREEZE_OK | NOEMPTY_OK);
        else
            ii = writelasti(setup, i, PASS_EMPTY | NOFREEZE_OK | NOEMPTY_OK);
        ld = &setup->confld[ii];
        ld->cAssignSect += avail;
        change++;
    }

    if (change)
        xprintf("%d changes\n", change);
    else
        xprintf("no change\n");

    return(change);
}

/*---------------------------------------------------- sc_sizesection() ------
/ determine the number of sectors available from ld'i' upto and including
/ ld'j'
/---------------------------------------------------------------------------*/
int sc_sizesection(setup, i, j)
register struct setup *setup;
{
    register struct confld *ld;
    register int beginblock, endblock;

    ld = &setup->confld[i];
    if (!ld->type)
        software_bug("sc_sizesection", "First logical disk %d is empty", i);
    beginblock = ld->start;

    j++;
    if (j == LOGDR)
        endblock = setup->cTotalSect;
    else {
        ld = &setup->confld[j];
        if (!ld->type)
            software_bug("sc_sizesection", "Second logical disk %d is empty", j);
        endblock = ld->start;
    }
    return(endblock - beginblock);
}

/*---------------------------------------------------- sc_applystart() -----*/
static void sc_applystart(setup)
register struct setup *setup;
{
    register struct confld *ld;
    register int nextdiskstart;
    register int i;

    /* CHECK PROTECT */

    xprintf("sc_applystart(), ");
    ld = &setup->confRsv;               /* reserved area */
    nextdiskstart = ld->cAssignSect;    /* reserved area size */
    for (ld = setup->confld, i = 0; i < LOGDR; ld++, i++) {
        switch (ld->type) {
            case LD_PWF:
            case LD_USERSPARE:
            case LD_SWAP:
                if (nextdiskstart != ld->start)
                    software_bug("sc_applystart",
                            "Logical disk %d, diskstart doesn't match", i);
                nextdiskstart += ld->cAssignSect;
                break;
            default: /* no logical disk marked */
                break;
        }
    }
}

/*---------------------------------------------------- sc_printapplysetup() */
static void sc_printapplysetup(setup, rsvtoo)
struct setup *setup;
{
    register int i;
    struct confld *ld;

    printf("    %d sectors available\n", setup->cAvail);
    printf(" ld type  st    size  start (CCC.HH)\n");
    printf(" -- ----- -- ------- ---------------\n");
    if (rsvtoo)
        sc_printapplyld(setup, -1, &setup->confRsv);
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++)
        if (ld->type)
            sc_printapplyld(setup, i, ld);
    sc_checkapplysetup(setup, warn, 0);
}

/*---------------------------------------------------- sc_printapplyld() ---*/
static void sc_printapplyld(setup, i, ld)
struct setup *setup;
struct confld *ld;
{
    register int c, h;

    printf(" %2d %5s ", i, ascldtype[ld->type]);
    if (ld->flag & CFLD_PROTECT)
        printf(" P");
    else if (ld->flag & CFLD_FREEZE)
        printf(" F");
    else
        printf("  ");
    printf(" %7d", ld->cAssignSect);

    h = ld->start / PD0(setup)->sechead;
    c = h / PD0(setup)->headcyl;
    h = h % PD0(setup)->headcyl;
    printf(" %7d(%03x.%02x)\n", ld->start, c, h);
}

/*---------------------------------------------------- sc_checkapplysetup() */
static int sc_checkapplysetup(setup, reportproc, escapebuf)
register struct setup *setup;
int (*reportproc)();
jmp_buf escapebuf;
{
    register int i;
    register struct confld *ld;
    register int check;                 /* total sector count */
    int allok = TRUE;
    int tsum;
    int countSWAP = 0;                  /* number of SWAP lds */
    int countPWF = 0;                   /* number of PWF lds */

    if (!(setup->flag & SET_FIG))       /* not changed - no need to check */
        return;

    /* check out reserved first */
    ld = &setup->confRsv;
    tsum = PD0(setup)->sechead * 3;
    if (ld->cAssignSect < tsum) {
        reperr(reportproc, "Reserved area too small, at least %d", tsum);
        allok = FALSE;
    }
    if (ld->cAssignSect > MAX_SIZE_RV) {
        reperr(reportproc, "Reserved area too big, %d max", MAX_SIZE_RV);
        allok = FALSE;
    }
    check = ld->cAssignSect;
    for (i = 0, ld = setup->confld; i < LOGDR; i++, ld++) {
        if (ld->type) {                 /* logical disk defined */
            if (ld->type == LD_SWAP)    /* swap type */
                countSWAP++;
            else if (ld->type == LD_PWF)/* power-fail type */
                countPWF++;
            else if (ld->type != LD_USERSPARE)
                software_bug("sc_checkapplysetup",
                        "Unsupported SCSI logical disk type %s",
                        ascldtype[ld->type]);
            if (ld->cAssignSect < MIN_LD_SIZE) {
                reperr(reportproc, "check: Logical disk %d - bad size %d",
                        i, ld->cAssignSect);
                allok = FALSE;
            }
            if (ld->cAssignSect % PD0(setup)->sechead) {
                reperr(reportproc, "check: Logical disk %d - bad size modulo %d",
                        i, ld->cAssignSect);
                allok = FALSE;
            }
            if (ld->start != check)
                reperr(reportproc, "Logical disk %d starting address ($%x) inconsistent",
                        i, ld->start);
            check += ld->cAssignSect;
            if (ld->cAssignSect != ld->cSect)
                warn("check: Logical disk %d - size changed from %d to %d",
                        i, ld->cSect, ld->cAssignSect);
        }
    }

    if (countSWAP > 1) {
        allok = FALSE;
        reperr(reportproc, "At most one SWAP area per disk");
    }
    if (countPWF > 1) {
        allok = FALSE;
        reperr(reportproc, "At most one PWRF area per disk");
    }
    if (setup->cTotalSect != check) {
        allok = FALSE;
        reperr(reportproc,
                "Total sector count (%d) does not match assigned sector count (%d)",
                setup->cTotalSect, check);
        if (check > setup->cTotalSect)
            errxit("Allocated sector count (%d) exceeds available sector count (%d).",
                    check, setup->cTotalSect);
    }

    if ((!allok) && escapebuf)
        longjmp(escapebuf, 1);
    return(allok);
}

/*---------------------------------------------------- sc_readphys0() ------*/
static void sc_readphys0(setup)
register struct setup *setup;
{
    register struct sector0 *pd;

    setup->flag |= SET_CLEAR;           /* in case an error */
    setup->memPhysdisk = memalloc(sizeof (struct sector0), MEM_ALLIGN);
    pd = PD0(setup);
#ifdef RO_UNIX
    setup->fd = NULL;	/* file descriptor for slice0 is kept in scsi_fd  */
#else
    setup->fd = pdopen(&setup->dev);    /* open physical disk */
#endif

#ifdef UNIXSTANDALONE
    UNIXfd = setup->fd;
#endif
    if (pdfread(setup->fd, 0, pd) < 0)  /* read sector 0 */
        error("Cannot access physical disk");
#ifdef UNIXSTANDALONE
    UNIXpdnext = pd->nextsec;
#endif
    if (strcmp(pd->id, "INIT"))
        error("Disk is not formatted.");
    if (pd->FORMAT_LEVEL)
        errxit("Version problem with %d, run a new dsetup", pd->FORMAT_LEVEL);
    if (pd->rv_strt != 0)
        errxit("Reserved area doesn't start at 0.  Cannot continue!");
    setup->flag &= ~SET_CLEAR;
}

/*---------------------------------------------------- sc_rvwriteconfig() --*/
static void sc_rvwriteconfig(setup)
register struct setup *setup;
{
    register struct confld *ld = setup->confld;
    register struct logtype *logdrive;
    register int i;
    register struct sector0 *pd = PD0(setup);
    int ldnum;
    int nextsect;

    /*-----------------------------------------------
    / verify protected disk parameters still the same
    /----------------------------------------------*/

    for (i = 0; i < LOGDR; i++, ld++)
        if (ld->flag & CFLD_PROTECT) {
            logdrive = &pd->logdrive[ld->logdriveindex];
            if (logdrive->ld_strt == ld->start
                    && logdrive->ld_size == ld->cAssignSect
                    && logdrive->ld_type == ld->type
                    && logdrive->ld_mirror == -1) {
                continue;
            }
            else
                software_bug("sc_rvwriteconfig",
                        "Protected logical disk data have been touched.");
        }

    /*-------------------------------------------
    / initialize the boot pointer if a new config
    /------------------------------------------*/

    if (setup->flag & SET_NEW) {
        struct memuse *xmem;
        struct boot_list *b;

        xmem = memalloc(0x400, MEM_ALLIGN);
        b = (struct boot_list *) (xmem->p);
        pdreadany(setup->fd, BOOT_SECTOR, b);
        b->boot_strt = BOOT_START;
        if (!(b->boot_size)) {
            b->boot_size = setup->confRsv.cAssignSect - BOOT_START;
            pdwrite(setup->fd, BOOT_SECTOR, b);
        }
        memfree(xmem);
    }

#ifdef UNIXSTANDALONE
    /*--------------------------------------------------------
    / Ask what the next avail sector is, plug this number into
    / the new sector 0 structure.  This will be written back.
    /-------------------------------------------------------*/

    ioctlck(setup->fd, NEXTRVSEC, &nextsect);
    pd->nextsec = nextsect;
#endif

    /*--------------------------------------------------------
    / fill in the new sector 0
    / what order should it happen in, consider the NEXTRVSEC
    / for now CLEAR MIRROR INFORMATION
    /-------------------------------------------------------*/

    ld = setup->confld;
    logdrive = pd->logdrive;
    for (i = 0; i < LOGDR; i++, ld++, logdrive++) {
        memset(logdrive, 0, sizeof (*logdrive));
        if (!ld->type)                  /* never mind this one - just blank */
            continue;
        ldnum = i;                      /* for count of valid ld table entries */
        logdrive->ld_strt = ld->start;  /* starting sector of logical disk */
        logdrive->ld_size = ld->cAssignSect;    /* size of logical disk */
        logdrive->ld_type = ld->type;   /* logical disk type */
        logdrive->ld_mirror = -1;       /* mirror logical disk num */
        logdrive->ld_skip = 0;          /* for format '0' only */
    }
    pd->pd_ldnum = ldnum + 1;           /* number of logical disks */
    pd->rv_size = setup->confRsv.cAssignSect;   /* reserved area size */
    pd->rv_skip = 0;                    /* for format '0' only */
    pd->DSETUP_LEVEL = CURRENT_DSETUP_LEVEL;    /* dsetup level */
    pdwrite(setup->fd, 0, pd);          /* write sector 0 */
    ioctlck(setup->fd, INIT, 0);        /* initialize the disk */
}

/*----------------------------- End of dsetscsi.c --------------------------*/
