/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) setacl.c: version 25.1 created on 12/2/91 at 13:30:03	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)setacl.c	25.1	12/2/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*							*/
/*	Copyright (c) 1989 ARIX	*/
/*	  All Rights Reserved  	*/

/*	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF ARIX	*/
/*	The copyright notice above does not evidence any   	*/
/*	actual or intended publication of such source code.	*/



/* setacl - Sets / replaces/ removes an ACL from a file */
/* Written 10-89, mer */
/* Re-written 9-91, mbm */

#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <stdio.h>
#include <limits.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <sys/acl.h>
#include <time.h>

#define	NUM_ACL_TYPES	4
#define	LINE_LEN  	80

#define	REPLACE	0x0001		/* Replace an acl with a new one */
#define	DEFAULT	0x0002		/* Assign default ACL to a directory */
#define	LINK 	0x0004		/* Share an ACL with another file */
#define	UNLINK	0x0008		/* Unlink an ACL */
#define	NAMED	0x0010		/* Create & add to a named ACL */
#define	IACTIVE	0x0011		/* interactive.......not done yet */

extern char	*optarg;
extern int	optind;

void 	exit ();
void 	qsort (); 
char  	*malloc();
long 	strtol();
struct	passwd	*getpwnam ();
struct	group	*getgrnam ();

void	get_new_acl(); 
void	set_new_acl();
void	link_the_acl (); 
void	unlink_the_acl ();
void	adjust_for_tz();
void	argsok();
int 	comp_it();


struct	type_list {
	char	*t_name;
	int	val;
	} type_list [NUM_ACL_TYPES]= {

	"A_USER", A_USER,
	"A_GROUP", A_GROUP,
	"A_TIME", A_TIME,
	"A_DATE", A_DATE };

int	flags = 0;		/* Used to store non-default settings */

main (argc, argv)
int	argc;
char	**argv;
{
	char	*file_name;		/* File whose ACL is to be set */
	char	*file_to_share;		/* If -l, file which wishes to share
					   an ACL */
	int	opt;

	while( (opt = getopt( argc, argv, "rdlun:")) != EOF ) {

		switch (opt) {

		case 'r':
			flags |= REPLACE;
			break;

		case 'd':
			flags |= DEFAULT;
			break;

		case 'l':
			flags |= LINK;
			break;

		case 'u':
			flags |= UNLINK;
			break;

		case 'n':
			flags |= NAMED | REPLACE;
			file_name = optarg;
			break;

		default:
			usage();
			break;
		}
	}

	if ( ! (flags & NAMED) ){

	     /* Get file name(s), or give insufficient argument message */
	     if ( (flags & LINK) && (argc > optind + 1) ){

		     file_name = argv[optind];
		     file_to_share = argv[optind + 1];
	     }

	     else if ( !(flags & LINK) && (argc > optind) )

		     file_name = argv[optind];
	     else
		     usage ();

	     if ( access (file_name, 00) == -1){
		     fprintf (stderr, "setacl: %s does not exist\n", file_name);
		     exit (1);
	     }
	}


	/*
	 * do some misc argument checking.
	 */
	 argsok(file_name,file_to_share);


	/*
	 * Either (un)link the ACLs at this time, or accept new information 
	 */

	if (flags & LINK)
		link_the_acl (file_name, file_to_share);

	else if (flags & UNLINK)
		unlink_the_acl (file_name);

	else
		get_new_acl (file_name);

	/* NOTREACHED */
}



/*
 *  get_new_acl  -- grab new acl entries from stdin, and either exit
 *                 with an error (1), or call set_new_acl.
 */
void
get_new_acl(filename)
char *filename;
{

	char input[LINE_LEN];
	acl_t tmp;
	uint num_el=0;           /*number of elements in the buffer*/
	acl_t *buffer;           /*needs room for 10k worth of acls */
	struct stat acl_stat;		/* Where stat of ACL is stored */

	if ( (buffer = (acl_t *) malloc(MAX_ACL_SIZE)) == NULL ) {
		fprintf (stderr, "setacl: Unable to allocate memory\n");
		exit (1);
	}


	if ( ! (flags & REPLACE)) {

		if ( acl( filename, (flags & DEFAULT) ? ACLSTAT_DEFAULT 
		     : ACLSTAT,(char *)  &acl_stat,0) != -1){

			if ( acl( filename, (flags & DEFAULT) ? GETACL_DEFAULT
			: GETACL,(char *)  buffer, acl_stat.st_size) == -1){

				fprintf (stderr,
				 "setacl: Unable to get ACL for file %s\n"
				 ,filename);
				exit (1);
			}

			num_el = acl_stat.st_size / sizeof(acl_t);
		}
	}

	buffer += num_el;     /* fun with pointers */

	while (fgets(input,LINE_LEN,stdin)) {

		if (num_el == MAX_ACL_SIZE / sizeof(acl_t)){
			fprintf(stderr,"EXECEDED MAX ACL SIZE. Nothing set\n");
			exit(1);
		}

		if (parse_input(input,&tmp))
			continue;                /* Bogus input don't accept */

		*buffer++ = tmp;
		num_el++;

	} 

	if (!num_el ) {
		fprintf(stderr,"No data, No ACL set\n");
		exit(1);
	}

	buffer = buffer - num_el;   /* back up to the start */
	qsort((char *) buffer, num_el , sizeof (acl_t), comp_it);
	set_new_acl((char *)buffer,(int ) (num_el * sizeof(acl_t)), filename);

}



/*
 *  parse_input - munge through the input line and fill in the acl_t
 *                with good data.  Return 0 for success, and 1 for fail.
 */
parse_input(line,tmp_acl)
char	*line;
acl_t *tmp_acl;
{
	uint	count;
	char *rest;      
	char start[LINE_LEN], end[LINE_LEN];

	if ((rest = strchr(line,' ') ) == NULL){
		fprintf(stderr,"Bad Input Data. \n");
		return (1);
	}

	*rest = '\0';
	rest++;

	for (count = 0, tmp_acl->type = 0; count < NUM_ACL_TYPES; count++)
		if (strcmp (line, type_list[count].t_name) == 0)
			tmp_acl->type = type_list[count].val;
	
	if (! tmp_acl->type)   /*didn't find one*/
		return (1);

	switch (tmp_acl->type) {

		case A_USER:
		case A_GROUP:

			if ( parse_user_group(rest,tmp_acl) )
				return (1);
			break;

		case A_DATE:

			if (sscanf(rest,"%s %s",start,end) != 2){
				fprintf(stderr, 
				"Wrong # of arguments for an A_DATE ACL!\n");
				return (1);
			}

			if (((tmp_acl->date_start= setdate(start)) == -1)
			   || ((tmp_acl->date_end= setdate(end)) == -1))
				return (1);

			break;

		case A_TIME:

			if (sscanf(rest,"%hu %hu %hu %hu",&tmp_acl->hour_start,
			    &tmp_acl->hour_end,&tmp_acl->day_start,
			    &tmp_acl->day_end) != 4){

				fprintf(stderr, 
				"Wrong # of arguments for an A_TIME ACL!\n");
				return (1);
			}

			if (check_time(tmp_acl->hour_start,tmp_acl->hour_end, 
			    tmp_acl->day_start,tmp_acl->day_end) == -1) 
				return (1);

			adjust_for_tz(&tmp_acl->hour_start,&tmp_acl->hour_end, 
				      &tmp_acl->day_start, &tmp_acl->day_end);
			
			break;

		default:
			fprintf(stderr,"Unknown Acl Type!\n");
			return (1);
	}

	return (0);   /* looks good */
}


/*
 * parse_user_group -- parse through the input string and assemble the 
 *                     data for an A_USER or A_GROUP acl.  Return 0 for 
 *                     success, or 1 for failure.
 */
parse_user_group(line,myacl)
char *line;
acl_t *myacl;
{

	char name[LINE_LEN];
	char *ptr;
	struct passwd *pw;
	struct group *gr;
	int id;

	if (sscanf(line,"%s %hu %hu %hu %hu %hu",name,&myacl->mode, 
	    &myacl->hour_start,&myacl->hour_end, &myacl->day_start, 
	    &myacl->day_end) != 6 ){

		fprintf(stderr, 
		"Wrong number of arguments for an A_USER/A_GROUP ACL!\n");
		return(1);
	}


	/*
	 * the name field may be a uid or uname (gid/gname). If it is all
	 * digits, we'll assume it is a uid, although there is a chance
	 * we could be wrong. :-(
	 */

	if ((id = (uint ) strtol(name, &ptr, 0)) && ptr)   /* if a number */
		if (myacl->type == A_USER) {

			if ((pw = getpwuid(id)) == NULL){
				fprintf(stderr,"Unknown userid: %d\n",id);
				return(1);
			}

			myacl->id = id;
		}
		else {

			if ((gr = getgrgid(id)) == NULL){
				fprintf(stderr,"Unknown groupid: %d\n",id);
				return(1);
			}

			myacl->id = id;
		}

	else            /* it's a uname/gname */

		if (myacl->type == A_USER) {

			if ((pw = getpwnam(name)) == NULL){
				fprintf(stderr,"Unknown username: %s\n",name);
				return(1);
			}

			myacl->id = pw->pw_uid;
		}
		else {

			if ((gr = getgrnam(name)) == NULL){
				fprintf(stderr,"Unknown group name: %s\n",name);
				return(1);
			}

			myacl->id = gr->gr_gid;
		}
		
			
	/*
	 * check the mode bit. (mode is a uint, so <0 is not needed.)
	 */
	if (myacl->mode > 7 ){
		fprintf(stderr, "Incorrect mode: %d",myacl->mode);
		return (1);
	}


	/*
	 * check the hour/day stuff
	 */
	if (check_time(myacl->hour_start,myacl->hour_end, 
	    myacl->day_start,myacl->day_end) == -1) 
		return (1);

	adjust_for_tz(&myacl->hour_start,&myacl->hour_end, 
			 &myacl->day_start, &myacl->day_end);
			
	return (0);
}





/*
 *  set_the_acl -- does an acl system call, based on flag. This guy 
 *                 does not return, it exits with:  1=fail, and 0=success.
 */
void
set_new_acl(buf,size,filename)
char *buf, *filename;
int size;
{
	if (flags & DEFAULT) {

		if ( acl( filename, SETACL_DEFAULT, buf, size) == -1){
			fprintf(stderr,"setacl: Can't set DEFAULT ACL on %s\n",
			        filename);
			exit (1);
		}

	}
	else if (flags & NAMED) {

		if ( acl( filename, SETNACL, buf, size) == -1){
			fprintf (stderr,"setacl: Can't create named ACL %s\n",
			         filename);
			exit (1);
		}

	}
	else {

		if ( acl( filename, SETACL, buf, size) == -1){
			fprintf (stderr,"setacl: Can't set ACL on %s\n",
			         filename);
			exit (1);
		}

	}

	exit (0);
}





/*
 *   link_the_acl -- nuf said.
 */
void
link_the_acl (file1, file2)
char	*file1, *file2;
{

	if ( flags & DEFAULT ){

		if ( acl( file1, ACL_LINK_DEFAULT, file2, 0) == -1 ){
			fprintf (stderr, 
			 "setacl: Unable to link default ACL to directory %s\n",
			 file2);
			exit (1);
		}

	}
	else {

		if ( acl (file1, ACL_LINK, file2, 0) == -1){
			fprintf (stderr, 
			  "setacl: Unable to link ACL to file %s\n", file2);
			exit (1);
		}

	}
}



/*
 *  unlink_the_acl -- well said.
 */
void
unlink_the_acl (fname)
char	*fname;
{
	struct	stat	acl_stat;

	/* Make sure that file #1 actually has a (default) acl */
	if ( acl( fname, (flags & DEFAULT) ? ACLSTAT_DEFAULT : ACLSTAT, 
	     (char *) &acl_stat,0) == -1){

		fprintf ( stderr, "setacl: %s has no acl associated with it\n",
			  fname);
		exit (1);
	}

	/* Unlink the ACL */
	if ( flags & DEFAULT ){

		if ( acl( fname, ACL_UNLINK_DEFAULT, (char *)0, 0) == -1){
			fprintf ( stderr, 
		"setacl: Unable to unlink default ACL from directory %s\n"
			 ,fname);
			exit (1);
		}
		
	}
	else {

		if ( acl( fname, ACL_UNLINK, (char *)0, 0) == -1){
			fprintf ( stderr, 
			"setacl: Unable to unlink ACL from file %s\n", fname);
			exit (1);
		}

	}
}



/*
 *  argsok -- Check the arguments to a LINK and DEFAULT. 
 *            If just DEFAULT, file1 must be a dir. If DEFAULT & LINK,
 *            file1 must be a dir or NAMED ACL, and have an ACL; also,
 *            file2 must be a dir. If just LINK, file1 must have an ACL,
 *            and file2 must exist.
 */
void
argsok(file1,file2)
char *file1,*file2;
{


	struct	stat	stbf, stbf2;
	int  	type = ACLSTAT;

	if ( stat ( file1, &stbf ) == -1){
		fprintf (stderr, "setacl: Can't stat %s\n", file1);
		exit (1);
	}

	if ( flags & DEFAULT ) {
		if (! ( flags & LINK )) {

			if ( (stbf.st_mode & S_IFMT) != S_IFDIR) {
				fprintf(stderr,
				"setacl: %s - not a directory\n",file1);
				exit(1);
			}
		}
		else {
			if ( (stbf.st_mode & S_IFMT) == S_IFDIR) 
				type = ACLSTAT_DEFAULT;

			else if ( (stbf.st_mode & S_IFMT) != S_IFACL) {
				fprintf(stderr, 
				"setacl: %s - not a directory or named ACL\n", file1);
				exit(1);
			}
		}
	}	

	if ( flags & LINK ) {

		if (acl(file1,type,&stbf,0) == -1){
			fprintf(stderr,"setacl: %s has no ACL.\n",file1);
			exit(1);
		}

		if ( stat ( file2, &stbf2 ) == -1){
			fprintf (stderr, "setacl: Can't stat %s\n", file2);
			exit (1);
		}

		if ( ( flags & DEFAULT) && 
		     (( stbf2.st_mode & S_IFMT) != S_IFDIR )) {
			fprintf (stderr,"setacl: %s - not a directory\n",file2);
			exit (1);
		}
	}

}


/* This function is called by qsort to help sort the acls. */
int
comp_it (record1, record2)
acl_t *record1, *record2;
{
	return ( (record1->type > record2->type) -
	   (record1->type < record2->type));
}



/* 
 *  Determines if times & days entered by users are legal, only 0 - 6 are legal
 *  for days, 0 - 23 for times. 
 */

int
check_time (time_start, time_end, day_start, day_end)
ushort	time_start, time_end, day_start, day_end;
{

	/*NOTE- These are all unsigneds. Thus no "< 0" checks */
	
	if (time_start > 23 || time_end > 23){
		fprintf (stderr, "Values for time of day must be 0 - 23\n");
		return (-1);
	}

	if (day_start > 6 || day_end > 6){
		fprintf (stderr, "Values for days of week must be 0 - 6\n");
		return (-1);
	}

	return (0);
}


/*
 * This function  changes the time-of-day / day-of-week pair into GMT.
 */
void
adjust_for_tz(st_hr,end_hr,st_day,end_day)
ushort *st_hr, *end_hr, *st_day, *end_day;
{

ushort tmp, hours_diff;
time_t tloc;

	tloc = time((long *)0);

	/* calc the hours we are off of GMT */
	if (localtime(&tloc)->tm_isdst)
		hours_diff =  altzone / (60*60);
	else
		hours_diff = timezone / (60*60);

	if ((tmp = (*end_hr + hours_diff) % 24) < *end_hr)
		*end_day =  (*end_day + 1) % 7;
	*end_hr = tmp;

	if ((tmp = (*st_hr + hours_diff) % 24) < *st_hr)
		*st_day =  (*st_day + 1) % 7;
	*st_hr=tmp;
}
	
/* 
	This function takes a date MM/DD/YY (11/21/71, 1/15/85, 
	12/3/86, 3/3/88) and convert it to seconds since the birth of
	UNIX.  It will also take a MM/DD (10/18, 1/20, 12/4, 4/4) and 
	assume a default of the current year and convert it to seconds 
	since the birth of UNIX.
*/

setdate(date)
char		*date;
{

#define	year_size(A)	(((A) % 4) ? 365 : 366)

	static short month_size[12] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
	register int i;

	struct tm *tmx;
	struct tm *localtime();
	time_t now;
	long clck;
	int mm, dd, yy;


	now = time((long *) 0);
	tmx = localtime(&now); 

	/*  Parse date string  */
	switch(strlen(date)) {
	case 8:
		yy = 1900 + atoi(&date[6]);
		date[5] = '\0';
		dd = atoi(&date[3]);
		date[2] = '\0';
		break;
	case 7:
		yy = 1900 + atoi(&date[5]);
		date[4] = '\0';

	/* checking for 1/23/84 or 12/5/99 */

		if ( !strncmp(&date[1],"/",1) ){
			dd = atoi(&date[2]);
			date[1] = '\0';
		}
		else {
			dd = atoi(&date[3]);
			date[2] = '\0';
		}
		break;
	case 6:
		yy = 1900 + atoi(&date[4]);
		date[3] = '\0';
		dd = atoi(&date[2]);
		date[1] = '\0';
		break;
	case 5:
		yy = 1900 + tmx->tm_year;
		dd = atoi(&date[3]);
		date[2] = '\0';
		break;
	case 4:
		yy = 1900 + tmx->tm_year;

	/* checking for 4/15 or 12/1 */
		if ( !strncmp(&date[1],"/",1) ){
			dd = atoi(&date[2]);
			date[1] = '\0';
		}
		else {
			dd = atoi(&date[3]);
			date[2] = '\0';
		}
		break;
	case 3:
		yy = 1900 + tmx->tm_year;
		dd = atoi(&date[2]);
		date[1] = '\0';
		break;
	default:
		(void) fprintf(stderr, "setacl: bad date/time conversion\n");
		return(-1);
	}
	mm = atoi(&date[0]);

	/*  Validate date elements  */
	if(!((mm >= 1 && mm <= 12) && (dd >= 1 && dd <= 31) && (yy >= 1970))) {
		(void) fprintf(stderr, "setacl: bad date/time conversion\n");
		return(-1);
	}
	
	/*  Build date and time number  */
	for(clck = 0, i = 1970; i < yy; i++)
		clck += year_size(i);

	/*  Adjust for leap year  */
	if (year_size(yy) == 366 && mm >= 3)
		clck += 1;

	/*  Adjust for different month lengths  */
	while(--mm)
		clck += month_size[mm - 1];

	/*  Load up the rest  */
	clck += (dd - 1); /* why I'm doing this I don't know?? */
	clck *= 24;
	clck *= 60;
	clck *= 60;

	/* convert to GMT assuming standard time */
	/* correction is made in localtime(3C) */

	clck += timezone;

	/* correct if daylight savings time in effect */

	if (localtime(&clck)->tm_isdst)
		clck = clck - (timezone - altzone); 

	return(clck);
}




int
usage ()
{
	fprintf (stderr, "Usage: setacl file_name | -d directory\n");
	fprintf (stderr, "       setacl -l (file_with_acl | named_acl) file_to_share_acl\n");
	fprintf (stderr, "       setacl -ld (dir_with_acl | named acl) dir_to_share_acl\n");
	fprintf (stderr, "       setacl -u file_name | named_acl\n");
	fprintf (stderr, "       setacl -r file_name | -d directory\n");
	fprintf (stderr, "       setacl -n named_acl\n");
	exit (1);
}



