/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) rwi.c: version 25.1 created on 11/27/91 at 15:40:35	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)rwi.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
/*
 * rwi.c -- do stuff on the Real World Interface
 */



#include "sensor.h"
#include "sys/ints.h"
#include "sys/uadmin.h"
#include "misc.h"
#include "spm.h"
#include "rwi.h"
#include "adc.h"
#include "novram.h"
#include "cio.h"
#include "spm_debug.h"
#include "sbus_conf.h"
#include "sbus_iom.h"

#define	NO_FAN_MINUTES	5 	/* time with no fan before shutdown */
#define FAN_TIMER_OK	(NO_FAN_MINUTES * 60 + 1)

extern uint	iomap_save();
extern uint	n_seconds;
extern uchar	*iomap();
extern uint	int_counter;
extern uint	pm_booting_done;
extern uint	local_scan_done;
extern uint	poll_local;
extern uint	clock_tick;
extern uint	spm_mem_copy_called;

extern struct sbus_config sbus_config;

static local_ps_t	local_ps = {
  /* reading,	    next,			warn,		time,	*/
  { { 0 }, &local_ps.l_aux_five_volt, &NOVRAM->five_volt_warning, 0,
  /*	name		*/
    "Local power supply Five Volt",
  /*mux,      channel,	state,  type			*/
    0, LOCAL_FIVE_VOLT, NORMAL, LOCAL_FIVE_VOLT },

  { { 0 }, NULL, NULL, 0,
    "Local Voltage Reference", 0, LOCAL_VREF, NORMAL, LOCAL_VREF },

  { { 0 }, &local_ps.l_twelve_volt, &NOVRAM->five_volt_warning, 0,
    "Local power supply Aux Five Volt",
    0, AUX_FIVE_VOLT, NORMAL, AUX_FIVE_VOLT },

  { { 0 }, &local_ps.l_aux_twelve_volt, &NOVRAM->twelve_volt_warning, 0,
    "Local power supply Twelve Volt",
    0, LOCAL_TWELVE_VOLT, NORMAL, LOCAL_TWELVE_VOLT },

  { { 0 }, &local_ps.l_neg_twelve_volt, &NOVRAM->twelve_volt_warning, 0,
    "Local power supply Aux Twelve Volt",
    0, AUX_TWELVE_VOLT, NORMAL, AUX_TWELVE_VOLT },

  { { 0 }, &local_ps.l_aux_neg_twelve_volt, &NOVRAM->neg_twelve_volt_warning, 0,
    "Local power supply Negative Twelve Volt",
    0, LOCAL_NEG_TWELVE_VOLT, NORMAL, LOCAL_NEG_TWELVE_VOLT },

  { { 0 }, &local_ps.l_cage_air, &NOVRAM->neg_twelve_volt_warning, 0,
    "Local power supply Aux Negative Twelve Volt",
    0, AUX_NEG_TWELVE_VOLT, NORMAL, AUX_NEG_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->cage_air_warning, 0,
    "CSS Cage Air Temperature", 0, CAGE_AIR, NORMAL, CAGE_AIR }
};

static main_ps_t	main_ps[NUM_PS_PER_SYS] = {
 {
	/* power supply 0 */
 /* reading, next, warn,		  time,   		*/
  { { 0 }, NULL, &NOVRAM->five_volt_warning, 0,
 /*		name				   		*/
    "Main power supply # 0 Five Volt",
 /*     mux,		channel,      state,     type		*/
    PS_0_5_VOLT_MUX, PS_0_5_VOLT_CHAN, NORMAL, MAIN_FIVE_VOLT },

  { { 0 }, NULL, &NOVRAM->twelve_volt_warning, 0,
    "Main power supply # 0 Twelve Volt",
    PS_0_12_VOLT_MUX, PS_0_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->neg_twelve_volt_warning, 0,
    "Main power supply # 0 Negative Twelve Volt",
    PS_0_NEG_12_VOLT_MUX, PS_0_NEG_12_VOLT_CHAN, NORMAL, MAIN_NEG_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->five_volt_current_kl, 0,
    "Main power supply # 0 Five Volt Current",
    PS_0_I_MON_MUX, PS_0_I_MON_CHAN, NORMAL, MAIN_FIVE_VOLT_CURRENT },

  { { 0 }, NULL, &NOVRAM->main_ps_temp_warning, 0,
    "Main power supply # 0 Temperature",
    PS_0_TEMP_MUX, PS_0_TEMP_CHAN, NORMAL, PS_TEMP }
 }, {
	/* power supply 1 */
  { { 0 }, NULL, &NOVRAM->five_volt_warning, 0,
    "Main power supply # 1 Five Volt",
    PS_1_5_VOLT_MUX, PS_1_5_VOLT_CHAN, NORMAL, MAIN_FIVE_VOLT },

  { { 0 }, NULL, &NOVRAM->twelve_volt_warning, 0,
    "Main power supply # 1 Twelve Volt",
    PS_1_12_VOLT_MUX, PS_1_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->neg_twelve_volt_warning, 0,
    "Main power supply # 1 Negative Twelve Volt",
    PS_1_NEG_12_VOLT_MUX, PS_1_NEG_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->five_volt_current_kl, 0,
    "Main power supply # 1 Five Volt Current",
    PS_1_I_MON_MUX, PS_1_I_MON_CHAN, NORMAL, MAIN_FIVE_VOLT_CURRENT },

  { { 0 }, NULL, &NOVRAM->main_ps_temp_warning, 0,
    "Main power supply # 1 Temperature",
    PS_1_TEMP_MUX, PS_1_TEMP_CHAN, NORMAL, PS_TEMP }
 }, {
	/* power supply 2 */
  { { 0 }, NULL, &NOVRAM->five_volt_warning, 0,
    "Main power supply # 2 Five Volt",
    PS_2_5_VOLT_MUX, PS_2_5_VOLT_CHAN, NORMAL, MAIN_FIVE_VOLT },

  { { 0 }, NULL, &NOVRAM->twelve_volt_warning, 0,
    "Main power supply # 2 Twelve Volt",
    PS_2_12_VOLT_MUX, PS_2_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->neg_twelve_volt_warning, 0,
    "Main power supply # 2 Negative Twelve Volt",
    PS_2_NEG_12_VOLT_MUX, PS_2_NEG_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->five_volt_current_kl, 0,
    "Main power supply # 2 Five Volt Current",
    PS_2_I_MON_MUX, PS_2_I_MON_CHAN, NORMAL, MAIN_FIVE_VOLT_CURRENT },

  { { 0 }, NULL, &NOVRAM->main_ps_temp_warning, 0,
    "Main power supply # 2 Temperature",
    PS_2_TEMP_MUX, PS_2_TEMP_CHAN, NORMAL, PS_TEMP }
 }, {
	/* power supply 3 */
  { { 0 }, NULL, &NOVRAM->five_volt_warning, 0,
    "Main power supply # 3 Five Volt",
    PS_3_5_VOLT_MUX, PS_3_5_VOLT_CHAN, NORMAL, MAIN_FIVE_VOLT },

  { { 0 }, NULL, &NOVRAM->twelve_volt_warning, 0,
    "Main power supply # 3 Twelve Volt",
    PS_3_12_VOLT_MUX, PS_3_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->neg_twelve_volt_warning, 0,
    "Main power supply # 3 Negative Twelve Volt",
    PS_3_NEG_12_VOLT_MUX, PS_3_NEG_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->five_volt_current_kl, 0,
    "Main power supply # 3 Five Volt Current",
    PS_3_I_MON_MUX, PS_3_I_MON_CHAN, NORMAL, MAIN_FIVE_VOLT_CURRENT },

  { { 0 }, NULL, &NOVRAM->main_ps_temp_warning, 0,
    "Main power supply # 3 Temperature",
    PS_3_TEMP_MUX, PS_3_TEMP_CHAN, NORMAL, PS_TEMP }
 }, {
	/* power supply 4 */
  { { 0 }, NULL, &NOVRAM->five_volt_warning, 0,
    "Main power supply # 4 Five Volt",
    PS_4_5_VOLT_MUX, PS_4_5_VOLT_CHAN, NORMAL, MAIN_FIVE_VOLT },

  { { 0 }, NULL, &NOVRAM->twelve_volt_warning, 0,
    "Main power supply # 4 Twelve Volt",
    PS_4_12_VOLT_MUX, PS_4_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->neg_twelve_volt_warning, 0,
    "Main power supply # 4 Negative Twelve Volt",
    PS_4_NEG_12_VOLT_MUX, PS_4_NEG_12_VOLT_CHAN, NORMAL, MAIN_TWELVE_VOLT },

  { { 0 }, NULL, &NOVRAM->five_volt_current_kl, 0,
    "Main power supply # 4 Five Volt Current",
    PS_4_I_MON_MUX, PS_4_I_MON_CHAN, NORMAL, MAIN_FIVE_VOLT_CURRENT },

  { { 0 }, NULL, &NOVRAM->main_ps_temp_warning, 0,
    "Main power supply # 4 Temperature",
    PS_4_TEMP_MUX, PS_4_TEMP_CHAN, NORMAL, PS_TEMP }
 }
};

static cabinet_t	cabinet[NUM_CAB_PER_SYS] = {
 {
	/* cabinet 0 */
 /* reading, next, warn,		  time,   		*/
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
 /*		name				   		*/
    "Cabinet # 0 Inlet Air Temperature",
 /*     mux,			channel,      state,    type	*/
    CAB_0_INLET_AIR_MUX, CAB_0_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 0 Exhaust Air Temperature",
    CAB_0_EXHAUST_AIR_MUX, CAB_0_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 1 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 1 Inlet Air Temperature",
    CAB_1_INLET_AIR_MUX, CAB_1_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 1 Exhaust Air Temperature",
    CAB_1_EXHAUST_AIR_MUX, CAB_1_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 2 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 2 Inlet Air Temperature",
    CAB_2_INLET_AIR_MUX, CAB_2_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 2 Exhaust Air Temperature",
    CAB_2_EXHAUST_AIR_MUX, CAB_2_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 3 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 3 Inlet Air Temperature",
    CAB_3_INLET_AIR_MUX, CAB_3_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 3 Exhaust Air Temperature",
    CAB_3_EXHAUST_AIR_MUX, CAB_3_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 4 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 4 Inlet Air Temperature",
    CAB_4_INLET_AIR_MUX, CAB_4_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 4 Exhaust Air Temperature",
    CAB_4_EXHAUST_AIR_MUX, CAB_4_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 5 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 5 Inlet Air Temperature",
    CAB_5_INLET_AIR_MUX, CAB_5_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 5 Exhaust Air Temperature",
    CAB_5_EXHAUST_AIR_MUX, CAB_5_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }, {
	/* cabinet 6 */
  { { 0 }, NULL, &NOVRAM->inlet_air_warning, 0,
    "Cabinet # 6 Inlet Air Temperature",
    CAB_6_INLET_AIR_MUX, CAB_6_INLET_AIR_CHAN, NORMAL, INLET_AIR },

  { { 0 }, NULL, &NOVRAM->exhaust_air_warning, 0,
    "Cabinet # 6 Exhaust Air Temperature",
    CAB_6_EXHAUST_AIR_MUX, CAB_6_EXHAUST_AIR_CHAN, NORMAL, EXHAUST_AIR },

  { { 0 }, NULL, NULL, 0, "", 99, 99, NORMAL, EXHAUST_DELTA }
 }
};

static sensor_t		*chk_head;	/* list of scanned RWI sensors	*/
static sensor_t		*rwi_head;	/* list of all RWI sensors	*/
static sensor_t		*rwi_ptr;	/* current pointer for main A/D	*/
static sensor_t		*local_head;	/* linked list of local sensors	*/
static sensor_t		*local_ptr;	/* current pointer for local A/D*/

uint			warning_state;	/* count of sensors (etc) in trouble */
uint			shutdown_state;

static uint		rwi_num;
static uint		local_num;
uint			poll_rwi;
uint			rwi_conversion_pending;
uint 			rwi_first_pass_done;
uint 			rwi_scan_done;

/* FIX THIS, HH - local stuff should be in local.c */
uint			get_local_reading();
uint			local_conversion_pending;
uint			local_first_pass_done;

static uint		acrw;
static uint		do_fan_check;

static char		*unit_names();
static void		check_limits();


check_fan_speed()	/* model 25 (ACRW) only */
{
	static uint 	warning_timer = FAN_TIMER_OK;

	K_ASSERT(acrw);
	
	if (RWI_CIO1->portdata[0].reg & ACRW_FAN_OK) {
		if (warning_timer != FAN_TIMER_OK) {
			change_err_state(WARNING, NORMAL);
			warning_timer = FAN_TIMER_OK;
			printf("\nNotice: peripheral fan speed OK\n");
		}
		return;
	}

	if (--warning_timer % 60)
		return;

	put_char('\n');
	print_time_stamp(0);
	printf("Warning: peripheral fan speed too low: ");

	switch (warning_timer / 60) {
	case 0:	
		printf("SYSTEM BEING SHUT DOWN NOW!\n");
		change_err_state(WARNING, SHUTDOWN);
		break;
	case NO_FAN_MINUTES: 		/* first time */
		change_err_state(NORMAL, WARNING);
		/* fall through! */
	default:
		printf("system shutdown in %u minute%s\n",
		  warning_timer / 60, (warning_timer / 60 > 1 ? "s" : ""));
		break;
	}
}


rwi_init()
{
	/* clear interrupt pending and int under service on CIO 0, Port B */
	RWI_CIO0->portcs[1].reg = ICW_CPUS;

	/*
	 * CIO 0, Port A:  interrupt on any transition of any bit.
	 */
	RWI_CIO0->cioport[0].patpolar.reg = 0x00;
	RWI_CIO0->cioport[0].pattrans.reg = 0xff;
	RWI_CIO0->cioport[0].patmask.reg  = 0x00;

	initscc();
	check_rwi_rev();
}

/*
 * check_rwi_rev -- check the Real World Interface board revision number
 */

check_rwi_rev()
{
	uchar	rwi_type;

	rwi_type = RWI_CIO1->portdata[0].reg & RWI_TYPE_MASK;
	acrw = rwi_type & RWI_TYPE_ACRW;
	do_fan_check = acrw && (NOVRAM->fan_status & FAN_ALRM_ENABLE);

	if (rwi_type < RWI_REV_2B) {
		poll_rwi = 0;
		printf("RWI revision level mismatch!\n");
		printf("RWI rev level is %d, expected at least rev level %d\n",
		  rwi_type, RWI_REV_2B);
		printf("Please don't enable RWI sampling.\n");
	}
}

/*
 * check_rwi_limits -- check the RWI sensors against their limits
 */

static void
check_rwi_limits()
{
	register sensor_t	*snp;

	for (snp = chk_head; snp; snp = snp->next)
		check_limits(snp);
}

/*
 * check_local_limits -- check the local sensors against their limits
 */

check_local_limits()
{
	register sensor_t	*snp;

	for (snp = local_head; snp; snp = snp->next)
		check_limits(snp);
}

/*
 * rwi_poll -- do RWI polling
 */

rwi_poll(sec_tick)
uint	sec_tick;
{
	if (acrw) { 
		if (pm_booting_done) {
			tty_timer();
			lp_timer();
		}
		if (sec_tick && do_fan_check)
			check_fan_speed();
	} else if (poll_rwi) {
	    	if (!rwi_scan_done)
			check_rwi();
		else if (sec_tick) {
			rwi_scan_done = 0;
			check_rwi_limits();
		}
	}
}


/*
 *	get_result(data port,pointer) - get the A/D converter result
 *
 *	return values:
 *			1 - the result is stored at pointer
 *			0 - A/D conversion has timed out
 *			  - otherwise, conversion still in progress
 */

static
get_result(port, result)
register ushort	*port;
register uchar	*result;
{
	register ushort	data;
	register int	ret_val = -1;

	switch ((uint)port) {
	case RWI_AD_DATA:
		/* check if done with conversion */
		if ((data = *port) & RWI_AD_END_CONV) {
			/* store it */
			*result = data & RWI_AD_RESULT_MASK;
			ret_val = 1;
		}
		else if (int_counter > (rwi_conversion_pending + 1))
			/* got a timeout */
			ret_val = 0;
		break;
	case LOCAL_AD_DATA:
		/* check if done with conversion */
		if ((data = *port) & LOCAL_AD_END_CONV) {
			/* store it */
			*result = data & LOCAL_AD_RESULT_MASK;
			ret_val = 1;
		}
		else if (int_counter > (local_conversion_pending + 1))
			/* got a timeout */
			ret_val = 0;
		break;
	default:
		(void) spl1();
		printf("get_result(0x%x, 0x%x): unknown port!\n", port, result);
		panic("get_result confused!");
	}
	return (ret_val);
}

/*
 * start_rwi_a2d -- start the RWI A/D using rwi_ptr
 */

start_rwi_a2d()
{
	*RWI_AD_MUX = rwi_ptr->mux;
	*RWI_AD_CNTL = LOCAL_AD_START_CONV | rwi_ptr->channel;
	rwi_conversion_pending = int_counter;
}

/*
 * start_rwi_conversion -- set up the next RWI A/D conversion, or flag the end
 *	of a series of conversions in rwi_first_pass_done and rwi_scan_done
 */

static void
start_rwi_conversion()
{
	if (rwi_ptr == NULL) {
		if ((rwi_ptr = rwi_head) == NULL)
			return;			/* no RWI at all!?!	*/
	}
	else if ((rwi_ptr = rwi_ptr->next) == NULL) {	/* last sensor?	*/
		/*
		 * all sensors read -- go on to the next reading, or stop
		 */
		if (++rwi_num >= READ_NUM_MAX) {
			rwi_num = 0;
			rwi_first_pass_done = 1;
			rwi_scan_done = 1;
			return;			/* finished with all readings */
		}
		rwi_ptr = rwi_head;
	}

	start_rwi_a2d();			/* do next conversion	*/
}

/*
 * check_rwi -- check if a RWI A/D conversion is in progress, or finished
 */

check_rwi()
{
	uchar	result;

	if (rwi_conversion_pending) {
		switch (get_result(RWI_AD_DATA, &result)) {
		case 1:	/* channel conversion is done */
			rwi_conversion_pending = 0;
			/* store the result */
			rwi_ptr->reading[rwi_num] = result;
			break;
		case 0: /* conversion has timed out */
			printf("RWI conversion timeout.\n");
			poll_rwi = 0;		/* disable sampling */
			return;
		default:/* conversion still in progress */
			return;
		}
	}

	start_rwi_conversion();		/* else, set up another conversion */
}


/*
 * start_local_a2d -- start the local A/D using local_ptr
 */

static void
start_local_a2d()
{
	*LOCAL_AD_CNTL = LOCAL_AD_START_CONV | local_ptr->channel;
	local_conversion_pending = int_counter;
}

/*
 * start_local_conversion -- set up the next local A/D conversion, or flag
 *	the end of a series of conversions
 */

static void
start_local_conversion()
{
	if (local_ptr == NULL) {
		if ((local_ptr = local_head) == NULL)
			return;			/* no local at all!?!	*/
	}
	else if ((local_ptr = local_ptr->next) == NULL) {	/* next sensor*/
		/* go on to the next reading */
		if (++local_num >= READ_NUM_MAX) {
			local_num = 0;
			local_first_pass_done = 1;
			local_scan_done = 1;
			return;			/* finished with all readings */
		}
		local_ptr = local_head;
	}

	start_local_a2d();
}

/*
 * check_local -- check if a local A/D conversion is in progress
 */

check_local()
{
	uchar	result;

	/* check if a conversion is in progress */
	if (local_conversion_pending) {
		switch (get_result(LOCAL_AD_DATA, &result)) {
		case 1:	/* channel conversion is done */
			local_conversion_pending = 0;
			/* store the result */
			local_ptr->reading[local_num] = result;
			break;
		case 0: /* conversion has timed out */
			printf("local conversion timeout\n");
			poll_local = 0;		/* disable sampling */
			return;
		default:/* conversion still in progress */
			return;
		}
	}

	start_local_conversion();	/* else, set up another conversion */
}

/*
 * frac_str -- turns a reading into a string formatted:  [-]NN.N
 *		Note:  uses several static buffers for the return value,
 *		so up to NUM_FRAC_STRS frac_str calls can be in the arguments
 *		of a printf.
 */

#define NUM_FRAC_STRS	4	/* must be a power of 2 */

char *
frac_str(type, reading)
uint	type, reading;
{
	frac_t		fnum;
	char		*str;

	static char	strs[NUM_FRAC_STRS][16];
	static uint	idx;

	convert_reading(type, reading, &fnum);

	if (fnum.hundredths >= 5 && ++fnum.tenths > 9) {
		fnum.tenths = 0;
		if (fnum.number >= 0)
			++fnum.number;
		else
			--fnum.number;
	}

	str = strs[idx];
	++idx;
	idx &= NUM_FRAC_STRS - 1;
	sprintf(str, "%3d.%d", fnum.number, fnum.tenths);
	return (str);
}

/*
 * unit_names -- return names for each of the channels in the pointers
 */

static char *
unit_names(type)
uint	type;
{
	char	*unit;

	switch (type) {
	case MAIN_FIVE_VOLT:
	case MAIN_TWELVE_VOLT:
	case MAIN_NEG_TWELVE_VOLT:
	case LOCAL_FIVE_VOLT:
	case AUX_FIVE_VOLT:
	case LOCAL_TWELVE_VOLT:
	case AUX_TWELVE_VOLT:
	case LOCAL_NEG_TWELVE_VOLT:
	case AUX_NEG_TWELVE_VOLT:
		unit = "V";
		break;
	case PS_TEMP:
	case INLET_AIR:
	case EXHAUST_AIR:
	case CAGE_AIR:
		unit = "C";
		break;
	case MAIN_FIVE_VOLT_CURRENT:
		unit = "A";
		break;
	default:
	case LOCAL_VREF:
		printf("unit_names: unknown type = 0x%x\n", type);
		return ("");
	}

	return (unit);
}

/*
 * lim_reading_msg -- turn the reading into a voltage message
 */

static void
lim_reading_msg(snp, reading, msg)
sensor_t	*snp;
uint		reading;
char		*msg;
{
	uint	spl_level;
	char	*units, *frac;

	units = unit_names(snp->type);
	frac = frac_str(snp->type, reading);
	spl_level = spl1();
	put_char('\n');
	print_time_stamp(0);
	printf("%s: %s = %s %s\n", msg, snp->name, frac, units);
	splx(spl_level);
}

/*
 * lim_err_msg -- print a voltage error message, including the limits
 */

static void
lim_err_msg(snp, reading, msg)
register sensor_t	*snp;
uint			reading;
char			*msg;
{
	register char	*up;
	register uint	spl_level;
	char		*shut, *warn, *units;

	units = unit_names(snp->type);
	up = units;
	shut = frac_str(snp->type, (snp->warn + 1)->low);
	warn = frac_str(snp->type, snp->warn->low);
	spl_level = spl1();
	lim_reading_msg(snp, reading, msg);
	printf("Low limits:  warning at %s %s, shutdown at %s %s\n",
	  warn, up, shut, up);
	shut = frac_str(snp->type, (snp->warn + 1)->high);
	warn = frac_str(snp->type, snp->warn->high);
	printf("High limits: warning at %s %s, shutdown at %s %s\n",
	  warn, up, shut, up);
	splx(spl_level);
}

/*
 * check_limits -- average and check real-world limits
 */

static void
check_limits(snp)
register sensor_t	*snp;
{
	register uchar		*rdp;
	register limit_t	*limp;
	register int		i;
	register uint		avg_reading, neg;
	uint			over_limit;

	neg = (snp->warn == &NOVRAM->neg_twelve_volt_warning);

	switch (snp->state) {
	case NORMAL:
		/*
		 * Check the readings; continue unless all of them
		 * are over the limit.  If so, average the readings,
		 * change state, and print an error message.
		 */
		limp = snp->warn;
		rdp = snp->reading;
		avg_reading = 0;
		for (i = READ_NUM_MAX; --i >= 0; rdp++) {
			if (neg) {
				if (!OVER_N_LIMIT(*rdp, limp))
					break;		/* in bounds */
			}
			else {
				if (!OVER_LIMIT(*rdp, limp))
					break;		/* in bounds */
			}
			avg_reading += *rdp;
		}
		if (i >= 0)
			break;			/* in bounds */
		avg_reading /= READ_NUM_MAX;

		snp->state = WARNING;
		snp->time = n_seconds;
		change_err_state(NORMAL, WARNING);
		lim_err_msg(snp, avg_reading, "Warning");
		break;

	case WARNING:
		/*
		 * Check the readings; if all of them are over the
		 * shutdown limit, then print a message and shutdown.
		 */
		limp = snp->warn + 1;
		rdp = snp->reading;
		avg_reading = 0;
		for (i = READ_NUM_MAX; --i >= 0; rdp++) {
			if (neg) {
				if (!OVER_N_LIMIT(*rdp, limp))
					break;		/* in bounds */
			}
			else {
				if (!OVER_LIMIT(*rdp, limp))
					break;		/* in bounds */
			}
			avg_reading += *rdp;
		}
		if (i < 0) {
			avg_reading /= READ_NUM_MAX;

			snp->state = SHUTDOWN;
			change_err_state(WARNING, SHUTDOWN);
			lim_err_msg(snp, avg_reading, "Shutdown");
			break;
		}
		/*
		 * else, check to see if all are under the warning limit
		 */
		rdp = snp->reading;
		avg_reading = 0;
		over_limit = 0;
		limp = snp->warn;
		for (i = READ_NUM_MAX; --i >= 0; rdp++) {
			if (neg) {
				if (OVER_N_LIMIT(*rdp, limp))
					over_limit = 1;
			}
			else {
				if (OVER_LIMIT(*rdp, limp))
					over_limit = 1;
			}
			avg_reading += *rdp;
		}
		if (!over_limit) {
			snp->state = NORMAL;
			change_err_state(WARNING, NORMAL);
		}
		else if (n_seconds - snp->time > WARNING_INTERVAL) {
			snp->time = n_seconds;
			avg_reading /= READ_NUM_MAX;
			lim_reading_msg(snp, avg_reading, "Warning");
		}
		break;
	case DECONFIGURED:
	default:
		break;
	}
}


/*
 * ex_delta_msg -- print a over-limit message
 */

static void
ex_delta_msg(delta, cab, limits)
uint	delta, cab, limits;
{
	uint	spl_level;
	char	*frac;

	frac = frac_str(EXHAUST_AIR, delta);
	spl_level = spl1();
	put_char('\n');
	print_time_stamp(0);
	printf("Warning: cabinet %d exhaust to inlet air temp = %s C\n",
	  cab, frac);
	splx(spl_level);

	if (limits) {
		frac = frac_str(EXHAUST_AIR, NOVRAM->cage_air_delta);
		printf("Allowed exhaust/inlet air temp rise is %s C\n", frac);
	}
	printf("Check for clogged filters or fan failure!\n");
}

/*
 * check_exhaust_delta -- check the air temperature difference for all cabinets
 */

check_exhaust_delta()
{
	register uchar		*erp, *irp;
	register sensor_t	*dsp;
	register cabinet_t	*cbp;
	register int		i;
	register int		delta;
	register uint		delta_avg;
	register uint		num;
	register uint		over_limit;

	/* check exhaust air temperature */

	for (num = 0, cbp = cabinet; num < NUM_CAB_PER_SYS; num++, cbp++) {
		if (!NOVRAM->cab_present[num].present)
			continue;

		dsp = &cbp->c_exhaust_delta;
		if (dsp->state == DECONFIGURED)
			continue;

		erp = cbp->c_exhaust_air.reading;
		irp = cbp->c_inlet_air.reading;
		switch (dsp->state) {
		case NORMAL:
			/*
			 * Check the readings; continue unless all of them
			 * are over the limit.  If so, average the readings,
			 * change state, and print an error message.
			 */
			delta_avg = 0;
		    	for (i = READ_NUM_MAX; --i >= 0; ) {
				delta = *erp++ - *irp++;
				if (delta < NOVRAM->cage_air_delta)
					break;		/* temp ok */
				delta_avg += delta;
		    	}
			if (i >= 0)
				continue;		/* temp ok */

			delta_avg /= READ_NUM_MAX;
			dsp->state = WARNING;
			dsp->time = n_seconds;
			change_err_state(NORMAL, WARNING);
			ex_delta_msg(delta_avg, num, 1);
			break;

		case WARNING:
			/*
			 * Check the readings; if all of them are under the
			 * limit, then change state.
			 */
			over_limit = 0;
			delta_avg = 0;
		    	for (i = READ_NUM_MAX; --i >= 0; ) {
				delta = *erp++ - *irp++;
				if (delta > NOVRAM->cage_air_delta)
					over_limit = 1;
				delta_avg += delta;
		    	}
			if (!over_limit) {
				dsp->state = NORMAL;
				change_err_state(WARNING, NORMAL);
			}
			else if (n_seconds - dsp->time > WARNING_INTERVAL) {
				delta_avg /= READ_NUM_MAX;
				dsp->time = n_seconds;
				ex_delta_msg(delta_avg, num, 0);
			}
			break;

		case DECONFIGURED:
		default:
			break;
		}
	}
}

/*
 * change_err_state -- send messages to the kernel based on how many sensors
 *	are over their limits, or other related problems
 */

change_err_state(from, to)
uint	from, to;
{
	if (from == to)
		return;		/* no op */

	switch (from) {
	case NORMAL:
		break;		/* do nothing */

	case WARNING:
		if (warning_state)
			--warning_state;
		break;

	case SHUTDOWN:
		if (shutdown_state)
			--shutdown_state;
		break;

	default:
		printf("change_err_state: unknown from state = %x\n", from);
		return;
	}

	switch (to) {
	case NORMAL:
		if (!warning_state && !shutdown_state) {
			if (pm_booting_done)
				spm2kern_cmd(S2K_PT_NORMAL_FLUSH);
			warning_light_off();
		}
		break;

	case WARNING:
		if (warning_state++ == 0) {
			if (pm_booting_done)
				spm2kern_cmd(S2K_PT_FAST_FLUSH);
			warning_light_off();
		}
		break;

	case SHUTDOWN:
		if (shutdown_state++ == 0) {
			if (pm_booting_done)
				spm2kern_cmd(S2K_PT_SHUTDOWN);
			else
				shutdown(MD_POWEROFF);
			warning_light_off();
		}
		break;

	default:
		printf("change_err_state: unknown to state = %x\n", to);
		return;
	}
}

/*
 * rem_sensors -- remove any sensors that may be out of limits
 */

rem_sensors(snp)
register sensor_t	*snp;
{

	for ( ; snp; snp = snp->next)
		if (snp->state == WARNING || snp->state == SHUTDOWN) {
			change_err_state(snp->state, NORMAL);
			snp->state = NORMAL;
		}
}

/*
 * rem_local_sensors -- remove any local sensors that may be out of limits
 */

rem_local_sensors()
{
	rem_sensors(local_head);
}

/*
 * rem_rwi_sensors -- remove any RWI sensors that may be out of limits
 */

rem_rwi_sensors()
{
	rem_sensors(rwi_head);
}

/*
 * do_uadmin -- do the SPM side of uadmin(2)
 */

do_uadmin(fcn, mdep)
uint	fcn, mdep;
{
	uint	spl_level;

	switch (fcn) {
	case AD_HALT:
	case AD_BOOT:
	case AD_IBOOT:
		shutdown(mdep);
		break;
	default:
		spl_level = spl1();
		printf("do_uadmin:  unknown fcn = 0x%x, mdep = 0x%x\n",
		  fcn, mdep);
		splx(spl_level);
	}
}

/*
 * shutdown -- kill the system
 */

shutdown(mdep)
register uint	mdep;
{
	register uint	sbus_slot;
	uint		saved_map = iomap_save();	/* saved map entry */

	NOVRAM->shutdown = (mdep & MD_CLEAN) ? 1 : 0;
	runtime_crc();

	if (pm_booting_done) {
		cpu_all_stop();

		/*
		 * reset all IOMs
		 */
		for (sbus_slot = 0; sbus_slot != SBUS_NUM_SLOT; sbus_slot++) {
			switch (sbus_config.slot_id[sbus_slot]) {
			case SBUS_IOM:
			case SBUS_NIOM:
				*(uint *)iomap(sbus_slot, IOM_RESET) = 0;
			default:
				break;
			}
		}

		/*
		 * turn off dispatcher
		 */
		*WRCNTL1 &= ~WR1_DREQ;
	}

	iomap_restore(saved_map);		/* restore temp map */

	switch (mdep = MD_GET_ACTION(mdep)) {
	case MD_POWEROFF:		/* turn off power */
		power_off();
		break;

	case MD_REBOOT:			/* set autoboot flag and reboot	*/
	case MD_PROM:			/* clear autoboot flag & reset	*/
		NOVRAM->autoboot = (mdep == MD_REBOOT) ? 1 : 0;
		/* fall through! */
	case MD_RESET:			/* reset, use existing autoboot	*/
	default:
		NOVRAM->rshutdown = 0;
		shared_crc();
		warning_light_on();
		ON_LIGHT_OFF;
		prom();
		break;
	}
}

/*
 * power_off -- kill system power, and if the shutdown is due to a power
 *		failure, turn off the UPS as well
 */

power_off()
{
	NOVRAM->rshutdown = 1;
	shared_crc();

	splhi();

	warning_light_on();
	ON_LIGHT_OFF;			/* turn off the "system on" light */

	*RWI_CIO0_PORTB &= ~ALL_CAB_OFF;

	if (*RWI_CIO0_PORTA & UPS_AC_FAIL)
		*RWI_CIO0_PORTB &= ~UPS_OFF;	/* turn off UPS, too */

	for (;;)
		;			/* spin */
}

/*
 * get_cabinet_reading -- return averaged data from a sensor in a cabinet
 */

uint
get_cabinet_reading(cab, sens_num)
uint	cab, sens_num;
{
	register uint	reading_total;
	register int	i;
	register uchar	*rdp;

	rdp = cabinet[cab].cab[sens_num].reading;
	reading_total = 0;
	i = READ_NUM_MAX;
	while (--i >= 0)
		reading_total += *rdp++;

	return (reading_total / READ_NUM_MAX);
}

/*
 * get_ps_reading -- return averaged data from a sensor in a power supply
 */

uint
get_ps_reading(ps, sens_num)
uint	ps, sens_num;
{
	register uint	reading_total;
	register int	i;
	register uchar	*rdp;

	rdp = main_ps[ps].pow[sens_num].reading;
	reading_total = 0;
	i = READ_NUM_MAX;
	while (--i >= 0)
		reading_total += *rdp++;

	return (reading_total / READ_NUM_MAX);
}

/*
 * get_ps_amps_type -- return the type of a power supply's current sensor
 */

uint
get_ps_amps_type(ps_num)
uint	ps_num;
{
	return (main_ps[ps_num].p_five_volt_current.type);
}

/*
 * get_local_reading -- return averaged data from a local sensor
 */

uint
get_local_reading(sens_num)
uint	sens_num;
{
	register uint	reading_total;
	register int	i;
	register uchar	*rdp;

	rdp = local_ps.loc_ps[sens_num].reading;
	reading_total = 0;
	i = READ_NUM_MAX;
	while (--i >= 0)
		reading_total += *rdp++;

	return (reading_total / READ_NUM_MAX);
}

/*
 * init_local -- setup local A/D data
 */

init_local()
{
	local_head = &local_ps.l_five_volt;
	start_local_conversion();	/* do one to init the A/D */
}

/*
 * init_rwi -- setup RWI A/D data for main power supplies and cabinets
 */

init_rwi()
{
	register main_ps_t	*mp;
	register cabinet_t	*cp;
	register sensor_t	*last_sp;
	register uint		i;
	sensor_t		start;

	last_sp = &start;

	for (i = 0, mp = main_ps; i < NUM_PS_PER_SYS; i++, mp++) {
		if (NOVRAM->main_pwr[i].five_volt) {
			last_sp = (last_sp->next = &mp->p_five_volt);
			mp->p_five_volt.state = NORMAL;
		}
		else
			mp->p_five_volt.state = DECONFIGURED;

		if (NOVRAM->main_pwr[i].twelve_volt) {
			last_sp = (last_sp->next = &mp->p_twelve_volt);
			mp->p_twelve_volt.state = NORMAL;
		}
		else
			mp->p_twelve_volt.state = DECONFIGURED;

		mp->p_neg_twelve_volt.type = MAIN_NEG_TWELVE_VOLT;
		if (NOVRAM->main_pwr[i].neg_twelve_volt) {
			last_sp = (last_sp->next = &mp->p_neg_twelve_volt);
			mp->p_neg_twelve_volt.state = NORMAL;
		}
		else
			mp->p_neg_twelve_volt.state = DECONFIGURED;

		if (NOVRAM->main_pwr[i].five_volt_current) {
			mp->p_five_volt_current.next = rwi_head;
			rwi_head = &mp->p_five_volt_current;
			/*
			 * use more descriptive types for 5V current
			 */
			switch (NOVRAM->main_pwr[i].ps_id) {
			case HC_POWER:
				mp->p_five_volt_current.type = MAIN_5V_HC_AMPS;
				break;
			case SWITCHING_POWER:
				mp->p_five_volt_current.type =
				  MAIN_5V_SWITCH_AMPS;
				break;
			default:
				printf("init_rwi: new current sensor type=%x\n",
				  NOVRAM->main_pwr[i].ps_id);
			}
			mp->p_five_volt_current.state = NORMAL;
		}
		else
			mp->p_five_volt_current.state = DECONFIGURED;

		if (NOVRAM->main_pwr[i].temp) {
			last_sp = (last_sp->next = &mp->p_ps_temp);
			mp->p_ps_temp.state = NORMAL;
		}
		else
			mp->p_ps_temp.state = DECONFIGURED;
	}

	for (i = 0, cp = cabinet; i < NUM_CAB_PER_SYS; i++, cp++) {
		if (NOVRAM->cab_present[i].inlet_air) {
			last_sp = (last_sp->next = &cp->c_inlet_air);
			cp->c_inlet_air.state = NORMAL;
		}
		else
			cp->c_inlet_air.state = DECONFIGURED;

		if (NOVRAM->cab_present[i].exhaust_air) {
			last_sp = (last_sp->next = &cp->c_exhaust_air);
			cp->c_exhaust_air.state = NORMAL;
		}
		else
			cp->c_exhaust_air.state = DECONFIGURED;

 		cp->c_exhaust_delta.state =
		  ((cp->c_exhaust_air.state == NORMAL) &&
		   (cp->c_inlet_air.state == NORMAL)) ? NORMAL : DECONFIGURED;
	}
	last_sp->next = NULL;
	chk_head = start.next;
	rwi_ptr = NULL;

	/*
	 * find end of rwi list and add checked sensors
	 */
	if (last_sp = rwi_head) {
		while (last_sp->next)
			last_sp = last_sp->next;

		last_sp->next = chk_head;
	}
	else
		rwi_head = chk_head;

	if (!acrw)
		start_rwi_conversion();		/* do one to init the A/D */
}

struct scc *
port_address(port_id)
uint port_id;
{
	switch (port_id) {
	case MODEM :
		return(MODEM_PORT);
	case CONSOLE :
		return(CONSOLE_PORT);
	case PRINTER :
		if (acrw)
			return(0);
		return(PRINTER_PORT);
	case UPS :
		if (acrw)
			return(0);
		return(UPS_PORT);
	}
	return(0);
}

static uint	warning_light_state;

warning_light_on()
{
	warning_light_state = 1;
	WARNING_LIGHT_ON;
}

warning_light_off()
{
	warning_light_state = 0;
	WARNING_LIGHT_OFF;
}

toggle_warning_light()
{
	if (warning_light_state ^= 1)
		WARNING_LIGHT_ON;
	else
		WARNING_LIGHT_OFF;
}
