/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) timewarp.c: version 25.1 created on 11/27/91 at 15:40:10	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)timewarp.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/*
 * timewarp.c -- routines to adjust the clock interrupt rate to synchronize
 *		kernel time to the clock chip time
 */

#include "sys/types.h"
#include "sys/spm_mem.h"
#include "spm.h"
#include "spm_debug.h"

/* #define WARP_DEBUG					/* debug mode */

#define ABS(x)		((x) < 0 ? -(x) : (x))

#define MINUTE		60
#define CORR_TIME	(5 * MINUTE)
#define STOP_DELAY	3				/* stay stopped */
#define TIME_DELAY	(5 * MINUTE)			/* timebase period */
#define LOCK_TIME	4				/* time to stay locked*/
#define LOCK_ERRORS	10				/* # of errors */
#define MAX_TIMER_VAL	(6 * LOC_CIO_TIMER0_TC / 5)	/* 120% of normal */
#define MIN_TIMER_VAL	(4 * LOC_CIO_TIMER0_TC / 5)	/*  80% of normal */

typedef enum warp_st {
	WS_Stopped = 0, WS_Timebase, WS_Correcting, WS_Locked
} WARP_ST;

static WARP_ST	warp_state;
static long	warp_delay;
static long	lock_error;
static time_t	last_real;
static time_t	last_kern;
static time_t	last_diff;
static ulong	last_timer = LOC_CIO_TIMER0_TC;
static ulong	best_timer = LOC_CIO_TIMER0_TC;

extern uint	tdb_check_state;
extern uint	kern_clock_enable;

extern time_t	rtc_time();

/*
 * timewarp -- update Spm_Mem->spm_time, and syncronize kernel time to SPM time
 */

timewarp()
{
	register time_t	real, diff, elapsed, val;

	Spm_Mem->spm_time = real = rtc_time();

	if (last_real == 0 || !tdb_check_state || !kern_clock_enable) {
		warp_state = WS_Stopped;
		warp_delay = STOP_DELAY;
		last_real = real;
#ifdef WARP_DEBUG
		printf("S:\n");
#endif
		return;				/* system stopped */
	}

	switch (warp_state) {
	case WS_Stopped:
		if (--warp_delay >= 0)
			return;			/* wait until stabilized */

		warp_state = WS_Timebase;	/* check crystal freq */
		warp_delay = TIME_DELAY;
		last_real = real;
		last_kern = Spm_Mem->time;
#ifdef WARP_DEBUG
		printf("T:\n");
#endif
		return;

	case WS_Timebase:
		if (--warp_delay >= 0)
			return;			/* still waiting */

		/*
		 * calculate new timer value that syncs the clocks
		 */

		elapsed = real - last_real;
		diff = Spm_Mem->time - last_kern;
		if (elapsed <= 0)
			elapsed = 1;
		val = (diff * last_timer) / elapsed;
#ifdef WARP_DEBUG
		printf("Best(diff=%6d, elapsed=%6d, old=%d, corr=%d, ",
		  diff, elapsed, best_timer, val);
#endif
		if (val > MAX_TIMER_VAL)
			val = MAX_TIMER_VAL;
		else if (val < MIN_TIMER_VAL)
			val = MIN_TIMER_VAL;
#ifdef WARP_DEBUG
		printf("new=%d)\n", val);
#endif
		best_timer = val;

		break;				/* calc best_timer, etc. */

	case WS_Correcting:
		if ((diff = Spm_Mem->time - real) == 0)
			break;
		if (--warp_delay >= 0 && last_diff == diff)
			return;			/* wait for change to work */

		if (ABS(diff) < ABS(last_diff)) {
			last_diff = diff;
			return;			/* still reducing gap */
		}
		break;				/* do further correction */

	case WS_Locked:
		if ((diff = Spm_Mem->time - real) == 0)
			return;			/* still locked */

		best_timer += diff;

		if (real - last_real < LOCK_TIME) {
			if (++lock_error > LOCK_ERRORS) {
				lock_error = 0;
				warp_state = WS_Stopped;
				return;		/* stop and try again */
			}
		}
		else if (lock_error)
			--lock_error;
		break;				/* do a correction */

	default:
		printf("timewarp: unknown warp_state = %d!\n", warp_state);
		warp_state = WS_Stopped;
		warp_delay = 0;
		return;
	}

	elapsed = real - last_real;
	last_kern = Spm_Mem->time;
	last_real = real;

	/*
	 * if diff is zero, go to locked state
	 */

	if ((last_diff = diff = last_kern - real) == 0) {
		warp_state = WS_Locked;		/* maintain lock */
		set_local_timer0(last_timer = best_timer);
#ifdef WARP_DEBUG
		printf("L: %d\n", best_timer);
#endif
		return;
	}

	/*
	 * else, do a correction
	 */

	warp_state = WS_Correcting;	/* slew until real == kern */
	warp_delay = CORR_TIME;

	val = (diff * LOC_CIO_TIMER0_TC) / CORR_TIME;
#ifdef WARP_DEBUG
	printf("C: diff=%3d, elapsed=%6d, corr=%6d,", diff, elapsed, val);
#endif
	val += last_timer;
	if (val > MAX_TIMER_VAL)
		val = MAX_TIMER_VAL;
	else if (val < MIN_TIMER_VAL)
		val = MIN_TIMER_VAL;

	set_local_timer0(last_timer = val);
#ifdef WARP_DEBUG
	printf(" newN=%5d\n", val);
#endif
}


/*
 * set_time -- set the clock chip, etc.
 */

set_time(time_val)
time_t	time_val;
{
	rtc_stime(time_val);
	last_real = last_kern = Spm_Mem->spm_time = time_val;
	last_diff = 0;
	warp_state = WS_Stopped;
}
