/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) parse.c: version 25.1 created on 11/27/91 at 15:39:47	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)parse.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/
 /*
 * parse.c
 *
 *	parse (c_ptr) - parse command string
 *	char *c_ptr
 *
 *	This is the main command line interpreter.  It checks for special
 *	command functions, then searches the current menu for a match for
 *	a match.  If no match is found, an error message is printed.
 *
 *	On a command match with the current menu, one of two courses of
 *	action is taken, based upon the command type.  A 'c' signifies that
 *	this is a specific command, to be executed.  A 'm' means to traverse
 *	to this sub-menu.
 */

#include "global.h"
#include "menu.h"
#include "spm_debug.h"

#define ARGBUF_SIZE	256
#define argbufend	&argbuf[sizeof(argbuf) - 1]

char 		*comm_args[MAXARGS];
static char	argbuf[ARGBUF_SIZE];
static char	tmp_size;

struct mstack	menu_stack[MENUSTACKSIZE];
int		menu_idx;

extern uint	pm_booting_done;
extern char	*gets();

/*
 * check_menu -- returns the first matching entry in the given menu, or NULL
 */

static struct menu *
check_menu(mp)
register struct menu	*mp;
{
	for (; mp->comm_text; mp++) {
		if (rubber_room && (mp->flags & MF_EXPERT))
			continue;
		if (mp->flags & (pm_booting_done ? MF_PREBOOT : MF_POSTBOOT))
			continue;
		if (!comm_cmp(comm_args[0], mp->comm_text))
			return (mp);

	}
	return (NULL);
}

/*
 * search_menu -- recursively search a menu tree for comm_arg[0]
 */

static struct menu *
search_menu(mp)
struct menu	*mp;
{
	struct menu	*rmp;

	if (rmp = check_menu(mp))
		return (rmp);

	for (; mp->comm_text; mp++) {
		if (rubber_room && (mp->flags & MF_EXPERT))
			continue;
		if (mp->flags & (pm_booting_done ? MF_PREBOOT : MF_POSTBOOT))
			continue;
		if ((mp->flags & MF_MENU) &&
		  (rmp = search_menu(mp->command.menu_p)))
			return (rmp);
	}
	return (NULL);
}

/*
 * parse -- parse the string and execute a command, returns non-zero on error
 */

parse(str)
char	*str;
{
	register int		n_arg;
	register struct menu	*mp;
	int			local_item;

	if ((n_arg = argsplit(str)) < 0)
		return (0);				/* no command */

	tmp_size = mref_siz;

	/*
	 * check the global commands, then the current menu, then all menus
	 */
	if (mp = check_menu(global_comm))
		local_item = 0;
	else if (mp = check_menu(menu_stack[menu_idx].m_ptr))
		local_item = 1;
	else {
		mp = search_menu(runtime_boot_level);
		local_item = 0;
	}

	/* check for a privilege level command */
	if (mp == NULL || (rubber_room && (mp->flags & MF_EXPERT))) {
		ctrlprint(comm_args[0]);
		printf(": not found\n");
		return (-1);
	}

	/* Check for correct number of arguments */
	if (n_arg < mp->lo_limit || n_arg > mp->hi_limit) {
		printf("Illegal number of arguments: ");
		if (mp->hi_limit == 0)
			printf("none allowed.\n");
		else {
			printf("was %d, should be ", n_arg);
			if (mp->lo_limit != mp->hi_limit)
				printf("between %d and ", mp->lo_limit);
			printf("%d.\n", mp->hi_limit);
		}
		return (-1);
	}

	/* Update memory reference size */
	mref_siz = tmp_size;  /* save this, if filled in. */
	/* Check for menu */
	if (mp->flags & MF_MENU) {
		/* Traverse to new menu */
		menu_idx++;
		menu_stack[menu_idx].m_ptr = mp->command.menu_p;
		menu_stack[menu_idx].name =
		  (mp->comm_desc[0] == '-') ?  mp->comm_desc+1 : mp->comm_desc;
	}
	else {
		/* Execute command */
		if (!local_item)
			printf("%s\n", mp->comm_text);
		(*mp->command.procedure)(n_arg);
		first_spm_cmd = 0;
	}
	return (0);
}

/*
 * Comm_cmp
 *
 *	This puppy compares the comm_in with comm_ref, returning a 0 on
 *	equality, and -1 on inequality.
 *
 *	Comm_in is the string to be checked.
 *
 *	Comm_ref is the reference string, and follows the following rules:
 *		'(' begins the optional part of the command identifier string.
 *			The following characters need not be included in the
 *			input string.
 *		')' ends the optional part of the command identifier.
 *		'<' begins the optional postfix for the command.
 *		'<' ends the optional postfix for the command.
 *		'[' begins a list of legal postfix characters.  Only one
 *			character per brace set is allowed.
 *		']' ends a list of legal postfix characters.
 *
 *		Example:  for the compare string "mm(emory)<.[bwl]>", legal
 *		input strings are "mm", "mm.b", "mmemory" and so forth.
 *		An illegal input would be "mmem.b" or "mm.bw".
 *		Note that when the postfix is used with this command, the
 *		'.' is mandatory.
 *
 *		Currently, only one of each type of field is supported.
 */

comm_cmp(comm_in, comm_ref)
register char	*comm_in, *comm_ref;
{
	register int	flag, option;

	option = 0;

	/* First keyletters */
	while (*comm_in && *comm_ref && (*comm_in == *comm_ref)) {
		comm_in++;
		comm_ref++;
	}
	if (!*comm_in && !*comm_ref)
		return (0);

	/* Optional keyletters begin with open paren */
	if (*comm_ref == '(') {
		comm_ref++;
		while (*comm_in && *comm_ref && (*comm_in == *comm_ref)) {
			comm_in++;
			comm_ref++;
		}
		if (!*comm_in && !*comm_ref)
			return (0);
		/* Space past the close paren */
		while (*comm_ref && (*comm_ref++ != ')'))
			;
	}

	/* Postfix =  String followed by optional character */
	if (*comm_ref == '<') {
		comm_ref++;
		option++;
		if (!*comm_in)
			return (0);
	}
	/* Check string */
	while (*comm_in && *comm_ref && (*comm_in == *comm_ref)) {
		comm_in++;
		comm_ref++;
	}
	if (*comm_ref == '>')
		comm_ref++;
	while (*comm_ref == ' ')
		comm_ref++;
	/* Check for equality */
	if (!*comm_in && !*comm_ref)
		return (0);
	if (!*comm_in && !option)
		return (-1);
					/* Compare postfix character table */
	if ((*comm_ref == '[') && (*comm_in || !option)) {
		flag = 0;
		comm_ref++;
		while (*comm_ref != ']') {
			if (*comm_in == *comm_ref) {
				flag++;
				tmp_size = *comm_in;	/* save size */
			}
			comm_ref++;
		}
		comm_ref++;
		if (!flag)
			return (-1);
		comm_in++;
	}
	/* Check for extraneous stuff */
	if (*comm_in)
			return (-1);
	return (0);
}

/*
 * Argsplit takes an input string and divides it into separate arguments,
 *	stored in the pointer array comm_args.  The number of arguments - 1 is
 *	returned.
 *
 *	Arguments are divided by white space.
 *
 *	If there are more than MAXARGS arguments, MAXARGS - 1 is returned.
 */

argsplit(str)
register char	*str;
{
	register char	*p;
	register int	c, wordcnt;

	p = argbuf;
	for (wordcnt = 0; wordcnt < MAXARGS && p < argbufend; wordcnt++) {
		while (isspace(*str))		/* eat white space */
			str++;

		if (*str == '\0')
			break;			/* end of string */

		comm_args[wordcnt] = p;
						/* Copy the argument */
		while ((c = *str) && !isspace(c) && p < argbufend) {
			*p++ = c;
			++str;
		}
		*p++ = '\0';			/* Terminate arg string */
	}
	return (wordcnt - 1);
}

/*
 * ctrlprint -- print a string, expanding control characters to '^X' or '\???'
 */

ctrlprint(str)
register char	*str;
{
	register int	c;

	while (c = *str++) {
		if (isprint(c) || c == ' ' || c == '\t')
			put_char(c);
		else if (c < ' ')
			printf("^%c", c + '@');
		else
			printf("\\%03o", c);
	}
}


/*
 *	print_menu (ptr) - Print current local menu
 *
 *	struct menu {
 *		char *comm_text;
 *		char *comm_desc;
 *		unsigned char flags;
 *		unsigned char lo_limit;
 *		unsigned char hi_limit;
 *		union {
 *			PFI procedure;
 *			struct menu *menu_p;
 *		} command;
 *		char *help;
 *	};
 *
 *	Menu.comm_text is the full text for the command name, such as 'l(ist)'.
 *	Menu.comm_desc is the description for the command.
 *	Menu.flags is the type (command or menu), plus display flags.
 *	Menu.lo_limit is the low limit of the number of arguments allowed for
 *		the command.
 *	Menu.hi_limit is the high limit of the number of arguments allowed for
 *		the command.
 *	Menu.command is a union containing either a pointer to a procedure or
 *		a pointer to another menu.
 *	Menu.command.procedure points to the related procedure.
 *	Menu.command.menu_p points to the related sub-menu.
 *	Menu.help is a text description for this command.
 *	Menus are terminated with a NULL value in the menu.comm_text field.
 */

print_menu()
{
	struct menu 	*mp;
	register char	*cp;
	register uint	col;

	for (mp = menu_stack[menu_idx].m_ptr; mp->comm_text; mp++) {
		if (rubber_room && (mp->flags & MF_EXPERT))
			continue;
		if (mp->flags & (pm_booting_done ? MF_PREBOOT : MF_POSTBOOT))
			continue;

		col = 0;

		for ( cp = mp->comm_text; *cp; cp++, col++ )
			put_char(*cp);

		put_char(' ');

		if ((cp = mp->comm_desc) == NULL) {
			put_char('\n');
			continue;
		}
		for ( ; *cp && *cp != '-'; cp++, col++ )
			put_char(*cp);

		if (mp->flags & MF_MENU) {
			while (++col <= 12)
				put_char(' ');
			if (*cp == '-')
				++cp;
		}
		else {
			while (++col <= 38)
				put_char(' ');
		}

		printf("%s\n", cp);
	}
}

print_help(arg_cnt)
int	arg_cnt;
{
	struct menu	*mp;
	char		combuf[256];

	if (arg_cnt <= 0) {  /* if there are no arguments. */
		combuf[0] = '\0';
		printf("Which Command? : ");
		if (gets(combuf, sizeof(combuf)) == NULL)
			return;	/* didn't type a command. */
		if (argsplit(combuf) < 0)
			return;	/* if no args to copy.. */
	}
	else
		comm_args[0] = comm_args[1];

	if (mp = search_menu(runtime_boot_level)) {
		printf("%s\n", mp->comm_text);
		printf("%s\n", mp->comm_desc);
		if (mp->help)
			printf("%s\n", mp->help);
	}
	else
		printf("'%s' not found\n", comm_args[0]);
}

/*
 * Global menu
 *
 * These are the global commands.
 *
 */

extern int	help();

struct init_menu i_global_comm[] = {
 { "h(elp)",	"", 0, 0, 1, print_help },
 { "?",		"", 0, 0, 0, help },
 { "l(ist)",	"", 0, 0, 0, print_menu },
 { 0 }
};

struct menu *global_comm = (struct menu *)i_global_comm;
