/*	START NEW ARIX SCCS HEADER			*/
/*							*/
/*	@(#) exceptions.c: version 25.1 created on 11/27/91 at 15:39:06	*/
/*							*/
/*	Copyright (c) 1990 by Arix Corporation		*/
/*	All Rights Reserved				*/
/*							*/
#ident	"@(#)exceptions.c	25.1	11/27/91 Copyright (c) 1990 by Arix Corporation"
/*							*/
/*	END NEW ARIX SCCS HEADER			*/
/*							*/

/* exceptions.c */

/*
 *  This module contains the interrupt service routines for the Service
 *  Processor board.
 */

/* This is for the spm, specifically, so use 68020 include files & defines
 * So, the 68020 flag should always be on. Hopefully, you don't need the   
 * 040 flag on at the same time.
 */
#ifdef	M68040
#undef	M68040
#define	M68020
#endif	/* M68040 */

#include "sys/param.h"
#include "misc.h"
#include "global.h"
#include "sys/sbus.h"
#include "sys/sbus_spm.h"
#include "sys/sbus_pm.h"
#include "sys/spm_mem.h"
#include "sys/ints.h"
#include "sys/types.h"
#include "sys/uadmin.h"
#include "spm_debug.h"
#include "spm.h"
#include "scc.h"
#include "cio.h"
#include "bus.h"
#include "csscmd.h"
#include "sys/state.h"
#include "setjmp.h"
#include "rwi.h"
#include "error.h"
#include "sbus_conf.h"
#include "sbus_iom.h"

uint		one_sec = TICKS_PER_SEC;/* clock tick counter		*/

#include "novram.h"

extern uint	iopm_cmd_unreset();

static uint	spm2kern_flag;		/* non-zero if a comand in progress */

extern uint	warning_state, shutdown_state;
extern uint	running_standalone;

uint		clock_tick;
uint		seconds_tick;
uint		n_seconds;
uint		unix_console_interrupt;
uint		int_counter;

/*
 * bus fault handler
 */

buserr(stk_frame)
struct	state stk_frame;		/* stk_frame is value/result */
{
	/* is there a probe in progress? */
	if (spm_caddrflt) {
		if (pm_booting_done) {	/* FIX JPC: use SPM state variable */
			showbuserr(&stk_frame);
			printf("Recoverable: spm_caddrflt=%x\n", spm_caddrflt);
		}
		stk_frame.s_pc = (ulong)spm_caddrflt;
		stk_frame.s_crunch = 1;	/* make a normal stack frame */ 
		spm_caddrflt= 0;
		return;
	}

	showbuserr(&stk_frame);

	if (!is_iomap(busviol(&stk_frame)))
		spm_dead();

	cpu_all_stop();

	unix_console_mode = 0;
	rtn_to_monitor();
}

/*
 * spm_dead -- unrecoverable SPM error:  blink the fault, on, and warning LEDs
 */

spm_dead()
{
	register int	i;

	spl7();
	for (;;) {
		SPMFAULT_LED_ON;
		ON_LIGHT_ON;
		WARNING_LIGHT_OFF;
		for (i = 0x40000; --i > 0;)
			;
		SPMFAULT_LED_OFF;
		ON_LIGHT_OFF;
		WARNING_LIGHT_ON;
		for (i = 0x40000; --i > 0;)
			;
	}
}

/* showbuserr() - display bus error information
 *
 * Parameter:
 *
 *	stk_ptr - stack pointer of bus error stack frame
 *
 */
showbuserr(stk_ptr)
register struct state *stk_ptr;
{
	register char	*p;
	register ulong	*r = &stk_ptr->s_d0;
	register ulong	sp;
	ulong		vaddr;
	int		i, is_map;
	static char	err_string[2][320];
	static int	err_index;
	extern uchar	stkfmtsz[];

	vaddr = busviol(stk_ptr);	/* fault address */

	spm_backtrace();

	i = err_index;
	err_index = (err_index + 1) & 1;
	p = err_string[i];
	p += sprintf(p, "\nBUS ERROR\n");
	sp = (ulong) &stk_ptr->s_proc + stkfmtsz[stk_ptr->s_format];

	p += sprintf(p, "data: %x %x %x %x %x %x %x %x\n",
		r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7]);
	p += sprintf(p, "addr: %08x %08x %08x %08x %08x %08x %08x %08x\n",
		r[8],r[9],r[10],r[11],r[12],r[13],r[14],sp);
	p += sprintf(p, "pc = %08x, sr = %04x, ", stk_ptr->s_pc, stk_ptr->s_ps);
	p += sprintf(p, "fault address = %08x\n",vaddr);

	if (is_map = is_iomap(vaddr))
		p += sprintf(p, "fault address mapped to %x:%08x\n",
		       iomap_to_slot(vaddr), iomap_to_offset(vaddr));
	p += sprintf(p, "special status word = %04x, ",
	       stk_ptr->s_proc.formatA.fA_ssw);

	break_out_ssw(stk_ptr->s_proc.formatA.fA_ssw, p);

	p = err_string[i];
	if (is_map && polling)
		queue_err(p);
	else
		printf("%s\n", p);
}

/*
 * break_out_ssw() - interprets the special status word from a bus error
 *		     stack frame.
 *
 * Parameter:
 *
 *	special status word from a bus error stack frame
 */

break_out_ssw(ssw, p)
register unsigned	ssw;
char	 		*p;
{
	if (DF(ssw)) {

		if (ssw & SSW_RMW)
			p += sprintf(p, "modify ");
		if (ssw & SSW_READ)
			p += sprintf(p, "read");
		else
			p += sprintf(p, "write");
	}
	else
		p += sprintf(p, "instruction fetch");

	*p++ = '\n';
}


/*
 * busviol() - returns the fault address on the bus fault stack frame
 *
 * Parameter:
 *
 *	stk_ptr - stack pointer of bus error stack frame
 *
 * Return value:
 *
 *	fault address
 */ 
busviol(stk_ptr)
register struct state *stk_ptr;
{
	ushort ssw;

	switch (stk_ptr->s_format) {

	case 0xA:	/* format A: short bus cycle fault stack frame */
		/* get special status word */
		ssw = stk_ptr->s_proc.formatA.fA_ssw;

		if (DF(ssw))
			/* data cycle fault address */
			return(stk_ptr->s_proc.formatA.fA_faultaddr);

		if (stageC(ssw))
			return((ulong)stk_ptr->s_pc + 2);

		if (stageB(ssw))
			return((ulong)stk_ptr->s_pc + 4);

		break;

	case 0xB:	/* format B: long bus cycle fault stack frame */
		/* get special status word */
		ssw = stk_ptr->s_proc.formatB.fB_ssw;

		if (DF(ssw))
			/* data cycle fault address */
			return(stk_ptr->s_proc.formatB.fB_faultaddr);

		if (stageC(ssw))
			return(stk_ptr->s_proc.formatB.fB_stBaddr - 2);

		if (stageB(ssw))
			return(stk_ptr->s_proc.formatB.fB_stBaddr);

		break;
	}

	panic("unknown cause of bus error (invalid stack frame format)");
	return(0);
}


/*
 * disp_err() - dispatcher error interrupt comes in at auto vector level 1
 */
disp_err()
{
	register uint	tmp;
	uint		disp_err_reg;

	disp_err_reg = *DISPERROR;

	cpu_all_stop();

	/* increase tail pointer */
	tmp = (error_tail == ERROR_SIZE) ? 0 : error_tail + 1;
	if(tmp != error_head) {
		error[error_tail].type = DISP_ERR;
		error[error_tail].data.disp_reg = disp_err_reg;
		error_tail = tmp;
	}

	dispatcher_error = 1;

	*WRCNTL1 &= ~WR1_DREQ;
	if (running_standalone)
		rtn_to_monitor();
}

/*
 * showdisperror() - interprets what was latched in the dispatcher error 
 *		     register.
 *
 * Parameters:
 *
 *	disp_err_reg - value read from the dispatcher error regsiter 
 */
show_disp_err(disp_err_reg)
register unsigned disp_err_reg;
{
	put_char('\n');
	print_time_stamp(0);
	printf("DISPATCHER ERROR: 0x%x\n", disp_err_reg);
	break_out_dispreg(disp_err_reg);
}

/*
 * break_out_dispreg() - explain what was latched in the dispatcher error
 *			 register
 * Parameter:
 *
 *	value read from the dispatcher error register
 */
break_out_dispreg(disp_err_reg)
register unsigned	disp_err_reg;
{
	register uint disp_err_code;

	printf("vector = 0x%x, ",(disp_err_reg & DISP_VEC_MASK) >> DISP_VEC_SH);

	if(disp_err_reg & DISP_DINT)
		printf("directed, ");
	else
		printf("non-directed, ");

	disp_err_code = (disp_err_reg & DISP_ERR_MASK) >> DISP_ERR_SH;

	switch(disp_err_code) {
		case DISP_HDWR0_ERR:
		case DISP_HDWR4_ERR:
			printf("hardware failure, ");
			break;
		case DISP_ACK_ERR:	/* insane ack error */
			printf("spurious ack received, ");
			break;
		case DISP_REQ2_ERR:	/* request error */
		case DISP_REQ6_ERR:	/* request error */
			printf(
		     "timeout or error on attempted interrupt level request, ");
			break;
		case DISP_RCV3_ERR:	/* receive error */
		case DISP_RCV7_ERR:	/* receive error */
			printf("new interrupt arrived, but queue full, ");
			break;
		case DISP_ACK_TOUT:	/* interrupt acknowledge response err */
			printf(
			"timeout or error on attempted acknowledge response, ");
			break;
		default:
			printf("undefined dispatcher error, ");	
			break;
	}

	printf("\npriority = 0x%x, ", 
		((disp_err_reg & DISP_PRI_MASK) >> DISP_PRI_SH) +
	  		((disp_err_reg & DISP_DINT) ? 5 : 0));

	/* 
	 * If the error is an ack error, report the destination field of the
	 * error register as the source of the ack error.
	*/
	if (disp_err_code == DISP_ACK_ERR || disp_err_code == DISP_ACK_TOUT) {
		printf("source = 0x%x\n",
			(disp_err_reg & DISP_DEST_MASK) >> DISP_DEST_SH);
		return;
	}

	printf("dest = 0x%x, ",(disp_err_reg & DISP_DEST_MASK) >> DISP_DEST_SH);

	if (disp_err_reg & DISP_IO_INT)
		printf("i/o src = 0x%x, ", disp_err_reg & DISP_IOSRC_MASK);
	else
		printf("CSS src = 0x%x, ",
			(disp_err_reg & DISP_CSSSRC_MASK) >> DISP_CSSSRC_SH);

	printf("\n");
}

/*
 * css_err() - CSS errors show up here. Read the status and CSS error registers
 *	       and display the symptom of the error.
 */
css_err()
{
	register uint 	css_err_reg,status_reg;
	register uint	tmp;

	/* read status reg 1st */
	status_reg = *STATUSREG;

	/* read error register 2nd */
	css_err_reg = *CSSERROR;

	/* are CSS errors being inhibited ?*/
	if( !(*WRCNTL0 & WR0_CSS_ERR_INHIBIT) || ignore_css_error) {
		return;
	}

	cpu_all_stop();

	/* increase tail pointer */
	tmp = (error_tail == ERROR_SIZE) ? 0 : error_tail + 1;
	if (tmp != error_head) {
		error[error_tail].type = CSS_ERR;
		error[error_tail].data.css.err_reg = css_err_reg;
		error[error_tail].data.css.status_reg = status_reg;
		error_tail = tmp;
	}

	if (running_standalone)
		rtn_to_monitor();
}


/*
 * showcsserror() - interprets what was latched in the css and status
 *		    registers.
 *
 * Parameters:
 *
 *	css_err_reg - value read from the css error regsiter 
 *	status_reg - value read from the status regsiter 
 */
show_css_err(css_err_reg,status_reg)
register unsigned css_err_reg,status_reg;
{
	put_char('\n');
	print_time_stamp(0);
	printf("CSS ERROR: 0x%x\n", css_err_reg);
	break_out_cssreg(css_err_reg);
	printf("status register 0x%x: ",status_reg);
	break_out_statreg(status_reg);
}

/*
 * break_out_cssreg() - explain what was latched in the css error register
 *
 * Parameter:
 *
 *	value read from the css error register
 */
break_out_cssreg(css_err_reg)
register uint css_err_reg;
{
	register uint i,type_field,modifier_field;
	register unchar bit_mask,bus_data_parity;
	
	if (css_err_reg & CSS_DERR)
		printf("destination error, ");
	if (css_err_reg & CSS_SERR)
		printf("source error, ");
	printf("dest %1x, ",(css_err_reg >>CSS_DEST_SH) & MASK4);
	printf("src %1x, ",(css_err_reg >>CSS_SRC_SH) & MASK4);

	printf("CSS data (57,43-40) %1x, ",(css_err_reg >> CSS_DATA_SH)); 

	/* 3 bit type field */
	type_field = (css_err_reg >> CSS_BTYPE_SH) & MASK3;  

	/* 3 bit modifier field */
	modifier_field = (css_err_reg >> CSS_MODIFIER_SH) & MASK3;

	printf("type %1o%1o, ", type_field, modifier_field);

	switch(type_field) {
	case (SBUS_TYPE_RESPONSE):
		printf("bus type: RESPONSE, ");
		switch(modifier_field) {
		case(SBUS_RESPONSE_ERR):
			printf("error\n");
			break;
		case(SBUS_RESPONSE_ERR_DATA0):
			printf("error data 0\n");
			break;
		case(SBUS_RESPONSE_ERR_DATA1):
			printf("error data 1\n");
			break;
		case(SBUS_RESPONSE_DATA):
			printf("normal\n");
			break;
		default:
			printf("undefined response type\n");
			break;
		}
		break;
	case (SBUS_TYPE_READ):
		printf("bus type: READ, ");
		switch(modifier_field) {
		case(SBUS_1BYTE_DATA):
			printf("1 byte\n");
			break;
		case(SBUS_2BYTES_DATA):
			printf("2 bytes\n");
			break;
		case(SBUS_3BYTES_DATA):
			printf("3 bytes\n");
			break;
		case(SBUS_4BYTES_DATA):
			printf("4 bytes\n");
			break;
		case(SBUS_8BYTES_DATA):
			printf("8 bytes\n");
			break;
		case(SBUS_16BYTES_DATA):
			printf("16 bytes\n");
			break;
		case(SBUS_32BYTES_DATA):
			printf("32 bytes\n");
			break;
		default:
			printf("undefined size\n");
			break;
		}
		break;
	case (SBUS_TYPE_WRITE):
		printf("bus type: WRITE, ");
		switch(modifier_field) {
		case(SBUS_1BYTE_DATA):
			printf("1 byte\n");
			break;
		case(SBUS_2BYTES_DATA):
			printf("2 bytes\n");
			break;
		case(SBUS_3BYTES_DATA):
			printf("3 bytes\n");
			break;
		case(SBUS_4BYTES_DATA):
			printf("4 bytes\n");
			break;
		default:
			printf("undefined size (%1o)\n", modifier_field);
			break;
		}
		break;
	case (SBUS_TYPE_CWRITE):
		printf("bus type: CONTROL WRITE, ");
		switch(modifier_field) {
		case(SBUS_MODULE_ENABLE_OFF):
			printf("module enable off\n");
			break;
		case(SBUS_MODULE_ENABLE_ON):
			printf("module enable on\n");
			break;
		case(SBUS_MODULE_INTERFACE_OFF):
			printf("module interface enable off\n");
			break;
		case(SBUS_MODULE_INTERFACE_ON):
			printf("module interface enable on\n");
			break;
		default:
			printf("undefined control signal\n");
		}
		break;
	default:
		printf("illegal bus type\n");
		break;
	}

	/* bus type parity error bit is active low */
  
	if(!(css_err_reg & CSS_BPAR_ERR))
		printf("type parity error, ");

	if((bus_data_parity = css_err_reg & MASK8) != 0xff) {
		printf("data parity error");
		for(bit_mask=0x01,i=0; i < 8; bit_mask <<= 1, i++)
			if(!(bit_mask & bus_data_parity))
				printf(", byte %1x",i);
		printf("\n");
	}
}


/*
 * break_out_statreg() - explain what was latched in the status register
 *
 * Parameter:
 *
 *	value read from the status register
 */
break_out_statreg(status_reg)
register unsigned status_reg;
{
	/* bad command bit */
	if(status_reg & STAT_BCMD)
		/* bad parity, or illegal type, or source/dest mismatch */
		printf("CSS bus protocol violation, ");


	/* css arbiter grant error ? */
	if(!(status_reg & STAT_GERR))
		printf("CSS arbiter grant error, ");

	/* css bus ack ? */
	if(status_reg & STAT_BACK)
		printf("CSS bus ack, ");
	
	/* CSS bus nack ? */
	if(status_reg & STAT_NACK)
		printf("CSS bus nack, ");

	if(!(status_reg & (STAT_NACK | STAT_BACK)))
		printf("no ack, ");

	printf("\n");
}

trap(stk_frame)
struct	state	stk_frame;	/* stk_frame is value/result */
{
	showtrap(&stk_frame);
	panic("");
}

static	char	bad_trap_name[] = "Undefined Exception";
static	char	reserved_trap_name[] = "Reserved Exception";
static	char	user_trap_name[] = "User Defined Exception";

char *nametrap[] = {
	bad_trap_name,
	bad_trap_name,
	"Bus Error",
	"Address Error",
	"Illegal Instruction",
	"Zero Divide",
	"CHK Instruction",
	"TRAPV Instruction",
	"Privilege Violation",
	"Trace Trap",
	bad_trap_name,
	bad_trap_name,
	reserved_trap_name,
	"Coprocessor Protocol Violation",
	"Format Error",
	"Uninitialized Interrupt",
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	"Spurious Interrupt",
	"Auto 1",
	"Auto 2",
	"Auto 3",
	"Auto 4",
	"Auto 5",
	"Auto 6",
	"Auto 7",
	"TRAP 0 Instruction",
	"TRAP 1 Instruction",
	"TRAP 2 Instruction",
	"TRAP 3 Instruction",
	"TRAP 4 Instruction",
	"TRAP 5 Instruction",
	"TRAP 6 Instruction",
	"TRAP 7 Instruction",
	"TRAP 8 Instruction",
	"TRAP 9 Instruction",
	"TRAP 10 Instruction",
	"TRAP 11 Instruction",
	"TRAP 12 Instruction",
	"TRAP 13 Instruction",
	"TRAP 14 Instruction",
	"TRAP 15 Instruction",
	"Floating Point Exception",
	"Floating Point Exception",
	"Floating Point Exception",
	"Floating Point Exception",
	"Floating Point Exception",
	"Floating Point Exception",
	"Floating Point Exception",
	reserved_trap_name,
	bad_trap_name,
	bad_trap_name,
	bad_trap_name,
	reserved_trap_name,
	reserved_trap_name,
	reserved_trap_name,
	reserved_trap_name,
	reserved_trap_name,
};

static	char *
get_except_name(vector_number)
uint	vector_number;
{
	if (vector_number >= (sizeof(nametrap) / sizeof(nametrap[0])))
		return(user_trap_name);
	return(nametrap[vector_number]);
}

showtrap(stk_ptr)
register struct state *stk_ptr;
{
	register ulong	*r = &stk_ptr->s_d0;
	register ulong	sp;
	extern	unsigned char	stkfmtsz[];

	put_char('\n');
	print_time_stamp(0);
	printf("SPM System Mode ");
	sp = (ulong) &stk_ptr->s_proc + stkfmtsz[stk_ptr->s_format];

	printf("trap type %s, vector offset 0x%x\n",
		 get_except_name(stk_ptr->s_vector), stk_ptr->s_vector * 4);

	printf("data regs: %x %x %x %x %x %x %x %x\n",
		r[0],r[1],r[2],r[3],r[4],r[5],r[6],r[7]);
	printf("addr regs: %x %x %x %x %x %x %x %x\n",
		r[8],r[9],r[10],r[11],r[12],r[13],r[14],sp);
	printf("pc = %x, sr = %x\n", stk_ptr->s_pc, stk_ptr->s_ps);

	/*
	 * Its always safe to longjmp out of these exceptions 
	 */
	switch (stk_ptr->s_vector) {
	case 5:		/* Zero Divide */
		longjmp(environbuf, 1);		/* jump back to monitor */
	}
	spm_dead();
}

/*	Called by the ASSERT macro in debug.h when an
**	assertion fails.
*/

assfail(a, f, l, jumpflag)
char *a, *f;
uint	jumpflag;
{
	put_char('\n');
	print_time_stamp(0);
	printf("SPM assertion failed: %s, file: %s, line: %d", a, f, l);

	if (jumpflag)
	{
		spm_backtrace();
		/* non level 0 guarantess that this error was
		 * generated from code called by the spm code,
		 * and not the user.  attempt to shut down the
		 * system.
		 */
		if ( get_spl() ) {
/* FIX THIS, DS:  all this should get lumped together in a shutdown routine */
			printf("assfail: shutting down system.\n");
			kern_clock_enable = 0;
			unix_console_mode = 0;
			cpu_all_stop();
		}
		rtn_to_monitor();
	}
	else
		panic("assert failure!");
}

/*
 * Panic is called on unresolvable fatal errors.
 */
panic(arg)
char	*arg;
{
	spm_backtrace();
	printf("\nPANIC: %s\n", arg);
	spm_dead();
}

cmd_int()
{
	register uint	status_reg, sbus_slot;
	uint		command_reg;

	status_reg = *STATUSREG;
	command_reg = *CSSCMD;
	iready();

	sbus_slot = (status_reg & STAT_SRC_SLOT_MASK) >> STAT_SRC_SLOT_SH;

	switch (sbus_config.slot_id[sbus_slot]) {
	case SBUS_DPM40_DBL:
	case SBUS_DPM40_SGL:
		if (!spm_mem_copy_called)
			return;		/* we expect these during boot */
	}

	print_time_stamp(0);
	printf("CSS command write to SPM of 0x%x from slot %u, byte 6 = %#x\n",
	  command_reg, sbus_slot, (status_reg & STAT_BYTE6) >> STAT_BYTE6_SH);
}

spm_int()
{
	disp_int_t 	interrupt_data;
	uint		saved_map = iomap_save();

	/* ACK the dispatcher */
	if (!probe_rd_long((caddr_t)iomap(spm_slot, SPM_ACK_LEVEL_FIVE),
	  &interrupt_data.entire_32_bits)) {
		printf("spm_int: SPM_ACK_LEVEL_FIVE failed!\n");
		iomap_restore(saved_map);
		return;
	}

	switch (interrupt_data.fields.vector) {
	case IOM_ERROR:
		cpu_all_stop();
		get_iom_err_info(interrupt_data.fields.src_slot);
		break;
	case IOSBA_ERROR:
		cpu_all_stop();
		get_iosba_err_info(interrupt_data.fields.src_slot);
		break;
	case MM_INTERRUPT:
		mm_interrupt(interrupt_data.fields.src_slot);
		break;
	case KERNEL_TO_SPM_CMD:
		kern2spm_cmd();
		break;
	case KERNEL_TO_SPM_OUTPUT:
		K_ASSERT(! unix_console_interrupt);
		unix_console_interrupt = 1;
		break;
	case SPM_TO_KERNEL_DONE:
		spm2kern_flag = 0;
		break;
	default:
		printf("spm_int: unknown interrupt vector = %d\n",
		  interrupt_data.fields.vector);
		panic("");
	}
	iomap_restore(saved_map);
}

/*
 * kern2spm_cmd -- do a generic command for the kernel
 */

kern2spm_cmd()
{
	uint		reply;
	extern uint	get_margin(), set_margin();
	extern uint	get_rtc_cal(), set_rtc_cal();

	reply = 0;

	switch (Spm_Mem->kern2spm_cmd) {
	case K2S_SETTIME:
		set_time(Spm_Mem->kern2spm_arg[0]);
		break;
	case K2S_SETMARGIN:
		reply =
		  set_margin(Spm_Mem->kern2spm_arg[0],Spm_Mem->kern2spm_arg[1]);
		break;
	case K2S_GETMARGIN:
		reply = get_margin(Spm_Mem->kern2spm_arg[0]);
		break;
	case K2S_UADMIN:
		do_uadmin(Spm_Mem->kern2spm_arg[0], Spm_Mem->kern2spm_arg[1]);
		break;
	case K2S_GET_CLK_CAL:
		reply = get_rtc_cal();
		break;
	case K2S_SET_CLK_CAL:
		reply = set_rtc_cal(Spm_Mem->kern2spm_arg[0]);
		break;
	case K2S_CCDEBUG:
		reply = set_spm_char(Spm_Mem->kern2spm_arg[0],
		  Spm_Mem->kern2spm_arg[1]);
		break;
	case K2S_RESETIOPM:
		/* kern2spm_arg[0] = IOPM slot */
		reply = iopm_cmd_reset(Spm_Mem->kern2spm_arg[0]);
		break;
	case K2S_STARTIOPM:
		/* param1 : ioslot, param2 : entry point */
		reply = iopm_cmd_unreset(Spm_Mem->kern2spm_arg[0],
			Spm_Mem->kern2spm_arg[1]);
		break;
	default:
		printf("kern2spm_smd: unknown command = 0x%x\n",
		  Spm_Mem->kern2spm_cmd);
	}
	*Spm_Mem->kern2spm_arg = reply;
	signal_kernel(KERNEL_TO_SPM_DONE);
}

/*
 * signal_kernel -- send an interrupt to a PM for SPM-to-Kernel communications
 */

signal_kernel(vector)
uint	vector;
{
	disp_int_t	d_int;

	d_int = Spm_Mem->spm2kern_int;
	ASSERT(d_int.entire_32_bits);

	switch (vector) {
	case KERNEL_TO_SPM_DONE:
	case SPM_TO_KERNEL_CMD:
		d_int.fields.vector = vector;
		break;
	default:
		printf("signal_kernel: unknown vector = 0x%x!\n", vector);
		return;
	}

	if (! probe_wr_long(interrupt_dispatcher, d_int.entire_32_bits)) {
		printf("signal_kernel: bus error while sending interrupt!\n");
		cpu_all_stop();
	}
}

/*
 * s2k_wait -- wait for command to complete, returns non-zero on timeout
 */

static uint
s2k_wait()
{
	int	i;
	uint	spl_value = 0;

	if (get_spl() >= 3)
		spl_value = spl2();

	/*
	 * time out in about ten milliseconds
	 */
	for (i = 10000; --i >= 0 && spm2kern_flag; )
		;

	if (spl_value)
		splx(spl_value);
	if (i < 0) {
		printf("spm2kern_cmd: timeout waiting for reply!\n");
		return (1);
	}
	return (0);
}

/*
 * spm2kern_cmd -- send a generic command to the kernel, returns BAD_UINT on err
 */
/*VARARGS1*/
uint
spm2kern_cmd(cmd, arg0, arg1, arg2, arg3)
uint	cmd, arg0, arg1, arg2, arg3;
{
	uint	old_reply = 0, reply;

	ASSERT(pm_booting_done);

	if (spm2kern_flag) {		/* wait for prev command to finish
		if (s2k_wait())
			return (BAD_UINT);

		old_reply = *Spm_Mem->spm2kern_arg;	/* prev return value */
	}

	Spm_Mem->spm2kern_cmd = cmd;
	Spm_Mem->spm2kern_arg[0] = arg0;
	Spm_Mem->spm2kern_arg[1] = arg1;
	Spm_Mem->spm2kern_arg[2] = arg2;
	Spm_Mem->spm2kern_arg[3] = arg3;
	spm2kern_flag = 1;

	signal_kernel(SPM_TO_KERNEL_CMD);

	if (s2k_wait())
		return (BAD_UINT);

	reply = *Spm_Mem->spm2kern_arg;		/* return value */
	*Spm_Mem->spm2kern_arg = old_reply;	/* old return value */
	return (reply);
}

LCIO_err()
{
	if (LOCCIO->ctcs[0].reg & ICR_IP)		/* timer 1 int? */
		LOCCIO->ctcs[0].reg = ICW_CPUS | CT_GCB;
	if (LOCCIO->ctcs[1].reg & ICR_IP)		/* timer 2 int? */
		LOCCIO->ctcs[1].reg = ICW_CPUS | CT_GCB;
	if (LOCCIO->ctcs[2].reg & ICR_IP) {		/* timer 3 int? */
		LOCCIO->ctcs[2].reg = ICW_CPUS | CT_GCB;
		LOCCIO->ctcs[2].reg = ICW_CIE;	/* interrupt disable	*/
	}
}

LCIOt1_int()
{
	uint	 spl_level;

	if (!(LOCCIO->ctcs[0].reg & ICR_IP)) {	/* timer 1 int pending ? */
		spl_level = spl1();
		printf("Spurious Interrupt, local CIO timer 1\n");
		splx(spl_level);
		return;
	}

	/* clear int pending and int under service */
	LOCCIO->ctcs[0].reg = ICW_CPUS | CT_GCB;

	if (kern_clock_enable) {
		kern_clock();
		Spm_Mem->lbolt++;
	}
	clock_tick = 1;				/* used in poll routine */

	int_counter++;
	if (--one_sec == 0) {			/* 1 sec */
		one_sec = TICKS_PER_SEC;
		seconds_tick = 1;
		n_seconds++;
	}
	if (shutdown_state) {
		if (!(int_counter & 3))			/* fast blink */
			toggle_warning_light();
	}
	else if (warning_state) {
		if (!(int_counter & 15))		/* slow blink */
			toggle_warning_light();
	}
}


LCIOt2_int()
{
	uint		spl_level;
	static uint	ticks_500;

	if (LOCCIO->ctcs[1].reg & ICR_IP) {	/* timer 2 int pending ? */

		/*
		 * clear int pending and int under service
		 */
		LOCCIO->ctcs[1].reg = ICW_CPUS | CT_GCB;

		if (kern_clock_enable)
			Spm_Mem->ticks_500 = ++ticks_500;
	}
	else {
		spl_level = spl1();
		printf("Spurious Interrupt, local CIO timer 2\n");
		splx(spl_level);
	}
}

LCIOt3_int()
{
	uint	spl_level;

	if (LOCCIO->ctcs[2].reg & ICR_IP) {	/* timer 3 int pending ? */

		/* clear int pending and int under service */
		LOCCIO->ctcs[3].reg = ICW_CPUS | CT_GCB;
	}
	else {
		spl_level = spl1();
		printf("Spurious Interrupt, local CIO timer 3\n");
		splx(spl_level);
	}
}


/*
 * kern_clock:
 *
 *	Sends interrupts to all cpus
 *	Special care however must be taken with the upkern cpu
 *
 *	Cpu_all_send_clock returns true if it was able to send
 *	the clock interrupt to a upkern cpu.
 *	This indicates that the cpu will be responsible for doing
 *	the upkern_dec.
 *
 *	If, however, cpu_all_send_clock returns false it means that
 *	no cpu took responsibility.  This is presumed to be caused by
 *	the upkern cpu not being ready for another interrupt.
 *	In this case, we keep the upkern information and use it next
 *	time we are called.
 */

kern_clock()
{
	static uint upkern_ready = 1;
	static uint upkern_id;

	if (upkern_ready) {
		if ((upkern_id = upkern_inc(&Spm_Mem->upkern)) == -1) {
			kern_clock_enable = 0;
			queue_err("SPM Bus error in upkern_inc");
			cpu_all_stop();
			return;
		}
	}
	upkern_ready = cpu_all_send_clock(upkern_id);
}



#define OP_ADDA_A7	0xdefc		/* 68020 opcode for adda.w #xx, a7 */
#define OP_ADDQ_A7	0x504f		/* 68020 opcode for addq.l #xx, a7 */
#define ADDQ_MASK	0x0e00		/* mask to remove data from addq */
#define ADDQ_SHIFT	9		/* shift to extract data from addq */

/*	SPM backtrace, called during spm panic				*/

/*VARARGS*/
spm_backtrace(arg)
uint	arg;		/* dummy arg to get starting fp */
{
	register uint	n, retpc;
	register int	args;
	register uint	*fp;
	register uint	*lower_limit;
	register ushort	*addr;
	extern uint	busint();
	extern uint	stack_end[];

	fp = &arg - 2;	/* stack frame is fp, pc, arg, .... */
	put_char('\n');
	print_time_stamp(0);
	printf("SPM Stack Backtrace:\n");

	retpc = fp[1];
	lower_limit = fp;
	fp = (uint *) *fp;

	for ( ; fp > lower_limit; lower_limit = fp, fp = (uint *) *fp) {

		if (&fp[1] >= stack_end)
			break;
		addr = (ushort *) fp[1];
		printf("%x (", retpc);	/* print return pc */
		retpc = (uint) addr;

		/*
		 * Guess at the number arguments by looking at the opcode
		 * pointed at by the return pc and checking how much is added
		 * to the stack.  Divide by 4 to get the number of arguments.
		 */
		n = *addr;
		if ((n & ~ADDQ_MASK) == OP_ADDQ_A7) {
			switch (n = (n & ADDQ_MASK) >> ADDQ_SHIFT) {
			case 0:		/* addq 8 */
				args = 2;
				break;
			case 4:		/* addq 4 */
				args = 1;
				break;
			default:
				printf("[addq %d]", n);
				args = 0;
			}
		}
		else if (n == OP_ADDA_A7)
			args = ((short *)addr)[1] / 4;
		else {
#ifdef BACKTRACE_DEBUG
			printf("[%x=%x] ", addr, n);
#endif
			args = 0;
		}

		if (&fp[2 + args] > stack_end) {
			printf(")\n");
			break;
		}
		if (args > 8) {
			printf("{%d args} ", args);
			args = 8;
		}
		for (n = 2; --args >= 0; n++)
			printf("%x%s", fp[n], (args ? ", " : ""));

		printf(")\n");
	}
	printf("%x\n", retpc);		/* print return pc */
}


queue_err(str)
char *str;
{
	uint tmp;

	tmp = (error_tail == ERROR_SIZE) ? 0 : error_tail + 1;

	if(tmp != error_head) {
		error[error_tail].type = SPM_ERR;
		error[error_tail].data.spm.str = str;
		error_tail = tmp;
	}
}
